/***************************************************************************
                              kstviewlegend.cpp
                             ------------------
    begin                : September 2005
    copyright            : (C) 2005 by cbn
                           (C) 2005 by The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kstviewlegend.h"

#include "enodes.h"
#include "ksdebug.h"
#include "kst.h"
#include "kstdatacollection.h"
#include "labelrenderer.h"
#include "kstsettings.h"
#include "ksttimers.h"
#include "kst2dplot.h"


#include <kdatastream.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpopupmenu.h>

#include <qapplication.h>
#include <qbitmap.h>
#include <qdeepcopy.h>
#include <qmetaobject.h>
#include <qptrstack.h>
#include <qstylesheet.h>

#include <stdlib.h>

#define MIN_FONT_SIZE 5

KstViewLegend::KstViewLegend()
: KstBorderedViewObject("Legend") {
  _rotation = 0;
  _vertical = true;
  _fontName = KstApp::inst()->defaultFont();
  _fontSize = 0;
  setBorderWidth(2);
  _absFontSize = _fontSize+KstSettings::globalSettings()->plotFontSize;
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | Edit;
  _standardActions |= Delete | Edit;
  computeTextSize();
  setDirty(false);
}


KstViewLegend::KstViewLegend(const QDomElement& e)
: KstBorderedViewObject(e) {

  // some defaults and invariants
  _type = "Legend";
  _rotation = 0.0;
  _fontName = KstApp::inst()->defaultFont();
  _fontSize = 0;
  _vertical = true;
  _absFontSize = _fontSize+KstSettings::globalSettings()->plotFontSize;
  _layoutActions |= Delete | Raise | Lower | RaiseToTop | LowerToBottom | Rename | Edit;
  _standardActions |= Delete | Edit;
  QStringList ctaglist;
  
  // read the properties
  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement el = n.toElement();
    if (!el.isNull()) {
      if (metaObject()->findProperty(el.tagName().latin1(), true) > -1) {
        setProperty(el.tagName().latin1(), QVariant(el.text()));
      } else if (el.tagName() == "curvetag") {
        ctaglist.append(el.text()); 
      }
    }
    n = n.nextSibling();
  }

  KstBaseCurveList l = kstObjectSubList<KstDataObject,KstBaseCurve>(KST::dataObjectList);
  KstBaseCurveList::ConstIterator end = l.end();
  for (QStringList::ConstIterator str = ctaglist.begin(); str != ctaglist.end(); ++str) {
    KstBaseCurveList::ConstIterator it = l.findTag(*str);
    if (it != end) {
      addCurve(*it);
    }
  }
}


KstViewLegend::~KstViewLegend() {
}


void KstViewLegend::resize(const QSize& size) {
  KstBorderedViewObject::resize(size);
  drawToBuffer();
}


int KstViewLegend::ascent() const {
  return _ascent;
}


void KstViewLegend::setFontName(const QString& fontName) {
  if (_fontName != fontName) {
    setDirty();
    _fontName = fontName;
  }
}


const QString& KstViewLegend::fontName() const {
  return _fontName;
}


void KstViewLegend::save(QTextStream &ts, const QString& indent) {
  ts << indent << "<" << type() << ">" << endl;
  KstBorderedViewObject::save(ts, indent + "  ");
  
  for (KstBaseCurveList::ConstIterator j = Curves.begin(); j != Curves.end(); ++j) {
    (*j)->readLock();
    ts << indent+"  " << "<curvetag>" << QStyleSheet::escape((*j)->tagName()) << "</curvetag>" << endl;
    (*j)->readUnlock();
  }

  ts << indent << "</" << type() << ">" << endl;
}


void KstViewLegend::drawToBuffer() {
  setDirty(false);

  _backBuffer.buffer().resize(size());
  _backBuffer.buffer().fill(backgroundColor());
  QPainter p(&_backBuffer.buffer());
  QPen pen;
  pen.setColor(foregroundColor());
  p.setPen(pen);
  drawToPainter(p);
}


void KstViewLegend::drawToPainter(QPainter& p) {
  RenderContext rc(_fontName, _absFontSize, &p);

  if (_vertical) {
    unsigned i=0;
    for (KstBaseCurveList::Iterator it = Curves.begin(); it != Curves.end(); ++it) {
      p.save();
      if ((*it)->parsedLegendTag()) {
        p.translate(_ascent/2, _ascent/4 + (i)*(rc.fontHeight()+_ascent/4));
        QRect symbolBound(QPoint(0,0),
                          QSize(16*_ascent/4, rc.fontHeight()));

        (*it)->paintLegendSymbol(&p, symbolBound);
        p.translate(9*_ascent/2, 0);
        rc.x = 0;
        rc.y = _ascent;
        rc.xStart = rc.x;
        renderLabel(rc, (*it)->parsedLegendTag()->chunk);
      }
      p.restore();
      ++i;
    }
  } else {
    p.save();
    p.translate(_ascent/2, _ascent/4);
    for (KstBaseCurveList::Iterator it = Curves.begin(); it != Curves.end(); ++it) {
      if ((*it)->parsedLegendTag()) {
        QRect symbolBound(QPoint(0,0),
                          QSize(16*_ascent/4, rc.fontHeight()));
        (*it)->paintLegendSymbol(&p, symbolBound);
        p.translate(9*_ascent/2, 0);
        rc.x = 0;//Curves[i]->legendLabelSize().width() / 2;
        rc.y = _ascent;
        rc.xStart = rc.x;
        renderLabel(rc, (*it)->parsedLegendTag()->chunk);
        p.translate((*it)->legendLabelSize().width() + _ascent,0);
      }
    }
    p.restore();
  }

  QApplication::syncX();
}


void KstViewLegend::computeTextSize() {
  _textWidth = 0;
  _ascent = 0;
  _textHeight = 0;

  for(KstBaseCurveList::iterator it = Curves.begin(); it != Curves.end(); it++) {
    if ((*it)->parsedLegendTag()) {
      RenderContext rc(_fontName, _absFontSize, 0L);
      renderLabel(rc, (*it)->parsedLegendTag()->chunk);
      if (_vertical) {
        if (rc.xMax>_textWidth) _textWidth = rc.xMax;
      } else {
        if (rc.fontHeight()>_textHeight) _textHeight = rc.fontHeight();
        _textWidth += rc.xMax;
      }
      (*it)->setLegendLabelSize(QSize(rc.xMax, rc.fontHeight()));
    } else {
      (*it)->setLegendLabelSize(QSize(0,0));
    }
  }
  RenderContext rc(_fontName, _absFontSize, 0L);
  _ascent = rc.fontAscent();
  if (_vertical) {
    _textHeight = Curves.count()*rc.fontHeight() + (Curves.count()-1)*_ascent/4;
  } else {
    _textWidth += Curves.count()*_ascent;
  }
}


void KstViewLegend::paint(KstPaintType type, QPainter& p, const QRegion& bounds) {
  if (type == P_PRINT) {
    p.save();
    adjustSizeForText(p.window());
    p.setViewport(geometry());
    p.setWindow(0,0,geometry().width(), geometry().height());
    drawToPainter(p);
    //setDirty();
    p.restore();
  } else {
    if (type == P_UPDATE) {
      setDirty();
    }
    bool d = dirty();
    if (d) {
      adjustSizeForText(p.window()); // calls computeTextSize and drawToBuffer
    }

    if (_transparent) {
      QRegion oldRegion = p.clipRegion();
      p.setClipRegion(oldRegion & clipRegion());
      _backBuffer.paintInto(p, geometry());
      p.setClipRegion(oldRegion);
    } else {
      _backBuffer.paintInto(p, geometry());
    }
  }
  KstBorderedViewObject::paint(type, p, bounds);
}

QRegion KstViewLegend::clipRegion() {
  if (!_transparent) {
    return KstBorderedViewObject::clipRegion();
  }

  if (_clipMask.isNull()) {
    QBitmap bm = _backBuffer.buffer().createHeuristicMask(false); // slow but preserves antialiasing...
    _clipMask = QRegion(bm);
    _clipMask.translate(geometry().topLeft().x(), geometry().topLeft().y());
  }

  return _clipMask;
}


void KstViewLegend::setFontSize(int size) {
  if (_fontSize != size) {
    _absFontSize = size + KstSettings::globalSettings()->plotFontSize;
    if (_absFontSize < MIN_FONT_SIZE) {
      _absFontSize = MIN_FONT_SIZE;
    }

    _fontSize = size;
    setDirty();
  }
}


int KstViewLegend::fontSize() const {
  return _fontSize;
}


void KstViewLegend::adjustSizeForText(QRect w) {
  double x_s, y_s, s;
  int width, height;

  x_s = y_s = _fontSize + (double)KstSettings::globalSettings()->plotFontSize;

  int x_pix = w.width();
  int y_pix = w.height();

  if (x_pix < y_pix) {
    x_s *= x_pix/540.0;
    y_s *= y_pix/748.0;
  } else {
    y_s *= y_pix/540.0;
    x_s *= x_pix/748.0;
  }

  s = (x_s + y_s)/2.0;

  if (s < MIN_FONT_SIZE) {
    s = MIN_FONT_SIZE;
  }
  _absFontSize = int(s);
  if (_absFontSize < MIN_FONT_SIZE) {
    _absFontSize = MIN_FONT_SIZE;
  }

  computeTextSize();

  if (_vertical) {
    width = _textWidth + _ascent/2 + 5*_ascent;
  } else {
    width = _textWidth + _ascent/2 + 9*_ascent*Curves.count()/2;
  }
  height = _textHeight + _ascent/2;

  QSize sz(width, height);
  if (_parent) {
    QRect r(position(), sz);
    resize(r.intersect(_parent->geometry()).size());
  } else {
    resize(sz);
  }
}


bool KstViewLegend::layoutPopupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent) {
  KstViewObject::layoutPopupMenu(menu, pos, topLevelParent);
  return true;
}


KstViewObjectPtr create_KstViewLegend() {
  return KstViewObjectPtr(new KstViewLegend());
}


KstViewObjectFactoryMethod KstViewLegend::factory() const {
  return &create_KstViewLegend;
}


/* FIXME: not written for legends yet. */
void KstViewLegend::writeBinary(QDataStream& str) {
}


/* FIXME: not written for legends yet. */
void KstViewLegend::readBinary(QDataStream& str) {
}


QMap<QString, QVariant> KstViewLegend::widgetHints(const QString& propertyName) const {
  QMap<QString, QVariant> map = KstBorderedViewObject::widgetHints(propertyName);
  if (!map.empty()) {
    return map;
  }

  if (propertyName == "text") {
    map.insert(QString("_kst_widgetType"), QString("QLineEdit"));
    map.insert(QString("_kst_label"), i18n("Text"));
  } else if (propertyName == "font") {
    map.insert(QString("_kst_widgetType"), QString("KFontCombo"));
    map.insert(QString("_kst_label"), i18n("Font"));
  } else if (propertyName == "foreColor") {
    map.insert(QString("_kst_widgetType"), QString("KColorButton"));
    map.insert(QString("_kst_label"), i18n("Font color"));
  } else if (propertyName == "backColor") {
    map.insert(QString("_kst_widgetType"), QString("KColorButton"));
    map.insert(QString("_kst_label"), i18n("Background color"));
  } else if (propertyName == "fontSize") {
    map.insert(QString("_kst_widgetType"), QString("QSpinBox"));
    map.insert(QString("_kst_label"), i18n("Font size"));
    map.insert(QString("minValue"), MIN_FONT_SIZE - (double)KstSettings::globalSettings()->plotFontSize);
  } else if (propertyName == "transparent") {
    map.insert(QString("_kst_widgetType"), QString("QCheckBox"));
    map.insert(QString("_kst_label"), QString::null);
    map.insert(QString("text"), i18n("Transparent fill"));
  }
  return map;
}


bool KstViewLegend::transparent() const {
  return KstViewObject::transparent();
}


void KstViewLegend::setTransparent(bool transparent) {
  KstViewObject::setTransparent(transparent);
}


void KstViewLegend::addCurve(KstBaseCurvePtr incurve) {
  Curves.append(incurve);
  setDirty();
}


void KstViewLegend::removeCurve(KstBaseCurvePtr incurve) {
  Curves.remove(incurve);
  setDirty();
}


void KstViewLegend::setCurveList(Kst2DPlot *pl) {
  Curves = QDeepCopy<KstBaseCurveList>(pl->Curves);
}

#include "kstviewlegend.moc"
// vim: ts=2 sw=2 et
