/***************************************************************************
                          kbudgetview.cpp  -  description
                             -------------------
    begin                : Sat Aug 10 16:03:35 CST 2002
    copyright            : (C) 2002 by Richard Garand
    email                : richard@garandnet.net
    $Id: kbudgetview.cpp,v 1.25 2002/09/10 01:02:50 richard Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// system includes
#include <time.h>
#include <string.h>
#include <stdio.h>

// include files for Qt
#include <qprinter.h>
#include <qpainter.h>
#include <klocale.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <kiconloader.h>

// application specific includes
#include "kbudgetview.h"
#include "kbudgetdoc.h"
#include "kbudget.h"
#include "kbudgeteditor.h"
#include "klocalizer.h"
#include "ktransactionwindow.h"
#include "knewaccount.h"

const char* months[] = { "January", "February", "March", "April", "May", "June", "July",
						 "August", "September", "October", "November", "December" };
const int COLUMNS = 4;

KBudgetView::KBudgetView(QWidget *parent, const char *name) : QVBox(parent, name), iColour(0,128,0),
  eColour(255,0,0), cColour(0,0,0), type(VT_NONE), selAccount(-1)
{
  setBackgroundMode(PaletteBase);

  periodDisplay = new QHBox(this);

  dtab = new QTable(30, COLUMNS, this);
  setStretchFactor(dtab, 1);
  dtab->setShowGrid(false);
  dtab->horizontalHeader()->hide();
  dtab->verticalHeader()->hide();
  dtab->setTopMargin(0);
  dtab->setLeftMargin(0);
  dtab->setSelectionMode(QTable::SingleRow);

  time_t curTime = time(0);
  tm* datetime = localtime(&curTime);

  editor = new KBudgetEditor(this, getDocument());

  selectMonth(1900 + datetime->tm_year, datetime->tm_mon);

  connect(getDocument(), SIGNAL(accountAddDel()), SLOT(slotAccountAddDel()));
  connect(getDocument(), SIGNAL(accountChanged(int)), SLOT(slotAccountChanged(int)));
  connect(dtab, SIGNAL(selectionChanged()), SLOT(slotSelectionChanged()));
  connect(dtab, SIGNAL(contextMenuRequested(int, int, const QPoint&)), SLOT(slotContextMenuRequested(int, int, const QPoint&)));
  connect(dtab, SIGNAL(valueChanged(int, int)), SLOT(slotValueChanged(int, int)));
  connect(editor, SIGNAL(valueChanged(int)), SLOT(slotAccountChanged(int)));
}

KBudgetView::~KBudgetView()
{
}

void KBudgetView::selectMonth(int yr, int mn)
{
  if ( !from.setYMD(yr, mn + 1, 1) )
    return;

  if ( type != VT_MONTH ) {
    cleanupPeriodDisplay();

    prevButton = new QPushButton(periodDisplay);
    QPixmap backPix = SmallIcon("back");
    QIconSet backSet = QIconSet(backPix);
    prevButton->setIconSet(backSet);
    connect(prevButton, SIGNAL(clicked()), this, SLOT(slotPrevClick()));

    QLabel* spacer1 = new QLabel(periodDisplay);
    periodDisplay->setStretchFactor(spacer1, 1);

    month = new KComboBox(periodDisplay);
    connect(month, SIGNAL(activated(int)), this, SLOT(slotMonthActivate(int)));
    for ( int i = 0; i < 12; i++ )
      month->insertItem(months[i]);

    year = new KIntNumInput(periodDisplay);
    connect(year, SIGNAL(valueChanged(int)), this, SLOT(slotYearChange(int)));
    year->setMinValue(1752);
    year->setMaxValue(8000);

    spacer2 = new QLabel(periodDisplay);
    periodDisplay->setStretchFactor(spacer2, 1);

    nextButton = new QPushButton(periodDisplay);
    QPixmap fwdPix = SmallIcon("forward");
    QIconSet fwdSet = QIconSet(fwdPix);
    nextButton->setIconSet(fwdSet);
    connect(nextButton, SIGNAL(clicked()), this, SLOT(slotNextClick()));

    type = VT_MONTH;
  }

  QString filler;

  int pmonth = mn - 1;
  if ( pmonth < 0 ) pmonth = 11;
  prevButton->setText((QString)months[pmonth] + filler.fill(' ', 10 - strlen(months[pmonth])));

  month->setCurrentItem(mn);

  year->setValue(yr);

  int nmonth = mn + 1;
  if ( nmonth > 11 ) nmonth = 0;
  nextButton->setText(filler.fill(' ', 10 - strlen(months[nmonth])) + (QString)months[nmonth]);

  KBudgetDoc* doc = getDocument();
  for ( int i = 0; i < 3; i++ ) {
    accounts[i] = doc->getAccounts((Account::types)i, QDate(yr, mn + 1, 1));
    for ( unsigned int j = 0; j < accounts[i].size(); j++ )
      accounts[i][j].dump();
  }
  createSummary();

  editor->selectMonth(QDate(yr,mn+1,1));
}

void KBudgetView::rebuildSummary()
{
	createSummary();
}

KBudgetDoc *KBudgetView::getDocument() const
{
  KBudgetApp *theApp=(KBudgetApp *) parentWidget();

  return theApp->getDocument();
}

void KBudgetView::print(QPrinter *pPrinter)
{
  QPainter printpainter;
  printpainter.begin(pPrinter);

  // TODO: add your printing code here

  printpainter.end();
}

int KBudgetView::currentAccount() const
{
  return selAccount;
}

void KBudgetView::slotPrevClick()
{
	if ( type == VT_MONTH ) {
		QDate newdate = from.addMonths(-1);
		selectMonth(newdate.year(), newdate.month() - 1);
	}
}

void KBudgetView::slotMonthActivate(int month)
{
	if ( type == VT_MONTH )
		selectMonth(from.year(), month);
}

void KBudgetView::slotYearChange(int year)
{
	if ( type == VT_MONTH )
		selectMonth(year, from.month() - 1);
}

void KBudgetView::slotNextClick()
{
	if ( type == VT_MONTH ) {
		QDate newdate = from.addMonths(1);
		selectMonth(newdate.year(), newdate.month() - 1);
	}
}

void KBudgetView::slotAccountAddDel()
{
  if ( type == VT_MONTH )
    selectMonth(from.year(), from.month() - 1);
}

void KBudgetView::slotAccountChanged(int it)
{
  slotAccountAddDel();
}

void KBudgetView::slotSelectionChanged()
{
  int account = accountAt(dtab->selection(dtab->currentSelection()).topRow());
  switch ( account ) {
    case -2:
      break;
    case -1:
      editor->hide();
      emit accountSelectionStatus(false);
      break;
    default:
      selAccount = accounts[account & 3][account >> 2].id();
      if ( (account & 3) < 2 ) {
        editor->selectAccount(selAccount);
        editor->show();
      }
      emit accountSelectionStatus(true);
      break;
  }
}

void KBudgetView::slotRenameAccount()
{
  int curRow = dtab->selection(dtab->currentSelection()).topRow();
  if ( accountAt(curRow) >= 0 )
    dtab->editCell(curRow, 0);
}

void KBudgetView::slotContextMenuRequested(int row, int col, const QPoint & pos)
{
  emit contextMenuRequested(pos);
}

void KBudgetView::slotValueChanged(int row, int col)
{
  int account = accountAt(row);
  if ( account >= 0 ) {
    int id = accounts[account&3][account>>2].id();
    getDocument()->getAccount(id).setName(dtab->text(row,col));
    getDocument()->setModified(true);
  }
}

void KBudgetView::slotTransactions()
{
  int curRow = dtab->selection(dtab->currentSelection()).topRow();
  int account = accountAt(curRow);
  if ( account >= 0 ) {
    KTransactionWindow* w = new KTransactionWindow(getDocument(), from, accounts[account&3][account>>2].id());
    w->show();
  }
}

void KBudgetView::slotEditAccount()
{
  int curRow = dtab->selection(dtab->currentSelection()).topRow();
  int account = accountAt(curRow);
  if ( account >= 0 )
    (new KNewAccount((KMainWindow*)parent(), getDocument(), accounts[account&3][account>>2].id()))->show();
}

int KBudgetView::accountAt(int row)
{
  if ( row < 0 )
    return -2;

  int r = 3;
  for ( int i = 0; i < 3; i++ ) {
    if ( r <= row && row - r < (int)accounts[i].size() ) {
      return ((row - r) << 2) + i;
    }
    r += accounts[i].size() + 4;
  }
  return -1;
}

void KBudgetView::cleanupPeriodDisplay()
{
	switch ( type ) {
	case VT_MONTH:
		delete prevButton;
		delete spacer1;
		delete month;
		delete year;
		delete spacer2;
		delete nextButton;
		break;
	case VT_YEAR:
		break;
	case VT_NONE:
		break;
	}
	type = VT_NONE;
}

void KBudgetView::createSummary()
{
  dtab->setNumRows(accounts[0].size()+accounts[1].size()+accounts[2].size()+12);
  for ( int i = 1; i < 4; i++ )
    dtab->setColumnReadOnly(i, true);
  dtab->setRowReadOnly(0, true);
  
  int row = 1;
  addSummaryHeader(row, "Income");
  dtab->setRowReadOnly(row, true);
  row++;
  addSummaryRow(row, "Account", "Budgeted", "Received", "Remaining");
  dtab->setRowReadOnly(row, true);
  row++;
  createSection(0, row);
  for ( int i = 0; i < COLUMNS; i++ )
    dtab->clearCell(row, i);
  dtab->setRowReadOnly(row, true);
  row++;

  addSummaryHeader(row, "Expenses");
  dtab->setRowReadOnly(row, true);
  row++;
  addSummaryRow(row, "Account", "Budgeted", "Spent", "Remaining");
  dtab->setRowReadOnly(row, true);
  row++;
  createSection(1, row);
  for ( int i = 0; i < COLUMNS; i++ )
    dtab->clearCell(row, i);
  dtab->setRowReadOnly(row, true);
  row++;

  addSummaryHeader(row, "Accounts");
  dtab->setRowReadOnly(row, true);
  row++;
  addSummaryRow(row, "Account", "Start", "Change", "End");
  dtab->setRowReadOnly(row, true);
  row++;  
  createSection(2, row);

  slotSelectionChanged();
}

void KBudgetView::addSummaryHeader(int row, QString text/*, QColor colour*/)
{
//  QLabel* label = new QLabel(text, vb);
//  label->setPaletteForegroundColor(colour);
//  QFont font = label->font();
//  font.setBold(true);
//  label->setFont(font);
//  label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
//  label->setBackgroundMode(Qt::PaletteBase);
//  dataGrid->addMultiCellWidget(label, row, row, 1, 4);
  dtab->setText(row, 0, text);
  dtab->item(row, 0)->setSpan(1,4);
}

void KBudgetView::addSummaryRow(int row, QString text1, QString text2, QString text3, QString text4/*, QColor colour*/)
{
//  QLabel* label;
  QString* text[4] = {&text1, &text2, &text3, &text4};
  for ( int i = 0; i < COLUMNS; i++ ) {
//    label = new QLabel(*text[i], 0);
//    label->setAlignment(Qt::AlignRight);
//    label->setPaletteForegroundColor(colour);
//    label->setBackgroundMode(Qt::PaletteBase);
//    dataGrid->addWidget(label, row, i + 1);
    dtab->setText(row, i, *text[i]);
//    dtab->setCellWidget(row, i, label);
    if ( i == 0 )
      dtab->item(row, 0)->setSpan(1,1);
  }
  dtab->setRowReadOnly(row, false);
}

void KBudgetView::createSection(int number, int &row)
{
  float sumStart = 0.0f, sumChange = 0.0f, sumEnd = 0.0f;
  float start, change, end;
  for ( unsigned int i = 0; i < accounts[number].size(); i++ ) {
    QString name = accounts[number][i].name();
    if ( number != 2 ) {
      start = accounts[number][i].startingBudget() + accounts[number][i].budgeted();
      change = accounts[number][i].change();
      end = accounts[number][i].budget();
    } else {
      start = accounts[number][i].startingBalance();
      end = accounts[number][i].getBalance();
      change = end - start;
    }
    sumStart += start;
    sumEnd += end;
    sumChange += change;
    addSummaryRow(row, name, l10n.formatMoney(start), l10n.formatMoney(change), l10n.formatMoney(end));
    row++;
  }
  addSummaryRow(row, "Total", l10n.formatMoney(sumStart), l10n.formatMoney(sumChange), l10n.formatMoney(sumEnd));
  dtab->setRowReadOnly(row, true);
  row++;
}
