// -*- Mode: C++; -*-
#ifndef _smart_pointer_h
#define _smart_pointer_h

#if defined WIN32
#include <windows.h>
#elif defined  _USE_PTHREAD
#include "cmutex.h"
#endif

#include <string>
#include "kmysqladmin/helpers/nullpointer_exception.h"
/*!
 * \file smart_pointer.h
 * \brief smart pointer and reference counter
 * \author Rajko Albrecht
 *
 */

//! simple reference counter class
class ref_count {
protected:
    //! reference count member
    long m_RefCount;
#ifdef _USE_PTHREAD
    CMutex m_RefcountMutex;
#endif
public:
    //! first reference must be added after "new" via Pointer()
    ref_count() : m_RefCount(0)
#ifdef _USE_PTHREAD
                  ,m_RefcountMutex()
#endif
    {}
    virtual ~ref_count() {}
    //! add a reference
#ifdef WIN32
    void Incr() {InterlockedIncrement(& m_RefCount); }
#else
    void Incr() {
#ifdef _USE_PTHREAD
        CAutoMutex a(&m_RefcountMutex);
#endif
        ++m_RefCount;
    }
#endif
    //! delete a reference
#ifdef WIN32
    void Decr() { InterlockedDecrement(& m_RefCount); }
#else
    void Decr() {
#ifdef _USE_PTHREAD
        CAutoMutex a(&m_RefcountMutex);
#endif
        --m_RefCount;
    }
#endif
    //! is it referenced
    bool Shared() { return (m_RefCount > 0); }
};

//! reference counting wrapper class
template<class T> class smart_pointer {
    //! pointer to object
    /*!
     * this object must contain Incr(), Decr() and Shared() 
     * methode as public members. The best way is, that it will be a child
     * class of RefCount
     */
    T *ptr;
public:
    //! standart constructor
    smart_pointer() { ptr = NULL; }
    //! standart destructor
    /*!
     * release the reference, if it were the last reference, destroys
     * ptr
     */
    ~smart_pointer()
    {
        if (ptr){
            ptr->Decr();
            if (!ptr->Shared())
                delete ptr;
        }
    }
    //! construction
    smart_pointer(T* t) { if (ptr = t) ptr->Incr(); }
    //! Pointer copy
    smart_pointer(const smart_pointer<T>& p) 
    { if (ptr = p.ptr) ptr->Incr(); }
    //! pointer copy by assignment
    smart_pointer<T>& operator= (const smart_pointer<T>& p) 
    {
        // already same: nothing to do
        if (ptr == p.ptr) return *this;
        // decouple reference
        if (ptr) { ptr->Decr(); if (!ptr->Shared()) delete ptr; } 
        // establish new reference
        if (ptr = p.ptr) ptr->Incr();
        return *this;
    }
    smart_pointer<T>& operator= (T*p)
    {
        if (ptr==p)return *this;
        if (ptr) {
            ptr->Decr();
            if (!ptr->Shared()) delete ptr;
        }
        if (ptr=p) ptr->Incr();
        return *this;
    }

    //! cast to conventional pointer
    operator T* () { return ptr; }
    //! deref: fails for NULL pointer
    T& operator* () { if (ptr == NULL) throw nullpointer_exception("Bad smartpointer operator *"); return *ptr; }
    //! deref with method call
    T* operator-> () { if (ptr == NULL) throw nullpointer_exception("Bad smartpointer operator ->"); return ptr; }
    //! deref with const method call
    T* operator-> ()const { if (ptr == NULL) throw nullpointer_exception("Bad smartpointer operator ->"); return ptr; }
    //! supports "if (pointer)"
    operator bool () const { return (ptr != NULL); }
    //! same as non const
    operator bool () { return ptr != NULL;}
    //! support if (!pointer)"
    bool operator! () const { return (ptr == NULL); }
    //! support if (!pointer)" as non const
    bool operator! () { return (ptr == NULL); }
};

//! A pre-defined class for a smart string
/*!
 * This class should help to avoid copy operations if more than one
 * classes are accessing the same string
 */
class SharedString:public ref_count
{
protected:
    //! the string itself
    std::string m_String;
public:
    //! constructor
    SharedString():ref_count(){};
    //! copy constructor
    SharedString(const SharedString&old):ref_count(){m_String = old.m_String;}
    //! copy constructor
    SharedString(const char*old):ref_count(){m_String = old?old:"";}
    //! copy constructor
    SharedString(const std::string&old):ref_count(){m_String = old;}
    //! destructor
    virtual ~SharedString(){}
    //! returns a reference to the string
    operator std::string&(){return m_String;}
    //! returns the value of the string
    operator const std::string()const{return m_String;}
};

#endif
