/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <stdlib.h>

#include <qglobal.h>
#include <qscrollbar.h>
#include <qlayout.h>
#include <qdrawutil.h>
#include <qbrush.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kaction.h>
#include <kshortcut.h>

#include "mlofile.h"
#include "printdlg.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "senderaliases.h"
#include "global.h"
#include "kmlofax.h"
#include "preview.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>


#define FAX_BAR_WIDTH 92


static ushort count_bits(ulong value)
{
	int i;
	ushort result = 32;

	for (i=0; i < 32; i++) {
		if (value & 0x1)
			result--;
		value >>= 1;
	}

	return result;
}


static char *shrinkPage(int w, int h, char *data)
{
	static const uchar shrink_tab[128] = {
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f
	};
	char *ndata;
	uchar *line, *nline;
	int nw, nh, i, j, k, bpl, nbpl;

	nw = w >> 1;
	nh = h >> 1;

	bpl = (((w-1) >> 4) + 1) << 1;
	nbpl = (((nw-1) >> 4) + 1) << 1;
	ndata = (char *)malloc(nbpl * nh);

	line = (uchar *)data;
	nline = (uchar *)ndata;

	for (i=0; i < nh; i++) {
		if (i & 1)
			line += bpl;
		for (j=0; j < nbpl; j++) {
			k = j << 1;
			if (j & 1)
				nline[j] = shrink_tab[line[k] >> 1] | (shrink_tab[line[k|1] >> 1] << 4);
			else
				nline[j] = shrink_tab[line[k] & 0x7f] | (shrink_tab[line[k|1] & 0x7f] << 4);
		}
		line += bpl;
		if (!(i & 1))
			line += bpl;
		nline += nbpl;
	}

	return ndata;
}


static char *rotatePage(int w, int h, char *data, bool lsb_first)
{
	char *ndata;
	uchar *line, *nline, *nline2, buff[8];
	int nw, nh, i, j, k, l, max_lines, bpl, nbpl;

	nw = h;
	nh = w;
	
	bpl = (((w-1) >> 4) + 1) << 1;
	nbpl = (((nw-1) >> 4) + 1) << 1;
	ndata = (char *)malloc(nbpl * nh);
	
	for (i = 0; i < w; i += 8) { // bpl
		line = (uchar *)(data + bpl * h + (i >> 3));
		nline = (uchar *)(ndata + nbpl * i);
		max_lines = nh - i;
		if (max_lines > 8)
			max_lines = 8;

		for (j = 0; j < nw; j += 8) { // nbpl
			nline2 = nline;
			for (k = 0; k < max_lines; k++) {
				line -= bpl;
				buff[k] = *line;
			}
			for (k = 0; k < max_lines; k++) {
				if (lsb_first) {
					for (l = 0; l < 8; l++) {
						*nline2 = (uchar)((*nline2 << 1) | (buff[l] & 1));
						buff[l] >>= 1;
					}
				} else {
					for (l = 0; l < 8; l++) {
						*nline2 = (uchar)((*nline2 >> 1) | (buff[l] & 128));
						buff[l] <<= 1;
					}
				}
				*nline2 = xchg_endian[*nline2];
				nline2 += nbpl;
			}
			nline++;
		}
	}

	return ndata;
}


FaxScrollBox::FaxScrollBox(QWidget *parent, const char *name): QFrame(parent, name)
{
	setFrameStyle(Panel | Sunken);
}


FaxScrollBox::~FaxScrollBox()
{
}


void FaxScrollBox::setPageSize(QSize size)
{
	pagesize = size;

	if (newGeometry()) {
		setBackgroundMode();
		update();
	}
}


void FaxScrollBox::setViewSize(QSize size)
{
	viewsize = size;

	if (newGeometry()) {
		setBackgroundMode();
		update();
	}
}


void FaxScrollBox::setViewPos(QPoint pos)
{
	viewpos = pos;

	if (newGeometry())
		update();
}


void FaxScrollBox::mousePressEvent(QMouseEvent *e)
{
	if (e->button() & LeftButton)
		mousepos = e->pos();
}


void FaxScrollBox::mouseMoveEvent(QMouseEvent *e)
{
	if (e->state() & LeftButton) {
		int dx, dy;

		dx = (e->pos().x() - mousepos.x()) * pagesize.width() / width();
		dy = (e->pos().y() - mousepos.y()) * pagesize.height() / height();

		emit move(dx ,dy);

		mousepos = e->pos();
	}
}


void FaxScrollBox::drawContents(QPainter *p)
{
	QBrush b(colorGroup().background());
	qDrawShadePanel(p, m_pos.x(), m_pos.y(), m_size.width(), m_size.height(), colorGroup(), false, 1, &b);
}


void FaxScrollBox::setBackgroundMode()
{
	if (pagesize.isNull() || (viewsize.rwidth() >= pagesize.rwidth() && viewsize.rheight() >= pagesize.rheight()))
		QFrame::setBackgroundMode(PaletteBackground);
	else
		QFrame::setBackgroundMode(PaletteMid);
}


bool FaxScrollBox::newGeometry()
{
	QPoint newpos;
	QSize newsize;

	if (pagesize.isEmpty())
		return false;

	QRect c(contentsRect());

	if (pagesize.width() > viewsize.width()) {
		newsize.setWidth(c.width() * viewsize.width() / pagesize.width());
		newpos.setX(c.x() + (c.width() - newsize.width()) * viewpos.x() / (pagesize.width() - viewsize.width()));
	}	else {
		newsize.setWidth(c.width());
		newpos.setX(c.x());
	}
		
	if (pagesize.height() > viewsize.height()) {
		newsize.setHeight(c.height() * viewsize.height() / pagesize.height());
		newpos.setY(c.y() + (c.height() - newsize.height()) * viewpos.y() / (pagesize.height() - viewsize.height()));
	}	else {
		newsize.setHeight(c.height());
		newpos.setY(c.y());
	}

	if ((newpos == m_pos) && (newsize == m_size))
		return false;

	m_pos = newpos;
	m_size = newsize;

	return true;
}


FaxIconListItem::FaxIconListItem(QListBox *listbox, const QPixmap &pixmap, int page): QListBoxItem(listbox)
{
	m_pixmap = pixmap;
	m_page = page;
	setText(i18n("Page %1").arg(page));
}


FaxIconListItem::~FaxIconListItem()
{
}


int FaxIconListItem::height(const QListBox *lb) const
{
	return m_pixmap.height() + lb->fontMetrics().lineSpacing() + 10;
}


void FaxIconListItem::paint(QPainter *painter)
{
	QFontMetrics fm = painter->fontMetrics();
	painter->drawPixmap((FAX_BAR_WIDTH - m_pixmap.width()) / 2 - 8, 5, m_pixmap);
	painter->drawText(0, m_pixmap.height() + 7, FAX_BAR_WIDTH - 16, fm.lineSpacing(), Qt::AlignCenter, text());
}



FaxIconList::FaxIconList(QWidget *parent, const char *name): KListBox(parent, name)
{
	setFixedWidth(FAX_BAR_WIDTH);
//	setBackgroundColor(gray);
	setFocusPolicy(NoFocus);
}


FaxIconList::~FaxIconList()
{
}


void FaxIconList::showFax(const QString &name, bool lsb_first)
{
	TiffFile file;
	int page, width, height, bpl;
	bool fine;
	int iwidth, iwidth2, iheight, icol;
	int i, line, stripheight, stripline, maxstripheight, countshift;
	ulong *buff;
	ushort *grays;
	uchar *ibits;
	QImage image;
	QPixmap pixmap;
	FaxIconListItem *li;

	clear();

	file.setName(expandPath(name));
	if (!file.open(IO_ReadOnly))
		return;

	kapp->setOverrideCursor(waitCursor);
	
	for (page = 1; page <= file.pages(); page++) {
		if (!file.gotoPage(page)) {
			file.close();
			return;
		}

		width = file.pageWidth();
		height = file.pageHeight();
		bpl = (((width-1) >> 5) + 1) << 2;
		buff = (ulong *)malloc(bpl);
		memset((void *)buff, 0, bpl);
		fine = file.fineResolution();

		iwidth = ((width-1) >> 5) + 3;
		iwidth2 = iwidth-2;
		grays = (ushort *)malloc(iwidth2 * 2);
		iheight = fine ? ((height-1) >> 5) + 3 : ((height-1) >> 4) + 3;
		
		maxstripheight = fine ? 32 : 16;
		countshift = fine ? 3 : 2;
	
		image.create(iwidth, iheight, 8, 129);
		for (i=0; i < 128; i++)
			image.setColor(i, qRgb(i << 1, i << 1, i << 1));
		image.setColor(128, qRgb(255, 255, 255));

		ibits = image.bits();
		memset((void *)ibits, 0, iwidth * iheight);

		ibits += iwidth + 1;
		line = 0;
		while (line < height) {
			if ((stripheight = (height - line)) > maxstripheight)
				stripheight = maxstripheight;
			else {
				countshift = 0;
				if (stripheight > 4)
					countshift++;
				if (stripheight > 8)
					countshift++;
				if (stripheight > 16)
					countshift++;
			}
			memset((void *)grays, 0, iwidth2 * 2);
			for (stripline = 0; stripline < stripheight; stripline++) {
				file.readImgLine((uchar *)buff, lsb_first);
				for (icol=0; icol < iwidth2; icol++)
					grays[icol] += count_bits(buff[icol]);
			}
			for (icol=0; icol < iwidth2; icol++)
				ibits[icol] = (uchar)(grays[icol] >> countshift);
			ibits += iwidth;
			line += stripheight;
		}
	
		free(grays);
		free(buff);

		pixmap.convertFromImage(image);
		image.reset();

		li = new FaxIconListItem(this, pixmap, page);
	}

	file.close();

	kapp->restoreOverrideCursor();

	setMinimumHeight(sizeHint().height());
}


void FaxIconList::selectPage(int page)
{
	setCurrentItem(page-1);
}



FaxWidget::FaxWidget(QWidget *parent, const char *name): QWidget(parent, name)
{
	int screen;

	sizex = 0;
	sizey = 0;
	scrollx = 0;
	scrolly = 0;

	initKeyAccel();

	display = x11Display();
	screen = x11Screen();

	lsb_first = (BitmapBitOrder(display) == LSBFirst);
	
	window = XCreateSimpleWindow(display, winId(), 0, 0, 1, 1, 0,
		 BlackPixel(display, screen),
		 WhitePixel(display, screen));
	gc = XCreateGC(display, window, 0L, (XGCValues *)0);
	XSetForeground(display, gc, BlackPixel(display, screen));
	XSetBackground(display, gc, WhitePixel(display, screen));
	XSetFunction(display, gc, GXcopy);
	XSelectInput(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask);
	XFlush(display);

	move_cursor = XCreateFontCursor(display, XC_fleur);
	moving = false;
	move_x = 0;
	move_y = 0;

	image = 0;
	im_width = 0;
	im_height = 0;

	hscroll = new QScrollBar(QScrollBar::Horizontal, this, "prehscroll");
	Q_CHECK_PTR(hscroll);
	hscroll->setFixedHeight(hscroll->sizeHint().height());
	hscroll->hide();
	vscroll = new QScrollBar(QScrollBar::Vertical, this, "prevscroll");
	Q_CHECK_PTR(vscroll);
	vscroll->setFixedWidth(vscroll->sizeHint().width());
	vscroll->hide();

	connect(hscroll, SIGNAL(valueChanged(int)), this, SLOT(setXScroll(int)));
	connect(vscroll, SIGNAL(valueChanged(int)), this, SLOT(setYScroll(int)));
}


FaxWidget::~FaxWidget()
{
	deleteImage();
	XFreeCursor(display, move_cursor);
	XFreeGC(display, gc);
	XDestroyWindow(display, window);
}


void FaxWidget::setImage(int w, int h, char *data)
{
	deleteImage();

	im_width = w;
	im_height = h;

	if ((w) && (h)) {
		image = XCreateImage(display, DefaultVisual(display, x11Screen()),
			1, XYBitmap, 0, data, im_width, im_height, 16, 0);
		XMapRaised(display, window);
	}
	else {
		XUnmapWindow(display, window);
		XResizeWindow(display, window, 1, 1);
	}
	updateGeometry();
	update();
}


void FaxWidget::deleteImage()
{
	if (image) {
		image->data = 0;
		XDestroyImage(image);
		image = 0;
	}
}


void FaxWidget::updateGeometry()
{
	int range;
	bool need_hscroll, need_vscroll;

	sizex = width();
	sizey = height();

	if ((need_hscroll = (im_width > sizex)))
		sizey -= hscroll->height();
	if ((need_vscroll = (im_height > sizey))) {
		sizex -= vscroll->width();
		if (!need_hscroll && (im_width > sizex)) {
			need_hscroll = true;
			sizey -= hscroll->height();
		}
	}
	
	if (need_hscroll) {
		range = im_width - sizex;
		hscroll->setRange(0, range);
		if (hscroll->value() > range)
			hscroll->setValue(range);
		hscroll->setSteps(12, sizex);
		hscroll->setGeometry(0, sizey, sizex, hscroll->height());
		hscroll->show();
	} else {
		hscroll->hide();
		hscroll->setRange(0, 0);
		hscroll->setValue(0);
	}
	
	if (need_vscroll) {
		range = im_height - sizey;
		vscroll->setRange(0, range);
		if (vscroll->value() > range)
			vscroll->setValue(range);
		vscroll->setSteps(12, sizey);
		vscroll->setGeometry(sizex, 0, vscroll->width(), sizey);
		vscroll->show();
	} else {
		vscroll->hide();
		vscroll->setRange(0, 0);
		vscroll->setValue(0);
	}

	emit resized(QSize(sizex, sizey));
}


bool FaxWidget::x11EventFilter(XEvent *ev)
{
	bool refresh = false;
	bool move_it = false;
	int mv_x = 0, mv_y = 0, wheel_steps = 0, wheel_state = 0;
	int x1 = INT_MAX, y1 = INT_MAX, x2 = 0, y2 = 0, xb, yb;

	if (ev->xany.window != window)
		return false;
	
	do {
		switch(ev->type) {
			case ButtonPress:
				if (ev->xbutton.button == Button1) {
					XDefineCursor(display, window, move_cursor);
					moving = true;
					move_x = ev->xbutton.x;
					move_y = ev->xbutton.y;
				}
				if ((ev->xbutton.button == Button4) || (ev->xbutton.button == Button5)) {
					if (ev->xbutton.state & Button1Mask)
						wheel_state |= Qt::LeftButton;
					if (ev->xbutton.state & Button2Mask)
						wheel_state |= Qt::MidButton;
					if (ev->xbutton.state & Button3Mask)
						wheel_state |= Qt::RightButton;
					if (ev->xbutton.state & ShiftMask)
						wheel_state |= Qt::ShiftButton;
					if (ev->xbutton.state & ControlMask)
						wheel_state |= Qt::ControlButton;
					if (ev->xbutton.state & Mod1Mask)
						wheel_state |= Qt::AltButton;
					if (ev->xbutton.button == Button4)
						wheel_steps++;
					else
						wheel_steps--;
				}
				break;
			case ButtonRelease:
				if (ev->xbutton.button == Button1) {
					XUndefineCursor(display, window);
					moving = false;
				}
				break;
			case MotionNotify:
				if (moving) {
					move_it = true;
					mv_x = ev->xmotion.x;
					mv_y = ev->xmotion.y;
				}
				break;
			case Expose:
				refresh = true;
				xb = ev->xexpose.x + ev->xexpose.width;
				yb = ev->xexpose.y + ev->xexpose.height;
				if (ev->xexpose.x < x1)
					x1 = ev->xexpose.x;
				if (ev->xexpose.y < y1)
					y1 = ev->xexpose.y;
				if (xb > x2)
					x2 = xb;
				if (yb > y2)
					y2 = yb;
			default:
				break;
		}
	} while (XCheckWindowEvent(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask, ev));

	if (refresh && image)
		XPutImage(display, window, gc, image, scrollx + x1, scrolly + y1, x1, y1, x2-x1, y2-y1);

	if (move_it) {
		move(move_x - mv_x, move_y - mv_y);
		move_x = mv_x;
		move_y = mv_y;
	}

	if (wheel_steps) {
		QWheelEvent e(QPoint(0, 0), 120 * wheel_steps, wheel_state);
		KApplication::sendEvent(vscroll, &e);
	}

	return true;
}	


void FaxWidget::setXScroll(int x)
{
	scrollx = x;
	if (scrollx + sizex > im_width)
		scrollx = im_width - sizex;
	if (scrollx < 0)
		scrollx = 0;
	update();
}


void FaxWidget::setYScroll(int y)
{
	scrolly = y;
	if (scrolly + sizey > im_height)
		scrolly = im_height - sizey;
	if (scrolly < 0)
		scrolly = 0;
	update();
}


void FaxWidget::move(int dx, int dy)
{
	hscroll->setValue(hscroll->value() + dx);
	vscroll->setValue(vscroll->value() + dy);
}


void FaxWidget::keyLeft()
{
	hscroll->setValue(hscroll->value() - hscroll->lineStep());
}


void FaxWidget::keyRight()
{
	hscroll->setValue(hscroll->value() + hscroll->lineStep());
}


void FaxWidget::keyUp()
{
	vscroll->setValue(vscroll->value() - vscroll->lineStep());
}


void FaxWidget::keyDown()
{
	vscroll->setValue(vscroll->value() + vscroll->lineStep());
}


void FaxWidget::keyPageUp()
{
	vscroll->setValue(vscroll->value() - vscroll->pageStep());
}


void FaxWidget::keyPageDown()
{
	vscroll->setValue(vscroll->value() + vscroll->pageStep());
}


void FaxWidget::keyHome()
{
	vscroll->setValue(vscroll->minValue());
}


void FaxWidget::keyEnd()
{
	vscroll->setValue(vscroll->maxValue());
}


void FaxWidget::initKeyAccel()
{
  keyAccel = new KAccel(this);
	Q_CHECK_PTR(keyAccel);

	keyAccel->insert("ScrollLeft", i18n("Scroll Left"), 0, Key_Left, this, SLOT(keyLeft()));
	keyAccel->insert("ScrollRight", i18n("Scroll Right"), 0, Key_Right, this, SLOT(keyRight()));
	keyAccel->insert("ScrollUp", i18n("Scroll Up"), 0, Key_Up, this, SLOT(keyUp()));
	keyAccel->insert("ScrollDown", i18n("Scroll Down"), 0, Key_Down, this, SLOT(keyDown()));
	keyAccel->insert("PageUp", i18n("Page Up"), 0, Key_PageUp, this, SLOT(keyPageUp()));
	keyAccel->insert("PageDown", i18n("Page Down"), 0, Key_PageDown, this, SLOT(keyPageDown()));
	keyAccel->insert(KStdAccel::Home, this, SLOT(keyHome()));
	keyAccel->insert(KStdAccel::End, this, SLOT(keyEnd()));

	keyAccel->readSettings();
}


void FaxWidget::paintEvent(QPaintEvent *)
{
	int w, h, x = 0, y = 0;

	w = width();
	if (vscroll->isVisible())
		w -= vscroll->width();

	h = height();
	if (hscroll->isVisible())
		h -= hscroll->height();
	
	if (w > im_width) {
		x = (w - im_width) >> 1;
		w = im_width;
	}
	if (h > im_height) {
		y = (h - im_height) >> 1;
		h = im_height;
	}
	if (image) {
		XMoveResizeWindow(display, window, x, y, w, h);
		XPutImage(display, window, gc, image, scrollx, scrolly, 0, 0, w, h);
		emit moved(QPoint(scrollx, scrolly));
	}
}


void FaxWidget::resizeEvent(QResizeEvent *)
{
	updateGeometry();
}


Preview::Preview(QWidget *parent, const char *name): KMainWindow(parent, name)
{
	int i;
	QBoxLayout *hbox, *vbox;
	QFrame *frame;

	config = kapp->config();

	for (i=0; i < 3; i++)
		page_datas[i] = 0;
	page = pages = 0;
	page_width = page_height = 0;
	page_fine = false;
	zoom_width = zoom_height = 0;
	zoom = 1;
	angle = 0;

	initActions();
	initMenuBar();
	initToolBar();
	initStatusBar();

	view = new QWidget(this, "faxview");
	Q_CHECK_PTR(view);
	setCentralWidget(view);

	hbox = new QHBoxLayout(view);
	
	vbox = new QVBoxLayout();
	hbox->addLayout(vbox);

	fax_scroll = new FaxScrollBox(view, "faxscrollbox");
	Q_CHECK_PTR(fax_scroll);
	fax_scroll->setFixedSize(FAX_BAR_WIDTH, FAX_BAR_WIDTH);
	vbox->addWidget(fax_scroll);
	
	fax_icons = new FaxIconList(view, "faxiconlist");
	Q_CHECK_PTR(fax_icons);
	vbox->addWidget(fax_icons);

	frame = new QFrame(view);
	Q_CHECK_PTR(frame);
	frame->setFrameStyle(QFrame::Panel | QFrame::Raised);
	frame->setLineWidth(1);
	frame->setFixedWidth(3);
	hbox->addWidget(frame);

	fax_widget = new FaxWidget(view, "faxwidget");
	Q_CHECK_PTR(fax_widget);
	fax_widget->setMinimumSize(100, 75);
	hbox->addWidget(fax_widget, 1);
	
	hbox->activate();

	readOptions();

	connect(fax_scroll, SIGNAL(move(int, int)), fax_widget, SLOT(move(int ,int)));
	connect(fax_icons, SIGNAL(highlighted(int)), SLOT(pageSelected(int)));
	connect(fax_widget, SIGNAL(resized(QSize)), fax_scroll, SLOT(setViewSize(QSize)));
	connect(fax_widget, SIGNAL(moved(QPoint)), fax_scroll, SLOT(setViewPos(QPoint)));

	((KMLOFaxApp *)kapp)->addPreview(this);
}


Preview::~Preview()
{
	deletePages();
	((KMLOFaxApp *)kapp)->removePreview(this);
}


void Preview::showFax(const QString &name)
{
	TiffFile file;
	QString s;

	m_name = name;

	file.setName(expandPath(m_name));
	if (!file.open(IO_ReadOnly)) {
		KMessageBox::sorry(this, i18n("Cannot open facsimile file."), i18n("File Error"));
		return;
	}
	pages = file.pages();
	file.gotoPage(1);
	s = SENDER_ALIAS(file.sender());
	slotStatusSender(s);
	setCaption(s.isEmpty() ? i18n("Facsimile") : i18n("Facsimile from %1").arg(s));
	slotStatusTime(file.time());
	file.close();

	fax_icons->showFax(m_name, fax_widget->lsbFirst());

//	This should also load page 1, but sometimes causes a segfault
//	fax_icons->selectPage(1);

//	Instead use this workaround	
	loadPage(1);
	disconnect(fax_icons, SIGNAL(highlighted(int)), this, SLOT(pageSelected(int)));
	fax_icons->selectPage(1);
	connect(fax_icons, SIGNAL(highlighted(int)), SLOT(pageSelected(int)));
//	End of workaround

	if (isVisible())
		raise();
	else
		show();
}


void Preview::readOptions()
{
	config->setGroup("Preview Options");

	settingsShowToolbar->setChecked(config->readBoolEntry("Show ToolBar", true));
	slotSettingsShowToolbar();
	settingsShowStatusbar->setChecked(config->readBoolEntry("Show StatusBar", true));
	slotSettingsShowStatusbar();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));

	QSize geoSize(650, 400);
	resize(config->readSizeEntry("Geometry", &geoSize));
}


void Preview::saveOptions()
{
	config->setGroup("Preview Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());
}


void Preview::initActions()
{
	filePrint = KStdAction::print(this, SLOT(slotFilePrint()), actionCollection());
	fileExport = new KAction(i18n("&Export..."), "filesave", KShortcut(CTRL+Key_E), this, SLOT(slotFileExport()), actionCollection(), "file_export");
	fileMail = new KAction(i18n("&Mail..."), "mail_send", KShortcut(CTRL+Key_M), this, SLOT(slotFileMail()), actionCollection(), "file_mail");
	fileClose = KStdAction::close(this, SLOT(close()), actionCollection());

	viewZoomIn = KStdAction::zoomIn(this, SLOT(slotViewZoomIn()), actionCollection());
	viewZoomOut = KStdAction::zoomOut(this, SLOT(slotViewZoomOut()), actionCollection());
	viewRotate = new KAction(i18n("&Rotate"), "rotate", KShortcut(CTRL+Key_R), this, SLOT(slotViewRotate()), actionCollection(), "view_rotate");

	goPreviousPage = new KAction(i18n("&Previous Page"), "back", KShortcut(CTRL+Key_PageUp), this, SLOT(slotGoPreviousPage()), actionCollection(), "go_previous_page");
	goNextPage = new KAction(i18n("&Next Page"), "forward", KShortcut(CTRL+Key_PageDown), this, SLOT(slotGoNextPage()), actionCollection(), "go_next_page");
	goFirstPage = new KAction(i18n("&First Page"), "start", KShortcut(CTRL+Key_Home), this, SLOT(slotGoFirstPage()), actionCollection(), "go_first_page");
	goLastPage = new KAction(i18n("&Last Page"), "finish", KShortcut(CTRL+Key_End), this, SLOT(slotGoLastPage()), actionCollection(), "go_last_page");

	settingsShowToolbar = KStdAction::showToolbar(this, SLOT(slotSettingsShowToolbar()), actionCollection());
	settingsShowStatusbar = KStdAction::showStatusbar(this, SLOT(slotSettingsShowStatusbar()), actionCollection());
	settingsSaveOptions = new KAction(i18n("Save &Options"), 0, 0, this, SLOT(saveOptions()), actionCollection(), "settings_save_options");
}


void Preview::initMenuBar()
{
	QPopupMenu *fileMenu, *viewMenu, *goMenu, *settingsMenu;

	fileMenu = new QPopupMenu(0, "prefilemenu");
	Q_CHECK_PTR(fileMenu);
	filePrint->plug(fileMenu);
	fileExport->plug(fileMenu);
	fileMail->plug(fileMenu);
	fileMenu->insertSeparator();
	fileClose->plug(fileMenu);

	viewMenu = new QPopupMenu(0, "previewmenu");
	Q_CHECK_PTR(viewMenu);
	viewZoomIn->plug(viewMenu);
	viewZoomOut->plug(viewMenu);
	viewRotate->plug(viewMenu);

	goMenu = new QPopupMenu(0, "pregomenu");
	Q_CHECK_PTR(goMenu);
	goPreviousPage->plug(goMenu);
	goNextPage->plug(goMenu);
	goFirstPage->plug(goMenu);
	goLastPage->plug(goMenu);

	settingsMenu = new QPopupMenu(0, "presettingsmenu");
	Q_CHECK_PTR(settingsMenu);
	settingsShowToolbar->plug(settingsMenu);
	settingsShowStatusbar->plug(settingsMenu);
	settingsMenu->insertSeparator();
	settingsSaveOptions->plug(settingsMenu);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&View"), viewMenu);
	menuBar()->insertItem(i18n("&Go"), goMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
}


void Preview::initToolBar()
{
	filePrint->plug(toolBar());
	fileExport->plug(toolBar());
	fileMail->plug(toolBar());
	toolBar()->insertSeparator();
  viewZoomIn->plug(toolBar());
  viewZoomOut->plug(toolBar());
  viewRotate->plug(toolBar());
	toolBar()->insertSeparator();
  goPreviousPage->plug(toolBar());
  goNextPage->plug(toolBar());
  goFirstPage->plug(toolBar());
  goLastPage->plug(toolBar());
	toolBar()->insertSeparator();
  toolBar()->alignItemRight(fileClose->itemId(fileClose->plug(toolBar())));
}


void Preview::initStatusBar()
{

	statusBar()->insertFixedItem(KGlobal::locale()->formatDate(QDate::currentDate(), true) + " (" + KGlobal::locale()->formatTime(QTime::currentTime(), true) + ")  ", ID_PRE_STATUS_TIME);
	statusBar()->insertItem("", ID_PRE_STATUS_SENDER, 1);
	statusBar()->insertFixedItem(i18n(" Page 999 of 999 "), ID_PRE_STATUS_PAGE);
	statusBar()->insertFixedItem(QString::fromLatin1(" %1%2, %3 ").arg(KGlobal::locale()->formatNumber(9999.9, 0)).arg(KGlobal::locale()->formatNumber(99999.9, 0)).arg(i18n("Normal")), ID_PRE_STATUS_SIZE);
	statusBar()->insertFixedItem(" 100% ", ID_PRE_STATUS_ZOOM);
	statusBar()->insertFixedItem(" 000 ", ID_PRE_STATUS_ANGLE);
	statusBar()->changeItem("", ID_PRE_STATUS_TIME);
	statusBar()->changeItem("", ID_PRE_STATUS_PAGE);
	statusBar()->changeItem("", ID_PRE_STATUS_SIZE);
	statusBar()->changeItem("", ID_PRE_STATUS_ZOOM);
	statusBar()->changeItem("", ID_PRE_STATUS_ANGLE);
}


void Preview::slotFilePrint()
{
	PrintDlg printdlg(this, "printdlg");
	printdlg.printFax(m_name);
	printdlg.exec();
}


void Preview::slotFileExport()
{
	ExportDlg exportdlg(this, "exportdlg");
	exportdlg.exportFax(m_name);
	exportdlg.exec();
}


void Preview::slotFileMail()
{
	MailDlg maildlg(this, "maildlg");
	maildlg.mailFax(m_name);
	maildlg.exec();
}


void Preview::slotViewZoomIn()
{
	if (zoom) {
		zoom--;
		showPage();
	}
}


void Preview::slotViewZoomOut()
{
	if (zoom < 2) {
		zoom++;
		showPage();
	}
}


void Preview::slotViewRotate()
{
	char *new_data;
	int i, w, h;

	if (angle & 1) {
		w = page_height;
		h = page_width;
	} else {
		w = page_width;
		h = page_height;
	}
	
	kapp->setOverrideCursor(waitCursor);

	new_data = rotatePage(w, h, page_datas[0], fax_widget->lsbFirst());

	for (i=0; i < 3; i++)
		free(page_datas[i]);
	
	page_datas[0] = new_data;
	page_datas[1] = shrinkPage(h, w, page_datas[0]);
	page_datas[2] = shrinkPage(h >> 1, w >> 1, page_datas[1]);

	kapp->restoreOverrideCursor();

	angle = (angle + 1) & 3;
	slotStatusAngle();

	showPage();
}


void Preview::slotGoPreviousPage()
{
	if (page > 1)
		fax_icons->selectPage(page-1);
}


void Preview::slotGoNextPage()
{
	if (page < pages)
		fax_icons->selectPage(page+1);
}


void Preview::slotGoFirstPage()
{
	if (page > 1)
		fax_icons->selectPage(1);
}


void Preview::slotGoLastPage()
{
	if (page < pages)
		fax_icons->selectPage(pages);
}


void Preview::slotSettingsShowToolbar()
{
	if (settingsShowToolbar->isChecked())
		toolBar()->show();
	else
		toolBar()->hide();
}


void Preview::slotSettingsShowStatusbar()
{
	if (settingsShowStatusbar->isChecked())
		statusBar()->show();
	else
		statusBar()->hide();
}


void Preview::slotStatusTime(const QDateTime &time)
{
	statusBar()->changeItem(KGlobal::locale()->formatDate(time.date(), true) + " (" + KGlobal::locale()->formatTime(time.time(), true) + ")", ID_PRE_STATUS_TIME);
}


void Preview::slotStatusSender(const QString &text)
{
	statusBar()->changeItem(text, ID_PRE_STATUS_SENDER);
}


void Preview::slotStatusPage()
{
	if (page)
		statusBar()->changeItem(i18n("Page %1 of %2").arg(page).arg(pages), ID_PRE_STATUS_PAGE);
	else
		statusBar()->changeItem("", ID_PRE_STATUS_PAGE);
}


void Preview::slotStatusSize()
{
	if (page_width)
		statusBar()->changeItem(QString("%1%2, %3").arg(KGlobal::locale()->formatNumber((double)page_width, 0)).arg(KGlobal::locale()->formatNumber((double)page_height, 0)).arg(page_fine ? i18n("Fine") : i18n("Normal")), ID_PRE_STATUS_SIZE);
	else
		statusBar()->changeItem("", ID_PRE_STATUS_SIZE);
}


void Preview::slotStatusZoom()
{
	static const int sizes[3] = { 100, 50, 25 };

	statusBar()->changeItem(QString("%1%").arg(sizes[zoom]), ID_PRE_STATUS_ZOOM);
}


void Preview::slotStatusAngle()
{
	static const int angles[4] = { 0, 90, 180, 270 };

	statusBar()->changeItem(QString("%1").arg(angles[angle]), ID_PRE_STATUS_ANGLE);
}


void Preview::pageSelected(int index)
{
	loadPage(index + 1);
}


void Preview::deletePages()
{
	int i;

	page_width = page_height = 0;

	for (i=0; i < 3; i++)
		if (page_datas[i]) {
			free(page_datas[i]);
			page_datas[i] = 0;
		}
	fax_widget->deleteImage();
}


void Preview::showPage()
{
	viewZoomIn->setEnabled(zoom > 0);
	viewZoomOut->setEnabled(zoom < 2);
	
	zoom_width = page_width >> zoom;
	zoom_height = page_height >> zoom;

	if (angle & 1) {
		fax_scroll->setPageSize(QSize(zoom_height, zoom_width));
		fax_widget->setImage(zoom_height, zoom_width, page_datas[zoom]);
	}
	else {
		fax_scroll->setPageSize(QSize(zoom_width, zoom_height));
		fax_widget->setImage(zoom_width, zoom_height, page_datas[zoom]);
	}

	slotStatusZoom();
}


void Preview::loadPage(int p)
{
	TiffFile file;
	char *pd;
	int i, bpl;
	bool lsb_first = fax_widget->lsbFirst();

	page = 0;
	slotStatusPage();
	
	page_width = page_height = 0;
	page_fine = false;
	slotStatusSize();

	angle = 0;
	slotStatusAngle();

	deletePages();

	page = p;

	goPreviousPage->setEnabled(page != 1);
	goNextPage->setEnabled(page != pages);
	goFirstPage->setEnabled(page != 1);
	goLastPage->setEnabled(page != pages);

	file.setName(expandPath(m_name));
	if (!file.open(IO_ReadOnly)) {
		KMessageBox::sorry(this, i18n("Cannot open facsimile file."), i18n("File Error"));
		showPage();
		return;
	}
	if (!file.gotoPage(p)) {
		KMessageBox::sorry(this, i18n("Cannot find page in file."), i18n("File Error"));
		file.close();
		showPage();
		return;
	}

	page_width = file.pageWidth();
	page_height = file.pageHeight();
	page_fine = file.fineResolution();

	slotStatusSize();
	slotStatusPage();

	kapp->setOverrideCursor(waitCursor);

	bpl = (((page_width-1) >> 4) + 1) << 1;

	if (page_fine) {
		page_datas[0] = (char *)malloc(bpl * page_height);
		pd = page_datas[0];
		for (i=0; i < page_height; i++) {
			file.readImgLine((uchar *)pd, lsb_first);
			pd += bpl;
		}
	} else {
		page_datas[0] = (char *)malloc(bpl * page_height * 2);
		pd = page_datas[0];
		for (i=0; i < page_height; i++) {
			file.readImgLine((uchar *)pd, lsb_first);
			memcpy((void *)(pd + bpl), (void *)pd, bpl);
			pd += (bpl << 1);
		}
		page_height <<= 1;
	}
	file.close();

	page_datas[1] = shrinkPage(page_width, page_height, page_datas[0]);
	page_datas[2] = shrinkPage(page_width >> 1, page_height >> 1, page_datas[1]);

	kapp->restoreOverrideCursor();

	showPage();
}
