/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <unistd.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qdir.h>

#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "mloloader.h"
#include "global.h"
#include "filters.h"



#define MOD_STAT_IDLE         0
#define MOD_STAT_WAIT4RESET   1
#define MOD_STAT_WAIT4MODEL   2
#define MOD_STAT_WAIT4DIR     3
#define MOD_STAT_WAIT4DNL     4
#define MOD_STAT_WAIT4DEL     5

#define MODEL_UNKNOWN         0
#define MODEL_OFFICE          1
#define MODEL_OFFICE_II       2
#define MODEL_ISDN_OFFICE     3

#define ID_MODEL_OFFICE       "MicroLink Office"
#define ID_MODEL_OFFICE_II    "MicroLink Office II"
#define ID_MODEL_ISDN_OFFICE  "MicroLink ISDN Office"



MLOLoader::MLOLoader(QWidget *parent, const char *name): QObject(parent, name)
{
	config = kapp->config();
	
	modem = new Modem(this, "modem");
	Q_CHECK_PTR(modem);

	init();

	connect(modem, SIGNAL(gotLine(const QCString &)), SLOT(fetchModemLine(const QCString &)));
	connect(modem, SIGNAL(xmodemStatus(int)), SLOT(fetchXModemStatus(int)));
	connect(modem, SIGNAL(xmodemDone(bool)), SLOT(fetchXModemDone(bool)));
	connect(modem, SIGNAL(timeout()), SLOT(fetchTimeout()));
}


MLOLoader::~MLOLoader()
{
	reset();
}


bool MLOLoader::startDownload()
{
	static int speeds[] = { 19200, 38400, 57600, 115200, 230400 };
	int speed;
	bool use_usb;

	if (status)
		return false;

	config->setGroup(ID_PREF_GROUP_MODEM);
	load_new = config->readBoolEntry(ID_PREF_MOD_LOADNEW, PREF_MOD_LOADNEW);
	delete_fax = config->readBoolEntry(ID_PREF_MOD_DELETEFAX, PREF_MOD_DELETEFAX);
	use_usb = config->readBoolEntry(ID_PREF_MOD_USB, PREF_MOD_USB);
	if (use_usb) {
		modem->setDevice(use_usb, config->readNumEntry(ID_PREF_MOD_USB_DEVICE, PREF_MOD_USB_DEVICE));
	} else {
		modem->setDevice(use_usb, config->readNumEntry(ID_PREF_MOD_SER_DEVICE, PREF_MOD_SER_DEVICE));
		speed = config->readNumEntry(ID_PREF_MOD_SER_SPEED, PREF_MOD_SER_SPEED);
		if ((speed < 0) || (speed > 4))
			speed = PREF_MOD_SER_SPEED;
		modem->setSpeed(speeds[speed]);
	}
	
	modem->setLineParams(8, 'N', 1);

	if (!modem->open()) {
		emit helpMessage(i18n("Cannot open modem device."));
		return false;
	}
	if (!use_usb && !modem->dsrOn()) {
		emit helpMessage(i18n("Modem is off."));
		modem->close();
		return false;
	}
	if (!modem->ctsOn()) {
		emit helpMessage(i18n("Modem is busy."));
		modem->close();
		return false;
	}

	modem->writeLine("");
	usleep(250000);
	modem->flush();
	writeCommand("AT&F", 2000);

	status = MOD_STAT_WAIT4RESET;

	return true;
}


void MLOLoader::cancel()
{
	if (status == MOD_STAT_WAIT4DNL)
		modem->abortXModem();
	else {
		reset();
		emit done();
		emit helpMessage(i18n("Aborted by user."));
	}
}



void MLOLoader::fetchModemLine(const QCString &line)
{
	if (line == command)	// Discard command echo
		return;

	if (line == "ERROR") {
		modem->timerStop();
		status = MOD_STAT_IDLE;
		emit helpMessage(i18n("Error after modem command '%1'.").arg(command));
		reset();
		emit done();
		return;
	}
	
	switch (status) {
		case MOD_STAT_WAIT4RESET:					/* "AT&F" sent	*/
			if (line == "OK") {
				modem->timerStop();
				model = MODEL_UNKNOWN;
				writeCommand("ATI6", 2000);
				status = MOD_STAT_WAIT4MODEL;
			}
			break;
		case MOD_STAT_WAIT4MODEL:					/* "ATI6" sent */
			if (line == ID_MODEL_OFFICE)
				model = MODEL_OFFICE;
			else if (line == ID_MODEL_OFFICE_II)
				model = MODEL_OFFICE_II;
			else if (line == ID_MODEL_ISDN_OFFICE)
				model = MODEL_ISDN_OFFICE;
			else if (line == "OK") {
				modem->timerStop();
				if (model == MODEL_UNKNOWN) {
					status = MOD_STAT_IDLE;
					emit done();
					emit helpMessage(i18n("Unknown modem model \"%1\".").arg(line));
					reset();
					return;
				}
				if (model == MODEL_OFFICE)
					writeCommand(load_new ? "AT$JDIR=,FAX,N" : "AT$JDIR=,FAX", 3000);
				if ((model == MODEL_OFFICE_II) || (model == MODEL_ISDN_OFFICE))
					writeCommand("AT$JDIR", 3000);
				status = MOD_STAT_WAIT4DIR;
			}
			break;
		case MOD_STAT_WAIT4DIR:					/* "AT$JDIR[=,FAX[,N]]" sent	*/
			char buff[81], *p;
			fitem_t *fitem;
			char *type, *flags, *datetime, *sizes, *clip;
			int year, month, day, hour, minute, second, size;

			if (line == "OK") {
				modem->timerStop();
				count = 0;
				status = MOD_STAT_WAIT4DNL;
				downloadFile();
				break;
			}

			strncpy(buff, line.data(), 80);
			buff[80] = 0;

			if (!(type = strchr(buff, ',')))
				break;
			p = type;
			type++;
			while ((p > buff) && (*(p-1) == ' '))
				p--;
			*p = 0;

			while (*type == ' ')
				type++;

			if (!(flags = strchr(type, ',')))
				break;
			*flags = 0;
			flags++;
			
			if (!(datetime = strchr(flags, ',')))
				break;
			*datetime = 0;
			datetime++;
			
			if (!(p = strchr(datetime, ',')))
				break;
			p++;
			
			if (!(sizes = strchr(p, ',')))
				break;
			*sizes = 0;
			sizes++;
			
			if ((clip = strchr(sizes, ','))) {
				*clip = 0;
				clip++;
			}

			if ((model == MODEL_OFFICE_II) || (model == MODEL_ISDN_OFFICE)) {
				if (strcmp(type, "TIFF"))
					break;
				if (load_new && !strchr(type, 'N'))
					break;
			}

			if (model == MODEL_OFFICE) {
				char timestr[13];

				if (sscanf(datetime, "%d.%d.%d, %d:%d", &day, &month, &year, &hour, &minute) != 5)
					break;
				sprintf(timestr, "%04d%02d%02d%02d%02d", year, month, day, hour, minute);
				if (!strncmp(buff, timestr, 12) && (buff[12] >= '0') && (buff[12] <= '9') &&  (buff[13] >= '0') && (buff[13] <= '9'))
					second = (int)(buff[12] & 0x0f) * 10 + (int)(buff[13] & 0x0f);
				else
					second = 0;
			}
			
			if ((model == MODEL_OFFICE_II) || (model == MODEL_ISDN_OFFICE)) {
				if (sscanf(datetime, " %d.%d.%d, %d:%d:%d", &day, &month, &year, &hour, &minute, &second) != 6)
					break;
			}
			if (sscanf(sizes, " %d", &size) != 1)
				break;
			
			fitem = (fitem_t *)malloc(sizeof(fitem_t));
			Q_CHECK_PTR(fitem);
			fitem->next = fitems;
			fitems = fitem;
			strncpy(fitems->name, buff, 20);
			fitems->name[20] = 0;
			fitems->year = year;
			fitems->month = month;
			fitems->day = day;
			fitems->hour = hour;
			fitems->minute = minute;
			fitems->second = second;
			fitems->size = size;
			nfiles++;
			break;
		case MOD_STAT_WAIT4DNL:					/* XModem confirmed	*/
			if (line == "OK") {
				modem->timerStop();

				if (delete_fax) {
					QCString com;
					com.sprintf("AT$JDEL=\"%s\"", fitems->name);
					writeCommand(com, 2000);
					status = MOD_STAT_WAIT4DEL;
				} else {
					fitem = fitems;
					fitems = fitems->next;
					free(fitem);
					downloadFile();
				}
			}
			break;
		case MOD_STAT_WAIT4DEL:					/* Delete confirmed	*/
			if (line == "OK") {
				modem->timerStop();
				fitem = fitems;
				fitems = fitems->next;
				free(fitem);
				status = MOD_STAT_WAIT4DNL;
				downloadFile();
			}
		default:
			break;
	}
}


void MLOLoader::fetchXModemStatus(int pos)
{
	emit progress(pos);
}


void MLOLoader::fetchXModemDone(bool success)
{
	QString s;

	if (success) {
		QDir d;
		QString fname = fitems->name;

		if (fname.right(4) == ".FAX")
			fname.replace(fname.length()-3, 3, "tif");

		if (model == MODEL_OFFICE) {
			MLO2TiffFilter filter;

			filter.setFile(expandPath(QString(TEMP_DIR) + "/" + fitems->name));
			filter.setTime(fitems->year, fitems->month, fitems->day, fitems->hour, fitems->minute, fitems->second);
			filter.convertFile(expandPath(QString(INBOX_DIR) + "/" + fname));
		} else {
			Tiff2TiffFilter filter;

			filter.setFile(expandPath(QString(TEMP_DIR) + "/" + fitems->name));
			filter.convertFile(expandPath(QString(INBOX_DIR) + "/" + fname));
		}
		d.remove(expandPath(QString(TEMP_DIR) + "/" + fitems->name));
		emit gotFile(fname);
		if ((model == MODEL_OFFICE_II) || (model == MODEL_ISDN_OFFICE))	// Work-around for too fast OK response
			modem->timerStart(1000);
	} else {
		emit done();
		emit helpMessage(i18n("Download failed."));
		reset();
	}
}


void MLOLoader::fetchTimeout()
{
	QString s;
	
	if (status == MOD_STAT_WAIT4DNL) {	// Work-around for too fast OK response
		modem->flush();
		fetchModemLine("OK");
		return;
	}

	emit done();
	emit helpMessage(i18n("Modem response timeout."));
	reset();
}


void MLOLoader::init()
{
	load_new = false;
	delete_fax = false;
	
	status = MOD_STAT_IDLE;
	model = MODEL_UNKNOWN;
	command = "";

	fitems = 0;
	nfiles = 0;
	count = 0;
}


void MLOLoader::reset()
{
	fitem_t *p;

	modem->close();

	while (fitems) {
		p = fitems;
		fitems = fitems->next;
		free(p);
	}

	init();
}


void MLOLoader::downloadFile()
{
	if (fitems) {
		QCString com;
		emit message(i18n("Receiving file %1 of %2...").arg(++count).arg(nfiles));
		emit totalSteps(fitems->size);
		com.sprintf("AT$JDNL=\"%s\"", fitems->name);
		writeCommand(com, 3000);
		modem->receiveXModem(expandPath(QString(TEMP_DIR) + "/" + fitems->name), fitems->size, model == MODEL_OFFICE);	// firmware problem (no crc allowed with isdn yet)!
	} else {
		emit done();
		if (nfiles)
			emit helpMessage(i18n("Received %1 file(s).").arg(nfiles));
		else
			emit helpMessage(i18n("No facsimile in modem."));
		reset();
	}
}


void MLOLoader::writeCommand(const QCString &com, int msec)
{
	command = com;
	usleep(10000);
	modem->writeLine(command, msec);
}
