/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 * PostScript(R) is a registered trademark of Adobe Systems Incorporated.
 */

#include <stdlib.h>
#include <string.h>

#include <qdatetime.h>

#include <kapp.h>
#include <klocale.h>

#include "global.h"
#include "senderaliases.h"
#include "filters.h"



MLOFilter::MLOFilter()
{
	init();
}


MLOFilter::MLOFilter(const QString &infile)
{
	setFile(infile);
}


MLOFilter::~MLOFilter()
{
}


void MLOFilter::setFile(const QString &infile)
{
	init();
	mlofile.setName(infile);
}


void MLOFilter::setTime(int year, int month, int day, int hour, int minute, int second)
{
	m_time.setDate(QDate(year, month, day));
	m_time.setTime(QTime(hour ,minute, second));
}


void MLOFilter::init()
{
	m_first = 0;
	m_last = 0;
	m_pages = 0;
	m_time = QDateTime::currentDateTime();
}




MLO2TiffFilter::MLO2TiffFilter(): MLOFilter()
{
}


MLO2TiffFilter::MLO2TiffFilter(const QString &infile): MLOFilter(infile)
{
}


MLO2TiffFilter::~MLO2TiffFilter()
{
}


bool MLO2TiffFilter::convertFile(const QString &outfile)
{
	int page;
	
	if (!mlofile.open(IO_ReadOnly)) {
		return false;
	}

	m_pages = mlofile.pages();
	
	if (m_first == 0) {
		m_first = 1;
		m_last = m_pages;
	}

	if ((m_first > m_last) || (m_first > m_pages)) {
		mlofile.close();
		return false;
	}

	if (m_last > m_pages)
		m_last = m_pages;
	
	m_pages = m_last - m_first + 1;

	tifffile.setName(outfile);
	
	if (!tifffile.open(IO_ReadWrite | IO_Truncate)) {
		mlofile.close();
		return false;
	}

	tifffile.setSender(mlofile.sender());
	tifffile.setTime(m_time);

	for (page = m_first; page <= m_last; page++)
		convertPage(page);

	tifffile.close();
	mlofile.close();

	return true;
}


bool MLO2TiffFilter::convertPage(int page)
{
	int h, i;
	short buffer[MAX_RLE_BUFF];

	if (!mlofile.gotoPage(page))
		return false;
	
	h = mlofile.pageHeight();

	tifffile.addPage(page, m_pages, mlofile.pageWidth(), h, mlofile.fineResolution());

	for (i=0; i < h; i++) {
		mlofile.readRleLine(buffer);
		tifffile.writeRleLine(buffer);
	}

	tifffile.finishPage();

	return true;
}



TiffFilter::TiffFilter()
{
	init();
}


TiffFilter::TiffFilter(const QString &infile)
{
	init();
	setFile(infile);
}


TiffFilter::~TiffFilter()
{
}


void TiffFilter::setFile(const QString &infile)
{
	tifffile.setName(infile);
}


bool TiffFilter::convertFile(FILE *stream)
{
	f = stream;

	return true;
}


void TiffFilter::init()
{
	m_first = 0;
	m_last = 0;
	m_pages = 0;
	f = 0;
}



Tiff2TiffFilter::Tiff2TiffFilter(): TiffFilter()
{
}


Tiff2TiffFilter::~Tiff2TiffFilter()
{
}


bool Tiff2TiffFilter::convertFile(const QString &outfile)
{
	int page;
	
	if (!tifffile.open(IO_ReadOnly)) {
		return false;
	}

	m_pages = tifffile.pages();
	
	if (m_first == 0) {
		m_first = 1;
		m_last = m_pages;
	}

	if ((m_first > m_last) || (m_first > m_pages)) {
		tifffile.close();
		return false;
	}

	if (m_last > m_pages)
		m_last = m_pages;
	
	m_pages = m_last - m_first + 1;

	tiffout.setName(outfile);

	if (!tiffout.open(IO_ReadWrite | IO_Truncate)) {
		tifffile.close();
		return false;
	}

	tiffout.setSender(tifffile.sender());
	tiffout.setTime(tifffile.time());

	for (page = m_first; page <= m_last; page++)
		convertPage(page);

	tiffout.close();
	tifffile.close();

	return true;
}


bool Tiff2TiffFilter::convertPage(int page)
{
	int h, i;
	short buffer[MAX_RLE_BUFF];

	if (!tifffile.gotoPage(page))
		return false;
	
	h = tifffile.pageHeight();

	tiffout.addPage(page, m_pages, tifffile.pageWidth(), h, tifffile.fineResolution());

	for (i=0; i < h; i++) {
		tifffile.readRleLine(buffer);
		tiffout.writeRleLine(buffer);
	}

	tiffout.finishPage();

	return true;
}



Tiff2PSFilter::Tiff2PSFilter(): TiffFilter()
{
	init();
}


Tiff2PSFilter::Tiff2PSFilter(const QString &infile): TiffFilter(infile)
{
	init();
}


Tiff2PSFilter::~Tiff2PSFilter()
{
}


void Tiff2PSFilter::setFormat(int format, bool landscp)
{
	/* A4, B4, B5, Letter, Legal, Executive */
	static int papers_x[] = { 595,  729, 516, 612,  612, 540 };
	static int papers_y[] = { 842, 1032, 729, 792, 1008, 720 };

	landscape = landscp;

	if (landscape) {
		paper_x = papers_y[format];
		paper_y = papers_x[format];
	}
	else {
		paper_x = papers_x[format];
		paper_y = papers_y[format];
	}
}


bool Tiff2PSFilter::convertFile(FILE *stream)
{
	int page, sub_page;
	int *size_y_buff;
	int maxlines;
	int ps_pages;
	QString s;

	f = stream;

	/* Size of viewport (in pixel, 1/72 dpi) */
	view_x = paper_x - (int)((double)(l_margin + r_margin) * 2.8346);
	view_y = paper_y - (int)((double)(t_margin + b_margin) * 2.8346);
	
	/* Bottom-left border (in pixel, 1/72 dpi) */
	trans_x = (int)((double)l_margin * 2.8346);
	trans_y = (int)((double)b_margin * 2.8346);
	
	/* Calculate real number of pages */
	if (!tifffile.open(IO_ReadOnly)) {
		return false;
	}

	m_pages = tifffile.pages();
	
	if (m_first == 0) {
		m_first = 1;
		m_last = m_pages;
	}

	if ((m_first > m_last) || (m_first > m_pages)) {
		tifffile.close();
		return false;
	}

	if (m_last > m_pages)
		m_last = m_pages;

	m_pages = 0;
	size_y_buff = (int *)malloc((m_last - m_first + 1) * sizeof(int));
	for (page = m_first; page <= m_last; page++) {
		tifffile.gotoPage(page);

		/* Horizontal size of image (in pixel, 204 dpi) */
		size_x = tifffile.pageWidth();
		/* Vertical size of image (in pixel, 98 or 196 lpi) */
		size_y = tifffile.pageUsedHeight();
		size_y_buff[page-m_first] = size_y;

		scale_x = size_x * 72 / 204;
		
		maxlines = view_y * (tifffile.fineResolution() ? 196 : 98) / 72;

		if (scale_x > view_x)
			maxlines = (maxlines * scale_x) / view_x;

		while (size_y > maxlines) {
			size_y -= maxlines;
			m_pages++;
		}
		m_pages++;
	}

	writeLine("%!PS-Adobe-3.0");
	if (ribbons)
		writeLine(QString("%%BoundingBox: %1 0 %2 %3").arg(trans_x).arg(trans_x + view_x).arg(paper_y));
	else
		writeLine(QString("%%BoundingBox: %1 %2 %3 %4").arg(trans_x).arg(trans_y).arg(trans_x + view_x).arg(trans_y + view_y));
	writeLine("%%Creator: KMLOFax Version " VERSION);
	writeLine(QString("%%CreationDate: ") + tifffile.time().toString());
	writeLine("%%DocumentData: Clean7Bit");
	if (level2)
		writeLine("%%LanguageLevel: 2");
	if (landscape)
		writeLine("%%Orientation: Landscape");
	else
		writeLine("%%Orientation: Portrait");
	writeLine(QString("%%Pages: %1").arg(m_pages));
	writeLine("%%PageOrder: Ascend");
	s = SENDER_ALIAS(tifffile.sender());
	if (s.isEmpty())
		writeLine(QString("%%Title: ") + i18n("Facsimile"));
	else
		writeLine(QString("%%Title: ") + i18n("Facsimile from %1").arg(s));
	writeLine("%%DocumentFonts: Helvetica-Bold");
	writeLine("%%EndComments");
	if (ribbons) {
		writeLine("%%BeginProlog");
		writeLine("/Ribbon {");
		writeLine(" gsave");
		writeLine(" dup");
		writeLine(" 0.7 setgray");
		writeLine(" newpath");
		writeLine(" 0 moveto");
		writeLine(QString(" 0 %1 rlineto").arg(paper_y));
		writeLine(" 15 0 rlineto");
		writeLine(QString(" 0 %1 rlineto").arg(-paper_y));
		writeLine(" closepath\n fill\n 1 setgray\n /Helvetica-Bold findfont\n 12 scalefont\n setfont");
		writeLine(" 12 add\n -40 moveto\n 90 rotate");
		writeLine(QString(" %1 {40 0 rmoveto gsave (FAX) show grestore} repeat").arg((paper_y + 39) / 40));
		writeLine(" grestore");
		writeLine("} bind def");
	}
	writeLine("%%EndProlog");
	writeLine("%%BeginSetup");
	if (level2)
		writeLine("/DeviceGray setcolorspace");
	writeLine(QString("/#copies %1 def").arg(doc_copies));
	writeLine("%%EndSetup");

	ps_pages = 0;
	for (page = m_first; page <= m_last; page++) {
		tifffile.gotoPage(page);

		size_x = tifffile.pageWidth();
		size_y = size_y_buff[page-m_first];

		scale_x = (size_x * 72) / 204;
		scale_y = (size_y * 72) / (tifffile.fineResolution() ? 196 : 98);
		maxlines = (view_y * (tifffile.fineResolution() ? 196 : 98)) / 72;

		if (scale_x > view_x) {
			scale_y = (scale_y * view_x) / scale_x;
			maxlines = (maxlines * scale_x) / view_x;
			scale_x = view_x;
		}

		sub_page = 0;
		while (size_y > maxlines) {
			if (!convertPage(page, ++sub_page, ++ps_pages, maxlines, view_y)) {
				tifffile.close();
				return false;
			}
			size_y -= maxlines;
			scale_y -= view_y;
		}
		if (!convertPage(page, ++sub_page, ++ps_pages, size_y, scale_y)) {
			tifffile.close();
			return false;
		}
	}

	free(size_y_buff);

	writeLine("%%Trailer\n%%EOF");

	tifffile.close();

	return true;
}


void Tiff2PSFilter::init()
{
	setFormat(PAPER_A4, false);
	level2 = true;
	interpolate = false;
	ribbons = true;
	doc_copies = 1;
	l_margin = 0;
	r_margin = 0;
	t_margin = 0;
	b_margin = 0;

	resetImageData();
}


bool Tiff2PSFilter::convertPage(int tiff_page, int sub_page, int ps_page, int lines, int strech_y)
{
	static const char n_hex[16] = {
		'f', '7', 'b', '3', 'd', '5', '9', '1',
		'e', '6', 'a', '2', 'c', '4', '8', '0'
	};
	int bpl, i, j;
	uchar img_in_buff[MAX_IMG_BUFF], huf_in_buff[MAX_HUF_BUFF], c;
	char out_buff[66];

	bpl = size_x >> 3;

	if (!writeLine(QString("%%Page: %1.%2 %3").arg(tiff_page).arg(sub_page).arg(ps_page)))
		return false;

	writeLine("%%BeginPageSetup\n/pagelevel save def\n%%EndPageSetup");

	if (ribbons)
		writeLine(QString("40 Ribbon\n%1 Ribbon").arg(paper_x - 55));
	
	writeLine(QString("%1 %2 translate").arg(trans_x).arg(trans_y + view_y - strech_y));
	writeLine(QString("%1 %2 scale").arg(scale_x).arg(strech_y));

	if (level2) {
		writeLine("{");
		writeLine("/G3DICT <<");
		writeLine(QString(" /Columns %1").arg(size_x));
		writeLine(" /K 0");
		writeLine(" /Uncompressed false");
		writeLine(" /EndOfLine true");
		writeLine(" /EndOfBlock true");
		writeLine(" /BlackIs1 false");
		writeLine(">> def");
		writeLine("/Data1 currentfile /ASCII85Decode filter def");
		writeLine("/Data2 Data1 G3DICT /CCITTFaxDecode filter def");
		writeLine("<<");
		writeLine(" /ImageType 1");
		writeLine(QString(" /Width %1").arg(size_x));
		writeLine(QString(" /Height %1").arg(lines));
		writeLine(" /BitsPerComponent 1");
		writeLine(" /Decode [ 0 1 ]");
		writeLine(QString(" /ImageMatrix [ %1 0 0 %2 0 %3 ]").arg(size_x).arg(-lines).arg(lines));
		if (interpolate)
			writeLine(" /Interpolate true");
		else
			writeLine(" /Interpolate false");
		writeLine(" /DataSource Data2");
		if (ribbons)
			writeLine(">> imagemask");
		else
			writeLine(">> image");
		writeLine("Data1 flushfile");
		writeLine("} exec");
	}
	else {
		writeLine(QString("/bpl %1 string def").arg(bpl));
		writeLine(QString("%1 %2 1 [ %3 0 0 %4 0 %5 ]").arg(size_x).arg(lines).arg(size_x).arg(-lines).arg(lines));
		writeLine("{currentfile bpl readhexstring pop}\nimage");	
	}
	
	if (level2) {
		writeBase85(0x00);
		writeBase85(0x01);

		while (lines--) {
			bpl = tifffile.readHuffLine(huf_in_buff);
			for (i=0; i < bpl; i++)
				writeBase85(huf_in_buff[i]);
		}

		writeBase85(0x00); /* Write End-Of-Block */
		writeBase85(0x01);
		writeBase85(0x00);
		writeBase85(0x10);
		writeBase85(0x01);
		writeBase85(0x00);
		writeBase85(0x10);
		writeBase85(0x01);

		flushBase85();
		flushImageData();
		resetImageData();
		
		writeLine("~>");
	}
	else {
		while (lines--) {
			tifffile.readImgLine(img_in_buff, true);
			i = 0;
			for (j=0; j < bpl; j++) {
				c = img_in_buff[j];
				out_buff[i++] = n_hex[c & 0x0f];
				out_buff[i++] = n_hex[c >> 4];
				if (i == 64) {
					out_buff[i] = 0;
					writeLine(out_buff);
					i = 0;
				}
			}
			if (i) {
				out_buff[i] = 0;
				writeLine(out_buff);
			}
		}
	}

	writeLine("pagelevel restore\nshowpage");

	return true;
}


bool Tiff2PSFilter::writeLine(const QString &s)
{
	return fprintf(f, "%s\n", s.latin1()) > 0;
}


void Tiff2PSFilter::writeBase85(uchar c)
{
	base85_buff[base85_ind] = c;

	if (base85_ind)
		base85_ind--;
	else
		flushBase85();
}


void Tiff2PSFilter::flushBase85()
{
	static ulong power85[5] = { 85*85*85*85, 85*85*85, 85*85, 85, 1 };
	ulong v, c;
	int i;

	if (base85_ind == 3)
		return;

	v = *((ulong *)base85_buff);

	if (v)
		for (i=0; i < 5; i++) {
			c = v / power85[i];
			writeImageData('!' + (char)c);
			v -= c * power85[i];
		}
	else
		writeImageData('z');

	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
}


void Tiff2PSFilter::writeImageData(char c)
{
	image_buff[image_ind] = c;

	if (++image_ind == 253)
		flushImageData();
}


void Tiff2PSFilter::flushImageData()
{
	if (!image_ind)
		return;

	image_buff[image_ind] = 0;
	
	if ((image_buff[0] == '%') && (image_buff[1] == '%'))
		writeLine(QString("% ") + &image_buff[1]);
	else
		writeLine(image_buff);

	image_ind = 0;
}


void Tiff2PSFilter::resetImageData()
{
	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
	
	image_ind = 0;
}
