#include <iostream.h>
#include <string>
#include <sstream>
#include <math.h>

#include "tagFunctions.h"
#include "exif_parser.h"

string tagString(const char *name, IFDEntry *entry, const char *units) 
{
  string str;
  str.assign((char*) entry->getValuePtr(), entry->getByteCount());

  if (units) str += units;
  return str;
}

string tagRational(const char *name, IFDEntry *entry, const char *units)
{
  string str;
  std::ostringstream o;      
  
  o << Exif_Parser::Get_ULong(entry->getValuePtr());
  o <<"/";
  o << Exif_Parser::Get_ULong(4+(char *)entry->getValuePtr());

  str = o.str();
  if (units) str += units;
  return str;

}

string tagRationalNormal(const char *name, IFDEntry *entry, const char *units)
{
  string str;
  std::ostringstream o;      
  
  double num = (double) Exif_Parser::Get_ULong(entry->getValuePtr());
  double den = (double) Exif_Parser::Get_ULong(4+(char *)entry->getValuePtr());
  o << num/den;

  str = o.str();
  if (units) str += units;
  return str;

}


string tagUShort(const char *name, IFDEntry *entry, const char *units) {

  string str;
  std::ostringstream o; 

  for (unsigned i=0; i<entry->getNumComponents(); i++) 
    o << Exif_Parser::Get_UShort(entry->getValuePtr());

  str = o.str();
  if (units) str += units;
  return str;
}

string tagULong(const char *name, IFDEntry *entry, const char *units) {

  string str;
  std::ostringstream o; 

  for (unsigned i=0; i<entry->getNumComponents(); i++)
    o << Exif_Parser::Get_ULong(entry->getValuePtr());

  str = o.str();
  if (units) str += units;
  return str;
}

string tagNumber(const char *name, IFDEntry *entry, const char *units) {

  string str;

  switch(entry->getFormat()) {
  case 3:
    return tagUShort(name, entry, units);
  case 4:
    return tagULong(name, entry, units);
  default:
    return NULL;
  }
  return NULL;
}

string tagOrientation(const char *name, IFDEntry *entry, const char *units)
{
  int orientation = (int) Exif_Parser::Get_UShort(entry->getValuePtr());
  string orient;

  switch (orientation) {
  case 1:
    orient = "Top (Row) & LeftSide (Column)";
    break;
  case 2:
    orient = "Top (Row) & RightSide (Column)";
    break;
  case 3:
    orient = "Bottom (Row) & RightSide (Column)";
    break;
  case 4:
    orient = "Bottom (Row) & LeftSide (Column)";
    break;
  case 5:
    orient = "LeftSide (Row) & Top (Column)";
    break;
  case 6:
    orient = "RightSide (Row) & Top (Column)";
    break;
  case 7:
    orient = "RightSide (Row) & Bottom (Column)";
    break;
  case 8:
    orient = "LeftSide (Row) & Bottom (Column)";
    break;
  }

  if (units) orient += units;
  return orient;

}


string tagResolutionUnit(const char *name, IFDEntry *entry, const char *units)
{
  int unit = (int) Exif_Parser::Get_UShort(entry->getValuePtr());
  string resUnit;

  switch(unit) {
  case 1:
    resUnit = "No-Unit";
    break;
  case 2:
    resUnit = "Inches";
    break;
  case 3:
    resUnit = "Centimeters";
    break;
  }

  return resUnit;
}


string tagWhitePoint(const char *name, IFDEntry *entry, const char *units)
{
  return NULL;
}

string tagPrimaryChromacities(const char *name, IFDEntry *entry, const char *units)
{ 
 return NULL;
}

string tagYCbCrCoefficients(const char *name, IFDEntry *entry, const char *units)
{
  return NULL;
}

string tagYCbCrPositioning(const char *name, IFDEntry *entry, const char *units)
{
  string position;
  int pos = Exif_Parser::Get_UShort(entry->getValuePtr());

  switch(pos) {
  case 1:
    position = "Center of Pixel Array";
    break;

  case 2:
    position = "Datum Point";
    break;
  }

  return position;
}

string tagReferenceBW(const char *name, IFDEntry *entry, const char *units)
{
  return NULL;
}


// ------------

string tagExpProgram(const char *name, IFDEntry *entry, const char *units) {
 
  int val = Exif_Parser::Get_UShort(entry->getValuePtr());
  string program;
  
  switch(val) {
  case 1: program = "Manual Control"; break;
  case 2: program = "Program Normal"; break;
  case 3: program = "Aperture Priority"; break;
  case 4: program = "Shutter Priority"; break;
  case 5: program = "Program Creative"; break;
  case 6: program = "Program Action"; break;
  case 7: program = "Portrait Mode"; break;
  case 8: program = "Landscape Mode"; break;
  }

  return program;
}

string tagISO(const char *name, IFDEntry *entry, const char *units) {

  int first = Exif_Parser::Get_UShort(entry->getValuePtr());
  int second = Exif_Parser::Get_UShort(2+entry->getValuePtr());

  
  string str;
  std::ostringstream o;
  o << first << " (" << second << ")";

  str = o.str();
  return str;
}


string tagShutterSpeed(const char *name, IFDEntry *entry, const char *units) {

  int num = Exif_Parser::Get_SLong(entry->getValuePtr());
  int den = Exif_Parser::Get_SLong(4+(char*)entry->getValuePtr());

  double val = (double)num/(double)den;

  std::ostringstream o; 
  o << "1/" << pow(2,val);

  string str= o.str();
  if (units) str += units;
  return str;
}

string tagAperture(const char *name, IFDEntry *entry, const char *units) {

  int num = Exif_Parser::Get_ULong(entry->getValuePtr());
  int den = Exif_Parser::Get_ULong(4+(char*)entry->getValuePtr());
  double val = (double)num/(double)den;

  std::ostringstream o; 
  o << "f" << floor(10*pow(sqrt(2), val))/10.0;
  string str= o.str();

  if (units) str += units;
  return str;
}



string tagMeteringMode(const char *name, IFDEntry *entry, const char *units) {

  int val = (int) Exif_Parser::Get_UShort(entry->getValuePtr());
  string str;

  switch(val) {
  case 0: str = "Unknown"; break;
  case 1: str = "Average"; break;
  case 2: str = "Center Weighted Average"; break;
  case 3: str = "Spot"; break;
  case 4: str = "Multi-Spot"; break;
  case 5: str = "Multi-Segment"; break;
  case 6: str = "Partial"; break;
  case 255: str = "Other"; break;
  }

  return str;
}

string tagLightSource(const char *name, IFDEntry *entry, const char *units) {

  int val = (int) Exif_Parser::Get_UShort(entry->getValuePtr());
  string str;

  switch(val) {
  case 0: str = "Automatic"; break;
  case 1: str = "Daylight"; break;
  case 2: str = "Fluorescent"; break;
  case 3: str = "Tungsten"; break;
  case 10: str = "Flash"; break;
  case 17: str = "Standard Light A"; break;
  case 18: str = "Standard Light B"; break;
  case 19: str = "Standard Light C"; break;
  case 20: str = "D55"; break;
  case 21: str = "D65"; break;
  case 22: str = "D75"; break;
  case 255:str = "Other"; break;
  }

  return str;
}

string tagFlash(const char *name, IFDEntry *entry, const char *units) {

  int val = (int) Exif_Parser::Get_UShort(entry->getValuePtr());
  string str;


  switch(val) {
  case 0: str = "No Flash"; break;
  case 1: str = "Yes"; break;
  case 5: str = "Yes (Strobe Return Light Not Detected)"; break;
  case 7: str = "Yes (Strobe Return Light Detected)"; break;
  }

  return str;

}

string tagExifVersion(const char *name, IFDEntry *entry, const char *units) {

  string str;

  for (unsigned int i=0; i<entry->getNumComponents(); i++) {
    unsigned char c = *(i+entry->getValuePtr());
    str += c;
  }
    
  return str;

}

string tagSensingMethod(const char *name, IFDEntry *entry, const char *units=0) {

  int val = (int) Exif_Parser::Get_UShort(entry->getValuePtr());
  string str;

  switch(val) {
  case 2: str = "1 Chip color area sensor"; break;
  default: str = "Unknown"; break;
  }

  return str;
  
}



string tagFileSource(const char *name, IFDEntry *entry, const char *units=0) {

  string str;
  
  int a = ( int) *(entry->getValuePtr());

  if (a==3)
    str = "Digital Still Camera";
  else
    str = "Unknown";

  return str;
}

string tagSceneType(const char *name, IFDEntry *entry, const char *units=0) {

  string str;

  int a = ( int) *(entry->getValuePtr());

  if (a==1)
    str = "Directly Photographed";
  else
    str = "Unknown";

  return str;

}

//----- IFD1 Tags  ----------------------------------------------------------

string tagCompression(const char *name, IFDEntry *entry, const char *units=0) {

  string str;
  int val = Exif_Parser::Get_UShort(entry->getValuePtr());

  switch(val) {

  case 1: str = "No Compression"; break;
  case 6: str = "JPEG Compression"; break;
  default: str = "Unknown"; break;
  }

  return str;
}

string tagPhotoMetric(const char *name, IFDEntry *entry, const char *units=0) {

  string str;
  int val = Exif_Parser::Get_UShort(entry->getValuePtr());

  switch(val) {
  case 1: str = "Monochrome"; break;
  case 2: str ="RGB"; break;
  case 6: str ="YCbCr"; break;
  default: str = "Unknown"; break;
  }

  return str;
    
}

//------Nikon Tags ----------------------------------------------------------


string tagNikonVersion(const char *name, IFDEntry *entry, const char *units) {

  string str;
  std::ostringstream o;
  for (unsigned int i=0; i<entry->getNumComponents(); i++) {
    o <<(int) *(i+entry->getValuePtr());
  }
  
  str = o.str();
  
  return str;

}


string tagNikonAFFocusPosition(const char *name, IFDEntry *entry, const char *units=0) {

  string str;
  
  int a = ( int) *(entry->getValuePtr());
  int b = ( int) *(1+entry->getValuePtr());
  int c = ( int) *(2+entry->getValuePtr());
  int d = ( int) *(3+entry->getValuePtr());
  
  if (a == 0 && c == 0 && d == 0) {
    if (b == 0)  
      str = "Center";
    else if (b == 1)  
      str = "Top";
    else if (b == 2)  
      str = "Bottom";
    else if (b == 3)  
      str = "Left";
    else if (b == 4)  
      str = "Right";
    else 
      str = "Unknown";
  }
  else 
    str = "Unknown";

  return str;

}


// --- Olympus Tags---------------------------------------------

string tagOlympusSpecialMode(const char *name, IFDEntry *entry, 
			     const char *units=0) {

  string str;
  int val = Exif_Parser::Get_ULong(entry->getValuePtr());

  switch(val) {
  case 0: str = "Normal"; break;
  case 1: str = "Unknown"; break;
  case 2: str = "Fast"; break;
  case 3: str = "Panorama"; break;
  default: break;
  }


  if (Exif_Parser::Get_ULong(4+entry->getValuePtr()) > 0) {
    std::ostringstream o;
    o << Exif_Parser::Get_ULong(4+entry->getValuePtr());
    str += ", Sequence Number: ";
    str += o.str();
  }

  val = Exif_Parser::Get_ULong(8+entry->getValuePtr());

  switch(val) {
  case 1: str += ", "; str += "Left to Right"; break;
  case 2: str += ", "; str += "Right to Left"; break;
  case 3: str += ", "; str += "Bottom to Top"; break;
  case 4: str += ", "; str += "Top to Bottom"; break;
  default: break;
  }
  
  return str;

}

string tagOlympusJpegQuality(const char *name, IFDEntry *entry, 
			     const char *units=0) {

  string str;
  int val = Exif_Parser::Get_UShort(entry->getValuePtr());
  
  switch(val) {
  case 1: str = "SQ"; break;
  case 2: str = "HQ"; break;
  case 3: str = "SHQ"; break;
  default: break;
  }

  return str;

}

string tagOlympusMacro(const char *name, IFDEntry *entry, 
			     const char *units=0) {

  string str;
  int val = Exif_Parser::Get_UShort(entry->getValuePtr());

  switch(val) {
  case 0: str = "Normal Mode"; break;
  case 1: str = "Macro Mode"; break;
  }
    
  return str;

}
