#include "sqlsyntax.h"
#ifndef NO_SQL_HIGHLIGHT
#include <qstring.h>
#include <qcolor.h>

const QRegExp sqlsyntax::keyWords = QRegExp("\\b(ALTER|CREATE|DROP|EXPLAIN|GRANT|OPTIMIZE|RENAME|REVOKE|SELECT|SHOW|UPDATE|INSERT|USE|FROM)\\b",false);
const QRegExp sqlsyntax::otherWords = QRegExp("\\b(ADD|ALL|ANALYZE|AND|AS|ASC|ASENSITIVE|AUTO_INCREMENT|BDB|BEFORE|BERKELEYDB|BETWEEN"
    "|BOTH|BTREE|BY|CALL|CASCADE|CASE|CHANGE|CHECK|COLLATE|COLUMN|COLUMNS|CONNECTION|CONSTRAINT"
    "|CROSS|CURRENT_DATE|CURRENT_TIME|CURRENT_TIMESTAMP|CURSOR|DATABASES?|DAY_HOUR|DAY_MICROSECOND|DAY_MINUTE"
    "|DAY_SECOND|DECLARE|DEFAULT|DELAYED|DELETE|DESC|DESCRIBE|DISTINCT|DISTINCTROW|DIV|ELSE"
    "|ELSEIF|ENCLOSED|ERRORS|ESCAPED|EXISTS|FALSE|FIELDS|FOR|FORCE|FOREIGN|FULLTEXT"
    "|GROUP|HASH|HAVING|HIGH_PRIORITY|HOUR_MICROSECOND|HOUR_MINUTE|HOUR_SECOND|IF|IGNORE|IN|INDEX|INFILE|INNER"
    "|INNODB|INOUT|INSENSITIVE|INTERVAL|INTO|IO_THREAD|IS|ITERATE|JOIN|KEY|KEYS|KILL|LEADING"
    "|LEAVE|LEFT|LIKE|LIMIT|LINES|LOAD|LOCALTIME|LOCALTIMESTAMP|LOCK|LOOP|LOW_PRIORITY"
    "|MASTER_SERVER_ID|MATCH|MINUTE_MICROSECOND|MINUTE_SECOND|MOD|MRG_MYISAM"
    "|NATURAL|NOT|NO_WRITE_TO_BINLOG|NULL|ON|OPTION|OPTIONALLY|OR|ORDER|OUT|OUTER|OUTFILE|PRECISION"
    "|PRIMARY|PRIVILEGES|PROCEDURE|PURGE|READ|REFERENCES|REGEXP|REPEAT|REPLACE|REQUIRE|RESTRICT|RETURN|RETURNS"
    "|RIGHT|RLIKE|RTREE|SECOND_MICROSECOND|SENSITIVE|SEPARATOR|SOME|SONAME|SPATIAL|SPECIFIC"
    "|SQL_BIG_RESULT|SQL_CALC_FOUND_ROWS|SQL_SMALL_RESULT|SSL|STARTING|STRAIGHT_JOIN|STRIPED|TABLES?|TERMINATED|THEN"
    "|TO|TRAILING|TRUE|TYPES?|UNION|UNIQUE|UNLOCK|UNSIGNED|UNTIL|USAGE|USER_RESOURCES"
    "|USING|UTC_DATE|UTC_TIME|UTC_TIMESTAMP|VALUES|VARYING|WARNINGS|WHEN|WHERE|WHILE|WITH"
    "|WRITE|XOR|YEAR_MONTH|ZEROFILL)\\b|[=!<>+\\-]+",false);
const QRegExp sqlsyntax::datatypeWords= QRegExp("\\b(BIGINT|BINARY|BLOB|CHAR|CHARACTER|DEC|DECIMAL|DOUBLE"
    "|FLOAT|INT|INTEGER|LONG|LONGBLOB|LONGTEXT|MEDIUMBLOB|MEDIUMINT|MEDIUMTEXT|MIDDLEINT|SMALLINT"
    "|TINYBLOB|TINYINT|TINYTEXT|VARBINARY|VARCHAR|VARCHARACTER|TIMESTAMP|SET|REAL|TEXT|DATE|TIME|ENUM|DATETIME"
    "|YEAR|NUMERIC|MyISAM)\\b",false);
const QRegExp sqlsyntax::singleComment = QRegExp("//.*$|#.*$");

const std::string sqlsyntax::STATEMENT="statement";
const std::string sqlsyntax::KEYWORD="keywords";
const std::string sqlsyntax::DATATYPE="datatypes";
const std::string sqlsyntax::STRING="strings";
const std::string sqlsyntax::COMMENT="comments";


using namespace std;


sqlsyntax::sqlsyntax(QTextEdit*aEdit)
    :QSyntaxHighlighter(aEdit),statementColor(255,0,0),keyColor(0,200,0),
     commentColor(120,120,120),datatypesColor(0,0,200),stringColor(0,0,255),m_RealParse(true)
{
}

sqlsyntax::~sqlsyntax()
{
}

int sqlsyntax::highlightParagraph ( const QString & aText, int lastPara)
{
    setFormat(0,(int)aText.length(),regularFont,QColor(0,0,0));
    if (!m_RealParse) {
        return 0;
    }
    
    int pos = 0;
    int subpos = 0;
    RangeVector stringRanges,commentRanges;
    bool lpushed = false;
    RangePair tRange = RangePair(-1,aText.length()+1);
    
    const char*quot="\'";
    const char*dquot = "\"";
    const char*comment="*/";
    const char*use_quot;
    
    /* first search for end multiline comment */
    if (lastPara==1) {
        pos = aText.find("*/");
        if (pos == -1) {
            setFormat(0,aText.length(),commentFont,commentColor);
            return 1;
        }
        setFormat(0,pos+2,commentFont,commentColor);
        commentRanges.push_back(RangePair(0,pos+2));
    }
    
    subpos = 0;
    bool comment_opened = false;
    int mp = 0;
    int p1 = aText.find(quot);
    if (p1==-1) p1=aText.length()+1;
    mp = p1;
    use_quot = quot;
    int p2 = aText.find(dquot);
    if (p2==-1) p2=aText.length()+1;
    if (mp>p2) {
        mp = p2;
        use_quot=dquot;
    }
    int p3 = aText.find("/*");
    if (p3==-1) p3=aText.length()+1;
    if (mp>p3) {
        mp = p3;
        use_quot=comment;
        comment_opened=true;
        tRange.first=mp;
        subpos = mp;
        lpushed = false;
    }
    
    while ( (pos=aText.find(use_quot,subpos))!=-1) {
        if (!comment_opened && pos>0 && aText[pos-1]=='\\') {
            subpos=pos+1;
            continue;
        }
        if (tRange.first==-1) {
            tRange.first = pos;
            lpushed=false;
        } else {
            tRange.second = pos+(comment_opened?2:1);
            if (!comment_opened) {
                stringRanges.push_back(tRange);
            } else {
                commentRanges.push_back(tRange);
                comment_opened=false;
            }
            tRange = RangePair(-1,aText.length()+1);
            lpushed=true;
            
            p1 = aText.find(quot,pos+1);
            if (p1==-1) p1=aText.length()+1;
            mp = p1;
            use_quot = quot;
            p2 = aText.find(dquot,pos+1);
            if (p2==-1) p2=aText.length()+1;
            if (mp>p2) {
                mp = p2;
                use_quot=dquot;
            }
            p3 = aText.find("/*",pos+1);
            if (p3==-1) p3=aText.length()+1;
            if (mp>p3) {
                mp = p3;
                use_quot=comment;
                comment_opened=true;
                tRange.first=mp;
                subpos = mp;
                lpushed = false;
            }     
        }
        subpos = pos+1;
    }
    
    if (comment_opened && tRange.first!=-1 && !lpushed) commentRanges.push_back(tRange);
    RangeVector::const_iterator it=commentRanges.begin();
    for (;it!=commentRanges.end();++it) {
        setFormat(it->first,it->second-it->first,commentFont,commentColor);
    }

    if (!comment_opened && tRange.first!=-1 && !lpushed) stringRanges.push_back(tRange);
    it=stringRanges.begin();
    for (;it!=stringRanges.end();++it) {
        setFormat(it->first,it->second-it->first,stringFont,stringColor);
    }
    stringRanges.insert(stringRanges.end(),commentRanges.begin(),commentRanges.end());
    
    // search for sql keywords
    subpos = 0;
    pos = 0;
    while ( (pos = keyWords.search(aText,pos))!=-1) {
        subpos = keyWords.matchedLength();
        if (subpos == 0) {
            ++pos;continue;
        }
        if (!pos_in_string(stringRanges,pos)) {
            setFormat(pos,subpos,statementFont,statementColor);
        }
        pos+=subpos;
    }
    pos = 0;
    subpos = 0;
    while ( (pos = otherWords.search(aText,pos))!=-1) {
        subpos = otherWords.matchedLength();
        if (subpos == 0) {
            ++pos;continue;
        }
        if (!pos_in_string(stringRanges,pos)) {
            setFormat(pos,subpos,keywordFont,keyColor);
        }
        pos+=subpos;
    }
    pos = 0;
    subpos = 0;
    while ( (pos = datatypeWords.search(aText,pos))!=-1) {
        subpos = datatypeWords.matchedLength();
        if (subpos == 0) {
            ++pos;continue;
        }
        setFormat(pos,subpos,datatypeFont,datatypesColor);
        pos+=subpos;
    }
    pos = 0;
    subpos = 0;
    while ( (pos = singleComment.search(aText,pos))!=-1) {
        if (!pos_in_string(stringRanges,pos)) {
            setFormat(pos,singleComment.matchedLength(),commentFont,commentColor);
            break;
        }
        pos+=1;
    }
    return comment_opened;
}

void sqlsyntax::setDefaultFont(const QFont&adefaultFont)
{
    regularFont=adefaultFont;
    bool it,bold;
    it = statementFont.italic();
    bold = statementFont.bold();
    statementFont=regularFont;
    statementFont.setBold(bold);
    statementFont.setItalic(it);
    
    it = keywordFont.italic();
    bold = keywordFont.bold();
    keywordFont=regularFont;
    keywordFont.setBold(bold);
    keywordFont.setItalic(it);
    
    it = datatypeFont.italic();
    bold = datatypeFont.bold();
    datatypeFont=regularFont;
    datatypeFont.setBold(bold);
    datatypeFont.setItalic(it);
    
    it = stringFont.italic();
    bold = stringFont.bold();
    stringFont=regularFont;
    stringFont.setBold(bold);
    stringFont.setItalic(it);

    it = commentFont.italic();
    bold = commentFont.bold();
    commentFont=regularFont;
    commentFont.setBold(bold);
    commentFont.setItalic(it);
}

bool sqlsyntax::pos_in_string(const RangeVector&aVector,int aPos)
{
    RangeVector::const_iterator it=aVector.begin();
    for(;it!=aVector.end();++it) {
        if (aPos>=it->first && aPos<=it->second) {
            return true;
        }
    }
    return false;
}
#endif