/***************************************************************************
                          funitem.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Aug 29 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FUNITEM_H
#define FUNITEM_H

#include <kurl.h>
#include "kplitem.h"

class KSimpleConfig;
class KplCheckListItem;

/**
  * Function item class. Class for function items showing ordinary functions
  * by lines or markers.
  * @author Werner Stille
  */

class FunItem : public ScaledItem {

public:
  /** Constructor */
	FunItem();
  /** Copy constructor */
  FunItem(const FunItem& f);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  FunItem(Kpl::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    * @param url URL of plot file.
    */
  FunItem(KSimpleConfig* plo, Kpl::AutoStruct* aut, const KURL& url);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param symb 0 for circles, < 0 for different markers,
    * >0 for different line types.
    * @param color color for function plot.
    * @param fx normalization factor for x values.
    * @param fy normalization factor for y values.
    * @param xmin minimum argument value.
    * @param xmax maximum argument value.
    * @param dx argument stepsize.
    * @param name function name.
    * @param url URL of the function library.
    */
	FunItem(bool active, int symb, const QString& color,
          double fx, double fy, double xmin, double xmax, double dx,
          const QString& name, const KURL& url);
  /** Destructor */
  virtual ~FunItem();
  /** Assignment operator */
  FunItem& operator=(const FunItem& f);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g) const;
  virtual void writePlo(FILE* f, const KURL& url, bool abs) const;
  virtual void setText(KplCheckListItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual KplItem* copy() const;
  virtual void expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const;
  virtual void minMax(double* xmi, double* xma, double* ymi, double* yma) const;
  virtual void setPar(int i, double value, bool yFun);
  virtual void exportTable(FILE* f, KplDoc* m) const;

protected:
  /**
   * Initializes several elements.
   */
  void init();
  /**
   * Calculates function value table.
   * @param logx true for logarithmic steps
   * @return number of rows.
   */
  int calcTable(bool logx) const;
  mutable bool logxo;
  double tmin, tmax, dt;
  mutable double tmino, tmaxo, dto;
  double (*fkty)(double, const double*);
  mutable double (*fktyo)(double, const double*);
  double py[KPL_NPMAX];
  mutable double pyo[KPL_NPMAX];
  mutable QArray<double> tv, yv;
  QString namey;
  KURL pathy;
  void* hmody;

  friend class FitDlg;
  friend class LMFit;
  friend class FuncDlg;
  friend class KplDoc;
  friend class KplApp;
};

/**
  * Parametric function item class.
  * @author Werner Stille
  */

class ParFunItem : public FunItem {

public:
  /** Constructor */
  ParFunItem();
  /** Copy constructor */
  ParFunItem(const ParFunItem& f);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  ParFunItem(Kpl::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    * @param url URL of plot file.
    */
  ParFunItem(KSimpleConfig* plo, Kpl::AutoStruct* aut, const KURL& url);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param symb 0 for circles, < 0 for different markers,
    * >0 for different line types.
    * @param color color for function plot.
    * @param fx normalization factor for x values.
    * @param fy normalization factor for y values.
    * @param tmin minimum argument value.
    * @param tmax maximum argument value.
    * @param dt argument stepsize.
    * @param namex x function name.
    * @param urlX URL of the x function library.
    * @param namey y function name.
    * @param urlY URL of the y function library.
    */
	ParFunItem(bool active, int symb, const QString& color,
             double fx, double fy, double tmin, double tmax, double dt,
             const QString& namex, const KURL& urlX,
             const QString& namey, const KURL& urlY);
  /** Destructor */
  virtual ~ParFunItem();
  /** Assignment operator */
  ParFunItem& operator=(const ParFunItem& f);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g) const;
  virtual void writePlo(FILE* f, const KURL& url, bool abs) const;
  virtual void setText(KplCheckListItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual KplItem* copy() const;
  virtual void expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const;
  virtual void minMax(double* xmi, double* xma, double* ymi, double* yma) const;
  virtual void setPar(int i, double value, bool yFun);
  virtual void exportTable(FILE* f, KplDoc* m) const;

protected:
  /**
   * Initializes several elements.
   */
  void init();
  /**
   * Calculates function value table.
   * @return number of rows.
   */
  int calcTable() const;
  double (*fktx)(double, const double*);
  mutable double (*fktxo)(double, const double*);
  double px[KPL_NPMAX];
  mutable double pxo[KPL_NPMAX];
  mutable QArray<double> xv;
  QString namex;
  KURL pathx;
  void* hmodx;
  friend class ParFuncDlg;
  friend class KplApp;
};

#endif
