/**************************************************************************
                          cmappath.cpp
                      -------------------
    description          : The path text element
    begin                : Fri Oct 22 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmappath.h"

#include <stdlib.h>
#include <math.h>

/** This is the constuctor for the map paths and
  * used to create a path
  */
CMapPath::CMapPath (direction SrcDir,direction DestDir,CMapRoom *DestRoom,CMapRoom *SrcRoom)
{
	srcDir = SrcDir;
	destDir = DestDir;
	destRoom = DestRoom;
	srcRoom  = SrcRoom;
	setSelected(false);
	bSpecialExit = false;
	specialCmd = "";
	bendList.setAutoDelete(true);
	pointEditing = 0;	
	opsitePath = NULL;
	setEditMode(false);
}	


bool CMapPath::elementIn(int x1,int y1,int x2,int y2,CMapZone *currentZone)
{
	if (srcDir == UP || srcDir == DOWN || srcDir ==SPECIAL || destDir==SPECIAL) return false;
	int startx1,startx2;
 	int starty1,starty2;
 	int endx1,endx2;
 	int endy1,endy2;
	
	QRect rect = getPathRect(currentZone);;

	startx1= rect.left();
	endx1 = rect.right();
	starty1 = rect.top();
	endy1 = rect.bottom();
	
	if (x1<x2)
	{
		startx2 = x1;
		endx2 = x2;	
	}
	else
	{
		startx2 = x2;
		endx2 = x1;
	}

	if (y1<y2)
	{
		starty2 = y1;
		endy2 = y2;	
	}
	else
	{
		starty2 = y2;
		endy2 = y1;
	}
	
	return (startx1>=startx2 && starty1>=starty2 && endx1<=endx2 && endy1<=endy2);
}


/** Check to see if the mouse is in a path edit move box */
signed int CMapPath::mouseInPathSeg(int x,int y,CMapZone *currentZone)
{
	int x1 = getLowX();
	int y1 = getLowY();
	int x2 = getHiX();
	int y2 = getHiY();
	int startx1,endx1,starty1,endy1;
			
	// Is this a inter zone path
	if (getSrcRoom()->getZone()!=currentZone || getDestRoom()->getZone()!=currentZone)
		return-1;
	
	if (bendList.count()==0)
		return -1;
	
	switch(getSrcDir())
	{
		case NORTH     : startx1 = x1; starty1=y1-5;  break;
		case SOUTH     : startx1 = x1; starty1=y1+5;  break;
		case WEST      : startx1 = x1-5; starty1=y1; break;
		case EAST      : startx1 = x1+5; starty1=y1; break;
		case NORTHEAST : startx1 = x1+5; starty1=y1-5; break;
		case SOUTHEAST : startx1 = x1+5; starty1=y1+5; break;
		case SOUTHWEST : startx1 = x1-5; starty1=y1+5; break;
		case NORTHWEST : startx1 = x1-5; starty1=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}		
		
	switch(getDestDir())
	{
		case NORTH     : endx1 = x2  ; endy1=y2-5;  break;
		case SOUTH     : endx1 = x2  ; endy1=y2+5;  break;
		case WEST      : endx1 = x2-5; endy1=y2; break;
		case EAST      : endx1 = x2+5; endy1=y2; break;
		case NORTHEAST : endx1 = x2+5; endy1 = y2-5; break;
		case SOUTHEAST : endx1 = x2+5; endy1 = y2+5; break;
		case SOUTHWEST : endx1 = x2-5; endy1 = y2+5; break;
		case NORTHWEST : endx1 = x2-5; endy1 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	
					
	x1=startx1,y1=starty1;
	int count = 0;
	int d;
	for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
	{
		x2 = point->x();
		y2 = point->y();
			
		x1-=5;
	 	x2+=5;
	 	 	
	 	y1-=5;
	 	y2+=5;

		d = distance (x,y,startx,endx,starty,endy);

		if (d<=5 && (x>=startx && x<=endx && y>=starty && y<=endy))
		{				
			return count;
		}
		x1 = point->x();
		y1 = point->y();
		count++;
	}	

	endx1+=5;
	endy1+=5;
	x1-=5;
	y2-=5;	
	d = distance (x,y,x1,endx1,y1,endy1);

	if (d<=5 && (x>=x1 && x<=endx1 && y>=y1 && y<=endy1))
	{
		return count;
	}
	
	return -1;
}

/** Check to see if the mouse is in a path edit move box */
signed int CMapPath::elementInEditBox(int x,int y,CMapZone *currentZone)
{
	if (srcDir == UP || srcDir == DOWN || srcDir ==SPECIAL || destDir==SPECIAL) return false;
	
	int x1,y1;
	signed int i = 0;

	// Is this a inter zone path
	if (getSrcRoom()->getZone()!=currentZone || getDestRoom()->getZone()!=currentZone)
		return -1;
		
	for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
	{
		x1 =point->x();
		y1 =point->y();		
		if (x>=x1-3 && x<=x1+3 && y>=y1-3 && y<=y1+3)
		{
			return i;
		}
		i++;
	}

	return -1;	
}

void CMapPath::setEditMode(bool edit)
{
	if (bendList.count()>0)
	{
		editMode = edit;
	}
	else
	{
		editMode = false;
	
		if (edit)
			setSelected(false);
	}
}

/** Used to move the path by an offset */
void CMapPath::moveBy(int x,int y)
{		
	for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
	{
		point->setX(point->x()+x);
		point->setY(point->y()+y);		
	}
}

bool CMapPath::mouseInResize(int x,int y,CMapZone *currentZone)
{
	if (srcDir == UP || srcDir == DOWN || srcDir ==SPECIAL || destDir==SPECIAL) return false;
	QRect rect = getPathRect(currentZone);	
 		
	return (x>=rect.right()-3 && x<=rect.right()+3 && y>=rect.bottom()-3 && y<=rect.bottom()+3);	
}

QRect CMapPath::getPathRect(CMapZone *currentZone)
{
	QRect pos;
	int startx1,starty1;
	int endx1,endy1;
	
	int x1,y1,x2,y2;
 	
	if (getDestRoom()->getZone()!=currentZone)
	{
		int room_x2 = getSrcRoom()->getX() * ROOM_SIZE;
		int room_y2 = getSrcRoom()->getY() * ROOM_SIZE;
		int room_x1 = room_x2 - ROOM_SIZE;
		int room_y1 = room_y2 - ROOM_SIZE;		

		switch (getSrcDir())
		{				
			case NORTH     : x1 = room_x1 + HALF_ROOM_SIZE; y1 = room_y1; x2 = room_x1 +HALF_ROOM_SIZE;y2=room_y1-18;break;
			case SOUTH     : x1 = room_x1 + HALF_ROOM_SIZE; y1 = room_y2 + 18; x2 = room_x1 +HALF_ROOM_SIZE;y2=room_y2; break;
			case WEST      : x1 = room_x1; y1 = room_y1 + HALF_ROOM_SIZE; x2 = room_x1 -18;y2=room_y1 + HALF_ROOM_SIZE; break;
			case EAST      : x1 = room_x2; y1 = room_y1 + HALF_ROOM_SIZE; x2 = room_x2 +18;y2=room_y1 + HALF_ROOM_SIZE; break;
			case NORTHEAST : x1 = room_x2; y1 = room_y1; x2 = room_x2 +18;y2=room_y1-18; break;
			case SOUTHEAST : x1 = room_x2; y1 = room_y2; x2 = room_x2 +18;y2=room_y2+18; break;
			case SOUTHWEST : x1 = room_x1; y1 = room_y2; x2 = room_x1 -18;y2=room_y2+18; break;
			case NORTHWEST : x1 = room_x1; y1 = room_y1; x2 = room_x1 -18;y2=room_y1-18; break;		
			case UP        : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}
	}
	else
	{	
		y1 = starty;
		x1 = startx;
		x2 = endx;
		y2 = endy;
	}
	
	if (y1<y2)
	{
		starty1 = y1;
		endy1 = y2;
	}
	else
	{
		starty1 = y2;
		endy1 = y1;
	}

 	if (x1<x2)
	{
		startx1 = x1;
		endx1 = x2;
	}
	else
	{
		startx1 = x2;
		endx1 = x1;
	}	
	
	for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
	{	
		if (point->x()<startx1) startx1 = point->x();
		if (point->y()<starty1) starty1 = point->y();		
		if (point->x()>endx1) endx1 = point->x();
		if (point->y()>endy1) endy1 = point->y();				
	}
	 	
	startx1-=3;
 	endx1+=3;
 	
 	starty1-=3;
 	endy1+=3;
	
	pos.setX(startx1);
	pos.setY(starty1);
	pos.setWidth(endx1-startx1);
	pos.setHeight(endy1-starty1);	
	
	return pos;		
}

int CMapPath::getLargest(int num1,int num2)
{
	if (num1>num2)
		return num1;
	else
		return num2;
}

int CMapPath::getSmallest(int num1,int num2)
{
	if (num1<num2)
		return num1;
	else
		return num2;
}

/** This method is used to calcualte the distance from a path segmeant */
int CMapPath::distance (int x,int y,int x1,int x2,int y1,int y2)
{
	int a = y1 - y2;
	int b = -(x1 - x2);

	double d = ( (x - x2) * a + (y-y2) * b) / sqrt (a*a + b*b);

	return abs((int)d);
}

bool CMapPath::mouseInElement(int x,int y,CMapZone *currentZone)
{
	if (srcDir == UP || srcDir == DOWN || srcDir ==SPECIAL || destDir==SPECIAL) return false;

	int startx1,starty1;
	int endx1,endy1;
	
	int x1,y1,x2,y2;
 	
	if (getDestRoom()->getZone()!=currentZone)
	{
		int room_x2 = getSrcRoom()->getX() * ROOM_SIZE;
		int room_y2 = getSrcRoom()->getY() * ROOM_SIZE;
		int room_x1 = room_x2 - ROOM_SIZE;
		int room_y1 = room_y2 - ROOM_SIZE;		

		switch (getSrcDir())
		{				
			case NORTH     : x1 = room_x1 + HALF_ROOM_SIZE; y1 = room_y1; x2 = room_x1 +HALF_ROOM_SIZE;y2=room_y1-18;break;
			case SOUTH     : x1 = room_x1 + HALF_ROOM_SIZE; y1 = room_y2 + 18; x2 = room_x1 +HALF_ROOM_SIZE;y2=room_y2; break;
			case WEST      : x1 = room_x1; y1 = room_y1 + HALF_ROOM_SIZE; x2 = room_x1 -18;y2=room_y1 + HALF_ROOM_SIZE; break;
			case EAST      : x1 = room_x2; y1 = room_y1 + HALF_ROOM_SIZE; x2 = room_x2 +18;y2=room_y1 + HALF_ROOM_SIZE; break;
			case NORTHEAST : x1 = room_x2; y1 = room_y1; x2 = room_x2 +18;y2=room_y1-18; break;
			case SOUTHEAST : x1 = room_x2; y1 = room_y2; x2 = room_x2 +18;y2=room_y2+18; break;
			case SOUTHWEST : x1 = room_x1; y1 = room_y2; x2 = room_x1 -18;y2=room_y2+18; break;
			case NORTHWEST : x1 = room_x1; y1 = room_y1; x2 = room_x1 -18;y2=room_y1-18; break;		
			case UP        : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}
	}
	else
	{	
		y1 = starty;
		x1 = startx;
		x2 = endx;
		y2 = endy;
	}

	bool result = false;
	
	startx1 = x1;
	endx1 = x2;
	starty1 = y1;
	endy1 = y2;

	// Draw the path
	int tmpx=x1,tmpy=y1;
    int d;
	for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
	{
		startx1 = getSmallest(tmpx,point->x())-3;
		starty1 = getSmallest(tmpy,point->y())-3;		
		endx1 = getLargest(tmpx,point->x())+3;
		endy1 = getLargest(tmpy,point->y())+3;		

		d = distance(x,y,startx1,endx1,starty1,endy1);

		if (d <=5 && (x>=startx1 && x<=endx1 && y>=starty1 && y<=endy1))		
			result = true;
			
		tmpx = point->x();
		tmpy = point->y();
	}

	startx1 = getSmallest(tmpx,x2)-3;
	starty1 = getSmallest(tmpy,y2)-3;		
	endx1 = getLargest(tmpx,x2)+3;
	endy1 = getLargest(tmpy,y2)+3;		

	d = distance (x,y,startx1,endx1,starty1,endy1);	

	if (d <=5 && (x>=startx1 && x<=endx1 && y>=starty1 && y<=endy1))
	{
		result = true;
	}

	return result;
}

CMapElement *CMapPath::copy(void)
{
	CMapPath *path = new CMapPath (srcDir,destDir,destRoom,srcRoom);
	path->setLowHiX(getLowX(),getHiX());
	path->setLowHiY(getLowY(),getHiY());
	path->setSelected(getSelected());
	path->setLowerColour(getLowerColour());
	path->setDefaultColour(getDefaultColour());
	path->setHigherColour(getHigherColour());
	path->setSelectColour(getSelectColour());
	path->setSpecialCmd(getSpecialCmd());
	path->setSpecialExit(getSpecialExit());
	path->setEditColour(getEditColour());
	
	for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
	{
		path->addBend(point->x(),point->y());
	}

	return path;
}

void CMapPath::lowerPaint(QPainter *p,CMapZone *currentZone)
{
	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;

	if (srcDir == UP || srcDir == DOWN || destDir == UP || destDir == DOWN || srcDir==SPECIAL || destDir==SPECIAL) return;
	if (getSrcRoom()->getZone()!=getDestRoom()->getZone()) return;	
	
	x1 = getLowX()-5;
	y1 = getLowY()-5;
	x2 = getHiX()-5;
	y2 = getHiY()-5;

	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	

	switch(getDestDir())
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	
			
	p->setPen(getLowerColour());
	p->setBrush(getLowerColour());
	if (!getDone())
	{		
		// Draw the path
		p->drawLine(x1,y1,x5,y5);
		int x=x5,y=y5;
		for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
		{
			p->drawLine(x,y,point->x()-5,point->y()-5);
			x = point->x()-5;
			y = point->y()-5;
		}
		p->drawLine(x,y,x6,y6);
		p->drawLine(x2,y2,x6,y6);
    }
	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	p->drawPolygon(a,FALSE,0,3);
	setDone(true);
	
	if (getOpsitePath())
		getOpsitePath()->setDone(true);
	
}
      	
void CMapPath::higherPaint(QPainter *p,CMapZone *currentZone)
{
	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;


	if (srcDir == UP || srcDir == DOWN || destDir == UP || destDir == DOWN || srcDir==SPECIAL || destDir==SPECIAL) return;
	if (getSrcRoom()->getZone()!=getDestRoom()->getZone()) return;


	x1 = getLowX()+5;
	y1 = getLowY()+5;
	x2 = getHiX()+5;
	y2 = getHiY()+5;

	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	

	switch(getDestDir())
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	
			
	p->setPen(getHigherColour() );
	p->setBrush(getHigherColour() );
	
	if (!getDone())
	{		
		// Draw the path
		p->drawLine(x1,y1,x5,y5);
	
		int x=x5,y=y5;
		for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
		{
			p->drawLine(x,y,point->x()+5,point->y()+5);
			x = point->x()+5;
			y = point->y()+5;
		}
		p->drawLine(x,y,x6,y6);				
		p->drawLine(x2,y2,x6,y6);
    }
	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	p->drawPolygon(a,FALSE,0,3);
	
	setDone(true);
	if (getOpsitePath())
		getOpsitePath()->setDone(true);
}

void CMapPath::paint(QPainter *p,CMapZone *currentZone)
{	
	if (srcDir == UP || srcDir == DOWN || destDir == UP || destDir == DOWN || srcDir==SPECIAL || destDir==SPECIAL) return;
	
	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;
	int zoneposx,zoneposy;
	direction tempDestDir;

	x1 = getLowX();
	y1 = getLowY();
	x2 = getHiX();
	y2 = getHiY();

	// Get the cords for the start of the path
	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	

	if (getDestRoom()->getZone()!=currentZone)
	{
		switch (getSrcDir())
		{		
			case NORTH     : x2 = x1; y2 = y1 - 10; zoneposx = x2-3; zoneposy = y2-6; tempDestDir = SOUTH; break;
			case SOUTH     : x2 = x1; y2 = y1 + 10; zoneposx = x2-3; zoneposy = y2; tempDestDir = NORTH;break;
			case WEST      : x2 = x1 - 10; y2 = y1; zoneposx = x2-6; zoneposy = y2 -3; tempDestDir = EAST ;break;
			case EAST      : x2 = x1 + 10; y2 = y1; zoneposx = x2; zoneposy = y2-3; tempDestDir = WEST; break;
			case NORTHEAST : x2 = x1 + 9; y2 = y1 - 9;zoneposx = x2+1; zoneposy = y2-7; tempDestDir = SOUTHWEST; break;
			case SOUTHEAST : x2 = x1 + 9; y2 = y1 + 9; zoneposx = x2+1; zoneposy = y2+1; tempDestDir = NORTHWEST; break;
			case SOUTHWEST : x2 = x1 - 9; y2 = y1 + 9; zoneposx = x2-7; zoneposy = y2+1; tempDestDir = NORTHEAST; break;
			case NORTHWEST : x2 = x1 - 9; y2 = y1 - 9; zoneposx = x2-7; zoneposy = y2-7; tempDestDir = SOUTHEAST; break;
			case UP		   : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}
	}
	else
		tempDestDir = getDestDir();
		
	if (getSrcRoom()->getZone()!=currentZone)
	{
	 	// FIXME: draw the map when the start zone is not the current zone
		return;
	}
		
	// Get the cords for the end of the path
	switch(tempDestDir)
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	
		
	p->setPen(getDefaultColour());
	p->setBrush(getDefaultColour() );
	
	// If the path has not already been then draw it
	if (!getDone())
	{		
		// Draw the path
		p->drawLine(x1,y1,x5,y5);	

		int x=x5,y=y5;
		for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
		{
			p->drawLine(x,y,point->x(),point->y());
			x = point->x();
			y = point->y();
		}
	
		p->drawLine(x,y,x6,y6);				
		p->drawLine(x2,y2,x6,y6);
	}

	// Draw the arrow head to indicate the direction of the path
	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	p->drawPolygon(a,FALSE,0,3);
	
	// If the path has not already been drawn then draw it
	if (!getDone())
	{
		// Draw the marker to show that the path goes to anthoer zone
		if (getDestRoom()->getZone()!=getSrcRoom()->getZone())
		{

			p->setPen(getDefaultColour());
			p->setBrush(getDefaultColour() );
			p->drawRect(zoneposx,zoneposy ,7,7);
			
		}
	
		// Draw the selected indicator if nescary
		if (getSelected())
		{
			
			QRect rect = getPathRect(currentZone);
						
			p->setPen(getSelectColour());
			p->setBrush(getSelectColour());
			p->drawRect(rect.right()-3,rect.bottom()-3,6,6);
			
			
		}
	
		// Draw the edit mode indicator if nescary
		if (getEditMode())
		{
			for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
			{
				p->setPen(getEditColour());
				p->setBrush(getEditColour());
			
				p->drawRect(point->x()-3,point->y()-3,6,6);
			}
		}
	}
	
	// Mark this path and the opsite (two way path) as having been drawn
	setDone(true);	
	
	if (getOpsitePath())
	{
		getOpsitePath()->setDone(true);
	}
}

void CMapPath::dragEditPaint(QPainter *p,int offsetx,int offsety,signed int pointNum,CMapZone *currentZone)
{
	if (srcDir == UP || srcDir == DOWN || srcDir ==SPECIAL || destDir==SPECIAL) return;
	
	int startx1,endx1,starty1,endy1;
	int xpos = bendList.at(pointNum)->x()+offsetx;
	int ypos = bendList.at(pointNum)->y()+offsety;	
	
	if (pointNum == 0)
	{
		int x1 = getLowX();
		int y1 = getLowY();

		switch(getSrcDir())
		{
			case NORTH     : startx1 = x1; starty1=y1-5;  break;
			case SOUTH     : startx1 = x1; starty1=y1+5;  break;
			case WEST      : startx1 = x1-5; starty1=y1; break;
			case EAST      : startx1 = x1+5; starty1=y1; break;
			case NORTHEAST : startx1 = x1+5; starty1=y1-5; break;
			case SOUTHEAST : startx1 = x1+5; starty1=y1+5; break;
			case SOUTHWEST : startx1 = x1-5; starty1=y1+5; break;
			case NORTHWEST : startx1 = x1-5; starty1=y1-5; break;
			case UP		   : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}		
	}
	else
	{
		startx1 = bendList.at(pointNum -1)->x();
		starty1 = bendList.at(pointNum -1)->y();		
	}
	
	if (pointNum == (signed int)bendList.count()-1 || bendList.count()==0)
	{		
		int x2 = getHiX();
		int y2 = getHiY();

		switch(getDestDir())
		{
			case NORTH     : endx1 = x2  ; endy1=y2-5;  break;
			case SOUTH     : endx1 = x2  ; endy1=y2+5;  break;
			case WEST      : endx1 = x2-5; endy1=y2; break;
			case EAST      : endx1 = x2+5; endy1=y2; break;
			case NORTHEAST : endx1 = x2+5; endy1 = y2-5; break;
			case SOUTHEAST : endx1 = x2+5; endy1 = y2+5; break;
			case SOUTHWEST : endx1 = x2-5; endy1 = y2+5; break;
			case NORTHWEST : endx1 = x2-5; endy1 = y2-5; break;
			case UP		   : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}	
		
		
	}
	else
	{
		endx1 = bendList.at(pointNum + 1)->x();
		endy1 = bendList.at(pointNum + 1)->y();	
	}
	
	p->drawLine(startx1,starty1,xpos,ypos);	
	p->drawLine(xpos,ypos,endx1,endy1);		
}

void CMapPath::dragPaint(QPainter *p,int offsetx,int offsety,CMapZone *currentZone)
{

	if (srcDir == UP || srcDir == DOWN || srcDir ==SPECIAL || destDir==SPECIAL) return;
	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;
	int zoneposx,zoneposy;
	direction tempDestDir;


	x1 = getLowX() + offsetx;
	y1 = getLowY()+ offsety;
	x2 = getHiX() + offsetx;
	y2 = getHiY()+ offsety;

	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	

	if (getSrcRoom()->getZone()!=currentZone)
	{
	 	// FIXME: draw the map when the start zone is not the current zone
		return;
	}
			
	if (getDestRoom()->getZone()!=currentZone)
	{
		switch (getSrcDir())
		{		
			case NORTH     : x2 = x1; y2 = y1 - 10; zoneposx = x2-3; zoneposy = y2-6; tempDestDir = SOUTH; break;
			case SOUTH     : x2 = x1; y2 = y1 + 10; zoneposx = x2-3; zoneposy = y2; tempDestDir = NORTH;break;
			case WEST      : x2 = x1 - 10; y2 = y1; zoneposx = x2-6; zoneposy = y2 -3; tempDestDir = EAST ;break;
			case EAST      : x2 = x1 + 10; y2 = y1; zoneposx = x2; zoneposy = y2-3; tempDestDir = WEST; break;
			case NORTHEAST : x2 = x1 + 9; y2 = y1 - 9;zoneposx = x2+1; zoneposy = y2-7; tempDestDir = SOUTHWEST; break;
			case SOUTHEAST : x2 = x1 + 9; y2 = y1 + 9; zoneposx = x2+1; zoneposy = y2+1; tempDestDir = NORTHWEST; break;
			case SOUTHWEST : x2 = x1 - 9; y2 = y1 + 9; zoneposx = x2-7; zoneposy = y2+1; tempDestDir = NORTHEAST; break;
			case NORTHWEST : x2 = x1 - 9; y2 = y1 - 9; zoneposx = x2-7; zoneposy = y2-7; tempDestDir = SOUTHEAST; break;
			case UP		   : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}
	}
	else
		tempDestDir = getDestDir();

	
	switch(tempDestDir)
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
		case SPECIAL   : break;
	}	
	
	// Draw the path
	p->drawLine(x1,y1,x5,y5);	
	if (bendList.count()==0)
	{
		p->drawLine(x5,y5,x6,y6);
	}
	else
	{	
		int x=x5,y=y5;
		for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
		{
			p->drawLine(x,y,point->x() + offsetx,point->y()+offsety);
			x = point->x()+ offsetx;
			y = point->y()+ offsety;
		}
		p->drawLine(x,y,x6,y6);
				
	}
	p->drawLine(x2,y2,x6,y6);

	// Draw the arrow head at the end of the path
	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	
	p->drawPolygon(a,FALSE,0,3);
	
	// Draw the marker to show that the path goes to anthoer zone
	if (getDestRoom()->getZone()!=getSrcRoom()->getZone())
	{
		p->setPen(getDefaultColour());
		p->setBrush(getDefaultColour() );
		p->drawRect(zoneposx,zoneposy ,7,7);		
	}
}

/** Delete a bend from the path */
void CMapPath::delBend(int bendNum)
{
	bendList.remove(bendList.at(bendNum));
}

/** Just to add a new bend to the math */
void CMapPath::addBend(int x,int y)
{
	if (bendList.count()==0)
	{
		bendList.append(new QPoint(x,y));
	}
	else
	{
		int x1 = getLowX();
		int y1 = getLowY();
		int x2 = getHiX();
		int y2 = getHiY();
	
		int startx1,endx1,starty1,endy1;
		
		switch(getSrcDir())
		{
			case NORTH     : startx1 = x1; starty1=y1-5;  break;
			case SOUTH     : startx1 = x1; starty1=y1+5;  break;
			case WEST      : startx1 = x1-5; starty1=y1; break;
			case EAST      : startx1 = x1+5; starty1=y1; break;
			case NORTHEAST : startx1 = x1+5; starty1=y1-5; break;
			case SOUTHEAST : startx1 = x1+5; starty1=y1+5; break;
			case SOUTHWEST : startx1 = x1-5; starty1=y1+5; break;
			case NORTHWEST : startx1 = x1-5; starty1=y1-5; break;
			case UP		   : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}		
		
		switch(getDestDir())
		{
			case NORTH     : endx1 = x2  ; endy1=y2-5;  break;
			case SOUTH     : endx1 = x2  ; endy1=y2+5;  break;
			case WEST      : endx1 = x2-5; endy1=y2; break;
			case EAST      : endx1 = x2+5; endy1=y2; break;
			case NORTHEAST : endx1 = x2+5; endy1 = y2-5; break;
			case SOUTHEAST : endx1 = x2+5; endy1 = y2+5; break;
			case SOUTHWEST : endx1 = x2-5; endy1 = y2+5; break;
			case NORTHWEST : endx1 = x2-5; endy1 = y2-5; break;
			case UP		   : break;
			case DOWN      : break;
			case SPECIAL   : break;
		}	
					
		x1=startx1,y1=starty1;
		int count = 0;
		for (QPoint *point = bendList.first(); point !=0 ; point= bendList.next())
		{
			x2 = point->x();
			y2 = point->y();
		 	if (x1 == x2)
		 	{
 				x1-=3;
		 		x2+=3;
		 	}
 	
		 	if (y1 == y2)        	
		 	{
		 		y1-=3;
		 		y2+=3;
		 	}
		
			if (x>=x1 && y>=y1 && x<=x2 && y<=y2)
			{
				bendList.insert(count+1,new QPoint(x,y));
				return;
			}
			x1 = point->x();
			y1 = point->y();
			count++;
		}	
		bendList.append(new QPoint(x,y));
	}
}

/** Moves the bend */
void CMapPath::moveBend(int bendNum, int x, int y)
{
	if (bendNum < (signed int)bendList.count())
	{
		QPoint *p=bendList.at(bendNum);
		p->setX(x);
		p->setY(y);
	}
}

void CMapPath::read(KCMapFile* kcmfile)
{
  QString bendnum;
  QString bendpre("bend-");
  int count;
  int num_bends;
  QPoint* bendPoint;

  bSpecialExit = kcmfile->readBoolEntry("special exit", false);
  specialCmd = kcmfile->readEntry("special command", "");

  num_bends = kcmfile->readNumEntry("bends", 0);
  bendList.clear();
  for (count = 0; count < num_bends; count++)
  {
    bendnum.setNum(count+1);
    bendPoint = new QPoint(kcmfile->readPointEntry(bendpre+bendnum));
    bendList.append(bendPoint);
  }
}

void CMapPath::write(KCMapFile* kcmfile)
{
  QString bendnum;
  QString bendpre("bend-");
  int count;
  QPoint* bendPoint;

  kcmfile->writeEntry("srcX", srcRoom->getX());
  kcmfile->writeEntry("srcY", srcRoom->getY());
  kcmfile->writeEntry("srcLevel", srcRoom->getLevel());
  kcmfile->writeEntry("srcDir", getSrcDir());

  kcmfile->writeEntry("destX", destRoom->getX());
  kcmfile->writeEntry("destY", destRoom->getY());
  kcmfile->writeEntry("destLevel", destRoom->getLevel());
  kcmfile->writeEntry("destDir", getDestDir());

  kcmfile->writeEntry("special exit", bSpecialExit);
  kcmfile->writeEntry("special command", specialCmd);

  kcmfile->writeEntry("bends", bendList.count());
  count = 0;
  for (bendPoint=bendList.first(); bendPoint != NULL; bendPoint=bendList.next())
  {
    count++;
    bendnum.setNum(count);
    kcmfile->writeEntry(bendpre+bendnum, *bendPoint);
  }
}
