/***************************************************************************
                          kaspabase.cpp  -  description
                             -------------------
    begin                : Tue Sep 7 1999
    copyright            : (C) 1999 by Jan Mueller
    email                : janmueller7@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kaspabase.h"
#include <assert.h>
#include <time.h>


#define Inherited Sql

#define TYPEAUTHOR 0
#define TYPEPUBL   1
#define TYPEPART   2
#define TYPENOTE   3
#define TYPEFILE   4


////////////////////////////////////////////////////////
//// 	Field

 	Field::Field():changed(false) {}
  Field::~Field() {}
  void Field::setChanged() { changed=TRUE; }
  void Field::setUnchanged() { changed=FALSE; }
  bool Field::hasChanged() const { return changed; }

////////////////////////////////////////////////////////
//// DataField

  DataField::DataField(DataField& T):Field() { set(T.get(), T.length()); }

  DataField::DataField():Field(), d(0), len(0) { }

  DataField::~DataField() { del(); }

  DataField& DataField::operator=(const DataField& T) {
		if(!(this==&T)) {
			Field::operator=(T);
			set(T.get(), T.length());
		}
		return *this;
	}

  void DataField::empty() { del(); d=0; }

  long DataField::length() const { return len; }

  const unsigned char *DataField::get() const { return d; }

  void DataField::set(const unsigned char *buf, long l) {
    del();
    len=l;
    if(!len) { d=0; return; }
    d=new unsigned char[len];
    memcpy(d, buf, len);
  }	

  void DataField::take(unsigned char *buf, long l) {
    len=l;
    d=buf;
  }	

  void DataField::del() { delete d; d=0; }


////////////////////////////////////////////////////////
//// TextField
  TextField::TextField(TextField& T): Field() { set(T.get()); }

  TextField::TextField():Field(), t(0) { empty(); }

  TextField::~TextField() { del(); }

	TextField& TextField::operator=(TextField &T) {
		if(!(this==&T)) {
			Field::operator=(T);
			set(T.get());
		}
		return *this;
	}

  void TextField::empty() { del(); t=new char[1]; *t=0; }

  const char *TextField::get() const { return t; }

	char *TextField::getSql() const { return esc4sql(t); }

	void TextField::getSql(Str &s) const { char *tmp=esc4sql(t); s+=tmp; delete tmp; }

  void TextField::set(const char *text) {
    del();
    if(!text) { t=0; return; }
    t=new char[strlen(text)+1];
    strcpy(t, text);
  }	

	void TextField::append(const char *text) {
		char *s=new char[strlen(t)+strlen(text)+1];
		strcpy(s, t); strcat(s, text);
		del();
		set(s); delete s;
	}

  void TextField::take(char *text) { t=text; }		

  void TextField::del() { if(t) delete t; t=0; }

////////////////////////////////////////////////////////////////////////////
//// IntField

  IntField::IntField(IntField& I):Field(), i(I.get()) {};
  IntField::IntField(): Field(), i(0) {}
  void IntField::empty() { i=0; }
  int IntField::get() const { return i; }
  void IntField::get(Str& s) const {
  	char *c;
  	asprintf(&c, "%i", i);
		s+=c;
		free(c);
	}	
  void IntField::set(int j) { i=j; }
  void IntField::set(const char *s) { i=strtol(s,0,0); }

////////////////////////////////////////////////////////////////////////////
//// BoolField

//  BoolField::BoolField(BoolField& I):Field(), i(I.get()) {};

  BoolField::BoolField(): Field(), i(false) {}

//	BoolField& BoolField::operator=(BoolField& f) {  Field::operator=(t); i=f.get(); };

  void BoolField::empty() { i=false; }

  void BoolField::get(Str &s) const { if(i) s+="t"; else s+="f"; }

  bool BoolField::get() const { return i; }

  void BoolField::set(bool j) { i=j; }

  void BoolField::set(const char *s) { if(!strcmp(s, "t")) i=true; else i=false; }


////////////////////////////////////////////////////////////////////////////
//// OidField

//  OidField::OidField(OidField& O): Field(), o(O.get()) {}

  OidField::OidField(): Field(), o(InvalidOid) {}

  void OidField::empty() { o=InvalidOid; }

  Oid OidField::get() const { return o; }

  void OidField::get(Str& s) const { s+=o; }

  void OidField::set(Oid i) { o=i; }

  void OidField::set(const char *s) { o=str2oid(s); }

////////////////////////////////////////////////////////////////////////////
//// KaspaBase

KaspaBase::KaspaBase(Sql *conn): Inherited(*conn) {
}

KaspaBase::KaspaBase(const char *dbname): Inherited(dbname) {
}

KaspaBase::~KaspaBase() {
}

LockTableItem *KaspaBase::locktable=0L;

/*******************************************************
*              Author
*******************************************************/
Oid KaspaBase::newAuthor() {
  Oid ret;
  try {
    exec("begin");
    Str s("insert into author values ('', '', '', '', '', '', 0, 0,'");
		s+=getTime(); s+="','";
		s+=getTime(); s+="')";
    exec(s);
    ret=oidStatus();
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deleteAuthor(const char *where) {
		// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("author", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteLo("author", "memo", where);
    deleteLo("author", "picture", where);
    deleteTuples("author", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

AuthorRec *KaspaBase::getAuthor(const char *where) {
  AuthorRec *last=0L;
  try {
    exec("begin");
    Str s("select oid, lastname, firstname, pseudonym, born, died, country, memo, picture, modified, created from author ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      AuthorRec *rec= new AuthorRec;
      rec->id.set(getValue(i, "oid"));
      rec->lastname.set(getValue(i, "lastname"));
      rec->firstname.set(getValue(i, "firstname"));
      rec->born.set(getValue(i, "born"));
      rec->died.set(getValue(i, "died"));
      rec->country.set(getValue(i, "country"));
			rec->modified.set(getValue(i, "modified"));
			rec->created.set(getValue(i, "created"));
      rec->memo.take(lo2str(str2oid(getValue(i, "memo"))));
      rec->pseudonym.set(getValue(i, "pseudonym"));
      lo2buf(str2oid(getValue(i, "picture")), rec->picture);
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}

void KaspaBase::updateAuthor(AuthorRec *author, const char *where, bool block) {
  try {
    if(author->lastname.hasChanged()+author->firstname.hasChanged()+author->pseudonym.hasChanged()+
       author->country.hasChanged()+author->born.hasChanged()+author->died.hasChanged()+
       author->picture.hasChanged()+author->memo.hasChanged()==0) return;

    exec("begin", block);

    Str s="update author set ";
    if(author->lastname.hasChanged())  { s+="lastname ='"; author->lastname.getSql(s);  s+="', ";}
    if(author->firstname.hasChanged()) { s+="firstname='"; author->firstname.getSql(s); s+="', ";}
    if(author->country.hasChanged())   { s+="country='";   author->country.getSql(s);   s+="', ";}
    if(author->born.hasChanged()) 		 { s+="born='";      author->born.getSql(s);      s+="', ";}
    if(author->died.hasChanged()) 		 { s+="died='"; 		 author->died.getSql(s); 		  s+="', ";}
    if(author->pseudonym.hasChanged()) { s+="pseudonym='"; author->pseudonym.getSql(s); s+="', ";}
    if(author->memo.hasChanged()) 		 {
	    deleteLo("author", "memo", where);
			s+="memo=";
			s+=str2lo(author->memo.get());
			s+=", ";
		}
    if(author->picture.hasChanged()) 	 {
	    deleteLo("author", "picture", where);
      s+="picture=";
      s+=buf2lo((char*)author->picture.get(), author->picture.length());
			s+=", ";
    }
		s+="modified='"; s+=getTime(); s+="', ";

    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    author->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*********************************************************************
 * Publications
 **********************************************************************/
Oid KaspaBase::newPubl() {
  Oid ret;
  try {
    exec("begin");
    Str s("insert into publication values ('','',0,'','','',0,'','','','','','','','','','',0,'','','','',0,0,'t','");
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=oidStatus();
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePubl(Oid o) {
		if(isLocked(o)) throw BaseDeleteLock("Can't delete records - one is locked!");
    Str s("where part.publicationno=");
    s+=o;
    deletePart(s);
  try {
    exec("begin");
		Str s="where oid=";
		s+=o;
    deleteLo("publication", "memo", s);
    deleteTuples("publication", s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::deletePubl(const char *where) {
		// Won't kill the whole table...
		if(!where || strlen(where)==0) abort();
		if(isLocked("publication", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
    Str s("where part.publicationno in (select oid from publication ");
		s+=where;
		s+=")";
    deletePart(s);
  try {
    exec("begin");
    deleteLo("publication", "memo", where);
    deleteTuples("publication", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

PublRec *KaspaBase::getPubl(const char *where) {
  PublRec *last=0L;
  try {
    exec("begin");
    Str s("select oid,* from publication ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PublRec *rec= new PublRec;
      rec->title.set(getValue(i, "title"));
      rec->subtitle.set(getValue(i, "subtitle"));
      rec->edition.set(getValue(i, "edition"));
      rec->editor.set(getValue(i, "editor"));
      rec->howpublished.set(getValue(i, "howpublished"));
      rec->organization.set(getValue(i, "organization"));
      rec->publisher.set(getValue(i, "publisher"));
      rec->year.set(getValue(i, "year"));
      rec->pages.set(getValue(i, "pages"));
      rec->translator.set(getValue(i, "translator"));
      rec->volume.set(getValue(i, "volume"));
      rec->number.set(getValue(i, "number"));
      rec->month.set(getValue(i, "month"));
      rec->series.set(getValue(i, "series"));
      rec->type.set(getValue(i, "type"));
      rec->key.set(getValue(i, "key"));
      rec->orgtitle.set(getValue(i, "orgtitle"));
      rec->orgpublisher.set(getValue(i, "orgpublisher"));
      rec->orgyear.set(getValue(i, "orgyear"));
      rec->isbn_issn.set(getValue(i, "isbn_issn"));
      rec->hidingplace.set(getValue(i, "hidingplace"));
      rec->bibtex.set(getValue(i, "bibtex"));
      rec->createbibtex.set(getValue(i, "createbibtex"));
      rec->entrytype.set(getValue(i, "entrytype"));
      rec->modified.set(getValue(i, "modified"));
      rec->created.set(getValue(i, "created"));
      rec->memo.take(lo2str(str2oid(getValue(i, "memo"))));
      rec->id.set(getValue(i, "oid"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


void KaspaBase::updatePubl(PublRec *publ, const char *where, bool block) {
  try {
    if(publ->title.hasChanged()  + publ->subtitle.hasChanged()     + publ->edition.hasChanged()+
	 publ->editor.hasChanged()     + publ->howpublished.hasChanged() + publ->organization.hasChanged()+
	 publ->publisher.hasChanged()  + publ->year.hasChanged()         + publ->pages.hasChanged()+
	 publ->translator.hasChanged() + publ->volume.hasChanged()       + publ->number.hasChanged()+
	 publ->month.hasChanged()      + publ->series.hasChanged()       + publ->type.hasChanged()+
	 publ->key.hasChanged()        + publ->orgtitle.hasChanged()     + publ->orgpublisher.hasChanged()+
	 publ->orgyear.hasChanged()    + publ->isbn_issn.hasChanged()    + publ->hidingplace.hasChanged()+
	 publ->memo.hasChanged()       + publ->entrytype.hasChanged()    + publ->bibtex.hasChanged()+
	 publ->createbibtex.hasChanged()== 0 ) return;

    exec("begin", block);
    Str s="update publication set ";
    if(publ->title.hasChanged())        {s+="title='";        publ->title.getSql(s);							 s+="', ";}
    if(publ->subtitle.hasChanged())     {s+="subtitle='"; 		publ->subtitle.getSql(s);					 s+="', ";}
    if(publ->edition.hasChanged())      {s+="edition=";  			publ->edition.get(s);	  							 s+=", ";}
    if(publ->editor.hasChanged())       {s+="editor='";   		publ->editor.getSql(s);  					 s+="', ";}
    if(publ->howpublished.hasChanged()) {s+="howpublished='"; publ->howpublished.getSql(s); 					 s+="', ";}
    if(publ->organization.hasChanged()) {s+="organization='"; publ->organization.getSql(s);					 s+="', ";}
    if(publ->publisher.hasChanged())    {s+="publisher="; 		publ->publisher.get(s); 	 						 s+=", ";}
    if(publ->year.hasChanged())         {s+="year='"; 				publ->year.getSql(s); 									 s+="', ";}
    if(publ->pages.hasChanged())        {s+="pages='"; 				publ->pages.getSql(s); 								 s+="', ";}
    if(publ->translator.hasChanged())   {s+="translator='"; 	publ->translator.getSql(s); 						 s+="', ";}
    if(publ->volume.hasChanged())       {s+="volume='"; 			publ->volume.getSql(s); 								 s+="', ";}
    if(publ->number.hasChanged())       {s+="number='"; 			publ->number.getSql(s); 								 s+="', ";}
    if(publ->month.hasChanged())        {s+="month='"; 				publ->month.getSql(s); 								 s+="', ";}
    if(publ->series.hasChanged())       {s+="series='";			  publ->series.getSql(s); 								 s+="', ";}
    if(publ->type.hasChanged())         {s+="type='"; 				publ->type.getSql(s); 									 s+="', ";}
    if(publ->key.hasChanged())          {s+="key='"; 					publ->key.getSql(s); 									 s+="', ";}
    if(publ->orgtitle.hasChanged())     {s+="orgtitle='"; 		publ->orgtitle.getSql(s); 							 s+="', ";}
    if(publ->orgpublisher.hasChanged()) {s+="orgpublisher=";  publ->orgpublisher.get(s); 						 s+=", ";}
    if(publ->orgyear.hasChanged())      {s+="orgyear='"; 			publ->orgyear.getSql(s); 							 s+="', ";}
    if(publ->isbn_issn.hasChanged())    {s+="isbn_issn='"; 		publ->isbn_issn.getSql(s); 						 s+="', ";}
    if(publ->hidingplace.hasChanged())  {s+="hidingplace='"; 	publ->hidingplace.getSql(s); 					 s+="', ";}
    if(publ->memo.hasChanged())         {
      deleteLo("publication", "memo", where);
			s+="memo="; 					
			s+=str2lo(publ->memo.get());
			s+=", ";
		}
    if(publ->bibtex.hasChanged())       {s+="bibtex='"; 			 s+=publ->bibtex.get();                 s+="', ";}
    if(publ->createbibtex.hasChanged()) { s+="createbibtex='"; publ->createbibtex.get(s);  					 s+="', ";}
    if(publ->entrytype.hasChanged() )   {s+="entrytype="; 		 publ->entrytype.get(s); 							 s+=", ";};
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    publ->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*************************************************************************
 *  Note
 **************************************************************************/
Oid KaspaBase::newNote(Oid o=InvalidOid) {
  Oid ret;
  try {
    exec("begin");
		Str s;
		if(o==InvalidOid)
			s="insert into note values ('',0,0,";
		else
			s="insert into note values ('',0,1,";
		s+=o;
		s+=",'";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=oidStatus();
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deleteNote(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("note", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteLo("note", "memo", where);
    deleteTuples("note", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

NoteRec *KaspaBase::getNote(const char *where) {
  NoteRec *last=0L;
  try {
    exec("begin");
    Str s("select oid, title, memo, type, publicationno, modified, created from note ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      NoteRec *rec= new NoteRec;
      rec->id.set(getValue(i, "oid"));
      rec->title.set(getValue(i, "title"));
      rec->publno.set(getValue(i, "publicationno"));
      rec->type.set(getValue(i, "type"));
      rec->created.set(getValue(i, "created"));
      rec->modified.set(getValue(i, "modified"));
      rec->memo.take(lo2str(str2oid(getValue(i, "memo"))));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}

void KaspaBase::updateNote(NoteRec *note, const char *where, bool block) {
  try {
    if(note->title.hasChanged()+note->memo.hasChanged()+note->type.hasChanged()+
       note->publno.hasChanged()==0) return;

    exec("begin", block);  		

    Str s="update note set ";
    if(note->title.hasChanged())  { s+="title ='"; note->title.getSql(s); s+="', ";}
    if(note->type.hasChanged())   { s+="type=";    note->type.get(s); s+=", ";}
    if(note->publno.hasChanged()) { s+="publicationno=";  note->publno.get(s); s+=", ";}
    if(note->memo.hasChanged())   {
			s+="memo=";
			s+=str2lo(note->memo.get());
			s+=", ";
	    deleteLo("note", "memo", where);
		}
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    note->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*************************************************************************
 *  Part
 **************************************************************************/
Oid KaspaBase::newPart(Oid publ) {
  Oid ret;
  try {
    exec("begin");
    Str s="insert into part values (";
    s+=publ;
    s+=",0,'','',0,'','','',0,'t','";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=oidStatus();
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePart(const char *where) {
		// Won't kill the whole table...
		if(!where || strlen(where)==0) abort();
		Str s;
		if(isLocked("part", where)) throw BaseDeleteLock("Can't delete records - one is locked!");

		if(strlen(where)>5) {
	    s="where partdata.partno=part.oid and ";
			s+=(where+5);
		} else {
//			debug("Kill all Parts?!?!?! - Abort");
			abort();
		}
    deletePartData(s);	
  try {
    exec("begin");
    deleteLo("part", "memo", where);
    deleteTuples("part", where);
    exec("commit");
  } catch(...) {
  	exec("rollback");
	throw;
  }
}

PartRec *KaspaBase::getPart(const char *where) {
  PartRec *last=0L;
  try {
    exec("begin");
    Str s("select oid, publicationno, partno, title, " \
					"pages, key, intro, bibtex, memo, language, "\
					"createbibtex, created, modified from part ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PartRec *rec= new PartRec;
      rec->id.set(getValue(i, "oid"));
      rec->partno.set(getValue(i, "partno"));
      rec->publno.set(getValue(i, "publicationno"));
      rec->title.set(getValue(i, "title"));
      rec->pages.set(getValue(i, "pages"));
      rec->key.set(getValue(i, "key"));
      rec->intro.set(getValue(i, "intro"));
      rec->bibtex.set(getValue(i, "bibtex"));
      rec->createbibtex.set(getValue(i, "createbibtex"));
      rec->memo.take(lo2str(str2oid(getValue(i, "memo"))));
      rec->modified.set(getValue(i, "modified"));
      rec->created.set(getValue(i, "created"));
      rec->language.set(getValue(i, "language"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


void KaspaBase::updatePart(PartRec *part, const char *where, bool block) {
  try {
    if(part->title.hasChanged()+part->key.hasChanged()+part->partno.hasChanged()+
       part->pages.hasChanged()+part->intro.hasChanged()+
       part->memo.hasChanged()+part->language.hasChanged()+
			 part->bibtex.hasChanged()+part->createbibtex.hasChanged()==0) return;

    exec("begin", block);			
    Str s="update part set ";
    if(part->title.hasChanged()) { s+="title ='"; part->title.getSql(s); s+="', ";}
    if(part->language.hasChanged()) { s+="language ='"; part->language.getSql(s); s+="', ";}
    if(part->key.hasChanged()) { s+="key='"; part->key.getSql(s); s+="', ";}
    if(part->partno.hasChanged()) { s+="partno="; part->partno.get(s); s+=", ";}
	  if(part->publno.hasChanged()) { s+="publno="; part->publno.get(s); s+=", ";}
    if(part->pages.hasChanged()) { s+="pages='"; part->pages.getSql(s); s+="', ";}
    if(part->intro.hasChanged()) { s+="intro="; part->intro.get(s); s+=", ";}
    if(part->memo.hasChanged()) {
      deleteLo("part", "memo", where);
			s+="memo=";
			s+=str2lo(part->memo.get());
			s+=", ";
		}

    if(part->bibtex.hasChanged()) { s+="bibtex='"; s+=part->bibtex.get(); s+="', ";}
    if(part->createbibtex.hasChanged()) { s+="createbibtex='"; part->createbibtex.get(s); s+="', ";}
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    part->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*************************************************************************
 *  PartData
 **************************************************************************/
Oid KaspaBase::newPartData(Oid part) {
  Oid ret;
  try {
    exec("begin");
		Str s="insert into partdata values (";
		s+=part;
		s+=", 0, '', '','";
		s+=getTime();
 		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=oidStatus();
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePartData(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("partdata", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteLo("partdata", "file", where);
    deleteTuples("partdata", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

PartDataRec *KaspaBase::getPartData(const char *where) {
  PartDataRec *last=0L;
  try {
    exec("begin");
    Str s("select oid, file, filename, type, partno from partdata ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PartDataRec *rec= new PartDataRec;
      rec->id.set(getValue(i, "oid"));
      lo2buf(str2oid(getValue(i, "file")), rec->file);
      rec->filename.set(getValue(i, "filename"));
      rec->type.set(getValue(i, "type"));
      rec->partno.set(getValue(i, "partno"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


void KaspaBase::updatePartData(PartDataRec *partdata, const char *where, bool block) {
  try {
    if(partdata->file.hasChanged()+partdata->filename.hasChanged()+partdata->type.hasChanged()+
       partdata->partno.hasChanged()==0) return;

    exec("begin", block);
    deleteLo("partdata", "file", where);

    Str s="update partdata set ";
    if(partdata->file.hasChanged())     {
      s+="file =";
      s+=buf2lo((char*)partdata->file.get(), partdata->file.length());
			s+=", ";
    }
    if(partdata->filename.hasChanged()) { s+="filename='"; partdata->filename.getSql(s); s+="', ";}
    if(partdata->type.hasChanged())     { s+="type='"; partdata->type.getSql(s); s+="', ";}
    if(partdata->partno.hasChanged())   { s+="partno="; partdata->partno.get(s); s+=", ";}
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    partdata->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}

unsigned char *KaspaBase::getPartDataImage(const char *name, Oid part, long *l) {
	try {
		exec("begin");
		Str s("select file from partdata where filename='");
		s+=name;
		s+="' and partno=";
		s+=part;
		exec(s);
		Oid partdata=InvalidOid;
		for(int i=0; i<tuples(); i++)
			partdata=str2oid(getValue(i, "file"));
		unsigned char *buf;

		if(partdata!=InvalidOid)
			buf=(unsigned char *)lo2buf(partdata, l);
		else {
			*l=0;
			buf=0L;
		}

		exec("commit");
		return buf;
  } catch (...) {
    exec("rollback");
    throw;
  }
}

/*************************************************************************
 *  Publisher
 **************************************************************************/
Oid KaspaBase::newPublisher() {
  Oid ret;
  try {
    exec("begin");
    Str s("insert into publisher values ('','', '','");
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=oidStatus();
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePublisher(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("publisher", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("publisher", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

PublisherRec *KaspaBase::getPublisher(const char *where) {
  PublisherRec *last=0L;
  try {
    exec("begin");
    Str s("select oid, name, city, serie, modified, created from publisher ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PublisherRec *rec= new PublisherRec;
      rec->id.set(getValue(i, "oid"));
      rec->name.set(getValue(i, "name"));
      rec->city.set(getValue(i, "city"));
      rec->serie.set(getValue(i, "serie"));
      rec->modified.set(getValue(i, "modified"));
      rec->created.set(getValue(i, "created"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


void KaspaBase::updatePublisher(PublisherRec *publisher, const char *where, bool block) {
  try {
    if(publisher->name.hasChanged()+publisher->city.hasChanged()+publisher->serie.hasChanged()==0) return;

    exec("begin", block);

    Str s="update publisher set ";
    if(publisher->name.hasChanged()) { s+="name='"; publisher->name.getSql(s); s+="', ";}
    if(publisher->city.hasChanged()) { s+="city='"; publisher->city.getSql(s); s+="', ";}
    if(publisher->serie.hasChanged()) { s+="serie='"; publisher->serie.getSql(s); s+="', "; }
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    publisher->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*************************************************************************
 *
 **************************************************************************/
void KaspaBase::newPublAuthor(Oid publ, Oid author) {
  try {
    exec("begin");
    Str s="insert into publication_author values (";
    s+=publ;
    s+=",";
		s+=author;
    s+=",'";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}	

void KaspaBase::deletePublAuthor(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("publication_author", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("publication_author", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::newPartAuthor(Oid part, Oid author) {
  try {
    exec("begin");
    Str s="insert into part_author values (";
    s+=part;
    s+=",";
    s+=author;
		s+=",'";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::deletePartAuthor(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("part_author", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("part_author", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::newLink(Oid obj1, Oid obj2, const char *table) {
  try {
    exec("begin");
    Str s="insert into link values (";
    s+=obj1;
    s+=",";
    s+=obj2;
    s+=",'";
    s+=table;
    s+="','";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::deleteLink(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("link", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("link", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}



//*******************************************************
void KaspaBase::buildGetLinkClause(const char *table, Oid obj, Str& s) {
  s="where (";
  s+=mystrdup(table);
  s+=".oid=link.obj1 and link.obj2=";
  s+=obj;
  s+=") or (";
  s+=table;
  s+=".oid=link.obj2 and link.obj1=";
  s+=obj;
  s+=")";
}

Title *KaspaBase::getLinks(Oid obj) {
  Title *rec=0L;
  try {
    Str s;
    buildGetLinkClause("author", obj, s);
    rec=getAuthorNames(s);

    buildGetLinkClause("publication", obj, s);
    if(rec) rec->append(getPublTitles(s));
		else rec=getPublTitles(s);

    buildGetLinkClause("part", obj, s);
    if(rec) rec->append(getPartTitles(s));
		else rec=getPartTitles(s);

    buildGetLinkClause("partdata", obj, s);
    if(rec) rec->append(getPartDataNames(s));
		else rec=getPartDataNames(s);

    buildGetLinkClause("note", obj, s);
    if(rec) rec->append(getNoteTitles(s));
		else rec=getNoteTitles(s);
  } catch(...) { throw; }

  return rec;
}
//******************************************************


struct oid_chain {
	Oid o;
	oid_chain *next;
	~oid_chain() { delete next; };
};

void KaspaBase::deleteLo(const char *table, const char *field, const char *where) {
  Str s("select ");
  s+=field;
  s+=" from ";
  s+=table;
  s+=" ";
  s+=where;
  exec(s);
  int i;
	oid_chain *last=0L;
  for(i=0; i<tuples(); i++){
    oid_chain *o_id=new oid_chain;
		o_id->o=str2oid(getValue(i, field));
		o_id->next=last;
		last=o_id;
	}
	for(oid_chain *oc=last; oc; oc=oc->next) {
    if(oc->o!=InvalidOid) {
      lUnlink(oc->o);
			Str t("delete from analyzed where obj=");
			t+=oc->o;
			exec(t);
			t="delete from word where obj=";
			t+=oc->o;
			exec(t);
		}
  }
	delete last;
} 	

bool KaspaBase::isLocked(const char *table, const char *where) {
	Str s("select oid from ");
	s+=table;
	s+=" ";
	s+=where;
	exec(s, false);
	bool locked=false;
	for(int i=0; i<tuples(); i++) {
		Oid o=str2oid(getValue(i, "oid"));
		locked=isLocked(o);
	}
  return locked;
}

bool KaspaBase::isLocked(Oid o) {

	LockTableItem *i;
	for(i=locktable; i; i=i->next)
		if(i->id==o) return true;
	return false;

/*	if(!lockObj(o))	return true; // old lock mechanism, needs user_locks
	else {
		unlockObj(o);
		return false;
	} */
}

void KaspaBase::deleteTuples(const char *table, const char *where) {
	// Won't delete the whole table...
	if(!where || strlen(where)==0) abort();

  Str s("delete from ");
  s+=table;
  s+=" ";
  s+=where;
  exec(s, false);
} 	


bool KaspaBase::lockObj(Oid o) {
	if(isLocked(o)) return false;
	LockTableItem *lock=new LockTableItem;
	lock->id=o;
	lock->next=locktable;
	locktable=lock;
	return true;

/*  int ret=0;             // old lock mechanism, needs user_locks
  exec("begin");
  Str s="select user_write_lock_oid(";
  s+=o;
  s+=") AS locked";
  exec(s);
  if(tuples()==1)
    ret=strcmp(getValue(0, "locked"), "0");
  exec("commit");
  return ret; */
}


bool KaspaBase::unlockObj(Oid o) {
	LockTableItem *prev=0L;
	for(LockTableItem *lock=locktable; lock; prev=lock, lock=lock->next)
		if(lock->id==o) {
			if(prev) prev->next=lock->next;
			else locktable=lock->next;
			delete lock;
			return true;
		}
	return false;


/*  int ret=0;       // old lock mechanism, needs user_locks
  exec("begin");
  Str s("select user_write_unlock_oid(");
  s+=o;
  s+=") AS unlocked";
  exec(s);
  if(tuples()==1)
    ret=strcmp(getValue(0, "unlocked"), "0");
  exec("commit");
  return ret; */
}

AuthorName *KaspaBase::getAuthorNames(const char *where) {
  AuthorName *last=0;
  try {
    exec("begin");
    Str s("select oid, firstname, lastname from author ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      AuthorName *rec=new AuthorName;
      rec->firstname.set(getValue(i, "firstname"));
      rec->lastname.set(getValue(i, "lastname"));
      rec->title.set(getValue(i, "firstname"));
      rec->title.append(" ");
      rec->title.append(getValue(i, "lastname"));
      rec->table.set("author");
      rec->o.set(getValue(i, "oid"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PublTitle *KaspaBase::getPublTitles(const char *where) {
  PublTitle *last=0;
  try {
    exec("begin");
    Str s("select oid, year, title, entrytype from publication ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      PublTitle *rec=new PublTitle;
      rec->title.set(getValue(i, "title"));
      rec->year.set(getValue(i, "year"));
      rec->table.set("publication");
      rec->o.set(getValue(i, "oid"));
			rec->entrytype.set(getValue(i, "entrytype"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PartTitle *KaspaBase::getPartTitles(const char *where) {
  PartTitle *last=0;
  try {
    exec("begin");
    Str s("select oid, title from part ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      PartTitle *rec=new PartTitle;
      rec->title.set(getValue(i, "title"));
      rec->table.set("part");
      rec->o.set(getValue(i, "oid"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PartDataName *KaspaBase::getPartDataNames(const char *where) {
  PartDataName *last=0;
  try {
		exec("begin");
		Str s("select oid, filename from partdata ");
		s+=where;
		exec(s);
		for(int i=0; i<tuples(); i++) {
			PartDataName *rec=new PartDataName;
			rec->title.set(getValue(i, "filename"));
			rec->filename.set(getValue(i, "filename"));
			rec->table.set("partdata");
			rec->o.set(getValue(i, "oid"));
			rec->next=last;
			last=rec;
		}
		exec("commit");
	} catch(...) {
		delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


NoteTitle *KaspaBase::getNoteTitles(const char *where) {
  NoteTitle *last=0;
  try {
    exec("begin");
    Str s("select oid, title, publicationno from note ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      NoteTitle *rec=new NoteTitle;
      rec->title.set(getValue(i, "title"));
      rec->table.set("note");
      rec->publno.set(getValue(i, "publicationno"));
      rec->o.set(getValue(i, "oid"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PublisherName *KaspaBase::getPublisherNames(const char *where) {
  PublisherName *last=0;
  try {
    exec("begin");
    Str s("select oid, name, city, serie from publisher ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      PublisherName *rec=new PublisherName;
			rec->title.set(getValue(i, "serie"));
			if(!strlen(rec->title.get()))
	      rec->title.set(getValue(i, "name"));
      rec->address.set(getValue(i, "city"));
      rec->table.set("publisher");
      rec->o.set(getValue(i, "oid"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

/**  */
void KaspaBase::search(const char *where, AuthorName **an, PartTitle **pt,
									PublTitle **pbt, NoteTitle **nt, PartDataName **pdt) {
	try {
		exec("drop table tmpsearch");
	} catch (KaspaErr) {};
	Str s("select distinct tab, type into table tmpsearch from word ");
	s+=where;
	exec(s);

	s="where author.oid = tmpsearch.tab";
	*an=getAuthorNames(s);
	s="where publication.oid = tmpsearch.tab";
	*pbt=getPublTitles(s);
	s="where note.oid = tmpsearch.tab";
	*nt=getNoteTitles(s);

	try {
	  exec("begin");
  	s = "select part.oid, part.title as pt, publication.title as pb";
		s+= " from part, publication where part.oid=tmpsearch.tab and";
		s+= " part.publicationno=publication.oid order by pb";
	  exec(s);

  	for(int i=tuples()-1; i>=0; i--) {
      PartTitle *rec=new PartTitle;
			Str s=getValue(i, "pb"); s+=" : ";
			s+=getValue(i, "pt");
			rec->title.set(s);
      rec->table.set("part");
      rec->o.set(getValue(i, "oid"));
      rec->next=*pt;
      *pt=rec;
    }
    exec("commit");
  } catch(...) {
    delete *pt;
    *pt=0L;
    exec("rollback");
    throw;
  }


	try {
	  exec("begin");
  	s = "select partdata.oid, partdata.filename as pd, part.title as pt, publication.title as pb";
		s+= " from part, partdata, publication where partdata.oid=tmpsearch.tab and";
		s+= " partdata.partno=part.oid and part.publicationno=publication.oid order by pb";
	  exec(s);

  	for(int i=tuples()-1; i>=0; i--) {
      PartDataName *rec=new PartDataName;
			Str s=getValue(i, "pb"); s+=" : ";
			s+=getValue(i, "pt"); s+=" : ";
			s+=getValue(i, "pd");
			rec->title.set(s);
      rec->table.set("partdata");
      rec->o.set(getValue(i, "oid"));
      rec->next=*pdt;
      *pdt=rec;
    }
    exec("commit");
  } catch(...) {
    delete *pdt;
    *pdt=0L;
    exec("rollback");
    throw;
  }

//	*pdt=getPartDataNames(s);

	exec("drop table tmpsearch");
}

/**  */
int KaspaBase::bibTexIndex(Oid author, const char *year, Oid publ){
	exec("begin");
  Str s("(select oid from publication where publication.year='");
	s+=year;
	s+="' and publication.oid=publication_author.publicationno and publication_author.authorno=";
	s+=author;
	s+=" union select oid from publication where publication.year='";
	s+=year;
	s+="' and publication.oid=part.publicationno and part.oid=part_author.partno ";
	s+="and part_author.authorno=";
	s+=author;
	s+=") order by publication.title";
	
	int j=0;
	exec(s);
	if(tuples()>1) j++;
	for(int i=0; i<tuples(); i++)
		if(publ==Oid(atoi(getValue(i, "oid")))) j=i;
	exec("end");
	return j;
}

void KaspaBase::lo2buf(Oid i, DataField& f) {
  long len=0L;
  char *buf=lo2buf(i, &len);
  f.take((unsigned char*)buf, len);
}



char *KaspaBase::lo2str(Oid o) {
  long i=0L;
  char *s=lo2buf(o, &i);
  if(!s) {
    s=new char[1];
    *s=0;
  }
  return s;
}

Oid KaspaBase::str2lo(const char *s) {
  return buf2lo(s, strlen(s));
}

Oid KaspaBase::buf2lo(const char *buf, long len) {
  if(!buf)
    return InvalidOid;
  Oid j=InvalidOid;
  j=lCreate(INV_READ || INV_WRITE);
  int lfd=lOpen(j, INV_WRITE);

  if(lfd>=0) {
    lSeek(lfd, 0, SEEK_SET);
    lWrite(lfd, buf, len);
    lClose(lfd);
  }
  return j;
}

char *KaspaBase::lo2buf(Oid i, long *len) {
  int lfd=0;
  char *buf=0L;

  if(i==InvalidOid) return 0L;
  lfd=lOpen(i, INV_READ);
  lSeek(lfd, 0, SEEK_END);
  *len=lTell(lfd);
  lSeek(lfd, 0, SEEK_SET);
  buf=new char[*len+1];
  assert(buf);
  buf[*len]=0;
  lRead(lfd, buf, *len);
  lClose(lfd);
  return buf;
}

const char *KaspaBase::getTime() {
	static char buffer[100];
  time_t curtime;
  struct tm *gtime;
  curtime = time (NULL);
  gtime = gmtime (&curtime);
	strftime(buffer, 100, "%Y-%m-%d %H:%M:%S", gtime);
	return buffer;
}
