/* arch-tag: 5cb5730e-30fe-4714-a9aa-8ef4af2cb251 */

/*  eXperience GTK engine: drawable.c
 *  
 *  Copyright (C) 2004-2005  Benjamin Berg <benjamin@sipsolutions.net>
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include "drawable.h"

#define CHECK_SET_AND_SET(drawable, FLAG, message) { \
	if (drawable->private->options_set & FLAG) { \
		g_printerr ("Error in %s #%i in group %s: %s\n", drawable->class->object_type, drawable->number, drawable->group_name, message); \
		return; \
	} else drawable->private->options_set |= FLAG; \
}

typedef enum {
	OPTION_DONT_INHERIT = 1 << 0,
	OPTION_PADDING      = 1 << 1,
	OPTION_DONT_DRAW    = 1 << 2,
} eXperienceDrawableOptions;

struct _eXperienceDrawablePrivate {
	guint refcount;
	
	eXperienceDrawableOptions options_set;
	
	gboolean dont_inherit;
	gboolean dont_draw;
	
	eXperiencePadding padding;
	
	gboolean group_settings_applyed;
};

eXperienceDrawable *
experience_drawable_create (eXperienceDrawableClass * class, gint number, gchar * group_name)
{
	eXperienceDrawable * drawable;
	g_assert (class != NULL);
	g_assert (class->create != NULL);
	
	drawable = class->create ();
	g_return_val_if_fail (drawable != NULL, NULL);
	
	drawable->private = g_new0 (eXperienceDrawablePrivate, 1);
	
	drawable->private->refcount = 1;
	
	drawable->class = class;
	
	drawable->group_name = group_name;
	
	drawable->number = number;
	
	return drawable;
}


eXperienceDrawable *
experience_drawable_duplicate (eXperienceDrawable * drawable, gchar * group_name)
{
	eXperienceDrawable * new_drawable;
	
	g_assert (drawable != NULL);
	
	new_drawable = experience_drawable_create (drawable->class, drawable->number, group_name);
	
	experience_drawable_inherit_from (new_drawable, drawable);
	
	return new_drawable;
}

void
experience_drawable_ref (eXperienceDrawable * drawable)
{
	g_assert (drawable != NULL);
	
	drawable->private->refcount++;
}

void
experience_drawable_unref (eXperienceDrawable * drawable)
{
	g_assert (drawable != NULL);
	
	drawable->private->refcount--;
	
	if (drawable->private->refcount == 0) {
		/* destroy object */
		
		g_free (drawable->private);
		
		drawable->class->destroy (drawable);
	}
}

/*#############*/

void
experience_drawable_set_dont_inherit (eXperienceDrawable * drawable, gboolean dont_inherit)
{
	g_assert (drawable != NULL);
	
	CHECK_SET_AND_SET (drawable, OPTION_DONT_INHERIT, "dont_inherit set more than once!");
	
	drawable->private->dont_inherit = dont_inherit;
}

void
experience_drawable_set_dont_draw (eXperienceDrawable * drawable, gboolean dont_draw)
{
	g_assert (drawable != NULL);
	
	CHECK_SET_AND_SET (drawable, OPTION_DONT_DRAW, "dont_draw set more than once!");
	
	drawable->private->dont_draw = dont_draw;
}

void
experience_drawable_set_padding (eXperienceDrawable * drawable, gint left, gint right, gint top, gint bottom)
{
	g_assert (drawable != NULL);
	
	CHECK_SET_AND_SET (drawable, OPTION_PADDING, "Padding set more than once!");
	
	drawable->private->padding.left   = left;
	drawable->private->padding.right  = right;
	drawable->private->padding.top    = top;
	drawable->private->padding.bottom = bottom;
}

/*-------------*/

void
experience_drawable_apply_group_settings (eXperienceDrawable * drawable, eXperienceGroup * group)
{
	eXperiencePadding padding_tmp;
	
	g_assert (drawable != NULL);
	g_assert (group    != NULL);
	
	if (group->filter.mirror & ORIENTATION_HORIZONTAL) {
		padding_tmp = drawable->private->padding;
		
		drawable->private->padding.right = padding_tmp.left;
		drawable->private->padding.left  = padding_tmp.right;
	}
	
	if (group->filter.mirror & ORIENTATION_VERTICAL) {
		padding_tmp = drawable->private->padding;
		
		drawable->private->padding.top    = padding_tmp.bottom;
		drawable->private->padding.bottom = padding_tmp.top;
	}
	
	switch (group->filter.rotation) {
		case ROTATE_CW:
			padding_tmp = drawable->private->padding;
			
			drawable->private->padding.top    = padding_tmp.left;
			drawable->private->padding.right  = padding_tmp.top;
			drawable->private->padding.bottom = padding_tmp.right;
			drawable->private->padding.left   = padding_tmp.bottom;
			
			break;
		case ROTATE_CCW:
			padding_tmp = drawable->private->padding;
			
			drawable->private->padding.top    = padding_tmp.right;
			drawable->private->padding.right  = padding_tmp.bottom;
			drawable->private->padding.bottom = padding_tmp.left;
			drawable->private->padding.left   = padding_tmp.top;
			
			break;
		case ROTATE_AROUND:
			padding_tmp = drawable->private->padding;
			
			drawable->private->padding.top    = padding_tmp.bottom;
			drawable->private->padding.right  = padding_tmp.left;
			drawable->private->padding.bottom = padding_tmp.top;
			drawable->private->padding.left   = padding_tmp.right;
			
			break;
		case ROTATE_NONE:
			break;
	}
	
	experience_filter_apply_group_filter (&drawable->filter, &group->filter);
	
	drawable->class->apply_group_settings (drawable, group);
}

void
experience_drawable_inherit_from (eXperienceDrawable * drawable, eXperienceDrawable * from)
{
	g_assert (drawable != NULL);
	g_assert (from     != NULL);
	
	/* nothing should be done, so return */
	if (drawable->private->dont_inherit) return;
	
	if (drawable->class != from->class) {
		experience_warning ("Could not inherit two objects, because they have diffrent types!\n");
		return;
	}
	
	if (!(drawable->private->options_set & OPTION_PADDING))
		drawable->private->padding = from->private->padding;
	
	if (!(drawable->private->options_set & OPTION_DONT_DRAW))
		drawable->private->dont_draw = from->private->dont_draw;
	
	drawable->private->options_set |= from->private->options_set;
	
	drawable->class->inherit_from_drawable (drawable, from);
	
	experience_filter_inherit_from (&drawable->filter, &from->filter);
}

gboolean
experience_drawable_draw (eXperienceDrawable * drawable, GdkPixbuf * dest, GdkRectangle * dest_area, GtkStyle * style)
{
	GdkRectangle real_dest_area;
	
	g_assert (drawable   != NULL);
	g_assert (dest_area  != NULL);
	g_assert (style != NULL);
	
	if (drawable->private->dont_draw) return TRUE;
	
	real_dest_area = *dest_area;
	
	real_dest_area.x += drawable->private->padding.left;
	real_dest_area.y += drawable->private->padding.top;
	real_dest_area.width  -= drawable->private->padding.left + drawable->private->padding.right;
	real_dest_area.height -= drawable->private->padding.top  + drawable->private->padding.bottom;
	
	return drawable->class->draw (drawable, dest, &real_dest_area, style);
}

