/*
 *	Interfacephilia GTK+ Theme Engine
 * 
 *  interface_theme_draw.c:
 *		Drawing routines.
 *
 *	Copyright  2000 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "Interface_theme.h"
#include "Interface_images.h"

#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif /* M_PI */
#ifndef M_PI_4
#define M_PI_4  0.78539816339744830962
#endif /* M_PI_4 */

#define DEBUG 0

/* internal functions */
static void    interface_draw_hline(GtkStyle * style,
						       GdkWindow * window,
						       GtkStateType state_type,
						       GdkRectangle * area,
						       GtkWidget * widget,
						       gchar * detail,
						       gint x1,
						       gint x2,
						       gint y);
static void    interface_draw_vline(GtkStyle * style,
						       GdkWindow * window,
						       GtkStateType state_type,
						       GdkRectangle * area,
						       GtkWidget * widget,
						       gchar * detail,
						       gint y1,
						       gint y2,
						       gint x);

static void    interface_draw_shadow(GtkStyle * style,
								GdkWindow * window,
								GtkStateType state_type,
								GtkShadowType shadow_type,
								GdkRectangle * area,
								GtkWidget * widget,
								gchar * detail,
								gint x,
								gint y,
								gint width,
								gint height);

static void    interface_draw_polygon(GtkStyle * style,
								 GdkWindow * window,
								 GtkStateType state_type,
								 GtkShadowType shadow_type,
								 GdkRectangle * area,
								 GtkWidget * widget,
								 gchar * detail,
								 GdkPoint * point,
								 gint npoints,
								 gint fill);
static void    interface_draw_arrow(GtkStyle * style,
						       GdkWindow * window,
						       GtkStateType state_type,
						       GtkShadowType shadow_type,
						       GdkRectangle * area,
						       GtkWidget * widget,
						       gchar * detail,
						       GtkArrowType arrow_type,
						       gint fill,
						       gint x,
						       gint y,
						       gint width,
						       gint height);
static void    interface_draw_diamond(GtkStyle * style,
								 GdkWindow * window,
								 GtkStateType state_type,
								 GtkShadowType shadow_type,
								 GdkRectangle * area,
								 GtkWidget * widget,
								 gchar * detail,
								 gint x,
								 gint y,
								 gint width,
								 gint height);
static void    interface_draw_oval(GtkStyle * style,
						      GdkWindow * window,
						      GtkStateType state_type,
						      GtkShadowType shadow_type,
						      GdkRectangle * area,
						      GtkWidget * widget,
						      gchar * detail,
						      gint x,
						      gint y,
						      gint width,
						      gint height);
static void    interface_draw_string(GtkStyle * style,
								GdkWindow * window,
								GtkStateType state_type,
								GdkRectangle * area,
								GtkWidget * widget,
								gchar * detail,
								gint x,
								gint y,
								const gchar * string);
static void    interface_draw_box(GtkStyle * style,
						     GdkWindow * window,
						     GtkStateType state_type,
						     GtkShadowType shadow_type,
						     GdkRectangle * area,
						     GtkWidget * widget,
						     gchar * detail,
						     gint x,
						     gint y,
						     gint width,
						     gint height);
static void    interface_draw_flat_box(GtkStyle * style,
								  GdkWindow * window,
								  GtkStateType state_type,
								  GtkShadowType shadow_type,
								  GdkRectangle * area,
								  GtkWidget * widget,
								  gchar * detail,
								  gint x,
								  gint y,
								  gint width,
								  gint height);
static void    interface_draw_check(GtkStyle * style,
						       GdkWindow * window,
						       GtkStateType state_type,
						       GtkShadowType shadow_type,
						       GdkRectangle * area,
						       GtkWidget * widget,
						       gchar * detail,
						       gint x,
						       gint y,
						       gint width,
						       gint height);
static void    interface_draw_option(GtkStyle * style,
								GdkWindow * window,
								GtkStateType state_type,
								GtkShadowType shadow_type,
								GdkRectangle * area,
								GtkWidget * widget,
								gchar * detail,
								gint x,
								gint y,
								gint width,
								gint height);
static void    interface_draw_cross(GtkStyle * style,
						       GdkWindow * window,
						       GtkStateType state_type,
						       GtkShadowType shadow_type,
						       GdkRectangle * area,
						       GtkWidget * widget,
						       gchar * detail,
						       gint x,
						       gint y,
						       gint width,
						       gint height);
static void    interface_draw_ramp(GtkStyle * style,
						      GdkWindow * window,
						      GtkStateType state_type,
						      GtkShadowType shadow_type,
						      GdkRectangle * area,
						      GtkWidget * widget,
						      gchar * detail,
						      GtkArrowType arrow_type,
						      gint x,
						      gint y,
						      gint width,
						      gint height);
static void    interface_draw_tab(GtkStyle * style,
						     GdkWindow * window,
						     GtkStateType state_type,
						     GtkShadowType shadow_type,
						     GdkRectangle * area,
						     GtkWidget * widget,
						     gchar * detail,
						     gint x,
						     gint y,
						     gint width,
						     gint height);
static void    interface_draw_shadow_gap(GtkStyle * style,
								    GdkWindow * window,
								    GtkStateType state_type,
								    GtkShadowType shadow_type,
								    GdkRectangle * area,
								    GtkWidget * widget,
								    gchar * detail,
								    gint x,
								    gint y,
								    gint width,
								    gint height,
								    GtkPositionType gap_side,
								    gint gap_x,
								    gint gap_width);
static void    interface_draw_box_gap(GtkStyle * style,
								 GdkWindow * window,
								 GtkStateType state_type,
								 GtkShadowType shadow_type,
								 GdkRectangle * area,
								 GtkWidget * widget,
								 gchar * detail,
								 gint x,
								 gint y,
								 gint width,
								 gint height,
								 GtkPositionType gap_side,
								 gint gap_x,
								 gint gap_width);
static void    interface_draw_extension(GtkStyle * style,
								   GdkWindow * window,
								   GtkStateType state_type,
								   GtkShadowType shadow_type,
								   GdkRectangle * area,
								   GtkWidget * widget,
								   gchar * detail,
								   gint x,
								   gint y,
								   gint width,
								   gint height,
								   GtkPositionType gap_side);
static void    interface_draw_focus(GtkStyle * style,
						       GdkWindow * window,
						       GdkRectangle * area,
						       GtkWidget * widget,
						       gchar * detail,
						       gint x,
						       gint y,
						       gint width,
						       gint height);
static void    interface_draw_slider(GtkStyle * style,
								GdkWindow * window,
								GtkStateType state_type,
								GtkShadowType shadow_type,
								GdkRectangle * area,
								GtkWidget * widget,
								gchar * detail,
								gint x,
								gint y,
								gint width,
								gint height,
								GtkOrientation orientation);
static void    interface_draw_handle(GtkStyle * style,
								GdkWindow * window,
								GtkStateType state_type,
								GtkShadowType shadow_type,
								GdkRectangle * area,
								GtkWidget * widget,
								gchar * detail,
								gint x,
								gint y,
								gint width,
								gint height,
								GtkOrientation orientation);

static void		interface_draw_lines(GtkStyle		*style,
								GdkWindow		*window,
								GtkStateType	state_type,
								GtkShadowType	shadow_type,
								GdkRectangle	*area,
								GtkWidget		*widget,
								gint			x,
								gint			y,
								gint			width,
								gint			height,
								GtkOrientation	orientation);

static void		interface_draw_buds (GtkStyle		*style,
								GdkWindow		*window,
								GtkStateType	state_type,
								GtkShadowType	shadow_type,
								GdkRectangle	*area,
								GtkWidget		*widget,
								gint			x,
								gint			y,
								gint			width,
								gint			height,
								GtkOrientation	orientation);

static void interface_draw_background (GtkStyle *style, GdkWindow *window, GdkGC *gc, GdkPixmap *bg_pixmap,
								  GtkStateType state_type, GdkRectangle *area, GtkWidget *widget, 
								  gint x, gint y, gint width, gint height);

static void
interface_draw_image (GdkWindow *window, GtkWidget *widget, GdkRectangle *area, InterfaceImageType image_type,
				 guint16 srcx, guint16 srcy,
				 guint16 x, guint16 y, guint16 width, guint16 height);
static void
interface_draw_pixmap (GdkWindow *window, GdkGC *gc, GdkRectangle *area, GdkPixmap *pixmap, GdkBitmap *mask,
				  guint16 srcx, guint16 srcy,
				  guint16 x, guint16 y, guint16 width, guint16 height);



/* internal data structs */
GtkStyleClass interface_class = {
	2,
	2,
	interface_draw_hline,
	interface_draw_vline,
	interface_draw_shadow,
	interface_draw_polygon,
	interface_draw_arrow,
	interface_draw_diamond,
	interface_draw_oval,
	interface_draw_string,
	interface_draw_box,
	interface_draw_flat_box,
	interface_draw_check,
	interface_draw_option,
	interface_draw_cross,
	interface_draw_ramp,
	interface_draw_tab,
	interface_draw_shadow_gap,
	interface_draw_box_gap,
	interface_draw_extension,
	interface_draw_focus,
	interface_draw_slider,
	interface_draw_handle
};

GtkStyleClass interface_thin_class = {
	1,
	1,
	interface_draw_hline,
	interface_draw_vline,
	interface_draw_shadow,
	interface_draw_polygon,
	interface_draw_arrow,
	interface_draw_diamond,
	interface_draw_oval,
	interface_draw_string,
	interface_draw_box,
	interface_draw_flat_box,
	interface_draw_check,
	interface_draw_option,
	interface_draw_cross,
	interface_draw_ramp,
	interface_draw_tab,
	interface_draw_shadow_gap,
	interface_draw_box_gap,
	interface_draw_extension,
	interface_draw_focus,
	interface_draw_slider,
	interface_draw_handle
};



static void
interface_draw_hline (GtkStyle     *style,
				 GdkWindow    *window,
				 GtkStateType  state_type,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 gchar         *detail,
				 gint          x1,
				 gint          x2,
				 gint          y)
{
	gint thickness_light;
	gint thickness_dark;
	gint i;

	GtkWidget *cur;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	thickness_light = style->klass->ythickness / 2;
	thickness_dark  = style->klass->ythickness - thickness_light;

	if (detail && !strcmp (detail, "label")) {
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
		gdk_draw_line (window, style->fg_gc[state_type], x1, y, x2-1, y);
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
	} else {
		if (area) {
			gdk_gc_set_clip_rectangle (style->white_gc, area);
			gdk_gc_set_clip_rectangle (style->dark_gc [state_type], area);
			gdk_gc_set_clip_rectangle (style->bg_gc [state_type], area);
		}
		if ((widget) && GTK_IS_MENU_ITEM(widget)) {
			goto draw_normal;
		} else if ((widget) && (cur = widget->parent)) while (cur) {
			if (   (GTK_IS_BOX(cur) || GTK_IS_TABLE(cur))
				&& GTK_CONTAINER(cur)->border_width == 0)
			{
				cur = cur->parent;
			} else if (GTK_IS_WINDOW(cur) && GTK_CONTAINER(cur)->border_width == 0) {
				goto draw_inframe;
			} else if (GTK_IS_FRAME(cur) && GTK_CONTAINER(cur)->border_width == 0) {
				goto draw_ugly;
			} else
				goto draw_normal;
		}

	  draw_ugly:
		gdk_draw_line (window, style->dark_gc[state_type], x1, y,   x2-1, y);
		gdk_draw_line (window, style->white_gc, x1, y+1, x2-1, y+1);
		goto draw_done;

	  draw_inframe:
		gdk_draw_line (window, style->dark_gc[state_type], x1+1, y,   x2-1, y);
		gdk_draw_line (window, style->white_gc, x1,   y+1, x2-2, y+1);
		gdk_draw_point (window, style->bg_gc[state_type], x1,   y);
		gdk_draw_point (window, style->bg_gc[state_type], x2-1, y+1);
		goto draw_done;
	  
	  draw_normal:
		gdk_draw_line (window, style->dark_gc[state_type], x1,   y,   x2-2, y);
		gdk_draw_line (window, style->white_gc, x1+1, y+1, x2-1, y+1);
		gdk_draw_point (window, style->bg_gc[state_type], x1,   y+1);
		gdk_draw_point (window, style->bg_gc[state_type], x2-1, y);
	  
	  draw_done:
		if (area) {
			gdk_gc_set_clip_rectangle (style->white_gc, NULL);
			gdk_gc_set_clip_rectangle (style->dark_gc[state_type], NULL);
			gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
		}
	}
}


static void
interface_draw_vline (GtkStyle     *style,
				 GdkWindow    *window,
				 GtkStateType  state_type,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 gchar         *detail,
				 gint          y1,
				 gint          y2,
				 gint          x)
{
	gint thickness_light;
	gint thickness_dark;
	gint i;

	GtkWidget *cur;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	thickness_light = style->klass->xthickness / 2;
	thickness_dark  = style->klass->xthickness - thickness_light;

	if (detail && !strcmp (detail, "label")) {
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
		gdk_draw_line (window, style->fg_gc[state_type], x, y1, x, y2-1);
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
	} else {
		if (area) {
			gdk_gc_set_clip_rectangle (style->light_gc [state_type], area);
			gdk_gc_set_clip_rectangle (style->dark_gc [state_type], area);
			gdk_gc_set_clip_rectangle (style->mid_gc [state_type], area);
		}
		if ((widget) && GTK_IS_MENU_ITEM(widget)) {
			goto draw_normal;
		} else if ((widget) && (cur = widget->parent)) while (cur) {
			if (   (GTK_IS_BOX(cur) || GTK_IS_TABLE(cur))
				&& GTK_CONTAINER(cur)->border_width == 0)
			{
				cur = cur->parent;
			} else if (GTK_IS_WINDOW(cur)) {
				goto draw_inframe;
			} else if (GTK_IS_FRAME(cur)) {
				goto draw_ugly;
			} else
				goto draw_normal;
		}

	  draw_ugly:
		gdk_draw_line (window, style->dark_gc [state_type], x,   y1, x,   y2-1);
		gdk_draw_line (window, style->white_gc, x+1, y1, x+1, y2-1);
		goto draw_done;

	  draw_inframe:
		gdk_draw_line (window, style->dark_gc [state_type], x,   y1,   x,   y2-2);
		gdk_draw_line (window, style->white_gc, x+1, y1+1, x+1, y2-1);
		gdk_draw_point (window, style->bg_gc[state_type], x+1, y1);
		gdk_draw_point (window, style->bg_gc[state_type], x,   y2-1);
		goto draw_done;
	  
	  draw_normal:
		gdk_draw_line (window, style->dark_gc [state_type], x,   y1+1, x,   y2-1);
		gdk_draw_line (window, style->white_gc, x+1, y1,   x+1, y2-2);
		gdk_draw_point (window, style->bg_gc[state_type], x,   y1);
		gdk_draw_point (window, style->bg_gc[state_type], x+1, y2-1);
	  
	  draw_done:
		if (area) {
			gdk_gc_set_clip_rectangle (style->white_gc, NULL);
			gdk_gc_set_clip_rectangle (style->dark_gc [state_type], NULL);
			gdk_gc_set_clip_rectangle (style->bg_gc [state_type], NULL);
		}
	}
}



static GdkGC *
interface_ghost_gc (GtkStyle     *style,
			   GtkStyle     *style2,
			   GdkWindow    *window,
			   GdkGC        *gc)
{
	if (gc == style->black_gc) {
		gc = style->dark_gc[GTK_STATE_INSENSITIVE];
	} else if (gc == style->white_gc) {
		gc = style->light_gc[GTK_STATE_INSENSITIVE];
	} else if (   gc == style->light_gc[GTK_STATE_INSENSITIVE]
			   || gc == style->dark_gc[GTK_STATE_INSENSITIVE]
			   || gc == style2->light_gc[GTK_STATE_INSENSITIVE]
			   || gc == style2->dark_gc[GTK_STATE_INSENSITIVE])
	{
		if (style->klass->xthickness==1 || style->klass->ythickness==1)
			return gc;
	
		gdk_gc_copy(gc = gdk_gc_new(window), gc);
		gdk_gc_set_line_attributes (gc, 1, GDK_LINE_ON_OFF_DASH, 0, 0);
		gdk_gc_set_dashes (gc, 0, "\1\1", 2);
	} else {
		gc = style->bg_gc[GTK_STATE_INSENSITIVE];
	}
	
	return gc;
}

static void
interface_unghost_gc (GtkStyle     *style,
				 GtkStyle     *style2,
				 GdkGC        *gc)
{
	if (gc != style->bg_gc[GTK_STATE_INSENSITIVE] &&
		gc != style->dark_gc[GTK_STATE_INSENSITIVE] &&
		gc != style->light_gc[GTK_STATE_INSENSITIVE] &&
		gc != style2->bg_gc[GTK_STATE_INSENSITIVE] &&
		gc != style2->dark_gc[GTK_STATE_INSENSITIVE] &&
		gc != style2->light_gc[GTK_STATE_INSENSITIVE])
	{
		gdk_gc_destroy(gc);
	}
}


static void
interface_draw_shadow (GtkStyle      *style,
				  GdkWindow     *window,
				  GtkStateType   state_type,
				  GtkShadowType  shadow_type,
				  GdkRectangle  *area,
				  GtkWidget     *widget,
				  gchar         *detail,
				  gint           x,
				  gint           y,
				  gint           width,
				  gint           height)
{
	GdkPoint points[5];
	GtkWidget *menu;
	InterfaceShadowType newshadow = (InterfaceShadowType)shadow_type;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	switch (shadow_type) {
	  case GTK_SHADOW_NONE:
		return;
	  case GTK_SHADOW_IN:
		if (GTK_IS_PROGRESS(widget)) {
			newshadow = INTERFACE_SHADOW_IN;
		} else if (GTK_IS_FRAME(widget) && !GTK_IS_PREVIEW(GTK_BIN(widget)->child)) {
			newshadow = INTERFACE_SHADOW_IN;
			if (detail && !strcmp(detail, "slider")) {
				x++;		y++;
				width-=2;	height-=2;
			}
		} else if (GTK_IS_BUTTON(widget)) {
			 if (detail && !strncmp("buttondefault", detail)) {
				newshadow = ((InterfaceThemeData *)style->engine_data)->buttondef_shadow;
			} else
			if (!GTK_IS_TOGGLE_BUTTON(widget) && state_type == GTK_STATE_ACTIVE) {
				if (((GtkButton *)(widget))->relief == GTK_RELIEF_NONE) {
					newshadow = INTERFACE_SHADOW_NONE_IN;
				} else {
					newshadow = INTERFACE_SHADOW_IN;
				}
			}
		} else if (GTK_IS_RANGE(widget) && 
				   (   GTK_RANGE(widget)->in_child == GTK_RANGE_CLASS (GTK_OBJECT (widget)->klass)->step_forw
				    || GTK_RANGE(widget)->in_child == GTK_RANGE_CLASS (GTK_OBJECT (widget)->klass)->step_back))
		{
			newshadow = INTERFACE_SHADOW_IN;
		} else if (GTK_IS_ENTRY(widget)) {
			if (GTK_WIDGET_STATE(widget)==GTK_STATE_INSENSITIVE) {
				state_type = GTK_STATE_INSENSITIVE;
			}
		} if (GTK_IS_TEXT(widget) && GTK_WIDGET_HAS_FOCUS(widget)) {
			/* x--;
			y--;
			width+=2;
			height+=2; */
			newshadow = INTERFACE_SHADOW_IN_FOCUSED;
		}
		break;
	
	  case GTK_SHADOW_OUT:
		if (   (GTK_IS_FRAME(widget) && !GTK_IS_EVENT_BOX(widget->parent))
			|| GTK_IS_RULER(widget)
			|| (detail && (   !strcmp(detail, "handlebox_bin")
						   || !strcmp(detail, "dockitem_bin")))
		   )
		{
			newshadow = INTERFACE_SHADOW_OUT;
		} else if (GTK_IS_MENU(widget)) {
			newshadow = INTERFACE_SHADOW_OUT;

			if (   GTK_IS_MENU(widget)
			   && !(widget->parent && GTK_IS_OPTION_MENU(widget->parent))
			   && ((GtkMenu *)widget)->torn_off)
			   newshadow = INTERFACE_SHADOW_OUT;
		} else if (GTK_IS_BUTTON(widget)) {
			if (   GTK_BUTTON(widget)->relief != GTK_RELIEF_NORMAL
				&& state_type == GTK_STATE_PRELIGHT)
			{
				newshadow = INTERFACE_SHADOW_NONE_OUT;
			}
		} else if (GTK_IS_MENU_ITEM(widget)) {
			newshadow = INTERFACE_SHADOW_OUT;
		} else if (GTK_IS_MENU_BAR(widget)) {
			if (widget->parent && GTK_IS_HANDLE_BOX(widget->parent)) {
				return;
			}
			
			if (style->klass->ythickness > 1) {
				newshadow = INTERFACE_SHADOW_OUT;
				/* x--;
				y--;
				height++;
				width+=2; */
				
			} else {
				newshadow = INTERFACE_SHADOW_THIN_OUT;
				/* x--;
				y--;
				height++;
				width+=2; */
			}
		} else if (GTK_IS_PROGRESS(widget)) {
			GtkProgressBarOrientation or = ((GtkProgressBar *)widget)->orientation;
			GtkProgressBarStyle		  st = ((GtkProgressBar *)widget)->bar_style;
			
			gdk_draw_rectangle (window, style->base_gc[GTK_STATE_SELECTED], TRUE,
				x, y,
				width  - (st==GTK_PROGRESS_DISCRETE && (or==GTK_PROGRESS_RIGHT_TO_LEFT || or==GTK_PROGRESS_LEFT_TO_RIGHT)),
				height - (st==GTK_PROGRESS_DISCRETE && (or==GTK_PROGRESS_BOTTOM_TO_TOP || or==GTK_PROGRESS_TOP_TO_BOTTOM))
				);
			return;
		}
		break;
	
	  case GTK_SHADOW_ETCHED_IN:
		if (GTK_IS_SPIN_BUTTON(widget)) {
			state_type = GTK_STATE_INSENSITIVE;
			newshadow = INTERFACE_SHADOW_OUT;
		}
	}

	points[0].x = x;				points[0].y = y;
	points[1].x = x;				points[1].y = y + height - 1;
	points[2].x = x + width - 1;	points[2].y = y + height - 1;
	points[3].x = x + width - 1;	points[3].y = y;
	points[4].x = x;				points[4].y = y;

	interface_draw_polygon (style, window, state_type, (GtkShadowType)newshadow, area,
					   widget, NULL, points, 5, FALSE);
}

#define SIGN(exp)	((exp)<0 ? -1 : 1)


static void
interface_draw_polygon (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   gchar         *detail,
				   GdkPoint      *points,
				   gint           npoints,
				   gboolean       fill)
{
	static const gdouble pi_over_4	 = M_PI_4;
	static const gdouble pi_over_2	 = M_PI_2;
	static const gdouble pi_3_over_4 = M_PI_4 * 3;
	static const gdouble epsilon	 = M_PI_4 / 2;

	InterfaceShadowType	shadow;
	GtkStyle		*style2;
	GtkStateType	state2;
	gdouble			angle;
	guint			thin=0;
	gint			x,y, x2,y2;
	gint			i, j;
	gboolean		sign, oldsign;
	InterfaceThemeData	*data;

	GdkGC			*bg, *gc0, *gc1, *gc2, *gc3, *gc4, *gc5;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	g_return_if_fail (points != NULL);

	shadow = (InterfaceShadowType)shadow_type;
	bg = style->bg_gc[state_type];

	if (widget && widget->parent) {
		style2 = widget->parent->style;
		state2 = widget->parent->state;
	} else {
		style2 = style;
		state2 = state_type;
	}
	if (state_type == GTK_STATE_INSENSITIVE)
		state2 = GTK_STATE_INSENSITIVE;

	data = NULL;
	if (style->klass->xthickness==1 || style->klass->ythickness==1) {
		if (shadow == INTERFACE_SHADOW_IN)
			shadow = INTERFACE_SHADOW_THIN_IN;
		else if (shadow == INTERFACE_SHADOW_OUT)
			shadow = INTERFACE_SHADOW_THIN_OUT; 
	}

	if (area)
		gdk_gc_set_clip_rectangle (bg, area);
	
	if (fill)
		gdk_draw_polygon (window, bg, TRUE, points, npoints);

	switch (shadow) {
	  case INTERFACE_SHADOW_IN:
		gc0 = style2->dark_gc[state2];
		gc1 = style->black_gc;
		gc2 = style2->bg_gc[state2];
		gc3 = style->white_gc;
		gc5 = gc1;
		gc4 = style2->mid_gc[state2];
		break;

	  case INTERFACE_SHADOW_OUT:
		gc0 = style->white_gc;
		gc1 = style->bg_gc[state_type];
		gc4 = style->mid_gc[state_type];
		gc5 = gc4;
		gc2 = style->dark_gc[state_type];
		gc3 = style->black_gc;
		break;

	  case GTK_SHADOW_ETCHED_IN:
		gc1 = style->white_gc;
		gc2 = style->dark_gc[state_type];
		gc5 = style->white_gc;
		gc0 = gc2;
		gc3 = gc1;
		gc4 = gc5;
		break;

	  case GTK_SHADOW_ETCHED_OUT:
		gc0 = style2->bg_gc[state2];
		gc1 = style2->dark_gc[state2];
		gc2 = style2->bg_gc[state2];
		gc3 = style2->dark_gc[state2];
		gc4 = style2->bg_gc[state2];
		gc5 = style2->bg_gc[state2];
		break;

	  case INTERFACE_SHADOW_BUTTON_IN:
		gc0 = style->black_gc;
		gc1 = style->dark_gc[state_type];
		gc2 = style->bg_gc[state_type];
		gc3 = gc1;
		gc4 = gc0;
		gc5 = gc3;
		break;

	  case INTERFACE_SHADOW_NONE_OUT:
		gc0 = style->white_gc;
		gc1 = style->bg_gc[state_type];
		gc4 = style->mid_gc[state_type];
		gc5 = gc4;
		gc2 = style->dark_gc[state_type];
		gc3 = style->black_gc;
		break;

	  case INTERFACE_SHADOW_NONE_IN:
		gc0 = style2->dark_gc[state2];
		gc1 = style2->black_gc;
		gc2 = style2->bg_gc[state_type];
		gc3 = style2->white_gc;
		gc5 = gc1;
		gc4 = style2->dark_gc[state2];
		break;

	  case INTERFACE_SHADOW_IN_FOCUSED:
		gc0 = style2->dark_gc[state2];
		gc1 = style2->black_gc;
		gc2 = style2->black_gc;
		gc3 = style2->white_gc;
		gc5 = gc1;
		gc4 = style2->dark_gc[state2];
		break;
	
	  case INTERFACE_SHADOW_THIN_OUT:
		thin = 1;
		gc0 = style->white_gc;
		gc1 = bg;
		gc2 = bg;
		gc3 = style->dark_gc[state_type];
		gc4 = style->bg_gc[state_type];
		gc5 = bg;
		break;

	  case INTERFACE_SHADOW_THIN_IN:
		thin = 1;
		gc0 = style2->black_gc;
		gc1 = bg;
		gc2 = bg;
		gc3 = style2->white_gc;
		gc4 = style2->bg_gc[state2];
		gc5 = bg;
		break;

	  case INTERFACE_SHADOW_NOTE:
		thin = 1;
		gc0 = style2->black_gc;
		gc1 = bg;
		gc2 = bg;
		gc3 = style2->black_gc;
		gc4 = gc0;
		gc5 = bg;
		break;
	
	  case INTERFACE_SHADOW_NONE:
	  default:
		return;
	}
	if (state_type == GTK_STATE_INSENSITIVE) {
		gc0 = interface_ghost_gc (style, style2, window, gc0);
		gc1 = interface_ghost_gc (style, style2, window, gc1);
		gc2 = interface_ghost_gc (style, style2, window, gc2);
		gc3 = interface_ghost_gc (style, style2, window, gc3);
		gc4 = interface_ghost_gc (style, style2, window, gc4);
		gc5 = interface_ghost_gc (style, style2, window, gc5);
	}
	if (area) {
		gdk_gc_set_clip_rectangle (gc0, area);
		gdk_gc_set_clip_rectangle (gc1, area);
		gdk_gc_set_clip_rectangle (gc2, area);
		gdk_gc_set_clip_rectangle (gc3, area);
		gdk_gc_set_clip_rectangle (gc4, area);
		gdk_gc_set_clip_rectangle (gc5, area);
	}


	npoints--;

	for (j = thin; j < 3; j++) {
		for (i = 0; i < npoints; i++) {
			x	= points[i].x;
			y	= points[i].y;
			x2	= points[i+1].x;
			y2	= points[i+1].y;
			
			if ((x == x2) && (y == y2))
				angle = 0;
			else
				angle = atan2 (y2 - y, x2 - x);

			if ((angle > -pi_3_over_4 - 0.0625) && (angle < pi_over_4 - 0.0625)) {
				sign = 0;
				if (j==1) {
					gdk_draw_line (window, gc3, x, y, x2, y2);
				} else {
				  /*
					if (angle < -pi_3_over_4 + epsilon) {
						x--;
						y2++;
						x2--;
					} else if (angle < -pi_over_4 - epsilon) {
						x--;
						x2--;
					} else if (angle < -pi_over_4 + epsilon) {
						y--;
						x2--;
					} else if (angle < epsilon) {
						y--;
						y2--;
					} else {
						x++;
						y2--;
					}
				  */
					if (angle > -pi_over_4) {
						y--;
						y2--;
					} else {
						x--;
						x2--;
					}

					if (j==0) {
						gdk_draw_line (window, gc2, x, y, x2, y2);
					} else if (sign != oldsign && i>0) {
						gdk_draw_point (window, gc4, points[i].x, points[i].y);
						if (!thin)
							gdk_draw_point (window, gc5, x+1, y);
					}
				}
			} else {
				sign = 1;
				if (j==1) {
					gdk_draw_line (window, gc0, x,y, x2,y2);
				} else {
				  /*
					if (angle > M_PI-epsilon || angle < -(M_PI-epsilon)) {
						y	+= 1;
						y2	+= 1;
					} else if (angle < -pi_3_over_4) {
						x	-= 1;
						y2	+= 1;
					} else if (angle < pi_over_2-epsilon) {
						x	+= 1;
						y2	-= 1;
					} else if (angle < pi_3_over_4 - epsilon) {
						x	+= 1;
						x2	+= 1;
					} else {
						x	+= 1;
						x2	+= 1;
					}
				  */
					if ((angle < -pi_3_over_4) || (angle > pi_3_over_4)) {
						y++;
						y2++;
					} else {
						x++;
						x2++;
					}

					if (j==0) {
						gdk_draw_line (window, gc1, x,y, x2,y2);
					} else if (sign != oldsign && i>0) {
						gdk_draw_point (window, gc4, points[i].x, points[i].y);
						if (!thin)
							gdk_draw_point (window, gc5, x-1, y);
					}
				}
			}
			oldsign = sign;
		}
		if (j==0 && state_type == GTK_STATE_INSENSITIVE) {
			gdk_draw_polygon (window, bg, FALSE, points, npoints);
		}
	}

	if (area) {
		gdk_gc_set_clip_rectangle (gc5, NULL);
		gdk_gc_set_clip_rectangle (gc4, NULL);
		gdk_gc_set_clip_rectangle (gc3, NULL);
		gdk_gc_set_clip_rectangle (gc2, NULL);
		gdk_gc_set_clip_rectangle (gc1, NULL);
		gdk_gc_set_clip_rectangle (gc0, NULL);
		gdk_gc_set_clip_rectangle (bg, NULL);
	}
	if (state_type == GTK_STATE_INSENSITIVE) {
		interface_unghost_gc (style, style2, gc5);
		interface_unghost_gc (style, style2, gc4);
		interface_unghost_gc (style, style2, gc3);
		interface_unghost_gc (style, style2, gc2);
		interface_unghost_gc (style, style2, gc1);
		interface_unghost_gc (style, style2, gc0);
	}
}


static void
interface_draw_arrow (GtkStyle      *style,
				 GdkWindow     *window,
				 GtkStateType   state_type,
				 GtkShadowType  shadow_type,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 gchar         *detail,
				 GtkArrowType   arrow_type,
				 gboolean       fill,
				 gint           x,
				 gint           y,
				 gint           width,
				 gint           height)
{
	gint	half_width, t;
	gint	d, s;
	gushort	i;
	
	/* gfloat	d, s; */
	GdkGC	*black = style->black_gc;
	GdkGC	*dark  = style->dark_gc [state_type];

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

/*	} else if (GTK_IS_SPIN_BUTTON(widget)) {
		gtk_paint_box (style, window, state_type,
					   shadow_type, area, widget, detail, x, y, width, height);
		x      += 2;
		y      += 2;
		width  -= 5;
		height -= 5;
		goto draw_black;
*/

	if (   widget
		&& GTK_IS_RANGE(widget)
		&& (style->engine_data != NULL))
	{
		if (((InterfaceThemeData *)style->engine_data)->stepper_box) {
			gint xt, yt;
			
			xt = style->klass->xthickness;
			yt = style->klass->ythickness;
		
			gtk_paint_box (style, window, state_type,
						   shadow_type, area, widget, detail, x, y, width, height);
			
			x += xt;
			y += yt;
			width -= xt + xt;
			height -= yt + yt;
			
			if (((InterfaceThemeData *)style->engine_data)->stepper_arrows) {
				shadow_type = GTK_SHADOW_OUT;
				
				if (state_type == GTK_STATE_NORMAL || state_type == GTK_STATE_ACTIVE)
					state_type = GTK_STATE_SELECTED;
				goto draw_normal;
			}
		}
		 
		if (((InterfaceThemeData *)style->engine_data)->stepper_arrows) {
			goto draw_normal;
		} else {
			x++;
			y++;
			width = (width - 3) << 1;
			height = (height - 3) << 1;
			goto draw_black;
		}
	}

	if (   widget
		&& GTK_IS_ARROW(widget)
		&& widget->parent
		&& widget->parent->parent
		&& GTK_IS_COMBO(widget->parent->parent))
	{
		fill = FALSE;
		/* This was always commented out, others == changed from Xeno/original Interface
		x		+= 2;
		y		+= 2;
		width	-= 4;
		height	-= 4;
		*/
				
		/* x += (width >> 1) - 4;
		y += (height >> 1) - 2; */
				
		x += (width >> 1) - 4;
		y += (height >> 1) - 3;
		
		/* width	= 0x12;
		height	= 0x09; */
		
		width	= 0x11;
		height	= 0x11;
		
		goto draw_black;
	
	} else if (detail && !strcmp("optionmenutab", detail)) {
		fill = FALSE;
		/*
		x		+= 2;
		y		+= 2;
		width	-= 4;
		height	-= 4;
		*/
		/* x += (width >> 1) - 6;
		y += (height >> 1) - 3; */
		
		x += (width >> 1) - 4;
		y += (height >> 1) - 4;
		
		/* width	= 0x17;
		height	= 0x0C; */
		
		width	= 0x11;
		height	= 0x11;
		
		goto draw_black;
	
	} else if (   (detail && !strcmp ("menuitem", detail))
			   || (widget && widget->parent && widget->parent && GTK_IS_BUTTON(widget->parent))
			   || (widget
			       && GTK_IS_ARROW(widget)
			       && widget->parent
				   && widget->parent->parent
				   && GTK_IS_BUTTON(widget->parent->parent)
				  )
			  )
	{
		x		+= 1;
		y		+= 1;
		width	-= 2;
		height	-= 2;
		
		t = style->font->ascent - height;
		if (t < 0) {
				x -= t>>1;
				y -= t>>1;
				height += t;
				width += t;
		}

		width  = width << 1;
		height = height << 1;

	  draw_black:
		if (state_type == GTK_STATE_INSENSITIVE) {
			black = style->dark_gc[state_type];
			dark  = style->mid_gc[state_type];
		}
		if (area) {
			gdk_gc_set_clip_rectangle (black, area);
			gdk_gc_set_clip_rectangle (dark, area);
		}
		
		if (height < 3)
			height = 3;
		if (width < 3)
			width = 3;
		
		if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_DOWN) {
			d = (width << 14) / height;
			x += width >> 2;
		} else {
			d = (height << 14) / width;
			y += height >> 2;
		}
		if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_LEFT) {
			s = d;
		} else {
			s = (height>>1) * d;
			d = -d;
		}

		if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_DOWN) {
			for (i = 0; i < (height>>1); i++) {
				gdk_draw_line (window, black,
							   (unsigned) (x - ((s-0x00003fff)>>15)), y + i,
							   (unsigned) (x + ((s-0x00003fff)>>15)), y + i);
				if ((s & 0x00004000)) {
					gdk_draw_point (window, dark, (unsigned) (x + (s>>15)), y + i);
					gdk_draw_point (window, dark, (unsigned) (x - (s>>15)), y + i);
				}
				s += d;
			}
		} else {
			for (i = 0; i < (width>>1); i++) {
				gdk_draw_line (window, black,
							   x + i, (unsigned) (y - ((s-0x00003fff)>>15)),
							   x + i, (unsigned) (y + ((s-0x00003fff)>>15)));
				if ((s & 0x00004000)) {
					gdk_draw_point (window, dark, x + i, (unsigned) (y + (s>>15)));
					gdk_draw_point (window, dark, x + i, (unsigned) (y - (s>>15)));
				}
				s += d;
			}
		}
		
		if (area) {
			gdk_gc_set_clip_rectangle (black, NULL);
			gdk_gc_set_clip_rectangle (dark, NULL);
		}
	} else {
		GdkPoint points[4];
		gint half_width, half_height, w;

	  draw_normal:

		if (GTK_IS_RANGE(widget)) {
			x++;		y++;
			width-=2;	height-=2;
		}
		if (state_type == GTK_STATE_INSENSITIVE) {
			black = style->dark_gc[state_type];
			dark  = style->mid_gc[state_type];
		}

	  draw_boxed_arrows:

		if (   GTK_IS_SPIN_BUTTON(widget)
			|| GTK_IS_NOTEBOOK(widget)
			|| GTK_IS_RANGE(widget))
		{
			if (shadow_type == GTK_SHADOW_IN) {
				shadow_type = (GtkShadowType)INTERFACE_SHADOW_IN;
			} else if (shadow_type == GTK_SHADOW_ETCHED_IN) {
				state_type = GTK_STATE_INSENSITIVE;
				shadow_type = GTK_SHADOW_OUT;
			}
		}
	
		width &= 0xfffe;
		height &= 0xfffe;

		w = MIN(width, height);
		x += (width-w)/2;
		y += (height-w)/2;
		width = height = w;
		
		half_width = width / 2;
		half_height = height / 2;

		switch (arrow_type) {
		  case GTK_ARROW_UP:
			points[0].x = x + half_width - 1;
			points[0].y = y;
			points[1].x = x;
			points[1].y = y + height - 1;
			points[2].x = x + width  - 1;
			points[2].y = y + height - 1;
			points[3].x = x + half_width;
			points[3].y = y;
			break;

		  case GTK_ARROW_DOWN:
			points[0].x = x + half_width ;
			points[0].y = y + height     - 1;
			points[1].x = x + width - 1;
			points[1].y = y;
			points[2].x = x;
			points[2].y = y;
			points[3].x = x + half_width - 1;
			points[3].y = y + height     - 1;
			break;

		  case GTK_ARROW_LEFT:
			points[0].x = x;
			points[0].y = y + half_height;
			points[1].x = x + width - 1;
			points[1].y = y + height - 1;
			points[2].x = x + width - 1;
			points[2].y = y;
			points[3].x = x;
			points[3].y = y + half_height - 1;
			break;

		  case GTK_ARROW_RIGHT:
			points[0].x = x + width - 1;
			points[0].y = y + half_height - 1;
			points[1].x = x;
			points[1].y = y;
			points[2].x = x;
			points[2].y = y + height - 1;
			points[3].x = x + width - 1;
			points[3].y = y + half_height;
			break;

		  default:
			return;
		}

		interface_draw_polygon (style, window, state_type, shadow_type, area, widget, detail,
						   points, 4, fill);
	}
}


static void
interface_draw_diamond (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   gchar         *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	GdkPoint	points[6];
	gint		half_width;
	gint		half_height;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	half_width	= width / 2;
	half_height	= height / 2;

	width = half_width * 2;
	height = half_height * 2 - 1;

	points[0].x = x+half_width-1;	points[0].y = y;
	points[1].x = x;				points[1].y = y+half_height-1;
	points[2].x = x+half_width-1;	points[2].y = y+height-1;
	points[3].x = x+half_width;		points[3].y = y+height-1;
	points[4].x = x+width-1;		points[4].y = y+half_height-1;
	points[5].x = x+half_width;		points[5].y = y;

	interface_draw_polygon (style, window, state_type, shadow_type, area, widget, detail,
					   points, 3, FALSE);
	interface_draw_polygon (style, window, state_type, shadow_type, area, widget, detail,
					   points+3, 3, FALSE);
}



static void
interface_draw_oval (GtkStyle      *style,
				GdkWindow     *window,
				GtkStateType   state_type,
				GtkShadowType  shadow_type,
				GdkRectangle  *area,
				GtkWidget     *widget,
				gchar         *detail,
				gint           x,
				gint           y,
				gint           width,
				gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	g_warning ("interface_draw_oval(): FIXME, this function is not implemented in GTK.");
}

static void
interface_draw_string (GtkStyle      *style,
				  GdkWindow     *window,
				  GtkStateType   state_type,
				  GdkRectangle  *area,
				  GtkWidget     *widget,
				  gchar         *detail,
				  gint           x,
				  gint           y,
				  const gchar   *string)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if (state_type == GTK_STATE_INSENSITIVE) {
		if (area) gdk_gc_set_clip_rectangle (style->white_gc, area);
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
		
		gdk_draw_string (window, style->font, style->white_gc, x+1, y+1, string);
		gdk_draw_string (window, style->font, style->fg_gc[state_type], x, y, string);
		
		if (area) gdk_gc_set_clip_rectangle (style->white_gc, NULL);
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
	} else {
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
		
		gdk_draw_string (window, style->font, style->fg_gc[state_type], x, y, string);
		
		if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
	}
}

static void 
interface_draw_box  (GtkStyle      *style,
				GdkWindow     *window,
				GtkStateType   state_type,
				GtkShadowType  shadow_type,
				GdkRectangle  *area,
				GtkWidget     *widget,
				gchar         *detail,
				gint           x,
				gint           y,
				gint           width,
				gint           height)
{
	GdkGC			*gc;
	GdkPixmap		*pixmap;
	InterfaceKnobStyle	knob_style;
	GtkOrientation	line_orientation;
	GtkOrientation	knob_orientation;
	gboolean		pad;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	gc = style->bg_gc[state_type];
	pixmap = style->bg_pixmap[state_type];
	
	knob_style = INTERFACE_KNOB_NONE;
	if (widget) {
		if (GTK_IS_BUTTON(widget)) {
			if ((detail) && !strcmp ("buttondefault", detail)) {
				x++; y++;
				width--; height--;
	
				if (widget && widget->parent) {
					style = widget->parent->style;
				
					gc = gtk_widget_get_style(widget->parent)->bg_gc[widget->parent->state];
					pixmap = widget->parent->style->bg_pixmap[widget->parent->state];
				} else {
					gc = style->bg_gc[GTK_STATE_INSENSITIVE];
					pixmap = style->bg_pixmap[GTK_STATE_INSENSITIVE];
				}
			} else if (   GTK_IS_TOGGLE_BUTTON(widget)
					   && ((GtkToggleButton *)widget)->active)
			{
				if (   ((GtkButton *)widget)->button_down
					&& ((GtkButton *)widget)->in_button)
				{
					gc = style->bg_gc[GTK_STATE_ACTIVE];
				} else {
					gc = style->base_gc[state_type];
				}
				pixmap = style->bg_pixmap[GTK_STATE_ACTIVE];
			} else if (GTK_IS_OPTION_MENU(widget) && detail && strcmp(detail, "optionmenu")==0) {
				y++;
				height -= 2;
			} else if (   ((GtkButton *)widget)->relief != GTK_RELIEF_NORMAL
					   && widget->parent)
			{
				gc = widget->parent->style->bg_gc[state_type];
			}
		} else if (GTK_IS_PANED(widget)) {
			if (style->engine_data)
				knob_style = ((InterfaceThemeData *)style->engine_data)->paned_knob;
			
			gdk_window_get_geometry(window, NULL, NULL, &width, &height, NULL);
			x = 0;
			y = 0;
			
			area = NULL;

			if (GTK_IS_HPANED(widget)) {
				y = ((GtkContainer*)widget)->border_width;
				height -= y*2;
			
				if (knob_style == INTERFACE_KNOB_NONE && width > 6) {
					x = (width-6)/2;
					width = 6;
				}
			} else {
				x = ((GtkContainer*)widget)->border_width;
				width -= x*2;

				if (knob_style == INTERFACE_KNOB_NONE && height > 6) {
					y = (height-6)/2;
					height = 6;
				}
			}

			if (knob_style == INTERFACE_KNOB_NONE) {
				shadow_type = INTERFACE_SHADOW_OUT;
			} else {
				knob_orientation = GTK_IS_VPANED(widget) ? GTK_ORIENTATION_HORIZONTAL
														 : GTK_ORIENTATION_VERTICAL;
				line_orientation = knob_orientation;
				shadow_type = INTERFACE_SHADOW_NONE;
				pad = TRUE;
				
				
			} 
		} else if (   GTK_IS_SCROLLBAR(widget)
				   && (detail && strcmp(detail, "slider")==0)
				   && (style->engine_data)
				   && (knob_style = ((InterfaceThemeData *)style->engine_data)->scrollbar_knob) != INTERFACE_KNOB_NONE)
		{
			knob_orientation = GTK_IS_VSCROLLBAR(widget) ? GTK_ORIENTATION_VERTICAL
														 : GTK_ORIENTATION_HORIZONTAL;
			line_orientation = knob_orientation ^ 1;
			pad = TRUE;
		}
	}
	
	interface_draw_background (style, window, gc, pixmap, state_type, area, widget, x, y, width, height);
	
	interface_draw_shadow (style, window, state_type, shadow_type, area,
					  widget, detail, x, y, width, height);

	if (knob_style != INTERFACE_KNOB_NONE) {
		if (knob_style == INTERFACE_KNOB_DIMPLE) {
			interface_draw_image (window, widget, area,
							 state_type == GTK_STATE_PRELIGHT ? INTERFACE_IMAGE_DIMPLE_PRELIGHT : INTERFACE_IMAGE_DIMPLE_NORMAL,
							 0, 0, x+width/2-3, y+height/2-3, 6, 6);
		} else {
			gint w, h, xt, yt;
			
			if (pad) {
				xt = style->klass->xthickness + 1;
				yt = style->klass->ythickness + 1;
			} else {
				xt = 0;
				yt = 0;
			}
			
			if (knob_orientation == GTK_ORIENTATION_VERTICAL) {
				h = MIN (height - xt - xt, width + height/7);
				y = y + height/2 - h/2;
				x = x + xt;
				w = width - xt - xt;
			} else {
				w = MIN (width - xt - xt, height + width/7);
				x = x + width/2 - w/2;
				y = y + yt;
				h = height - yt - yt;
			}
			
			if (knob_style == INTERFACE_KNOB_BUDS) {
				interface_draw_buds (style, window, state_type, GTK_SHADOW_ETCHED_OUT,
								area, widget, x, y, w, h, line_orientation);
			} else if (knob_style == INTERFACE_KNOB_HOLES) {
				interface_draw_buds (style, window, state_type, GTK_SHADOW_ETCHED_IN,
								area, widget, x, y, w, h, line_orientation);
			} else if (knob_style == INTERFACE_KNOB_LINES) {
				interface_draw_lines (style, window, state_type, GTK_SHADOW_OUT,
								 area, widget, x, y, w, h,
								 line_orientation);
			}
		}
	}
}


static void 
interface_draw_flat_box (GtkStyle      *style,
				    GdkWindow     *window,
				    GtkStateType   state_type,
				    GtkShadowType  shadow_type,
				    GdkRectangle  *area,
				    GtkWidget     *widget,
				    gchar         *detail,
				    gint           x,
				    gint           y,
				    gint           width,
				    gint           height)
{
	GdkGC *gc;
	GdkPixmap *pixmap;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	if (detail && !strcmp(detail, "scale trough")) {
		state_type = GTK_STATE_NORMAL;
	}

	gc = style->bg_gc[state_type];
	pixmap = style->bg_pixmap[state_type];

	if (GTK_IS_RADIO_BUTTON(widget) || GTK_IS_CHECK_BUTTON(widget))
		return;
	if (GTK_IS_ENTRY(widget)) {
		pixmap = NULL;
		gc = style->base_gc[state_type];
		if (!GTK_EDITABLE(widget)->editable && state_type != GTK_STATE_SELECTED)
			gc = style->base_gc[GTK_STATE_INSENSITIVE];
	}

	interface_draw_background(style, window, gc, pixmap, state_type, area, widget, x, y, width, height);

	if (detail) {
		/* if (GTK_IS_WINDOW(widget) && !strcmp("base", detail)) {
			interface_draw_shadow (style, window, GTK_STATE_NORMAL,
							  (GTK_WINDOW(widget)->type == GTK_WINDOW_POPUP)
							  ? INTERFACE_SHADOW_OUT : INTERFACE_SHADOW_THIN_OUT,
							  area, widget, NULL, x, y, width, height);
		} else */
		
		if (!strcmp("tooltip", detail)) {
			InterfaceShadowType shadow_type = ((InterfaceThemeData *)style->engine_data)->tooltip_shadow;
			interface_draw_shadow (style, window, GTK_STATE_NORMAL, shadow_type, area, widget,
							  NULL, x, y, width, height);
		}
	}
}

static void 
interface_draw_check   (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   gchar         *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	GdkGC *gc = style->bg_gc[state_type];

	if ((detail) && ((!strcmp ("checkbutton", detail)))) {
		GdkPixmap	*base_pixmap, *norm_pixmap, *prel_pixmap, *down_pixmap, *inse_pixmap;
		GdkBitmap	*mask;
		GdkPixmap	*pixmap;
		gboolean	down=FALSE;
		gboolean	active=FALSE;
		
		state_type = GTK_WIDGET_STATE(widget);

		if (shadow_type == GTK_SHADOW_IN)
			down = TRUE;

		if (   GTK_IS_BUTTON(widget)
			&& ((GtkButton *)widget)->button_down
			&& ((GtkButton *)widget)->in_button)
		{
			active = TRUE;
			shadow_type = GTK_SHADOW_IN;
		}

		norm_pixmap = interface_pixmap_get(window, style, style, INTERFACE_IMAGE_CHECK_BUTTON_NORMAL);
		prel_pixmap = interface_pixmap_get(window, style, style, INTERFACE_IMAGE_CHECK_BUTTON_PRELIGHT);
		down_pixmap = interface_pixmap_get(window, style, style, INTERFACE_IMAGE_CHECK_BUTTON_ACTIVE);
		inse_pixmap = interface_pixmap_get(window, style, style, INTERFACE_IMAGE_CHECK_BUTTON_INSENSITIVE);
		mask = interface_masks[INTERFACE_MASK_CHECK];

		y  = widget->allocation.y + widget->allocation.height/2 - height/2;
		
		interface_draw_box (style, window, state_type, shadow_type,
					   area, widget, detail, x, y, width, height);
		
		if (down) {
			if (state_type == GTK_STATE_INSENSITIVE) {
				pixmap = inse_pixmap;
			} else if (active) {
				pixmap = down_pixmap;
			} else {
				pixmap = (state_type==GTK_STATE_PRELIGHT)? prel_pixmap : norm_pixmap;
			}

			if (pixmap)
				interface_draw_pixmap (window, gc, area, pixmap, mask, 0, 0, x+3, y+3,
								  INTERFACE_CHECK_MENU_SIZE, INTERFACE_CHECK_MENU_SIZE);
		}
	} else if ((detail) && ((!strcmp ("check", detail)))) {
		GdkPixmap	*norm_pixmap, *prel_pixmap, *inse_pixmap;
		GdkBitmap	*mask;
		GdkPixmap	*pixmap;
		gint		y2;
		gboolean	down;

		norm_pixmap = interface_pixmap_get(window, style, style, INTERFACE_IMAGE_CHECK_MENU_NORMAL);
		prel_pixmap = interface_pixmap_get(window, style, style, INTERFACE_IMAGE_CHECK_MENU_PRELIGHT);
		inse_pixmap = interface_pixmap_get(window, style, style, INTERFACE_IMAGE_CHECK_MENU_INSENSITIVE);
		mask = interface_masks[INTERFACE_MASK_CHECK];

		y  = (widget->allocation.height - INTERFACE_CHECK_MENU_SIZE)/2;
		y2 = GTK_CONTAINER(widget)->border_width
			 + style->klass->ythickness
			 + 1
			 + style->font->ascent
			 - INTERFACE_CHECK_MENU_SIZE;
		y = MAX(y, y2);
		
		width  = INTERFACE_CHECK_MENU_SIZE;
		height = INTERFACE_CHECK_MENU_SIZE;

		interface_draw_background (style, window, NULL, NULL, state_type, area, widget, x, y, width, height);
	/*
		interface_draw_flat_box (style, window, state_type, shadow_type, area,
							widget, detail, x, y, width, height);
	*/
		if (GTK_IS_MENU_ITEM(widget)) {
			down = GTK_CHECK_MENU_ITEM(widget)->active;
		} else {
			if (   (shadow_type == GTK_SHADOW_IN  && state_type != GTK_STATE_PRELIGHT)
				|| (shadow_type == GTK_SHADOW_OUT && state_type == GTK_STATE_PRELIGHT))
				down = TRUE;
			else
				down = FALSE;
		}		
		
		if (down) {
			switch (state_type) {
			  case GTK_STATE_INSENSITIVE:	pixmap = inse_pixmap; break;
			  case GTK_STATE_PRELIGHT:		pixmap = prel_pixmap; break;
			  default:						pixmap = norm_pixmap;
			}
			if (pixmap)
				interface_draw_pixmap (window, gc, area, pixmap, mask, 0, 0, x, y,
								  INTERFACE_CHECK_MENU_SIZE, INTERFACE_CHECK_MENU_SIZE);
		}
	} else
		gtk_paint_box (style, window, state_type, shadow_type, area, widget, detail,
					   x, y, width, height);
}


static void 
interface_draw_option  (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   gchar         *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	GdkGC *gc = style->bg_gc[state_type];
	if ((detail) && ((!strcmp ("radiobutton", detail))))
	{
		GdkPixmap *pixmap[] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL};
		GdkBitmap *mask;
		gint index;
		
		GtkStyle *style2 = style;

		if (widget->parent && widget->parent->style)
			style2 = widget->parent->style;
		else
			g_print("widget does not have any parent\n");

		pixmap[0] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_NORMAL_OUT);
		pixmap[1] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_PRELIGHT_OUT);
		pixmap[2] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_INSENSITIVE_OUT);

		pixmap[5] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_INSENSITIVE_IN);
		pixmap[6] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_ACTIVE_IN);
		pixmap[3] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_NORMAL_IN);
		pixmap[4] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_PRELIGHT_IN);
		mask = interface_masks[INTERFACE_MASK_RADIO];
		
		if (shadow_type == GTK_SHADOW_IN) {
			index = 3;
		} else {
			index = 0;
		}
		if (state_type == GTK_STATE_INSENSITIVE) {
			index += 2;
		} else if (GTK_BUTTON(widget)->button_down && GTK_BUTTON(widget)->in_button) {
			index = 6;
		} else if (state_type == GTK_STATE_PRELIGHT) {
			index += 1;
		}

		if (pixmap[0]) {
			y += (height-INTERFACE_RADIO_SIZE)/2;
			interface_draw_pixmap (window, gc, area, pixmap[index], mask, 0, 0, x, y,
							  INTERFACE_RADIO_SIZE, INTERFACE_RADIO_SIZE);
		}
	} else if ((detail) && ((!strcmp ("option", detail)))) {
		GdkPixmap	*pixmaps[] = {NULL, NULL, NULL};
		GdkBitmap	*mask;
		GdkPixmap	*pixmap;
		GtkStyle	*style2 = style;
		gint		y2;
		
		style2 = (widget->parent && widget->parent->style) ? widget->parent->style : style;

		pixmaps[0] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_MENU_NORMAL);
		pixmaps[1] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_MENU_PRELIGHT);
		pixmaps[2] = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_RADIO_MENU_INSENSITIVE);
		
		y  = (widget->allocation.height - INTERFACE_RADIO_MENU_SIZE)/2;
		y2 = GTK_CONTAINER(widget)->border_width
			 + style->klass->ythickness
			 + 1
			 + style->font->ascent
			 - INTERFACE_RADIO_MENU_SIZE;
		y = MIN(y, y2);
		
		x = x + (width - INTERFACE_RADIO_MENU_SIZE)/2;
		
		width  = INTERFACE_RADIO_MENU_SIZE;
		height = INTERFACE_RADIO_MENU_SIZE;

		interface_draw_flat_box (style, window, state_type, shadow_type, area,
							widget, detail, x, y, width, height);
		if (shadow_type == GTK_SHADOW_IN) {
			switch (state_type) {
			  case GTK_STATE_INSENSITIVE:	pixmap = pixmaps[2]; break;
			  case GTK_STATE_PRELIGHT:		pixmap = pixmaps[1]; break;
			  default:						pixmap = pixmaps[0];
			}
			mask = interface_masks[INTERFACE_MASK_RADIO_ITEM];
			
			if (pixmap)
				interface_draw_pixmap(window, gc, area, pixmap, mask, 0, 0, x, y, width, height);
		}
	} else
		gtk_paint_diamond (style, window, state_type, shadow_type, area, widget, 
						   detail, x, y, width, height);
}


static void 
interface_draw_cross   (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   gchar         *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	g_warning ("interface_draw_cross(): FIXME, this functionality is not implemented in GTK+.");
}

static void 
interface_draw_ramp    (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   gchar         *detail,
				   GtkArrowType   arrow_type,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	g_warning ("interface_draw_ramp(): FIXME, this functionality is not implemented in GTK+.");
}

static void interface_draw_tab     (GtkStyle      *style,
						       GdkWindow     *window,
						       GtkStateType   state_type,
						       GtkShadowType  shadow_type,
						       GdkRectangle  *area,
						       GtkWidget     *widget,
						       gchar         *detail,
						       gint           x,
						       gint           y,
						       gint           width,
						       gint           height)
{
	gint w, h;
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((detail) && (!strcmp ("optionmenutab", detail))) {	
		/* h = MIN(height, (width/2 + 0x0001) & 0xfffe);
		w = width + (h & 0x0001);
		y += (height - h)/2;
		width = w;
		height = h; */
		
		gtk_paint_arrow (style, window, state_type, shadow_type, area, widget, detail, GTK_ARROW_DOWN, 0, x, y, width, height);
	} else {
		gtk_paint_box (style, window, state_type, shadow_type, area, widget, detail,
					   x, y, width, height);
	}
}


static void 
interface_draw_shadow_gap (GtkStyle       *style,
				      GdkWindow      *window,
				      GtkStateType    state_type,
				      GtkShadowType   shadow_type,
				      GdkRectangle   *area,
				      GtkWidget      *widget,
				      gchar          *detail,
				      gint            x,
				      gint            y,
				      gint            width,
				      gint            height,
				      GtkPositionType gap_side,
				      gint            gap_x,
				      gint            gap_width)
{
	GdkPoint points[7];
	int i=0, n=4;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	switch (gap_side) {
	  case GTK_POS_TOP:
		if (gap_x > 0) {
			n++;
			points[i].x   = x+gap_x;
			points[i++].y = y;
		}
		goto cont_top;
		
	  case GTK_POS_BOTTOM:
	    if (gap_x+gap_width < width) {
			n++;
			points[i].x   = x + gap_x + gap_width;
			points[i++].y = y + height-1;
		}
		goto cont_bottom;
	
	  case GTK_POS_LEFT:
		if (gap_x+gap_width < height) {
			n++;
			points[i].x   = x;
			points[i++].y = y + gap_x + gap_width;
		}
		goto cont_left;
	
	  case GTK_POS_RIGHT:
	    if (gap_x > 0) {
			n++;
			points[i].x   = x+width-1;
			points[i++].y = y+gap_x;
		}
		goto cont_right;
	}

	while(1) {
	  cont_top:
		points[i].x	  = x;
		points[i++].y = y;
		if (i==n) {
			if (gap_x > 0) {
				n++;
				points[i].x = x;
				points[i].y = y + gap_x;
			}
			goto done;
		}

	  cont_left:
		points[i].x	  = x;
		points[i++].y = y+height-1;
		if (i==n) {
			if (gap_x > 0) {
				n++;
				points[i].x = x + gap_x;
				points[i].y = y + height - 1;
			}
			goto done;
		}

	  cont_bottom:
		points[i].x	  = x+width-1;
		points[i++].y = y+height-1;
		if (i==n) {
			if (gap_x + gap_width < width) {
				n++;
				points[i].x = x + width - 1;
				points[i].y = y + gap_x + gap_width;
			}
			goto done;
		}

	  cont_right:
		points[i].x	  = x+width-1;
		points[i++].y = y;
		if (i==n) {
			if (gap_x + gap_width < width) {
				n++;
				points[i].x = x + gap_x + gap_width;;
				points[i].y = y;
			}
			goto done;
		}
	};
  done:
	interface_draw_polygon (style, window, state_type, shadow_type, area,
					   widget, detail, points, n, FALSE);

	return;
}


static void 
interface_draw_box_gap (GtkStyle       *style,
				   GdkWindow      *window,
				   GtkStateType    state_type,
				   GtkShadowType   shadow_type,
				   GdkRectangle   *area,
				   GtkWidget      *widget,
				   gchar          *detail,
				   gint            x,
				   gint            y,
				   gint            width,
				   gint            height,
				   GtkPositionType gap_side,
				   gint            gap_x,
				   gint            gap_width)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	interface_draw_background(style, window, NULL, NULL, state_type, area, widget, x, y, width, height);

	interface_draw_shadow_gap(style, window, state_type, shadow_type, area, widget, detail,
						 x, y, width, height, gap_side, gap_x, gap_width);

	return;
}


static void 
interface_draw_extension (GtkStyle       *style,
				     GdkWindow      *window,
				     GtkStateType    state_type,
				     GtkShadowType   shadow_type,
				     GdkRectangle   *area,
				     GtkWidget      *widget,
				     gchar          *detail,
				     gint            x,
				     gint            y,
				     gint            width,
				     gint            height,
				     GtkPositionType gap_side)
{
	GtkStyle	*style2;
	GdkPoint	points[8];
	GdkPixmap	*pixmap;
	GdkBitmap	*mask;
	GtkStateType state2;
	guint		xt, yt;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	style2 = (widget->parent) ? widget->parent->style : style;
	state2 = (widget->parent) ? widget->parent->state : GTK_STATE_INSENSITIVE;

	xt = style->klass->xthickness;
	yt = style->klass->ythickness;

	switch (gap_side) {
	  case GTK_POS_BOTTOM:
		points[0].x = x+width-1;	points[0].y = y+height-1;
		points[1].x = x+width-1;	points[1].y = y+5;
		
		points[2].x = x+width-3;	points[2].y = y+2;
		points[3].x = x+width-6;	points[3].y = y;
		
		points[4].x = x+5;			points[4].y = y;
		points[5].x = x+2;			points[5].y = y+2;

		points[6].x = x;			points[6].y = y+5;
		points[7].x = x;			points[7].y = y+height-1;

		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x, y, 5, 5);
		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x+width-5, y, 5, 5);
		
		x += xt;	width	-= xt+xt;
		y += yt;	height	-= yt;
		break;

	  case GTK_POS_RIGHT:
		points[0].y = y;			points[0].x = x+width-1;
		points[1].y = y;			points[1].x = x+5;
		points[2].y = y+2;			points[2].x = x+2;
		points[3].y = y+5;			points[3].x = x;
		points[4].y = y+height-6;	points[4].x = x;
		points[5].y = y+height-3;	points[5].x = x+2;
		points[6].y = y+height-1;	points[6].x = x+5;
		points[7].y = y+height-1;	points[7].x = x+width-1;

		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x, y, 5, 5);
		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x, y+height-5, 5, 5);
		
		x += xt;	width 	-= xt;
		y += yt;	height	-= yt+yt;
		break;

	  case GTK_POS_LEFT:
		points[0].y = y+height-1;	points[0].x = x;
		points[1].y = y+height-1;	points[1].x = x+width-6;
		points[2].y = y+height-3;	points[2].x = x+width-3;
		points[3].y = y+height-6;	points[3].x = x+width-1;
		points[4].y = y+5;			points[4].x = x+width-1;
		points[5].y = y+2;			points[5].x = x+width-3;
		points[6].y = y;			points[6].x = x+width-6;
		points[7].y = y;			points[7].x = x;

		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x+width-5, y, 5, 5);
		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x+width-5, y+height-5, 5, 5);
		
					width 	-= xt;
		y += yt;	height	-= yt+yt;
		break;

	  case GTK_POS_TOP:
		points[0].x = x;			points[0].y = y;
		points[1].x = x;			points[1].y = y+height-6;
		points[2].x = x+2;			points[2].y = y+height-3;
		points[3].x = x+5;			points[3].y = y+height-1;
		points[4].x = x+width-6;	points[4].y = y+height-1;
		points[5].x = x+width-3;	points[5].y = y+height-3;
		points[6].x = x+width-1;	points[6].y = y+height-6;
		points[7].x = x+width-1;	points[7].y = y;

		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x, y+height-5, 5, 5);
		interface_draw_background (style2, window, NULL, NULL, state2, area, widget, x+width-5, y+height-5, 5, 5);
		
		x += xt;	width	-= xt+xt;
					height	-= yt;
		break;

	  default:
		return;
	}

	interface_draw_background(style, window, NULL, NULL, state_type, area, widget, x, y, width, height);
	
	interface_draw_polygon (style, window, state_type, shadow_type,
					   area, widget, detail, points, 8, FALSE);

	/*
	if (xt<2 || yt<2)
		return;
	*/
	
	switch (state_type) {
	  case GTK_STATE_ACTIVE:
		pixmap = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_NOTEBOOK_CORNER_ACTIVE);
		break;
	  case GTK_STATE_INSENSITIVE:
		pixmap = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_NOTEBOOK_CORNER_INSENSITIVE);
		break;
	  default:
		pixmap = interface_pixmap_get(window, style, style2, INTERFACE_IMAGE_NOTEBOOK_CORNER_NORMAL);
	}
	mask = interface_masks[INTERFACE_MASK_CORNER];

	if (pixmap && mask) {
		if (gap_side == GTK_POS_BOTTOM || gap_side == GTK_POS_RIGHT) {
			interface_draw_pixmap (window, style->bg_gc[state_type], area,
							  pixmap, mask, 0, 0, x-xt, y-yt, 7, 7);
		}
		if (gap_side == GTK_POS_BOTTOM || gap_side == GTK_POS_LEFT) {
			interface_draw_pixmap (window, style->bg_gc[state_type], area,
							  pixmap, mask, 6, 0, x+width-(7-xt), y-yt, 7, 7);
		}
		if (gap_side == GTK_POS_TOP || gap_side == GTK_POS_RIGHT) {
			interface_draw_pixmap (window, style->bg_gc[state_type], area,
							  pixmap, mask, 0, 6, x-xt, y+height-(7-yt), 7, 7);
		}
		if (gap_side == GTK_POS_TOP || gap_side == GTK_POS_LEFT) {
			interface_draw_pixmap (window, style->bg_gc[state_type], area,
							  pixmap, mask, 6, 6, x+width-(7-xt), y+height-(7-yt), 7, 7);
		}
	} else {
		g_print("no pixmap\n");
	}
}


static void 
interface_draw_focus (GtkStyle      *style,
				 GdkWindow     *window,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 gchar         *detail,
				 gint           x,
				 gint           y,
				 gint           width,
				 gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (GTK_WIDGET_STATE(widget) == GTK_STATE_INSENSITIVE)
		return;

	if ((width == -1) && (height == -1)) {
		gdk_window_get_size (window, &width, &height);
		width -= 1;
		height -= 1;
	} else if (width == -1) {
		gdk_window_get_size (window, &width, NULL);
		width -= 1;
	} else if (height == -1) {
		gdk_window_get_size (window, NULL, &height);
		height -= 1;
	}
	if (area) {
		gdk_gc_set_clip_rectangle (style->white_gc, area);
		gdk_gc_set_clip_rectangle (style->black_gc, area);
	}
	if ((detail) && !strcmp (detail, "add-mode")) {
		gdk_gc_set_line_attributes (style->black_gc, 1, GDK_LINE_ON_OFF_DASH, 0, 0);
		gdk_gc_set_dashes (style->black_gc, 0, "\4\4", 2);
		
		gdk_draw_rectangle (window,
						   style->black_gc, FALSE,
						   x, y, width, height);

		gdk_gc_set_line_attributes (style->black_gc, 1, GDK_LINE_SOLID, 0, 0);
	} else {
		if (GTK_IS_OPTION_MENU(widget)) {
			y++;
			height-=2;
		}
		if (GTK_IS_CHECK_BUTTON(widget) || GTK_IS_RADIO_BUTTON(widget) || 
			(detail && !strcmp("tab", detail)))
		{
			GdkGC		*gc;
			static const gchar data[] = { 0x01, 0x02 };
			
			static GdkBitmap *bm = NULL;
			
			if (!bm)
				bm = gdk_bitmap_create_from_data(window, data, 2, 2);

			if (bm) {
				gc = style->black_gc;
				
				width  = (width -  1) | 1;
				height = (height - 1) | 1;
				
				gdk_gc_set_stipple(gc, bm);
				gdk_gc_set_fill(gc, GDK_STIPPLED);
				gdk_gc_set_ts_origin(gc, x, y);

				gdk_draw_rectangle (window, gc, FALSE, x, y, width-1, height-1);
	
				gdk_gc_set_fill(gc, GDK_SOLID);
			}
		} else if (GTK_IS_EDITABLE(widget)) {
			interface_draw_box (style, window, GTK_STATE_NORMAL, GTK_SHADOW_IN, area, widget, detail,
							  x,y, width+1,height+1);
			
			if (style->engine_data && ((InterfaceThemeData *)style->engine_data)->thin) {
				/* ? */
			} else {
				gdk_draw_line (window, style->white_gc, x+1, y+height, x+width-1, y+height);
				gdk_draw_line (window, style->white_gc, x+width, y+1, x+width, y+height);
				gdk_draw_line (window, style->black_gc, x+1, y+height-1, x+width-2, y+height-1);
				gdk_draw_line (window, style->black_gc, x+width-1, y+1, x+width-1, y+height-1);
			}
		} else {
			gdk_draw_rectangle (window,
								style->black_gc, FALSE,
								x, y, width, height);
		}
	}

	if (area) {
		gdk_gc_set_clip_rectangle (style->white_gc, NULL);
		gdk_gc_set_clip_rectangle (style->black_gc, NULL);
	}
}


static void 
interface_draw_slider (GtkStyle      *style,
				  GdkWindow     *window,
				  GtkStateType   state_type,
				  GtkShadowType  shadow_type,
				  GdkRectangle  *area,
				  GtkWidget     *widget,
				  gchar         *detail,
				  gint           x,
				  gint           y,
				  gint           width,
				  gint           height,
				  GtkOrientation orientation)
{
	GdkGC *lgc, *dgc, *mgc;
	gint i, w, t, xt, yt;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	xt = MAX(0, style->klass->xthickness - 1);
	yt = MAX(0, style->klass->ythickness - 1);
	
	t = 0;
	if (GTK_WIDGET_HAS_FOCUS(widget)) {
		t = 1;
		xt++;
		yt++;
		
		interface_draw_focus (style, window, area, widget, detail, x, y, width, height);
	}
	
	gtk_paint_box (style, window, state_type, shadow_type,
				   area, widget, detail, x+t, y+t, width-t-t, height-t-t);
	
	lgc = style->light_gc[state_type];
	dgc = style->dark_gc[state_type];
	mgc = style->mid_gc[state_type];
	
	if (area) {
		gdk_gc_set_clip_rectangle (lgc, area);
		gdk_gc_set_clip_rectangle (dgc, area);
		gdk_gc_set_clip_rectangle (mgc, area);
	}
	
	if (orientation == GTK_ORIENTATION_HORIZONTAL) {
		w = MIN(width, height + width/6);
		x = x + width/2 - w/2;
		y += yt;
		height -= yt + yt;
		for (i=x; i<x+w; i+=3) {
			gdk_draw_line (window, dgc,		i,  	y+1,	i,   	y+height-2);
			gdk_draw_line (window, lgc,		i+1,	y+1,	i+1,	y+height-2);
			gdk_draw_point (window, mgc,	i,		y);
			gdk_draw_point (window, mgc,	i+1,	y+height-1);
		}
	} else {
		height += 32;
		w = MIN(height, width + height/6);
		y = y + height/2 - w/2;
		x += xt;
		width -= xt + xt;
		for (i=y - 13; i<x+w; i+=3) {
			gdk_draw_line (window, dgc,		x+1,	i,		x+width-2,	i  );
			gdk_draw_line (window, lgc,		x+1,	i+1,	x+width-2,	i+1);
			gdk_draw_point (window, mgc,	x,		i);
			gdk_draw_point (window, mgc,	x+width-1, i+1);
		}
	}
	if (area) {
		gdk_gc_set_clip_rectangle (lgc, NULL);
		gdk_gc_set_clip_rectangle (dgc, NULL);
		gdk_gc_set_clip_rectangle (mgc, NULL);
	}
}

static void 
interface_draw_handle  (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   gchar         *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height,
				   GtkOrientation orientation)
{
	gint	xx, yy;
	gint	xthick, ythick;
	GdkGC *light_gc, *dark_gc, *mid_gc, *white_gc;
	InterfaceKnobStyle knob_style;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	light_gc	= style->light_gc[state_type];
	dark_gc		= style->dark_gc[state_type];
	mid_gc		= style->mid_gc[state_type];
	white_gc	= style->white_gc;
	
	xthick = style->klass->xthickness;
	ythick = style->klass->ythickness;
	
	knob_style = INTERFACE_KNOB_NONE;
	if (style->engine_data)
		knob_style = ((InterfaceThemeData *)style->engine_data)->handle_knob;

	if (knob_style == INTERFACE_KNOB_LINES)
		shadow_type = GTK_SHADOW_NONE;

	if (shadow_type != GTK_SHADOW_NONE) {
		switch (shadow_type) {
		  case GTK_SHADOW_IN:
			shadow_type = (GtkShadowType)INTERFACE_SHADOW_IN;
			break;
	
		  case GTK_SHADOW_OUT:
			shadow_type = (GtkShadowType)INTERFACE_SHADOW_OUT;
			break;
	
		  case GTK_SHADOW_ETCHED_IN:
			shadow_type = (GtkShadowType)INTERFACE_SHADOW_OUT;
			state_type = GTK_STATE_INSENSITIVE;
			break;
		
		  default:
			shadow_type = GTK_SHADOW_NONE;
		}
		interface_draw_box (style, window, state_type, shadow_type, area, widget, NULL, x, y,
					   width  - ((orientation==GTK_ORIENTATION_VERTICAL)?1:0),
					   height - ((orientation==GTK_ORIENTATION_HORIZONTAL)?1:0));
	}
	
	if (   shadow_type != GTK_SHADOW_NONE
		&& detail
		&& (   !strcmp(detail, "handlebox")
			|| !strcmp(detail, "dockitem")))
		{
		gdk_gc_set_clip_rectangle (mid_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		if (orientation == GTK_ORIENTATION_VERTICAL) {
			gdk_draw_line(window, light_gc, x+width-1, y, x+width-1, y+height-2);
			gdk_draw_point(window, mid_gc, x+width-1, y+height-1);
			width--;
		} else {
			gdk_draw_line(window, light_gc, x+1, y+height-1, x+width-2, y+height-1);
			gdk_draw_point(window, mid_gc, x+width-1, y+height-1);
			height--;
		}
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
	}

	if (knob_style == INTERFACE_KNOB_LINES) {
		interface_draw_lines (style, window, state_type, GTK_SHADOW_OUT, area, widget,
						 x+2, y+2, width-4, height-4, orientation);
	} else {
		if (knob_style == INTERFACE_KNOB_HOLES) {
			shadow_type = GTK_SHADOW_ETCHED_IN;
		} else if (knob_style == INTERFACE_KNOB_BUDS) {
			shadow_type = GTK_SHADOW_ETCHED_OUT;
		} else {
			shadow_type = GTK_SHADOW_OUT;
		}	
		
		interface_draw_buds (style, window, state_type, shadow_type, area, widget,
						x+2, y+2, width-4, height-4, orientation);
	}
}

static void
interface_draw_lines (GtkStyle		*style,
				 GdkWindow		*window,
				 GtkStateType	state_type,
				 GtkShadowType	shadow_type,
				 GdkRectangle	*area,
				 GtkWidget		*widget,
				 gint			x,
				 gint			y,
				 gint			width,
				 gint			height,
				 GtkOrientation	orientation)
{
	GdkGC	*light_gc, *dark_gc, *mid_gc;
	gint	i;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if (shadow_type == GTK_SHADOW_IN) {
		dark_gc  = style->dark_gc[state_type];
		light_gc = style->light_gc[state_type];
	} else {
		light_gc = style->dark_gc[state_type];
		dark_gc	 = style->light_gc[state_type];
	}
	mid_gc = style->mid_gc[state_type];
	
	if (area) {
		gdk_gc_set_clip_rectangle (dark_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		gdk_gc_set_clip_rectangle (mid_gc, area);
	}

	if (orientation == GTK_ORIENTATION_HORIZONTAL) {
		y += (height % 3) & 0x01;
		for (i = y; i < y+height; i += 3) {
			gdk_draw_line  (window, dark_gc,	x,			i,		x+width-2,	i  );
			gdk_draw_line  (window, light_gc,	x+1, 		i+1,	x+width-1,	i+1);
			gdk_draw_point (window, mid_gc,		x,			i+1);
			gdk_draw_point (window, mid_gc,		x+width-1,	i);
		}
	} else {
		x += (width % 3) & 0x01;
		for (i = x; i < x+width; i += 3) {
			gdk_draw_line  (window, dark_gc,	i,		y,   	i,   y+height-2);
			gdk_draw_line  (window, light_gc,	i+1,	y+1, 	i+1, y+height-1);
			gdk_draw_point (window, mid_gc,		i+1,	y);
			gdk_draw_point (window, mid_gc,		i,		y+height-1);
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
	}
}

static void
interface_draw_buds (GtkStyle		*style,
				GdkWindow		*window,
				GtkStateType	state_type,
				GtkShadowType	shadow_type,
				GdkRectangle	*area,
				GtkWidget		*widget,
				gint			x,
				gint			y,
				gint			width,
				gint			height,
				GtkOrientation	orientation)
{
	GdkGC	*light_gc, *dark_gc, *mid_gc;
	gint	x2, y2;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (shadow_type == GTK_SHADOW_OUT || shadow_type == GTK_SHADOW_ETCHED_OUT) {
		light_gc = style->white_gc;
		mid_gc	 = style->mid_gc[state_type];
		dark_gc  = style->dark_gc[state_type];
	} else {
		light_gc = style->dark_gc[state_type];;
		mid_gc = style->mid_gc[state_type];
		dark_gc	 = style->white_gc;
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (dark_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		gdk_gc_set_clip_rectangle (mid_gc, area);
	}

	if (   shadow_type == GTK_SHADOW_OUT
		|| shadow_type == GTK_SHADOW_IN
		&& orientation == GTK_ORIENTATION_VERTICAL)
	{
		/* netscape - style */
		for (y2 = y+1; y2 < y+height-1; y2 += 3) {
			for (x2 = x; x2 < x+width-1; x2 += 6) {
				gdk_draw_point (window, light_gc,	x2,		y2);
				gdk_draw_point (window, dark_gc,	x2 + 1, y2 + 1);
				gdk_draw_point (window, mid_gc,		x2 + 1,	y2);
				gdk_draw_point (window, mid_gc,		x2,		y2 + 1);
			}
		}
		for (y2 = y; y2 < y+height-1; y2 += 3) {
			for (x2 = x+3; x2 < x+width-1; x2 += 6) {
				gdk_draw_point (window, light_gc,	x2,		y2    );
				gdk_draw_point (window, dark_gc,	x2 + 1, y2 + 1);
				gdk_draw_point (window, mid_gc,		x2 + 1,	y2    );
				gdk_draw_point (window, mid_gc,		x2,		y2 + 1);
			}
		}
	} else {
		/* mac - style */
		x += (width % 3) & 0x01;
		y += (height% 3) & 0x01;
		
		for (y2 = y; y2 < y+height-1; y2 += 3) {
			for (x2 = x; x2 < x+width-1; x2 += 3) {
				gdk_draw_point (window, light_gc,	x2,		y2	);
				gdk_draw_point (window, mid_gc,		x2+1,	y2	);
				gdk_draw_point (window, mid_gc,		x2,		y2+1);
				gdk_draw_point (window, dark_gc,	x2+1,	y2+1);
			}
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
	}
}


static void interface_draw_background (GtkStyle *style, GdkWindow *window, GdkGC *gc, GdkPixmap *pixmap,
								  GtkStateType state_type, GdkRectangle *area, GtkWidget *widget,
								  gint x, gint y, gint width, gint height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	if (!gc) {
		gc = style->bg_gc[state_type];
		if (!pixmap)
			pixmap = style->bg_pixmap[state_type];
	}
	
	if (   (   gdk_window_get_type(window) == GDK_WINDOW_PIXMAP
		    || !pixmap || widget || GTK_WIDGET_NO_WINDOW(widget))
		&& (pixmap != (GdkPixmap *)GDK_PARENT_RELATIVE))
	{
		if (area) gdk_gc_set_clip_rectangle (gc, area);
		if (   pixmap
			&& (gdk_window_get_type(window) != GDK_WINDOW_PIXMAP)
			&& (style->bg[state_type].red   == style->rc_style->bg[state_type].red)
			&& (style->bg[state_type].green == style->rc_style->bg[state_type].green)
			&& (style->bg[state_type].blue  == style->rc_style->bg[state_type].blue))
		{
			gdk_gc_set_fill(gc, GDK_TILED);
			gdk_gc_set_tile(gc, pixmap);
			gdk_gc_set_ts_origin(gc, 0, 0);
		}
		gdk_draw_rectangle(window, gc, TRUE, x, y, width, height);
		gdk_gc_set_fill(gc, GDK_SOLID);
		
		if (area) gdk_gc_set_clip_rectangle (gc, NULL);
	} else {
		if (widget && !GTK_WIDGET_NO_WINDOW(widget)) {
			if (pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE)
				gdk_window_set_back_pixmap(window, NULL, TRUE);
			else
				gdk_window_set_back_pixmap(window, pixmap, FALSE);
		}
		gdk_window_clear_area (window, x, y, width, height);
	}
/*
	if (   state_type == GTK_STATE_NORMAL
		&& (pixmap == NULL || pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE)
		&& (gc == NULL || gc == style->bg_gc[GTK_STATE_NORMAL]))
	{
		gdk_window_clear_area (window, x, y, width, height);
	} else 
	{
		if (!gc) {
			gc = style->bg_gc[state_type];
			if (!pixmap)
				pixmap = style->bg_pixmap[state_type];
		}
		if (pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE)
			pixmap = NULL;
		
		if (area) gdk_gc_set_clip_rectangle (gc, area);
		if (   pixmap
			&& (gdk_window_get_type(window) != GDK_WINDOW_PIXMAP)
			&& (style->bg[state_type].red   == style->rc_style->bg[state_type].red)
			&& (style->bg[state_type].green == style->rc_style->bg[state_type].green)
			&& (style->bg[state_type].blue  == style->rc_style->bg[state_type].blue))
		{
			gdk_gc_set_fill(gc, GDK_TILED);
			gdk_gc_set_tile(gc, pixmap);
			gdk_gc_set_ts_origin(gc, 0, 0);
		}
		gdk_draw_rectangle(window, gc, TRUE, x, y, width, height);
		gdk_gc_set_fill(gc, GDK_SOLID);
		
		if (area) gdk_gc_set_clip_rectangle (gc, NULL);
	}
*/
}

static void
interface_draw_pixmap (GdkWindow *window, GdkGC *gc, GdkRectangle *area, GdkPixmap *pixmap, GdkBitmap *mask,
				  guint16 srcx, guint16 srcy,
				  guint16 x, guint16 y, guint16 width, guint16 height)
{
	GdkRectangle	cliprect;
	gint			xd, yd, wd, hd;

	if (area) {
		xd = MAX(area->x, x) - x; if (xd < 0 || xd >= width) return;
		yd = MAX(area->y, y) - y; if (yd < 0 || yd >= height) return;

		wd = width  - (MIN(area->x+area->width,  x+width)  -x -xd); if (wd < 0) return;
		hd = height - (MIN(area->y+area->height, y+height) -y -yd); if (hd < 0) return;
	} else {
		xd = yd = wd = hd = 0;
	}

	gdk_gc_set_clip_mask (gc, mask);
	gdk_gc_set_clip_origin (gc, x-srcx, y-srcy);
	gdk_draw_pixmap (window, gc, pixmap, srcx+xd, srcy+yd, x+xd, y+yd, width-wd, height-hd);
	gdk_gc_set_clip_mask (gc, NULL);
}


static void
interface_draw_image (GdkWindow *window, GtkWidget *widget, GdkRectangle *area, InterfaceImageType image_type,
				 guint16 srcx, guint16 srcy,
				 guint16 x, guint16 y, guint16 width, guint16 height)
{
	GdkPixmap	*pixmap;
	GtkStyle	*style, *style2;
	
	style = widget->style;
	style2 = (widget->parent) ? widget->parent->style : style;
	if (!window)
		window = widget->window;
	
	pixmap = interface_pixmap_get(window, style, style2, image_type);
	if (pixmap)
		interface_draw_pixmap (window, style->bg_gc[widget->state], area,
						  pixmap, interface_mask_get(image_type), srcx, srcy, x, y, width, height);
}


