/** \ingroup popt
 * \file popt/poptconfig.c
 */

/* (C) 1998-2002 Red Hat, Inc. -- Licensing details are in the COPYING
   file accompanying popt source distributions, available from 
   ftp://ftp.rpm.org/pub/rpm/dist. */

#include "system.h"
#include "poptint.h"
#ifdef _WIN32
#include <io.h>
#include <windows.h>
#include <shlobj.h>
#endif

#ifndef O_BINARY
#define O_BINARY 0
#endif

/*@access poptContext @*/

/*@-compmempass@*/	/* FIX: item->option.longName kept, not dependent. */
static void configLine(poptContext con, char * line)
	/*@modifies con @*/
{
    size_t nameLength;
    const char * entryType;
    const char * opt;
    poptItem item = alloca(sizeof(*item));
    int i, j;

    if (con->appName == NULL)
	return;
    nameLength = strlen(con->appName);
    
/*@-boundswrite@*/
    memset(item, 0, sizeof(*item));

    if (strncmp(line, con->appName, nameLength)) return;

    line += nameLength;
    if (*line == '\0' || !isspace(*line)) return;

    while (*line != '\0' && isspace(*line)) line++;
    entryType = line;
    while (*line == '\0' || !isspace(*line)) line++;
    *line++ = '\0';

    while (*line != '\0' && isspace(*line)) line++;
    if (*line == '\0') return;
    opt = line;
    while (*line == '\0' || !isspace(*line)) line++;
    *line++ = '\0';

    while (*line != '\0' && isspace(*line)) line++;
    if (*line == '\0') return;

    /*@-temptrans@*/ /* FIX: line alias is saved */
    if (opt[0] == '-' && opt[1] == '-')
	item->option.longName = opt + 2;
    else if (opt[0] == '-' && opt[2] == '\0')
	item->option.shortName = opt[1];
    /*@=temptrans@*/

    if (poptParseArgvString(line, &item->argc, &item->argv)) return;

    /*@-modobserver@*/
    item->option.argInfo = POPT_ARGFLAG_DOC_HIDDEN;
    for (i = 0, j = 0; i < item->argc; i++, j++) {
	const char * f;
	if (!strncmp(item->argv[i], "--POPTdesc=", sizeof("--POPTdesc=")-1)) {
	    f = item->argv[i] + sizeof("--POPTdesc=");
	    if (f[0] == '$' && f[1] == '"') f++;
	    item->option.descrip = f;
	    item->option.argInfo &= ~POPT_ARGFLAG_DOC_HIDDEN;
	    j--;
	} else
	if (!strncmp(item->argv[i], "--POPTargs=", sizeof("--POPTargs=")-1)) {
	    f = item->argv[i] + sizeof("--POPTargs=");
	    if (f[0] == '$' && f[1] == '"') f++;
	    item->option.argDescrip = f;
	    item->option.argInfo &= ~POPT_ARGFLAG_DOC_HIDDEN;
	    item->option.argInfo |= POPT_ARG_STRING;
	    j--;
	} else
	if (j != i)
	    item->argv[j] = item->argv[i];
    }
    if (j != i) {
	item->argv[j] = NULL;
	item->argc = j;
    }
    /*@=modobserver@*/
/*@=boundswrite@*/
	
    /*@-nullstate@*/ /* FIX: item->argv[] may be NULL */
    if (!strcmp(entryType, "alias"))
	(void) poptAddItem(con, item, 0);
    else if (!strcmp(entryType, "exec"))
	(void) poptAddItem(con, item, 1);
    /*@=nullstate@*/
}
/*@=compmempass@*/

static int poptReadConfigFD(poptContext con, int fd)
{
    const char * file, * chptr, * end;
    char * buf;
/*@dependent@*/ char * dst;
    int rc;
    off_t fileLength;

    fileLength = lseek(fd, 0, SEEK_END);
    if (fileLength == -1 || lseek(fd, 0, 0) == -1) {
	rc = errno;
	(void) close(fd);
	errno = rc;
	return POPT_ERROR_ERRNO;
    }

    file = alloca(fileLength + 1);
    if (read(fd, (char *)file, fileLength) != fileLength) {
	rc = errno;
	(void) close(fd);
	errno = rc;
	return POPT_ERROR_ERRNO;
    }
    if (close(fd) == -1)
	return POPT_ERROR_ERRNO;

/*@-boundswrite@*/
    dst = buf = alloca(fileLength + 1);

    chptr = file;
    end = (file + fileLength);
    /*@-infloops@*/	/* LCL: can't detect chptr++ */
    while (chptr < end) {
	switch (*chptr) {
	  case '\r':
	    chptr++;		/* Skip CR */
	    /*@switchbreak@*/ break;
	  case '\n':
	    *dst = '\0';
	    dst = buf;
	    while (*dst && isspace(*dst)) dst++;
	    if (*dst && *dst != '#')
		configLine(con, dst);
	    chptr++;
	    /*@switchbreak@*/ break;
	  case '\\':
	    *dst++ = *chptr++;
	    if (chptr < end) {
		if (*chptr == '\n') 
		    dst--, chptr++;	
		    /* \ at the end of a line does not insert a \n */
		else
		    *dst++ = *chptr++;
	    }
	    /*@switchbreak@*/ break;
	  default:
	    *dst++ = *chptr++;
	    /*@switchbreak@*/ break;
	}
    }
    /*@=infloops@*/
/*@=boundswrite@*/

    return 0;
}

int poptReadConfigFile(poptContext con, const char * fn)
{
    int fd;

    fd = open(fn, O_RDONLY|O_BINARY);
    if (fd < 0)
	return (errno == ENOENT ? 0 : POPT_ERROR_ERRNO);

    return poptReadConfigFD(con, fd);
}

#ifdef _WIN32

static int poptReadConfigFileW(poptContext con, const wchar_t * fn)
{
    int fd;

    fd = _wopen(fn, O_RDONLY|O_BINARY);
    if (fd < 0)
	return (errno == ENOENT ? 0 : POPT_ERROR_ERRNO);

    return poptReadConfigFD(con, fd);
}

static char *global_config_file = NULL;
static wchar_t *global_config_fileW = NULL;

static char *user_config_file = NULL;
static wchar_t *user_config_fileW = NULL;

#ifdef PIC

/* DllMain function needed to fetch the DLL name and deduce the
 * installation directory from that.
 */
BOOL WINAPI
DllMain (HINSTANCE hinstDLL,
	 DWORD     fdwReason,
	 LPVOID    lpvReserved)
{
  wchar_t wcbfr[1000];
  char cpbfr[1000];
  
  switch (fdwReason) {
    case DLL_PROCESS_ATTACH:
      if (GetVersion() < 0x80000000) {
	  /* NT-based Windows has wide char API */
	  if (GetModuleFileNameW((HMODULE) hinstDLL,
				 wcbfr, sizeof(wcbfr)/sizeof(wcbfr[0]))) {
	      wchar_t *p = wcsrchr(wcbfr, L'\\');
	  
	      if (p)
		  *p = '\0';
	  
	      p = wcsrchr(wcbfr, L'\\');
	      if (p && (wcsicmp(p + 1, L"bin") == 0))
		  *p = '\0';
	  
	      global_config_fileW = malloc((wcslen(wcbfr) +
					    wcslen(L"\\etc\\popt") +
					    1) * sizeof(wchar_t));
	      wcscpy(global_config_fileW, wcbfr);
	      wcscat(global_config_fileW, L"\\etc\\popt");
	  }
      } else {
	  /* Win9x, yecch */
	  if (GetModuleFileNameA((HMODULE) hinstDLL,
				 cpbfr, sizeof(cpbfr))) {
	      char *p = _mbsrchr(cpbfr, '\\');
	  
	      if (p)
		  *p = '\0';
	  
	      p = _mbsrchr(cpbfr, '\\');
	      if (p && (stricmp(p + 1, "bin") == 0))
		  *p = '\0';
	  
	      global_config_file = malloc(strlen(cpbfr) +
					  strlen("\\etc\\popt") +
					  1);
	      strcpy(global_config_file, cpbfr);
	      strcat(global_config_file, "\\etc\\popt");
	  }
	  
      }
      break;
  }

  return TRUE;
}

#else

#warning Should build as DLL

#endif

static void get_user_config_file(void)
{
    char cpbfr[MAX_PATH+1];
    wchar_t wcbfr[MAX_PATH+1];
    HRESULT hr;
    LPITEMIDLIST pidl = NULL;
    BOOL b;

    hr = SHGetSpecialFolderLocation (NULL, CSIDL_APPDATA, &pidl);
    if (hr == S_OK)
	{
	    if (GetVersion() < 0x80000000)
		{
		    if (SHGetPathFromIDListW(pidl, wcbfr)) {
			user_config_fileW = malloc((wcslen (wcbfr) +
						    wcslen (L"\\popt.conf") +
						    1) * sizeof(wchar_t));
			if (user_config_fileW) {
			    wcscpy(user_config_fileW, wcbfr);
			    wcscat(user_config_fileW, L"\\popt.conf");
			}
		    }
		}
	    else
		{
		    if (SHGetPathFromIDListA(pidl, cpbfr)) {
			user_config_file = malloc(strlen (cpbfr) +
						  strlen ("\\popt.conf") +
						  1);
			if (user_config_file) {
			    strcpy(user_config_file, cpbfr);
			    strcat(user_config_file, "\\popt.conf");
			}
		    }
		}
	    CoTaskMemFree (pidl);
	}
}

#endif

int poptReadDefaultConfig(poptContext con, /*@unused@*/ int useEnv)
{
    char * fn, * home;
    int rc;

    if (con->appName == NULL) return 0;

#ifndef _WIN32
    rc = poptReadConfigFile(con, "/etc/popt");
#else
    if (global_config_fileW)
	rc = poptReadConfigFileW(con, global_config_fileW);
    else if (global_config_file)
	rc = poptReadConfigFile(con, global_config_file);
    else
	rc = 0;
#endif
    if (rc) return rc;

#ifndef _WIN32
    if ((home = getenv("HOME"))) {
	fn = alloca(strlen(home) + 20);
	strcpy(fn, home);
	strcat(fn, "/.popt");
	rc = poptReadConfigFile(con, fn);
	if (rc) return rc;
    }
#else
    get_user_config_file();
    if (user_config_fileW) {
	rc = poptReadConfigFileW(con, user_config_fileW);
	if (rc) return rc;
    } else if (user_config_file) {
	rc = poptReadConfigFile(con, user_config_file);
	if (rc) return rc;
    }
#endif

    return 0;
}
