/* gok-page-keysizespace.c
*
* Copyright 2002 Sun Microsystems, Inc.,
* Copyright 2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-page-keysizespace.h"
#include "gok-data.h"
#include "support.h"
#include "gok-log.h"

/* settings for the key size/spacing */
static gint m_keywidth, m_keyheight, m_spacing;
static gboolean m_bUseGtkPlusTheme;

/* backup of the settings (in case the "cancel" button is clicked) */
static gint m_keywidthBackup, m_keyheightBackup, m_spacingBackup; 
static gboolean m_bUseGtkPlusThemeBackup;

/* pointer to the settings dialog window */
static GtkWidget* m_pWindowSettings;

/**
* gok-settings-page-keysizespace-initialize
* @pWindowSettings: Pointer to the settings dialog window.
*
* Initializes this page of the gok settings dialog. This must be called
* prior to any calls on this page.
*
* returns: TRUE if the page was initialized OK, FALSE if not.
**/
gboolean gok_settings_page_keysizespace_initialize (GtkWidget* pWindowSettings)
{
	GtkWidget* pButtonExample1;
	GtkWidget* pButtonExample2;
	GtkWidget* pButtonExample3;
	GtkWidget* pButtonExample4;
	GtkWidget* fixedKeySpacing;
	GtkTooltips *tooltips;
	
	g_assert (pWindowSettings != NULL);
	
	m_pWindowSettings = pWindowSettings;

	/* initialize this data */
	m_keywidth = gok_data_get_key_width();
	m_keyheight = gok_data_get_key_height();
	m_spacing = gok_data_get_key_spacing();
	m_bUseGtkPlusTheme = gok_data_get_use_gtkplus_theme();
	
	/* backup the initial settings */
	gok_settings_page_keysizespace_backup();

	tooltips = gtk_tooltips_new ();

	/* create the example buttons for key size/spacing */
	/* I'm doing this here because Glade generates a setusize that I don't want */
	
	/* first, get the fixed area that holds the keys */
	fixedKeySpacing = lookup_widget (m_pWindowSettings, "fixedKeySpacing");
	g_assert (fixedKeySpacing != NULL);
	
	pButtonExample1 = gtk_button_new_with_label (_("Key 1"));
	gtk_widget_ref (pButtonExample1);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonExample1", pButtonExample1,
                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonExample1);
	gtk_fixed_put (GTK_FIXED (fixedKeySpacing), pButtonExample1, 80, 232);
	GTK_WIDGET_UNSET_FLAGS (pButtonExample1, GTK_CAN_FOCUS);
	gtk_tooltips_set_tip (tooltips, pButtonExample1, _("example key 1"), NULL);
	gtk_widget_set_name (pButtonExample1, "StyleButtonNormal"); 
	gok_settings_page_keysizespace_set_label_name (pButtonExample1);

	pButtonExample2 = gtk_button_new_with_label (_("Key 2"));
	gtk_widget_ref (pButtonExample2);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonExample2", pButtonExample2,
                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonExample2);
	gtk_fixed_put (GTK_FIXED (fixedKeySpacing), pButtonExample2, 144, 232);
	GTK_WIDGET_UNSET_FLAGS (pButtonExample2, GTK_CAN_FOCUS);
	gtk_tooltips_set_tip (tooltips, pButtonExample2, _("example key 2"), NULL);
	gtk_widget_set_name (pButtonExample2, "StyleButtonNormal"); 
	gok_settings_page_keysizespace_set_label_name (pButtonExample2);

	pButtonExample3 = gtk_button_new_with_label (_("Key 3"));
	gtk_widget_ref (pButtonExample3);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonExample3", pButtonExample3,
                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonExample3);
	gtk_fixed_put (GTK_FIXED (fixedKeySpacing), pButtonExample3, 80, 264);
	GTK_WIDGET_UNSET_FLAGS (pButtonExample3, GTK_CAN_FOCUS);
	gtk_tooltips_set_tip (tooltips, pButtonExample3, _("example key 3"), NULL);
	gtk_widget_set_name (pButtonExample3, "StyleButtonNormal"); 
	gok_settings_page_keysizespace_set_label_name (pButtonExample3);

	pButtonExample4 = gtk_button_new_with_label (_("Key 4"));
	gtk_widget_ref (pButtonExample4);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonExample4", pButtonExample4,
                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonExample4);
	gtk_fixed_put (GTK_FIXED (fixedKeySpacing), pButtonExample4, 144, 264);
	GTK_WIDGET_UNSET_FLAGS (pButtonExample4, GTK_CAN_FOCUS);
	gtk_tooltips_set_tip (tooltips, pButtonExample4, _("example key 4"), NULL);
	gtk_widget_set_name (pButtonExample4, "StyleButtonNormal"); 
	gok_settings_page_keysizespace_set_label_name (pButtonExample4);

	/* get the settings from the gok_data and update the controls */
	gok_settings_page_keysizespace_refresh();
	
	return TRUE;
}

/**
* gok_settings_page_keysizespace_set_label_name
* @pButton: Pointer to the button that gets the new label name.
*
* Helper function that sets the 'name' of the button label.
**/
void gok_settings_page_keysizespace_set_label_name (GtkWidget* pButton)
{
	GList* pListContainerChildren;

	pListContainerChildren = gtk_container_get_children (GTK_CONTAINER(pButton));
	while (pListContainerChildren != NULL)
	{
		if (GTK_IS_LABEL (pListContainerChildren->data) == TRUE)
		{
			gtk_widget_set_name (GTK_WIDGET(pListContainerChildren->data), "StyleTextNormal"); 
			break;
		}
		pListContainerChildren = pListContainerChildren->next;
	}
	
	g_list_free (pListContainerChildren);
}

/**
* gok_settings_page_keysizespace_refresh
* 
* Refreshes the key size and key spacing controls on the key size/spacing page 
* from the gok_data.
*
* returns: void.
**/
void gok_settings_page_keysizespace_refresh ()
{
	GtkWidget* pSpinSpacing;
	GtkWidget* pSpinWidth;
	GtkWidget* pSpinHeight;
	GtkWidget* pCheckUseTheme;

	g_assert (m_pWindowSettings != NULL);
	
	/* initialize the spin control for the key width */
	pSpinWidth = lookup_widget (m_pWindowSettings, "spinKeyWidth");
	g_assert (pSpinWidth != NULL);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(pSpinWidth), gok_data_get_key_width());

	/* initialize the spin control for the key height */
	pSpinHeight = lookup_widget (m_pWindowSettings, "spinKeyHeight");
	g_assert (pSpinHeight != NULL);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(pSpinHeight), gok_data_get_key_height());

	/* initialize the spin control for the key spacing */
	pSpinSpacing = lookup_widget (m_pWindowSettings, "spinKeySpacing");
	g_assert (pSpinSpacing != NULL);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(pSpinSpacing), gok_data_get_key_spacing());

	/* position the example keys to show key size and key spacing */
	gok_settings_page_keysizespace_display_keysizespacing (gok_data_get_key_width(),
															gok_data_get_key_height(),
															gok_data_get_key_spacing());

	pCheckUseTheme = lookup_widget (m_pWindowSettings, "checkUseTheme");
	g_assert (pCheckUseTheme != NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pCheckUseTheme), gok_data_get_use_gtkplus_theme());
}

/**
* gok_settings_page_keysizespace_apply
* 
* Stores the current settings in the gok_data.
*
* returns: TRUE if any settings have changed, FALSE if no settings have changed.
**/
gboolean gok_settings_page_keysizespace_apply ()
{
	gboolean bDataChanged;
	GtkWidget* pCheckUseTheme;
	
	g_assert (m_pWindowSettings != NULL);
	
	bDataChanged = FALSE;

	/* update the gok data with any new settings */
	if (gok_data_get_key_spacing() != m_spacing)
	{
		bDataChanged = TRUE;
		gok_data_set_key_spacing (m_spacing);
	}

	if (gok_data_get_key_width() != m_keywidth)
	{
		bDataChanged = TRUE;
		gok_data_set_key_width (m_keywidth);
	}

	if (gok_data_get_key_height() != m_keyheight)
	{
		bDataChanged = TRUE;
		gok_data_set_key_height (m_keyheight);
	}

	pCheckUseTheme = lookup_widget (m_pWindowSettings, "checkUseTheme");
	g_assert (pCheckUseTheme != NULL);
	m_bUseGtkPlusTheme = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(pCheckUseTheme));
	if (gok_data_get_use_gtkplus_theme() != m_bUseGtkPlusTheme)
	{
		bDataChanged = TRUE;
		gok_data_set_use_gtkplus_theme (m_bUseGtkPlusTheme);
	}
	
	return bDataChanged;
}

/**
* gok_settings_page_keysizespace_revert
* 
* Revert to the backup settings for this page and store them in the gok_data.
*
* returns: TRUE if any of the settings have changed, FALSE 
* if they are all still the same.
**/
gboolean gok_settings_page_keysizespace_revert ()
{
	gboolean bDataChanged;

	bDataChanged = FALSE;
	
	/* get the original settings */
	if (m_spacing != m_spacingBackup)
	{
		bDataChanged = TRUE;
		m_spacing = m_spacingBackup;
		gok_data_set_key_spacing (m_spacing);
	}

	if (m_keywidth != m_keywidthBackup)
	{
		bDataChanged = TRUE;
		m_keywidth = m_keywidthBackup;
		gok_data_set_key_width (m_keywidth);
	}

	if (m_keyheight != m_keyheightBackup)
	{
		bDataChanged = TRUE;
		m_keyheight = m_keyheightBackup;
		gok_data_set_key_height (m_keyheight);
	}

	if (m_bUseGtkPlusTheme != m_bUseGtkPlusThemeBackup)
	{
		bDataChanged = TRUE;
		m_bUseGtkPlusTheme = m_bUseGtkPlusThemeBackup;
		gok_data_set_use_gtkplus_theme (m_bUseGtkPlusTheme);
	}

	if (bDataChanged == TRUE)
	{
		gok_settings_page_keysizespace_refresh();
	}
	
	return bDataChanged;
}

/**
* gok_settings_page_keysizespace_display_keysizespacing
* @KeyWidth: Width of the example keys.
* @KeyHeight: Height of the example keys.
* @Space: Spacing between the example keys.
*
* Displays the example key size and key spacing.
**/
void gok_settings_page_keysizespace_display_keysizespacing (gint KeyWidth, gint KeyHeight, gint Space)
{

	GtkWidget* pButtonKey1;
	GtkWidget* pButtonKey2;
	GtkWidget* pButtonKey3;
	GtkWidget* pButtonKey4;
	GtkWidget* pFixedKeyspacing;
	GtkRequisition requisitionSizeFixed;
	gint left;
	gint top;
	gint width;

	g_assert (m_pWindowSettings != NULL);
	
	if (KeyWidth < 0)
	{
		KeyWidth = m_keywidth;
	}
	if (KeyHeight < 0)
	{
		KeyHeight = m_keyheight;
	}
	if (Space < 0)
	{
		Space = m_spacing;
	}

	/* keep the values within range */
	if (KeyWidth < MIN_KEY_WIDTH)
	{
		KeyWidth = MIN_KEY_WIDTH;
	}
	else if (KeyWidth > MAX_KEY_WIDTH)
	{
		KeyWidth = MAX_KEY_WIDTH;
	}
	if (KeyHeight < MIN_KEY_HEIGHT)
	{
		KeyHeight = MIN_KEY_HEIGHT;
	}
	else if (KeyHeight > MAX_KEY_HEIGHT)
	{
		KeyHeight = MAX_KEY_HEIGHT;
	}
	if ((Space <0) ||
		(Space > MAX_KEY_SPACING))
	{
		Space = MAX_KEY_SPACING;
	}

	/* get pointers to the example keys */
	pButtonKey1 = lookup_widget (m_pWindowSettings, "pButtonExample1");
	g_assert (pButtonKey1 != NULL);
	pButtonKey2 = lookup_widget (m_pWindowSettings, "pButtonExample2");
	g_assert (pButtonKey2 != NULL);
	pButtonKey3 = lookup_widget (m_pWindowSettings, "pButtonExample3");
	g_assert (pButtonKey3 != NULL);
	pButtonKey4 = lookup_widget (m_pWindowSettings, "pButtonExample4");
	g_assert (pButtonKey4 != NULL);

	/* get the container that holds the 'keyspace' buttons */
	pFixedKeyspacing = lookup_widget (m_pWindowSettings, "fixedKeySpacing");
	g_assert (pFixedKeyspacing != NULL);

	/* get the size of the fixed container */
	gtk_widget_size_request (pFixedKeyspacing, &requisitionSizeFixed);

	/* calculate the positions of the buttons */
	left = (requisitionSizeFixed.width - ((KeyWidth * 2) + Space)) / 2;
	top = (requisitionSizeFixed.height - ((KeyHeight * 2) + Space)) / 2;

	/* calculate the left side based upon the window size */
	if (GDK_IS_DRAWABLE(m_pWindowSettings->window) == TRUE)
	{
		gdk_window_get_size (m_pWindowSettings->window, &width, NULL);
		left = ((width - 32) - ((KeyWidth * 2) + Space)) / 2;
	}

	/* resize the buttons */
	gtk_widget_set_size_request (pButtonKey1, KeyWidth, KeyHeight);
	gtk_widget_set_size_request (pButtonKey2, KeyWidth, KeyHeight);
	gtk_widget_set_size_request (pButtonKey3, KeyWidth, KeyHeight);
	gtk_widget_set_size_request (pButtonKey4, KeyWidth, KeyHeight);

	/* position the buttons */
	gtk_fixed_move (GTK_FIXED(pFixedKeyspacing), pButtonKey1, left, top);
	gtk_fixed_move (GTK_FIXED(pFixedKeyspacing), pButtonKey2, left + Space + KeyWidth, top);
	gtk_fixed_move (GTK_FIXED(pFixedKeyspacing), pButtonKey3, left, top + Space + KeyHeight);
	gtk_fixed_move (GTK_FIXED(pFixedKeyspacing), pButtonKey4, left + Space + KeyWidth, top + Space + KeyHeight);

	m_keywidth = KeyWidth;
	m_keyheight = KeyHeight;
	m_spacing = Space;
}

/**
* gok_settings_page_keysizespace_backup
* 
* Copies all the member settings to backup.
**/
void gok_settings_page_keysizespace_backup ()
{
	m_spacingBackup = m_spacing;
	m_keywidthBackup = m_keywidth;
	m_keyheightBackup = m_keyheight;
	m_bUseGtkPlusThemeBackup = m_bUseGtkPlusTheme;
}

