/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* GNOME Personal Web Server Preferences Widget
 *
 * Authors:
 *	lvaro Pea <apg@esware.com>
 *
 * Copyright (C) 2003 GNOME Foundation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pws-preferences-widget.h"

#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/libgnomeui.h>
#include "gnetwork-http-server.h"
#include "pws-preferences-support.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _PwsPreferencesWidgetPrivate {
	GtkWidget                *disclosure;
	GtkWidget                *document_root;
	GtkWidget                *port;
	GtkWidget                *time_out;
	GtkWidget                *keepalive;
	GNetworkHttpServerConfig *config;
};

static void pws_preferences_widget_init       (PwsPreferencesWidget      *pref);
static void pws_preferences_widget_class_init (PwsPreferencesWidgetClass *klass);
static void pws_preferences_widget_finalize   (GObject                   *object);
static void port_changed_cb                   (GtkSpinButton             *p_sp);
static void timeout_changed_cb                (GtkSpinButton             *t_sp);
static void document_root_changed_cb          (GtkEntry                  *entry);
static void keepalive_toggled_cb              (GtkToggleButton           *tb);

static GObjectClass *parent_class = NULL;
extern GConfClient  *gconf_client;

/*
 * PwsPreferencesWidget class implementation
 */

GType
pws_preferences_widget_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (PwsPreferencesWidgetClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) pws_preferences_widget_class_init,
			NULL,
			NULL,
			sizeof (PwsPreferencesWidget),
			0,
			(GInstanceInitFunc) pws_preferences_widget_init
		};
		type = g_type_register_static (PARENT_TYPE, "PwsPreferencesWidget", &info, 0);
	}
	return type;
}


static void
pws_preferences_widget_init (PwsPreferencesWidget *pref)
{
	GtkWidget *hbox, *label, *hbox_additional, *vbox, *table;
	GtkWidget *disclosure;
	
	g_return_if_fail (PWS_IS_PREFERENCES_WIDGET (pref));
	
	pref->priv = g_new0 (PwsPreferencesWidgetPrivate, 1);

	gtk_box_set_spacing (GTK_BOX (pref), 6);
	
	hbox = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (pref), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);

	label = gtk_label_new (_("Web folder:"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	
	pref->priv->document_root = gnome_file_entry_new (NULL, _("Select the folder to be published"));
	gnome_file_entry_set_directory_entry (GNOME_FILE_ENTRY (pref->priv->document_root), TRUE);
	g_signal_connect (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (pref->priv->document_root)),
			  "changed",
			  G_CALLBACK (document_root_changed_cb), NULL);
	gtk_box_pack_start (GTK_BOX (hbox), pref->priv->document_root, TRUE, TRUE, 0);
	gtk_widget_show (pref->priv->document_root);

	pref->priv->disclosure = cddb_disclosure_new (_("Additional options"),
						      _("Additional options"));
	gtk_box_pack_start (GTK_BOX (pref), pref->priv->disclosure, FALSE, FALSE, 0);
	gtk_widget_show (pref->priv->disclosure);
	
	hbox_additional = gtk_hbox_new (FALSE, 0);
	cddb_disclosure_set_container (CDDB_DISCLOSURE (pref->priv->disclosure), hbox_additional);
	gtk_box_pack_start (GTK_BOX (pref), hbox_additional, FALSE, FALSE, 0);
	gtk_widget_show (hbox_additional);

	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox_additional), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	
	vbox = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (hbox_additional), vbox, TRUE, TRUE, 0);

	table = gtk_table_new (2, 2, FALSE);
	gtk_table_set_col_spacings (GTK_TABLE (table), 6);
	gtk_table_set_row_spacings (GTK_TABLE (table), 6);
	gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
	gtk_widget_show (table);

	label = gtk_label_new (_("Port number:"));
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label,
			  0, 1,
			  0, 1,
			  GTK_FILL, GTK_FILL,
			  0, 0);
	gtk_widget_show (label);
	
	pref->priv->port = gtk_spin_button_new_with_range  (1025, 9999, 1);
	g_signal_connect (pref->priv->port,
			  "value-changed",
			  G_CALLBACK (port_changed_cb), NULL);
	gtk_table_attach_defaults (GTK_TABLE (table), pref->priv->port,
				   1, 2,
				   0, 1);
	gtk_widget_show (pref->priv->port);

	label = gtk_label_new (_("Time out:"));
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label,
			  0, 1,
			  1, 2,
			  GTK_FILL, GTK_FILL,
			  0, 0);
	gtk_widget_show (label);
	
	pref->priv->time_out = gtk_spin_button_new_with_range (100, 1000, 1);
	g_signal_connect (pref->priv->port,
			  "value-changed",
			  G_CALLBACK (timeout_changed_cb), NULL);
	gtk_table_attach_defaults (GTK_TABLE (table), pref->priv->time_out,
				   1, 2,
				   1, 2);
	gtk_widget_show (pref->priv->time_out);
	
	pref->priv->keepalive = gtk_check_button_new_with_label (_("Keep alive"));
	g_signal_connect (pref->priv->keepalive,
			  "toggled",
			  G_CALLBACK (keepalive_toggled_cb), NULL);
	gtk_box_pack_start (GTK_BOX (vbox), pref->priv->keepalive, FALSE, FALSE, 0);
	gtk_widget_show (pref->priv->keepalive);
}

static void
pws_preferences_widget_class_init (PwsPreferencesWidgetClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	
	parent_class = g_type_class_peek_parent (klass);
	
	object_class->finalize = pws_preferences_widget_finalize;
}

static void
pws_preferences_widget_finalize (GObject *object)
{
	PwsPreferencesWidget *pref;
	
	g_return_if_fail (PWS_IS_PREFERENCES_WIDGET (object));
	pref = PWS_PREFERENCES_WIDGET (object);
	
	if (pref->priv) {
		g_free (pref->priv);
		pref->priv = NULL;
	}
	
	parent_class->finalize (object);
}

static void
port_changed_cb (GtkSpinButton *p_sp)
{
	gint port_number;

	port_number = gtk_spin_button_get_value_as_int (p_sp);
	gconf_client_set_int (gconf_client, "/apps/gnome-pws/prefs/port", port_number, NULL);
}

static void
timeout_changed_cb (GtkSpinButton *t_sp)
{
	gint timeout;

	timeout = gtk_spin_button_get_value_as_int (t_sp);
	gconf_client_set_int (gconf_client, "/apps/gnome-pws/prefs/timeout", timeout, NULL);
}

static void
document_root_changed_cb (GtkEntry *entry)
{
	const gchar *str;
	
	str = gtk_entry_get_text (entry);
	gconf_client_set_string (gconf_client, "/apps/gnome-pws/prefs/documentroot", g_strdup (str), NULL);
}


static void
keepalive_toggled_cb (GtkToggleButton *tb)
{
	gboolean a;
	
	a = gtk_toggle_button_get_active (tb);
	gconf_client_set_bool (gconf_client, "/apps/gnome-pws/prefs/keepalive", a, NULL);
}

/**
 * pws_preferences_widget_new
 *
 */
GtkWidget*
pws_preferences_widget_new (GNetworkHttpServerConfig *config)
{
	PwsPreferencesWidget *pref;
	
	pref = g_object_new (PWS_TYPE_PREFERENCES_WIDGET, NULL);
	if (config != NULL)
		pws_preferences_widget_set_server_config (pref, config);
	
	return GTK_WIDGET (pref);
}

void
pws_preferences_widget_set_show_additional (PwsPreferencesWidget *pref, gboolean show)
{
	g_return_if_fail (PWS_IS_PREFERENCES_WIDGET (pref));

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->priv->disclosure), show);
}

gboolean
pws_preferences_widget_set_server_config (PwsPreferencesWidget     *pref,
					  GNetworkHttpServerConfig *config)
{
	g_return_val_if_fail (PWS_IS_PREFERENCES_WIDGET (pref), FALSE);
	g_return_val_if_fail (config != NULL, FALSE);

	if (!pref->priv->config)
		pref->priv->config = g_new0 (GNetworkHttpServerConfig, 1);

	pref->priv->config->document_root = g_strdup (config->document_root);
	gnome_file_entry_set_filename (GNOME_FILE_ENTRY (pref->priv->document_root),
				       g_strdup (pref->priv->config->document_root));

	pref->priv->config->port = config->port;
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->priv->port),
				   pref->priv->config->port);
	
	pref->priv->config->timeout = config->timeout;
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (pref->priv->time_out),
				   pref->priv->config->timeout);

	pref->priv->config->keepalive = config->keepalive;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->priv->keepalive),
				      pref->priv->config->keepalive);
		

	return TRUE;
}

GNetworkHttpServerConfig*
pws_preferences_widget_get_server_config (PwsPreferencesWidget *pref)
{
	GNetworkHttpServerConfig *config;
	
	g_return_val_if_fail (PWS_IS_PREFERENCES_WIDGET (pref), NULL);

	config = g_new0 (GNetworkHttpServerConfig, 1);

	config->document_root = g_strdup (gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (pref->priv->document_root), TRUE));
	config->port = gtk_spin_button_get_value (GTK_SPIN_BUTTON (pref->priv->port));
	config->timeout = gtk_spin_button_get_value (GTK_SPIN_BUTTON (pref->priv->time_out));
	config->keepalive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->priv->keepalive));

	return config;
}
