/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/bytestream/bytestream.h>

#ifdef HAVE_MAD
#include <id3tag.h>
#endif

#include <string.h>
#include "gstid3types.h"

/* elementfactory information */
static GstElementDetails id3types_details = GST_ELEMENT_DETAILS (
  "ID3v1/v2 tag parser",
  "Codec/Parser",
  "parse metadata tags often found in mp3s",
  "Erik Walthinsen <omega@cse.ogi.edu>, "
  "Ronald Bultje <rbultje@ronald.bitfreak.net>"
);

GST_PAD_TEMPLATE_FACTORY (sink_templ,
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "id3types_sink",
    "application/x-id3",
      "id3version", GST_PROPS_INT_RANGE (1, 2)
  )
);

GST_PAD_TEMPLATE_FACTORY (src_templ,
  "src",
  GST_PAD_SRC,
  GST_PAD_SOMETIMES,
  NULL
);

/* signals */
enum {
  /* FILL ME */
  LAST_SIGNAL,
};

/* args */
enum {
  ARG_0,
  ARG_METADATA,
  /* FILL ME */
};


static void	gst_id3types_base_init		(gpointer g_class);
static void	gst_id3types_class_init		(GstID3TypesClass *klass);
static void	gst_id3types_init		(GstID3Types  *mp3parse);

static const GstEventMask *
		gst_id3types_event_mask		(GstPad       *pad);
static gboolean	gst_id3types_src_event		(GstPad       *pad,
						 GstEvent     *event);

static const GstQueryType *
		gst_id3types_get_query_types	(GstPad       *pad);
static gboolean gst_id3types_handle_query 	(GstPad       *pad,
						 GstQueryType  type, 
						 GstFormat    *format,
						 gint64       *value);

static void	gst_id3types_chain		(GstPad       *pad,
						 GstData      *_data);
static void	gst_id3types_loop		(GstElement   *element);

static void	gst_id3types_get_property	(GObject      *object,
						 guint         prop_id,
						 GValue       *value,
						 GParamSpec   *pspec);
static GstElementStateReturn
		gst_id3types_change_state	(GstElement   *element);

static GstElementClass *parent_class = NULL;
/*static guint gst_id3types_signals[LAST_SIGNAL] = { 0 };*/

GType
gst_id3types_get_type (void)
{
  static GType id3types_type = 0;

  if (!id3types_type) {
    static const GTypeInfo id3types_info = {
      sizeof (GstID3TypesClass),
      gst_id3types_base_init,
      NULL,
      (GClassInitFunc) gst_id3types_class_init,
      NULL,
      NULL,
      sizeof (GstID3Types),
      0,
      (GInstanceInitFunc) gst_id3types_init,
    };

    id3types_type = g_type_register_static (GST_TYPE_ELEMENT,
					    "GstID3Types",
					    &id3types_info, 0);
  }

  return id3types_type;
}

static gboolean
gst_id3types_detect (GstByteStream *bs,
		     guint         *version,
		     guint         *size)
{
  gboolean res = FALSE;
  GstBuffer *buf = NULL;

  if (gst_bytestream_peek (bs, &buf, 10) == 10) {
    guint8 *data = GST_BUFFER_DATA (buf);

    /* gracefully ripped from libid3 */
    if (data[0] == 'T' && data[1] == 'A' && data[2] == 'G') {
      /* ID3v1 tags */
      res = TRUE;
      if (size)
        *size = 128;
      if (version)
        *version = 1;
    } else if (data[0] == 'I' && data[1] == 'D' && data[2] == '3') {
      /* ID3v2 tags */
      if (data[3] < 0xff && data[4] < 0xff &&
	  data[6] < 0x80 && data[7] < 0x80 &&
	  data[8] < 0x80 && data[9] < 0x80) {
        guint32 skip = ((data[6] & 0x7f) << 21) |
		       ((data[7] & 0x7f) << 14) |
		       ((data[8] & 0x7f) << 7) |
		       ((data[9] & 0x7f) << 0);

        /* include size of header */
        skip += 10;

        /* footer present? (only available since version 4) */
        if (data[3] >= 4 && (data[5] & 0x10))
          skip += 10;

        res = TRUE;
        if (size)
          *size = skip;
        if (version)
          *version = 2;
      }
    }
  }

  if (buf != NULL) {
    gst_buffer_unref (buf);
  }

  return res;
}

static void
gst_id3types_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);
  
  gst_element_class_add_pad_template (element_class, GST_PAD_TEMPLATE_GET (sink_templ));
  gst_element_class_add_pad_template (element_class, GST_PAD_TEMPLATE_GET (src_templ));
  gst_element_class_set_details (element_class, &id3types_details);
}
static void
gst_id3types_class_init (GstID3TypesClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  g_object_class_install_property (gobject_class, ARG_METADATA,
    g_param_spec_boxed ("metadata", "Metadata", "Metadata",
                        GST_TYPE_CAPS, G_PARAM_READABLE));

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gobject_class->get_property = gst_id3types_get_property;

  gstelement_class->change_state = gst_id3types_change_state;
}

static inline void
gst_id3types_start (GstID3Types *id3)
{
  id3->pass_through = FALSE;
  id3->id3_tag_size = 0;
  id3->metadata = NULL;
}

static inline void
gst_id3types_cont (GstID3Types *id3)
{
  id3->pass_through = TRUE;
}

static inline void
gst_id3types_stop (GstID3Types *id3)
{
  gst_caps_replace (&id3->metadata, NULL);
  id3->pass_through = TRUE;
}

static void
gst_id3types_init (GstID3Types *id3)
{
  GST_FLAG_SET (id3, GST_ELEMENT_EVENT_AWARE);

  id3->sinkpad = gst_pad_new_from_template (GST_PAD_TEMPLATE_GET (sink_templ),
					    "sink");
  gst_element_set_loop_function (GST_ELEMENT (id3), gst_id3types_loop);
  gst_id3types_start (id3);
  gst_element_add_pad (GST_ELEMENT (id3), id3->sinkpad);

  id3->metadata = NULL;
  id3->id3_tag_size = 0;

  GST_FLAG_SET (id3, GST_ELEMENT_EVENT_AWARE);
}

static const GstEventMask *
gst_id3types_event_mask (GstPad *pad)
{
  static const GstEventMask my_masks[] = {
    { GST_EVENT_SEEK, GST_SEEK_METHOD_SET },
    { 0, }
  };

  return my_masks;
}
static void
gst_id3types_sink_event (GstPad *pad, GstEvent *event)
{
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_DISCONTINUOUS: {
      /* comes from the sink pad */
      GstEvent *new = NULL;
      guint64 offset = 0;
      gint n;

      /* get new offset in bytes */
      for (n = 0; n < GST_EVENT_DISCONT_OFFSET_LEN (event); n++) {
        GstFormatValue *value = &GST_EVENT_DISCONT_OFFSET (event, n);
        if (value->format == GST_FORMAT_BYTES) {
          offset = value->value;
          break;
        }
      }

      if (GST_EVENT_DISCONT_NEW_MEDIA (event) ||
	  offset < id3->id3_tag_size) {
        gst_id3types_stop (id3);
        gst_id3types_start (id3);
        new = gst_event_new_discontinuous (GST_EVENT_DISCONT_NEW_MEDIA (event),
					   GST_FORMAT_BYTES, 0, 0);
      } else {
        gst_id3types_cont (id3);
        new = gst_event_new_discontinuous (FALSE, GST_FORMAT_BYTES,
					   offset - id3->id3_tag_size, 0);
      }
      gst_pad_event_default (pad, new);
      break;
    }
    default:
      gst_pad_event_default (pad, event);
      break;
  }
}
static gboolean
gst_id3types_src_event (GstPad *pad, GstEvent *event)
{
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));
  gboolean res = TRUE;

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_SEEK:
      /* comes from the src pad... */
      switch (GST_EVENT_SEEK_FORMAT (event)) {
        case GST_FORMAT_BYTES: {
          gint64 offset = GST_EVENT_SEEK_OFFSET (event);
          GstEvent *new = gst_event_new_seek (GST_SEEK_METHOD_SET | GST_FORMAT_BYTES,
					      offset + id3->id3_tag_size);

          gst_pad_send_event (GST_PAD_PEER (id3->sinkpad), new);
          break;
        }
        default:
          g_warning ("Non-bytes seek event in id3types");
          res = FALSE;
          break;
      }
      break;
    default:
      res = FALSE;
      break;
  }

  gst_event_unref (event);

  return res;
}

static const GstQueryType *
gst_id3types_get_query_types (GstPad *pad)
{
  static const GstQueryType types[] = {
    GST_QUERY_TOTAL,
    GST_QUERY_POSITION,
    0
  };

  return types;
}

static gboolean
gst_id3types_handle_query (GstPad       *pad,
			   GstQueryType  type, 
			   GstFormat    *format,
			   gint64       *value)
{
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));

  g_return_val_if_fail (type == GST_QUERY_TOTAL ||
			type == GST_QUERY_POSITION, FALSE);

  if (*format == GST_FORMAT_BYTES) {
    gst_pad_query (id3->sinkpad, type, format, value);

    if (*value >= id3->id3_tag_size)
      *value -= id3->id3_tag_size;
    else
      *value = 0;

    return TRUE;
  }

  return FALSE;
}

static void
gst_id3types_chain (GstPad    *pad,
		    GstData *_data)
{
  GstBuffer *buf = GST_BUFFER (_data);
  GstID3Types *id3 = GST_ID3TYPES (gst_pad_get_parent (pad));

  if (GST_IS_EVENT (buf)) {
    gst_id3types_sink_event (id3->sinkpad, GST_EVENT (buf));
  } else {
    /* passthrough mode */
    gst_pad_push (id3->srcpad, GST_DATA (buf));
  }
}

#ifdef HAVE_MAD
/* gracefuly ripped from madplay */
static GstCaps *
id3_to_caps (struct id3_tag const *tag)
{
  unsigned int i;
  struct id3_frame const *frame;
  id3_ucs4_t const *ucs4;
  id3_utf8_t *utf8;
  GstProps *props;
  GstPropsEntry *entry;
  GstCaps *caps;
  GList *values;

  struct {
    char const *id;
    char const *name;
  } const info[] = {
    { ID3_FRAME_TITLE,   "Title"        },
    { "TIT3",            "Subtitle"     },
    { "TCOP",            "Copyright"    },
    { "TPRO",            "Produced"     },
    { "TCOM",            "Composer"     },
    { ID3_FRAME_ARTIST,  "Artist"       },
    { "TPE2",            "Orchestra"    },
    { "TPE3",            "Conductor"    },
    { "TEXT",            "Lyricist"     },
    { ID3_FRAME_ALBUM,   "Album"        },
    { ID3_FRAME_YEAR,    "Year"         },
    { ID3_FRAME_TRACK,   "Track"        },
    { "TPUB",            "Publisher"    },
    { ID3_FRAME_GENRE,   "Genre"        },
    { "TRSN",            "Station"      },
    { "TENC",            "Encoder"      },
  };

  /* text information */
  props = gst_props_empty_new ();

  for (i = 0; i < sizeof(info) / sizeof(info[0]); ++i) {
    union id3_field const *field;
    unsigned int nstrings, namelen, j;
    char const *name;

    frame = id3_tag_findframe(tag, info[i].id, 0);
    if (frame == 0)
      continue;

    field    = &frame->fields[1];
    nstrings = id3_field_getnstrings(field);

    name = info[i].name;

    if (name) {
      namelen = name ? strlen(name) : 0;

      values = NULL;
      for (j = 0; j < nstrings; ++j) {
        ucs4 = id3_field_getstrings(field, j);
        g_assert(ucs4);

        if (strcmp(info[i].id, ID3_FRAME_GENRE) == 0)
	  ucs4 = id3_genre_name(ucs4);

        utf8 = id3_ucs4_utf8duplicate(ucs4);
        if (utf8 == 0)
	  goto fail;

        entry = gst_props_entry_new (name, GST_PROPS_STRING_TYPE, utf8);
	values = g_list_prepend (values, entry);
        free(utf8);
      }
      if (values) {
        values = g_list_reverse (values);

        if (g_list_length (values) == 1) {
          gst_props_add_entry (props, (GstPropsEntry *) values->data);
        }
        else {
          entry = gst_props_entry_new(name, GST_PROPS_GLIST_TYPE, values);
          gst_props_add_entry (props, (GstPropsEntry *) entry);
        }
        g_list_free (values);
      }
    }
  }

  values = NULL;
  i = 0;
  while ((frame = id3_tag_findframe(tag, ID3_FRAME_COMMENT, i++))) {
    ucs4 = id3_field_getstring(&frame->fields[2]);
    g_assert(ucs4);

    if (*ucs4)
      continue;

    ucs4 = id3_field_getfullstring(&frame->fields[3]);
    g_assert(ucs4);

    utf8 = id3_ucs4_utf8duplicate(ucs4);
    if (utf8 == 0)
      goto fail;

    entry = gst_props_entry_new ("Comment", GST_PROPS_STRING_TYPE, utf8);
    values = g_list_prepend (values, entry);
    free(utf8);
  }
  if (values) {
    values = g_list_reverse (values);

    if (g_list_length (values) == 1) {
      gst_props_add_entry (props, (GstPropsEntry *) values->data);
    }
    else {
      entry = gst_props_entry_new("Comment", GST_PROPS_GLIST_TYPE, values);
      gst_props_add_entry (props, (GstPropsEntry *) entry);
    }
    g_list_free (values);
  }

  gst_props_debug (props);

  caps = gst_caps_new ("mad_metadata",
		       "application/x-gst-metadata",
		       props);
  if (0) {
fail:
    g_warning ("mad: could not parse ID3 tag");

    return NULL;
  }

  return caps;
}
#endif
typedef struct {
  GstByteStream *bs;
  guint best_probability;
  GstCaps *caps;
  GstBuffer *buffer;
} SimpleTypeFind;
guint8 *
simple_find_peek (gpointer data, gint64 offset, guint size)
{
  guint result;
  SimpleTypeFind *find = (SimpleTypeFind *) data;
  
  if (offset < 0)
    return NULL;
  
  if (find->buffer) {
    if (GST_BUFFER_SIZE (find->buffer) >= offset + size) {
      return GST_BUFFER_DATA (find->buffer) + offset;
    } else {
      gst_data_unref (GST_DATA (find->buffer));
    }
  }
  if (offset >= 0) {
    while ((result = gst_bytestream_peek (find->bs, &find->buffer, offset + size)) == 0) {
      GstEvent *event = NULL;
      gst_bytestream_get_status (find->bs, NULL, &event);
      g_assert (event);
      gst_id3types_sink_event (find->bs->pad, event);
    }
    if (result == offset + size) {
      return GST_BUFFER_DATA (find->buffer) + offset;
    }
  }
  return NULL;
}
static void
simple_find_suggest (gpointer data, guint probability, GstCaps *caps)
{
  SimpleTypeFind *find = (SimpleTypeFind *) data;

  if (probability > find->best_probability) {
    gst_caps_replace (&find->caps, caps);
    find->best_probability = probability;
  }
}
static void
gst_id3types_loop (GstElement *element)
{
  GList *type_list, *walk;
  GstID3Types *id3;
  GstByteStream *bs;
  GstTypeFind gst_find;
  SimpleTypeFind find;
  GstBuffer *buf = NULL;
  guint size = 0, version = 0;
#ifdef HAVE_MAD
  struct id3_tag *tag = NULL;
#endif

  g_return_if_fail (GST_IS_ID3TYPES (element));
  id3 = GST_ID3TYPES (element);

  if (id3->pass_through) {
    gst_id3types_chain (id3->sinkpad, gst_pad_pull (id3->sinkpad));
    return;
  }

  bs = gst_bytestream_new (id3->sinkpad);

  /* first loop... */
  if (gst_id3types_detect (bs, &version, &size)) {
    /* OK: so we've now got a ID3 tag of that size, let's read it */
    if (gst_bytestream_peek (bs, &buf, size) != size) {
      gst_element_error (element,
			 "Failed to read ID3 tag");
      goto error;
    }
    gst_bytestream_flush (bs, size);
#ifdef HAVE_MAD
    /* here, we should parse the ID3 header and notify the app */
    if (!(tag = id3_tag_parse (GST_BUFFER_DATA (buf), size))) {
      gst_element_error (element,
			 "Failed to parse ID3v1/2 tags");
      goto error;
    } else {
      gst_caps_replace_sink (&id3->metadata, id3_to_caps (tag));
      id3_tag_delete (tag);
      g_object_notify (G_OBJECT (id3), "metadata");
    }
#endif
  }

  /* done */
  id3->id3_tag_size = size;

  /* this will help us detecting the media stream type after
   * this id3 thingy... Please note that this is a cruel hack
   * for as long as spider doesn't support multi-type-finding.
   */
  walk = type_list = gst_type_find_factory_get_list ();
  
  find.bs = bs;
  find.best_probability = 0;
  find.caps = NULL;
  find.buffer = NULL;
  gst_find.data = &find;
  gst_find.peek = simple_find_peek;
  gst_find.suggest = simple_find_suggest;
  while (walk) {
    GstTypeFindFactory *factory = GST_TYPE_FIND_FACTORY (walk->data);
    
    gst_type_find_factory_call_function (factory, &gst_find);
    if (find.best_probability >= GST_TYPE_FIND_MAXIMUM)
      break;
    walk = g_list_next (walk);
  }
  g_list_free (type_list);
  if (find.best_probability > 0) {
    id3->srcpad = gst_pad_new_from_template (GST_PAD_TEMPLATE_GET (src_templ), "src");
    gst_pad_set_event_function (id3->srcpad, gst_id3types_src_event);
    gst_pad_set_event_mask_function (id3->srcpad, gst_id3types_event_mask);
    gst_pad_set_query_type_function (id3->srcpad, gst_id3types_get_query_types);
    gst_pad_set_query_function (id3->srcpad, gst_id3types_handle_query);
    g_assert (gst_pad_try_set_caps (id3->srcpad, find.caps) == GST_PAD_LINK_OK);
    gst_element_add_pad (GST_ELEMENT (id3), id3->srcpad);
    if (bs->listavail) {
      gst_bytestream_read (bs, &buf, bs->listavail);
      gst_pad_push (id3->srcpad, GST_DATA (buf));
    }
    gst_id3types_cont (id3);
  } else {
    gst_element_error (element,
		       "Media stream type not detected after id3v1/2 header");
  }
  if (find.caps)
    gst_caps_unref (find.caps);
  if (find.buffer)
    gst_data_unref (GST_DATA (find.buffer));

error:
  gst_bytestream_destroy (bs);
}

static void
gst_id3types_get_property (GObject    *object,
			   guint       prop_id,
			   GValue     *value,
			   GParamSpec *pspec)
{
  GstID3Types *id3;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_ID3TYPES (object));
  id3 = GST_ID3TYPES (object);

  switch (prop_id) {
    case ARG_METADATA:
      g_value_set_boxed (value, id3->metadata);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstElementStateReturn 
gst_id3types_change_state (GstElement *element) 
{
  GstID3Types *id3;

  g_return_val_if_fail (GST_IS_ID3TYPES (element), GST_STATE_FAILURE);
  id3 = GST_ID3TYPES (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_READY_TO_PAUSED:
      gst_id3types_start (id3);
      break;
    case GST_STATE_PAUSED_TO_READY:
      gst_id3types_stop (id3);
      break;
    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

static gboolean
plugin_init (GstPlugin *plugin)
{
  if (!gst_library_load ("gstbytestream"))
    return FALSE;
      
  /* create an elementfactory for the id3types element */
  if (!gst_element_register (plugin, "id3types", GST_RANK_SECONDARY, GST_TYPE_ID3TYPES))
    return FALSE;
  return TRUE;
}

#ifdef HAVE_MAD
GST_PLUGIN_DEFINE (
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "id3types",
  "Parses ID3v1 and ID3v2 tags",
  plugin_init,
  VERSION,
  "GPL", /* grmbl, where's a good C LGPL ID3v1/2 reader? */
  GST_COPYRIGHT,
  GST_PACKAGE,
  GST_ORIGIN
)
#else
GST_PLUGIN_DEFINE (
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "id3types",
  "Parses ID3v1 and ID3v2 tags",
  plugin_init,
  VERSION,
  "LGPL",
  GST_COPYRIGHT,
  GST_PACKAGE,
  GST_ORIGIN
)
#endif
