/*
* main.c
*
* Main file for the mighty GOK
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <dirent.h>
#include <gnome.h>
#include <cspi/spi.h>
#include <libbonobo.h>
#include <bonobo-activation/bonobo-activation.h>
#include <atk/atkobject.h>
#include <libspi/Accessibility.h>
#include <libspi/accessible.h>
#include <libspi/application.h>
#include <gconf/gconf-client.h>
#include <locale.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include "main.h"
#include "gok.h"
#include "gok-word-complete.h"
#include "gok-branchback-stack.h"
#include "gok-data.h"
#include "gok-scanner.h"
#include "gok-settings-dialog.h"
#include "callbacks.h"
#include "switchapi.h"
#include "gok-sound.h"
#include "gok-log.h"
#include "gok-gconf-keys.h"
#include "gok-predictor.h"
#include "gok-editor.h"
#include "gok-spy.h"
#include "gok-feedback.h"
#include "gok-modifier.h"
#include "gok-action.h"
#include <signal.h>
#include "gok-bounds.h"

#define USE_GCONF 1

#define APP_STATIC_BUFF_SZ 30
#define MAX_DELETABLE_KEYBOARDS 50

/* gok error exit codes */

#define GOK_ERROR_SETTINGSDIALOG_OPEN 3
#define GOK_ERROR_MAIN_CREATE_WINDOW  4
#define GOK_ERROR_DISPLAY_SCAN_MAIN   5
#define GOK_ERROR_ACTION_OPEN         6
#define GOK_ERROR_FEEDBACK_OPEN       7
#define GOK_ERROR_NO_XKB_EXTENSION    8
#define GOK_ERROR_NOSTICKYKEYS		  9

#define GOKMAINKEYBOARDNAME "main"
#define GOKLOGINKEYBOARDNAME "Keyboard"

#define KEYBOARD_ACCESSIBILITY_STICKY_KEYS_KEY "/desktop/gnome/accessibility/keyboard/stickykeys_enable"
#define GCONF_ACCESSIBILITY_KEY "/desktop/gnome/interface/accessibility"

/* pointer to the first keyboard in the list of keyboards */
static GokKeyboard* m_pKeyboardFirst = NULL;

/* pointer to the keyboard that is currently displayed */
static GokKeyboard* m_pCurrentKeyboard = NULL;

/* pointer to the window that holds the GOK keyboard */
static GtkWidget* m_pWindowMain = NULL;

/* pointer to the foreground window's accessible interface */
static Accessible* m_pForegroundWindowAccessible = NULL;

/* will be zero if the window center location should be stored */
static gint m_countIgnoreConfigure = 0;

/* width and height that we have resized the window to */
static gint m_OurResizeWidth = 0;
static gint m_OurResizeHeight = 0;

/* size and location of GOK window specified by given geometry */
static gboolean m_bUseGivenGeometry = FALSE;
static gint m_GeometryWidth = -1;
static gint m_GeometryHeight = -1;
static gint m_GeometryX = -1;
static gint m_GeometryY = -1;

/* pointer to the gok command predictor */
static Gok_Predictor m_pCommandPredictor = NULL;

/* Private functions */
gint gok_main_open(gint argc, gchar *argv[]);
void gok_main_close (void);
void gok_main_read_rc (void);
void gok_main_initialize_access_methods (void);
void gok_main_initialize_wordcomplete (void);
void gok_main_initialize_commandprediction (void);
gboolean gok_main_display_scan_main(void);
void gok_main_display_geometry_error (void);

/* poptOption storage */
static char* accessmethodname = NULL;
static char* inputdevicename = NULL;
static char* mainkeyboardname = NULL;
static int bDisplayKeyboardEditor = 0;
static char *geometry = NULL;
static int geometry_bitmask;
static int bLoginOption = 0;
static int bRememberGeometryOption = 0;
static int bDisplaySettingsDialog = 0;
static int bExtras = 0;

static struct poptOption options[] = {
	{
		"access-method", 
		'a',
		POPT_ARG_STRING, 
		&accessmethodname, 
		0,
		N_("Use the specified access method. NAME is a string and can be found in the various access method files (.xam) assigned to the \"name\" property of <gok:accessmethod> tag. Note this is not necessarily the same as the name of the .xam file."), 
		N_("NAME")
	},
	{
		"editor", 
		'e', 
		POPT_ARG_NONE, 
		&bDisplayKeyboardEditor, 
		0,
		N_("Start the GOK keyboard editor"), 
		NULL},
	{
		"extras", 
		'\0', 
		POPT_ARG_NONE, 
		&bExtras, 
		0,
		N_("Use special, but possibly unstable, gok stuff"),
		NULL},
	{
		"geometry", 
		'\0', 
		POPT_ARG_STRING, 
		&geometry, 
		0,
		N_("Whenever --geometry is not used gok remembers its position between invocations and starts in the position that it had when it was last shutdown.  When --geometry is used gok positions itself within the rectangular area of screen described by the given X11 geometry specification.  When --geometry is used gok does not remember its position when it shuts down.  This behaviour can be changed with the --remembergeometry flag which forces gok to remember its position when shutdown even when it was started with --geometry."),
		N_("GEOMETRY")
	},
	{
		/* TODO: NOT USED */
		"input-device", 
		'i', 
		POPT_ARG_STRING, 
		&inputdevicename, 
		0,
		N_("Use the specified input device"), 
		N_("DEVICENAME")
	},
	{
		"keyboard", 
		'k', 
		POPT_ARG_STRING, 
		&mainkeyboardname, 
		0,
		N_("Start GOK with the specified keyboard."), 
		N_("KEYBOARDNAME")
	},
	{
		"login", 
		'l', 
		POPT_ARG_NONE, 
		&bLoginOption, 
		0,
		N_("GOK will be used to login"),
		NULL
	},
	{
		"remembergeometry", 
		'\0', 
		POPT_ARG_NONE,
		&bRememberGeometryOption, 
		0,
		N_("Can be used with --geometry.  Forces GOK to remember its position when shutdown even when it was started with --geometry.  Please see the discussion under the --geometry flag for more information."),
		NULL
	},
	{
		"settings", 
		's', 
		POPT_ARG_NONE, 
		&bDisplaySettingsDialog, 
		0,
		N_("Open the settings dialog box when GOK starts"),
		NULL
	},

	/* End the list */
	{NULL, '\0', 0, NULL, 0, NULL, NULL}
};

static int segfaults = 0;
/* private */
static int gok_sig_handler(int sig)
{
	switch (sig) {
	case SIGSEGV:
		segfaults++;
		fprintf (stderr, "gok: Segmentation fault\n");
		break;
	case SIGTERM:
		fprintf (stderr, "gok: exiting (terminated)\n");
		break;
	case SIGINT:
	default:
		break;
	}
	
	if (segfaults == 1)
		gok_main_close();

	_exit(1);
	return -1;  /* this line gets rid of compiler warning */ 
}

/**
* main
* @argc: ignored
* @argv: ignored
*
* The GOK main function.
*
* returns: Program exit code.
**/
gint main (gint argc, gchar *argv[])
{
	gint result;
	
#ifdef ENABLE_NLS
    setlocale (LC_ALL, "");
    bindtextdomain (GETTEXT_PACKAGE, GOK_LOCALEDIR);
    textdomain (GETTEXT_PACKAGE);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif

	/* intialization */
	result = gok_main_open(argc, argv);
	
	if (result != 0)
	{
		return result;
	}
	
	gok_spy_add_idle_handler ();
	
	/* start the main hook */
	SPI_event_main();	

	/* cleanup */
	gok_main_close();
	 
	return 0;
}

/**
* gok_main_open
* @argc: ignored
* @argv: ignored
*
* The GOK initialization function.
*
* returns: a program exit code.
**/
gint gok_main_open(gint argc, gchar *argv[])
{
	GnomeProgram *gok_program;
	/* initialize member data */
	m_pForegroundWindowAccessible = NULL;
	m_pKeyboardFirst = NULL;
	m_pWindowMain = NULL;
	m_pCurrentKeyboard = NULL;
	m_countIgnoreConfigure = 0;
	
	/* Initialize Gnome */
	gok_program = gnome_program_init (PACKAGE, VERSION,
		LIBGNOMEUI_MODULE, argc, argv,
		GNOME_PARAM_POPT_TABLE, options,
		GNOME_PROGRAM_STANDARD_PROPERTIES,
	        GNOME_PARAM_APP_DATADIR, DATADIR"/gok", 
		LIBGNOMEUI_PARAM_CRASH_DIALOG, FALSE,
		NULL);

	/* Parse geometry */
	if (geometry != NULL)
	{
		geometry_bitmask = XParseGeometry (geometry, &m_GeometryX,
		                                   &m_GeometryY,
		                                   (unsigned int *)&m_GeometryWidth,
		                                   (unsigned int *)&m_GeometryHeight);

		if ( (geometry_bitmask & XValue)
		     && (geometry_bitmask & YValue)
		     && (geometry_bitmask & WidthValue)
		     && (geometry_bitmask & HeightValue) )
		{
			gok_bounds_get_upper_left (geometry_bitmask,
							m_GeometryX, m_GeometryY,
							m_GeometryWidth,
							m_GeometryHeight,
							gdk_screen_width (),
							gdk_screen_height (),
							&m_GeometryX, &m_GeometryY);

			gok_log_x ("gok screen space = %dx%d+%d+%d",
					m_GeometryWidth, m_GeometryHeight,
					m_GeometryX, m_GeometryY);

			/* given geometry is good so use it for GOK location */
			m_bUseGivenGeometry = TRUE;
		}
		else
		{
			fprintf (stderr, _("gok: Unsupported geometry specification\n"));
			fprintf (stderr, _("gok: Currently GOK requires that the x, y, width and height all be given\n"));
			gok_main_display_geometry_error ();
		}
	}

	/* initialise GConf */
	gconf_init (argc, argv, NULL);

	/* read the user profile data */
	gok_data_initialize ();

	gok_log ("gok_data_initialize has finished");

	/* load the ".rc" files used for the key styles */
	if (gok_data_get_use_gtkplus_theme () == FALSE)
	{
		gok_main_read_rc ();
		gok_log ("gok_main_read_rc has finished");
	}

	/* display the keyboard editor not the GOK */
	if (bDisplayKeyboardEditor)
	{
		SPI_init ();
		gok_editor_run();
		SPI_event_main();	
		gok_editor_close();
		SPI_exit();
		return 1;
	}
	
	/* check for xkb extension */
	if (!gok_main_has_xkb_extension())
	{
		gok_main_display_error (_("XKB extension is required."));
		return GOK_ERROR_NO_XKB_EXTENSION;
	}		

	/* read the keyboards */
	gok_main_read_keyboards ();

	/* initialize the actions */
	if (gok_action_open() == FALSE)
	{
		gok_main_display_error (_("Can't initialize actions."));
		return GOK_ERROR_ACTION_OPEN;
	}

	/* initialize the button callbacks */
	gok_control_button_callback_open();
	
	/* initialize the feedbacks */
	if (gok_feedback_open() == FALSE)
	{
		gok_main_display_error (_("Can't initialize feedbacks."));
		return GOK_ERROR_FEEDBACK_OPEN;
	}

	/* initialize the access methods */
	gok_main_initialize_access_methods ();

	/* initialize the SPI */
	SPI_init ();
	
	/* initialize the switch API */
	initSwitchApi();
	
	/* initialize the spy routines */
	gok_spy_open();
	gok_spy_register_windowchangelistener ((void *)gok_main_window_change_listener);
	gok_spy_register_mousebuttonlistener ((void *)gok_main_mousebutton_listener);	

	/* initialize sound */
	gok_sound_initialize();

	/* create the main window */
	m_pWindowMain = gok_main_create_window (
		gok_data_get_dock_type () != GOK_DOCK_NONE);
	if (m_pWindowMain == NULL)
	{
		gok_main_display_error (_("Can't create the main GOK window!"));
		return GOK_ERROR_MAIN_CREATE_WINDOW;
	}
	gtk_widget_show (m_pWindowMain);

	/* initialize these things */
	gok_chunker_initialize();
	gok_keyboard_initialize();
	gok_branchbackstack_initialize();
	
	if (!bLoginOption) {
		gok_main_initialize_wordcomplete ();
		gok_main_initialize_commandprediction ();
	}
	
	/* create the settings dialog */
	if (gok_settingsdialog_open (bDisplaySettingsDialog) == FALSE)
	{
		gok_main_display_error (_("Can't create the settings dialog window!"));
		return GOK_ERROR_SETTINGSDIALOG_OPEN;
	}


	/* a specified keyboard takes precedance, even if the login flag is specifed
	   we use the login keyboard, otherwise the regular main keyboard */
	if (mainkeyboardname == NULL) {
		if (bLoginOption) {
			mainkeyboardname = (gchar*) 
				g_malloc (strlen (GOKLOGINKEYBOARDNAME) + 1);
			strcpy (mainkeyboardname, (gchar *) GOKLOGINKEYBOARDNAME);
		}
		else {
			mainkeyboardname = (gchar*) 
				g_malloc (strlen (GOKMAINKEYBOARDNAME) + 1);
			strcpy (mainkeyboardname, (gchar *) GOKMAINKEYBOARDNAME);
		}
	}

	if (!bLoginOption) {
		gok_main_check_accessibility();
	}
	
	if (!gok_main_check_sticky_keys (m_pWindowMain)) {
		return GOK_ERROR_NOSTICKYKEYS;
	}

	/* display the "main" keyboard */
	if (gok_main_display_scan_main () == FALSE)
	{
		return GOK_ERROR_DISPLAY_SCAN_MAIN;
	}

gok_log ("check for currently active frame");
	if (m_pForegroundWindowAccessible == NULL) {
		/* check for currently active frame */
		Accessible* accessible;
		accessible = gok_spy_get_active_frame();
		if (accessible)
			gok_main_window_change_listener (accessible);
	}
gok_log ("finished check");
	
	/* connect signal handlers; don't trap SIGKILL */ 
	signal(SIGSEGV, (void (*)(int))gok_sig_handler);
	signal(SIGTERM, (void (*)(int))gok_sig_handler);
	signal(SIGINT,  (void (*)(int))gok_sig_handler);

	return 0;
}

/* private */
gboolean gok_main_display_scan_main()
{
	if (gok_main_display_scan ((GokKeyboard*)NULL, mainkeyboardname, KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL, KEYBOARD_SHAPE_BEST) == FALSE)
	{
		/* if no "main" keyboard then display first keyboard in list */
		if (gok_main_display_scan ((GokKeyboard*)NULL, NULL, KEYBOARD_TYPE_UNSPECIFIED, KEYBOARD_LAYOUT_NORMAL, KEYBOARD_SHAPE_BEST) == FALSE)
		{
			gok_main_display_error (_("No keyboards to display!"));
			return FALSE;
		}
	}
	return TRUE;
}

/**
* gok_main_get_first_keyboard
* 
* Accessor function.
*
* returns: A pointer to the first keyboard in the list of keyboards.
*/
GokKeyboard* gok_main_get_first_keyboard ()
{
	return m_pKeyboardFirst;
}

/**
* gok_main_set_first_keyboard
* @pKeyboard: Pointer to the keyboard that will be set as the first keyboard.
*
* Sets the first keyboard in the list of keyboards.
*/
void gok_main_set_first_keyboard (GokKeyboard* pKeyboard)
{
	m_pKeyboardFirst = pKeyboard;
}

/**
* gok_main_get_current_keyboard
* 
* Accessor function.
*
* returns: A pointer to the keyboard that is currently displayed.
*/
GokKeyboard* gok_main_get_current_keyboard ()
{
	return m_pCurrentKeyboard;
}

/**
* gok_main_get_main_window
* 
* Accessor function.
*
* returns: A pointer to main GOK window.
*/
GtkWidget* 
gok_main_get_main_window ()
{
	return m_pWindowMain;
}

static char cursor_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
 
void
gok_main_set_cursor (GdkCursor *cursor)
{
	GdkPixmap *source;
	GdkColor color = { 0, 0, 0, 0 };
	static GdkCursor *nilCursor = NULL;
	if (!cursor) {
		if (!nilCursor) {
			GdkPixmap *source = 
				gdk_bitmap_create_from_data (NULL, cursor_bits,
							     8, 8);
			nilCursor = gdk_cursor_new_from_pixmap (source, source, 
							     &color, &color, 1, 1);
		}		
		cursor = nilCursor;
	}
	gdk_window_set_cursor (gok_main_get_main_window ()->window, cursor);
}

/**
* gok_main_get_command_predictor
* 
* Accessor function.
*
* returns: A pointer to command prediction engine
*/
Gok_Predictor gok_main_get_command_predictor ()
{
	return m_pCommandPredictor;
}

/**
* gok_main_get_foreground_window_accessible
* 
* Accessor function.
*
* returns: A pointer to foreground window's accessible interface.
*/
Accessible* gok_main_get_foreground_window_accessible()
{
	gok_log_enter();
	gok_log ("address:%d",m_pForegroundWindowAccessible);
	gok_log_leave();
	return m_pForegroundWindowAccessible;
}


/**
* gok_main_window_change_listener
* @pAccessible: Pointer to the foreground window's accessible interface.
* 
* This function is called each time the foreground window changes.
*
*/
void gok_main_window_change_listener (Accessible* pAccessible)
{
	GokKeyboard* pKeyboard = NULL;

	gok_log_enter();	

	/* note: ref counting happens in gok-spy's listeners */
	m_pForegroundWindowAccessible = pAccessible;
	
	/* we might currently have a dynamic keyboard so we need to branch to be safe */
	gok_main_display_scan_previous_premade();
	
	/*if (*/gok_keyboard_validate_dynamic_keys (pAccessible);/* == TRUE)*/
	{
		/* rechunk the keyboard because keys have changed state */
		gok_scanner_stop();
		pKeyboard = gok_main_get_current_keyboard();
		
		gok_chunker_chunk (pKeyboard);
		pKeyboard->bRequiresChunking = FALSE;
	
		/* highlight the first chunk */
		gok_chunker_highlight_first_chunk();
		
		/* restart the scanning process */
		gok_scanner_start();
	}

	gok_main_raise_window ();

	gok_log_leave();	
}

/**
* gok_main_motion_listener
* @n_axes: Number of axes on which motion may have occurred
* @motion_data: An array of long ints containing the motion data. 
* @mods: long int, ignored
* @timestamp: long int, ignored
*
* This handler is called each time there is a motion event from the connected input device.
*/
void gok_main_motion_listener (gint n_axes, int *motion_data, long mods, long timestamp)
{
        GtkWidget *window = gok_main_get_main_window ();
	Display *display;
	GdkWindow *root;
	gint dx, dy;
	if (window) 
	{
		display = GDK_WINDOW_XDISPLAY (window->window);
		root = gdk_screen_get_root_window (gdk_drawable_get_screen (window->window));
		gok_log_enter();
		gok_log ("%d axes: [%d] [%d] mods[%lx] timestamp[%ld]", n_axes,
			motion_data[0], (n_axes > 1) ? motion_data[1] : 0, 
			mods, timestamp);
		gok_log_leave ();
		gok_scanner_input_motion (motion_data, n_axes);

		dx = motion_data[0] * gok_data_get_valuator_sensitivity ();
		dy = motion_data[1] * gok_data_get_valuator_sensitivity ();
		if (gok_data_get_drive_corepointer ()) {
			XWarpPointer (display, None, 
				      GDK_WINDOW_XWINDOW (root),
				      0, 0, 0, 0, dx, dy);
		}
	}
}

/**
* gok_main_button_listener
* @button: Switch number that has changed state.
* @state: State of the switch.
* @mods: long int, ignored
* @timestamp: long int, ignored
*
* This handler is called each time there is a button event.
*/
void gok_main_button_listener (gint button, gint state, long mods, long timestamp)
{
	gok_log_enter();
	gok_log("button[%d] state[%d] mods[%lx] timestamp[%ld]",button,state,mods,timestamp);

	if (gok_data_get_drive_corepointer ())
	{
		int x, y, win_x, win_y;
		Window root_ret, child_ret;
		unsigned int mask_ret;
		GtkWidget *window;
		char ename[12];
		snprintf (ename, 12, "b%d%c", button, (state) ? 'p' : 'r');
		window = gok_main_get_main_window ();
		XQueryPointer (GDK_WINDOW_XDISPLAY (window->window), 
			       GDK_WINDOW_XWINDOW (window->window), 
			       &root_ret, &child_ret, &x, &y,
			       &win_x, &win_y, &mask_ret);
		SPI_generateMouseEvent (x, y, ename);
	}

	if (state == 0)
	{
		gok_main_raise_window ();

		switch (button)
		{
			case 1:
				gok_scanner_on_switch1_up();
				break;
			case 2:
				gok_scanner_on_switch2_up();
				break;
			case 3:
				gok_scanner_on_switch3_up();
				break;
			case 4:
				gok_scanner_on_switch4_up();
				break;
			case 5:
				gok_scanner_on_switch5_up();
				break;
			default:
				gok_log_x("this mouse button wasted by gok!");
				break;
		}
	}
	else
	{
		switch (button)
		{
			case 1:
				gok_scanner_on_switch1_down();
				break;
			case 2:
				gok_scanner_on_switch2_down();
				break;
			case 3:
				gok_scanner_on_switch3_down();
				break;
			case 4:
				gok_scanner_on_switch4_down();
				break;
			case 5:
				gok_scanner_on_switch5_down();
				break;
			default:
				gok_log_x("this mouse button wasted by gok!");
				break;
		}
	}

	gok_log_leave();	
}

/**
 * gok_main_raise_window:
 * hack to make sure gok is not occluded - TODO: seems to be unstable
 *
 **/
void
gok_main_raise_window (void)
{
	if (m_pWindowMain && m_pWindowMain->window) 
	{
		gdk_window_raise (m_pWindowMain->window);
	}
}

/**
* gok_main_mouse_button_listener
* @button: Mouse button number that has changed state.
* @state: State of the button.
* @mods: long int, ignored
* @timestamp: long int, ignored
*
* This handler is called each time there is a mouse button event.
*/
void gok_main_mousebutton_listener (gint button, gint state, long mods, long timestamp)
{
	gok_log_enter();
	gok_log("mouse button[%d] state[%d] mods[%lx] timestamp[%ld]",button,state,mods,timestamp);

	if (state == 0)
	{
		gok_main_raise_window ();

		/* TODO: account for reversed button order, which is possible! 
		 * i.e. leftbutton == 3, etc.
		 */
		/*
		 * Since we now get global mouse button notifications
		 * from at-spi, we need to filter out those that occur
		 * _outside_ the GOK window if the current access
		 * method is "Direct Selection."
		 */
		switch (button)
		{
			case 1:
				gok_scanner_left_button_up();
				break;
			case 2:
				gok_scanner_middle_button_up();
				break;
			case 3:
				gok_scanner_right_button_up();
				break;
			case 4:
				gok_scanner_on_button4_up();
				break;
			case 5:
				gok_scanner_on_button5_up();
				break;
			default:
				gok_log_x("this mouse button wasted by gok!");
				break;
		}
	}
	else
	{
		switch (button)
		{
			case 1:
				gok_scanner_left_button_down();
				break;
			case 2:
				gok_scanner_middle_button_down();
				break;
			case 3:
				gok_scanner_right_button_down();
				break;
			case 4:
				gok_scanner_on_button4_down();
				break;
			case 5:
				gok_scanner_on_button5_down();
				break;
			default:
				gok_log_x("this mouse button wasted by gok!");
				break;
		}
	}

	gok_log_leave();	
}

/**
* gok_main_display_scan_reset
* 
* Display and scan the current keyboard. This should be called after the user settings 
* have been changed.
*
* returns: TRUE if the keyboard was displayed, FALSE if not.
*/
gboolean gok_main_display_scan_reset ()
{
	return gok_main_ds (m_pCurrentKeyboard);
}

/**
* gok_main_display_scan_previous_premade
* 
* Display and scan the first keyboard pulled from the branch-back-stack
* that is premade. Delete any dynamic keyboards that are on the stack before
* the first premade keyboard.
* Don't push the current keyboard on the branch-back-stack.
*
* returns: TRUE if the previous keyboard was displayed, FALSE if not.
*/
gboolean gok_main_display_scan_previous_premade ()
{
	gboolean bReturnCode;
	GokKeyboard* arrayKeyboardsToDelete[MAX_DELETABLE_KEYBOARDS];
	GokKeyboard* pKeyboard;
	int index;
	
	gok_log_enter();

	pKeyboard = m_pCurrentKeyboard;
	if (pKeyboard->bDynamicallyCreated != TRUE)
	{
		gok_log_leave();
		return gok_main_display_scan_previous();
	}

	/* check this flag before branching back */
/*
	if (gok_data_get_dynamic_branchback() == FALSE)
	{
		return;
	}
*/
	
	/* initialize this array */
	for (index = 0; index < MAX_DELETABLE_KEYBOARDS; index++)
	{
		arrayKeyboardsToDelete[index] = NULL;
	}

	if (gok_branchbackstack_is_empty() == TRUE)
	{
		/* no keyboard on the branch back stack */
		gok_log("no dynamic keyboards to remove during branch");
		gok_log_leave();
		return FALSE;
	}

	/* start pulling off the branch back stack */
	/* stop when we get to a premade keyboard */
	index = -1;
	
	while ((pKeyboard->bDynamicallyCreated == TRUE) && (index < MAX_DELETABLE_KEYBOARDS))
	{
		/* store the dynamic keyboards in the array */
		gok_log("adding [%s] to the list of keyboards to delete", pKeyboard->Name);
		arrayKeyboardsToDelete[++index] = pKeyboard;
		pKeyboard = gok_branchbackstack_pop();
		if (pKeyboard == NULL)
		{
			break;
		}
	}
	
	if (pKeyboard == NULL)
	{
		gok_log_x ("No premade keyboards in stack in gok_main_display_scan_previous_premade!\n");
		gok_log_leave();
		return FALSE;
	}
		
	/* display and scan the premade keyboard */
	bReturnCode = gok_main_ds (pKeyboard);
	
	/* delete all the dynamic keyboards that were on the stack */
	while (index >= 0)
	{
		pKeyboard = arrayKeyboardsToDelete[index];
		if (m_pCurrentKeyboard != pKeyboard)  /* hack to fix potentially nastiness */
		{
			gok_log("deleting a dynamic keyboard with index %d",
			        index);
			gok_keyboard_delete (arrayKeyboardsToDelete[index],FALSE);
		}
		else
		{
			gok_log_x("an attempt was made to delete the currently displayed keyboard!");
		}

		index--;
	}
	
	gok_log_leave();
	return bReturnCode;
}

/**
* gok_main_display_scan_previous
* 
* Display and scan the previous keyboard (pulled from the branch-back-stack).
* Don't push the current keyboard on the branch-back-stack.
*
* returns: TRUE if the previous keyboard was displayed, FALSE if not.
*/
gboolean gok_main_display_scan_previous ()
{
	gboolean bReturnCode;
	GokKeyboard* pKeyboardToDelete;
	
	gok_log_enter();
	
	bReturnCode = TRUE;
	pKeyboardToDelete = NULL;

	if (gok_branchbackstack_is_empty() == TRUE)
	{
		/* no keyboard on the branch back stack */
		gok_log_leave();
		return FALSE;
	}

	/* are we leaving a dynamically created keyboard? */
	if (m_pCurrentKeyboard->bDynamicallyCreated == TRUE)
	{
		/* delete this keyboard after we switch to the new keyboard */
		pKeyboardToDelete = m_pCurrentKeyboard;
	}
	
	/* display and scan the previous keyboard */
	bReturnCode = gok_main_ds (gok_branchbackstack_pop());
	
	if (pKeyboardToDelete != NULL)
	{
		/* delete the dynamically created keyboard */
		/* make sure we branched back before deleting it */
		if (bReturnCode == TRUE)
		{
			if (m_pCurrentKeyboard != pKeyboardToDelete) /* hack to fix potentially nastiness */
			{
				gok_keyboard_delete (pKeyboardToDelete,FALSE);
			}
			else
			{
				gok_log_x("an attempt was made to delete the currently displayed keyboard!");
			}
		}
	}
	
	gok_log_leave();
	return bReturnCode;
}

GokKeyboard *
gok_main_keyboard_find_byname (const gchar *NameKeyboard)
{
	GokKeyboard *pKB = NULL;
	
	if (NameKeyboard == NULL)
	{
		/* if keyboard name is NULL then use first keyboard in list */
		pKB = m_pKeyboardFirst;
	}
	else
	{
		/* find the keyboard in the list (according to its name) */
		pKB = m_pKeyboardFirst;
		while (pKB != NULL)
		{
			if (g_strcasecmp (pKB->Name, NameKeyboard) == 0)
			{
				/* found the keyboard */
				break;
			}
			else
			{
				pKB = pKB->pKeyboardNext;
			}
		}
	}
	return pKB;
}

/**
* gok_main_display_scan
* @pKeyboard: If this is supplied it takes precedence over the name parameter (useful for dynamic keyboards with the same name)
* @nameKeyboard: Name of the keyboard you want displayed (must be in the list of keyboards.)
# @typeKeyboard: Can be used to describe a runtime keyboard type (e.g. a Menus keyboard)
* @layout: Can be used to specify a particular type of keyboard (Example: center-weighted or upper-left-weighted).
* @shape: Can be used to specify a particular shape of keyboard (Example: square).
*
* Display a keyboard and allow the user to make selections from it.
* The keyboard must have already been created and in the list of keyboards.
* The keyboard is specified by name and keyboard type. A name of NULL means 
* the first keyboard in the list. 
* The previous keyboard is stored on the "branch back stack".
*
* returns: TRUE if keyboard is displayed, FALSE if not.
*/
gboolean 
gok_main_display_scan (GokKeyboard* pKeyboard, gchar* nameKeyboard, KeyboardTypes typeKeyboard, KeyboardLayouts layout, KeyboardShape shape)
{
	/* FIXME we're not using the type params */
	
	GokKeyboard* pKB;
	gboolean bPushed;
	gboolean bReturnCode;

	gok_log_enter();
	g_assert (m_pKeyboardFirst != NULL);

	bPushed = FALSE;
	bReturnCode = FALSE;
	
	if (pKeyboard != NULL)
	{
		pKB = pKeyboard;
	}
	else
	{
		pKB = gok_main_keyboard_find_byname (nameKeyboard);
	}

	if (pKB == NULL)
	{
		/* keyboard not found in list! */
		gok_log_x("keyboard not found in list!");
		gok_log_leave();
		return FALSE;
	}

	/* push the current keyboard on the branch back stack */
	if ((m_pCurrentKeyboard != NULL) && (m_pCurrentKeyboard->bDynamicallyCreated != TRUE))
	{
		bPushed = TRUE;
		gok_branchbackstack_push (m_pCurrentKeyboard);
	}

	gok_log_leave();
	
	/* display the keyboard and start the scanning process */
	bReturnCode = gok_main_ds (pKB);
	
	/* hack */
	if ((bReturnCode == FALSE) && (bPushed == TRUE))
	{
		gok_branchbackstack_pop ();
	}
		
	return bReturnCode;
}

/**
* gok_main_ds
* @pKeyboard: Pointer to the keyboard that will be displayed and scanned.
*
* Does the actual work of displaying and scanning the keyboard.
*
* returns: TRUE if the keyboard was displayed, FALSE if not.
*/
gboolean gok_main_ds (GokKeyboard* pKeyboard)
{
	gok_log_enter();
	g_assert (pKeyboard != NULL);

	/* stop any key flashing */
	gok_feedback_timer_stop_key_flash();
	
	/* stop the scanning process */
	gok_scanner_stop();

	/* raise the window as a precaution */
	gok_main_raise_window ();

	/* create new keys for dynamic keyboards */
	if (pKeyboard->bDynamicallyCreated == TRUE)
	{
		if (gok_keyboard_update_dynamic (pKeyboard) == FALSE)
		{
			gok_log_leave();
			return FALSE;
		}
	}
	
	/* layout the keys on the keyboard */
	if (gok_keyboard_layout (pKeyboard, KEYBOARD_LAYOUT_NORMAL, 
							KEYBOARD_SHAPE_SQUARE, FALSE) == FALSE)
	{
		gok_log_leave();
		return FALSE;
	}
		
	/* display the keyboard */
	gok_keyboard_display (pKeyboard, m_pCurrentKeyboard, m_pWindowMain, TRUE);
	m_pCurrentKeyboard = pKeyboard;
	
	/* enable/disable the menu and toolbar keys */
	gok_keyboard_validate_dynamic_keys (gok_main_get_foreground_window_accessible());

	/* if there are command prediction keys - fill them in */
	gok_predictor_predict(m_pCommandPredictor);

	/* initialize the current access method */
	gok_scanner_reset_access_method ();
	
	/* chunk the keyboard if required */
	if (pKeyboard->bRequiresChunking == TRUE)
	{
		gok_chunker_chunk (pKeyboard);
		pKeyboard->bRequiresChunking = FALSE;
	}

	/* highlight the first chunk */
	gok_chunker_highlight_first_chunk();
	
	/* restart the scanning process */
	gok_scanner_start();

	gok_log_leave();
	return TRUE;
}

/**
* gok_main_create_window
* 
* Creates the window that holds the keyboards.
*
* returns: A pointer to the window, NULL if it couldn't be created.
*/
GtkWidget* gok_main_create_window (gboolean is_dock)
{
	GtkWidget *window1;
	GtkWidget *fixed1;

	gok_log_enter();
						     
	window1 = g_object_connect (gtk_widget_new (gtk_window_get_type (),
				"user_data", NULL,
				"can_focus", FALSE,
				"type", GTK_WINDOW_TOPLEVEL,
				"window-position", GTK_WIN_POS_CENTER,
				"title","GOK",
				"allow_grow", TRUE,
				"allow_shrink", TRUE,
				"border_width", 0,
				NULL),
			     "signal::realize", on_window1_realize, is_dock,
			     "signal::destroy", on_window1_destroy, NULL,
			     NULL);

	m_OurResizeWidth = 200;
	m_OurResizeHeight = 100;
	gtk_window_set_default_size (GTK_WINDOW (window1), m_OurResizeWidth, m_OurResizeHeight);


	fixed1 = gtk_fixed_new ();
	gtk_widget_ref (fixed1);
	gtk_object_set_data_full (GTK_OBJECT (window1), "fixed1", fixed1,
                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (fixed1);
	gtk_container_add (GTK_CONTAINER (window1), fixed1);

	gtk_signal_connect (GTK_OBJECT (window1), "motion_notify_event",
                      GTK_SIGNAL_FUNC (on_window1_motion_notify_event),
                      NULL);

	gtk_signal_connect (GTK_OBJECT (window1), "leave_notify_event",
                      GTK_SIGNAL_FUNC (on_window1_leave_notify_event),
                      NULL);

	gtk_signal_connect (GTK_OBJECT (window1), "enter_notify_event",
                      GTK_SIGNAL_FUNC (on_window1_enter_notify_event),
                      NULL);

	gtk_widget_add_events (window1, GDK_POINTER_MOTION_MASK);

	gtk_signal_connect (GTK_OBJECT (window1), "size_allocate",
                      GTK_SIGNAL_FUNC (on_window1_size_allocate),
                      NULL);
                      
	gtk_signal_connect (GTK_OBJECT (window1), "delete_event",
                      GTK_SIGNAL_FUNC (on_window1_delete_event),
                      NULL);

	gtk_signal_connect (GTK_OBJECT (window1), "configure_event",
                      GTK_SIGNAL_FUNC (on_window1_configure_event),
                      NULL);

	gok_log_leave();
	return window1;
}

/**
* gok_main_read_rc:
*
* Retreives the name of the directory containing gok.rc from
* GConf and loads it.
*/
void gok_main_read_rc ()
{
    GConfClient *gconf_client = NULL;
    GError *gconf_err = NULL;
    gchar *directory_name = NULL;
    gchar *complete_path = NULL;

    gconf_client = gconf_client_get_default ();
    
    directory_name = gconf_client_get_string (gconf_client,
					      GOK_GCONF_RESOURCE_DIRECTORY,
					      &gconf_err);
    if (directory_name == NULL)
    {
	gok_log_x ("Got NULL resource directory key from GConf");
    }
    else if (gconf_err != NULL)
    {
	gok_log_x ("Error getting resource directory key from GConf");
	g_error_free (gconf_err);
    }
    else
    {
        complete_path = g_build_filename (directory_name, "gok.rc", NULL);
	gtk_rc_parse (complete_path);
	g_free (complete_path);
	g_free (directory_name);
    }
}

/**
* gok_main_read_keyboards:
*
* Retrieves the name of the directory containing keyboards from GConf.
* Then loads the keyboards in that directory using
* gok_main_read_keyboards_from_dir.
*/
void gok_main_read_keyboards ()
{
    GConfClient *gconf_client = NULL;
    GError *gconf_err = NULL;
    gchar *directory_name = NULL;
    gboolean gconf_keyboard_directory_error = FALSE;

    gconf_client = gconf_client_get_default ();
    
    directory_name = gconf_client_get_string (gconf_client,
					      GOK_GCONF_KEYBOARD_DIRECTORY,
					      &gconf_err);
    if (directory_name == NULL)
    {
	gok_log_x ("Got NULL keyboard directory key from GConf");
        gok_main_display_gconf_error ();
	exit (1);
    }
    else if (gconf_err != NULL)
    {
	gok_log_x ("Error getting keyboard directory key from GConf");
	g_error_free (gconf_err);
	gconf_keyboard_directory_error = TRUE;
    }
    else
    {
	/* read the keyboard files from directory_name */
	gok_modifier_open();
	m_pKeyboardFirst = gok_main_read_keyboards_from_dir (directory_name);
    }

    g_free (directory_name);
    
    if ( gconf_keyboard_directory_error || (m_pKeyboardFirst == NULL) )
    {
	gok_main_display_error (_("Can't read any keyboards!"));
	exit (1);
    }
}

/**
* gok_main_read_keyboards_from_dir:
* @directory: The name of the directory to read the keyboard files from.
* 
* Reads all the keyboard files from the given directory.
*
* Returns: A pointer to the first keyboard, NULL if no keyboards could be read.
*/
GokKeyboard* gok_main_read_keyboards_from_dir (const char *directory)
{
	GokKeyboard* pKeyboardFirst;
	GokKeyboard* pKeyboard;
	GokKeyboard* pKeyboardPrevious;
	DIR* pDirectoryKeyboards;
	struct dirent* pDirectoryEntry;
	gchar* filename;
	gchar* complete_path;

	gok_log_enter();
	gok_log ("directory: %s", directory);
	pKeyboardFirst = NULL;
	pKeyboard = NULL;
	pKeyboardPrevious = NULL;

	/* get a list of all the files from the given directory name */
	/* first, open the keyboard directory */
	pDirectoryKeyboards = opendir (directory);
	if (pDirectoryKeyboards == NULL)
	{
		gok_log_x ("Can't open keyboard directory in gok_main_read_files!");
		gok_log_leave();
		return FALSE;
	}	

	/* create the default "core" keyboard (aka qwerty keyboard) */
	if (gok_data_get_use_xkb_kbd ()) {
		pKeyboard = pKeyboardFirst = gok_keyboard_get_core ();
		/* pKeyboard = pKeyboardFirst = gok_keyboard_get_alpha_by_frequency (); *//* TESTING ONLY!!!! */
		if (pKeyboard)
			pKeyboard->pKeyboardPrevious = NULL;
		pKeyboardPrevious = pKeyboard;
	}

	/* look at each file in the directory */
	while ((pDirectoryEntry = readdir (pDirectoryKeyboards)) != NULL)
	{
		/* is this a keyboard file? */
		filename = pDirectoryEntry->d_name;
		if ( (strlen(filename) >= 4)
		&& (strcmp(filename + (strlen(filename)-4), ".kbd") == 0) 
		&& ((strlen(filename) < 12) ||
		    (strcmp(filename + (strlen(filename)-12), 
			    "Keyboard.kbd") != 0) || 
		    (gok_data_get_use_xkb_kbd () == FALSE)))
		{
			/* read the keyboard file */
                        complete_path = g_build_filename (directory, filename,
                                                          NULL);
			gok_log ("complete_path = %s", complete_path);
			pKeyboard = gok_keyboard_read (complete_path);
			g_free (complete_path);
			
			/* add this keyboard into the list of keyboards */
			pKeyboard->pKeyboardPrevious = pKeyboardPrevious;
			if (pKeyboardFirst == NULL)
				pKeyboardFirst = pKeyboard;
			if (pKeyboardPrevious != NULL)
				pKeyboardPrevious->pKeyboardNext = pKeyboard;
			
			/* all predefined keyboards are assumed to be laid out */
			pKeyboard->bRequiresLayout = FALSE;
			pKeyboard->bLaidOut = TRUE;
			
			pKeyboardPrevious = pKeyboard;
		}
	}
	
	gok_log_leave();
	return pKeyboardFirst;
}

/**
* gok_main_initialize_access_methods:
*
* Retrieves the name of the directory containing access methods from GConf.
* Then calls gok_scanner_initialize passing the directory it
* got from GConf.
*/
void gok_main_initialize_access_methods ()
{
    GConfClient *gconf_client = NULL;
    GError *gconf_err = NULL;
    gchar *directory_name = NULL;

    gconf_client = gconf_client_get_default ();
    
    directory_name = gconf_client_get_string (gconf_client,
					      GOK_GCONF_ACCESS_METHOD_DIRECTORY,
					      &gconf_err);
    if (directory_name == NULL) {
		gok_log_x ("Got NULL access method directory key from GConf");
        gok_main_display_gconf_error ();
		exit (2);
    }
    else if (gconf_err != NULL) {
		gok_log_x ("Error getting access method directory key from GConf");
		g_error_free (gconf_err);
 		gok_main_display_error
			(_("could not access method directory key from GConf!"));
		exit(2);		
    }
    else if (!(gok_scanner_initialize(directory_name))) {
 		gok_main_display_error
			(_("possibly unknown access method!"));
		exit(2);
			
    }
    
    g_free (directory_name);
    
}

/**
* gok_main_initialize_wordcomplete:
*
* Retreives the name of the directory containing dictionary.txt from
* GConf and calls gok_wordcomplete_open passing that directory.
*/
void gok_main_initialize_wordcomplete ()
{
    GConfClient *gconf_client = NULL;
    GError *gconf_err = NULL;
    gchar *directory_name = NULL;

    gconf_client = gconf_client_get_default ();
    
    directory_name = gconf_client_get_string (gconf_client,
					      GOK_GCONF_DICTIONARY_DIRECTORY,
					      &gconf_err);
    if (directory_name == NULL)
    {
	gok_log_x ("Got NULL dictionary directory key from GConf");
    }
    else if (gconf_err != NULL)
    {
	gok_log_x ("Error getting dictionary directory key from GConf");
	g_error_free (gconf_err);
    }
    else
    {
	if ( !(gok_wordcomplete_open (gok_wordcomplete_get_default (),
				      directory_name)) ||
		! (gok_keyslotter_on (gok_wordcomplete_get_default (), KEYTYPE_WORDCOMPLETE))) {
	    gok_log_x ("Error initializing word completion");
	}
	
	g_free (directory_name);
    }
}

/**
* gok_main_initialize_commandpredictione:
*
* Retreives the name of the directory containing dictionary.txt from
* GConf and calls gok_wordcomplete_open passing that directory.
*/
void gok_main_initialize_commandprediction ()
{
	gok_log_enter();
    /* intialize command predictor */
    m_pCommandPredictor = gok_predictor_open();
   /*
    GConfClient *gconf_client = NULL;
    GError *gconf_err = NULL;
    gchar *directory_name = NULL;

    gconf_client = gconf_client_get_default ();
    
    directory_name = gconf_client_get_string (gconf_client,
					      GOK_GCONF_DICTIONARY_DIRECTORY,
					      &gconf_err);
    if (directory_name == NULL)
    {
	gok_log_x ("Got NULL dictionary directory key from GConf");
    }
    else if (gconf_err != NULL)
    {
	gok_log_x ("Error getting dictionary directory key from GConf");
	g_error_free (gconf_err);
    }
    else
    {
	if ( !(gok_wordcomplete_open (directory_name)) ) {
	    gok_log_x ("Error initializing word completion");
	}
	
	g_free (directory_name);
    }
    */
	gok_log_leave();
}

/**
* gok_main_resize_window
* @pWindow: Pointer to the main window.
* @pKeyboard: Pointer to the relevant (pending) keyboard (may be NULL).
* @Width: Width of the new window.
* @Height: Height of the new window.
* 
* Resizes the main window to the given width and height.
* The main window is centered over the the center location in gok_data
* The new window will not be resized so it appears off screen.
* When the new window is resized it generates 2 calls (configure events)to 
* gok_main_store_window_center. If the new window is not centered over the old 
* window then m_countIgnoreConfigure will be set to 2 so that the window 
* center is not changed.
*/
void gok_main_resize_window (GtkWidget* pWindow, GokKeyboard *pKeyboard, gint Width, gint Height)
{
	gint left;
	gint top;
	gint winX;
	gint winY;
	gint frameX;
	gint frameY;
	gint screenX;
	gint screenY;
	GdkRectangle rectFrame;

	/* ensure the window is at least this big */
	if (Width < 50)
	{
		Width = 50;
	}
	if (Height < 50)
	{
		Height = 50;
	}

	/* this may be the editor window so just resize it */
	if (pWindow != m_pWindowMain)
	{
		gtk_window_resize (GTK_WINDOW(pWindow), Width, Height);
		return;
	}
	
	/* let the keyboard know that the resize was caused by us */
	gok_keyboard_set_ignore_resize (TRUE);

	
	/* calculate the upper left corner of the window */
	switch (gok_data_get_dock_type () ) { /* if we're a dock */
	case GOK_DOCK_BOTTOM:
	  top = gdk_screen_height ();
	  left = 0;
	  if ((pKeyboard->expand == GOK_EXPAND_ALWAYS) || 
	      (gok_data_get_expand () && (pKeyboard->expand != GOK_EXPAND_NEVER))) {
		  Width = gdk_screen_width ();
		  gok_keyboard_set_ignore_resize (FALSE);
	  }
	  break;
	case GOK_DOCK_TOP:
	  top = 0;
	  left = 0;
	  if ((pKeyboard->expand == GOK_EXPAND_ALWAYS) || 
	      (gok_data_get_expand () && (pKeyboard->expand != GOK_EXPAND_NEVER))) {
		  Width = gdk_screen_width ();
		  gok_keyboard_set_ignore_resize (FALSE);
	  }
	  break;
	default:
	  top = gok_data_get_keyboard_y() - (Height / 2);
	  left = gok_data_get_keyboard_x() - (Width / 2);
	  break;
	}

	/* add the frame dimension to the window dimension */
	/* get the frame location */
	gdk_window_get_frame_extents ((GdkWindow*)pWindow->window, &rectFrame);
	/* get the window location */
	gdk_window_get_position (pWindow->window, &winX, &winY);

	if ((winX != 0) &&
		(winY != 0))
	{
		frameX = (winX - rectFrame.x);
		frameY = (winY - rectFrame.y);
	}
	else if (gok_data_get_dock_type () == GOK_DOCK_NONE)
	{
		frameX = 5;/* this is usually true */
		frameY = 22;/* this is usually true */
	}
	else {
	        frameX = frameY = 0;
	}
	left -= frameX;
	top -= frameY;
	
	/* make sure the window does not go off screen */
	screenX = gdk_screen_width();
	screenY = gdk_screen_height();
	m_countIgnoreConfigure = 0;	
	if (left < 0)
	{
		m_countIgnoreConfigure = 2;
		left = 0;
	}
	if (top < 0)
	{
		m_countIgnoreConfigure = 2;
		top = 0;
	}
	if ((Width + left + frameX) > screenX)
	{
		m_countIgnoreConfigure = 2;
		left = screenX - Width - (frameX * 2);
	}
	if ((Height + top + frameY) > screenY)
	{
		m_countIgnoreConfigure = 2;
		top = screenY - Height - frameY - (frameX * 2);
	}
	
	/* ensure the window is within the given geometry */
	if (m_bUseGivenGeometry == TRUE)
	{
		if (left < m_GeometryX)
		{
			left = m_GeometryX;
		}
		if (top < m_GeometryY)
		{
			top = m_GeometryY;
		}
	}

	/* move/resize the main window */
	gdk_window_move_resize ((GdkWindow*)pWindow->window, left, top, Width, Height);

	if (gok_data_get_dock_type () != GOK_DOCK_NONE) {
		/* if this is a "never expand" keyboard, unset struts! */
		if (pKeyboard && (pKeyboard->expand == GOK_EXPAND_NEVER))
			gok_main_update_struts (Width, 1);
		else
			gok_main_update_struts (Width, Height);
	}

	/* store these values */
	m_OurResizeWidth = Width;
	m_OurResizeHeight = Height;
}

void
gok_main_set_wm_dock (gboolean is_dock)
{
  Atom atom_type[1], atom_window_type;
  GtkWidget *widget = gok_main_get_main_window ();

  if (widget) gtk_widget_hide (widget);
  gdk_error_trap_push ();
  atom_window_type = gdk_x11_get_xatom_by_name ("_NET_WM_WINDOW_TYPE");
  if (is_dock) 
	  atom_type[0] = gdk_x11_get_xatom_by_name ("_NET_WM_WINDOW_TYPE_DOCK");
  else
	  atom_type[0] = gdk_x11_get_xatom_by_name ("_NET_WM_WINDOW_TYPE_NORMAL");

  XChangeProperty (GDK_WINDOW_XDISPLAY (widget->window), 
		   GDK_WINDOW_XWINDOW (widget->window), 
		   atom_window_type,
		   XA_ATOM, 32, PropModeReplace,
		   (guchar *) &atom_type, 1);
  gdk_error_trap_pop ();
  if (widget) gtk_widget_show (widget);

  if (m_pCurrentKeyboard) {
	  gok_keyboard_display (m_pCurrentKeyboard, m_pCurrentKeyboard,
				gok_main_get_main_window(), TRUE); 
          /* this should move us to appropriate screen edge */
  }

}

/*
 * TODO: GtkDoc this!
 */
void
gok_main_update_struts (gint width, gint height)
{
	GtkWidget *widget = gok_main_get_main_window ();
	Atom atom_strut = gdk_x11_get_xatom_by_name ("_NET_WM_STRUT");
	guint32 struts[4];

	switch (gok_data_get_dock_type ()) {
	case GOK_DOCK_TOP:
		struts[0] = struts[2] = struts[3] = 0;
		struts[1] = height;
		break;
	case GOK_DOCK_BOTTOM:
		struts[0] = struts[1] = struts[2]= 0;
		struts[3] = height;
		break;
	default:
		return;
		break;
	}
	gdk_error_trap_push ();
	/* this means that we are responsible for placing ourselves appropriately on screen */
	if (widget && widget->window)
		XChangeProperty (GDK_WINDOW_XDISPLAY (widget->window), 
				 GDK_WINDOW_XWINDOW (widget->window), 
				 atom_strut,
				 XA_CARDINAL, 32, PropModeReplace,
				 (guchar *) &struts, 4);
	gdk_error_trap_pop ();
}

/**
* gok_main_get_our_window_size
* @pWidth: Pointer to a variable that will hold the window width.
* @pHeight: Pointer to a variable that will hold the window height.
*
* Retreives the width and height that we last resized the window to.
*/
void gok_main_get_our_window_size (gint* pWidth, gint* pHeight)
{
	*pWidth = m_OurResizeWidth;
	*pHeight = m_OurResizeHeight;
}

/**
* gok_main_store_window_center
* 
* Stores in gok_data the center location of the current keyboard.
*/
void gok_main_store_window_center ()
{
	gint winX;
	gint winY;
	gint winWidthCurrent;
	gint winHeightCurrent;

	/* don't store the window center if geometry is specified */
	if (m_bUseGivenGeometry == TRUE)
	{
		return;
	}
	
	/* this flag may be set in gok_main_resize_window so ignore this call */
	if (m_countIgnoreConfigure != 0)
	{
		m_countIgnoreConfigure--;
		return;
	}

	/* get the center of the current window */	
	gdk_window_get_position (m_pWindowMain->window, &winX, &winY);
	gdk_window_get_size (m_pWindowMain->window, &winWidthCurrent, &winHeightCurrent);
	winX +=  winWidthCurrent / 2;	
	winY +=  winHeightCurrent / 2;

	/* update the gok_data with keyboard center */
	gok_data_set_keyboard_x (winX);
	gok_data_set_keyboard_y (winY);
}

/**
* gok_main_close
* 
* Delete any keyboards that were created.
* This must be called at the end of the program.
*/
void gok_main_close()
{
	/* NOTE: the order in which things are cleaned up is very important */

	GokKeyboard* pKeyboard;
	GokKeyboard* pKeyboardTemp;

	/* unhook listeners/callbacks */
	gok_spy_deregister_windowchangelistener ((void *)gok_main_window_change_listener);
	/* the call to deregister_mousebuttonlistener is in on_window1_destroy */
	
	gok_spy_stop();
	
	gok_log("BEFORE DELETING KEYBOARDS");
	gok_log("Keyboards news - deletes: [%d]",gok_keyboard_get_keyboards());
	gok_log("Stack pushes - pops:      [%d]",gok_branchbackstack_pushes_minus_pops());

	pKeyboard = gok_main_get_first_keyboard ();
	while (pKeyboard != NULL)
	{
		pKeyboardTemp = pKeyboard->pKeyboardNext;
		gok_keyboard_delete (pKeyboard, TRUE);
		pKeyboard = pKeyboardTemp;
	}
	
	/* TODO delete the current keyboard? */

	gok_log("AFTER DELETING KEYBOARDS");
	gok_log("Keyboards news - deletes: [%d]",gok_keyboard_get_keyboards());
	gok_log("Stack pushes - pops:      [%d]",gok_branchbackstack_pushes_minus_pops());

	gok_spy_close();
	SPI_exit();

	gok_settingsdialog_close();
	
	gok_scanner_close();
	gok_data_close();

	closeSwitchApi();
	gok_sound_shutdown();
	gok_modifier_close();
	gok_action_close();
	gok_feedback_close();
	gok_control_button_callback_close();
	gok_wordcomplete_close(gok_wordcomplete_get_default ());
	gok_predictor_close(m_pCommandPredictor);
	
	_exit(0);
}

/**
* gok_main_display_error
* @ErrorString: Fatal error message.
* 
* Displays a fatal error dialog (modal).
*/
void gok_main_display_error (gchar* ErrorString)
{
	GtkWidget* pDialog;
	gchar buffer [1024];
	
	strcpy (buffer, _("Sorry, GOK can't run because:\n"));
	strcat (buffer, ErrorString);
	
	pDialog = gtk_message_dialog_new (NULL,
		GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_MESSAGE_ERROR,
		GTK_BUTTONS_CLOSE,
		buffer);
	
	gtk_window_set_title (GTK_WINDOW (pDialog), _("GOK Fatal Error"));
	gtk_dialog_run (GTK_DIALOG (pDialog));
	gtk_widget_destroy (pDialog);
}

/**
* gok_main_display_gconf_error
* 
* Displays a gconf error message.
*/
void gok_main_display_gconf_error ()
{
    gok_main_display_error ( _("GOK uses GConf 2 to store its settings and requires certain settings to be in GConf to run.  GOK is currently unable to retrieve those settings.  If this is the first time that you have run gok after installing it you may need to restart gconfd, you can use this command: 'gconftool-2 --shutdown' or log out and back in."));
}

/**
* gok_main_display_geometry_error
* 
* Displays a dialog informing user that the given geometry was not correct.
*/
void gok_main_display_geometry_error ()
{
	GtkWidget* dialog;
        
	dialog = gtk_message_dialog_new (NULL,
	                                 GTK_DIALOG_DESTROY_WITH_PARENT,
	                                 GTK_MESSAGE_ERROR,
	                                 GTK_BUTTONS_CLOSE,
	                                 _("Currently GOK requires that the x, y, width and height all be given.  Sorry, your geometry specification will not be used."));
        
	gtk_window_set_title (GTK_WINDOW (dialog), _("gok: Unsupported geometry specification"));

	g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
	                          G_CALLBACK (gtk_widget_destroy),
	                          GTK_OBJECT (dialog));

	gtk_widget_show_all (dialog);
}

/**
* gok_main_get_use_geometry
* 
* returns: TRUE if the user has specified window geometry for the GOK.
* Returns FALSE if the GOK can use the whole screen.
*/
gboolean gok_main_get_use_geometry ()
{
	return m_bUseGivenGeometry;
}

/**
* gok_main_get_geometry
* @pRectangle: Pointer to a rectangle that will be populated with screen
* geometry that should be used by the GOK.
*/
void gok_main_get_geometry (GdkRectangle* pRectangle)
{
	g_assert (pRectangle != NULL);
	g_assert (m_bUseGivenGeometry == TRUE);

	pRectangle->x = m_GeometryX;
	pRectangle->y = m_GeometryY;
	pRectangle->width = m_GeometryWidth;
	pRectangle->height = m_GeometryHeight;
}

gboolean
gok_main_window_contains_pointer (void)
{
	gint x, y;
	GdkModifierType mask;
	GtkWidget *widget = gok_main_get_main_window ();
	if (gdk_window_get_pointer (widget->window, &x, &y, &mask) != NULL)
		return TRUE;
	return FALSE;
}

Display *
gok_main_display (void)
{
	if (gok_main_get_main_window () != NULL)
		return GDK_WINDOW_XDISPLAY (gok_main_get_main_window ()->window);
	return NULL;
}



/**
* gok_main_get_access_method_override
*
* Call this to see if an access method was passed in from the command line
*
* returns: name of access method or NULL
*/
gchar* 
gok_main_get_access_method_override (void)
{
	return accessmethodname;
}

/**
* gok_main_get_geometry
*
* Call this to see if the extras argument was passed in from the command line
*
* returns: gboolean
*/
gboolean 
gok_main_get_extras (void)
{
	return bExtras;
}

/**
* gok_main_get_login
*
* Call this to see if the extras argument was passed in from the command line
*
* returns: gboolean
*/
gboolean 
gok_main_get_login (void)
{
	return bLoginOption;
}


/**
* gok_main_has_xkb_extension
*
* Call this to see if xkb is enabled.
* 
* returns: gboolean
*/
gboolean
gok_main_has_xkb_extension ()
{
	/* TODO/revisit - is this a good check? */
	if (gok_keyboard_get_xkb_desc() != NULL)
	{
		return TRUE;
	}
	return FALSE;
}

gboolean
gok_main_check_sticky_keys (GtkWidget *widget)
{
	int op_rtn, event_rtn, error_rtn;
	Display *display = GDK_WINDOW_XDISPLAY (widget->window);
	GtkWidget *dialog;
	gboolean xkb_ok = FALSE;

	g_assert (display);
	if (XkbQueryExtension (display,
			       &op_rtn, &event_rtn, &error_rtn, NULL, NULL)) {
		XkbDescRec *xkb;
		xkb = XkbGetMap (display, XkbAllComponentsMask, XkbUseCoreKbd);
		if (xkb) {
			XkbGetControls (display,
					XkbControlsEnabledMask,
					xkb);
			xkb_ok = TRUE;
		} 
		if (!xkb || !(xkb->ctrls->enabled_ctrls & XkbStickyKeysMask)) {
			/* Turn on sticky keys and warn user! */
#if USE_GCONF
			if (1) {
				GConfClient *gconf_client = gconf_client_get_default ();
				gok_gconf_set_bool (gconf_client, KEYBOARD_ACCESSIBILITY_STICKY_KEYS_KEY, 
						    TRUE); /* turn on gconf key */
#else
			if (XkbChangeEnabledControls (display,
						      XkbUseCoreKbd,
						      XkbStickyKeysMask,
						      XkbStickyKeysMask)) {
#endif
				xkb_ok = TRUE;
				if (!bLoginOption) {
				  dialog = gtk_message_dialog_new (GTK_WINDOW (widget),
								 GTK_DIALOG_DESTROY_WITH_PARENT,
								 GTK_MESSAGE_INFO,
								 GTK_BUTTONS_OK,
								 "GOK has enabled Sticky Keys, which it requires.\n");
				
				  /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
				  g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
							  G_CALLBACK (gtk_widget_destroy),
							  GTK_OBJECT (dialog));
				  gtk_widget_show_all (dialog);
				}
			}
			else
				xkb_ok = FALSE;
		}
		XkbFreeKeyboard (xkb, 0, True);
	}
	if (xkb_ok == FALSE) {
		if (!bLoginOption) {
                /* post an error dialog */
			dialog = gtk_message_dialog_new (GTK_WINDOW (widget),
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 "GOK cannot run because XKB display extension is missing.\n");		
			g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
					  G_CALLBACK (gtk_widget_destroy),
					  GTK_OBJECT (dialog));
			gtk_widget_show_all (dialog);
		}
		return FALSE;
	}
	return TRUE;
}



static GtkWidget* acd = NULL; /* accessibility check dialog */

void check_accessibility_cb ( GObject* o, gpointer* data )
{
	gtk_widget_destroy(acd);	
	
	if (strcmp((gchar*)data,"logout") == 0) {
		gok_gconf_set_bool ( gok_data_get_gconf_client(),
			GCONF_ACCESSIBILITY_KEY, TRUE );
			gok_main_close();  /* FIXME: need to logout */
			_exit (0);
	}
	else if (strcmp((gchar*)data,"quit") == 0) {
		gok_main_close();
		_exit (0);
	}
	else if (strcmp((gchar*)data,"continue") == 0) {
		/* maybe change gok somehow to show user weakened status */
	}

}

void
gok_main_check_accessibility ()
{
    gboolean     accessibility_on;
    GConfClient* client;
	
	client = gok_data_get_gconf_client();

    /* check if accessibility flag is TRUE */
	gok_gconf_get_bool ( client, GCONF_ACCESSIBILITY_KEY, &accessibility_on );
        
    if (!accessibility_on) {
		GtkWidget*	button;
		gint         response_id;
		gboolean 	returnCode = TRUE;
		
		acd = gtk_message_dialog_new (
			GTK_WINDOW(m_pWindowMain), 
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_WARNING,
			GTK_BUTTONS_NONE,			
			_("Assistive Technology Support Is Not Enabled." 
		"\n\n" 
		"You can start GOK without enabling support for assistive technologies. "
		"However, some of the features of the application might not be available." 
		"\n\n"
		"To enable support for assistive technologies "
		"and log in to a new session with the change enabled, "
		"click "
		"Enable and Log Out" "."
		"\n\n"
		"To continue using GOK, " 
		"click "
		"Continue" "."
		"\n\n"
		"To quit GOK, " 
		"click " 
		"Close" "." 
		"\n\n"));
		   
		button = gtk_button_new_with_label(_("Enable and Log Out"));
		gtk_container_add (GTK_CONTAINER (GTK_DIALOG(acd)->action_area),
						  button);
		g_signal_connect(G_OBJECT (button), "clicked", 
			G_CALLBACK (check_accessibility_cb), (gpointer) "logout");
			
		button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
		gtk_container_add (GTK_CONTAINER (GTK_DIALOG(acd)->action_area),
						  button);
		g_signal_connect(G_OBJECT (button), "clicked", 
			G_CALLBACK (check_accessibility_cb), (gpointer) "quit");

		button = gtk_button_new_with_label(_("Continue"));
		gtk_container_add (GTK_CONTAINER (GTK_DIALOG(acd)->action_area),
						  button);
		g_signal_connect(G_OBJECT (button), "clicked", 
			G_CALLBACK (check_accessibility_cb), (gpointer) "continue");
						  

		gtk_widget_show_all (acd);
		
	}
}
