/*
 * AT-SPI - Assistive Technology Service Provider Interface
 * (Gnome Accessibility Project; http://developer.gnome.org/projects/gap)
 *
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <popt.h>
#include <gdk/gdkwindow.h>
#include <libbonobo.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <X11/Xatom.h>
#include "magnifier.h"
#include "magnifier-private.h"
#include "zoom-region.h"
#include "zoom-region-private.h"
#include "GNOME_Magnifier.h"

static GObjectClass *parent_class = NULL;

enum {
	MAGNIFIER_SOURCE_DISPLAY_PROP,
	MAGNIFIER_TARGET_DISPLAY_PROP,
	MAGNIFIER_SOURCE_SIZE_PROP,
	MAGNIFIER_TARGET_SIZE_PROP,
	MAGNIFIER_CURSOR_SET_PROP,
	MAGNIFIER_CURSOR_SIZE_PROP,
	MAGNIFIER_CURSOR_ZOOM_PROP,
	MAGNIFIER_CURSOR_COLOR_PROP,
	MAGNIFIER_CURSOR_HOTSPOT_PROP,
	MAGNIFIER_CURSOR_DEFAULT_SIZE_PROP,
	MAGNIFIER_CROSSWIRE_SIZE_PROP,
	MAGNIFIER_CROSSWIRE_CLIP_PROP,
	MAGNIFIER_CROSSWIRE_COLOR_PROP
} PropIdx;

static int _x_error = 0;

gboolean
magnifier_error_check (void)
{
	if (_x_error) {
		_x_error = 0;
		return TRUE;
	}
	return FALSE;
}

static int
magnifier_x_error_handler (Display	 *display,
			   XErrorEvent *error)
{
	if (error->error_code == BadAlloc) {
		_x_error = error->error_code;
	}
	else {
		return -1;
	}
	return 0;
}

static void
magnifier_realize (GtkWidget *widget)
{
	XWMHints wm_hints;
	Atom wm_window_protocols[2];
	Atom wm_type_atoms[1];
	Atom net_wm_window_type;
	GdkDisplay *target_display = gdk_drawable_get_display (widget->window);
	
	static gboolean initialized = FALSE;
	
	if (!initialized) {
		wm_window_protocols[0] = gdk_x11_get_xatom_by_name_for_display (target_display,
										"WM_DELETE_WINDOW");
		wm_window_protocols[1] = gdk_x11_get_xatom_by_name_for_display (target_display,
										"_NET_WM_PING");
		/* use DOCK until Metacity RFE for new window type goes in */
                wm_type_atoms[0] = gdk_x11_get_xatom_by_name_for_display (target_display,
									  "_NET_WM_WINDOW_TYPE_DOCK");
	}
  
	wm_hints.flags = InputHint;
	wm_hints.input = False;
	
	XSetWMHints (GDK_WINDOW_XDISPLAY (widget->window),
		     GDK_WINDOW_XWINDOW (widget->window), &wm_hints);
	
	XSetWMProtocols (GDK_WINDOW_XDISPLAY (widget->window),
			 GDK_WINDOW_XWINDOW (widget->window), wm_window_protocols, 2);

	net_wm_window_type = gdk_x11_get_xatom_by_name_for_display 
		(target_display, "_NET_WM_WINDOW_TYPE");

	if (net_wm_window_type && wm_type_atoms[0])
		XChangeProperty (GDK_WINDOW_XDISPLAY (widget->window),
				 GDK_WINDOW_XWINDOW (widget->window),
				 net_wm_window_type,
				 XA_ATOM, 32, PropModeReplace,
				 (guchar *)wm_type_atoms,
				 1);

	/* TODO: make sure this works/is reset if the DISPLAY 
	 * (as well as the SCREEN) changes.
	 */

	XSetErrorHandler (magnifier_x_error_handler);
	fprintf (stderr, "Realize complete.\n");
}

GdkWindow*
magnifier_get_root (Magnifier *magnifier)
{
	if (!magnifier->priv->root) {
		magnifier->priv->root = gdk_screen_get_root_window (
			gdk_display_get_screen (magnifier->source_display,
						magnifier->source_screen_num));
	}
	return magnifier->priv->root;
}

static void
magnifier_exit (GtkObject *object)
{
	gtk_main_quit ();
	exit (0);
}

static gint
magnifier_parse_display_name (Magnifier *magnifier, gchar *full_display_string,
			      gchar **display_name)
{
	gchar *screen_ptr;
	gchar **strings;
	
	if (display_name != NULL) {
		strings = g_strsplit (full_display_string, ":", 2);
		*display_name = strings [0];
		if (strings [1] != NULL)
			g_free (strings [1]);
	}

	screen_ptr = rindex (full_display_string, '.');
	if (screen_ptr != NULL) {
		return (gint) strtol (++screen_ptr, NULL, 10);
	}
	return 0;
}


#define GET_PIXEL(a,i,j,s,b) \
(*(guint32 *)(memcpy (b,(a) + ((j) * s + (i) * pixel_size_t), pixel_size_t)))

#define PUT_PIXEL(a,i,j,s,b) \
(memcpy (a + ((j) * s + (i) * pixel_size_t), &(b), pixel_size_t))

static void
magnifier_recolor_pixbuf (Magnifier *magnifier, GdkPixbuf *pixbuf)
{
	int rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	int i, j;
	int w = gdk_pixbuf_get_width (pixbuf);
	int h = gdk_pixbuf_get_height (pixbuf);
	guchar *pixels = gdk_pixbuf_get_pixels (pixbuf);
	guint32 pixval = 0, cursor_color = 0;
	size_t pixel_size_t = 3; /* FIXME: invalid assumption ? */

	cursor_color = ((magnifier->cursor_color & 0xFF0000) >> 16) +
		(magnifier->cursor_color & 0x00FF00) +
		((magnifier->cursor_color & 0x0000FF) << 16);
	for (j = 0; j < h; ++j) {
		for (i = 0; i < w; ++i) {
			pixval = GET_PIXEL (pixels, i, j, rowstride, &pixval);
			if ((pixval & 0xFFFFFF) == 0)
			{
				pixval = cursor_color;
				PUT_PIXEL (pixels, i, j, rowstride,
					   pixval);
			}
		}
	}
}

static void
magnifier_transform_cursor (Magnifier *magnifier)
{
	if (magnifier->priv->cursor) /* don't do this if cursor isn't intialized yet */
	{
		int width, height;
		int size;
		GdkPixbuf *scaled_cursor_pixbuf;
		GdkPixbuf *scaled_mask_pixbuf;
		GdkPixbuf *cursor_pixbuf;
		GdkPixbuf *mask_pixbuf;
		GdkPixmap *cursor_pixmap = magnifier->priv->cursor;
		GdkPixmap *mask_pixmap = magnifier->priv->cursor_mask;
		GdkGC *cgc;
		GdkGC *mgc;

		if (magnifier->cursor_size)
			size = magnifier->cursor_size;
		else
			size = magnifier->priv->cursor_default_size * 
			       magnifier->cursor_scale_factor;
		gdk_drawable_get_size (magnifier->priv->cursor, &width, &height);
		if ((size == width) 
		    && (magnifier->cursor_color == 0xFF000000))
			return; /* nothing changes */

		cgc = gdk_gc_new (cursor_pixmap);
		mgc = gdk_gc_new (mask_pixmap);
		cursor_pixbuf = gdk_pixbuf_get_from_drawable (NULL, cursor_pixmap,
							      NULL, 0, 0, 0, 0,
							      width, height);
		if (magnifier->cursor_color != 0xFF000000)
			magnifier_recolor_pixbuf (magnifier, cursor_pixbuf);
		mask_pixbuf = gdk_pixbuf_get_from_drawable (NULL,
							    mask_pixmap,
							    NULL, 0, 0, 0, 0,
							    width, height);
		scaled_cursor_pixbuf = gdk_pixbuf_scale_simple (
			cursor_pixbuf, size, size, GDK_INTERP_NEAREST);
		
		magnifier->cursor_hotspot.x = magnifier->priv->cursor_hotspot_x * size 
					    / magnifier->priv->cursor_default_size;
		magnifier->cursor_hotspot.y = magnifier->priv->cursor_hotspot_y * size 
					    / magnifier->priv->cursor_default_size;
					    
		scaled_mask_pixbuf = gdk_pixbuf_scale_simple (
			mask_pixbuf, size, size, GDK_INTERP_NEAREST);
		g_object_unref (cursor_pixbuf);
		g_object_unref (mask_pixbuf);
		g_object_unref (cursor_pixmap);
		g_object_unref (mask_pixmap);
		magnifier->priv->cursor = gdk_pixmap_new (
			magnifier->priv->w->window,
			size, size,
			-1);
		magnifier->priv->cursor_mask = gdk_pixmap_new (
			magnifier->priv->w->window,
			size, size,
			1);
		gdk_pixbuf_render_to_drawable (scaled_cursor_pixbuf,
					       magnifier->priv->cursor,
					       cgc,
					       0, 0, 0, 0, size, size,
					       GDK_RGB_DITHER_NONE, 0, 0 );
		scaled_mask_pixbuf = gdk_pixbuf_add_alpha (
			scaled_mask_pixbuf, True, 0, 0, 0);
		gdk_pixbuf_render_threshold_alpha (scaled_mask_pixbuf,
						   magnifier->priv->cursor_mask,
						   0, 0, 0, 0, size, size,
						   0x80);
		g_object_unref (scaled_cursor_pixbuf);
		g_object_unref (scaled_mask_pixbuf);
		g_object_unref (mgc);
		g_object_unref (cgc);
	}	
}

static void
magnifier_unref_zoom_region (gpointer data, gpointer user_data)
{
/*	Magnifier *magnifier = user_data; NOT USED */
	CORBA_Environment ev;
	GNOME_Magnifier_ZoomRegion zoom_region = data;
	CORBA_exception_init (&ev);
	GNOME_Magnifier_ZoomRegion_dispose (zoom_region, &ev);
}

static void
magnifier_init_cursor (Magnifier *magnifier)
{
	GdkPixbuf	*cursor_pixbuf = NULL;
	/*
	 * we check the cursor-set property string here,
	 * and create/apply the appropriate cursor settings
	 */
	    
	if (!strcmp (magnifier->cursor_set, "none"))
		magnifier->priv->cursor = NULL;
	else 
	{
		gchar *xpm_data_file = g_strconcat (CURSORSDIR,"/",
						    magnifier->cursor_set,
						    "-cursor.xpm", NULL);

		cursor_pixbuf = gdk_pixbuf_new_from_file (xpm_data_file, NULL);
		
		if (cursor_pixbuf)
		{
		    magnifier->cursor_hotspot.x = atoi (gdk_pixbuf_get_option (cursor_pixbuf,"x_hot") );
		    magnifier->cursor_hotspot.y = atoi (gdk_pixbuf_get_option (cursor_pixbuf,"y_hot") );

		    fprintf (stderr,"\nHOTSPOT  %d %d\n",
				magnifier->cursor_hotspot.x,
				magnifier->cursor_hotspot.y);
		    g_object_unref (G_OBJECT (cursor_pixbuf));
		}
		else
		    fprintf (stderr,"\nHOTSPOT failed\n");

		if (magnifier->priv->cursor) {
			g_object_unref (magnifier->priv->cursor);
			magnifier->priv->cursor = NULL;
		}
		if (magnifier->priv->w->window)
		{
			magnifier->priv->cursor = 
				gdk_pixmap_create_from_xpm (
					magnifier->priv->w->window,
					&magnifier->priv->cursor_mask,
					NULL,
					xpm_data_file);

			if (magnifier->priv->cursor) {
				gdk_drawable_get_size (magnifier->priv->cursor,
						       &magnifier->priv->cursor_default_size,
						       &magnifier->priv->cursor_default_size);
				magnifier->priv->cursor_hotspot_x = magnifier->cursor_hotspot.x;
				magnifier->priv->cursor_hotspot_y = magnifier->cursor_hotspot.y;
				fprintf (stderr, "getting cursor %p, mask %p\n",
					 magnifier->priv->cursor, magnifier->priv->cursor_mask);
			}
		}
		g_free (xpm_data_file);
	}
	
	magnifier_transform_cursor (magnifier);
}

static void
magnifier_get_property (BonoboPropertyBag *bag,
			BonoboArg *arg,
			guint arg_id,
			CORBA_Environment *ev,
			gpointer user_data)
{
	Magnifier *magnifier = user_data;
	GNOME_Magnifier_RectBounds rect_bounds;
	int csize = 0;
	
	switch (arg_id) {
	case MAGNIFIER_SOURCE_SIZE_PROP:
		rect_bounds.x1 = 0;
		rect_bounds.x2 = gdk_screen_get_width (
			gdk_display_get_screen (magnifier->source_display,
						magnifier->source_screen_num));
		rect_bounds.y1 = 0;
		rect_bounds.y2 = gdk_screen_get_height (
			gdk_display_get_screen (magnifier->source_display,
						magnifier->source_screen_num));
		BONOBO_ARG_SET_GENERAL (arg, rect_bounds,
					TC_GNOME_Magnifier_RectBounds,
					GNOME_Magnifier_RectBounds, NULL);
		break;
	case MAGNIFIER_TARGET_SIZE_PROP:
		rect_bounds.x1 = 0;
		rect_bounds.x2 = gdk_screen_get_width (
			gdk_display_get_screen (magnifier->target_display,
						magnifier->target_screen_num));
		rect_bounds.y1 = 0;
		rect_bounds.y2 = gdk_screen_get_height (
			gdk_display_get_screen (magnifier->target_display,
						magnifier->target_screen_num));
		BONOBO_ARG_SET_GENERAL (arg, rect_bounds,
					TC_GNOME_Magnifier_RectBounds,
					GNOME_Magnifier_RectBounds, NULL);

		break;
	case MAGNIFIER_CURSOR_SET_PROP:
		BONOBO_ARG_SET_STRING (arg, magnifier->cursor_set);
		break;
	case MAGNIFIER_CURSOR_SIZE_PROP:
		BONOBO_ARG_SET_INT (arg, magnifier->cursor_size);
		break;
	case MAGNIFIER_CURSOR_ZOOM_PROP:
		BONOBO_ARG_SET_FLOAT (arg, magnifier->cursor_scale_factor);
		break;
	case MAGNIFIER_CURSOR_COLOR_PROP:
		BONOBO_ARG_SET_GENERAL (arg, magnifier->cursor_color,
					TC_CORBA_unsigned_long,
					CORBA_unsigned_long, NULL);
		break;
	case MAGNIFIER_CURSOR_HOTSPOT_PROP:
		BONOBO_ARG_SET_GENERAL (arg, magnifier->cursor_hotspot,
					TC_GNOME_Magnifier_Point,
					GNOME_Magnifier_Point, NULL);

		break;
	case MAGNIFIER_CURSOR_DEFAULT_SIZE_PROP:
		if (magnifier->priv->cursor)
			gdk_drawable_get_size (magnifier->priv->cursor,
					       &csize, &csize);
		BONOBO_ARG_SET_INT (arg, csize);
		break;
	case MAGNIFIER_CROSSWIRE_SIZE_PROP:
		BONOBO_ARG_SET_INT (arg, magnifier->crosswire_size);
		break;
	case MAGNIFIER_CROSSWIRE_CLIP_PROP:
		BONOBO_ARG_SET_BOOLEAN (arg, magnifier->crosswire_clip);
		break;
	case MAGNIFIER_CROSSWIRE_COLOR_PROP:
		BONOBO_ARG_SET_LONG (arg, magnifier->crosswire_color);
		break;
	default:
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_NotFound);
	};
}

static void
magnifier_set_property (BonoboPropertyBag *bag,
			BonoboArg *arg,
			guint arg_id,
			CORBA_Environment *ev,
			gpointer user_data)
{
	Magnifier *magnifier = user_data;
	gchar *full_display_string;
	
	switch (arg_id) {
	case MAGNIFIER_SOURCE_DISPLAY_PROP:
		full_display_string = BONOBO_ARG_GET_STRING (arg);
		magnifier->source_screen_num =
			magnifier_parse_display_name (magnifier,
						      full_display_string,
						      NULL);
		magnifier->source_display =
			gdk_display_open (full_display_string);
		magnifier->priv->root =
			gdk_screen_get_root_window (
				gdk_display_get_screen (
					magnifier->source_display,
					magnifier->source_screen_num));
		break;
	case MAGNIFIER_TARGET_DISPLAY_PROP:
		full_display_string = BONOBO_ARG_GET_STRING (arg);
		magnifier->target_screen_num =
			magnifier_parse_display_name (magnifier,
						      full_display_string,
						      NULL);
		magnifier->target_display =
			gdk_display_open (full_display_string);
		if (GTK_IS_WINDOW (magnifier->priv->w)) 
			gtk_window_set_screen (GTK_WINDOW (magnifier->priv->w), 
					       gdk_display_get_screen (
						       magnifier->target_display,
						       magnifier->target_screen_num));
		break;
	case MAGNIFIER_SOURCE_SIZE_PROP:
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_ReadOnly);
		break;
	case MAGNIFIER_TARGET_SIZE_PROP:
	        magnifier->target_bounds = BONOBO_ARG_GET_GENERAL (arg,
								   TC_GNOME_Magnifier_RectBounds,
								   GNOME_Magnifier_RectBounds,
								   NULL);
		gtk_window_move (GTK_WINDOW (magnifier->priv->w),
				 magnifier->target_bounds.x1,
				 magnifier->target_bounds.y1);
		
		gtk_window_resize (GTK_WINDOW (magnifier->priv->w),
				   magnifier->target_bounds.x2 - magnifier->target_bounds.x1,
				   magnifier->target_bounds.y2 - magnifier->target_bounds.y1);
		
		break;
	case MAGNIFIER_CURSOR_SET_PROP:
		magnifier->cursor_set = g_strdup (BONOBO_ARG_GET_STRING (arg));
		magnifier_init_cursor (magnifier);
		break;
	case MAGNIFIER_CURSOR_SIZE_PROP:
		magnifier->cursor_size = BONOBO_ARG_GET_INT (arg);
		magnifier_transform_cursor (magnifier);
		break;
	case MAGNIFIER_CURSOR_ZOOM_PROP:
		magnifier->cursor_scale_factor = BONOBO_ARG_GET_FLOAT (arg);
		magnifier_transform_cursor (magnifier);
		break;
	case MAGNIFIER_CURSOR_COLOR_PROP:
		magnifier->cursor_color = BONOBO_ARG_GET_GENERAL (arg,
								  TC_CORBA_unsigned_long, 
								  CORBA_unsigned_long, 
								  NULL);
		magnifier_init_cursor (magnifier);
		break;
	case MAGNIFIER_CURSOR_HOTSPOT_PROP:
		magnifier->cursor_hotspot = BONOBO_ARG_GET_GENERAL (arg,
								    TC_GNOME_Magnifier_Point,
								    GNOME_Magnifier_Point,
								    NULL);
		/* TODO: notify zoomers */
                /* FIXME: don't call init_cursor, it overwrites this property! */
		magnifier_transform_cursor (magnifier); 
		break;
	case MAGNIFIER_CURSOR_DEFAULT_SIZE_PROP:
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_ReadOnly);
		break;
	case MAGNIFIER_CROSSWIRE_SIZE_PROP:
		magnifier->crosswire_size = BONOBO_ARG_GET_INT (arg);
		/* TODO: notify zoomers */
		break;
	case MAGNIFIER_CROSSWIRE_CLIP_PROP:
		magnifier->crosswire_clip = BONOBO_ARG_GET_BOOLEAN (arg);
		break;
	case MAGNIFIER_CROSSWIRE_COLOR_PROP:
		magnifier->crosswire_color = BONOBO_ARG_GET_LONG (arg);
		break;
	default:
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_NotFound);
	};
}

static void
magnifier_do_dispose (Magnifier *magnifier)
{
	/* FIXME: this is dead ropey code structuring */
	bonobo_activation_active_server_unregister (
		MAGNIFIER_OAFIID, BONOBO_OBJREF (magnifier));

	if (magnifier->zoom_regions)
		g_list_free (magnifier->zoom_regions);
	magnifier->zoom_regions = NULL;
	
	bonobo_main_quit ();
}

static void
magnifier_gobject_dispose (GObject *object)
{
	magnifier_do_dispose (MAGNIFIER (object));

	BONOBO_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
impl_magnifier_set_source_display (PortableServer_Servant servant,
				   const CORBA_char *display,
				   CORBA_Environment *ev)
{
	Magnifier *magnifier = MAGNIFIER (bonobo_object_from_servant (servant));
	BonoboArg *arg = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (arg, display);

	magnifier_set_property (magnifier->property_bag,
				arg,
				MAGNIFIER_SOURCE_DISPLAY_PROP,
				ev,
				magnifier);

	bonobo_arg_release (arg);
}

static void
impl_magnifier_set_target_display (PortableServer_Servant servant,
				   const CORBA_char *display,
				   CORBA_Environment *ev)
{
	Magnifier *magnifier = MAGNIFIER (bonobo_object_from_servant (servant));
	BonoboArg *arg = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (arg, display);

	magnifier_set_property (magnifier->property_bag,
				arg,
				MAGNIFIER_TARGET_DISPLAY_PROP,
				ev,
				magnifier);

	bonobo_arg_release (arg);
}

static GNOME_Magnifier_ZoomRegion
impl_magnifier_create_zoom_region (PortableServer_Servant servant,
				   const CORBA_float zx,
				   const CORBA_float zy,
				   const GNOME_Magnifier_RectBounds *roi,
				   const GNOME_Magnifier_RectBounds *viewport,
				   CORBA_Environment *ev)
{
	Magnifier *magnifier = MAGNIFIER (bonobo_object_from_servant (servant));
	CORBA_any viewport_any;
	ZoomRegion *zoom_region = zoom_region_new ();
	Bonobo_PropertyBag properties;
	GNOME_Magnifier_ZoomRegion retval;

	/* FIXME:
	 * shouldn't do this here, since it causes the region to get
	 * mapped onto the parent, if if it's not explicitly added!
	 */
	zoom_region->priv->parent = magnifier;

	retval = BONOBO_OBJREF (zoom_region);
	/* XXX: should check ev after each call, below */
	CORBA_exception_init (ev);
	GNOME_Magnifier_ZoomRegion_setMagFactor (retval, zx, zy, ev);

	if (ev->_major != CORBA_NO_EXCEPTION)
		fprintf (stderr, "EXCEPTION setMagFactor\n");

	CORBA_exception_init (ev);
	properties = GNOME_Magnifier_ZoomRegion_getProperties (retval, ev);
	if (ev->_major != CORBA_NO_EXCEPTION)
		fprintf (stderr, "EXCEPTION getProperties\n");

	viewport_any._type = TC_GNOME_Magnifier_RectBounds;
	viewport_any._value = (gpointer) viewport;
	Bonobo_PropertyBag_setValue (
		properties, "viewport", &viewport_any, ev);

	GNOME_Magnifier_ZoomRegion_setROI (retval, roi, ev);
	if (ev->_major != CORBA_NO_EXCEPTION)
		fprintf (stderr, "EXCEPTION setROI\n");

	CORBA_exception_init (ev);

	gtk_widget_set_size_request (magnifier->priv->canvas,
			   viewport->x2 - viewport->x1,
			   viewport->y2 - viewport->y1);
	gtk_widget_show (magnifier->priv->canvas);
	gtk_widget_show (magnifier->priv->w);

	bonobo_object_release_unref (properties, ev);
	
	return CORBA_Object_duplicate (retval, ev);
}

static
CORBA_boolean
impl_magnifier_add_zoom_region (PortableServer_Servant servant,
				const GNOME_Magnifier_ZoomRegion region,
				CORBA_Environment * ev)
{
	Magnifier *magnifier = MAGNIFIER (bonobo_object_from_servant (servant));

	/* FIXME: this needs proper lifecycle management */
	magnifier->zoom_regions = g_list_append (magnifier->zoom_regions, region);

	return CORBA_TRUE;
}

static Bonobo_PropertyBag
impl_magnifier_get_properties (PortableServer_Servant servant,
			       CORBA_Environment *ev)
{
	Magnifier *magnifier = MAGNIFIER (bonobo_object_from_servant (servant));

	return bonobo_object_dup_ref (
		BONOBO_OBJREF (magnifier->property_bag), ev);
}

GNOME_Magnifier_ZoomRegionList *
impl_magnifier_get_zoom_regions (PortableServer_Servant servant,
				 CORBA_Environment * ev)
{
	Magnifier *magnifier =
		MAGNIFIER (bonobo_object_from_servant (servant));

	GNOME_Magnifier_ZoomRegionList *list;
	CORBA_Object objref;
	int i, len;

	len = g_list_length (magnifier->zoom_regions);
	list = GNOME_Magnifier_ZoomRegionList__alloc ();
	list->_length = len;
	list->_buffer =
		GNOME_Magnifier_ZoomRegionList_allocbuf (list->_length);
	for (i = 0; i < len; ++i) {
		objref = g_list_nth_data (magnifier->zoom_regions, i);
		list->_buffer [i] =
			CORBA_Object_duplicate (objref, ev);
	}
	CORBA_sequence_set_release (list, CORBA_TRUE);
	
	return list; 
}

static void
impl_magnifier_clear_all_zoom_regions (PortableServer_Servant servant,
				       CORBA_Environment * ev)
{
	Magnifier *magnifier = MAGNIFIER (bonobo_object_from_servant (servant));

	fprintf (stderr, "Destroying all zoom regions!\n");
	g_list_foreach (magnifier->zoom_regions,
			magnifier_unref_zoom_region, magnifier);
	g_list_free (magnifier->zoom_regions);
	magnifier->zoom_regions = NULL;
}

static void
impl_magnifier_dispose (PortableServer_Servant servant,
			CORBA_Environment *ev)
{
	magnifier_do_dispose (
		MAGNIFIER (bonobo_object_from_servant (servant)));
}

static void
magnifier_class_init (MagnifierClass *klass)
{
        GObjectClass * object_class = (GObjectClass *) klass;
        POA_GNOME_Magnifier_Magnifier__epv *epv = &klass->epv;

	object_class->dispose = magnifier_gobject_dispose;

        epv->_set_SourceDisplay = impl_magnifier_set_source_display;
	epv->_set_TargetDisplay = impl_magnifier_set_target_display;
	epv->getProperties = impl_magnifier_get_properties;
	epv->getZoomRegions = impl_magnifier_get_zoom_regions;
	epv->createZoomRegion = impl_magnifier_create_zoom_region;
	epv->addZoomRegion = impl_magnifier_add_zoom_region;
	epv->clearAllZoomRegions = impl_magnifier_clear_all_zoom_regions;
	epv->dispose = impl_magnifier_dispose;
}

static void
magnifier_properties_init (Magnifier *magnifier)
{
	BonoboArg *def;
	
	magnifier->property_bag =
		bonobo_property_bag_new_closure (
			g_cclosure_new_object (
				G_CALLBACK (magnifier_get_property),
				G_OBJECT (magnifier)),
			g_cclosure_new_object (
				G_CALLBACK (magnifier_set_property),
				G_OBJECT (magnifier)));

	/* Aggregate so magnifier implements Bonobo_PropertyBag */
	bonobo_object_add_interface (BONOBO_OBJECT (magnifier),
				     BONOBO_OBJECT (magnifier->property_bag));

	def = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (def, getenv ("DISPLAY"));

	bonobo_property_bag_add (magnifier->property_bag,
				 "source-display-screen",
				 MAGNIFIER_SOURCE_DISPLAY_PROP,
				 BONOBO_ARG_STRING,
				 NULL,
				 "source display screen",
				 Bonobo_PROPERTY_WRITEABLE);

	bonobo_property_bag_add (magnifier->property_bag,
				 "target-display-screen",
				 MAGNIFIER_TARGET_DISPLAY_PROP,
				 BONOBO_ARG_STRING,
				 NULL,
				 "target display screen",
				 Bonobo_PROPERTY_WRITEABLE);

	bonobo_arg_release (def);

	bonobo_property_bag_add (magnifier->property_bag,
				 "source-display-size",
				 MAGNIFIER_SOURCE_SIZE_PROP,
				 TC_GNOME_Magnifier_RectBounds,
				 NULL,
				 "source display bounds/size",
				 Bonobo_PROPERTY_READABLE);
	
	bonobo_property_bag_add (magnifier->property_bag,
				 "target-display-bounds",
				 MAGNIFIER_TARGET_SIZE_PROP,
				 TC_GNOME_Magnifier_RectBounds,
				 NULL,
				 "target display bounds/size",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);

	bonobo_property_bag_add (magnifier->property_bag,
				 "cursor-set",
				 MAGNIFIER_CURSOR_SET_PROP,
				 BONOBO_ARG_STRING,
				 NULL,
				 "name of cursor set",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);

	def = bonobo_arg_new (BONOBO_ARG_INT);
	BONOBO_ARG_SET_INT (def, 64);
	
	bonobo_property_bag_add (magnifier->property_bag,
				 "cursor-size",
				 MAGNIFIER_CURSOR_SIZE_PROP,
				 BONOBO_ARG_INT,
				 def,
				 "cursor size, in pixels",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);
	bonobo_arg_release (def);
	
	bonobo_property_bag_add (magnifier->property_bag,
				 "cursor-scale-factor",
				 MAGNIFIER_CURSOR_ZOOM_PROP,
				 BONOBO_ARG_FLOAT,
				 NULL,
				 "scale factor for cursors (overrides size)",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);
	
	bonobo_property_bag_add (magnifier->property_bag,
				 "cursor-color",
				 MAGNIFIER_CURSOR_COLOR_PROP,
				 TC_CORBA_unsigned_long,
				 NULL,
				 "foreground color for 1-bit cursors, as ARGB",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);	

	bonobo_property_bag_add (magnifier->property_bag,
				 "cursor-hotspot",
				 MAGNIFIER_CURSOR_HOTSPOT_PROP,
				 TC_GNOME_Magnifier_Point,
				 NULL,
				 "hotspot relative to cursor's upper-left-corner, at default resolition",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);
	
	bonobo_property_bag_add (magnifier->property_bag,
				 "cursor-default-size",
				 MAGNIFIER_CURSOR_DEFAULT_SIZE_PROP,
				 BONOBO_ARG_INT,
				 NULL,
				 "default size of current cursor set",
				 Bonobo_PROPERTY_READABLE);

	bonobo_property_bag_add (magnifier->property_bag,
				 "crosswire-size",
				 MAGNIFIER_CROSSWIRE_SIZE_PROP,
				 BONOBO_ARG_INT,
				 NULL,
				 "thickness of crosswire cursor, in target pixels",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);
	
	bonobo_property_bag_add (magnifier->property_bag,
				 "crosswire-color",
				 MAGNIFIER_CROSSWIRE_COLOR_PROP,
				 BONOBO_ARG_LONG,
				 NULL,
				 "color of crosswire, as A-RGB; note that alpha is required. (use 0 for XOR wire)",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);

	bonobo_property_bag_add (magnifier->property_bag,
				 "crosswire-clip",
				 MAGNIFIER_CROSSWIRE_CLIP_PROP,
				 BONOBO_ARG_BOOLEAN,
				 NULL,
				 "whether to inset the cursor over the crosswire or not",
				 Bonobo_PROPERTY_READABLE |
				 Bonobo_PROPERTY_WRITEABLE);
}

static void
magnifier_init_window (Magnifier *magnifier)
{
	magnifier->priv->w =
		g_object_connect (gtk_widget_new (gtk_window_get_type (),
						  "user_data", NULL,
						  "can_focus", FALSE,
						  "type", GTK_WINDOW_TOPLEVEL,
						  "title", "magnifier",
						  "allow_grow", TRUE,
						  "allow_shrink", TRUE,
						  "border_width", 0,
						  NULL),
				  "signal::realize", magnifier_realize, NULL,
				  "signal::destroy", magnifier_exit, NULL,
				  NULL);
	gtk_window_set_decorated (GTK_WINDOW (magnifier->priv->w), FALSE);
	magnifier->priv->canvas = gtk_fixed_new ();
	gtk_container_add (GTK_CONTAINER (magnifier->priv->w),
			   magnifier->priv->canvas);
	magnifier->priv->root = NULL;
}

static void
magnifier_init (Magnifier *magnifier)
{
	int ret;
	
	magnifier->priv = g_new0 (MagnifierPrivate, 1);
	magnifier_properties_init (magnifier);
	magnifier->zoom_regions = NULL;
	magnifier->source_screen_num = 0;
	magnifier->target_screen_num = 0;
	magnifier->cursor_size = 0;
	magnifier->cursor_scale_factor = 1.0F;
	magnifier->cursor_color = 0xFF000000;
	magnifier->crosswire_size = 1;
	magnifier->crosswire_color = 0;
	magnifier->crosswire_clip = FALSE;
	magnifier->cursor_hotspot.x = 0;
	magnifier->cursor_hotspot.y = 0;
	magnifier->priv->cursor = NULL;
	magnifier->priv->w = NULL;
	magnifier_init_window (magnifier);
	
	ret = bonobo_activation_active_server_register (
		MAGNIFIER_OAFIID, BONOBO_OBJREF (magnifier));

	magnifier->cursor_set = "default";
}

GdkDrawable *
magnifier_get_cursor (Magnifier *magnifier)
{
	if (!magnifier->priv->cursor)
		magnifier_init_cursor (magnifier);

	return magnifier->priv->cursor;
}

Magnifier *
magnifier_new (void)
{
	return g_object_new (magnifier_get_type(), NULL);
}

BONOBO_TYPE_FUNC_FULL (Magnifier, 
		       GNOME_Magnifier_Magnifier,
		       BONOBO_TYPE_OBJECT,
		       magnifier);
