/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * PDF viewer Bonobo container.
 *
 * Author:
 *   Michael Meeks <michael@ximian.com>
 */
#include <aconf.h>
#include "gpdf-window.h"
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <gnome.h>
#include <bonobo-activation/bonobo-activation.h>
#include <libbonoboui.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

static gboolean
create_window (gpointer data)
{
	GtkWidget *win;
	const gchar *fname;

	win = gpdf_window_new ();
	fname = (const gchar *)data;

	gtk_widget_show (win);

	if (fname)
		gpdf_window_open (GPDF_WINDOW (win), fname);

	return FALSE;
}

static gboolean
handle_cmdline_args (gpointer data)
{
	poptContext ctx;
	const gchar **startup_files;

	ctx = data;
	startup_files = poptGetArgs (ctx);

	if (startup_files) {
		gint i;
		for (i = 0; startup_files [i]; ++i)
			g_idle_add (create_window, 
				    (gchar *) startup_files [i]);
	} else {
		g_idle_add (create_window, NULL);
	}
	
	poptFreeContext (ctx);

	return FALSE;
}

int
main (int argc, char **argv)
{
	GnomeProgram *program;
	GnomeClient *client;
	GValue value = { 0 };
	poptContext ctx;

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	program = gnome_program_init ("gpdf", VERSION,
				      LIBGNOMEUI_MODULE, argc, argv,
				      GNOME_PARAM_APP_DATADIR, DATADIR,
				      NULL);

	if (bonobo_ui_init ("GNOME PDF Viewer", VERSION, &argc, argv) == FALSE)
		g_error (_("Could not initialize Bonobo!\n"));

	client = gnome_master_client ();
	
	g_value_init (&value, G_TYPE_POINTER);
	g_object_get_property (G_OBJECT (program), GNOME_PARAM_POPT_CONTEXT,
			       &value);
	ctx = g_value_get_pointer (&value);
	g_value_unset (&value);

	gtk_idle_add (handle_cmdline_args, ctx);

	bonobo_main ();

	return 0;
}


/* GPdfWindow class */
#define GPDF_CONTROL_IID    "OAFIID:GNOME_PDF_Control"

#define PARENT_TYPE BONOBO_TYPE_WINDOW

GNOME_CLASS_BOILERPLATE (GPdfWindow, gpdf_window, BonoboWindow, PARENT_TYPE);

struct _GPdfWindowPrivate {
	BonoboControlFrame *ctrl_frame;

	GtkWidget *ctrl_widget;
};

/* The list of all open windows */
static GList *window_list;

/* Drag target types */
enum {
	TARGET_URI_LIST
};

#define GPDF_IS_NON_NULL_WINDOW(obj) \
(((obj) != NULL) && (GPDF_IS_WINDOW ((obj))))


static gboolean
gw_has_contents (GPdfWindow *gpdf_window)
{
	return (bonobo_control_frame_get_control (
			gpdf_window->priv->ctrl_frame) != CORBA_OBJECT_NIL);
}


static Bonobo_Control
gw_activate_control (void)
{
	Bonobo_Control control;
	CORBA_Environment ev;

	CORBA_exception_init (&ev);

	control = bonobo_get_object (GPDF_CONTROL_IID, "Bonobo/Control", &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("%s", bonobo_exception_get_text (&ev));
		control = CORBA_OBJECT_NIL;
	}

	CORBA_exception_free (&ev);

	return control;
}

static void
gw_add_control_to_ui (GPdfWindow *gpdf_window, Bonobo_Control control)
{
	GPdfWindowPrivate *priv;
	CORBA_Environment ev;

	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	priv = gpdf_window->priv;
	CORBA_exception_init (&ev);

	/* bind and view new control widget */
	bonobo_control_frame_bind_to_control (priv->ctrl_frame, control, &ev);
	bonobo_control_frame_control_activate (priv->ctrl_frame);
	if (control != CORBA_OBJECT_NIL && priv->ctrl_widget == NULL) {
		priv->ctrl_widget = 
			bonobo_control_frame_get_widget (priv->ctrl_frame);

		g_assert (priv->ctrl_widget != NULL);

		gtk_container_add (GTK_CONTAINER (gpdf_window->slot), 
				   priv->ctrl_widget);
		gtk_widget_show (priv->ctrl_widget);
	} else if (control == CORBA_OBJECT_NIL && priv->ctrl_widget != NULL) {
		gtk_container_remove (GTK_CONTAINER (gpdf_window->slot),
				      priv->ctrl_widget);
		priv->ctrl_widget = NULL;
	}			

	CORBA_exception_init (&ev);
}

static gboolean
gw_control_load_pdf (GPdfWindow *gpdf_window, const char *name)
{
	GPdfWindowPrivate *priv;
	CORBA_Environment ev;
	Bonobo_Control control;	
	Bonobo_PersistStream persist;
	Bonobo_Stream stream;
	char *uri;
	char *moniker_str;
	gboolean success = FALSE;

	g_return_val_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window), FALSE);

	priv = gpdf_window->priv;
	g_return_val_if_fail (priv->ctrl_frame != NULL, FALSE);
	
	control = bonobo_control_frame_get_control (priv->ctrl_frame);
	CORBA_exception_init (&ev);
	persist = Bonobo_Unknown_queryInterface (
		control, "IDL:Bonobo/PersistStream:1.0", &ev);
	

	if (BONOBO_EX (&ev) || persist == CORBA_OBJECT_NIL) {
		gnome_error_dialog (
			"Panic: component doesn't implement PersistStream.");
		goto error_persist;
	}

	uri = g_strconcat ("file:", name, NULL);
	moniker_str = gnome_vfs_escape_set (uri, "!");	
	stream = bonobo_get_object (moniker_str, "IDL:Bonobo/Stream:1.0", &ev);
	g_free (moniker_str);
	g_free (uri);

	if (stream == CORBA_OBJECT_NIL) {
		char *err = g_strdup_printf (_("Could not open %s"), name);
		gnome_error_dialog /* FIXME _parented */ (err);
		g_free (err);
		goto error_stream;
	}

	Bonobo_PersistStream_load (persist, stream, "application/pdf", &ev);
	success = TRUE;

	bonobo_object_release_unref (stream, &ev);
error_stream:
	bonobo_object_release_unref (persist, &ev);
error_persist:
	CORBA_exception_free (&ev);

	return success;
}


gboolean
gpdf_window_open (GPdfWindow *gpdf_window, const char *name)
{
	Bonobo_Control control;

	g_return_val_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window), FALSE);
	g_return_val_if_fail (!gw_has_contents (gpdf_window), FALSE);

	control = gw_activate_control ();
	if (control == CORBA_OBJECT_NIL)
		return FALSE;

	gw_add_control_to_ui (gpdf_window, control);
	bonobo_object_release_unref (control, NULL);

	return gw_control_load_pdf (gpdf_window, name);
}


static void
gw_drag_data_received (GtkWidget *widget, GdkDragContext *context,
		       gint x, gint y, GtkSelectionData *selection_data,
		       guint info, guint time)
{
	GList *uri_list;
	GList *tmp_list;
	GPdfWindow *gpdf_window = GPDF_WINDOW (widget);
	gboolean need_new_window;

	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));
	g_return_if_fail (info == TARGET_URI_LIST);

	uri_list = gnome_vfs_uri_list_parse ((gchar *)selection_data->data);

	need_new_window = gw_has_contents (gpdf_window);
	for (tmp_list = uri_list; tmp_list; tmp_list = tmp_list->next) {
		GtkWidget *new_window;
		const gchar *fname = 
			gnome_vfs_uri_get_path ((GnomeVFSURI *)tmp_list->data);

		g_assert (fname != NULL);
		
		if (need_new_window)
			new_window = gpdf_window_new ();
		else
			new_window = GTK_WIDGET (gpdf_window);

		if (gpdf_window_open (GPDF_WINDOW (new_window), fname)) {
			gtk_widget_show_now (new_window);
			need_new_window = TRUE;
		} else {
			if (new_window != GTK_WIDGET (gpdf_window))
				gtk_widget_destroy (new_window);
		}
	}

	gnome_vfs_uri_list_unref (uri_list);
}


void
gpdf_window_close (GPdfWindow *gpdf_window)
{
	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	gtk_widget_destroy (GTK_WIDGET (gpdf_window));

	if (window_list == NULL)
		bonobo_main_quit ();
}

static void
gw_close_all (void)
{
	while (window_list != NULL)
		gpdf_window_close (GPDF_WINDOW (window_list->data));
}



static void
file_dialog_ok (GtkWidget *widget, gboolean *dialog_result)
{
	*dialog_result = TRUE;
	gtk_main_quit ();
}

static guint
file_dialog_delete_event (GtkWidget *widget, GdkEventAny *event)
{
	gtk_main_quit ();
	return TRUE;
}

static gboolean
gw_ask_for_filename (GPdfWindow *gpdf_window, gchar **filename)
{
	GtkFileSelection *fsel;
	gboolean accepted = FALSE;

	g_return_val_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window), FALSE);
	g_return_val_if_fail (*filename == NULL, FALSE);

	fsel = GTK_FILE_SELECTION (gtk_file_selection_new (_("Load file")));
	gtk_window_set_modal (GTK_WINDOW (fsel), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (fsel),
				      GTK_WINDOW (gpdf_window));

	/* Connect the signals for Ok and Cancel */
	g_signal_connect (G_OBJECT (fsel->ok_button), "clicked",
			  G_CALLBACK (file_dialog_ok), &accepted);
	g_signal_connect (G_OBJECT (fsel->cancel_button), "clicked",
			  G_CALLBACK (gtk_main_quit), NULL);

	gtk_window_set_position (GTK_WINDOW (fsel), GTK_WIN_POS_MOUSE);

	/* Make sure that we quit the main loop if the window is destroyed */
	g_signal_connect (G_OBJECT (fsel), "delete_event",
			  G_CALLBACK (file_dialog_delete_event), NULL);

	/* Run the dialog */
	gtk_widget_show (GTK_WIDGET (fsel));
	gtk_grab_add (GTK_WIDGET (fsel));
	gtk_main ();

	if (accepted)
		*filename = g_strdup (gtk_file_selection_get_filename (fsel));
	
	gtk_widget_destroy (GTK_WIDGET (fsel));

	return accepted;
}

static void
gw_open_dialog (GPdfWindow *gpdf_window)
{
	gchar *filename = NULL;

	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	if (gw_ask_for_filename (gpdf_window, &filename)) {
		g_assert (filename != NULL);

		if (filename [strlen (filename) - 1] != '/') {
			GtkWidget *new_window;

			if (gw_has_contents (gpdf_window))
				new_window = gpdf_window_new ();
			else
				new_window = GTK_WIDGET (gpdf_window);

			if (gpdf_window_open (GPDF_WINDOW (new_window), 
					      filename)) {
				gtk_widget_show_now (new_window);
			} else {
				if (new_window != GTK_WIDGET (gpdf_window))
					gtk_widget_destroy (new_window);
			}
		} else {
			GtkWidget *dialog = gnome_message_box_new (
				_("Can't open a directory"),
				GNOME_MESSAGE_BOX_ERROR,
				GNOME_STOCK_BUTTON_OK,
				NULL);
			gnome_dialog_set_parent (GNOME_DIALOG (dialog),
						 GTK_WINDOW (gpdf_window));
			gnome_dialog_run (GNOME_DIALOG (dialog));
		}			
		g_free (filename);
	}
}



static void
verb_FileOpen_cb (BonoboUIComponent *uic, gpointer user_data,
		  const char *cname)
{
	gw_open_dialog (GPDF_WINDOW (user_data));
}

static void
verb_FileClose_cb (BonoboUIComponent *uic, gpointer user_data,
		   const char *cname)
{
	gpdf_window_close (GPDF_WINDOW (user_data));
}

static void
verb_FileExit_cb (BonoboUIComponent *uic, gpointer user_data,
		  const char *cname)
{
	gw_close_all ();
}

/* Brings attention to a window by raising it and giving it focus */
static void
raise_and_focus (GtkWidget *widget)
{
	g_assert (GTK_WIDGET_REALIZED (widget));
	gdk_window_show (widget->window);
	gtk_widget_grab_focus (widget);
}

static void
verb_HelpAbout_cb (BonoboUIComponent *uic, gpointer user_data,
		   const char *cname)
{
	static GtkWidget *about;

	static const gchar *authors[] = {
		N_("Derek B. Noonburg, Xpdf author."),
		N_("Martin Kretzschmar, GNOME port maintainer."),
		N_("Michael Meeks, GNOME port creator."),
		N_("Miguel de Icaza."),
		N_("Nat Friedman."),
		N_("Ravi Pratap."),
		NULL
	};

	static const gchar *translators;

	if (!about) {
#ifdef ENABLE_NLS
		int i;
		
		for (i = 0; authors[i] != NULL; i++)
			authors [i] = _(authors [i]);
#endif

		translators = _("translator_credits-PLEASE_ADD_YOURSELF_HERE");
		
		about = gnome_about_new (
			_("GNOME PDF Viewer"),
			VERSION,
			_("Copyright (C) 1996-2002 Glyph & Cog, LLC and authors"),
			_("A PDF viewer based on Xpdf"),
			authors,
			NULL,
			(strcmp (translators,
				 "translator_credits-PLEASE_ADD_YOURSELF_HERE")
			 ? translators : NULL),
			NULL);
	
		g_signal_connect (about, "destroy",
				  G_CALLBACK (gtk_widget_destroyed),
				  &about);
	}

	gtk_widget_show_now (about);
	raise_and_focus (about);
}


static gint
gw_delete_event (GtkWidget *widget, GdkEventAny *event)
{
	gpdf_window_close (GPDF_WINDOW (widget));
	return TRUE;
}

static void
gw_destroy (GtkObject *object)
{
	GPdfWindow *gpdf_window = GPDF_WINDOW (object);
	GPdfWindowPrivate *priv;

	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	priv = gpdf_window->priv;
	window_list = g_list_remove (window_list, gpdf_window);

	if (priv->ctrl_frame != NULL) {
		bonobo_object_unref (BONOBO_OBJECT (priv->ctrl_frame));
		priv->ctrl_frame = NULL;
	}
		
	GNOME_CALL_PARENT (GTK_OBJECT_CLASS, destroy, (object));
}

static void
gw_finalize (GObject *object)
{
	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (object));

	g_free (GPDF_WINDOW (object)->priv);

	GNOME_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
gpdf_window_class_init (GPdfWindowClass *klass)
{
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	G_OBJECT_CLASS (klass)->finalize = gw_finalize;
	GTK_OBJECT_CLASS (klass)->destroy = gw_destroy;

	widget_class->delete_event = gw_delete_event;
	widget_class->drag_data_received = gw_drag_data_received;
}

static void
gpdf_window_instance_init (GPdfWindow *gpdf_window)
{
	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));
	
	gpdf_window->priv = g_new0 (GPdfWindowPrivate, 1);
	window_list = g_list_prepend (window_list, gpdf_window);
}


/* Construction */

static void
gw_setup_window (GPdfWindow *gpdf_window)
{
	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	bonobo_window_set_name (BONOBO_WINDOW (gpdf_window), "gpdf");
	gtk_window_set_title (GTK_WINDOW (gpdf_window),
			      _("GNOME PDF Viewer"));

	gtk_window_set_default_size (GTK_WINDOW (gpdf_window), 600, 600);
}

static void
gw_setup_dnd (GPdfWindow *gpdf_window)
{
	static GtkTargetEntry drag_types[] = {
		{ "text/uri-list", 0, TARGET_URI_LIST },
	};
	static gint n_drag_types =
		sizeof (drag_types) / sizeof (drag_types [0]);

	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	gtk_drag_dest_set (GTK_WIDGET (gpdf_window),
			   GTK_DEST_DEFAULT_ALL,
			   drag_types, n_drag_types, GDK_ACTION_COPY);
}

static void
gw_setup_local_contents (GPdfWindow *gpdf_window)
{
	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	gpdf_window->slot = gtk_event_box_new ();
	gtk_widget_show (gpdf_window->slot);
	bonobo_window_set_contents (BONOBO_WINDOW (gpdf_window),
				    gpdf_window->slot);
	gtk_widget_show_all (gpdf_window->slot);
}

static void
gw_setup_control_frame (GPdfWindow *gpdf_window)
{
	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));      

	gpdf_window->priv->ctrl_frame = bonobo_control_frame_new (
		bonobo_ui_component_get_container (gpdf_window->ui_component));
	bonobo_control_frame_set_autoactivate (gpdf_window->priv->ctrl_frame,
					       FALSE);
}
		
/*
 * The menus.
 */
static BonoboUIVerb gw_verbs [] = {
	BONOBO_UI_VERB ("FileOpen",  verb_FileOpen_cb),
	BONOBO_UI_VERB ("FileClose", verb_FileClose_cb),
	BONOBO_UI_VERB ("FileExit",  verb_FileExit_cb),

	BONOBO_UI_VERB ("HelpAbout", verb_HelpAbout_cb),

	BONOBO_UI_VERB_END
};

static void
gw_setup_toplevel_ui (GPdfWindow *gpdf_window)
{
	BonoboUIContainer *ui_container;

	g_return_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window));

	ui_container = bonobo_window_get_ui_container (
		BONOBO_WINDOW (gpdf_window));

	gpdf_window->ui_component = bonobo_ui_component_new ("gpdf");
	bonobo_ui_component_set_container (gpdf_window->ui_component,
					   BONOBO_OBJREF (ui_container),
					   NULL);
	bonobo_ui_component_add_verb_list_with_data (gpdf_window->ui_component,
						     gw_verbs, gpdf_window);
	bonobo_ui_util_set_ui (gpdf_window->ui_component, 
			       DATADIR, "gpdf-window-ui.xml", "GPDF",
			       NULL);
}

GPdfWindow *gpdf_window_construct (GPdfWindow *gpdf_window)
{
	g_return_val_if_fail (GPDF_IS_NON_NULL_WINDOW (gpdf_window), NULL);

	gw_setup_window (gpdf_window);
	gw_setup_dnd (gpdf_window);
	gw_setup_local_contents (gpdf_window);
	gw_setup_toplevel_ui (gpdf_window);
	gw_setup_control_frame (gpdf_window);

	return gpdf_window;
}

GtkWidget *gpdf_window_new (void)
{
	GPdfWindow *gpdf_window = GPDF_WINDOW (gtk_type_new (GPDF_TYPE_WINDOW));

	return GTK_WIDGET (gpdf_window_construct (gpdf_window));
}
