/* Eye Of Gnome - abstract item factory
 *
 * Copyright (C) 2000-2001 The Free Software Foundation
 *
 * Author: Federico Mena-Quintero <federico@gnu.org>
 *         Jens Finke <jens@gnome.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gtk/gtksignal.h>
#include <libgnome/gnome-macros.h>

#include "eog-item-factory.h"
#include "eog-collection-marshal.h"



/* Signal IDs */
enum {
	CREATE_ITEM,
	UPDATE_ITEM,
	GET_ITEM_SIZE,
	CONFIGURATION_CHANGED,
	LAST_SIGNAL
};

static void eog_item_factory_class_init (EogItemFactoryClass *class);
static void eog_item_factory_instance_init (EogItemFactory *obj);

static guint ei_factory_signals[LAST_SIGNAL];

GNOME_CLASS_BOILERPLATE (EogItemFactory, eog_item_factory,
			 GObject, G_TYPE_OBJECT);


/* Class initialization function for the abstract list item factory */
static void
eog_item_factory_class_init (EogItemFactoryClass *class)
{
	GObjectClass *object_class;

	object_class = (GObjectClass *) class;

	ei_factory_signals[CREATE_ITEM] =
		g_signal_new ("create_item",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (EogItemFactoryClass, create_item),
			      NULL, NULL,
			      eog_collection_marshal_OBJECT__OBJECT_UINT,
			      GNOME_TYPE_CANVAS_ITEM, 2,
			      GNOME_TYPE_CANVAS_GROUP,
			      G_TYPE_UINT);
	ei_factory_signals[UPDATE_ITEM] =
		g_signal_new ("update_item",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (EogItemFactoryClass, update_item),
			      NULL, NULL,
			      eog_collection_marshal_VOID__OBJECT_OBJECT_INT,
			      G_TYPE_NONE, 3,
			      EOG_TYPE_COLLECTION_MODEL,
			      GNOME_TYPE_CANVAS_ITEM,
			      G_TYPE_INT);
	ei_factory_signals[GET_ITEM_SIZE] =
		g_signal_new ("get_item_size",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (EogItemFactoryClass, get_item_size),
			      NULL, NULL,
			      eog_collection_marshal_VOID__POINTER_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_POINTER,
			      G_TYPE_POINTER);
	ei_factory_signals[CONFIGURATION_CHANGED] =
		g_signal_new ("configuration_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (EogItemFactoryClass, configuration_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
}

static void
eog_item_factory_instance_init (EogItemFactory *factory)
{
}

/* Exported functions */

/**
 * eog_item_factory_create_item:
 * @factory: A list item factory.
 * @parent: Canvas group to act as the item's parent.
 *
 * Makes a list item factory create an empty item.
 *
 * Return value: An canvas item representing an empty list item.
 **/
GnomeCanvasItem *
eog_item_factory_create_item (EogItemFactory *factory, GnomeCanvasGroup *parent, 
			      guint id)
{
	GnomeCanvasItem *retval;

	g_return_val_if_fail (factory != NULL, NULL);
	g_return_val_if_fail (EOG_IS_ITEM_FACTORY (factory), NULL);
	g_return_val_if_fail (parent != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_CANVAS_GROUP (parent), NULL);

	retval = NULL;
	g_signal_emit_by_name (G_OBJECT (factory), 
			       "create_item",
			       parent, id, &retval);
	return retval;
}

/**
 * eog_item_factory_configure_item:
 * @factory: A list item factory.
 * @model: A list model.
 * @item: A list item created by this factory.

 * Requests that a list item generated by a list item factory be configured for
 * a particular element in the list model, as well as with a certain selection
 * and focus state.
 **/
void
eog_item_factory_update_item (EogItemFactory *factory, 
			      EogCollectionModel *model, 
			      GnomeCanvasItem *item,
			      EogItemUpdateHint hint)
{
	g_return_if_fail (factory != NULL);
	g_return_if_fail (EOG_IS_ITEM_FACTORY (factory));
	g_return_if_fail (item != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_ITEM (item));
	g_return_if_fail (model != NULL);
	g_return_if_fail (EOG_IS_COLLECTION_MODEL (model));

	g_signal_emit_by_name (G_OBJECT (factory), 
			       "update_item",
			       model, item, hint);
}

/**
 * eog_item_factory_get_item_size:
 * @factory: A list item factory.
 * @item: If non-NULL, a list item created by this factory.
 * @model: A list model.
 * @n: Index in the list model of the item to measure.
 * @width: Return value for the item's width.
 * @height: Return value for item's height.
 *
 * Queries the size in pixels of a list item that was or would be created by the
 * specified factory.  If the specified @item is not NULL, then the factory can
 * use it to compute the size more efficiently.  Otherwise, the factory will
 * have to compute the size from scratch.
 **/
void
eog_item_factory_get_item_size (EogItemFactory *factory,
				int *width, int *height)
{
	int w, h;

	g_return_if_fail (factory != NULL);
	g_return_if_fail (EOG_IS_ITEM_FACTORY (factory));

	w = h = 0;
	g_signal_emit_by_name (G_OBJECT (factory), 
			       "get_item_size", 
			       &w, &h);

	if (width)
		*width = w;

	if (height)
		*height = h;
}

void
eog_item_factory_configuration_changed (EogItemFactory *factory)
{
	g_return_if_fail (factory != NULL);
	g_return_if_fail (EOG_IS_ITEM_FACTORY (factory));

	g_signal_emit_by_name (G_OBJECT (factory), "configuration_changed");
}
