/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-utils.c:
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Jose M. Celorio <chema@ximian.com>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc. and Jose M. Celorio
 *  Contains pieces of code from glib, Copyright GLib Team and others 1997-2000.
 *
 */

#define __GPA_UTILS_C__

#include "config.h"
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <libxml/parser.h>
#include "gpa-node-private.h"
#include "gpa-utils.h"
#include "gpa-value.h"
#include "gpa-reference.h"
#include "gpa-list.h"
#include "gpa-key.h"

#define noGPA_QUARK_DEBUG

guchar *
gpa_id_new (const guchar *key)
{
	static gint counter = 0;
	guchar *id;

	/* FIXME: This is not very intelligent (Lauris) */
	id = g_strdup_printf ("%s-%.12d%.6ld%.6d",
			      key,
			      (gint) time (NULL),
			      (long) getpid (),
			      counter++);

	return id;
}

GPANode *
gpa_node_attach (GPANode *parent, GPANode *child)
{
	g_return_val_if_fail (parent != NULL, NULL);
	g_return_val_if_fail (GPA_IS_NODE (parent), NULL);
	g_return_val_if_fail (child != NULL, NULL);
	g_return_val_if_fail (GPA_IS_NODE (child), NULL);
	g_return_val_if_fail (child->parent == NULL, NULL);
	g_return_val_if_fail (child->next == NULL, NULL);

	child->parent = parent;

	return child;
}

GPANode *
gpa_node_attach_ref (GPANode *parent, GPANode *child)
{
	g_return_val_if_fail (parent != NULL, NULL);
	g_return_val_if_fail (GPA_IS_NODE (parent), NULL);
	g_return_val_if_fail (child != NULL, NULL);
	g_return_val_if_fail (GPA_IS_NODE (child), NULL);
	g_return_val_if_fail (child->parent == NULL, NULL);
	g_return_val_if_fail (child->next == NULL, NULL);

	gpa_node_ref (child);

	child->parent = parent;

	return child;
}

GPANode *
gpa_node_detach (GPANode *parent, GPANode *child)
{
	g_return_val_if_fail (parent != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (parent), child);
	g_return_val_if_fail (child != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (child), child);
	g_return_val_if_fail (child->parent == parent, child);

	child->parent = NULL;
	child->next = NULL;

	return NULL;
}

GPANode *
gpa_node_detach_unref (GPANode *parent, GPANode *child)
{
	g_return_val_if_fail (parent != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (parent), child);
	g_return_val_if_fail (child != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (child), child);
	g_return_val_if_fail (child->parent == parent, child);

	child->parent = NULL;
	child->next = NULL;

	gpa_node_unref (child);

	return NULL;
}

GPANode *
gpa_node_detach_next (GPANode *parent, GPANode *child)
{
	GPANode *next;

	g_return_val_if_fail (parent != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (parent), child);
	g_return_val_if_fail (child != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (child), child);
	g_return_val_if_fail (child->parent == parent, child);

	next = child->next;
	child->parent = NULL;
	child->next = NULL;

	return next;
}

GPANode *
gpa_node_detach_unref_next (GPANode *parent, GPANode *child)
{
	GPANode *next;

	g_return_val_if_fail (parent != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (parent), child);
	g_return_val_if_fail (child != NULL, child);
	g_return_val_if_fail (GPA_IS_NODE (child), child);
	g_return_val_if_fail (child->parent == parent, child);

	next = child->next;
	child->parent = NULL;
	child->next = NULL;

	gpa_node_unref (child);

	return next;
}

/**
 * gpa_node_lookup_check:
 * @path: 
 * @starts_with: 
 *
 * Checks if @path contains @start with, and if found it returns
 * a pointer to where the rest of the path (the part that didn't matched)
 * starts.
 *
 * This is the check path and stars_with part of _lookup_helper
 *
 * For example: given a path "A.B.C.D" for a starts_with of "A.B", "C.D" is returned
 * 
 * Return Value: A pointer to the rest of path that is not @starts_with. NULL if
 *               @path does not start with @starts_with
 **/
const guchar *
gpa_node_lookup_check (const guchar *path, const guchar *starts_with)
{
	gint len;

	g_return_val_if_fail (path != NULL, NULL);
	g_return_val_if_fail (*path != '\0', NULL);
	g_return_val_if_fail (starts_with != NULL, NULL);
	g_return_val_if_fail (*starts_with != '\0', NULL);

	len = strlen (starts_with);

	if (!strncmp (path, starts_with, len)) {
		if (!path[len])
			return path + len;
		if (path[len] == '.')
			return path + len + 1;
	}

	return NULL;
}



/**
 * gpa_node_lookup_helper:
 * @node: The node where the search starts
 * @path: The path we are looking for
 * @starts_with: Only if the path starts with this string the lookup is performed this 
 *               allow us to simplify the calling code by doing the compare over here
 * @child: Return the child over here.
 *
 * For example. Say we are searing from the GPAConfig node for "Printer.Name"
 * if you look at gpa_config_lookup we will first try to look for it in the Globals
 * node, so this function will be called with path="Printer.Name" and starts_with="Globals"
 * because the paths don't start with the same string it returns false. Then gpa_config_lookup
 * tries the same search with path="Printer.Name" and starts_with="Printer", the search
 * continues after we check that we have not found the complete path, we call gpa_node_lookup
 * with path="Name" (from path + len + 1). Since name is of type GPAKey, the gpa_key_lookup is
 * preformed with path= "Name".
 * 
 * Return Value: TRUE if the complete path was found. The child found is returned in @child
 **/
gboolean
gpa_node_lookup_helper (GPANode *node, const guchar *path,
			const guchar *starts_with,
			GPANode **child)
{
	gint len;

	g_return_val_if_fail (*child == NULL, TRUE);
	g_return_val_if_fail (node != NULL, TRUE);
	g_return_val_if_fail (GPA_IS_NODE (node), TRUE);
	g_return_val_if_fail (path != NULL, TRUE);
	g_return_val_if_fail (*path != '\0', TRUE);
	g_return_val_if_fail (starts_with != NULL, TRUE);
	g_return_val_if_fail (*starts_with != '\0', TRUE);

	len = strlen (starts_with);

	/* Check if @path starts with @starts_with if not, this
	 * is not the leaf we are looking for.
	 */
	if (strncmp (path, starts_with, len) != 0) {
		return FALSE;
	}

	/* If we are at the end of path, then we have a match already */
	if (!path[len]) {
		gpa_node_ref (node);
		*child = node;
		return TRUE;
	} else {
		/* We still have part of the path to walk. But check for the
		 * case where we Some.Leaf should not match Some.Leaflet since
		 * we base the compares on len, which comes from @start with
		 */
		if (path[len] != '.') {
			return FALSE;
		}
		
		/* Recurse to look for the child on the next segment of the path */
		*child = gpa_node_lookup (node, path + len + 1);

		return TRUE;
	}

	return FALSE;
}

/* XML helpers */

xmlChar *
gpa_xml_node_get_name (xmlNodePtr node)
{
	xmlNodePtr child;

	g_return_val_if_fail (node != NULL, NULL);

	for (child = node->xmlChildrenNode; child != NULL; child = child->next) {
		if (!strcmp (child->name, "Name")) {
			return xmlNodeGetContent (child);
		}
	}

	return NULL;
}

xmlNodePtr
gpa_xml_node_get_child (xmlNodePtr node, const guchar *name)
{
	xmlNodePtr child;

	g_return_val_if_fail (node != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);

	for (child = node->xmlChildrenNode; child != NULL; child = child->next) {
		if (!strcmp (child->name, name)) return child;
	}

	return NULL;
}

/* Private quarks */

#define GPA_QUARK_BLOCK_SIZE 128
static GHashTable *qdict = NULL;
static const guchar **qarray = NULL;
static GQuark qseq = 0;

static GPAQuark
gpa_quark_new (const guchar *string)
{
	if ((qseq % GPA_QUARK_BLOCK_SIZE) == 0) {
		qarray = g_renew (const guchar *, qarray, qseq + GPA_QUARK_BLOCK_SIZE);
#ifdef GPA_QUARK_DEBUG
		g_print ("GPAQuark: Increasing buffer, now %d\n", qseq + GPA_QUARK_BLOCK_SIZE);
#endif
	}

	qarray[qseq++] = string;

	g_hash_table_insert (qdict, (gpointer) string, GUINT_TO_POINTER (qseq));

#ifdef GPA_QUARK_DEBUG
	g_print ("GPAQuark: New quark %d:%s\n", qseq, string);
#endif

	return qseq;
}

GPAQuark
gpa_quark_try_string (const guchar *string)
{
	GPAQuark q;

	g_return_val_if_fail (string != NULL, 0);

	q = (qdict) ? GPOINTER_TO_INT (g_hash_table_lookup (qdict, string)) : 0;

	return q;
}

GPAQuark
gpa_quark_from_string (const guchar *string)
{
	GPAQuark q;

	g_return_val_if_fail (string != NULL, 0);

	if (!qdict) qdict = g_hash_table_new (g_str_hash, g_str_equal);

	q = GPOINTER_TO_INT (g_hash_table_lookup (qdict, string));

	if (!q) q = gpa_quark_new (g_strdup (string));

	return q;
}

GPAQuark
gpa_quark_from_static_string (const guchar *string)
{
	GPAQuark q;

	g_return_val_if_fail (string != NULL, 0);

	if (!qdict) qdict = g_hash_table_new (g_str_hash, g_str_equal);

	q = GPOINTER_TO_INT (g_hash_table_lookup (qdict, string));

	if (!q) q = gpa_quark_new (string);

	return q;
}

const guchar *
gpa_quark_to_string (GPAQuark quark)
{
	if ((quark > 0) && (quark <= qseq)) return qarray[quark - 1];

	return NULL;
}

/**
 * gpa_utils_dump_tree_with_level:
 * @node: The node to dump its contents
 * @level: How deep we are in the tree, used so that we know how many spaces to print
 *         so that it really looks like a tree
 * @follow_references: How deep in the tree do we dump info for the refrences
 * 
 * Recursively prints a node and it childs.
 **/
static void
gpa_utils_dump_tree_with_level (GPANode *node, gint level, gint follow_references)
{
	GPANode *previous_child = NULL;
	GPANode *child;
	int i;

	if (level > 20) {
		g_error ("Level too deep. Aborting\n");
	}
	
	g_print ("[%2d]", level);
	/* Print this leave indentation */
	for (i = 0; i < level; i++) {
		g_print ("   ");
	}

	/* Print the object itself */
	g_print ("%s [%s] (%d)", gpa_node_id (node),
		 G_OBJECT_TYPE_NAME (node), GPOINTER_TO_INT (node));

	if (strcmp (G_OBJECT_TYPE_NAME (node), "GPAReference") == 0) {
		GPANode *tmp = GPA_REFERENCE (node)->ref;
		g_print ("****");
		if (tmp)
			g_print ("     reference to a:%s\n", G_OBJECT_TYPE_NAME (tmp));
		else
			g_print ("     empty reference\n");

		if (level > follow_references)   /* We follow references only on level 0 & 1 so that we */
			return; /* see Global & Printer objects */
	} else {
		if (strcmp (G_OBJECT_TYPE_NAME (node), "GPAKey") == 0)
			g_print (" {%s}", ((GPAKey *) node)->value);
		if (strcmp (G_OBJECT_TYPE_NAME (node), "GPAValue") == 0)
			g_print (" {%s}", ((GPAValue *) node)->value);
		if (strcmp (G_OBJECT_TYPE_NAME (node), "GPAList") == 0) {
			g_print (" {CanHaveDefault:%s}", GPA_LIST (node)->can_have_default ?
				 "Yes" : "No");
		}
		g_print ("\n");
	}

	previous_child = NULL;
	while (TRUE) {
		child = gpa_node_get_child (node, previous_child);
		if (child == node) {
			g_error ("Error: child is the same as parent. Aborting.\n");
		}
		if (!child)
			break;
		previous_child = child;
		gpa_utils_dump_tree_with_level (child, level + 1, follow_references);
	}
}

/**
 * gpa_utils_dump_tree:
 * @node:
 * @follow_references: How deep in the tree do we follow references
 * 
 * Dump the tree pointed by @node to the console. Used for debuging purposeses
 **/
void
gpa_utils_dump_tree (GPANode *node, gint follow_references)
{
	g_print ("\n"
		 "-------------\n"
		 "Dumping a tree\n\n");

	gpa_utils_dump_tree_with_level (node, 0, follow_references);
	
	g_print ("-------------\n");
}


static gint enter_leave_level = 0;
static gboolean enter_leave_enabled = FALSE;

void
gpa_enter_leave_enable (void)
{
	enter_leave_level = 0;
	enter_leave_enabled = TRUE;
}

void
gpa_enter_leave_disable (void)
{
	enter_leave_enabled = FALSE;
}

void
gpa_enter_real (gint line, const gchar *file, const gchar *func, const gchar *format, ...)
{
	gchar *string;
	va_list args;
	gint i;

	if (!enter_leave_enabled) {
		enter_leave_level++;
		return;
	}
	
	g_print (".");
	
	for (i = 0; i <= enter_leave_level; i++)
		g_print ("   ");

	va_start (args, format);
	string = g_strdup_vprintf (format, args);
	g_print (string);
	va_end (args);
	g_free (string);

	g_print ("                 "
		 "                 "
		 "[%s:%s]\n", func, file);

	enter_leave_level++;
}

void
gpa_leave_real (gint line, const gchar *file, const gchar *func, const gchar *format, ...)
{
	gchar *string;
	va_list args;
	gint i;

	enter_leave_level--;	
	
	
	if (!enter_leave_enabled) {
		return;
	}
	
	g_print (".");
	
	for (i = 0; i <= enter_leave_level; i++)
		g_print ("   ");

	g_print ("DONE:");
	
	va_start (args, format);
	string = g_strdup_vprintf (format, args);
	g_print (string);
	va_end (args);
	g_free (string);

	g_print ("                 "
		 "                 "
		 "[%s:%s]\n", func, file);
}

