#include <config.h>

#include <string.h>
#include <gtk/gtkobject.h>
#include <gtk/gtksignal.h>
#include <gtk/gtksocket.h>
#include <gtk/gtkmain.h>

#include "gstvideowidget.h"

static void gst_video_widget_class_init		(GstVideoWidgetClass *klass);
static void gst_video_widget_init		(GstVideoWidget *vw);
static void gst_video_widget_set_property	(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void gst_video_widget_get_property	(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static void gst_video_widget_realize		(GtkWidget *vw);
static gint gst_video_widget_expose		(GtkWidget *widget, GdkEventExpose *event);

static void gst_video_widget_frame_displayed	(GstElement *element, GstVideoWidget *vw);

/* signals and args */
enum {
	LAST_SIGNAL
};

enum {
	ARG_0,
	ARG_SCALE_FACTOR,
};

static GtkObject *parent_class = NULL;
static guint gst_video_widget_signals[LAST_SIGNAL] = {0};

GtkType
gst_video_widget_get_type (void)
{
	static GtkType vw_type = 0;

	if (!vw_type) {
		static const GtkTypeInfo vw_info = {
			"GstVideoWidget",
			sizeof (GstVideoWidget),
			sizeof (GstVideoWidgetClass),
			(GtkClassInitFunc) gst_video_widget_class_init,
			(GtkObjectInitFunc) gst_video_widget_init,
			NULL,
			NULL,
			(GtkClassInitFunc)NULL,
		};
		vw_type = gtk_type_unique (gtk_aspect_frame_get_type (), &vw_info);
	}
	return vw_type;
}

static void
gst_video_widget_class_init (GstVideoWidgetClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	gobject_class->set_property = gst_video_widget_set_property;
	gobject_class->get_property = gst_video_widget_get_property;

	g_object_class_install_property (gobject_class, ARG_SCALE_FACTOR,
		g_param_spec_float ("scale-factor", "scale factor", "size the video should be scaled to",
		                    0, G_MAXFLOAT / G_MAXINT, 1, G_PARAM_READWRITE));

	widget_class->realize = gst_video_widget_realize;
	widget_class->expose_event = gst_video_widget_expose;

}

static void
gst_video_widget_init (GstVideoWidget *vw)
{
	vw->source_width = 0;
	vw->source_height = 0;
	vw->scale_factor = 1.0;
	vw->video_socket = gtk_socket_new ();
	gtk_container_add (GTK_CONTAINER (vw), vw->video_socket);
}

GtkWidget *
gst_video_widget_new ()
{
	GstVideoWidget *widget = g_object_new (GST_TYPE_VIDEO_WIDGET, NULL);

	return GTK_WIDGET(widget);
}

static void
gst_video_widget_realize (GtkWidget *widget)
{
	GstVideoWidget *vw = GST_VIDEO_WIDGET (widget);

	g_return_if_fail (vw != NULL);

	if (GTK_WIDGET_CLASS (parent_class)->realize)
		GTK_WIDGET_CLASS (parent_class)->realize (widget);
}

static gint
gst_video_widget_expose(GtkWidget *widget, GdkEventExpose *event)
{

	g_return_val_if_fail (event != NULL, FALSE);
		
	if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget)) {
		gdk_draw_rectangle (widget->window, widget->style->black_gc, TRUE, 
		                    event->area.x, event->area.y, event->area.width, event->area.height);
	}
	return FALSE;
}

static void
gst_video_widget_resize_request (GstVideoWidget *vw)
{
	gint width, height;
	gfloat temp;

	gdk_threads_enter ();

	temp = (vw->scale_factor * vw->source_width + 0.5);
	width = (gint) temp > G_MAXINT ? G_MAXINT : (gint) temp;
	temp = (vw->scale_factor * vw->source_height + 0.5);
	height = (gint) temp > G_MAXINT ? G_MAXINT : (gint) temp;
	/* don't make us want to be bigger than the screen */
	if (width > gdk_screen_width()) {
		height = height * gdk_screen_width() / width;
		width = gdk_screen_width();
	}
	if (height > gdk_screen_height()) {
		width = width * gdk_screen_height() / height;
		height = gdk_screen_height();
	}
	gtk_widget_set_size_request (vw->video_socket, width, height);
	gdk_threads_leave ();
}

void
gst_video_widget_set_source_size (GstVideoWidget *vw, gint width, gint height)
{
	vw->source_width = width;
	vw->source_height = height;
	gst_video_widget_resize_request(vw);
}

void
gst_video_widget_reset_socket(GstVideoWidget *vw)
{
	if (vw->video_socket){
		gtk_container_remove (GTK_CONTAINER (vw), vw->video_socket);
	}
	vw->video_socket = gtk_socket_new ();
	gtk_container_add (GTK_CONTAINER (vw), vw->video_socket);
}

void
gst_video_widget_set_xembed_xid(GstVideoWidget *vw, gint xid)
{
	gdk_threads_enter ();

	gtk_widget_realize (vw->video_socket);
	gtk_widget_show (vw->video_socket);

	gtk_socket_add_id (GTK_SOCKET (vw->video_socket), xid);

	xid = vw->xembed_xid;
	gdk_threads_leave ();

}

gint
gst_video_widget_get_xembed_xid(GstVideoWidget *vw)
{
	g_return_val_if_fail (GST_IS_VIDEO_WIDGET (vw), 0);
	return vw->xembed_xid;

}

gint
gst_video_widget_get_source_width (GstVideoWidget *vw)
{
	g_return_val_if_fail (GST_IS_VIDEO_WIDGET (vw), 0);

	return vw->source_width;
}

gint
gst_video_widget_get_source_height (GstVideoWidget *vw)
{
	g_return_val_if_fail (GST_IS_VIDEO_WIDGET (vw), 0);

	return vw->source_height;
}

static void
gst_video_widget_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	GstVideoWidget *vw;

	g_return_if_fail (object != NULL);

	vw = GST_VIDEO_WIDGET (object);

	switch (prop_id) {
	case ARG_SCALE_FACTOR:
		vw->scale_factor = g_value_get_float (value);
		gst_video_widget_resize_request(vw);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
gst_video_widget_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	GstVideoWidget *vw;

	g_return_if_fail (object != NULL);

	vw = GST_VIDEO_WIDGET (object);

	switch (prop_id) {
	case ARG_SCALE_FACTOR:
		g_value_set_float (value, vw->scale_factor);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}
