/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "bookmarks-bonoboui-menu.h"
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-bonobo-extensions.h"
#include "gul-string.h"

#undef DEBUG_BUIM

/**
 * Private data
 */
struct _GbBonoboUIMenuPrivate {
	GbFolder *root;
	gchar *path;
	BonoboUIComponent *uic;

	GHashTable *dirty_bits;
	guint dirty_timeout;

	GHashTable *bookmark_to_bit;
	GHashTable *all_bits;  /* there may be here more items than in bookmark_to_bit, because of
				  aliases to folders. The key is the BuimItem itself */

	GbLocationSource *location_source;
};

typedef struct _BuimItem BuimItem;
struct _BuimItem {
	GbBonoboUIMenu *buim;
	GbBookmark *bm;
	gchar *path;
	gboolean built_submenu : 1;
	gboolean built_children_submenu : 1;
	gboolean dirty : 1;     /* needs to be rebuilt. Can't be destroyed now */
	gboolean destroy : 1;   /* should be destroyed instead of rebuilt */
};

#define REBUILD_TIMEOUT 500

/**
 * Private functions, only availble from this file
 */
static void		gb_bonobo_ui_menu_class_init	(GbBonoboUIMenuClass *klass);
static void		gb_bonobo_ui_menu_init		(GbBonoboUIMenu *e);
static void		gb_bonobo_ui_menu_finalize_impl (GObject *o);
static void		gb_bonobo_ui_menu_hash_to_slist (gpointer key, gpointer data, gpointer user_data);
static void		gb_bonobo_ui_menu_build_submenu	(GbBonoboUIMenu *buim, GbFolder *item, 
							 const gchar *path, BuimItem *bit);
static void		gb_bonobo_ui_menu_build_root (GbBonoboUIMenu *buim, GbFolder *root);
static void		gb_bonobo_ui_menu_build_folder	(GbBonoboUIMenu *buim, GbFolder *item, 
							 const gchar *path, guint index);
static void		gb_bonobo_ui_menu_build_site	(GbBonoboUIMenu *buim, GbSite *item,
							 const gchar *path, guint index);
static void		gb_bonobo_ui_menu_build_childs	(GbBonoboUIMenu *buim, GbFolder *folder);
static gboolean		gb_bonobo_ui_menu_rebuild_timeout_cb (gpointer data);
static void		gb_bonobo_ui_menu_rebuild_submenu_with_timeout (GbBonoboUIMenu *buim,
									BuimItem *bit);
static void		gb_bonobo_ui_menu_rebuild_submenu (GbBonoboUIMenu *buim, BuimItem *bit);
static void		gb_bonobo_ui_menu_cb		(BonoboUIComponent *uic,
							 BuimItem *bit, const char *path);
static void 		gb_bonobo_ui_menu_child_modified_cb (GbFolder *f, GbBookmark *b, 
							     BuimItem *bit);
static void 		gb_bonobo_ui_menu_child_removed_cb (GbFolder *f, GbBookmark *b, int pos,
							    BuimItem *bit);
static void 		gb_bonobo_ui_menu_child_added_cb (GbFolder *f, GbBookmark *b, int pos,
							  BuimItem *bit);

static BuimItem *	buim_item_new			(GbBonoboUIMenu *buim, GbBookmark *b);
static void		buim_item_destroy		(BuimItem *bit, GClosure *closure);


static gpointer g_object_class;

enum GbBonoboUIMenuSignalsEnum {
	GB_BONOBO_UI_MENU_BOOKMARK_ACTIVATED,
	GB_BONOBO_UI_MENU_LAST_SIGNAL
};
static gint GbBonoboUIMenuSignals[GB_BONOBO_UI_MENU_LAST_SIGNAL];

/**
 * BonoboUIMenu object
 */

MAKE_GET_TYPE (gb_bonobo_ui_menu, "GbBonoboUIMenu", GbBonoboUIMenu, gb_bonobo_ui_menu_class_init,
	       gb_bonobo_ui_menu_init, G_TYPE_OBJECT);

static void
gb_bonobo_ui_menu_class_init (GbBonoboUIMenuClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gb_bonobo_ui_menu_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);

	GbBonoboUIMenuSignals[GB_BONOBO_UI_MENU_BOOKMARK_ACTIVATED] = g_signal_new (
		"bookmark-activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GbBonoboUIMenuClass, gb_bonobo_ui_menu_bookmark_activated), 
		NULL, NULL, 
		galeon_marshal_VOID__OBJECT_STRING_INT,
		G_TYPE_NONE, 3, GB_TYPE_BOOKMARK, G_TYPE_STRING, G_TYPE_UINT);
}

static void 
gb_bonobo_ui_menu_init (GbBonoboUIMenu *m)
{
	GbBonoboUIMenuPrivate *p = g_new0 (GbBonoboUIMenuPrivate, 1);

	m->priv = p;
	
}

static void
gb_bonobo_ui_menu_hash_to_slist (gpointer key, gpointer data, gpointer user_data)
{
	GSList **l = user_data;
	*l = g_slist_prepend (*l, data);
}

static void
gb_bonobo_ui_menu_finalize_impl (GObject *o)
{
	GbBonoboUIMenu *buim = GB_BONOBO_UI_MENU (o);
	GbBonoboUIMenuPrivate *p = buim->priv;
	GSList *l = NULL;
	GSList *li;

	gb_bonobo_ui_menu_set_location_source (buim, NULL);

	if (p->dirty_timeout)
	{
		g_source_remove (p->dirty_timeout);
		p->dirty_timeout = 0;
	}

	g_object_unref (G_OBJECT (p->root));
	g_object_unref (G_OBJECT (p->uic));

	/* destroy the bits left. can't use g_hash_table_foreach because buim_item_destroy 
	   modifies the hashtable */
	g_hash_table_foreach (p->all_bits, 
			      gb_bonobo_ui_menu_hash_to_slist, &l);
	for (li = l; li; li = li->next)
	{
		buim_item_destroy (li->data, NULL);
	}
	g_slist_free (l);

	g_hash_table_destroy (p->dirty_bits);
	g_hash_table_destroy (p->bookmark_to_bit);
	g_hash_table_destroy (p->all_bits);
	g_free (p->path);
	g_free (p);

	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

GbBonoboUIMenu *
gb_bonobo_ui_menu_new (GbFolder *root, BonoboUIComponent *uic, const gchar *path)
{
	GbBonoboUIMenu *ret = g_object_new (GB_TYPE_BONOBO_UI_MENU, NULL);
	GbBonoboUIMenuPrivate *p = ret->priv;

	p->root = root;
	g_object_ref (G_OBJECT (root));

	p->uic = uic;
	g_object_ref (G_OBJECT (uic));

	p->path = g_strdup (path);

	p->all_bits = g_hash_table_new (NULL, NULL);
	p->bookmark_to_bit = g_hash_table_new (NULL, NULL);
	p->dirty_bits = g_hash_table_new (NULL, NULL);
	p->dirty_timeout = 0;

	gb_bonobo_ui_menu_build_root (ret, root);

	return ret;
}

static void
gb_bonobo_ui_menu_build_root (GbBonoboUIMenu *buim, GbFolder *root)
{
	BuimItem *bit;

	bit = buim_item_new (buim, GB_BOOKMARK (root));
	bit->path = g_strdup (buim->priv->path);
	bit->built_submenu = FALSE;
	bit->built_children_submenu = FALSE;

	gb_bonobo_ui_menu_rebuild_submenu (buim, bit);
	gb_bonobo_ui_menu_build_childs (buim, root);

	g_signal_connect (root, "child-modified", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_modified_cb), bit);
	g_signal_connect (root, "child-added", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_added_cb), bit);
	g_signal_connect (root, "child-removed", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_removed_cb), bit);
}

static void
gb_bonobo_ui_menu_build_submenu (GbBonoboUIMenu *buim, GbFolder *item, const gchar *path, BuimItem *bit)
{
	guint index = 0;
	GbBookmark *i;

	if (bit->built_submenu)
	{
#ifdef DEBUG_BUIM
		g_print ("tried to rebuild a submenu...\n");
#endif
		return;
	}
	bit->built_submenu = TRUE;

	for (i = item->child; i != NULL; i = i->next)
	{
		if (GB_IS_SITE (i))
		{
			gb_bonobo_ui_menu_build_site (buim, GB_SITE (i), path, index);
		}
		else if (GB_IS_FOLDER (i))
		{
			gb_bonobo_ui_menu_build_folder (buim, GB_FOLDER (i), path, index);
		}
		else if (GB_IS_SEPARATOR (i))
		{
			gul_bonobo_add_menu_separator (buim->priv->uic, path);
		}
		else
		{
			g_warning ("something skipped when building bookmarks menu");
		}
		index++;
	}

}

static void
gb_bonobo_ui_menu_build_folder (GbBonoboUIMenu *buim, GbFolder *item, const gchar *path, guint index)
{
	GbBonoboUIMenuPrivate *p = buim->priv;
	BuimItem *bit;
	gchar *verb;
	gchar *name = gul_string_shorten (GB_BOOKMARK (item)->name, GB_BONOBO_UI_MENU_NAME_MAX_LENGTH);
	GdkPixbuf *pb = gb_bookmark_get_icon (GB_BOOKMARK (item));

	gul_bonobo_add_numbered_submenu (p->uic, path, index, name, pb);
	
	if (pb)
	{
		g_object_unref (G_OBJECT (pb));
	}
	
	bit = buim_item_new (buim, GB_BOOKMARK (item));
	bit->path = gul_bonobo_get_numbered_menu_item_path (p->uic, path, index);
	verb = gul_bonobo_get_numbered_menu_item_command (p->uic, path, index);
	bit->built_submenu = FALSE;
	bit->built_children_submenu = FALSE;
	
	bonobo_ui_component_add_verb_full (p->uic, verb, 
					   g_cclosure_new (
						   G_CALLBACK (gb_bonobo_ui_menu_cb), bit,
						   (GClosureNotify) buim_item_destroy));
	g_free (verb);
	g_free (name);

	g_signal_connect (item, "child-modified", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_modified_cb), bit);
	g_signal_connect (item, "child-added", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_added_cb), bit);
	g_signal_connect (item, "child-removed", 
			  G_CALLBACK (gb_bonobo_ui_menu_child_removed_cb), bit);

}

static void
gb_bonobo_ui_menu_build_site (GbBonoboUIMenu *buim, GbSite *item, const gchar *path, guint index)
{
	GbBonoboUIMenuPrivate *p = buim->priv;
	BuimItem *bit;

	gchar *name = gul_string_shorten (GB_BOOKMARK (item)->name, GB_BONOBO_UI_MENU_NAME_MAX_LENGTH);
	gchar *tip = gul_string_shorten (item->url, GB_BONOBO_UI_MENU_TIP_MAX_LENGTH);
	gchar *verb;
	GdkPixbuf *pb = gb_bookmark_get_icon (GB_BOOKMARK (item));

	gul_bonobo_add_numbered_menu_item (p->uic, path, index, name, pb);
	
	if (pb)
	{
		g_object_unref (G_OBJECT (pb));
	}

	bit = buim_item_new (buim, GB_BOOKMARK (item));
	bit->path = gul_bonobo_get_numbered_menu_item_path (p->uic, path, index);
	verb = gul_bonobo_get_numbered_menu_item_command (p->uic, path, index);
	
	gul_bonobo_set_tip (p->uic, bit->path, tip);
				
	bonobo_ui_component_add_verb_full (p->uic, verb, 
					   g_cclosure_new (
						   G_CALLBACK (gb_bonobo_ui_menu_cb), bit,
						   (GClosureNotify) buim_item_destroy));

	g_free (verb);
	g_free (name);
	g_free (tip);
}

static void
buim_item_destroy (BuimItem *bit, GClosure *closure)
{
#ifdef DEBUG_BUIM
	int t = 
#endif
		g_signal_handlers_disconnect_matched (bit->bm, G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, bit);

#ifdef DEBUG_BUIM
	g_print ("Destroying bit: %x, %s, %d handlers disconnected\n", (guint32) bit, bit->bm->name, t);
#endif

	if (g_hash_table_lookup (bit->buim->priv->bookmark_to_bit, bit->bm) == bit)
	{
		g_hash_table_remove (bit->buim->priv->bookmark_to_bit, bit->bm);
	}

	if (!bit->dirty)
	{
		g_hash_table_remove (bit->buim->priv->all_bits, bit);
		g_hash_table_remove (bit->buim->priv->dirty_bits, bit);
		g_object_unref (bit->bm);
		g_free (bit->path);
		g_free (bit);
	}
	else
	{
		bit->destroy = TRUE;
	}
}

static BuimItem *
buim_item_new (GbBonoboUIMenu *buim, GbBookmark *b)
{
	BuimItem *ret = g_new0 (BuimItem, 1);
	ret->buim = buim;
	ret->bm = b;
	g_hash_table_insert (buim->priv->bookmark_to_bit, b, ret);
	g_hash_table_insert (buim->priv->all_bits, ret, ret);
	g_object_ref (b);
	
	return ret;
}

static void
gb_bonobo_ui_menu_cb (BonoboUIComponent *uic, BuimItem *bit, const char *path)
{
	if (GB_IS_FOLDER (bit->bm))
	{
		if (!bit->built_submenu)
		{
			gb_bonobo_ui_menu_build_submenu (bit->buim, GB_FOLDER (bit->bm), bit->path, bit);
		}
		if (!bit->built_children_submenu)
		{
			gb_bonobo_ui_menu_build_childs (bit->buim, GB_FOLDER (bit->bm));
			bit->built_children_submenu = TRUE;
		}
	}
	else if (GB_IS_SITE (bit->bm))
	{
		g_signal_emit (bit->buim, GbBonoboUIMenuSignals[GB_BONOBO_UI_MENU_BOOKMARK_ACTIVATED], 0, 
			       bit->bm, GB_SITE (bit->bm)->url, GB_BAF_DEFAULT);
	}
	else
	{
		g_warning ("unexpected bookmark type");
	}
}

static void
gb_bonobo_ui_menu_build_childs (GbBonoboUIMenu *buim, GbFolder *folder)
{
	GbBookmark *i;
	for (i = folder->child; i != NULL; i = i->next)
	{
		if (GB_IS_FOLDER (i))
		{
			BuimItem *bit = g_hash_table_lookup (buim->priv->bookmark_to_bit, i);
			if (bit)
			{
				gb_bonobo_ui_menu_build_submenu (bit->buim, GB_FOLDER (bit->bm), 
								 bit->path, bit);
			}
		}
	}
}


/* dealing with changes in the menu structure */

/* in an ideal world, this should not rebuild the whole submenu, but for now it's good
   enough to do so. The difference will not be noticed unless the menu has been torn off */

static void
gb_bonobo_ui_menu_child_modified_cb (GbFolder *f, GbBookmark *b, 
				     BuimItem *bit)
{
	gb_bonobo_ui_menu_rebuild_submenu_with_timeout (bit->buim, bit);
}

static void
gb_bonobo_ui_menu_child_removed_cb (GbFolder *f, GbBookmark *b, int pos,
				    BuimItem *bit)
{
	gb_bonobo_ui_menu_rebuild_submenu_with_timeout (bit->buim, bit);
}

static void
gb_bonobo_ui_menu_child_added_cb (GbFolder *f, GbBookmark *b, int pos,
				  BuimItem *bit)
{
	gb_bonobo_ui_menu_rebuild_submenu_with_timeout (bit->buim, bit);
}

static void
gb_bonobo_ui_menu_rebuild_submenu_with_timeout (GbBonoboUIMenu *buim, BuimItem *bit)
{
	g_return_if_fail (GB_IS_BONOBO_UI_MENU (buim));

	bit->dirty = TRUE;
	g_hash_table_insert (buim->priv->dirty_bits, bit, bit);

	if (!buim->priv->dirty_timeout)
	{
		buim->priv->dirty_timeout = g_timeout_add (REBUILD_TIMEOUT, 
							   gb_bonobo_ui_menu_rebuild_timeout_cb, buim);
	}

}

static gboolean
gb_bonobo_ui_menu_rebuild_timeout_cb (gpointer data)
{
	GSList *l = NULL;
	GSList *li;
	GbBonoboUIMenu *buim = data;
#ifdef DEBUG_BUIM
	g_print ("Rebuild timeout %x\n", (guint32) buim);
	g_print ("I'm going to rebuild %d items\n", g_hash_table_size (buim->priv->dirty_bits));
#endif

	g_hash_table_foreach (buim->priv->dirty_bits, gb_bonobo_ui_menu_hash_to_slist, &l);
	for (li = l; li; li = li->next)
	{
		BuimItem *bit = li->data;
#ifdef DEBUG_BUIM
		g_print ("+");
		g_assert (bit->dirty);
#endif
		if (!bit->destroy)
		{
			gb_bonobo_ui_menu_rebuild_submenu (buim, bit);
		}
		else
		{
			bit->dirty = FALSE;
			buim_item_destroy (bit, NULL);
		}
#ifdef DEBUG_BUIM
		g_print ("-\n");
#endif
	}
	g_slist_free (l);

	g_hash_table_destroy (buim->priv->dirty_bits);
	buim->priv->dirty_bits = g_hash_table_new (NULL, NULL);

	buim->priv->dirty_timeout = 0;
	return FALSE;
}

static void
gb_bonobo_ui_menu_rebuild_submenu (GbBonoboUIMenu *buim, BuimItem *bit)
{
#ifdef DEBUG_BUIM
	g_print ("Rebuilding bit: %x, %s\n", (guint32) bit, bit->bm->name);
#endif
	
	gul_bonobo_clear_path (buim->priv->uic, bit->path);
	bit->built_submenu = FALSE;
	bit->built_children_submenu = FALSE;

	if (buim->priv->root == GB_FOLDER (bit->bm))
	{
		gb_bonobo_ui_menu_build_submenu (buim, GB_FOLDER (bit->bm), bit->path, bit);
	}
}

void
gb_bonobo_ui_menu_set_location_source (GbBonoboUIMenu *buim, GbLocationSource *src)
{
	GbBonoboUIMenuPrivate *p = buim->priv;

	if (p->location_source)
	{
		g_object_remove_weak_pointer (G_OBJECT (p->location_source),
					      (gpointer *) &p->location_source);
	}

	p->location_source = src;

	if (p->location_source)
	{
		g_object_add_weak_pointer (G_OBJECT (p->location_source), 
					   (gpointer *) &p->location_source);
	}
}
