/*
 *  Copyright (C) 2002 Christophe Fergeau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * This GTK2 widget extends the standard GtkNotebook with tab dragging support.
 * Even if this is called GulNotebook, please try not to add galeon specific
 * code in there if possible, so other apps like gedit or profterm can use 
 * it easily if they want.
 */

#include "gul-notebook.h"
#include "eel-gconf-extensions.h"
#include "prefs-strings.h"

#include <gtk/gtk.h>
#include <glib-object.h>
#include <libgnome/gnome-i18n.h>

#define AFTER_ALL_TABS -1

#define TAB_MIN_SIZE 60
#define TAB_MAX_SIZE 8

struct GulNotebookPrivate 
{
	GtkStyle *loading_text_style;
	GtkStyle *new_text_style;
	gboolean colors_loaded;
	GList *focused_pages;
	
	/* Used during tab drag'n'drop */
	gulong motion_notify_handler_id;
	gboolean drag_in_progress;
	GulNotebook *src_notebook;
	gint src_page;
};

/* GObject boilerplate code */
static void gul_notebook_init         (GulNotebook *notebook);
static void gul_notebook_class_init   (GulNotebookClass *klass);
static void gul_notebook_finalize     (GObject *object);

/* Local variables */
static GdkCursor *cursor = NULL;

GType 
gul_notebook_get_type (void)
{
        static GType gul_notebook_type = 0;

        if (gul_notebook_type == 0)
        {
                static const GTypeInfo our_info =
			{
				sizeof (GulNotebookClass),
				NULL, /* base_init */
				NULL, /* base_finalize */
				(GClassInitFunc) gul_notebook_class_init,
				NULL,
				NULL, /* class_data */
				sizeof (GulNotebook),
				0, /* n_preallocs */
				(GInstanceInitFunc) gul_notebook_init
			};
		
                gul_notebook_type = g_type_register_static (GTK_TYPE_NOTEBOOK,
							     "GulNotebook",
							     &our_info, 0);
        }

        return gul_notebook_type;
}

static void
gul_notebook_class_init (GulNotebookClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = gul_notebook_finalize;
}

static gboolean 
is_in_notebook_window (GulNotebook *notebook, 
		       gint abs_x, gint abs_y)
{
	gint x, y;
	gint rel_x, rel_y;
	gint width, height;
	GtkWidget *toplevel = gtk_widget_get_toplevel (GTK_WIDGET(notebook));
	GdkWindow *window = GTK_WIDGET(toplevel)->window;
	
	gdk_window_get_origin (window, &x, &y);
	rel_x = abs_x - x;
	rel_y = abs_y - y;
	g_assert (rel_x>=0 && rel_y>=0);

	x = GTK_WIDGET(notebook)->allocation.x;
	y = GTK_WIDGET(notebook)->allocation.y;
	height = GTK_WIDGET(notebook)->allocation.height;
	width  = GTK_WIDGET(notebook)->allocation.width;
	return ((rel_x>=x) && (rel_y>=y) && (rel_x<=x+width) && (rel_y<=y+height));
}

static gint
find_tab_num_at_pos (GulNotebook *notebook, gint abs_x, gint abs_y)
{
	GtkPositionType tab_pos;
	int page_num = 0;
	GtkNotebook *nb = GTK_NOTEBOOK(notebook);
	GtkWidget *page;
	
	tab_pos = gtk_notebook_get_tab_pos (GTK_NOTEBOOK(notebook));

	if (GTK_NOTEBOOK(notebook)->first_tab == NULL) 
	{
		return AFTER_ALL_TABS;
	}
	
	g_assert (is_in_notebook_window(notebook, abs_x, abs_y));
	
	while ((page = gtk_notebook_get_nth_page (nb, page_num)))  
	{
		GtkWidget *tab;
		gint max_x, max_y;
		gint x_root, y_root;
		
		tab = gtk_notebook_get_tab_label (nb, page);
		g_return_val_if_fail(tab != NULL, -1);
		
		if (!GTK_WIDGET_MAPPED(GTK_WIDGET(tab))) 
		{
			page_num++;
			continue;
		}

		gdk_window_get_origin (GDK_WINDOW(tab->window), 
				       &x_root, &y_root);

		max_x = x_root + tab->allocation.x + tab->allocation.width;
		max_y = y_root + tab->allocation.y + tab->allocation.height;

		if (((tab_pos == GTK_POS_TOP) 
		     || (tab_pos == GTK_POS_BOTTOM)) 
		    &&(abs_x<=max_x)) 
		{
			return page_num;
		} else if (((tab_pos == GTK_POS_LEFT) 
			    || (tab_pos == GTK_POS_RIGHT))
			   && (abs_y<=max_y)) 
		{
			return page_num;
		}		      

		page_num++;
	}
	return AFTER_ALL_TABS;
}

static gboolean
is_notebook_at_pointer (GulNotebook *nb,
			gint abs_x, gint abs_y)
{
	gint x, y;
	GdkWindow *win_at_pointer = gdk_window_at_pointer (&x, &y);
	GdkWindow *parent_at_pointer = NULL;
	GdkWindow *win;
	
	if (win_at_pointer == NULL) 
	{
		return FALSE;
	}

	gdk_window_get_toplevel(win_at_pointer);
	/* When we are in the notebook event window, win_at_pointer will be
	   this event window, and the toplevel window we are interested in
	   will be its parent
	*/
	parent_at_pointer = gdk_window_get_parent (win_at_pointer);

	win = GTK_WIDGET(nb)->window;
	win = gdk_window_get_toplevel (win);

	return ((win == win_at_pointer || win == parent_at_pointer)
	        && is_in_notebook_window (nb, abs_x, abs_y));
}

static void 
move_tab (GulNotebook *notebook, gint dest_page_num)
{
	gint cur_page_num;

	cur_page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK(notebook));

	if (dest_page_num != cur_page_num)
	{
		GtkWidget *cur_page = 
			gtk_notebook_get_nth_page (GTK_NOTEBOOK(notebook),
						   cur_page_num);
		gtk_notebook_reorder_child (GTK_NOTEBOOK(notebook), cur_page,
					    dest_page_num);
	}
}

static void 
drag_start (GulNotebook *notebook, 
	    GulNotebook *src_notebook, 
	    gint src_page)
{
	notebook->priv->drag_in_progress = TRUE;
	notebook->priv->src_notebook = src_notebook;
	notebook->priv->src_page = src_page;

	/* get a new cursor, if necessary */
	if (!cursor) cursor = gdk_cursor_new (GDK_FLEUR);
	
	/* grab the pointer */
	gtk_grab_add (GTK_WIDGET(notebook));
	if (!gdk_pointer_is_grabbed ()) {
		gdk_pointer_grab (GDK_WINDOW(GTK_WIDGET(notebook)->window), 
				  FALSE,
				  GDK_BUTTON1_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
				  NULL, cursor, GDK_CURRENT_TIME);
	}
}

static void 
drag_stop (GulNotebook *notebook)
{
	notebook->priv->drag_in_progress = FALSE;
	notebook->priv->src_notebook = NULL;
	notebook->priv->src_page = -1;
	if (notebook->priv->motion_notify_handler_id != 0) 
	{
		g_signal_handler_disconnect (G_OBJECT(notebook), 
					     notebook->priv->motion_notify_handler_id);
		notebook->priv->motion_notify_handler_id = 0;
	}       
}

/* Callbacks */
static gboolean
button_release_cb (GulNotebook *notebook, GdkEventButton *event,
		   gpointer data)
{
	if (notebook->priv->drag_in_progress) 
	{
		/* ungrab the pointer if it's grabbed */
		if (gdk_pointer_is_grabbed ())
		{
			gdk_pointer_ungrab (GDK_CURRENT_TIME);
			gtk_grab_remove (GTK_WIDGET(notebook));
		}
	}

	/* This must be called even if a drag isn't happening */
	drag_stop(notebook);
	return FALSE;
}

static gboolean 
motion_notify_cb (GulNotebook *notebook, GdkEventMotion *event, 
		  gpointer data)
{
	int page_num;
	int result;
	int abs_x;
	int abs_y;

	abs_x = (int)event->x_root;
	abs_y = (int)event->y_root;
	
	if (notebook->priv->drag_in_progress == FALSE) 
	{
		gint cur_page;
		
		cur_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(notebook));
		drag_start (notebook, notebook, cur_page); 
	}

	result = is_notebook_at_pointer (notebook, abs_x, abs_y);
	if (result) 
	{
		page_num = find_tab_num_at_pos (notebook, abs_x, abs_y);
		g_assert (page_num >= -1);
		move_tab (notebook, page_num);
	} 

	return FALSE;
}

static gboolean 
button_press_cb (GulNotebook *notebook, 
		 GdkEventButton *event,
		 gpointer data)
{
	gint tab_clicked = find_tab_num_at_pos(notebook, 
					       event->x_root, 
					       event->y_root);
	
	if (notebook->priv->drag_in_progress)
	{
		return TRUE;
	}

	if ((event->button == 1) && (event->type == GDK_BUTTON_PRESS)
	    && (tab_clicked != -1))
	{
		notebook->priv->motion_notify_handler_id = 
			g_signal_connect (G_OBJECT (notebook), 
					  "motion-notify-event",
					  G_CALLBACK (motion_notify_cb), NULL);
	}

	return FALSE;
}

GtkWidget *
gul_notebook_new (void)
{
	return GTK_WIDGET (g_object_new (GUL_NOTEBOOK_TYPE, NULL));
}

static void
load_colours (GulNotebook *nb, GtkWidget *label)
{
	GtkStyle  *rcstyle;
	char *tmp;
	
	rcstyle = gtk_rc_get_style (label);

        /* this is needed for some (broken?) themes */
        if (rcstyle == NULL)
        {
                rcstyle = gtk_style_new ();
        }
        else
        {
                g_object_ref (rcstyle);
        }

	/* make styles */
        nb->priv->loading_text_style = gtk_style_copy (rcstyle);
        nb->priv->new_text_style = gtk_style_copy (rcstyle);
	
	tmp = eel_gconf_get_string (CONF_TABS_TABBED_LOADING_COLOR);
        gdk_color_parse (tmp, &(nb->priv->loading_text_style->fg[0]));
        gdk_color_parse (tmp, &(nb->priv->loading_text_style->fg[2]));
       	g_free (tmp);
	
	tmp = eel_gconf_get_string (CONF_TABS_TABBED_NEW_COLOR);
        gdk_color_parse (tmp, &(nb->priv->new_text_style->fg[0]));
        gdk_color_parse (tmp, &(nb->priv->new_text_style->fg[2]));
        g_free (tmp);
}

static void
free_colours (GulNotebook *nb)
{
	g_object_unref (nb->priv->loading_text_style);
	g_object_unref (nb->priv->new_text_style);
}

static GList *
tab_remove_from_list (GList *list, GtkWidget *child)
{
	GList *rl;
	GList *l = list;
	
	rl = g_list_find (l, child);
        if (rl)
	{
		l = g_list_remove (l, rl->data);
	}

	return l;
}

static void
gul_notebook_switch_page_cb (GtkNotebook *notebook,
                             GtkNotebookPage *page,
                             guint page_num,
                             gpointer data)
{
	GulNotebook *nb = GUL_NOTEBOOK (notebook);
	GtkWidget *child;
	
	child = gtk_notebook_get_nth_page (notebook, page_num);

	/* Remove the old page, we dont want to grow unnecessarily
	 * the list */
	if (nb->priv->focused_pages)
	{
		nb->priv->focused_pages = tab_remove_from_list 
			(nb->priv->focused_pages, child);
	}

	nb->priv->focused_pages = g_list_append (nb->priv->focused_pages,
						 child);
}

static void
gul_notebook_init (GulNotebook *notebook)
{
	notebook->priv = g_new(GulNotebookPrivate, 1);

	notebook->priv->drag_in_progress = FALSE;
	notebook->priv->motion_notify_handler_id = 0;
	notebook->priv->src_notebook = NULL;
	notebook->priv->src_page = -1;
	notebook->priv->colors_loaded = FALSE;
	notebook->priv->focused_pages = NULL;
	
	g_signal_connect (notebook, "button-press-event", 
			  (GCallback)button_press_cb, NULL);
	g_signal_connect (notebook, "button-release-event", 
			  (GCallback)button_release_cb, NULL);
	gtk_widget_add_events (GTK_WIDGET(notebook), GDK_BUTTON1_MOTION_MASK);

	g_signal_connect_after (G_OBJECT (notebook), "switch_page",
                                G_CALLBACK (gul_notebook_switch_page_cb),
                                NULL);
}

static void 
gul_notebook_finalize (GObject *object)
{
	GulNotebook *notebook = GUL_NOTEBOOK(object);

	if (notebook->priv->colors_loaded)
	{
		free_colours (notebook);
	}

	if (notebook->priv->focused_pages)
	{
		g_list_free (notebook->priv->focused_pages);
	}
	
	g_free (notebook->priv);
}

static void
tab_label_set_size (GtkWidget *notebook, GtkWidget *label)
{
	int label_width;
		
	label_width = notebook->allocation.width/TAB_MAX_SIZE;

	if (label_width < TAB_MIN_SIZE) label_width = TAB_MIN_SIZE;
	
	gtk_widget_set_size_request (label, label_width, -1);
}

static GtkWidget *
tab_get_label (GulNotebook *nb, GtkWidget *child)
{
	GtkWidget *hbox, *label;

	hbox = gtk_notebook_get_tab_label (GTK_NOTEBOOK (nb),
					   child);
	label = g_object_get_data (G_OBJECT (hbox), "label");

	return label;
}

static void
tab_label_size_request_cb (GtkWidget *widget,
			   GtkRequisition *requisition,
			   GtkWidget *child)
{
	GtkWidget *hbox;
	
	hbox = gtk_notebook_get_tab_label (GTK_NOTEBOOK (widget),
					   child);
	tab_label_set_size (widget, hbox);
}

void
gul_notebook_set_page_status (GulNotebook *nb,
			      GtkWidget *child,
			      GulNotebookPageLoadStatus status)
{
	GtkWidget *label;

	label = tab_get_label (nb, child);

	switch (status)
	{
		case GUL_NOTEBOOK_TAB_LOAD_ACTIVATED:
       	        	gtk_widget_set_style (label, NULL);
		break;
		
		case GUL_NOTEBOOK_TAB_LOAD_STARTED:
         	       gtk_widget_set_style 
			       (label, nb->priv->loading_text_style);
		break;
		
        	case GUL_NOTEBOOK_TAB_LOAD_COMPLETED:
	                gtk_widget_set_style 
				(label, nb->priv->new_text_style);
		break;
	}
}

static void
galeon_tab_close_button_clicked_cb (GtkWidget *widget,
				    GtkWidget *child)
{
	GulNotebook *notebook;

	notebook = g_object_get_data (G_OBJECT (widget), "notebook");
	
	gul_notebook_remove_page (notebook, child);
}

static GtkWidget *
tab_build_label (GulNotebook *nb, GtkWidget *child)
{
	GtkWidget *label, *hbox, *close_button, *image;
	int h, w;
	GClosure *closure;
	
	gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &w, &h);
	
	hbox = gtk_hbox_new (FALSE, 0);

	/* setup close button */
	close_button = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON(close_button), 
			       GTK_RELIEF_NONE);
	image = gtk_image_new_from_stock (GTK_STOCK_CLOSE,
					  GTK_ICON_SIZE_MENU);
	gtk_widget_set_size_request (close_button, w, h);
	gtk_container_add (GTK_CONTAINER (close_button), 
			   image);
		
	/* setup label */ 
        label = gtk_label_new (_("Untitled"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.00, 0.5);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

	tab_label_set_size (GTK_WIDGET (nb), label);
		
	closure = g_cclosure_new (G_CALLBACK(tab_label_size_request_cb), 
				  child, NULL);
  	g_object_watch_closure (G_OBJECT (label), closure);
  	g_signal_connect_closure_by_id (G_OBJECT(nb),
		g_signal_lookup ("size_request", 
				 G_OBJECT_TYPE (G_OBJECT(nb))), 0,
                                 closure,
                                 FALSE);
	
	if (!nb->priv->colors_loaded)
	{
		load_colours (nb, label);
	}
	
	/* setup button */
	gtk_box_pack_start (GTK_BOX (hbox), close_button,
                            FALSE, FALSE, 0);

	g_signal_connect (G_OBJECT (close_button), "clicked",
                          G_CALLBACK (galeon_tab_close_button_clicked_cb),
                          child);
	g_object_set_data (G_OBJECT (close_button), "notebook", nb);
		
	gtk_widget_show (hbox);
	gtk_widget_show (label);
	gtk_widget_show (image);
	gtk_widget_show (close_button);

	g_object_set_data (G_OBJECT (hbox), "label", label);
	
	return hbox;
}

/*
 * update_tabs_visibility: Hide tabs if there is only one tab
 * and the pref is not set.
 * HACK We need to show tabs before inserting the second. Otherwise
 * gtknotebook go crazy.
 */
static void
update_tabs_visibility (GulNotebook *nb, gboolean before_inserting)
{
	gboolean show_tabs;
	guint tabs_num = 1;
	
	if (before_inserting) tabs_num--;
	
	show_tabs = gtk_notebook_get_nth_page (GTK_NOTEBOOK (nb), tabs_num)
		|| eel_gconf_get_boolean (CONF_TABS_TABBED_ALWAYS_SHOW);
	
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), show_tabs);
}

void
gul_notebook_insert_page (GulNotebook *nb,
			  GtkWidget *child,
			  int position,
			  gboolean jump_to)
{
	GtkWidget *tab_hbox;
	
	tab_hbox = tab_build_label (nb, child);
		
	update_tabs_visibility (nb, TRUE);
	
	gtk_notebook_insert_page (GTK_NOTEBOOK (nb), 
				  child,
				  tab_hbox, position);

	if (jump_to)
	{
		gtk_notebook_set_current_page (GTK_NOTEBOOK (nb),
					       position);
		g_object_set_data (G_OBJECT (child), "jump_to", 
				   GINT_TO_POINTER (jump_to));
	}
}

static void
smart_tab_switching_on_closure (GulNotebook *nb,
			  	GtkWidget *child)
{
	gboolean jump_to;
	
	jump_to = GPOINTER_TO_INT (g_object_get_data 
				   (G_OBJECT (child), "jump_to"));

	if (!jump_to || !nb->priv->focused_pages)
	{
		gtk_notebook_next_page (GTK_NOTEBOOK (nb));
	}
	else
	{
		GList *l;
		GtkWidget *child;
		int page_num;

		/* activate the last focused tab */
		l = g_list_last (nb->priv->focused_pages);
		child = GTK_WIDGET (l->data);
		page_num = gtk_notebook_page_num (GTK_NOTEBOOK (nb),
						  child);
		gtk_notebook_set_current_page 
			(GTK_NOTEBOOK (nb), page_num);
	}
}

void
gul_notebook_remove_page (GulNotebook *nb,
			  GtkWidget *child)
{
	int position, cur;
	gboolean last_tab;

	last_tab = gtk_notebook_get_nth_page (GTK_NOTEBOOK (nb), 1) == NULL;
	if (last_tab)
	{
		GtkWidget *window;
		window = gtk_widget_get_toplevel (GTK_WIDGET (nb));
		gtk_widget_destroy (window);
		return;
	}
	
	/* Remove the page from the focused pages list */
	nb->priv->focused_pages = tab_remove_from_list 
		(nb->priv->focused_pages, child);
		
	position = gtk_notebook_page_num (GTK_NOTEBOOK (nb), 
					  child);
	
	cur = gtk_notebook_get_current_page (GTK_NOTEBOOK (nb));
	if (position == cur)
	{
		smart_tab_switching_on_closure (nb, child);
	}
	
	gtk_notebook_remove_page (GTK_NOTEBOOK (nb), position);

	update_tabs_visibility (nb, FALSE);
}

void
gul_notebook_set_page_title (GulNotebook *nb,
			     GtkWidget *child,
			     const char *title)
{
	GtkWidget *label;
	
	label = tab_get_label (nb, child);
	gtk_label_set_label (GTK_LABEL (label), title);
}
