#define __GNOME_PRINT_MASTER_PREVIEW_C__

/*
 *  Copyright (C) 2000-2001 Ximian Inc.
 *
 *  Authors:
 *    Michael Zucchi <notzed@helixcode.com>
 *    Miguel de Icaza <miguel@gnu.org>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  A system print preview window.  Based on print-preview.c
 *  from gnumeric.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define noGPMP_VERBOSE

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <libart_lgpl/art_affine.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtkaccelgroup.h>
#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtklabel.h>
#include <libgnomeprint/gnome-print-private.h>
#include <libgnomeprint/gnome-print-meta.h>

#include "gnome-print-i18n.h"

#include "gnome-print-master-preview.h"
#if 0
#include "gnome-print-preview-icons.h"
#endif

#define GPMP_ZOOM_IN_FACTOR M_SQRT2
#define GPMP_ZOOM_OUT_FACTOR M_SQRT1_2
#define GPMP_ZOOM_MIN 0.0625
#define GPMP_ZOOM_MAX 16.0

#define GPMP_A4_WIDTH (210.0 * 72.0 / 25.4)
#define GPMP_A4_HEIGHT (297.0 * 72.0 / 2.54)

#define TOOLBAR_BUTTON_BASE 5
#define MOVE_INDEX 5

#if 0
typedef enum {
	GPMP_MODE_MOVE,
	GPMP_MODE_MOVE_DRAGGING,
	GPMP_MODE_ZOOM_IN,
	GPMP_MODE_ZOOM_OUT
} GPMPPreviewMode;
#endif

struct _GnomePrintMasterPreview {
	GtkWindow window;
	/* Main VBox */
	GtkWidget *vbox;
	/* Some interesting buttons */
	GtkWidget *bpf, *bpp, *bpn, *bpl;
	GtkWidget *bz1, *bzf, *bzi, *bzo;
	/* Number of pages master has */
	gint pages;
	/* Zoom factor */
	gdouble zoom;

	/* Physical area dimensions */
	gdouble paw, pah;
	/* Calculated Physical Area -> Layout */
	gdouble pa2ly[6];

	/* State */
	guint dragging : 1;
	gint anchorx, anchory;
	gint offsetx, offsety;

	gpointer priv;
};

struct _GnomePrintMasterPreviewClass {
	GtkWindowClass parent_class;
};

typedef struct _GPMPPrivate GPMPPrivate;

struct _GPMPPrivate {
	/* Our GnomePrintMaster */
	GnomePrintMaster *master;
	/* Our GnomePrintPreview */
	GnomePrintContext *preview;

	GtkWidget *page_entry;
	GtkWidget *scrolled_window;
	GtkWidget *last;
	GnomeCanvas *canvas;

	gint current_page;
	gint pagecount;

#if 0
	GPMPPreviewMode mode;
#endif
};

static void gpmp_parse_layout (GnomePrintMasterPreview *mp);

/*
 * Padding in points around the simulated page
 */

#define PAGE_PAD 4

static gint
render_page (GnomePrintMasterPreview *mp, gint page)
{
	GPMPPrivate *priv;
	GnomePrintConfig *config;
	ArtDRect region;
	gdouble transform[6];
	GnomePrintContext *meta;
	const guchar *data;
	gint len;

	priv = (GPMPPrivate *) mp->priv;

	if (priv->preview) {
		g_object_unref (G_OBJECT (priv->preview));
		priv->preview = NULL;
	}

	/* Set page transformation */
	transform[0] =  1.0;
	transform[1] =  0.0;
	transform[2] =  0.0;
	transform[3] = -1.0;
	transform[4] =  0.0;
	transform[5] =  mp->pah;
	art_affine_multiply (transform, mp->pa2ly, transform);

	/* Reset scrolling region always */
	region.x0 = region.y0 = 0.0 - PAGE_PAD;
	region.x1 = mp->paw + PAGE_PAD;
	region.y1 = mp->pah + PAGE_PAD;

	config = gnome_print_master_get_config (priv->master);
	priv->preview = gnome_print_preview_new_full (config, priv->canvas, transform, &region);
	gnome_print_config_unref (config);

	gnome_print_master_render_page (priv->master, priv->preview, page, TRUE);

	return GNOME_PRINT_OK;
}

static gint
goto_page (GnomePrintMasterPreview *mp, gint page)
{
	GPMPPrivate *priv;
	guchar c[32];

	priv = (GPMPPrivate *) mp->priv;

	g_snprintf (c, 32, "%d", page + 1);
	gtk_entry_set_text (GTK_ENTRY (priv->page_entry), c);
	
	gtk_widget_set_sensitive (mp->bpf, (page != 0));
	gtk_widget_set_sensitive (mp->bpp, (page != 0));
	gtk_widget_set_sensitive (mp->bpn, (page != (mp->pages - 1)));
	gtk_widget_set_sensitive (mp->bpl, (page != (mp->pages - 1)));

	if (page != priv->current_page) {
		priv->current_page = page;
		return render_page (mp, page);
	}

	return GNOME_PRINT_OK;
}

static gint
change_page_cmd (GtkEntry *entry, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	const gchar *text;
	gint page;

	priv = (GPMPPrivate *) pmp->priv;

	text = gtk_entry_get_text (entry);

	page = CLAMP (atoi (text), 1, priv->pagecount) - 1;

	return goto_page (pmp, page);
}

#define CLOSE_ENOUGH(a,b) (fabs (a - b) < 1e-6)

static void
gpmp_zoom (GnomePrintMasterPreview *mp, gdouble factor, gboolean relative)
{
	GPMPPrivate *priv;
	gdouble zoom;

	priv = (GPMPPrivate *) mp->priv;
	
	if (relative) {
		zoom = mp->zoom * factor;
	} else {
		zoom = factor;
	}

	mp->zoom = CLAMP (zoom, GPMP_ZOOM_MIN, GPMP_ZOOM_MAX);

	gtk_widget_set_sensitive (mp->bz1, (!CLOSE_ENOUGH (mp->zoom, 1.0)));
	gtk_widget_set_sensitive (mp->bzi, (!CLOSE_ENOUGH (mp->zoom, GPMP_ZOOM_MAX)));
	gtk_widget_set_sensitive (mp->bpn, (!CLOSE_ENOUGH (mp->zoom, GPMP_ZOOM_MIN)));

	gnome_canvas_set_pixels_per_unit (priv->canvas, mp->zoom);
}

/* Button press handler for the print preview canvas */

static gint
preview_canvas_button_press (GtkWidget *widget, GdkEventButton *event, GnomePrintMasterPreview *mp)
{
	gint retval;

	retval = FALSE;

	if (event->button == 1) {
		GdkCursor *cursor;

		mp->dragging = TRUE;

		gnome_canvas_get_scroll_offsets (GNOME_CANVAS (widget), &mp->offsetx, &mp->offsety);

		mp->anchorx = event->x - mp->offsetx;
		mp->anchory = event->y - mp->offsety;

		cursor = gdk_cursor_new (GDK_FLEUR);
		gdk_pointer_grab (widget->window, FALSE,
				  (GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK | GDK_BUTTON_RELEASE_MASK),
				  NULL, cursor, event->time);
		gdk_cursor_unref (cursor);

		retval = TRUE;
	}

	return retval;
}

/* Motion notify handler for the print preview canvas */

static gint
preview_canvas_motion (GtkWidget *widget, GdkEventMotion *event, GnomePrintMasterPreview *mp)
{
	GdkModifierType mod;
	gint retval;

	retval = FALSE;

	if (mp->dragging) {
		gint x, y, dx, dy;

		if (event->is_hint) {
			gdk_window_get_pointer (widget->window, &x, &y, &mod);
		} else {
			x = event->x;
			y = event->y;
		}

		dx = mp->anchorx - x;
		dy = mp->anchory - y;

		gnome_canvas_scroll_to (((GPMPPrivate *) mp->priv)->canvas, mp->offsetx + dx, mp->offsety + dy);

		/* Get new anchor and offset */
		mp->anchorx = event->x;
		mp->anchory = event->y;
		gnome_canvas_get_scroll_offsets (GNOME_CANVAS (widget), &mp->offsetx, &mp->offsety);

		retval = TRUE;
	}

	return retval;
}

/* Button release handler for the print preview canvas */

static gint
preview_canvas_button_release (GtkWidget *widget, GdkEventButton *event, GnomePrintMasterPreview *mp)
{
	gint retval;

	retval = TRUE;

	if (event->button == 1) {
		mp->dragging = FALSE;
		gdk_pointer_ungrab (event->time);
		retval = TRUE;
	}

	return retval;
}


static void
preview_close_cmd (gpointer unused, GnomePrintMasterPreview *mp)
{
	gtk_widget_destroy (GTK_WIDGET (mp));
}

static gboolean
preview_delete_event (gpointer unused, GdkEventAny *event, GnomePrintMasterPreview *mp)
{
	gtk_widget_destroy (GTK_WIDGET (mp));

	return TRUE;
}

static void
preview_file_print_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;
	
	gnome_print_master_print (priv->master);

	/* fixme: should we clean ourselves up now? */
}

static void
preview_first_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	goto_page (pmp, 0);
}

static void
preview_next_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;

	if (priv->current_page < (priv->pagecount - 1)) {
		goto_page (pmp, priv->current_page + 1);
	}
}

static void
preview_prev_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;

	if (priv->current_page > 0) {
		goto_page (pmp, priv->current_page - 1);
	}
}

static void
preview_last_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;

	priv = (GPMPPrivate *) pmp->priv;
	
	goto_page (pmp, priv->pagecount - 1);
}

static void
gpmp_zoom_in_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	gpmp_zoom (pmp, GPMP_ZOOM_IN_FACTOR, TRUE);
}

static void
gpmp_zoom_out_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	gpmp_zoom (pmp, GPMP_ZOOM_OUT_FACTOR, TRUE);
}

static void
preview_zoom_fit_cmd (GtkToggleButton *t, GnomePrintMasterPreview *mp)
{
	GPMPPrivate *priv;
	gdouble zoomx, zoomy;
	gint width, height;

	priv = (GPMPPrivate *) mp->priv;

	width = GTK_WIDGET (priv->canvas)->allocation.width;
	height = GTK_WIDGET (priv->canvas)->allocation.height;

	zoomx = width / (mp->paw + 5.0 + PAGE_PAD);
	zoomy = height / (mp->pah + 5.0 + PAGE_PAD);
	
	gpmp_zoom (mp, MIN (zoomx, zoomy), FALSE);
}

static void
preview_zoom_100_cmd (GtkToggleButton *t, GnomePrintMasterPreview *mp)
{
	gpmp_zoom (mp, 1.0, FALSE);
}

#if 0
static void
preview_zoom_fit_wide_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	GPMPPrivate *priv;
	gdouble zoom;
	gint width;

	priv = (GPMPPrivate *) pmp->priv;

	width = GTK_WIDGET (priv->canvas)->allocation.width;

	zoom = width / (mp->paw + 5.0 + PAGE_PAD);
	gnome_canvas_set_pixels_per_unit (priv->canvas, zoom);
}

static void
preview_zoom_fit_tall_cmd (GtkToggleButton *t, GnomePrintMasterPreview *mp)
{
	GPMPPrivate *priv;
	gdouble zoom;
	gint height;

	priv = (GPMPPrivate *) pmp->priv;

	height = GTK_WIDGET (priv->canvas)->allocation.height;

	zoom = height / (mp->pah + 5.0 + PAGE_PAD);
	gnome_canvas_set_pixels_per_unit (priv->canvas, zoom);
}
#endif

static gint
preview_canvas_key (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	GnomePrintMasterPreview *pmp;
	GPMPPrivate *priv;
	gint x,y;
	gint height, width;
	gint domove = 0;

	pmp = (GnomePrintMasterPreview *) data;

	priv = (GPMPPrivate *) pmp->priv;

	gnome_canvas_get_scroll_offsets (priv->canvas, &x, &y);
	height = GTK_WIDGET (priv->canvas)->allocation.height;
	width = GTK_WIDGET (priv->canvas)->allocation.width;

	switch (event->keyval) {
	case '1':
		preview_zoom_100_cmd (0, pmp);
		break;
#if 0
	case '2':		/* as good a key as any? */
		preview_zoom_fit_wide_cmd (0, pmp);
		break;
	case '3':
		preview_zoom_fit_tall_cmd (0, pmp);
		break;
#endif
	case '+':
	case '=':
	case GDK_KP_Add:
		gpmp_zoom_in_cmd (NULL, pmp);
		break;
	case '-':
	case '_':
	case GDK_KP_Subtract:
		gpmp_zoom_out_cmd (NULL, pmp);
		break;
	case GDK_KP_Right:
	case GDK_Right:
		if (event->state & GDK_SHIFT_MASK)
			x+=width;
		else
			x+=10;
		domove=1;
		break;
	case GDK_KP_Left:
	case GDK_Left:
		if (event->state & GDK_SHIFT_MASK)
			x-=width;
		else
			x-=10;
		domove=1;
		break;
	case GDK_KP_Up:
	case GDK_Up:
		if (event->state & GDK_SHIFT_MASK)
			goto page_up;
		y-=10;
		domove=1;
		break;
	case GDK_KP_Down:
	case GDK_Down:
		if (event->state & GDK_SHIFT_MASK)
			goto page_down;
		y+=10;
		domove=1;
		break;
	case GDK_KP_Page_Up:
	case GDK_Page_Up:
	case GDK_Delete:
	case GDK_KP_Delete:
	case GDK_BackSpace:
	page_up:
		if (y<=0) {
			if (priv->current_page > 0) {
				goto_page (pmp, priv->current_page - 1);
				y = GTK_LAYOUT (priv->canvas)->height - height;
			}
		} else {
			y -= height;
		}
		domove=1;
		break;
	case GDK_KP_Page_Down:
	case GDK_Page_Down:
	case ' ':
	page_down:
		if (y >= GTK_LAYOUT (priv->canvas)->height - height) {
			if (priv->current_page < priv->pagecount - 1) {
				goto_page (pmp, priv->current_page + 1);
				y = 0;
			}
		} else {
			y += height;
		}
		domove=1;
		break;
	case GDK_KP_Home:
	case GDK_Home:
		goto_page (pmp, 0);
		y=0;
		domove=1;
		break;
	case GDK_KP_End:
	case GDK_End:
		goto_page (pmp, priv->pagecount-1);
		y=0;
		domove=1;
		break;
	case GDK_Escape:
		g_object_unref (G_OBJECT (pmp));
		return TRUE;
	default:
		return FALSE;
	}

	if (domove)
		gnome_canvas_scroll_to (priv->canvas, x, y);

	gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
	return TRUE;
}

static void
create_preview_canvas (GnomePrintMasterPreview *mp)
{
	GPMPPrivate *priv;
	GtkWidget *status;
	GnomeCanvasItem *item;
	GnomePrintConfig *config;

	priv = (GPMPPrivate *) mp->priv;

	priv->scrolled_window = gtk_scrolled_window_new (NULL, NULL);

	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	priv->canvas = GNOME_CANVAS (gnome_canvas_new_aa ());
	gtk_widget_pop_colormap ();

	gtk_signal_connect (GTK_OBJECT (priv->canvas), "button_press_event",
			    GTK_SIGNAL_FUNC (preview_canvas_button_press), mp);
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "button_release_event",
			    GTK_SIGNAL_FUNC (preview_canvas_button_release), mp);
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "motion_notify_event",
			    GTK_SIGNAL_FUNC (preview_canvas_motion), mp);
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "key_press_event",
			    GTK_SIGNAL_FUNC (preview_canvas_key), mp);

	gtk_container_add (GTK_CONTAINER (priv->scrolled_window), GTK_WIDGET (priv->canvas));

	config = gnome_print_master_get_config (priv->master);
	priv->preview = gnome_print_preview_new (config, priv->canvas);
	gnome_print_config_unref (config);

	/*
	 * Now add some padding above and below and put a simulated
	 * page on the background
	 */

	item = gnome_canvas_item_new (gnome_canvas_root (priv->canvas),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", 0.0, "y1", 0.0, "x2", (gdouble) mp->paw, "y2", (gdouble) mp->pah,
				      "fill_color", "white",
				      "outline_color", "black",
				      "width_pixels", 1, NULL);
	gnome_canvas_item_lower_to_bottom (item);
	item = gnome_canvas_item_new (gnome_canvas_root (priv->canvas),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", 3.0, "y1", 3.0, "x2", (gdouble) mp->paw + 3, "y2", (gdouble) mp->pah + 3,
				      "fill_color", "black", NULL);
	gnome_canvas_item_lower_to_bottom (item);
	gnome_canvas_set_scroll_region (priv->canvas, 0 - PAGE_PAD, 0 - PAGE_PAD, mp->paw + PAGE_PAD, mp->pah + PAGE_PAD);

	status = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (status), gtk_label_new (_ ("Page: ")), FALSE, FALSE, 0);
	priv->page_entry = gtk_entry_new ();
	gtk_widget_set_usize (priv->page_entry, 40, 0);
	gtk_signal_connect (GTK_OBJECT (priv->page_entry), "activate", GTK_SIGNAL_FUNC (change_page_cmd), mp);
	gtk_box_pack_start (GTK_BOX (status), priv->page_entry, FALSE, FALSE, 0);
	priv->last = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (status), priv->last, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (mp->vbox), status, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (mp->vbox), priv->scrolled_window, TRUE, TRUE, 0);

	gtk_widget_show_all (mp->vbox);

	gtk_widget_grab_focus (GTK_WIDGET (priv->canvas));
}

#ifdef GPMP_VERBOSE
#define PRINT_2(s,a,b) g_print ("GPMP %s %g %g\n", s, (a), (b))
#define PRINT_DRECT(s,a) g_print ("GPMP %s %g %g %g %g\n", (s), (a)->x0, (a)->y0, (a)->x1, (a)->y1)
#define PRINT_AFFINE(s,a) g_print ("GPMP %s %g %g %g %g %g %g\n", (s), *(a), *((a) + 1), *((a) + 2), *((a) + 3), *((a) + 4), *((a) + 5))
#else
#define PRINT_2(s,a,b)
#define PRINT_DRECT(s,a)
#define PRINT_AFFINE(s,a)
#endif

static void
create_toplevel (GnomePrintMasterPreview *mp)
{
	GPMPPrivate *priv;
	GtkWidget *tb;
	gint width, height;

	priv = (GPMPPrivate *) mp->priv;

	width  = MIN (mp->paw + PAGE_PAD * 3, gdk_screen_width () - 40);
	height = MIN (mp->pah + PAGE_PAD * 3, gdk_screen_height () - 40);

	g_print ("width %d height %d\n", width, height);

	gtk_window_set_policy (GTK_WINDOW (mp), TRUE, TRUE, FALSE);
	gtk_widget_set_usize (GTK_WIDGET (mp), width, height);

	/* This is not very beautiful, but works for now */

	tb = gtk_toolbar_new ();
	gtk_widget_show (tb);
	gtk_box_pack_start (GTK_BOX (mp->vbox), tb, FALSE, FALSE, 0);

	gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_PRINT, _("Prints the current file"), "", 
				  GTK_SIGNAL_FUNC (preview_file_print_cmd), mp, 0);
	gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_CLOSE, _("Closes print preview window"), "", 
				  GTK_SIGNAL_FUNC (preview_close_cmd), mp, 1);
	mp->bpf = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_GOTO_FIRST, _("Shows the first page"), "", 
					    GTK_SIGNAL_FUNC (preview_first_page_cmd), mp, 2);
	mp->bpp = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_GO_BACK, _("Shows the previous page"), "", 
					    GTK_SIGNAL_FUNC (preview_prev_page_cmd), mp, 3);
	mp->bpn = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_GO_FORWARD, _("Shows the next page"), "", 
					    GTK_SIGNAL_FUNC (preview_next_page_cmd), mp, 4);
	mp->bpl = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_GOTO_LAST, _("Shows the last page"), "", 
					    GTK_SIGNAL_FUNC (preview_last_page_cmd), mp, 5);
	mp->bz1 = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_ZOOM_100, _("Zooms 1:1"), "", 
					    GTK_SIGNAL_FUNC (preview_zoom_100_cmd), mp, 6);
	mp->bzf = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_ZOOM_FIT, _("Zooms to fit the whole page"), "", 
					    GTK_SIGNAL_FUNC (preview_zoom_fit_cmd), mp, 7);
	mp->bzi = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_ZOOM_IN, _("Zooms the page in"), "", 
					    GTK_SIGNAL_FUNC (gpmp_zoom_in_cmd), mp, 8);
	mp->bzo = gtk_toolbar_insert_stock (GTK_TOOLBAR (tb), GTK_STOCK_ZOOM_OUT, _("Zooms the page out"), "", 
					    GTK_SIGNAL_FUNC (gpmp_zoom_out_cmd), mp, 9);
}

/*
  The GnomePrintMasterPreview object
*/

static void gnome_print_master_preview_class_init (GnomePrintMasterPreviewClass *klass);
static void gnome_print_master_preview_init (GnomePrintMasterPreview *pmp);

static GtkWindowClass *parent_class;

GtkType
gnome_print_master_preview_get_type (void)
{
	static GtkType print_master_preview_type = 0;
	
	if (!print_master_preview_type) {
		GtkTypeInfo print_master_preview_info = {
			"GnomePrintMasterPreview",
			sizeof (GnomePrintMasterPreview),
			sizeof (GnomePrintMasterPreviewClass),
			(GtkClassInitFunc) gnome_print_master_preview_class_init,
			(GtkObjectInitFunc) gnome_print_master_preview_init,
			NULL, NULL, NULL
		};
		
		print_master_preview_type = gtk_type_unique (GTK_TYPE_WINDOW, &print_master_preview_info);
	}
	
	return print_master_preview_type;
}

static void
gnome_print_master_preview_finalize (GObject *object)
{
	GnomePrintMasterPreview *pmp;
	GPMPPrivate *priv;

	pmp = GNOME_PRINT_MASTER_PREVIEW (object);
	priv = pmp->priv;

	if (priv && priv->preview != NULL) {
		g_object_unref (G_OBJECT (priv->preview));
		priv->preview = NULL;
	}

	if (priv && priv->master != NULL) {
		g_object_unref (G_OBJECT (priv->master));
		priv->master = NULL;
	}

	if (priv) {
		g_free (priv);
		pmp->priv = NULL;
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_print_master_preview_class_init (GnomePrintMasterPreviewClass *klass)
{
	GObjectClass *object_class;
	
	object_class = (GObjectClass *) klass;

	parent_class = gtk_type_class (GTK_TYPE_WINDOW);

	object_class->finalize = gnome_print_master_preview_finalize;
}

static void
gnome_print_master_preview_init (GnomePrintMasterPreview *mp)
{
	GPMPPrivate *priv;

	mp->priv = priv = g_new0 (GPMPPrivate, 1);
	priv->current_page = -1;

	/* Main VBox */
	mp->vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (mp->vbox);
	gtk_container_add (GTK_CONTAINER (mp), mp->vbox);

	mp->zoom = 1.0;
}

GtkWidget *
gnome_print_master_preview_new (GnomePrintMaster *gpm, const guchar *title)
{
	GnomePrintMasterPreview *gpmp;
	GPMPPrivate *priv;
	GnomePrintConfig *config;
	gint orientation;
	gchar *text;

	g_return_val_if_fail (gpm != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINT_MASTER (gpm), NULL);

	if (!title) title = _("Gnome Print Preview");

	gpmp = gtk_type_new (GNOME_TYPE_PRINT_MASTER_PREVIEW);
	gtk_signal_connect (GTK_OBJECT (gpmp), "delete_event", GTK_SIGNAL_FUNC (preview_delete_event), gpmp);
	gtk_window_set_title (GTK_WINDOW (gpmp), title);

	priv = (GPMPPrivate *) gpmp->priv;

	priv->master = gpm;
	g_object_ref (G_OBJECT (gpm));

	gpmp_parse_layout (gpmp);
	create_toplevel (gpmp);
	create_preview_canvas (gpmp);

	/* this zooms to fit, once we know how big the window actually is */
	gtk_signal_connect (GTK_OBJECT (priv->canvas), "realize",
			    GTK_SIGNAL_FUNC (preview_zoom_fit_cmd), gpmp);

	gpmp->pages = gnome_print_master_get_pages (gpm);

	priv->pagecount = gnome_print_master_get_pages (gpm);

	goto_page (gpmp, 0);
	
	text = g_strdup_printf ("/%d", priv->pagecount);
	gtk_label_set_text (GTK_LABEL (priv->last), text);
	g_free (text);

	return (GtkWidget *) gpmp;
}

#if 0
/**
 * gnome_print_master_preview_new:
 * @gpm: A GnomePrintMaster which has been printed to and completed.
 * @title: Window title for the preview window.
 * 
 * Create a new preview window widget.  The preview has inbuilt
 * buttons for zooming and panning, and menu's to print.
 * 
 * Return value: A newly created GnomePrintMasterPreview widget.
 **/
GnomePrintMasterPreview *
gnome_print_master_preview_new (GnomePrintMaster *gpm, const guchar *title)
{
	return gnome_print_master_preview_new_with_orientation (gpm, title, FALSE);
}

/**
 * gnome_print_master_preview_new_with_orientation:
 * @gpm: A GnomePrintMaster which has been printed to and completed.
 * @title: Window title for the preview window.
 * @landscape: TRUE to rotate paper.
 * 
 * Create a new preview window widget.  The preview has inbuilt
 * buttons for zooming and panning, and menu's to print.
 * 
 * Return value: A newly created GnomePrintMasterPreview widget.
 **/
GnomePrintMasterPreview *
gnome_print_master_preview_new_with_orientation (GnomePrintMaster *gpm,
						 const char *title,
						 gboolean landscape)
{
	GnomePrintMasterPreview *pmp;
	Private *pp;
	char text[4 * sizeof (int)];

	pmp = GNOME_PRINT_MASTER_PREVIEW (gtk_type_new (gnome_print_master_preview_get_type ()));

	gnome_app_construct (GNOME_APP (pmp), "preview_window", title);

	pp = pmp->priv;
	pp->landscape = landscape;

	pp->master = gpm;
	gtk_object_ref (GTK_OBJECT (gpm));

	create_toplevel (pmp);
	create_preview_canvas (pmp);

	/* this zooms to fit, once we know how big the window actually is */
	gtk_signal_connect (GTK_OBJECT (pp->canvas), "realize",
			   GTK_SIGNAL_FUNC (preview_zoom_fit_cmd), pmp);

	pp->pagecount = gnome_print_master_get_pages (gpm);

	goto_page (pmp, 0);
	
	g_snprintf (text, 4 * sizeof (int), "/%d", pp->pagecount);
	gtk_label_set_text (GTK_LABEL (pp->last), text);

	return pmp;
}
#endif

static void
gpmp_parse_layout (GnomePrintMasterPreview *mp)
{
	GPMPPrivate *priv;
	GnomePrintConfig *config;
	GnomePrintLayoutData *lyd;

	priv = (GPMPPrivate *) mp->priv;

	/* Calculate layout-compensated page dimensions */
	mp->paw = GPMP_A4_WIDTH;
	mp->pah = GPMP_A4_HEIGHT;
	art_affine_identity (mp->pa2ly);
	config = gnome_print_master_get_config (priv->master);
	lyd = gnome_print_config_get_layout_data (config, NULL, NULL, NULL, NULL);
	gnome_print_config_unref (config);
	if (lyd) {
		GnomePrintLayout *ly;
		ly = gnome_print_layout_new_from_data (lyd);
		if (ly) {
			gdouble pp2lyI[6], pa2pp[6];
			gdouble expansion;
			ArtDRect pp, ap, tp;
			/* Find paper -> layout transformation */
			art_affine_invert (pp2lyI, ly->LYP[0].matrix);
			PRINT_AFFINE ("pp2ly:", &pp2lyI[0]);
			/* Find out, what the page dimensions should be */
			expansion = art_affine_expansion (pp2lyI);
			if (expansion > 1e-6) {
				/* Normalize */
				pp2lyI[0] /= expansion;
				pp2lyI[1] /= expansion;
				pp2lyI[2] /= expansion;
				pp2lyI[3] /= expansion;
				pp2lyI[4] = 0.0;
				pp2lyI[5] = 0.0;
				PRINT_AFFINE ("pp2lyI:", &pp2lyI[0]);
				/* Find page dimensions relative to layout */
				pp.x0 = 0.0;
				pp.y0 = 0.0;
				pp.x1 = lyd->pw;
				pp.y1 = lyd->ph;
				art_drect_affine_transform (&tp, &pp, pp2lyI);
				/* Compensate with expansion */
				mp->paw = tp.x1 - tp.x0;
				mp->pah = tp.y1 - tp.y0;
				PRINT_2 ("Width & Height", mp->paw, mp->pah);
			}
			/* Now compensate with feed orientation */
			art_affine_invert (pa2pp, ly->PP2PA);
			PRINT_AFFINE ("pa2pp:", &pa2pp[0]);
			art_affine_multiply (mp->pa2ly, pa2pp, pp2lyI);
			PRINT_AFFINE ("pa2ly:", &mp->pa2ly[0]);
			/* Finally we need translation factors */
			/* Page box in normalized layout */
			pp.x0 = 0.0;
			pp.y0 = 0.0;
			pp.x1 = lyd->pw;
			pp.y1 = lyd->ph;
			art_drect_affine_transform (&ap, &pp, ly->PP2PA);
			art_drect_affine_transform (&tp, &ap, mp->pa2ly);
			PRINT_DRECT ("RRR:", &tp);
			mp->pa2ly[4] -= tp.x0;
			mp->pa2ly[5] -= tp.y0;
			PRINT_AFFINE ("pa2ly:", &mp->pa2ly[0]);
			/* Now, if master does PA2LY LY2PA concat it ends with scaled identity */
			gnome_print_layout_free (ly);
		}
		gnome_print_layout_data_free (lyd);
	}
}
