#include "sound-view.h"
#include <gtk/gtkctree.h>
#include <gtk/gtkscrolledwindow.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>
#include <config.h>

static GtkVBoxClass *sound_view_parent_class;

struct _SoundViewPrivate
{
	GtkCTree *ctree;
	SoundProperties *props;
	GtkWidget *entry;
	GHashTable *data;
	gboolean ignore_changed;
};

typedef struct 
{
	gchar *sort_data;

	enum { CATEGORY, EVENT } type;
	union {
		gchar *category;
		SoundEvent *event;
	} d;
} RowData;

static void sound_view_reload (SoundView *view);
static void select_row_cb (GtkCTree *ctree, GtkCTreeNode *row,
			   gint column, SoundView *view);

static void
sound_view_destroy (GtkObject *object)
{
	SoundView *view = SOUND_VIEW (object);

	g_hash_table_destroy (view->priv->data);
	g_free (view->priv);

	if (GTK_OBJECT_CLASS (sound_view_parent_class)->destroy)
		GTK_OBJECT_CLASS (sound_view_parent_class)->destroy (object);
}

static void
sound_view_class_init (GtkObjectClass *object_class)
{
	sound_view_parent_class = gtk_type_class (gtk_vbox_get_type ());

	object_class->destroy = sound_view_destroy;
}

static gint 
compare_func (GtkCList *l, gconstpointer ptr1, gconstpointer ptr2)
{
	const GtkCListRow *row1 = ptr1, *row2 = ptr2;
	const RowData *a = row1->data, *b = row2->data;

	if (!strcmp (a->sort_data, "gnome"))
		return -1;
	else if (!strcmp (b->sort_data, "gnome"))
		return 1;
	else
		return g_ascii_strcasecmp (a->sort_data, b->sort_data);
}

static void
play_cb (GtkButton *button, SoundView *view)
{
	RowData *data;
	SoundEvent *event;
	gchar *filename;

	if (!GTK_CLIST (view->priv->ctree)->selection)
		return;
	
	data = GTK_CLIST_ROW (&(GTK_CTREE_NODE (GTK_CLIST (view->priv->ctree)->selection->data))->list)->data;
	event = data->d.event;
	g_assert (event->file);
	if (!event->file)
		return;

	g_print ("%s\n", event->file);
	if ((event->file[0] == '/') || g_file_exists (event->file))
		filename = g_strdup (event->file);
	else
		filename = gnome_sound_file (event->file);
	
	g_print ("%s\n", filename);
	if (filename && g_file_exists (filename))
		gnome_sound_play (filename);
	else
	{
		GtkWidget *md;
		gchar *msg;

		if (filename && filename[0] == '/')
			msg = _("The sound file for this event does not exist.");
		else
			msg = _("The sound file for this event does not exist.\n"
				"You may want to install the gnome-audio package\n"
				"for a set of default sounds.");
		md = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, msg);
		gtk_dialog_run (GTK_DIALOG (md));
	}

	g_free (filename);
}

static void
entry_changed_cb (GtkEditable *entry, SoundView *view)
{
	const gchar *file;
	GtkCTreeNode *node;
	GtkCListRow *row;
	RowData *data;
	SoundEvent *event;
	
	if (view->priv->ignore_changed)
		return;

	if (!GTK_CLIST (view->priv->ctree)->selection)
		return;

	node = GTK_CLIST (view->priv->ctree)->selection->data;
	row = node->list.data;
	data = row->data;
	event = data->d.event;

	file = gtk_entry_get_text (GTK_ENTRY (entry));
	sound_event_set_file (event, (gchar *) file);
	sound_properties_event_changed (view->priv->props, event);
}

static void
event_changed_cb (SoundProperties *props, SoundEvent *event, SoundView *view)
{
	RowData *data = g_hash_table_lookup (view->priv->data, event);
	int row = gtk_clist_find_row_from_data (GTK_CLIST (view->priv->ctree),
						data);
	gtk_clist_set_text (GTK_CLIST (view->priv->ctree), row, 1,
			    event->file);
}

static void
sound_view_init (GtkObject *object)
{
	SoundView *view = SOUND_VIEW (object);
	GtkWidget *widget, *hbox, *bbox, *button;
	GtkWidget *vbox;
	gchar *path;
	int i;
	static const gchar *titles[] = { N_("Event"), N_("File to Play"), NULL };
	
	for (i = 0; titles[i] != NULL; i++)
		titles[i] = gettext (titles[i]);

	view->priv = g_new0 (SoundViewPrivate, 1);

	view->priv->data = g_hash_table_new (g_direct_hash, g_direct_equal);
	
	view->priv->ctree = GTK_CTREE (gtk_ctree_new_with_titles (2, 0, (gchar**) titles));
	gtk_clist_set_selection_mode (GTK_CLIST (view->priv->ctree), GTK_SELECTION_BROWSE);
	gtk_clist_set_column_width (GTK_CLIST (view->priv->ctree), 0, 200);
	//gtk_clist_set_column_auto_resize (GTK_CLIST (view->priv->ctree), 0, TRUE);
	gtk_ctree_set_line_style (view->priv->ctree, GTK_CTREE_LINES_DOTTED);

	gtk_clist_set_compare_func (GTK_CLIST (view->priv->ctree), compare_func);
	g_signal_connect (G_OBJECT (view->priv->ctree), "tree_select_row",
			  (GCallback) select_row_cb, view);

	widget = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (widget),
					GTK_POLICY_ALWAYS,
					GTK_POLICY_ALWAYS);
	gtk_widget_set_usize (widget, 350, -1);
	gtk_container_add (GTK_CONTAINER (widget), GTK_WIDGET (view->priv->ctree));
	gtk_box_pack_start (GTK_BOX (view), widget, TRUE, TRUE, 0);
	gtk_widget_show_all (widget);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (view), hbox, FALSE, FALSE, GNOME_PAD);
	button = gtk_button_new ();
	bbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (bbox), gtk_image_new_from_stock (GNOME_STOCK_PIXMAP_VOLUME, GTK_ICON_SIZE_BUTTON), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (bbox), gtk_label_new (_("Play")), FALSE, FALSE, GNOME_PAD);
	gtk_container_add (GTK_CONTAINER (button), bbox);
	gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, FALSE, GNOME_PAD_SMALL);
	gtk_object_set_data (GTK_OBJECT (button), "gnome_disable_sound_events", GINT_TO_POINTER(1));
	g_signal_connect (G_OBJECT (button), "clicked", (GCallback) play_cb, view);

	view->priv->entry = gnome_file_entry_new (NULL, _("Select sound file"));
	path = gnome_datadir_file ("sounds/");
	gnome_file_entry_set_default_path (GNOME_FILE_ENTRY (view->priv->entry), path);
	g_free (path);

	g_signal_connect (G_OBJECT (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (view->priv->entry))), "changed",
			  (GCallback) entry_changed_cb, view);
				       
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), view->priv->entry, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, FALSE, GNOME_PAD_SMALL);
	gtk_widget_show_all (hbox);
}

GtkType
sound_view_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		GtkTypeInfo info =
		{
			"SoundView",
			sizeof (SoundView),
			sizeof (SoundViewClass),
			(GtkClassInitFunc) sound_view_class_init,
			(GtkObjectInitFunc) sound_view_init,
			NULL,
			NULL
		};
		
		type = gtk_type_unique (gtk_vbox_get_type (), &info);
	}

	return type;
}

GtkWidget*
sound_view_new (SoundProperties *props)
{
	SoundView *view = gtk_type_new (sound_view_get_type ());
	view->priv->props = props;
	sound_view_reload (view);

	g_signal_connect (GTK_OBJECT (props), "event_changed",
			  (GCallback) event_changed_cb, view);

	return GTK_WIDGET (view);
}

static void
foreach_cb (gchar *category, gchar *desc, GList *events, SoundView *view)
{
	GtkCTreeNode *node, *prev = NULL;
	GList *l;
	RowData *data;
	gchar *text[] = { NULL, NULL, NULL};

	text[0] = desc;
	node = gtk_ctree_insert_node (view->priv->ctree,
			              NULL, NULL,
				      text, GNOME_PAD_SMALL,
				      NULL, NULL, NULL, NULL,
				      FALSE, TRUE);
	gtk_ctree_node_set_selectable (view->priv->ctree,
				       node, FALSE);
	data = g_new0 (RowData, 1);
	data->type = CATEGORY;
	/* We want the "gnome" category to be first, so we don't set the
	 * sort data to description for that */
	data->sort_data = (!strcmp (category, "gnome")) ? category : desc;
	data->d.category = category;
	gtk_ctree_node_set_row_data_full (view->priv->ctree,
				    	  node, data, g_free);

	for (l = events; l != NULL; l = l->next)
	{
		SoundEvent *event = l->data;
		text[0] = event->desc;
		text[1] = event->file;
		prev = gtk_ctree_insert_node (view->priv->ctree,
					      node, prev,
					      text, GNOME_PAD_SMALL,
					      NULL, NULL, NULL, NULL,
					      TRUE, FALSE);

		data = g_new0 (RowData, 1);
		data->type = EVENT;
		data->sort_data = event->desc;
		data->d.event = event;
		gtk_ctree_node_set_row_data_full (view->priv->ctree,
						  prev, data, g_free);
		g_hash_table_insert (view->priv->data, event, data);
	}
}

static void
sound_view_reload (SoundView *view)
{
	g_return_if_fail (SOUND_IS_VIEW (view));
	
	gtk_clist_freeze (GTK_CLIST (view->priv->ctree));
	gtk_clist_clear (GTK_CLIST (view->priv->ctree));
	sound_properties_category_foreach (view->priv->props,
					   foreach_cb, view);
	gtk_ctree_sort_recursive (view->priv->ctree, NULL);
	gtk_clist_thaw (GTK_CLIST (view->priv->ctree));
}

static void
select_row_cb (GtkCTree *ctree, GtkCTreeNode *row,
	       gint column, SoundView *view)
{
	RowData *data = gtk_ctree_node_get_row_data (ctree, row);

	/* Weird ctree events */
	if (!data)
		return;

	if (data->type == EVENT)
	{
		gchar *text = data->d.event->file;
		if (text && text[0] == '/')
		{
			gchar *dirname;
			dirname = g_path_get_dirname (text);
			gnome_file_entry_set_default_path (GNOME_FILE_ENTRY (view->priv->entry), dirname);
			g_free (dirname);
			text = g_path_get_basename (text);
		} else {
			text = g_strdup (text);
		}
	
		view->priv->ignore_changed = TRUE;
		gtk_entry_set_text (GTK_ENTRY (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (view->priv->entry))), text);
		view->priv->ignore_changed = FALSE;
		g_free (text);
	}
}
