/* GnomENIUS Calculator
 * Copyright (C) 1997, 1998 the Free Software Foundation.
 *
 * Author: George Lebl
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the  Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */
#include <gnome.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <glib.h>

#include "calc.h"
#include "eval.h"
#include "util.h"
#include "dict.h"
#include "funclib.h"

#include "mpwrap.h"


extern FILE *yyin;
extern char *yytext;
extern int yydebug;

/* stack ... has to be global:-( */
evalstack_t evalstack=NULL;

/*error .. global as well*/
calc_error_t error_num;

/*the current state of the calculator*/
calcstate_t calcstate;

/*error reporting function*/
void (*errorout)(char *)=NULL;

/*spit out ordinary primitives, such as plus, minus etc*/
char *
str_normalprim(char *s,char *p, tree_t *l, tree_t *r)
{
	s=makeexprstr(s,l);
	s=appendstr(s,p);
	s=makeexprstr(s,r);
	return s;
}

/*appends to S the primitive op and it's arguments*/
char *
str_prim(char *s, int op, tree_t *l, tree_t *r, tree_t *sr)
{
	char tmp[3];
	s=appendstr(s,"(");
	switch(op) {
		case E_SEPAR: s=str_normalprim(s,";",l,r); break;
		case E_EQUALS: s=str_normalprim(s,"=",l,r); break;
		case E_PLUS: s=str_normalprim(s,"+",l,r); break;
		case E_MINUS: s=str_normalprim(s,"-",l,r); break;
		case E_MUL: s=str_normalprim(s,"*",l,r); break;
		case E_DIV: s=str_normalprim(s,"/",l,r); break;
		case E_MOD: s=str_normalprim(s,"%",l,r); break;
		case E_NEG: s=str_normalprim(s,"~",l,r); break;
		case E_EXP: s=str_normalprim(s,"^",l,r); break;
		case E_FACT: s=str_normalprim(s,"!",l,r); break;
		case E_FUNCDEF:
			if(l->type!=ACTION_NODE ||
				l->data.action.type!=FUNCTION_TYPE) {
				s=appendstr(s,"?error?)");
				return s;
			}

			tmp[2]='\0';
			tmp[1]='\\';
			tmp[0]=(char)(l->data.action.data.function.func->nargs)
				+'0';
			s=appendstr(s,tmp);
			s=appendstr(s,l->data.action.data.function.func->id);
			s=appendstr(s,"{");
			s=makeexprstr(s,r);
			s=appendstr(s,"}");
			break;
		case E_IF_CONS:
			s=appendstr(s,"if(");
			s=makeexprstr(s,l);
			s=appendstr(s,")(");
			s=makeexprstr(s,r);
			s=appendstr(s,")");
			break;
		case E_IFELSE_CONS:
			s=appendstr(s,"ifelse(");
			s=makeexprstr(s,l);
			s=appendstr(s,")(");
			s=makeexprstr(s,r);
			s=appendstr(s,")(");
			s=makeexprstr(s,sr);
			s=appendstr(s,")");
			break;
		case E_WHILE_CONS:
			s=appendstr(s,"while(");
			s=makeexprstr(s,l);
			s=appendstr(s,")(");
			s=makeexprstr(s,r);
			s=appendstr(s,")");
			break;
		case E_EQ_CMP: s=str_normalprim(s,"==",l,r); break;
		case E_NE_CMP: s=str_normalprim(s,"!=",l,r); break;
		case E_CMP_CMP: s=str_normalprim(s,"<=>",l,r); break;
		case E_LT_CMP: s=str_normalprim(s,"<",l,r); break;
		case E_GT_CMP: s=str_normalprim(s,">",l,r); break;
		case E_LE_CMP: s=str_normalprim(s,"<=",l,r); break;
		case E_GE_CMP: s=str_normalprim(s,">=",l,r); break;
		case E_LOGICAL_AND: s=str_normalprim(s,"and",l,r); break;
		case E_LOGICAL_OR: s=str_normalprim(s,"or",l,r); break;
		case E_LOGICAL_XOR: s=str_normalprim(s,"xor",l,r); break;
		case E_LOGICAL_NOT: s=str_normalprim(s,"not",l,r); break;
		default:
			s=appendstr(s,"<???>");
			break;
	}
	s=appendstr(s,")");
	return s;
}

/*make a string representation of an expression*/
char *
makeexprstr(char *s,tree_t *n)
{
	char *p;
	int i;

	if(!n)
		return s;

	if(n->type==NUMBER_NODE) {
		p=mpw_getstring(n->data.val,calcstate.max_digits,
			calcstate.scientific_notation,
			calcstate.results_as_floats);
		if(!s)
			return p;
		s=appendstr(s,p);
		g_free(p);
		return s;
	} else if(n->type==ACTION_NODE) {
		if(n->data.action.type==PRIMITIVE_TYPE) {
			s=str_prim(s,n->data.action.data.primitive,
				n->left,n->right,n->secondright);
			return s;
		} else { /*type==FUNCTION_TYPE*/
			s=appendstr(s,n->data.action.data.function.func->id);
			if(n->data.action.data.function.func->nargs<=0)
				return s;
			if(n->data.action.data.function.args) {
				s=appendstr(s,"(");
				s=makeexprstr(s,
					n->data.action.data.function.args[0]);
				for(i=1;i<n->data.action.data.function.func->
					nargs;i++) {
					s=appendstr(s,",");
					s=makeexprstr(s,
						n->data.action.data.
						function.args[i]);
				}
				s=appendstr(s,")");
			} else {
				s=appendstr(s,"(?args?)");
			}
			return s;
		}
	}
	s=appendstr(s,"(???)");
	return s;
}

/*add the right parenthesis and brackets to the end of the expression*/
char *
addparenth(char *s)
{
	GList *stack = NULL;
	char *p;

	for(p=s;*p!='\0';p++) {
		switch(*p) {
			case '(':
				stack_push(&stack,")");
				break;
			case '{':
				stack_push(&stack,"}");
				break;
			case ')':
				if(!stack)
					return s; /*too many trailing*/
				if(*(char *)stack_pop(&stack) != ')') {
					if(stack) g_list_free(stack);
					return s; /*non matching*/
				}
				break;
			case '}':
				if(!stack)
					return s; /*too many trailing*/
				if(*(char *)stack_pop(&stack) != '}') {
					if(stack) g_list_free(stack);
					return s; /*non matching*/
				}
				break;
		}
	}
	while(stack)
		s=appendstr(s,(char *)stack_pop(&stack));
	return s;
}

char *
evalexp(char * str, calcstate_t state,void (*errorfunc)(char *))
{
	int fd[2];
	char * p;

	/*init the context stack and clear out any stale dictionaries
	  except the global one, if this is the first time called it
	  will also register the builtin routines with the global
	  dictionary*/
	d_singlecontext();
	
	errorout=errorfunc;

	error_num=NO_ERROR;

	/*set the state variable for calculator*/
	calcstate=state;
	
	mpw_init_mp();
	mpw_set_default_prec(state.float_prec);

	pipe(fd);
	yyin=fdopen(fd[0],"r");

	if(calcstate.notation_in==INFIX_NOTATION)
		write(fd[1],"infix ",6);
	else if(calcstate.notation_in==POSTFIX_NOTATION)
		write(fd[1],"postfix ",8);
	else
		write(fd[1],"prefix ",7);
	write(fd[1],str,strlen(str));
	close(fd[1]);

	/*yydebug=TRUE; */ /*turn debugging of parsing on here!*/
	while(yyparse() && !feof(yyin))
		;

	close(fd[0]);

	/*catch parsing errors*/
	if(error_num!=NO_ERROR) {
		while(evalstack)
			freetree(stack_pop(&evalstack));
		return NULL;
	}

	/*stack is supposed to have only ONE entry*/
	if(g_list_length(evalstack)!=1) {
		printf("stack size: %d\n",(int)g_list_length(evalstack));
		while(evalstack)
			freetree(stack_pop(&evalstack));
		(*errorout)(_("ERROR: Probably corrupt stack!"));
		return NULL;
	}

	evalstack->data = evalnode(evalstack->data);

	/*catch evaluation errors*/
	if(error_num!=NO_ERROR) {
		while(evalstack)
			freetree(stack_pop(&evalstack));
		return NULL;
	}

	p=makeexprstr(NULL,evalstack->data);

	freetree(stack_pop(&evalstack));

	return p;
}

void
yyerror(char *s)
{
	char *out=NULL;
	out=appendstr(out,_("ERROR: "));
	out=appendstr(out,s);
	out=appendstr(out,_(" before '"));
	out=appendstr(out,yytext);
	out=appendstr(out,"'");
	
	(*errorout)(out);
	g_free(out);
	error_num=PARSE_ERROR;
}

