/*
 * Copyright (C) 2011 Collabora Ltd.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors: Raul Gutierrez Segales <raul.gutierrez.segales@collabora.co.uk>
 *
 */

using Tracker.Sparql;
using TrackerTest;
using Folks;
using Gee;

public class SetPostalAddressesTests : Folks.TestCase
{
  private GLib.MainLoop _main_loop;
  private TrackerTest.Backend _tracker_backend;
  private IndividualAggregator _aggregator;
  private string _persona_fullname;
  private bool _postal_address_found;
  private PostalAddressFieldDetails _postal_address_fd;

  public SetPostalAddressesTests ()
    {
      base ("SetPostalAddressesTests");

      this._tracker_backend = new TrackerTest.Backend ();

      this.add_test ("test setting postal addresses ",
          this.test_set_postal_addresses);
    }

  public override void set_up ()
    {
    }

  public override void tear_down ()
    {
    }

  public void test_set_postal_addresses ()
    {
      this._main_loop = new GLib.MainLoop (null, false);
      Gee.HashMap<string, string> c1 = new Gee.HashMap<string, string> ();
      this._persona_fullname = "persona #1";

      c1.set (Trf.OntologyDefs.NCO_FULLNAME, this._persona_fullname);
      this._tracker_backend.add_contact (c1);

      var pa = new PostalAddress (null, null, null, null, null,
          null, null, null, null);
      pa.po_box = "12345";
      pa.locality = "locality";
      pa.postal_code = "code";
      pa.street = "some street";
      pa.extension = "some extension";
      pa.country = "some country";
      pa.region = "some region";
      this._postal_address_fd = new PostalAddressFieldDetails (pa);

      this._tracker_backend.set_up ();

      this._postal_address_found = false;

      this._test_set_postal_addresses_async.begin ();

      Timeout.add_seconds (5, () =>
        {
          this._main_loop.quit ();
          assert_not_reached ();
        });

      this._main_loop.run ();

      assert (this._postal_address_found);

     this._tracker_backend.tear_down ();
    }

  private async void _test_set_postal_addresses_async ()
    {
      var store = BackendStore.dup ();
      yield store.prepare ();
      this._aggregator = new IndividualAggregator ();
      this._aggregator.individuals_changed_detailed.connect
          (this._individuals_changed_cb);
      try
        {
          yield this._aggregator.prepare ();
        }
      catch (GLib.Error e)
        {
          GLib.warning ("Error when calling prepare: %s\n", e.message);
        }
    }

  private void _individuals_changed_cb (
       MultiMap<Individual?, Individual?> changes)
    {
      var added = changes.get_values ();
      var removed = changes.get_keys ();

      foreach (var i in added)
        {
          assert (i != null);

          if (i.full_name == this._persona_fullname)
            {
              i.notify["postal-addresses"].connect (this._notify_postal_cb);

              var addresses = new HashSet<PostalAddressFieldDetails> (
                  AbstractFieldDetails<PostalAddress>.hash_static,
                  AbstractFieldDetails<PostalAddress>.equal_static);
              var pa = new PostalAddress (null, null, null, null, null,
                null, null, null, null);
              pa.po_box = this._postal_address_fd.value.po_box;
              pa.locality = this._postal_address_fd.value.locality;
              pa.postal_code = this._postal_address_fd.value.postal_code;
              pa.street = this._postal_address_fd.value.street;
              pa.extension = this._postal_address_fd.value.extension;
              pa.country = this._postal_address_fd.value.country;
              pa.region = this._postal_address_fd.value.region;
              var pafd = new PostalAddressFieldDetails (pa);

              addresses.add (pafd);

              foreach (var p in i.personas)
                {
                  ((PostalAddressDetails) p).postal_addresses = addresses;
                }
            }
        }

      assert (removed.size == 1);

      foreach (var i in removed)
        {
          assert (i == null);
        }
    }

  private void _notify_postal_cb (Object individual_obj, ParamSpec ps)
    {
      Folks.Individual i = (Folks.Individual) individual_obj;
      if (i.full_name == this._persona_fullname)
        {
          foreach (var pafd in i.postal_addresses)
            {
              /* we don't care if UIDs differ for this test */
              this._postal_address_fd.id = pafd.id;
              if (pafd.equal (this._postal_address_fd))
                {
                  this._postal_address_found = true;
                  this._main_loop.quit ();
                }
            }
        }
    }
}

public int main (string[] args)
{
  Test.init (ref args);

  TestSuite root = TestSuite.get_root ();
  root.add_suite (new SetPostalAddressesTests ().get_suite ());

  Test.run ();

  return 0;
}
