# This is a shell script that calls functions and scripts from
# tml@iki.fi's personal work environment. It is not expected to be
# usable unmodified by others, and is included only for reference.

MOD=evolution-exchange
VER=2.22.2
THIS=$MOD-$VER
HEX=`echo $THIS | md5sum | cut -d' ' -f1`

DEPS=`/devel/src/tml/latest.sh intltool glib atk pixman cairo pango gtk+ libglade libIDL ORBit2 libbonobo GConf libbonoboui gnome-mime-data libgnome libgnomeui gnome-vfs gnutls libsoup evolution-data-server gnome-icon-theme libart_lgpl libgnomecanvas gail gtkhtml evolution`

sed -e 's/need_relink=yes/need_relink=no # no way --tml/' <ltmain.sh >ltmain.temp && mv ltmain.temp ltmain.sh

# Avoid using "file" in libtool. Otherwise libtool won't create a
# shared library, and give the extremely confusing and annoying
# warning "Trying to link with static lib archive [...] But I can only
# do this if you have shared version of the library, which you do not
# appear to have." Screw libtool. I know what I am doing. Yes, I do
# want to link with a static libintl now. (The proxy-libintl libintl
# wrapper.)

sed -e 's!file /!dont-want-to-use-file!' <configure >configure.temp && mv configure.temp configure

patch -p0 <<'EOF'
Index: camel/camel-stub.h
===================================================================
--- camel/camel-stub.h	(revision 1672)
+++ camel/camel-stub.h	(working copy)
@@ -35,6 +35,7 @@
 
 	CamelOperation *op;      /* for cancelling */
 	pthread_t status_thread;
+	gboolean have_status_thread;
 } CamelStub;
 
 typedef struct {
Index: camel/camel-stub-marshal.c
===================================================================
--- camel/camel-stub-marshal.c	(revision 1672)
+++ camel/camel-stub-marshal.c	(working copy)
@@ -27,6 +27,20 @@
 #include <stdlib.h>
 #include <unistd.h>
 
+
+#include <glib.h>
+
+#ifndef G_OS_WIN32
+#define CLOSESOCKET(s) close (s)
+#else
+#define CLOSESOCKET(s) closesocket (s)
+#include <winsock2.h>
+/* Note: pthread.h from pthreads-win32 also defines ETIMEDOUT (!), as
+ * this same value.
+ */
+#define ETIMEDOUT WSAETIMEDOUT
+#endif
+
 #include <camel/camel-file-utils.h>
 
 #include "camel-stub-marshal.h"
@@ -77,7 +91,7 @@
 void
 camel_stub_marshal_free (CamelStubMarshal *marshal)
 {
-	close (marshal->fd);
+	CLOSESOCKET (marshal->fd);
 	g_byte_array_free (marshal->out, TRUE);
 	g_byte_array_free (marshal->in, TRUE);
 	g_free (marshal);
@@ -90,7 +104,7 @@
 	ssize_t n;
 
 	do {
-		if ((n = camel_read (marshal->fd, buf + nread, len - nread)) <= 0) {
+		if ((n = camel_read_socket (marshal->fd, buf + nread, len - nread)) <= 0) {
 			if (errno != ETIMEDOUT)
 				break;
 			else
@@ -100,7 +114,7 @@
 	} while (nread < len);
 
 	if (nread < len) {
-		close (marshal->fd);
+		CLOSESOCKET (marshal->fd);
 		marshal->fd = -1;
 		return FALSE;
 	}
@@ -452,8 +466,8 @@
 	marshal->out->data[2] = (left >> 16) & 0xFF;
 	marshal->out->data[3] = (left >> 24) & 0xFF;
 
-	if (camel_write (marshal->fd, (char *) marshal->out->data, marshal->out->len) == -1) {
-		close (marshal->fd);
+	if (camel_write_socket (marshal->fd, (char *) marshal->out->data, marshal->out->len) == -1) {
+		CLOSESOCKET (marshal->fd);
 		marshal->fd = -1;
 		return -1;
 	}
Index: camel/Makefile.am
===================================================================
--- camel/Makefile.am	(revision 1672)
+++ camel/Makefile.am	(working copy)
@@ -9,6 +9,7 @@
 	$(CAMEL_CFLAGS)				\
 	$(SHELL_CFLAGS)				\
 	$(LIBEXCHANGE_CFLAGS)			\
+	-DPREFIX=\"$(prefix)\"			\
 	-DCONNECTOR_LOCALEDIR=\"$(localedir)\"	\
 	-DG_LOG_DOMAIN=\"camel-exchange-provider\"	
 
@@ -34,11 +35,13 @@
 	camel-stub-marshal.h			\
 	camel-stub.h
 
-libcamelexchange_la_LDFLAGS = -avoid-version -module
+libcamelexchange_la_LDFLAGS = -avoid-version -module $(NO_UNDEFINED)
 
 libcamelexchange_la_LIBADD = 			\
 	$(LDAP_LIBS)				\
 	$(LIBEXCHANGE_LIBS)			\
-	$(EXCHANGE_STORAGE_LIBS)
+	$(EXCHANGE_STORAGE_LIBS)		\
+	$(SOCKET_LIBS)				\
+	$(PTHREAD_LIB)
 
 EXTRA_DIST = libcamelexchange.urls
Index: camel/camel-exchange-provider.c
===================================================================
--- camel/camel-exchange-provider.c	(revision 1672)
+++ camel/camel-exchange-provider.c	(working copy)
@@ -25,6 +25,8 @@
 
 #include <string.h>
 
+#include <libedataserver/e-data-server-util.h>
+
 #include <glib/gi18n-lib.h>
 
 #include <camel/camel-provider.h>
@@ -36,6 +38,19 @@
 static guint exchange_url_hash (gconstpointer key);
 static gint exchange_url_equal (gconstpointer a, gconstpointer b);
 
+#ifdef G_OS_WIN32
+
+static const char *
+get_localedir (void)
+{
+	return e_util_replace_prefix (PREFIX, e_util_get_cp_prefix (), CONNECTOR_LOCALEDIR);
+}
+
+#undef CONNECTOR_LOCALEDIR
+#define CONNECTOR_LOCALEDIR get_localedir ()
+
+#endif
+
 CamelProviderConfEntry exchange_conf_entries[] = {
 	{ CAMEL_PROVIDER_CONF_SECTION_START, "mailcheck", NULL,
 	  N_("Checking for New Mail") },
Index: camel/camel-exchange-store.c
===================================================================
--- camel/camel-exchange-store.c	(revision 1672)
+++ camel/camel-exchange-store.c	(working copy)
@@ -27,11 +27,13 @@
 
 #include <glib/gi18n-lib.h>
 
+#include <camel/camel-session.h>
+#include <camel/camel-url.h>
+#include <libedataserver/e-data-server-util.h>
+
 #include "camel-exchange-store.h"
 #include "camel-exchange-folder.h"
 #include "camel-exchange-summary.h"
-#include <camel/camel-session.h>
-#include <camel/camel-url.h>
 
 #define SUBFOLDER_DIR_NAME     "subfolders"
 #define SUBFOLDER_DIR_NAME_LEN 10
@@ -320,30 +322,7 @@
 
 #define EXCHANGE_STOREINFO_VERSION 1
 
-/* SURF : Picked this from gal/util/e-util.c */
-/* This only makes a filename safe for usage as a filename.
-	It still may have shell meta-characters in it. */
 static void
-e_filename_make_safe (gchar *string)
-{
-	gchar *p, *ts;
-	gunichar c;
-
-	g_return_if_fail (string != NULL);
-	p = string;
-
-	while(p && *p) {
-		c = g_utf8_get_char (p);
-		ts = p;
-		p = g_utf8_next_char (p);
-		if (!g_unichar_isprint(c) || ( c < 0xff && strchr (" /'\"`&();|<>$%{}!", c&0xff ))) {
-			while (ts<p)
-				*ts++ = '_';
-		}
-	}
-}
-
-static void
 camel_exchange_get_password (CamelService *service, CamelException *ex)
 {
 	CamelSession *session = camel_service_get_session (service);
@@ -388,7 +367,7 @@
 exchange_connect (CamelService *service, CamelException *ex)
 {
 	CamelExchangeStore *exch = CAMEL_EXCHANGE_STORE (service);
-	char *real_user, *socket_path;
+	char *real_user, *socket_path, *dot_exchange_username, *user_at_host;
 	gchar *password = NULL;
 	guint32 connect_status;
 	gboolean online_mode = FALSE;
@@ -407,10 +386,15 @@
 			real_user++;
 		else
 			real_user = service->url->user;
-		socket_path = g_strdup_printf ("/tmp/.exchange-%s/%s@%s",
-					       g_get_user_name (),
-					       real_user, service->url->host);
-		e_filename_make_safe (strchr (socket_path + 5, '/') + 1);
+		dot_exchange_username = g_strdup_printf (".exchange-%s", g_get_user_name ());
+		user_at_host = g_strdup_printf ("%s@%s", real_user, service->url->host);
+		e_filename_make_safe (user_at_host);
+		socket_path = g_build_filename (g_get_tmp_dir (),
+						dot_exchange_username,
+						user_at_host,
+						NULL);
+		g_free (dot_exchange_username);
+		g_free (user_at_host);
 
 		exch->stub = camel_stub_new (socket_path, _("Evolution Exchange backend process"), ex);
 		g_free (socket_path);
@@ -505,7 +489,7 @@
 	folder_dir = exchange_path_to_physical (exch->storage_path, folder_name);
 
 	if (!camel_exchange_store_connected (exch, ex)) {
-		if (!folder_dir || access (folder_dir, F_OK) != 0) {
+		if (!folder_dir || !g_file_test (folder_dir, G_FILE_TEST_IS_DIR)) {
 			g_free (folder_dir);
 			camel_exception_setv (ex, CAMEL_EXCEPTION_STORE_NO_FOLDER,
 					      _("No such folder %s"), folder_name);
Index: camel/camel-exchange-summary.c
===================================================================
--- camel/camel-exchange-summary.c	(revision 1672)
+++ camel/camel-exchange-summary.c	(working copy)
@@ -22,7 +22,6 @@
 #endif
 
 #include <sys/stat.h>
-#include <sys/uio.h>
 #include <unistd.h>
 #include <errno.h>
 #include <string.h>
Index: camel/camel-stub.c
===================================================================
--- camel/camel-stub.c	(revision 1672)
+++ camel/camel-stub.c	(working copy)
@@ -32,9 +32,14 @@
 
 #include <errno.h>
 #include <string.h>
+
+#ifndef G_OS_WIN32
 #include <sys/socket.h>
 #include <sys/un.h>
 #include <unistd.h>
+#else
+#include <winsock2.h>
+#endif
 
 CamelStub *das_global_camel_stub;
 
@@ -49,6 +54,7 @@
 {
 	stub->read_lock = g_mutex_new ();
 	stub->write_lock = g_mutex_new ();
+	stub->have_status_thread = FALSE;
 }
 
 static void
@@ -57,7 +63,7 @@
 	if (stub->cmd)
 		camel_stub_marshal_free (stub->cmd);
 
-	if (stub->status_thread) {
+	if (stub->have_status_thread) {
 		void *unused;
 
 		/* When we close the command channel, the storage will
@@ -116,6 +122,8 @@
 	CamelStubMarshal *status_channel = stub->status;
 	guint32 retval;
 
+	stub->have_status_thread = TRUE;
+
 	stub->op = camel_operation_new (NULL, NULL);
 	camel_operation_register (stub->op);
 
@@ -138,9 +146,13 @@
 
 	camel_operation_unregister (stub->op);
 
+	stub->have_status_thread = FALSE;
+
 	return NULL;
 }
 
+#ifndef G_OS_WIN32
+
 static int
 connect_to_storage (CamelStub *stub, struct sockaddr_un *sa_un,
 		    CamelException *ex)
@@ -182,6 +194,68 @@
 	return fd;
 }
 
+#else
+
+static int
+connect_to_storage (CamelStub *stub, const char *socket_path,
+		    CamelException *ex)
+{
+	SOCKET fd;
+	struct sockaddr_in *sa_in;
+	gsize contents_length;
+	GError *error = NULL;
+	int rc;
+
+	if (!g_file_get_contents (socket_path, (gchar **) &sa_in,
+				  &contents_length, &error)) {
+		camel_exception_setv (ex, CAMEL_EXCEPTION_SERVICE_UNAVAILABLE,
+				      _("Count not read file '%s': %s"),
+				      socket_path, error->message);
+		g_error_free (error);
+		return -1;
+	}
+
+	if (contents_length != sizeof (*sa_in)) {
+		camel_exception_setv (ex, CAMEL_EXCEPTION_SERVICE_UNAVAILABLE,
+				      _("Wrong size file '%s'"),
+				      socket_path);
+		g_free (sa_in);
+		return -1;
+	}
+
+	fd = socket (AF_INET, SOCK_STREAM, 0);
+	if (fd == INVALID_SOCKET) {
+		camel_exception_setv (ex, CAMEL_EXCEPTION_SERVICE_UNAVAILABLE,
+				      _("Could not create socket: %s"),
+				      g_win32_error_message (WSAGetLastError ()));
+		return -1;
+	}
+	rc = connect (fd, (struct sockaddr *)sa_in, sizeof (*sa_in));
+	g_free (sa_in);
+
+	if (rc == SOCKET_ERROR) {
+		closesocket (fd);
+		if (WSAGetLastError () == WSAECONNREFUSED) {
+			/* The user has an account configured but the
+			 * backend isn't listening, which probably means that
+			 * he doesn't have a license.
+			 */
+			camel_exception_set (ex, CAMEL_EXCEPTION_USER_CANCEL,
+					     "Cancelled");
+		} else {
+			camel_exception_setv (ex, CAMEL_EXCEPTION_SERVICE_UNAVAILABLE,
+					      _("Could not connect to %s: %s"),
+					      stub->backend_name,
+					      g_win32_error_message (WSAGetLastError ()));
+		}
+		return -1;
+	}
+	return fd;
+}
+
+#endif
+
+
 /**
  * camel_stub_new:
  * @socket_path: path to the server's UNIX domain socket
@@ -198,9 +272,12 @@
 		CamelException *ex)
 {
 	CamelStub *stub;
+#ifndef G_OS_WIN32
 	struct sockaddr_un sa_un;
+#endif
 	int fd;
 
+#ifndef G_OS_WIN32
 	if (strlen (socket_path) > sizeof (sa_un.sun_path) - 1) {
 		camel_exception_setv (ex, CAMEL_EXCEPTION_SERVICE_UNAVAILABLE,
 				      _("Path too long: %s"), socket_path);
@@ -208,18 +285,27 @@
 	}
 	sa_un.sun_family = AF_UNIX;
 	strcpy (sa_un.sun_path, socket_path);
+#endif
 
 	stub = (CamelStub *)camel_object_new (CAMEL_STUB_TYPE);
 	stub->backend_name = g_strdup (backend_name);
 
+#ifndef G_OS_WIN32
 	fd = connect_to_storage (stub, &sa_un, ex);
+#else
+	fd = connect_to_storage (stub, socket_path, ex);
+#endif
 	if (fd == -1) {
 		camel_object_unref (CAMEL_OBJECT (stub));
 		return NULL;
 	}
 	stub->cmd = camel_stub_marshal_new (fd);
 
+#ifndef G_OS_WIN32
 	fd = connect_to_storage (stub, &sa_un, ex);
+#else
+	fd = connect_to_storage (stub, socket_path, ex);
+#endif
 	if (fd == -1) {
 		camel_object_unref (CAMEL_OBJECT (stub));
 		return NULL;
Index: mail/mail-stub-listener.c
===================================================================
--- mail/mail-stub-listener.c	(revision 1672)
+++ mail/mail-stub-listener.c	(working copy)
@@ -23,10 +23,17 @@
 #include <config.h>
 #endif
 
+#include <glib.h>
+#include <glib/gstdio.h>
+
+#ifndef G_OS_WIN32
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/un.h>
 #include <unistd.h>
+#else
+#include <winsock2.h>
+#endif
 
 #include "mail-stub-listener.h"
 #include <e2k-marshal.h>
@@ -72,7 +79,7 @@
 	if (listener->channel)
 		g_io_channel_unref (listener->channel);
 	if (listener->socket_path) {
-		unlink (listener->socket_path);
+		g_unlink (listener->socket_path);
 		g_free (listener->socket_path);
 	}
 
@@ -86,12 +93,9 @@
 new_connection (GIOChannel *source, GIOCondition condition, gpointer data)
 {
 	MailStubListener *listener = data;
-	struct sockaddr_un sa_un;
-	socklen_t len;
 	int fd;
 
-	len = sizeof (sa_un);
-	fd = accept (g_io_channel_unix_get_fd (source), (struct sockaddr *)&sa_un, &len);
+	fd = accept (g_io_channel_unix_get_fd (source), NULL, NULL);
 	if (fd == -1)
 		return TRUE;
 
@@ -110,9 +114,16 @@
 gboolean
 mail_stub_listener_construct (MailStubListener *listener, const char *socket_path)
 {
+#ifndef G_OS_WIN32
 	struct sockaddr_un sa_un;
+#else
+	struct sockaddr_in sa_in;
+	int addr_len;
+	GError *error = NULL;
+#endif
 	int fd;
 
+#ifndef G_OS_WIN32
 	g_return_val_if_fail (strlen (socket_path) < sizeof (sa_un.sun_path), FALSE);
 
 	fd = socket (AF_UNIX, SOCK_STREAM, 0);
@@ -130,6 +141,52 @@
 	}
 
 	listener->channel = g_io_channel_unix_new (fd);
+#else
+	fd = socket (AF_INET, SOCK_STREAM, 0);
+	if (fd == SOCKET_ERROR)
+		return FALSE;
+	memset (&sa_in, 0, sizeof (sa_in));
+	sa_in.sin_family = AF_INET;
+	sa_in.sin_port = 0;
+	sa_in.sin_addr.s_addr = htonl (INADDR_LOOPBACK);
+
+	if (bind (fd, (struct sockaddr *) &sa_in, sizeof (sa_in)) == SOCKET_ERROR) {
+		g_warning ("bind() failed: %s",
+			   g_win32_error_message (WSAGetLastError ()));
+		closesocket (fd);
+		return FALSE;
+	}
+
+	if (listen (fd, 1) == SOCKET_ERROR) {
+		g_warning ("listen() failed: %s",
+		g_win32_error_message (WSAGetLastError ()));
+		closesocket (fd);
+		return FALSE;
+	}	   
+
+	addr_len = sizeof (sa_in);
+	if (getsockname (fd, (struct sockaddr *) &sa_in, &addr_len) == SOCKET_ERROR) {
+		g_warning ("getsockname() failed: %s",
+			   g_win32_error_message (WSAGetLastError ()));
+		closesocket (fd);
+		return FALSE;
+	}
+
+	g_unlink (socket_path);
+	listener->socket_path = g_strdup (socket_path);
+
+	printf ("listening on port %d\n", ntohs(sa_in.sin_port));
+
+	if (!g_file_set_contents (socket_path, (char *) &sa_in, addr_len, &error)) {
+		g_warning ("Could not save socket address in '%s': %s",
+			   socket_path, error->message);
+		g_error_free (error);
+		closesocket (fd);
+		return FALSE;
+	}
+
+	listener->channel = g_io_channel_win32_new_socket (fd);
+#endif
 	g_io_add_watch (listener->channel, G_IO_IN, new_connection, listener);
 
 	listener->cmd_fd = -1;
Index: mail/mail-stub-exchange.c
===================================================================
--- mail/mail-stub-exchange.c	(revision 1672)
+++ mail/mail-stub-exchange.c	(working copy)
@@ -2894,7 +2894,7 @@
 	MailStubExchange *mse = MAIL_STUB_EXCHANGE (stub);
 	SoupMessage *msg;
 	E2kHTTPStatus status;
-	char *timestamp, hostname[256], *errmsg;
+	char *timestamp, *errmsg;
 	GString *data;
 	int i;
 
@@ -2914,11 +2914,9 @@
 	/* Exchange doesn't add a "Received" header to messages
 	 * received via WebDAV.
 	 */
-	if (gethostname (hostname, sizeof (hostname)) != 0)
-		strcpy (hostname, "localhost");
 	timestamp = e2k_make_timestamp_rfc822 (time (NULL));
 	g_string_append_printf (data, "Received: from %s by %s; %s\r\n",
-				hostname, mse->account->exchange_server,
+				g_get_host_name (), mse->account->exchange_server,
 				timestamp);
 	g_free (timestamp);
 
Index: mail/mail-stub.c
===================================================================
--- mail/mail-stub.c	(revision 1672)
+++ mail/mail-stub.c	(working copy)
@@ -24,8 +24,6 @@
 #endif
 
 #include <sys/types.h>
-#include <sys/socket.h>
-#include <sys/un.h>
 #include <unistd.h>
 
 #include <e2k-uri.h>
@@ -766,7 +764,11 @@
 void
 mail_stub_construct (MailStub *stub, int cmd_fd, int status_fd)
 {
+#ifndef G_OS_WIN32
 	stub->channel = g_io_channel_unix_new (cmd_fd);
+#else
+	stub->channel = g_io_channel_win32_new_socket (cmd_fd);
+#endif
 	g_io_add_watch (stub->channel, G_IO_IN | G_IO_ERR | G_IO_HUP,
 			connection_handler, stub);
 	stub->cmd = camel_stub_marshal_new (cmd_fd);
Index: configure.in
===================================================================
--- configure.in	(revision 1672)
+++ configure.in	(working copy)
@@ -97,6 +97,7 @@
     ;;
 esac
 AC_MSG_RESULT([$os_win32])
+AM_CONDITIONAL(OS_WIN32, [test $os_win32 = yes])
 AC_SUBST(NO_UNDEFINED)
 AC_SUBST(SOCKET_LIBS)
 
@@ -237,7 +238,6 @@
 	esac
 	;;
 esac
-AM_CONDITIONAL(SUNLDAP, test "$with_sunldap" != no)
 
 SAVE_CFLAGS="$CFLAGS"
 SAVE_LIBS="$LIBS"
@@ -256,6 +256,8 @@
 AM_CONDITIONAL(ENABLE_LDAP, true)
 fi # Win32
 
+AM_CONDITIONAL(SUNLDAP, test "$with_sunldap" != no)
+
 AC_MSG_CHECKING(for LDAP Paged Control)
 AC_TRY_RUN([
 #include <ldap.h>
Index: ChangeLog
===================================================================
--- ChangeLog	(revision 1672)
+++ ChangeLog	(working copy)
@@ -28,7 +28,151 @@
 
 	* storage/exchange-storage.c: (account_new_folder): Another leak fix.
 	
+2008-05-13  Tor Lillqvist  <tml@novell.com>
 
+	Final outstanding Win32 portability changes.
+
+	* storage/Makefile.am: Link with SOCKET_LIBS, too. On Windows, use
+	a relative path for CONNECTOR_PATH in the server file instead of
+	the absolute compile-time EVOLUTION_privlibexecdir. Replace also
+	EXEEXT in the server file.
+
+	* storage/exchage-storage.h: On Windows, declare three char*
+	variables that will hold the datadir, gladedir and imagesdir
+	constructed at run-time.
+
+	* storage/main.c: Define and initialise the three char* variables
+	mentioned above. Assume that evolution-exchange is installed in
+	the same run-time prefix as evolution-data-server. Use
+	g_get_tmp_dir() instead of hardcoding /tmp. Use gstdio wrappers to
+	cope with arbitrary Unicode pathnames on Windows. Don't bother
+	with the folder mode checks on Windows, as the stat::st_mode field
+	is mostly meaningless.
+
+	* storage/exchange-autoconfig-wizard.c
+	* storage/exchange-change-password.c: Construct glade and png file
+	pathnames at run-time. Don't use compile-time gladedir and
+	imagesdir on Windows.
+
+	* storage/exchange-config-listener.c: Reorder includes into a more
+	logical order. To cope with arbitrary Unicode pathnames on
+	Windows, use gstdio wrappers, use GDir instead of dirent API, and
+	use e_xml_parse_file() and e_xml_save_file() instead of libxml2
+	API directly.
+
+	* storage/ximian-connector-setup.c: Define and initialise the
+	datadir, gladedir and imagesdir variables on Windows here,
+	too. Use GETTEXT_PACKAGE instead of PACKAGE.
+
+	* storage/GNOME_Evolution_Exchange_Storage.server.in.in: Add
+	EXEEXT to the executable name.
+
+	* calendar/e-cal-backend-exchange.c: Avoid more characters in file
+	names on Windows. Use g_path_get_dirname() instead of searching
+	for '/'. Use g_lstat() instead of lstat(). There are no symlinks
+	on Windows, so unclear what to actually do on Windows where the
+	code creates a symlink on Unix. Use gstdio wrappers. Use binary
+	mode when opening files. Use g_filename_to/from_uri() instead of
+	manipulating and constructing URIs manually.
+
+2008-05-12  Tor Lillqvist  <tml@novell.com>
+
+	Further outstanding Win32 portability changes.
+
+	* camel/camel-stub.h: Add a separate have_status_thread boolean
+	flag to the CamelStub struct. A pthread_t is not necessarily a
+	scalar type, it can also be a struct.
+
+	* camel/camel-stub.c: Corresponding changes.
+
+	* camel/camel-stub.c: Include <winsock2.h> on Windows.
+	(connect_to_storage): Different implementation on Unix and
+	Windows. On Windows, there are no Unix domain sockets, so instead
+	we just have the sockaddr_in of a TCP socket bound to the loopback
+	interface stored in a plain file. Yes, this is insecure on
+	multiple-user machines. Will have to spend more time on this issue
+	sometime. Presumably Win32 named pipes are what should be used
+	instead.
+	(camel_stub_new): Corresponding ifdefs.
+
+	* mail/mail-stub-listener.c: Corresponding changes.
+	(finalize): Use g_unlink().
+	(new_connection): Drop unused sa_un and len variables. Just pass
+	NULL as parameters 2 and 3 to accept().
+	(mail_stub_listener_construct): On Windows, instead of creating a
+	Unix domain socket, create a TCP socket bound to the loopback
+	interface and write its sockaddr_in to a regular file from which
+	the client will read the address.
+
+2008-05-08  Tor Lillqvist  <tml@novell.com>
+
+	Win32 portability changes from a long time ago, now finally
+	starting to commit them a little at a time. Whether
+	evolution-exchange actually works on Windows is not clear... but
+	at least once these changes are in there is a possibility.
+
+	* configure.in: Define Automake conditional OS_WIN32.
+
+	* camel/Makefile.am: Pass also PREFIX to the compiler. Use
+	-no-undefined on Windows. Link with SOCKET_LIBS and PTHREAD_LIB.
+
+	* camel/camel-exchange-provider.c: Include
+	libedataserver/e-data-server-util.h for some Windows-specific
+	functions. Construct the localedir at run-time on Windows.
+
+	* camel/camel-exchange-store.c (e_filename_make_safe)
+	* storage/exchange-component.c (e_filename_make_safe): Remove
+	these two copies of this function. Just use the one in
+	libedataserver.
+
+	* camel/camel-exchange-store.c (exchange_connect)
+	* storage/exchange-component.c (config_listener_account_created):
+	Construct the path to the socket in the same way in these two
+	functions. Use g_get_tmp_dir() instead of hardcoding /tmp. Use
+	g_build_filename().
+
+	* camel/camel-exchange-store.c (exchange_get_folder): Use
+	g_file_test() instead of access().
+
+	* camel/camel-exchange-summary.c: Drop unnecessary inclusion of
+	<sys/uio.h>.
+
+	* camel/camel-stub-marshal.c: Use closesocket() instead of close()
+	for sockets on Windows. Use camel_read_socket() and
+	camel_write_socket() instead of camel_read() and camel_write().
+
+	* mail/mail-stub.c No need to include <sys/socket.h> and
+	<sys/un.h> in this file.
+	(mail_stub_construct): Use g_io_channel_win32_new_socket() on
+	Windows as we know that the fd is actually a socket.
+
+	* mail/mail-stub-exchange.c (send_message): Use g_get_host_name()
+	instead of gethostname().
+
+	* addressbook/e-book-backend-exchange.c (update_cache): Remove
+	pointless code that constructed a timestamp for the cache file
+	name but didn't use it for anything.
+
+	* addressbook/e-book-backend-gal.c: Don't include <ldap.h> here,
+	it or its Windows counterpart <winldap.h> has already been
+	included through <e2k-global-catalog-ldap.h> anyway.
+	(ber_dupbv): Need this on Windows, too.
+	(ldap_op_finished): Use ldap_abandon() instead of the nonstandard
+	ldap_abandon_ext(). The extra parameters were NULL anyway, meaning
+	it is equivalent to calling ldap_abandon().
+
+	* storage/exchange-component.c
+	(default_linestatus_notify_handler): Use guint instead of uint.
+
+	* storage/exchange-migrate.c: Bypass migration code on Windows as
+	there are no old evolution-exchange 1.x deployments on Windows to
+	migrate from.
+
+2008-05-07  Tor Lillqvist  <tml@novell.com>
+
+	* configure.in: Move the AM_CONDITIONAL for
+	SUNLDAP. AM_CONDITIONAL should not be inside shell if statements.
+
 2008-05-02  Johnny Jacob <jjohnny@novell.com>
 
 	* NEWS, Configure.in : Evolution Exchange 2.22.1.1 release.
Index: addressbook/e-book-backend-exchange.c
===================================================================
--- addressbook/e-book-backend-exchange.c	(revision 1672)
+++ addressbook/e-book-backend-exchange.c	(working copy)
@@ -505,20 +505,7 @@
 	E2kResultIter *iter;
 	E2kResult *result;
 	EContact *contact;
-	const char *cache_file_name;
-	time_t mod_time;
-	char time_string[50]; /* Donno what could be right. 50 should be safe for sure.*/
-	const struct tm *tm;
-	struct stat buf;
 
-	cache_file_name =
-		e_file_cache_get_filename (E_FILE_CACHE(bepriv->cache));
-
-	stat (cache_file_name, &buf);
-	mod_time = buf.st_mtime;
-	tm = gmtime (&mod_time);
-	strftime (time_string, 50, "%Y-%m-%dT%H:%M:%SZ", tm);
-
 	/* build hash table from storage file */
 
 	/* FIXME: extract the difference and build update cache with changes */
Index: addressbook/e-book-backend-gal.c
===================================================================
--- addressbook/e-book-backend-gal.c	(revision 1672)
+++ addressbook/e-book-backend-gal.c	(working copy)
@@ -37,15 +37,6 @@
 #include <exchange-account.h>
 #include "exchange-component.h"
 
-#ifdef DEBUG
-#define LDAP_DEBUG
-#define LDAP_DEBUG_ADD
-#endif
-#include <ldap.h>
-#ifdef DEBUG
-#undef LDAP_DEBUG
-#endif
-
 #define d(x) x
 
 #include <sys/time.h>
@@ -1961,7 +1952,7 @@
 	}
 }
 
-#ifdef SUNLDAP
+#if defined (SUNLDAP) || defined (G_OS_WIN32)
 static struct berval *
 ber_dupbv( struct berval *dst, struct berval *src )
 {
@@ -2338,7 +2329,7 @@
 	/* ignore errors, its only best effort? */
 	g_mutex_lock (bl->priv->ldap_lock);
 	if (bl->priv->ldap)
-		ldap_abandon_ext (bl->priv->ldap, op->id, NULL, NULL);
+		ldap_abandon (bl->priv->ldap, op->id);
 	g_mutex_unlock (bl->priv->ldap_lock);
 }
 
Index: storage/exchange-change-password.c
===================================================================
--- storage/exchange-change-password.c	(revision 1672)
+++ storage/exchange-change-password.c	(working copy)
@@ -32,7 +32,15 @@
 #include <gtk/gtkentry.h>
 #include <gtk/gtklabel.h>
 
-#define FILENAME CONNECTOR_GLADEDIR "/exchange-change-password.glade"
+#include "exchange-storage.h"
+
+#ifdef G_OS_WIN32
+
+#undef CONNECTOR_GLADEDIR
+#define CONNECTOR_GLADEDIR _exchange_storage_gladedir
+
+#endif
+
 #define ROOTNODE "pass_dialog"
 #define STARTNODE "pass_vbox"
 
@@ -80,8 +88,13 @@
 	GtkResponseType response;
 	GtkLabel *top_label;
 	char *new_pass;
+	gchar *gladefile;
 
-	xml = glade_xml_new (FILENAME, ROOTNODE, NULL);
+	gladefile = g_build_filename (CONNECTOR_GLADEDIR,
+				      "exchange-change-password.glade",
+				      NULL);
+	xml = glade_xml_new (gladefile, ROOTNODE, NULL);
+	g_free (gladefile);
 	top_widget = glade_xml_get_widget (xml, ROOTNODE);
 
         cur_entry = GTK_ENTRY (glade_xml_get_widget (xml, "current_pass_entry"));
Index: storage/exchange-config-listener.c
===================================================================
--- storage/exchange-config-listener.c	(revision 1672)
+++ storage/exchange-config-listener.c	(working copy)
@@ -22,35 +22,36 @@
  * default folders are updated as needed.
  */
 
-#ifdef HAVE_CONFIG_H
 #include <config.h>
-#endif
 
-#include "exchange-config-listener.h"
+#include <sys/types.h>
+#include <sys/stat.h>
+#include <unistd.h>
+#include <stdlib.h>
+#include <string.h>
 
-#include <exchange-account.h>
-#include <e-folder-exchange.h>
-#include <e2k-marshal.h>
-#include <e2k-uri.h>
-#include "mail-stub-listener.h"
+#include <glib.h>
+#include <glib/gstdio.h>
 
-#include <e-util/e-dialog-utils.h>
-
 #include <libedataserver/e-source.h>
 #include <libedataserver/e-source-list.h>
 #include <libedataserver/e-source-group.h>
+#include <libedataserver/e-xml-utils.h>
 #include <libedataserver/e-xml-hash-utils.h>
 
 #include <camel/camel-url.h>
 
-#include <sys/types.h>
-#include <dirent.h>
-#include <sys/stat.h>
-#include <unistd.h>
-#include <stdlib.h>
-#include <string.h>
+#include <e2k-marshal.h>
+#include <e2k-uri.h>
+#include <exchange-account.h>
+#include <e-folder-exchange.h>
 
+#include <e-util/e-dialog-utils.h>
 
+#include "mail-stub-listener.h"
+
+#include "exchange-config-listener.h"
+
 struct _ExchangeConfigListenerPrivate {
 	GConfClient *gconf;
 	guint idle_id;
@@ -194,18 +195,17 @@
 update_foreign_uri (const char *path, const char *account_uri)
 {
 	char *file_path, *phy_uri, *foreign_uri, *new_phy_uri;
-	struct stat file_stat;
 	GHashTable *old_props = NULL;
-	xmlDoc *old_doc, *new_doc = NULL;
+	xmlDoc *old_doc = NULL, *new_doc = NULL;
 
 	if (!path)
 		return;
 
 	file_path = g_build_filename (path, "hierarchy.xml", NULL);
-	if (stat (file_path, &file_stat) < 0)
+	if (!g_file_test (file_path, G_FILE_TEST_EXISTS))
 		goto cleanup;
 
-	old_doc = xmlParseFile (file_path);
+	old_doc = e_xml_parse_file (file_path);
 	if (!old_doc)
 		goto cleanup;
 
@@ -231,7 +231,7 @@
 	g_hash_table_insert (old_props, (char *)g_strdup ("physical_uri_prefix"), new_phy_uri);
 
 	new_doc = e_xml_from_hash (old_props, E_XML_HASH_TYPE_PROPERTY, "foreign-hierarchy");
-	xmlSaveFile (file_path, new_doc);
+	e_xml_save_file (file_path, new_doc);
 
 	xmlFreeDoc (new_doc);
 	g_free (new_phy_uri);
@@ -244,21 +244,21 @@
 static void
 migrate_foreign_hierarchy (ExchangeAccount *account)
 {
-	DIR *d;
-	struct dirent *dentry;
+	GDir *d;
+	const char *dentry;
 	char *dir;
 
-	d = opendir (account->storage_dir);
+	d = g_dir_open (account->storage_dir, 0, NULL);
 	if (d) {
-		while ((dentry = readdir (d))) {
-			if (!strchr (dentry->d_name, '@'))
+		while ((dentry = g_dir_read_name (d))) {
+			if (!strchr (dentry, '@'))
 				continue;
 			dir = g_strdup_printf ("%s/%s", account->storage_dir,
-							dentry->d_name);
+							dentry);
 			update_foreign_uri (dir, account->account_filename);
 			g_free (dir);
 		}
-		closedir (d);
+		g_dir_close (d);
 	}
 }
 
Index: storage/exchange-migrate.c
===================================================================
--- storage/exchange-migrate.c	(revision 1672)
+++ storage/exchange-migrate.c	(working copy)
@@ -47,6 +47,14 @@
 CORBA_short min=0;
 CORBA_short rev=0;
 
+#ifndef G_OS_WIN32
+
+/* No need for migration code from evolution-exchange 1.x on Win32.
+ * If and when the same code is partly used for migrating between
+ * future evolution-exchange versions, that code will have to be
+ * revised and ported as necessary.
+ */
+
 gchar *label_string = NULL;
 
 static void
@@ -449,6 +457,8 @@
 	return ret;
 }
 
+#endif
+
 void
 exchange_migrate (const CORBA_short major,
 		  const CORBA_short minor,
@@ -464,7 +474,7 @@
 
 	/* FIXME: version check */
 	if (maj == 1 && min <= 5) {
-
+#ifndef G_OS_WIN32
 		if (!base_dir)
 			return;
 
@@ -507,5 +517,6 @@
 		if (ret == FALSE)
 			show_error_dialog ();
 
+#endif
 	}
 }
Index: storage/ximian-connector-setup.c
===================================================================
--- storage/ximian-connector-setup.c	(revision 1672)
+++ storage/ximian-connector-setup.c	(working copy)
@@ -25,19 +25,51 @@
 #include <libgnomeui/gnome-ui-init.h>
 
 #include <e-util/e-dialog-utils.h>
+#include <libedataserver/e-data-server-util.h>
 #include <libedataserverui/e-passwords.h>
 
 #include <e2k-utils.h>
 
 #include "exchange-autoconfig-wizard.h"
 
+#ifdef G_OS_WIN32
+const char *_exchange_storage_datadir;
+const char *_exchange_storage_gladedir;
+const char *_exchange_storage_imagesdir;
+#endif
+
 int
 main (int argc, char **argv)
 {
-	bindtextdomain (PACKAGE, CONNECTOR_LOCALEDIR);
-	bind_textdomain_codeset (PACKAGE, "UTF-8");
-	textdomain (PACKAGE);
+#ifdef G_OS_WIN32
+	{
+		char *localedir;
 
+		/* We assume evolution-exchange is installed in the
+		 * same run-time prefix as evolution-data-server.
+		 */
+		_exchange_storage_datadir = e_util_replace_prefix (PREFIX, e_util_get_prefix (), DATADIR);
+		_exchange_storage_gladedir = e_util_replace_prefix (PREFIX, e_util_get_prefix (), CONNECTOR_GLADEDIR);
+		_exchange_storage_imagesdir = e_util_replace_prefix (PREFIX, e_util_get_prefix (), CONNECTOR_IMAGESDIR);
+		
+		localedir = e_util_replace_prefix (CONNECTOR_LOCALEDIR, e_util_get_cp_prefix (), CONNECTOR_LOCALEDIR);
+		bindtextdomain (GETTEXT_PACKAGE, localedir);
+	}
+
+/* PREFIX and DATADIR are part of GNOME_PROGRAM_STANDARD_PROPERTIES */
+
+#undef PREFIX
+#define PREFIX e_util_get_prefix ()
+
+#undef DATADIR
+#define DATADIR _exchange_storage_datadir
+
+#else
+	bindtextdomain (GETTEXT_PACKAGE, CONNECTOR_LOCALEDIR);
+#endif
+	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
+	textdomain (GETTEXT_PACKAGE);
+
 	gnome_program_init ("ximian-connector-setup", VERSION,
 			    LIBGNOMEUI_MODULE, argc, argv,
 			    GNOME_PROGRAM_STANDARD_PROPERTIES,
Index: storage/exchange-autoconfig-wizard.c
===================================================================
--- storage/exchange-autoconfig-wizard.c	(revision 1672)
+++ storage/exchange-autoconfig-wizard.c	(working copy)
@@ -41,6 +41,18 @@
 #include <libgnomeui/gnome-druid.h>
 #include <libgnomeui/gnome-druid-page-standard.h>
 
+#include "exchange-storage.h"
+
+#ifdef G_OS_WIN32
+
+#undef CONNECTOR_GLADEDIR
+#define CONNECTOR_GLADEDIR _exchange_storage_gladedir
+
+#undef CONNECTOR_IMAGESDIR
+#define CONNECTOR_IMAGESDIR _exchange_storage_imagesdir
+
+#endif
+
 typedef struct {
 	GnomeDruid *druid;
 
@@ -109,10 +121,15 @@
 autoconfig_gui_new (void)
 {
 	ExchangeAutoconfigGUI *gui;
+	char *gladefile;
 
 	gui = g_new0 (ExchangeAutoconfigGUI, 1);
 
-	gui->xml = glade_xml_new (CONNECTOR_GLADEDIR "/exchange-autoconfig-wizard.glade", NULL, NULL);
+	gladefile = g_build_filename (CONNECTOR_GLADEDIR,
+				      "exchange-autoconfig-wizard.glade",
+				      NULL);
+	gui->xml = glade_xml_new (gladefile, NULL, NULL);
+	g_free (gladefile);
 	if (!gui->xml) {
 		g_warning ("Could not find exchange-autoconfig-wizard.glade");
 		g_free (gui);
@@ -605,6 +622,7 @@
 	GtkWidget *page;
 	GdkPixbuf *icon;
 	int i;
+	gchar *pngfile;
 
 	gui = autoconfig_gui_new ();
 	g_return_if_fail (gui);
@@ -614,7 +632,11 @@
 	gui->window = (GtkWindow *)glade_xml_get_widget (gui->xml, "window");
 	gui->pages = g_ptr_array_new ();
 
-	icon = gdk_pixbuf_new_from_file (CONNECTOR_IMAGESDIR "/connector.png", NULL);
+	pngfile = g_build_filename (CONNECTOR_IMAGESDIR,
+				    "connector.png",
+				    NULL);
+	icon = gdk_pixbuf_new_from_file (pngfile, NULL);
+	g_free (pngfile);
 	for (i = 0; i < num_autoconfig_pages; i++) {
 		page = glade_xml_get_widget (gui->xml, autoconfig_pages[i].page_name);
 		g_ptr_array_add (gui->pages, page);
Index: storage/exchange-component.c
===================================================================
--- storage/exchange-component.c	(revision 1672)
+++ storage/exchange-component.c	(working copy)
@@ -21,6 +21,8 @@
 #include "config.h"
 #endif
 
+#include <libedataserver/e-data-server-util.h>
+
 #include "exchange-component.h"
 #include "shell/e-component-view.h"
 
@@ -44,7 +46,6 @@
 
 #define PARENT_TYPE bonobo_object_get_type ()
 static BonoboObjectClass *parent_class = NULL;
-static void e_filename_make_safe (gchar *string);
 static void exchange_component_update_accounts (ExchangeComponent *component,
 							gboolean status);
 static void ex_migrate_esources (ExchangeComponent *component,
@@ -314,31 +315,7 @@
 	}
 }
 
-/* SURF : Picked this from gal/util/e-util.c */
-/* This only makes a filename safe for usage as a filename.  It still may have shell meta-characters in it. */
 static void
-e_filename_make_safe (gchar *string)
-{
-	gchar *p, *ts;
-	gunichar c;
-
-	g_return_if_fail (string != NULL);
-	p = string;
-
-	while(p && *p) {
-		c = g_utf8_get_char (p);
-		ts = p;
-		p = g_utf8_next_char (p);
-		if (!g_unichar_isprint(c) || ( c < 0xff && strchr (" /'\"`&();|<>$%{}!", c&0xff ))) {
-			while (ts<p)
-				*ts++ = '_';
-		}
-	}
-}
-
-
-
-static void
 new_connection (MailStubListener *listener, int cmd_fd, int status_fd,
 		ExchangeComponentAccount *baccount)
 {
@@ -387,17 +364,21 @@
 	ExchangeComponent *component = user_data;
 	ExchangeComponentPrivate *priv = component->priv;
 	ExchangeComponentAccount *baccount;
-	char *path, *account_filename;
+	char *path, *dot_exchange_username, *account_filename;
 
 	baccount = g_new0 (ExchangeComponentAccount, 1);
 	baccount->account = g_object_ref (account);
 
+	dot_exchange_username = g_strdup_printf (".exchange-%s", g_get_user_name ());
+
 	account_filename = strrchr (account->storage_dir, '/') + 1;
 	e_filename_make_safe (account_filename);
 
-	path = g_strdup_printf ("/tmp/.exchange-%s/%s",
-				g_get_user_name (),
-				account_filename);
+	path = g_build_filename (g_get_tmp_dir (),
+				 dot_exchange_username,
+				 account_filename,
+				 NULL);
+	g_free (dot_exchange_username);
 	baccount->msl = mail_stub_listener_new (path);
 	g_signal_connect (baccount->msl, "new_connection",
 			  G_CALLBACK (new_connection), baccount);
@@ -427,7 +408,7 @@
 }
 
 static void
-default_linestatus_notify_handler (ExchangeComponent *component, uint status)
+default_linestatus_notify_handler (ExchangeComponent *component, guint status)
 {
 	CORBA_Environment ev;
 
Index: storage/main.c
===================================================================
--- storage/main.c	(revision 1672)
+++ storage/main.c	(working copy)
@@ -29,6 +29,9 @@
 #include <string.h>
 #include <unistd.h>
 
+#include <glib.h>
+#include <glib/gstdio.h>
+
 #include <bonobo/bonobo-main.h>
 #include <bonobo/bonobo-generic-factory.h>
 #include <bonobo/bonobo-exception.h>
@@ -41,6 +44,7 @@
 #include <libedata-book/e-data-book-factory.h>
 #include <libedata-cal/e-data-cal-factory.h>
 #include <libedataserver/e-data-server-module.h>
+#include <libedataserver/e-data-server-util.h>
 
 #include <e2k-utils.h>
 #include <exchange-constants.h>
@@ -61,6 +65,12 @@
 
 ExchangeComponent *global_exchange_component;
 
+#ifdef G_OS_WIN32
+const char *_exchange_storage_datadir;
+const char *_exchange_storage_gladedir;
+const char *_exchange_storage_imagesdir;
+#endif
+
 static BonoboObject *
 exchange_component_factory (BonoboGenericFactory *factory,
 			    const char *component_id, void *component)
@@ -171,10 +181,35 @@
 int
 main (int argc, char **argv)
 {
-	char *path;
+	char *userdir, *path;
 	char *config_directory;
 
+#ifdef G_OS_WIN32
+	{
+		char *localedir;
+
+		/* We assume evolution-exchange is installed in the
+		 * same run-time prefix as evolution-data-server.
+		 */
+		_exchange_storage_datadir = e_util_replace_prefix (PREFIX, e_util_get_prefix (), DATADIR);
+		_exchange_storage_gladedir = e_util_replace_prefix (PREFIX, e_util_get_prefix (), CONNECTOR_GLADEDIR);
+		_exchange_storage_imagesdir = e_util_replace_prefix (PREFIX, e_util_get_prefix (), CONNECTOR_IMAGESDIR);
+		
+		localedir = e_util_replace_prefix (CONNECTOR_LOCALEDIR, e_util_get_cp_prefix (), CONNECTOR_LOCALEDIR);
+		bindtextdomain (GETTEXT_PACKAGE, localedir);
+	}
+
+/* PREFIX and DATADIR are part of GNOME_PROGRAM_STANDARD_PROPERTIES */
+
+#undef PREFIX
+#define PREFIX e_util_get_prefix ()
+
+#undef DATADIR
+#define DATADIR _exchange_storage_datadir
+
+#else
 	bindtextdomain (GETTEXT_PACKAGE, CONNECTOR_LOCALEDIR);
+#endif
 	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
 	textdomain (GETTEXT_PACKAGE);
 
@@ -188,20 +223,24 @@
 	camel_init (config_directory, FALSE);
 	g_free(config_directory);
 
-	path = g_strdup_printf ("/tmp/.exchange-%s", g_get_user_name ());
-	if (mkdir (path, 0700) == -1) {
+	userdir = g_strdup_printf (".exchange-%s", g_get_user_name ());
+	path = g_build_filename (g_get_tmp_dir (), userdir, NULL);
+	g_free (userdir);
+	if (g_mkdir (path, 0700) == -1) {
 		if (errno == EEXIST) {
 			struct stat st;
 
-			if (stat (path, &st) == -1) {
+			if (g_stat (path, &st) == -1) {
 				g_warning ("Could not stat %s", path);
 				return 1;
 			}
+#ifdef G_OS_UNIX
 			if (st.st_uid != getuid () ||
 			    (st.st_mode & 07777) != 0700) {
 				g_warning ("Bad socket dir %s", path);
 				return 1;
 			}
+#endif
 		} else {
 			g_warning ("Can't create %s", path);
 			return 1;
Index: storage/Makefile.am
===================================================================
--- storage/Makefile.am	(revision 1672)
+++ storage/Makefile.am	(working copy)
@@ -45,7 +45,8 @@
 	$(top_builddir)/camel/camel-stub-marshal.lo		\
 	$(LDAP_LIBS)						\
 	$(EXCHANGE_STORAGE_LIBS)				\
-	$(LIBEXCHANGE_LIBS)
+	$(LIBEXCHANGE_LIBS)					\
+	$(SOCKET_LIBS)
 
 
 exchange_connector_setup_SOURCES =				\
@@ -79,10 +80,17 @@
 server_in_files = GNOME_Evolution_Exchange_Storage.server.in.in
 server_DATA = $(server_in_files:.server.in.in=_$(BASE_VERSION).server)
 
+if OS_WIN32
+CONNECTOR_PATH_IN_SERVER_FILE=../../../libexec/evolution/${BASE_VERSION}
+else
+CONNECTOR_PATH_IN_SERVER_FILE=${EVOLUTION_privlibexecdir}
+endif
+
 %_$(BASE_VERSION).server.in: %.server.in.in
-	sed -e "s;\@CONNECTOR_PATH\@;${EVOLUTION_privlibexecdir};g"	\
+	sed -e "s;\@CONNECTOR_PATH\@;${CONNECTOR_PATH_IN_SERVER_FILE};g"	\
 	    -e "s;\@IMAGESDIR\@;${imagesdir};g"				\
 	    -e "s;\@BASE_VERSION\@;${BASE_VERSION};g"			\
+	    -e "s;\@EXEEXT\@;${EXEEXT};g"				\
 	    -e "s;\@API_VERSION\@;${API_VERSION};g"			\
 	    -e "s;\@EDS_BASE_VERSION\@;${EDS_BASE_VERSION};g"		\
 	    < $< > $@
Index: storage/exchange-storage.h
===================================================================
--- storage/exchange-storage.h	(revision 1672)
+++ storage/exchange-storage.h	(working copy)
@@ -33,6 +33,14 @@
 
 EStorage         *exchange_storage_new      (ExchangeAccount *account);
 
+#ifdef G_OS_WIN32
+
+extern const char *_exchange_storage_datadir;
+extern const char *_exchange_storage_gladedir;
+extern const char *_exchange_storage_imagesdir;
+
+#endif
+
 #ifdef __cplusplus
 }
 #endif /* __cplusplus */
Index: storage/GNOME_Evolution_Exchange_Storage.server.in.in
===================================================================
--- storage/GNOME_Evolution_Exchange_Storage.server.in.in	(revision 1672)
+++ storage/GNOME_Evolution_Exchange_Storage.server.in.in	(working copy)
@@ -2,7 +2,7 @@
 
 <oaf_server iid="OAFIID:GNOME_Evolution_Exchange_Component_Factory:@BASE_VERSION@"
 	    type="exe"
-	    location="@CONNECTOR_PATH@/evolution-exchange-storage">
+	    location="@CONNECTOR_PATH@/evolution-exchange-storage@EXEEXT@">
 
 	<oaf_attribute name="repo_ids" type="stringv">
 		<item value="IDL:GNOME/ObjectFactory:1.0"/>
@@ -40,7 +40,7 @@
 
 <oaf_server iid="OAFIID:GNOME_Evolution_Exchange_Connector_BookFactory:@API_VERSION@"
             type="exe"
-            location="@CONNECTOR_PATH@/evolution-exchange-storage">
+            location="@CONNECTOR_PATH@/evolution-exchange-storage@EXEEXT@">
 
 	<oaf_attribute name="repo_ids" type="stringv">
 		<item value="IDL:GNOME/Evolution/DataServer/BookFactory:@API_VERSION@"/>
@@ -58,7 +58,7 @@
 
 <oaf_server iid="OAFIID:GNOME_Evolution_Exchange_Connector_CalFactory:@API_VERSION@"
             type="exe"
-            location="@CONNECTOR_PATH@/evolution-exchange-storage">
+            location="@CONNECTOR_PATH@/evolution-exchange-storage@EXEEXT@">
 
 	<oaf_attribute name="repo_ids" type="stringv">
 		<item value="IDL:GNOME/Evolution/DataServer/CalFactory:@API_VERSION@"/>
Index: evolution-exchange-zip.in
===================================================================
--- evolution-exchange-zip.in	(revision 1672)
+++ evolution-exchange-zip.in	(working copy)
@@ -31,4 +31,4 @@
 share/evolution-exchange/@BASE_VERSION@/ui
 EOF
 
-zip $ZIP lib/locale/*/LC_MESSAGES/evolution-exchange-@BASE_VERSION@.mo
+zip $ZIP share/locale/*/LC_MESSAGES/evolution-exchange-@BASE_VERSION@.mo
Index: calendar/e-cal-backend-exchange.c
===================================================================
--- calendar/e-cal-backend-exchange.c	(revision 1672)
+++ calendar/e-cal-backend-exchange.c	(working copy)
@@ -30,6 +30,9 @@
 #include <unistd.h>
 #include <libedataserver/e-time-utils.h>
 
+#include <glib.h>
+#include <glib/gstdio.h>
+
 #include <libgnomevfs/gnome-vfs-mime-utils.h>
 
 #include <camel/camel-mime-message.h>
@@ -47,9 +50,13 @@
 
 #include <e-folder-exchange.h>
 #include <exchange-account.h>
-#include "exchange-component.h"
+#include <exchange-component.h>
 #include <exchange-hierarchy.h>
 
+#ifndef O_BINARY
+#define O_BINARY 0
+#endif
+
 struct ECalBackendExchangePrivate {
 	gboolean read_only;
 
@@ -158,7 +165,7 @@
 	struct icaltimetype comp_last_mod, folder_last_mod;
 	icalcomponent_kind kind;
 	icalproperty *prop;
-	char *lastmod, *mangled_uri, *storage_dir, *end;
+	char *lastmod, *mangled_uri, *storage_dir;
 	const char *uristr;
 	int i;
 	struct stat buf;
@@ -177,15 +184,27 @@
 		switch (mangled_uri[i]) {
 		case ':' :
 		case '/' :
+#ifdef G_OS_WIN32
+		case '\\' :
+		case '<' :
+		case '>' :
+		case '|' :
+#endif
 			mangled_uri[i] = '_';
 		}
 	}
 	cbex->priv->local_attachment_store = g_strdup_printf ("%s/.evolution/exchange/%s", g_get_home_dir (), mangled_uri);
-	end = strrchr (cbex->priv->object_cache_file, '/');
-	storage_dir = g_strndup (cbex->priv->object_cache_file, end - cbex->priv->object_cache_file);
-	if (lstat(cbex->priv->local_attachment_store , &buf) < 0) {
+	storage_dir = g_path_get_dirname (cbex->priv->object_cache_file);
+
+	if (g_lstat(cbex->priv->local_attachment_store , &buf) < 0) {
+#ifdef G_OS_UNIX
 		if (symlink (storage_dir, cbex->priv->local_attachment_store) < 0)
 			g_warning ("%s: symlink() failed: %s", G_STRFUNC, g_strerror (errno));
+#else
+		g_warning ("should symlink %s->%s, huh?",
+			   cbex->priv->local_attachment_store,
+			   storage_dir);
+#endif
 	}
 	g_free (storage_dir);
 	g_free (mangled_uri);
@@ -281,7 +300,7 @@
 	icalcomponent_free (vcalcomp);
 
 	tmpfile = g_strdup_printf ("%s~", cbex->priv->object_cache_file);
-	f = fopen (tmpfile, "w");
+	f = g_fopen (tmpfile, "wb");
 	if (!f)
 		goto error;
 
@@ -290,8 +309,8 @@
 	if (fclose (f) != 0 || nwrote != len)
 		goto error;
 
-	if (rename (tmpfile, cbex->priv->object_cache_file) != 0)
-		unlink (tmpfile);
+	if (g_rename (tmpfile, cbex->priv->object_cache_file) != 0)
+		g_unlink (tmpfile);
 error:
 	g_free (tmpfile);
 	g_free (data);
@@ -1593,7 +1612,7 @@
 	d(printf ("dest_file is :%s\n", dest_file));
 
 	/* Write it to our local exchange store in .evolution */
-	fd = open (dest_file, O_RDWR | O_CREAT | O_TRUNC, 0600);
+	fd = g_open (dest_file, O_RDWR | O_CREAT | O_TRUNC | O_BINARY, 0600);
 	if (fd < 0) {
 		d(printf ("open of destination file for attachments failed\n"));
 		goto end;
@@ -1604,8 +1623,9 @@
 		goto end;
 	}
 	/* FIXME : Add a ATTACH:CID:someidentifier here */
-	dest_url = g_strdup_printf ("file://%s", dest_file);
+	dest_url = g_filename_to_uri (dest_file, NULL, NULL);
 
+
 end :
 	close (fd);
 	return dest_url;
@@ -1672,7 +1692,7 @@
 	struct stat sb;
 	char *file_contents = NULL;
 
-	fd = open (filename, O_RDONLY);
+	fd = g_open (filename, O_RDONLY | O_BINARY, 0);
 	if (fd < 0) {
 		d(printf ("Could not open the attachment file : %s\n", filename));
 		goto end;
@@ -1799,10 +1819,11 @@
 		char *mime_type;
 
 		if (!strncmp ((char *)l->data, "file://", 7)) {
-			fname = (char *)(l->data) + strlen ("file://");
-			filename = g_strrstr (fname, "/") + 1;
-			mime_filename = filename + strlen(uid) + 1;
-			attach_file = g_strdup (fname);
+			fname = g_filename_from_uri ((char *)l->data, NULL, NULL);
+			filename = g_path_get_basename (fname);
+			mime_filename = g_strdup (filename + strlen(uid) + 1);
+			g_free (filename);
+			attach_file = fname;
 		} else {
 			fname = (char *)(l->data);
 			filename = g_strrstr (fname, "/");
@@ -1813,19 +1834,25 @@
 				 */
 				continue;
 			}
-			filename++;
-			mime_filename = filename;
+			mime_filename = g_strdup (filename+1);
 			attach_file = g_strdup_printf ("%s/%s-%s", cbex->priv->local_attachment_store, uid, filename);
 		}
 
+		/* mime_filename and attach_file should be g_freed */
+
 		file_contents = get_attach_file_contents (fname, &len);
-			if (!file_contents)
+		if (!file_contents) {
+			g_free (attach_file);
+			g_free (mime_filename);
 			continue;
+		}
 
 		dest_url = save_attach_file (attach_file, file_contents, len);
 		g_free (attach_file);
-		if (!dest_url)
+		if (!dest_url) {
+			g_free (mime_filename);
 			continue;
+		}
 		new_attach_list = g_slist_append (new_attach_list, dest_url);
 
 		/* Content */
@@ -1834,7 +1861,10 @@
 		camel_data_wrapper_construct_from_stream (wrapper, stream);
 		camel_object_unref (stream);
 
-		mime_type = gnome_vfs_get_mime_type (dest_url + strlen ("file://"));
+		filename = g_filename_from_uri (dest_url, NULL, NULL);
+		mime_type = gnome_vfs_get_mime_type (filename);
+		g_free (filename);
+
 		type = camel_content_type_decode (mime_type);
 		camel_data_wrapper_set_mime_type_field (wrapper, type);
 		camel_content_type_unref (type);
@@ -1847,6 +1877,7 @@
 		cid = camel_header_msgid_generate ();
 		camel_mime_part_set_content_id (mime_part, cid);
 		camel_mime_part_set_description (mime_part, mime_filename);
+		g_free (mime_filename);
 		camel_mime_part_set_disposition (mime_part, "attachment");
 		camel_multipart_set_boundary (multipart, NULL);
 		*boundary = g_strdup (camel_multipart_get_boundary (multipart));
EOF

patch -p0 <<'EOF'
--- evolution-exchange-zip.in
+++ evolution-exchange-zip.in
@@ -11,5 +11,5 @@
 # prefixes, and there is no way to override that in configure.in. Oh
 # well.
-cd `pkg-config --variable=prefix evolution-shell-@BASE_VERSION@`
+cd `pkg-config --variable=prefix evolution-shell`
 zip $ZIP -@ <<EOF
 libexec/evolution/@BASE_VERSION@/evolution-exchange-storage.exe
--- camel/Makefile.in
+++ camel/Makefile.in
@@ -291,6 +291,7 @@
 	$(CAMEL_CFLAGS)				\
 	$(SHELL_CFLAGS)				\
 	$(LIBEXCHANGE_CFLAGS)			\
+	-DPREFIX=\"$(prefix)\"			\
 	-DCONNECTOR_LOCALEDIR=\"$(localedir)\"	\
 	-DG_LOG_DOMAIN=\"camel-exchange-provider\"	
 
@@ -317,11 +317,13 @@
 	camel-stub-marshal.h			\
 	camel-stub.h
 
-libcamelexchange_la_LDFLAGS = -avoid-version -module
+libcamelexchange_la_LDFLAGS = -avoid-version -module $(NO_UNDEFINED)
 libcamelexchange_la_LIBADD = \
 	$(LDAP_LIBS)				\
 	$(LIBEXCHANGE_LIBS)			\
-	$(EXCHANGE_STORAGE_LIBS)
+	$(EXCHANGE_STORAGE_LIBS)		\
+	$(SOCKET_LIBS)				\
+	$(PTHREAD_LIB)
 
 EXTRA_DIST = libcamelexchange.urls
 all: all-am
--- storage/Makefile.in
+++ storage/Makefile.in
@@ -344,7 +344,8 @@
 	$(top_builddir)/camel/camel-stub-marshal.lo		\
 	$(LDAP_LIBS)						\
 	$(EXCHANGE_STORAGE_LIBS)				\
-	$(LIBEXCHANGE_LIBS)
+	$(LIBEXCHANGE_LIBS)					\
+	$(SOCKET_LIBS)
 
 exchange_connector_setup_SOURCES = \
 	ximian-connector-setup.c				\
@@ -786,9 +786,10 @@
 	$(LIBTOOL) --mode=install $(INSTALL_PROGRAM) exchange-connector-setup $(DESTDIR)$(bindir)/exchange-connector-setup-$(BASE_VERSION)
 
 %_$(BASE_VERSION).server.in: %.server.in.in
-	sed -e "s;\@CONNECTOR_PATH\@;${EVOLUTION_privlibexecdir};g"	\
+	sed -e "s;\@CONNECTOR_PATH\@;../../../libexec/evolution/${BASE_VERSION};g"	\
 	    -e "s;\@IMAGESDIR\@;${imagesdir};g"				\
 	    -e "s;\@BASE_VERSION\@;${BASE_VERSION};g"			\
+	    -e "s;\@EXEEXT\@;${EXEEXT};g"				\
 	    -e "s;\@API_VERSION\@;${API_VERSION};g"			\
 	    -e "s;\@EDS_BASE_VERSION\@;${EDS_BASE_VERSION};g"		\
 	    < $< > $@
--- addressbook/Makefile.in
+++ addressbook/Makefile.in
@@ -608,7 +608,7 @@
 install-data-local:
 	if test -z "$(DESTDIR)" ; then \
 		for p in $(schema_DATA) ; do \
-			GCONF_CONFIG_SOURCE=$(GCONF_SCHEMA_CONFIG_SOURCE) $(GCONFTOOL) --makefile-install-rule $$p; \
+			(echo set GCONF_CONFIG_SOURCE=$(GCONF_SCHEMA_CONFIG_SOURCE); echo $(GCONFTOOL) --makefile-install-rule $$p) | cmd; \
 		done \
 	fi
 # Tell versions [3.59,3.63) of GNU make to not export all variables.
EOF

/opt/autotools/bin/autoconf

usedev

MY_PKG_CONFIG_PATH=""
for D in $DEPS; do
    PATH=/devel/dist/$D/bin:$PATH
    MY_PKG_CONFIG_PATH=/devel/dist/$D/lib/pkgconfig:$MY_PKG_CONFIG_PATH
done

PKG_CONFIG_PATH=$MY_PKG_CONFIG_PATH:$PKG_CONFIG_PATH CC='gcc -mtune=pentium3 -mms-bitfields -mthreads' CPPFLAGS='-I/opt/win_iconv/include -I/opt/proxy-libintl/include -I/opt/gnuwin32/include -I/opt/misc/include -I/devel/dist/popt-1.10.2-tml-20050828/include  -I/opt/pthread/include' LDFLAGS='-Wl,--enable-runtime-pseudo-reloc -L/opt/proxy-libintl/lib -Wl,--exclude-libs=libintl.a -L/opt/gnuwin32/lib -L/opt/misc/lib -L/devel/dist/popt-1.10.2-tml-20050828/lib -L/opt/pthread/lib' CFLAGS=-O ./configure --disable-gtk-doc --disable-static --with-e2k-debug --with-gconf-source='xml::${sysconfdir}/gconf/gconf.xml.defaults' --prefix=c:/devel/target/$HEX &&
libtoolcacheize &&

PATH=/devel/target/$HEX/bin:.libs:$PATH make install &&
PKG_CONFIG_PATH=$MY_PKG_CONFIG_PATH:$PKG_CONFIG_PATH ./evolution-exchange-zip &&

# the evolution-exchange-zip script doesn't create a dev zipfile, so do a dummy one
zip /tmp/$MOD-dev-$VER.zip nul &&
zip -d /tmp/$MOD-dev-$VER.zip nul &&

(cd /devel/src/tml && zip /tmp/$MOD-dev-$VER.zip make/$THIS.sh) &&

manifestify /tmp/$MOD-$VER.zip /tmp/$MOD-dev-$VER.zip
