/*
 * e-memo-shell-view-private.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the program; if not, see <http://www.gnu.org/licenses/>
 *
 *
 * Copyright (C) 1999-2008 Novell, Inc. (www.novell.com)
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "e-util/e-util-private.h"

#include "e-memo-shell-view-private.h"

static void
memo_shell_view_model_row_appended_cb (EMemoShellView *memo_shell_view,
                                       ECalModel *model)
{
	EMemoShellSidebar *memo_shell_sidebar;
	ECalClient *client;

	/* This is the "Click to Add" handler. */

	client = e_cal_model_ref_default_client (model);

	memo_shell_sidebar = memo_shell_view->priv->memo_shell_sidebar;
	e_memo_shell_sidebar_add_client (memo_shell_sidebar, E_CLIENT (client));

	g_object_unref (client);
}

static void
memo_shell_view_table_popup_event_cb (EShellView *shell_view,
                                      GdkEvent *button_event)
{
	const gchar *widget_path;

	widget_path = "/memo-popup";
	e_shell_view_show_popup_menu (shell_view, widget_path, button_event);
}

static void
memo_shell_view_selector_client_added_cb (EMemoShellView *memo_shell_view,
                                          ECalClient *client)
{
	EMemoShellContent *memo_shell_content;
	EMemoTable *memo_table;
	ECalModel *model;

	memo_shell_content = memo_shell_view->priv->memo_shell_content;
	memo_table = e_memo_shell_content_get_memo_table (memo_shell_content);
	model = e_memo_table_get_model (memo_table);

	e_cal_model_add_client (model, client);
}

static void
memo_shell_view_selector_client_removed_cb (EMemoShellView *memo_shell_view,
                                            ECalClient *client)
{
	EMemoShellContent *memo_shell_content;
	EMemoTable *memo_table;
	ECalModel *model;

	memo_shell_content = memo_shell_view->priv->memo_shell_content;
	memo_table = e_memo_shell_content_get_memo_table (memo_shell_content);
	model = e_memo_table_get_model (memo_table);

	e_cal_model_remove_client (model, client);
}

static gboolean
memo_shell_view_selector_popup_event_cb (EShellView *shell_view,
                                         ESource *primary_source,
                                         GdkEvent *button_event)
{
	const gchar *widget_path;

	widget_path = "/memo-list-popup";
	e_shell_view_show_popup_menu (shell_view, widget_path, button_event);

	return TRUE;
}

static void
memo_shell_view_backend_error_cb (EClientCache *client_cache,
                                  EClient *client,
                                  EAlert *alert,
                                  EMemoShellView *memo_shell_view)
{
	EMemoShellContent *memo_shell_content;
	ESource *source;
	const gchar *extension_name;

	memo_shell_content = memo_shell_view->priv->memo_shell_content;

	source = e_client_get_source (client);
	extension_name = E_SOURCE_EXTENSION_MEMO_LIST;

	/* Only submit alerts from memo list backends. */
	if (e_source_has_extension (source, extension_name)) {
		EAlertSink *alert_sink;

		alert_sink = E_ALERT_SINK (memo_shell_content);
		e_alert_sink_submit_alert (alert_sink, alert);
	}
}

static void
memo_shell_view_load_view_collection (EShellViewClass *shell_view_class)
{
	GalViewCollection *collection;
	GalViewFactory *factory;
	ETableSpecification *spec;
	const gchar *base_dir;
	gchar *filename;

	collection = shell_view_class->view_collection;

	base_dir = EVOLUTION_ETSPECDIR;
	spec = e_table_specification_new ();
	filename = g_build_filename (base_dir, ETSPEC_FILENAME, NULL);
	if (!e_table_specification_load_from_file (spec, filename))
		g_critical (
			"Unable to load ETable specification file "
			"for memos");
	g_free (filename);

	factory = gal_view_factory_etable_new (spec);
	gal_view_collection_add_factory (collection, factory);
	g_object_unref (factory);
	g_object_unref (spec);

	gal_view_collection_load (collection);
}

static void
memo_shell_view_notify_view_id_cb (EMemoShellView *memo_shell_view)
{
	EMemoShellContent *memo_shell_content;
	GalViewInstance *view_instance;
	const gchar *view_id;

	memo_shell_content = memo_shell_view->priv->memo_shell_content;
	view_instance =
		e_memo_shell_content_get_view_instance (memo_shell_content);
	view_id = e_shell_view_get_view_id (E_SHELL_VIEW (memo_shell_view));

	/* A NULL view ID implies we're in a custom view.  But you can
	 * only get to a custom view via the "Define Views" dialog, which
	 * would have already modified the view instance appropriately.
	 * Furthermore, there's no way to refer to a custom view by ID
	 * anyway, since custom views have no IDs. */
	if (view_id == NULL)
		return;

	gal_view_instance_set_current_view_id (view_instance, view_id);
}

void
e_memo_shell_view_private_init (EMemoShellView *memo_shell_view,
                                EShellViewClass *shell_view_class)
{
	if (!gal_view_collection_loaded (shell_view_class->view_collection))
		memo_shell_view_load_view_collection (shell_view_class);

	g_signal_connect (
		memo_shell_view, "notify::view-id",
		G_CALLBACK (memo_shell_view_notify_view_id_cb), NULL);
}

void
e_memo_shell_view_private_constructed (EMemoShellView *memo_shell_view)
{
	EMemoShellViewPrivate *priv = memo_shell_view->priv;
	EMemoShellContent *memo_shell_content;
	EMemoShellSidebar *memo_shell_sidebar;
	EShellBackend *shell_backend;
	EShellContent *shell_content;
	EShellSidebar *shell_sidebar;
	EShellWindow *shell_window;
	EShellView *shell_view;
	EShell *shell;
	EMemoTable *memo_table;
	ECalModel *model;
	ESourceSelector *selector;
	gulong handler_id;

	shell_view = E_SHELL_VIEW (memo_shell_view);
	shell_backend = e_shell_view_get_shell_backend (shell_view);
	shell_content = e_shell_view_get_shell_content (shell_view);
	shell_sidebar = e_shell_view_get_shell_sidebar (shell_view);
	shell_window = e_shell_view_get_shell_window (shell_view);
	shell = e_shell_window_get_shell (shell_window);

	e_shell_window_add_action_group (shell_window, "memos");
	e_shell_window_add_action_group (shell_window, "memos-filter");

	/* Cache these to avoid lots of awkward casting. */
	priv->memo_shell_backend = g_object_ref (shell_backend);
	priv->memo_shell_content = g_object_ref (shell_content);
	priv->memo_shell_sidebar = g_object_ref (shell_sidebar);

	memo_shell_content = E_MEMO_SHELL_CONTENT (shell_content);
	memo_table = e_memo_shell_content_get_memo_table (memo_shell_content);
	model = e_memo_table_get_model (memo_table);

	memo_shell_sidebar = E_MEMO_SHELL_SIDEBAR (shell_sidebar);
	selector = e_memo_shell_sidebar_get_selector (memo_shell_sidebar);

	/* Keep our own reference to this so we can
	 * disconnect our signal handler in dispose(). */
	priv->client_cache = g_object_ref (e_shell_get_client_cache (shell));

	handler_id = g_signal_connect (
		priv->client_cache, "backend-error",
		G_CALLBACK (memo_shell_view_backend_error_cb),
		memo_shell_view);
	priv->backend_error_handler_id = handler_id;

	g_signal_connect_swapped (
		model, "row-appended",
		G_CALLBACK (memo_shell_view_model_row_appended_cb),
		memo_shell_view);

	g_signal_connect_swapped (
		memo_table, "open-component",
		G_CALLBACK (e_memo_shell_view_open_memo),
		memo_shell_view);

	g_signal_connect_swapped (
		memo_table, "popup-event",
		G_CALLBACK (memo_shell_view_table_popup_event_cb),
		memo_shell_view);

	g_signal_connect_swapped (
		memo_table, "selection-change",
		G_CALLBACK (e_memo_shell_view_update_sidebar),
		memo_shell_view);

	g_signal_connect_swapped (
		memo_table, "selection-change",
		G_CALLBACK (e_shell_view_update_actions),
		memo_shell_view);

	g_signal_connect_swapped (
		memo_table, "status-message",
		G_CALLBACK (e_memo_shell_view_set_status_message),
		memo_shell_view);

	g_signal_connect_swapped (
		model, "model-changed",
		G_CALLBACK (e_memo_shell_view_update_sidebar),
		memo_shell_view);

	g_signal_connect_swapped (
		model, "model-rows-deleted",
		G_CALLBACK (e_memo_shell_view_update_sidebar),
		memo_shell_view);

	g_signal_connect_swapped (
		model, "model-rows-inserted",
		G_CALLBACK (e_memo_shell_view_update_sidebar),
		memo_shell_view);

	g_signal_connect_swapped (
		memo_shell_sidebar, "client-added",
		G_CALLBACK (memo_shell_view_selector_client_added_cb),
		memo_shell_view);

	g_signal_connect_swapped (
		memo_shell_sidebar, "client-removed",
		G_CALLBACK (memo_shell_view_selector_client_removed_cb),
		memo_shell_view);

	g_signal_connect_swapped (
		selector, "popup-event",
		G_CALLBACK (memo_shell_view_selector_popup_event_cb),
		memo_shell_view);

	g_signal_connect_swapped (
		selector, "primary-selection-changed",
		G_CALLBACK (e_shell_view_update_actions),
		memo_shell_view);

	e_categories_add_change_hook (
		(GHookFunc) e_memo_shell_view_update_search_filter,
		memo_shell_view);

	/* Keep the ECalModel in sync with the sidebar. */
	g_object_bind_property (
		shell_sidebar, "default-client",
		model, "default-client",
		G_BINDING_SYNC_CREATE);

	e_memo_shell_view_actions_init (memo_shell_view);
	e_memo_shell_view_update_sidebar (memo_shell_view);
	e_memo_shell_view_update_search_filter (memo_shell_view);
}

void
e_memo_shell_view_private_dispose (EMemoShellView *memo_shell_view)
{
	EMemoShellViewPrivate *priv = memo_shell_view->priv;

	if (priv->backend_error_handler_id > 0) {
		g_signal_handler_disconnect (
			priv->client_cache,
			priv->backend_error_handler_id);
		priv->backend_error_handler_id = 0;
	}

	g_clear_object (&priv->memo_shell_backend);
	g_clear_object (&priv->memo_shell_content);
	g_clear_object (&priv->memo_shell_sidebar);

	g_clear_object (&priv->client_cache);

	if (memo_shell_view->priv->activity != NULL) {
		/* XXX Activity is not cancellable. */
		e_activity_set_state (
			memo_shell_view->priv->activity,
			E_ACTIVITY_COMPLETED);
		g_object_unref (memo_shell_view->priv->activity);
		memo_shell_view->priv->activity = NULL;
	}
}

void
e_memo_shell_view_private_finalize (EMemoShellView *memo_shell_view)
{
	/* XXX Nothing to do? */
}

void
e_memo_shell_view_open_memo (EMemoShellView *memo_shell_view,
                             ECalModelComponent *comp_data)
{
	EShell *shell;
	EShellView *shell_view;
	EShellWindow *shell_window;
	ESourceRegistry *registry;
	CompEditor *editor;
	CompEditorFlags flags = 0;
	ECalComponent *comp;
	icalcomponent *clone;
	const gchar *uid;

	g_return_if_fail (E_IS_MEMO_SHELL_VIEW (memo_shell_view));
	g_return_if_fail (E_IS_CAL_MODEL_COMPONENT (comp_data));

	shell_view = E_SHELL_VIEW (memo_shell_view);
	shell_window = e_shell_view_get_shell_window (shell_view);
	shell = e_shell_window_get_shell (shell_window);

	registry = e_shell_get_registry (shell);

	uid = icalcomponent_get_uid (comp_data->icalcomp);
	editor = comp_editor_find_instance (uid);

	if (editor != NULL)
		goto exit;

	comp = e_cal_component_new ();
	clone = icalcomponent_new_clone (comp_data->icalcomp);
	e_cal_component_set_icalcomponent (comp, clone);

	if (e_cal_component_has_organizer (comp))
		flags |= COMP_EDITOR_IS_SHARED;

	if (itip_organizer_is_user (registry, comp, comp_data->client))
		flags |= COMP_EDITOR_USER_ORG;

	editor = memo_editor_new (comp_data->client, shell, flags);
	comp_editor_edit_comp (editor, comp);

	g_object_unref (comp);

exit:
	gtk_window_present (GTK_WINDOW (editor));
}

void
e_memo_shell_view_set_status_message (EMemoShellView *memo_shell_view,
                                      const gchar *status_message,
                                      gdouble percent)
{
	EActivity *activity;
	EShellView *shell_view;
	EShellBackend *shell_backend;

	g_return_if_fail (E_IS_MEMO_SHELL_VIEW (memo_shell_view));

	activity = memo_shell_view->priv->activity;
	shell_view = E_SHELL_VIEW (memo_shell_view);
	shell_backend = e_shell_view_get_shell_backend (shell_view);

	if (status_message == NULL || *status_message == '\0') {
		if (activity != NULL) {
			e_activity_set_state (activity, E_ACTIVITY_COMPLETED);
			g_object_unref (activity);
			activity = NULL;
		}

	} else if (activity == NULL) {
		activity = e_activity_new ();
		e_activity_set_percent (activity, percent);
		e_activity_set_text (activity, status_message);
		e_shell_backend_add_activity (shell_backend, activity);

	} else {
		e_activity_set_percent (activity, percent);
		e_activity_set_text (activity, status_message);
	}

	memo_shell_view->priv->activity = activity;
}

void
e_memo_shell_view_update_sidebar (EMemoShellView *memo_shell_view)
{
	EMemoShellContent *memo_shell_content;
	EShellView *shell_view;
	EShellSidebar *shell_sidebar;
	EMemoTable *memo_table;
	ECalModel *model;
	GString *string;
	const gchar *format;
	gint n_rows;
	gint n_selected;

	shell_view = E_SHELL_VIEW (memo_shell_view);
	shell_sidebar = e_shell_view_get_shell_sidebar (shell_view);

	memo_shell_content = memo_shell_view->priv->memo_shell_content;
	memo_table = e_memo_shell_content_get_memo_table (memo_shell_content);

	model = e_memo_table_get_model (memo_table);

	n_rows = e_table_model_row_count (E_TABLE_MODEL (model));
	n_selected = e_table_selected_count (E_TABLE (memo_table));

	string = g_string_sized_new (64);

	format = ngettext ("%d memo", "%d memos", n_rows);
	g_string_append_printf (string, format, n_rows);

	if (n_selected > 0) {
		format = _("%d selected");
		g_string_append_len (string, ", ", 2);
		g_string_append_printf (string, format, n_selected);
	}

	e_shell_sidebar_set_secondary_text (shell_sidebar, string->str);

	g_string_free (string, TRUE);
}

