/****************************************************************************
 * boards/arm/sam34/sam4l-xplained/src/sam_ug2832hsweg04.c
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/* OLED1 Connector:
 *
 * See the board README file for connector pin-out.
 *
 * OLED1 signals
 *
 * DATA_CMD_SEL - Data/command select. Used to choose whether the
 *   communication is data to the display memory or a command to the LCD
 *   controller. High = data, low = command
 * DISPLAY_RESET - Reset signal to the OLED display, active low. Used during
 *   initialization of the display.
 * DISPLAY_SS - SPI slave select signal, must be held low during SPI
 *   communication.
 * SPI_MOSI - SPI master out, slave in signal. Used to write data to the
 *   display
 * SPI_SCK SPI - clock signal, generated by the master.
 */

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <debug.h>

#include <nuttx/board.h>
#include <nuttx/spi/spi.h>
#include <nuttx/lcd/lcd.h>
#include <nuttx/lcd/ssd1306.h>

#include "sam_gpio.h"
#include "sam_spi.h"

#include "sam4l-xplained.h"

#ifdef CONFIG_SAM4L_XPLAINED_OLED1MODULE

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

/* Configuration ************************************************************/

/* The pin configurations here require that SPI1 is selected */

#ifndef CONFIG_LCD_SSD1306
#  error "The OLED driver requires CONFIG_LCD_SSD1306 in the configuration"
#endif

#ifndef CONFIG_LCD_UG2832HSWEG04
#  error "The OLED driver requires CONFIG_LCD_UG2832HSWEG04 in the configuration"
#endif

#ifndef CONFIG_SAM34_SPI0
#  error "The OLED driver requires CONFIG_SAM34_SPI0 in the configuration"
#endif

#ifndef CONFIG_SPI_CMDDATA
#  error "The OLED driver requires CONFIG_SPI_CMDDATA in the configuration"
#endif

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: board_graphics_setup
 *
 * Description:
 *   Called by NX initialization logic to configure the OLED.
 *
 ****************************************************************************/

FAR struct lcd_dev_s *board_graphics_setup(unsigned int devno)
{
  FAR struct spi_dev_s *spi;
  FAR struct lcd_dev_s *dev;

  /* Configure the OLED GPIOs. This initial configuration is RESET low,
   * putting the OLED into reset state.
   */

  sam_configgpio(GPIO_OLED_RST);

  /* Wait a bit then release the OLED from the reset state */

  up_mdelay(20);
  sam_gpiowrite(GPIO_OLED_RST, true);

  /* Get the SPI1 port interface */

  spi = sam_spibus_initialize(OLED_CSNO);
  if (!spi)
    {
      lcderr("ERROR: Failed to initialize SPI port 1\n");
    }
  else
    {
      /* Bind the SPI port to the OLED */

      dev = ssd1306_initialize(spi, NULL, devno);
      if (!dev)
        {
          lcderr("ERROR: Failed to bind SPI port 1 to OLED %d\n", devno);
        }
     else
        {
          lcdinfo("Bound SPI port 1 to OLED %d\n", devno);

          /* And turn the OLED on */

          dev->setpower(dev, CONFIG_LCD_MAXPOWER);
          return dev;
        }
    }

  return NULL;
}
#endif /* CONFIG_SAM4L_XPLAINED_OLED1MODULE */
