/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.core.svg;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import org.apache.commons.lang.SystemUtils;
import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.assertEquals;

public class HopSvgGraphics2DTest {

  @Before
  public void notOnWindows() {
    org.junit.Assume.assumeFalse(SystemUtils.IS_OS_WINDOWS);
  }

  private static final String BASIC_SVG_XML =
      "<?xml version=\"1.0\" encoding=\"UTF-8\"?><svg fill-opacity=\"1\" color-rendering=\"auto\" color-interpolation=\"auto\" text-rendering=\"auto\" "
          + "stroke=\"black\" stroke-linecap=\"square\" stroke-miterlimit=\"10\" shape-rendering=\"auto\" stroke-opacity=\"1\" fill=\"black\" stroke-dasharray=\"none\" font-weight=\"normal\" "
          + "stroke-width=\"1\" font-family=\"'Dialog'\" font-style=\"normal\" stroke-linejoin=\"miter\" font-size=\"12px\" stroke-dashoffset=\"0\" image-rendering=\"auto\" xmlns=\"http://www.w3"
          + ".org/2000/svg\">\n"
          + "  <!--Generated by the Batik Graphics2D SVG Generator-->\n"
          + "  <defs id=\"genericDefs\"/>\n"
          + "  <g/>\n"
          + "</svg>\n";

  private static final String BASIC_CIRCLE_XML =
      "<?xml version=\"1.0\" encoding=\"UTF-8\"?><svg fill-opacity=\"1\" color-rendering=\"auto\" color-interpolation=\"auto\" text-rendering=\"auto\" "
          + "stroke=\"black\" stroke-linecap=\"square\" stroke-miterlimit=\"10\" shape-rendering=\"auto\" stroke-opacity=\"1\" fill=\"black\" stroke-dasharray=\"none\" font-weight=\"normal\" "
          + "stroke-width=\"1\" font-family=\"'Dialog'\" font-style=\"normal\" stroke-linejoin=\"miter\" font-size=\"12px\" stroke-dashoffset=\"0\" image-rendering=\"auto\" xmlns=\"http://www.w3"
          + ".org/2000/svg\">\n"
          + "  <!--Generated by the Batik Graphics2D SVG Generator-->\n"
          + "  <defs id=\"genericDefs\"/>\n"
          + "  <g>\n"
          + "    <g>\n"
          + "      <circle fill=\"none\" r=\"12.5\" cx=\"62.5\" cy=\"62.5\"/>\n"
          + "    </g>\n"
          + "  </g>\n"
          + "</svg>\n";

  @Test
  public void testNewDocumentXml() throws Exception {
    HopSvgGraphics2D graphics2D = HopSvgGraphics2D.newDocument();
    XmlMapper xmlMapper = new XmlMapper();
    JsonNode graphic2DNode = xmlMapper.readTree(graphics2D.toXml());
    JsonNode basicSVGNode = xmlMapper.readTree(BASIC_SVG_XML);
    assertEquals(basicSVGNode, graphic2DNode);
  }

  @Test
  public void testNewDocumentSimpleXml() throws Exception {
    HopSvgGraphics2D graphics2D = HopSvgGraphics2D.newDocument();

    graphics2D.drawOval(50, 50, 25, 25);

    XmlMapper xmlMapper = new XmlMapper();
    JsonNode graphic2DNode = xmlMapper.readTree(graphics2D.toXml());
    JsonNode basicSVGNode = xmlMapper.readTree(BASIC_CIRCLE_XML);
    assertEquals(basicSVGNode, graphic2DNode);
  }
}
