/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.store.connector.source;

import org.apache.flink.api.connector.source.Boundedness;
import org.apache.flink.api.connector.source.SplitEnumerator;
import org.apache.flink.api.connector.source.SplitEnumeratorContext;
import org.apache.flink.table.store.file.predicate.Predicate;
import org.apache.flink.table.store.table.Table;
import org.apache.flink.table.store.table.source.TableScan;

import javax.annotation.Nullable;

import java.util.Collection;

/** A {@link FlinkSource} for system table. */
public class SimpleSystemSource extends FlinkSource {

    private static final long serialVersionUID = 2L;

    public SimpleSystemSource(
            Table table,
            @Nullable int[][] projectedFields,
            @Nullable Predicate predicate,
            @Nullable Long limit) {
        super(table, projectedFields, predicate, limit);
    }

    @Override
    public Boundedness getBoundedness() {
        return Boundedness.BOUNDED;
    }

    @Override
    public SplitEnumerator<FileStoreSourceSplit, PendingSplitsCheckpoint> restoreEnumerator(
            SplitEnumeratorContext<FileStoreSourceSplit> context,
            PendingSplitsCheckpoint checkpoint) {
        TableScan scan = table.newScan();
        if (predicate != null) {
            scan.withFilter(predicate);
        }

        Collection<FileStoreSourceSplit> splits =
                checkpoint == null
                        ? new FileStoreSourceSplitGenerator().createSplits(scan.plan())
                        : checkpoint.splits();

        return new StaticFileStoreSplitEnumerator(context, null, splits);
    }
}
