package org.apache.torque.generator.template.groovy;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import groovy.lang.Binding;
import groovy.lang.GroovyShell;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.torque.generator.GeneratorException;
import org.apache.torque.generator.configuration.ConfigurationException;
import org.apache.torque.generator.configuration.ConfigurationProvider;
import org.apache.torque.generator.control.ControllerState;
import org.apache.torque.generator.outlet.OutletResult;
import org.apache.torque.generator.qname.QualifiedName;
import org.apache.torque.generator.template.TemplateOutletImpl;


/**
 * A Outlet which uses a groovy script for generation.
 */
public class GroovyOutlet extends TemplateOutletImpl
{
    /**
     * The name under which the Torque generator interface will be put
     * into the context.
     */
    public static final String TORQUE_GEN_CONTEXT_NAME = "torqueGen";

    /** The log. */
    private static Log log = LogFactory.getLog(GroovyOutlet.class);

    /**
     * Constructs a new GroovyOutlet.
     *
     * @param name the name of this outlet, not null.
     * @param configurationProvider the provider for reading the templates,
     *        not null.
     * @param path the path to the templates, not null.
     *
     * @throws NullPointerException if name, path or directories are null.
     * @throws ConfigurationException if an error occurs while reading the
     *         template.
     */
    public GroovyOutlet(
            QualifiedName name,
            ConfigurationProvider configurationProvider,
            String path)
        throws ConfigurationException
    {
        super(name,
              configurationProvider,
              path,
              null,
              null);
    }

    /**
     * Executes the generation process; the result is returned.
     *
     * @param controllerState the current controller state.
     *
     * @return the result of the generation, not null.
     *
     * @see org.apache.torque.generator.outlet.Outlet#execute(ControllerState)
     */
    @Override
    public OutletResult execute(ControllerState controllerState)
        throws GeneratorException

    {
        if (log.isDebugEnabled())
        {
            log.debug("Start executing GroovyOutlet " + getName());
        }

        try
        {
            Binding binding = new Binding();
            GroovyShell shell = new GroovyShell(binding);

            String result = (String) shell.evaluate(
                    getContent(controllerState));
            return new OutletResult(result);
        }
        finally
        {
            if (log.isDebugEnabled())
            {
                log.debug("End executing GroovyOutlet " + getName());
            }
        }
    }
}
