/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* UNITS.C -- LaTeX units. */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "window.h"
#include "settings.h"
#include "canvas.h"

/* Latex units */
static char *units_tbl[] = { 
  "cm", "em", "ex", "in", "mm", "pc", "pt", 
};

#define F(N,D)	((long)((65536.0 * N) / D + .5))
#define I(N,D)	((long)((1e10 * D) / (65536.0 * N) + .5))
static long fix2sp_tbl[] ={ F(7227,	254),	/* cm */
			    F(10,	1),	/* em */
			    F(43,	10),	/* ex */
			    F(7227,	100),	/* in */
			    F(7227,	2540),	/* mm */
			    F(12,	1),	/* pc */
			    F(1,	1) };	/* pt */

static long sp2fix_tbl[] = {I(7227,	254),	/* cm */
			    I(10,	1),	/* em */
			    I(43,	10),	/* ex */
			    I(7227,	100),	/* in */
			    I(7227,	2540),	/* mm */
			    I(12,	1),	/* pc */
			    I(1,	1) };	/* pt */

/* See if string is a valid units string. */
LOCAL(UNITS_TYPE) units(char *s)
{
  char **p;

  p = str_search(s, units_tbl);
  return (p == NULL) ? uERROR : p - units_tbl;
}

/* Get a maximal string of digits, decimals, and letters. */
void read_TeXlength_str(FILE *f, char *buf, int buf_len)
{
  int i, ch;

  i = 0;
  while (isdigit(ch = getc(f)) || ch == '.' || isalpha(ch))
    if (i < buf_len)
      buf[i++] = ch;
  buf[i] = '\0';
}

/* Multiply two fixed point numbers stored in longs. */
LOCAL(long) fix_mul(long x, long y)
{
  ldiv_t xx = ldiv(x, 100000), yy = ldiv(y, 100000);
  return xx.quot * yy.quot * 100000 +
	 xx.rem * yy.quot +
	 xx.quot * yy.rem +
	 (((xx.rem/2ul) * (yy.rem/2ul) + 12500) / 25000ul);
}

/* Parse a string to obtain a TeX length in sp.
   Return 0 if the parse fails, else non-0. */
int str2TeXlength(char *str, SP *length_rtn, UNITS_TYPE *units_rtn)
{
  int ch, i;
  long w, n, d;
  UNITS_TYPE u;

  /* Start of `str' */
  i = 0;

  /* Read decimal number into three longs. */
  w = 0;	/* whole part */
  n = 0;	/* numerator of fraction */
  d = 0;	/* denominator of fraction (power of 10) */

  /* whole part */
  while (isdigit(ch = str[i++]))
    w = 10 * w + (ch - '0');

  /* fraction */
  if (ch == '.') 
    while (isdigit(ch = str[i++])) 
      if (d < 5) {
	n = 10 * n + (ch - '0');
	++d;
      }

  if ((u = units(&str[i - 1])) == uERROR)
    return 0;

  /* Adjust until den is 10^5. */
  while (d < 5) {
    n *= 10;
    ++d;
  }

  if (length_rtn != NULL)
    *length_rtn = fix_mul(w * 100000 + n, fix2sp_tbl[u]);
  if (units_rtn != NULL)
    *units_rtn = u;

  return 1;
}

/* Convert a length in TeX sp units to a 
   string with the given type of units. */
void TeXlength2str(SP len, char *str, UNITS_TYPE u)
{
  int i;
  ldiv_t r;

  /* Find fixed point units and split into whole and fraction. */
  r = ldiv(fix_mul(len, sp2fix_tbl[u]), 100000);

  /* Put whole part in string. */
  i = sprintf(str, "%ld", r.quot);

  /* If there is a frac part, write its digits. */
  if (r.rem) {
    str[i++] = '.';
    do {
      r = ldiv(r.rem * 10, 100000);
      str[i++] = r.quot + '0';
    } while (r.rem);
  }
  strcpy(&str[i], units_tbl[u]);
}

#ifdef TEST

main()
{
  char buf[80];
  SP len;

  while (!feof(stdin)) {

    gets(buf);

    if (!str2TeXlength(buf, &len))
      printf("bad len\n");
    else {
      printf("%ldsp", len);
      TeXlength2str(len, buf, uMM);
      printf(" = %s\n", buf);
    }
  }
  return 0;
}

#endif