#!/usr/bin/env perl

# File        : makeglossariesgui.perl
# Author      : Nicola Talbot
# Date        : 16 Sept 2011
# Version     : 1.0
# Description : LaTeX2HTML implementation of makeglosssariesgui package.

package main;

@GRAPHICS_PATH = ('../images');

# read in dictionary
&read_app_dictionary();

&unlink_tmp_idx_map();

sub unlink_tmp_idx_map{
   # delete tmp IDX file at start if it exists.
   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (-f $tmpidxfile)
   {
      unlink $tmpidxfile;
   }

   # delete tmp Map file at start if it exists.
   local($tmpmapfile) = "Map.tmp";

   if (-f $tmpmapfile)
   {
      unlink $tmpmapfile;
   }
}

# adapt definitions in glossaries.perl to use popup windows

@{$gls_entries{'main'}} = ();
$gls_displayfirst{'main'} = "glsdisplayfirst";
$gls_display{'main'} = "glsdisplay";

%glsentry = ();

$acronymtype = 'main';

sub do_cmd_acronymtype{
   join('', $acronymtype, $_[0]);
}

sub get_firstkeyval{
   local($key,$_) = @_;
   local($value);

   s/\b$key\s*=$OP(\d+)$CP(.*)$OP\1$CP\s*(,|$)/$value=$2;','/es;
   undef($value) if $`=~/\b$key\s*=/;

   unless (defined($value))
   {
      s/(^|,)\s*$key\s*=\s*([^,]*)\s*(,|$)/,/s;
      $value=$2;
   }

   ($value,$_);
}

# need to get the value of the last key of a given name
# in the event of multiple occurences.
sub get_keyval{
   local($key,$_) = @_;
   local($value);

   while (/\b$key\s*=/)
   {
      ($value,$_) = &get_firstkeyval($key, $_);
      last unless (defined($value));
   }

   ($value,$_);
}

sub do_cmd_glsdisplay{
   local($_) = @_;
   local($text,$description,$symbol,$insert);

   $text = &missing_braces unless
        (s/$next_pair_pr_rx/$text=$2;''/eo);

   $description = &missing_braces unless
        (s/$next_pair_pr_rx/$description=$2;''/eo);

   $symbol = &missing_braces unless
        (s/$next_pair_pr_rx/$symbol=$2;''/eo);

   $insert = &missing_braces unless
        (s/$next_pair_pr_rx/$insert=$2;''/eo);

   "$text$insert" . $_;
}

sub do_cmd_glsdisplayfirst{
   local($_) = @_;
   local($text,$description,$symbol,$insert);

   $text = &missing_braces unless
        (s/$next_pair_pr_rx/$text=$2;''/eo);

   $description = &missing_braces unless
        (s/$next_pair_pr_rx/$description=$2;''/eo);

   $symbol = &missing_braces unless
        (s/$next_pair_pr_rx/$symbol=$2;''/eo);

   $insert = &missing_braces unless
        (s/$next_pair_pr_rx/$insert=$2;''/eo);

   "$text$insert" . $_;
}

# initialise a new glossary entry

sub gls_entry_init{
   local($label, $type, $name, $desc) = @_;

   %{$glsentry{$label}} =
     ( type => $type,
       name => $name,
       'sort' => $name, # not required for helpset
       description => $description,
       text => $name,
       first => $name,
       plural => "${name}s",
       firstplural => "${name}s",
       symbol => '',
       flag => 0
     );
}

# set the symbol for given glossary entry
sub gls_set_symbol{
   local($label, $symbol) = @_;

   if (&gls_entry_defined($label))
   {
      $glsentry{$label}{'symbol'} = $symbol;
   }
   else
   {
      &write_warnings("gls_set_symbol: glossary entry '$label' has not been defined");
   }
}

sub gls_get_symbol{
   local($label) = @_;
   local($symbol) = '';

   if (&gls_entry_defined($label))
   {
      $symbol = $glsentry{$label}{'symbol'};
   }
   else
   {
      &write_warnings("gls_get_symbol: glossary entry '$label' has not been defined");
   }

   $symbol;
}

sub gls_get_text{
   local($label) = @_;
   local($text) = '';

   if (&gls_entry_defined($label))
   {
      $text = $glsentry{$label}{'text'};
   }
   else
   {
      &write_warnings("gls_get_text: glossary entry '$label' has not been defined");
   }

   $text;
}

sub gls_set_text{
   local($label, $text) = @_;

   if (&gls_entry_defined($label))
   {
      $glsentry{$label}{'text'} = $text;
   }
   else
   {
      &write_warnings("gls_set_text: glossary entry '$label' has not been defined");
   }
}

sub gls_get_plural{
   local($label) = @_;
   local($plural) = '';

   if (&gls_entry_defined($label))
   {
      $plural = $glsentry{$label}{'plural'};
   }
   else
   {
      &write_warnings("gls_get_plural: glossary entry '$label' has not been defined");
   }

   $plural;
}

sub gls_set_plural{
   local($label, $plural) = @_;

   if (&gls_entry_defined($label))
   {
      $glsentry{$label}{'plural'} = $plural;
   }
   else
   {
      &write_warnings("gls_set_plural: glossary entry '$label' has not been defined");
   }
}

sub gls_get_firstplural{
   local($label) = @_;
   local($firstplural) = '';

   if (&gls_entry_defined($label))
   {
      $firstplural = $glsentry{$label}{'firstplural'};
   }
   else
   {
      &write_warnings("gls_get_firstplural: glossary entry '$label' has not been defined");
   }

   $firstplural;
}

sub gls_set_firstplural{
   local($label, $firstplural) = @_;

   if (&gls_entry_defined($label))
   {
      $glsentry{$label}{'firstplural'} = $firstplural;
   }
   else
   {
      &write_warnings("gls_set_firstplural: glossary entry '$label' has not been defined");
   }
}

sub gls_get_first{
   local($label) = @_;
   local($first) = '';

   if (&gls_entry_defined($label))
   {
      $first = $glsentry{$label}{'first'};
   }
   else
   {
      &write_warnings("gls_get_first: glossary entry '$label' has not been defined");
   }

   $first;
}

sub gls_set_first{
   local($label, $first) = @_;

   if (&gls_entry_defined($label))
   {
      $glsentry{$label}{'first'} = $first;
   }
   else
   {
      &write_warnings("gls_set_first: glossary entry '$label' has not been defined");
   }
}

sub gls_get_type{
   local($label) = @_;
   local($type) = '';

   if (&gls_entry_defined($label))
   {
      $type = $glsentry{$label}{'type'};
   }
   else
   {
      &write_warnings("gls_get_type: glossary entry '$label' has not been defined");
   }

   $type;
}

sub gls_get_displayfirst{
   local($type) = @_;
   local($display)= $gls_displayfirst{$type};

   if (not defined($display))
   {
      &write_warnings("Glossary '$type' is not defined");
      $display='';
   }
   elsif ($display eq '')
   {
      &write_warnings("glsdisplayfirst not set for glossary '$type'");
   }
   else
   {
      $display = "\\$display ";
   }

   $display;
}

sub gls_get_display{
   local($type) = @_;
   local($display)= $gls_display{$type};

   if (not defined($display))
   {
      &write_warnings("Glossary '$type' is not defined");
      $display = '';
   }
   elsif ($display eq '')
   {
      &write_warnings("glsdisplay not set for glossary '$type'");
   }
   else
   {
      $display = "\\$display ";
   }

   $display;
}


sub gls_used{
   local($label) = @_;
   local($flag) = 0;

   if (&gls_entry_defined($label))
   {
      $flag = $glsentry{$label}{'flag'};
   }
   else
   {
      &write_warnings("gls_used: glossary entry '$label' has not been defined");
   }

   $flag;
}

sub gls_entry_defined{
   local($label) = @_;

   defined(%{$glsentry{$label}});
}

sub reset_entry{
   local($label) = @_;

   $glsentry{$label}{'flag'} = 0;
}

sub unset_entry{
   local($label) = @_;

   $glsentry{$label}{'flag'} = 1;
}

sub do_cmd_glsreset{
   local($_) = @_;
   local($label);

   $label = &missing_braces unless
              s/$next_pair_pr_rx/$label=$2;''/eo;

   &reset_entry($label);

   $_;
}

sub do_cmd_glsunset{
   local($_) = @_;
   local($label);

   $label = &missing_braces unless
              s/$next_pair_pr_rx/$label=$2;''/eo;

   &unset_entry($label);

   $_;
}

sub do_cmd_ifglsused{
   local($_) = @_;
   local($label,$true,$false);

   $label = &missing_braces unless
              s/$next_pair_pr_rx/$label=$2;''/eo;

   $true = &missing_braces unless
              s/$next_pair_pr_rx/$true=$2;''/eo;

   $false = &missing_braces unless
              s/$next_pair_pr_rx/$false=$2;''/eo;

   (&gls_used($label) ? $true : $false) . $_;
}

sub do_cmd_ifglsentryexists{
   local($_) = @_;
   local($label,$true,$false);

   $label = &missing_braces unless
              s/$next_pair_pr_rx/$label=$2;''/eo;

   $true = &missing_braces unless
              s/$next_pair_pr_rx/$true=$2;''/eo;

   $false = &missing_braces unless
              s/$next_pair_pr_rx/$false=$2;''/eo;

   (&gls_entry_defined($label) ? $true : $false) . $_;
}

# define a new glossary entry

sub make_newglossaryentry{
   local($label,$name,$description,$symbol,$sort,$text,$first,
         $plural,$firstplural,$type) = @_;

   &gls_entry_init($label, $type, $name, $description);

   &gls_set_symbol($label, defined($symbol)?$symbol:'');

   $text = $name unless (defined($text) and $text);

   &gls_set_text($label, $text);

   $first = $text unless (defined($first) and $first);

   &gls_set_first($label, $first);

   $plural = "${text}s" unless (defined($plural) and $plural);

   &gls_set_plural($label, $plural);

   $firstplural = "${first}s" unless (defined($firstplural) and $firstplural);

   &gls_set_firstplural($label, $firstplural);

   push @{$gls_entries{$type}}, $label;

   # create an html file containing the information

   # remove links
   $description=~s/<\/?A([^>]*)>//sg;

   if (open FD, ">glo_$label.html")
   {
      print FD "<HTML>\n";
      print FD "<TITLE>$name</TITLE>\n";
      print FD "<BODY>\n";
      print FD "<H1>$name</H1>\n";
      print FD "$description.\n";
      print FD "</BODY>\n";
      print FD "</HTML>\n";
      close FD;
   }
   else
   {
      &write_warnings("unable to open 'glo_$label.html' $!");
   }
}

sub do_cmd_newglossaryentry{
   local($_) = @_;
   local($label,$name,$description,$symbol,$sort,$text,$first,
         $plural,$firstplural,$type,$keyval);

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   $keyval = &missing_braces unless
             s/$next_pair_pr_rx/$keyval=$2;''/eo;

   ($name,$keyval) = &get_keyval('name', $keyval);
   ($description,$keyval) = &get_keyval('description', $keyval);
   ($symbol,$keyval) = &get_keyval('symbol', $keyval);
   ($sort,$keyval) = &get_keyval('sort', $keyval);
   ($text,$keyval) = &get_keyval('text', $keyval);
   ($first,$keyval) = &get_keyval('first', $keyval);
   ($firstplural,$keyval) = &get_keyval('firstplural', $keyval);
   ($plural,$keyval) = &get_keyval('plural', $keyval);
   ($type,$keyval) = &get_keyval('type', $keyval);

   if (defined($type))
   {
      $type = &translate_commands($type);
   }
   else
   {
      $type = 'main';
   }

   &make_newglossaryentry($label,$name,$description,$symbol,$sort,
       $text,$first,$plural,$firstplural,$type);

   $_;
}

sub make_glslink{
   local($type, $label, $linktext) = @_;
   local($str) = '';

   if (&gls_entry_defined($label))
   {
      $str = &popupWindow($linktext, "glo_$label.html");
   }  
   else
   {
      &write_warnings("glossary entry '$label' undefined");
   }

   $str;
}

# redefine \xspace to produce &#032; instead of space character
sub do_cmd_xspace {
    local($_) = @_;
    local($space) = "&#032;";
    $space = "" if /^([{}~.!,:;?\/'\)-]|\\\/|\\ )/;
    $space.$_;
}

sub do_cmd_glslink{
   local($_)=@_;
   local($optarg,$pat) = &get_next_optional_argument;
   local($label,$linktext);

   $label = &missing_braces unless
             (s/$next_pair_pr_rx/$label=$2;''/eo);

   $linktext = &missing_braces unless
             (s/$next_pair_pr_rx/$linktext=$2;''/eo);

   local ($space) = '';
   if (s/^\s+//) {$space = '&#32;';}

   $type = &gls_get_type($label);

   &make_glslink($type, $label, $linktext).$space.$_;
}

sub do_cmd_glslinkstar{
   local($_)=@_;
   local($optarg,$pat) = &get_next_optional_argument;
   local($label,$linktext);

   $label = &missing_braces unless
             (s/$next_pair_pr_rx/$label=$2;''/eo);

   $linktext = &missing_braces unless
             (s/$next_pair_pr_rx/$linktext=$2;''/eo);

   local ($space) = '';
   if (s/^\s+//) {$space = '&#32;';}

   $linktext.$space.$_;
}

sub do_cmd_gls{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_text($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_first($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   &make_glslink($type, $label, $link_text).$space.$_;
}

sub do_cmd_glsstar{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_text($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_first($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   $link_text.$space.$_;
}

sub do_cmd_glspl{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_plural($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_firstplural($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   &make_glslink($type, $label, $link_text).$space.$_;
}

sub do_cmd_glsplstar{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_plural($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_firstplural($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   $link_text.$space.$_;
}

sub do_cmd_Gls{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_text($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_first($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   &make_glslink($type, $label, ucfirst($link_text)).$space.$_;
}

sub do_cmd_Glsstar{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_text($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_first($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   ucfirst($link_text).$space.$_;
}

sub do_cmd_Glspl{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_plural($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_firstplural($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   &make_glslink($type, $label, ucfirst($link_text)).$space.$_;
}

sub do_cmd_Glsplstar{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_plural($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_firstplural($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   ucfirst($link_text).$space.$_;
}

sub do_cmd_GLS{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_text($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_first($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   &make_glslink($type, $label, uc($link_text)).$space.$_;
}

sub do_cmd_GLSpl{
   local($_) = @_;
   local($label, $text, $insert);

   local($optarg, $pat) = &get_next_optional_argument;

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   local($space) = '';
   $space = '&#32;' if (/^\s+[^\[]/ or /^\s*\[.*\]\s/);

   $insert = '';
   ($insert,$pat) = &get_next_optional_argument;

   local($display) = '';

   local($type) = &gls_get_type($label);

   if (&gls_used($label))
   {
      # entry has already been used
      $text = &gls_get_plural($label);
      $display = &gls_get_display($type);
   }
   else
   {
      # entry hasn't been used

      $text = &gls_get_firstplural($label);
      $display = &gls_get_displayfirst($type);
      &unset_entry($label);
   }

   local($args) = '';

   local($id) = ++$global{'max_id'};
   $args .= "$OP$id$CP$text$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{description}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$glsentry{$label}{symbol}$OP$id$CP";

   $id = ++$global{'max_id'};
   $args .= "$OP$id$CP$insert$OP$id$CP";

   local($link_text) = &translate_commands("$display$args");

   &make_glslink($type, $label, uc($link_text)).$space.$_;
}

sub do_cmd_newacronym{
   local($_) = @_;
   local($label,$abbrev,$long,$keyval);

   ($keyval,$pat) = &get_next_optional_argument;

   $label = &missing_braces unless
        (s/$next_pair_pr_rx/$label=$2;''/eo);
   $abbrv = &missing_braces unless
        (s/$next_pair_pr_rx/$abbrv=$2;''/eo);
   $long = &missing_braces unless
        (s/$next_pair_pr_rx/$long=$2;''/eo);

   local($type) = &translate_commands("\\acronymtype");

   local($name,$description,$text,$first,$plural,$firstplural,
         $symbol);

   ($name,$keyval) = &get_keyval('name', $keyval);

   $name = $abbrv unless defined($name);

   ($description,$keyval) = &get_keyval('description', $keyval);

   $description= $long unless defined($description);

   ($symbol,$keyval) = &get_keyval('symbol', $keyval);
   ($sort,$keyval) = &get_keyval('sort', $keyval);

   ($text,$keyval) = &get_keyval('text', $keyval);

   $text = $abbrv unless defined($text);

   ($first,$keyval) = &get_keyval('first', $keyval);

   $first = "$long ($abbrv)" unless defined($first);

   ($firstplural,$keyval) = &get_keyval('firstplural', $keyval);

   $firstplural = "${long}s (${abbrv}s)" unless defined($firstplural);

   ($plural,$keyval) = &get_keyval('plural', $keyval);

   $plural = "${abbrv}s" unless defined($plural);

   &make_newglossaryentry($label,$name,$description,$symbol,$sort,
       $text,$first,$plural,$firstplural,$type);

   $_;
}

# end of glossaries stuff

sub popupWindow{
   local($text,$html)=@_;

   "<object classid=\"java:com.sun.java.help.impl.JHSecondaryViewer\">"
    . "   <param name=\"viewerStyle\" value=\"javax.help.Popup\">"
    . "   <param name=\"viewerSize\" value=\"400,200\">"
    . "   <param name=\"content\" value=\"$html\">"
    . "   <param name=\"viewerActivator\" value=\"javax.help.LinkLabel\">"
    . "   <param name=\"text\" value=\"$text\">" 
    . "   <param name=\"textColor\" value=\"blue\">"
    . "</object>"
}

sub secondaryWindow{
   local($text,$mapid)=@_;

   "$text<object classid=\"java:com.sun.java.help.impl.JHSecondaryViewer\">"
    . "   <param name=\"viewerStyle\" value=\"javax.help.SecondaryWindow\">"
    . "   <param name=\"viewerSize\" value=\"400,400\">"
    . "   <param name=\"id\" value=\"$mapid\">"
    . "   <param name=\"viewerActivator\" value=\"javax.help.LinkLabel\">"
    . "   <param name=\"text\" value=\"&gt;&gt;\">" 
    . "   <param name=\"textColor\" value=\"blue\">"
    . "</object>"
}

sub do_cmd_secondaryWindow{
   local($_)=@_;
   local($text,$mapid);

   $text = &missing_braces unless
                (s/$next_pair_pr_rx/$text=$2;''/eo);

   $mapid = &missing_braces unless
                (s/$next_pair_pr_rx/$mapid=$2;''/eo);

   join('', &secondaryWindow($text, $mapid), $_);
}

# override do_cmd_label so that it adds information to Map.tmp
sub do_cmd_label{
   local($_) = @_;
   local($label);

   $label = &missing_braces unless (
            (s/$next_pair_pr_rx\n?/$label = $2;''/eo)
            || (s/$next_pair_rx\n?/$label = $2;''/eo));

   &do_label_and_map($label, $_)
}

sub do_cmd_labelappitem{
    local($_) = @_;
    local($label, $title);

    $title = &missing_braces unless (
        (s/$next_pair_pr_rx\n?/$title = $2;''/eo)
        ||(s/$next_pair_rx\n?/$title = $2;''/eo));

    $label = &missing_braces unless (
        (s/$next_pair_pr_rx\n?/$label = $2;''/eo)
        ||(s/$next_pair_rx\n?/$label = $2;''/eo));

   &do_label_and_map($label, $_)
}

sub do_label_and_map{
   local($label, $_) = @_;

   if (open MAP, ">>Map.tmp")
   {
      print MAP "<mapID target=\"$label\" url=\"$CURRENT_FILE#$label\"/>\n";
   }
   else
   {
      &write_warnings("unable to append to file 'Map.tmp' $!");
   }

   &anchor_label($label, $CURRENT_FILE,$_);
}

sub do_cmd_cloptionNI{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   join('', '<TT>', $text, '</TT>', $_);
}

sub do_cmd_cloption{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   join('', '<TT>', $text, '</TT>', $_);
}

sub do_cmd_cloptionstar{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#$text><2><command line options><$text>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', '<A NAME="', $text,'">', '<TT>', $text, '</TT></A>', $_);
}

sub do_cmd_configkey{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   join('', '<TT>', $text, '</TT>', $_);
}

sub do_cmd_configkeystar{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#$text><2><configuration file (makeglossariesgui.conf)><$text>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', '<A NAME="', $text,'">', '<TT>', $text, '</TT></A>', $_);
}

sub do_cmd_configfile{
   local($_) = @_;

   "<TT>makeglossariesgui.conf</TT>" . $_;
}

sub do_cmd_configfilestar{
   local($_) = @_;

   local($this_file) = $CURRENT_FILE;
   local($label)="makeglossariesguiconf";

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#$label><2><configuration file (makeglossariesgui.conf)><$text>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }


   join('', '<A NAME="$label">', '<TT>makeglossariesgui.conf</TT></A>', $_);
}

sub do_cmd_Index{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   join('', $text, $_);
}

sub do_cmd_Indexstar{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#$text><1><$text>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', '<A NAME="', $text,'">', $text, '</A>', $_);
}

sub do_cmd_index{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   $_;
}

sub do_cmd_indexstar{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#$text><1><$text>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   "<A NAME=\"$text\"></A>$_";
}

sub do_cmd_subIndex{
   local($_) = @_;
   local($item,$subitem);

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$item=$2;''/eo;

   $subitem = &missing_braces unless
             s/$next_pair_pr_rx/$subitem=$2;''/eo;

   join('', $subitem, $_);
}

sub do_cmd_subIndexstar{
   local($_) = @_;
   local($item,$subitem);

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$item=$2;''/eo;

   $subitem = &missing_braces unless
             s/$next_pair_pr_rx/$subitem=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#${item}_$subitem><2><$item><$subitem>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"${item}_$subitem\">", $subitem, '</A>', $_);
}

sub do_cmd_subindex{
   local($_) = @_;
   local($item,$subitem);

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$item=$2;''/eo;

   $subitem = &missing_braces unless
             s/$next_pair_pr_rx/$subitem=$2;''/eo;

   $_;
}

sub do_cmd_subindexstar{
   local($_) = @_;
   local($item,$subitem);

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$item=$2;''/eo;

   $subitem = &missing_braces unless
             s/$next_pair_pr_rx/$subitem=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#${item}_$subitem><2><$item><$subitem>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   "<A NAME=\"${item}_$subitem\"></A>$_";
}

sub do_cmd_appname{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   local($id) = ++$global{'max_id'};
   join('', '<A NAME="app_', $text,'_',$id,'">',
        '<TT>', $text, '</TT></A>', $_);
}

sub do_cmd_appnameNI{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   join('', '<TT>', $text, '</TT>', $_);
}

sub do_cmd_menu{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   join('', '<B>', $text, '</B>', $_);
}

sub do_cmd_menustar{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#menu_$text><1><$text menu>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"menu_$text>", '"<B>', $text, '</B></A>', $_);
}

sub do_cmd_appmenu{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   local $key = "$text.label";

   if (defined($applang{$key}))
   {
      $text = $applang{$key};
   }
   else
   {
      &write_warnings("Unknown key '$key'");
   }

   join('', '<B>', $text, '</B>', $_);
}

sub do_cmd_appmenustar{
   local($_) = @_;
   local($text,$id);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$id=$1;$text=$2;''/eo;

   local $key = "$text.label";

   if (defined($applang{$key}))
   {
      $text = $applang{$key};
   }
   else
   {
      &write_warnings("Unknown key '$key'");
   }

   $_ = "$OP$id$CP$text$OP$id$CP". $_;

   &do_cmd_menustar($_);
}

sub do_cmd_menuitem{
   local($_) = @_;
   local($menuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   join('', '<B>', $menuname, '-&gt;',
        $itemname, '</B>', $_);
}

sub do_cmd_menuitemNI{
   local($_) = @_;
   local($menuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   join('', '<B>', $menuname, '-&gt;',
        $itemname, '</B>', $_);
}

sub do_cmd_menuitemstar{
   local($_) = @_;
   local($menuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#menu_${menuname}_$itemname><2><$menuname menu><$itemname>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"menu_${menuname}_$itemname\">", '<B>',
       $menuname, '-&gt;', $itemname, '</B></A>', $_);
}

sub do_cmd_appmenuitem{
   local($_) = @_;
   local($menu,$item);

   $menu = &missing_braces unless
             s/$next_pair_pr_rx/$menu=$2;''/eo;

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$item=$2;''/eo;

   local $key1 = "$menu.label";
   local $key2 = "$menu.$item";

   if (defined($applang{$key1}))
   {
      $menu = $applang{$key1};
   }
   else
   {
      &write_warnings("Unknown key '$key1'");
   }

   if (defined($applang{$key2}))
   {
      $item = $applang{$key2};
   }
   else
   {
      &write_warnings("Unknown key '$key2'");
   }

   join('', '<B>', $menu, '-&gt;', $item, '</B>', $_);
}

sub do_cmd_appmenuitemstar{
   local($_) = @_;
   local($menu, $item,$id1, $id2);

   $menu = &missing_braces unless
             s/$next_pair_pr_rx/$id1=$1;$menu=$2;''/eo;

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$id2=$1;$item=$2;''/eo;

   local $key1 = "$menu.label";
   local $key2 = "$menu.$item";

   if (defined($applang{$key1}))
   {
      $menu = $applang{$key1};
   }
   else
   {
      &write_warnings("Unknown key '$key1'");
   }

   if (defined($applang{$key2}))
   {
      $item = $applang{$key2};
   }
   else
   {
      &write_warnings("Unknown key '$key2'");
   }

   $_ = "$OP$id1$CP$menu$OP$id1$CP$OP$id2$CP$item$OP$id2$CP". $_;

   &do_cmd_menuitemstar($_);
}

sub do_cmd_submenuitem{
   local($_) = @_;
   local($menuname,$submenuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $submenuname = &missing_braces unless
             s/$next_pair_pr_rx/$submenuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   join('', '<B>', $menuname, '-&gt;',
        $submenuname, '-&gt;',
        $itemname, '</B>', $_);
}

sub do_cmd_submenuitemstar{
   local($_) = @_;
   local($menuname,$submenuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $submenuname = &missing_braces unless
             s/$next_pair_pr_rx/$submenuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#menu_${menuname}_${submenuname}_$itemname><3><$menuname menu><$submenuname><$itemname>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"menu_${menuname}_${submenuname}_$itemname\">", '<B>',
       $menuname, '-&gt;', $submenuname, '-&gt;',
       $itemname, '</B></A>', $_);
}

sub do_cmd_appsubmenuitem{
   local($_) = @_;
   local($menu,$submenu,$item);

   $menu = &missing_braces unless
             s/$next_pair_pr_rx/$menu=$2;''/eo;

   $submenu = &missing_braces unless
             s/$next_pair_pr_rx/$submenu=$2;''/eo;

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$item=$2;''/eo;

   local $key1 = "$menu.label";
   local $key2 = "$menu.$submenu";
   local $key3 = "$menu.$submenu.$item";

   if (defined($applang{$key1}))
   {
      $menu = $applang{$key1};
   }
   else
   {
      &write_warnings("Unknown key '$key1'");
   }

   if (defined($applang{$key2}))
   {
      $submenu = $applang{$key2};
   }
   else
   {
      &write_warnings("Unknown key '$key2'");
   }

   if (defined($applang{$key3}))
   {
      $item = $applang{$key3};
   }
   else
   {
      &write_warnings("Unknown key '$key3'");
   }

   join('', '<B>', $menu, '-&gt;', $submenu, '-&gt;',
        $item, '</B>', $_);
}

sub do_cmd_appsubmenuitemstar{
   local($_) = @_;
   local($menu, $submenu, $item,$id1, $id2, $id3);

   $menu = &missing_braces unless
             s/$next_pair_pr_rx/$id1=$1;$menu=$2;''/eo;

   $submenu = &missing_braces unless
             s/$next_pair_pr_rx/$id2=$1;$submenu=$2;''/eo;

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$id3=$1;$item=$2;''/eo;

   local $key1 = "$menu.label";
   local $key2 = "$menu.$submenu";
   local $key3 = "$menu.$submenu.$item";

   if (defined($applang{$key1}))
   {
      $menu = $applang{$key1};
   }
   else
   {
      &write_warnings("Unknown key '$key1'");
   }

   if (defined($applang{$key2}))
   {
      $submenu = $applang{$key2};
   }
   else
   {
      &write_warnings("Unknown key '$key2'");
   }

   if (defined($applang{$key3}))
   {
      $item = $applang{$key3};
   }
   else
   {
      &write_warnings("Unknown key '$key3'");
   }

   $_ = "$OP$id1$CP$menu$OP$id1$CP"
      . "$OP$id2$CP$submenu$OP$id2$CP"
      . "$OP$id3$CP$item$OP$id3$CP". $_;

   &do_cmd_submenuitemstar($_);
}

sub do_cmd_subsubmenuitem{
   local($_) = @_;
   local($menuname,$submenuname,$subsubmenuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $submenuname = &missing_braces unless
             s/$next_pair_pr_rx/$submenuname=$2;''/eo;

   $subsubmenuname = &missing_braces unless
             s/$next_pair_pr_rx/$subsubmenuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   join('', '<B>', $menuname, '-&gt;',
        $submenuname, '-&gt;',
        $subsubmenuname, '-&gt;',
        $itemname, '</B>', $_);
}

sub do_cmd_subsubmenuitemstar{
   local($_) = @_;
   local($menuname,$submenuname,$subsubmenuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $submenuname = &missing_braces unless
             s/$next_pair_pr_rx/$submenuname=$2;''/eo;

   $subsubmenuname = &missing_braces unless
             s/$next_pair_pr_rx/$subsubmenuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#menu_${menuname}_${submenuname}_$itemname><4><$menuname menu><$submenuname><$subsubmenuname><$itemname>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"menu_${menuname}_${submenuname}_${subsubmenuname}_$itemname\">", '<B>',
       $menuname, '-&gt;',
       $submenuname, '-&gt;',
       $subsubmenuname, '-&gt;',
       $itemname, '</B></A>', $_);
}

sub do_cmd_appsubsubmenuitem{
   local($_) = @_;
   local($menu,$submenu, $subsubmenu, $item);

   $menu = &missing_braces unless
             s/$next_pair_pr_rx/$menu=$2;''/eo;

   $submenu = &missing_braces unless
             s/$next_pair_pr_rx/$submenu=$2;''/eo;

   $subsubmenu = &missing_braces unless
             s/$next_pair_pr_rx/$subsubmenu=$2;''/eo;

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$item=$2;''/eo;

   local $key1 = "$menu.label";
   local $key2 = "$menu.$submenu";
   local $key3 = "$menu.$submenu.$subsubmenu";
   local $key4 = "$menu.$submenu.$subsubmenu.$item";

   if (defined($applang{$key1}))
   {
      $menu = $applang{$key1};
   }
   else
   {
      &write_warnings("Unknown key '$key1'");
   }

   if (defined($applang{$key2}))
   {
      $submenu = $applang{$key2};
   }
   else
   {
      &write_warnings("Unknown key '$key2'");
   }

   if (defined($applang{$key3}))
   {
      $subsubmenu = $applang{$key3};
   }
   else
   {
      &write_warnings("Unknown key '$key3'");
   }

   if (defined($applang{$key4}))
   {
      $item = $applang{$key4};
   }
   else
   {
      &write_warnings("Unknown key '$key4'");
   }

   join('', '<B>', $menu, '-&gt;', $submenu, '-&gt;',
        $subsubmenu, '-&gt;', $item, '</B>', $_);
}

sub do_cmd_appsubsubmenuitemstar{
   local($_) = @_;
   local($menu, $submenu, $subsubmenu, $item,$id1, $id2, $id3,$id4);

   $menu = &missing_braces unless
             s/$next_pair_pr_rx/$id1=$1;$menu=$2;''/eo;

   $submenu = &missing_braces unless
             s/$next_pair_pr_rx/$id2=$1;$submenu=$2;''/eo;

   $subsubmenu = &missing_braces unless
             s/$next_pair_pr_rx/$id3=$1;$subsubmenu=$2;''/eo;

   $item = &missing_braces unless
             s/$next_pair_pr_rx/$id4=$1;$item=$2;''/eo;

   local $key1 = "$menu.label";
   local $key2 = "$menu.$submenu";
   local $key3 = "$menu.$submenu.$subsubmenu";
   local $key4 = "$menu.$submenu.$subsubmenu.$item";

   if (defined($applang{$key1}))
   {
      $menu = $applang{$key1};
   }
   else
   {
      &write_warnings("Unknown key '$key1'");
   }

   if (defined($applang{$key2}))
   {
      $submenu = $applang{$key2};
   }
   else
   {
      &write_warnings("Unknown key '$key2'");
   }

   if (defined($applang{$key3}))
   {
      $subsubmenu = $applang{$key3};
   }
   else
   {
      &write_warnings("Unknown key '$key3'");
   }

   if (defined($applang{$key4}))
   {
      $item = $applang{$key4};
   }
   else
   {
      &write_warnings("Unknown key '$key4'");
   }

   $_ = "$OP$id1$CP$menu$OP$id1$CP"
      . "$OP$id2$CP$submenu$OP$id2$CP"
      . "$OP$id3$CP$subsubmenu$OP$id3$CP"
      . "$OP$id4$CP$item$OP$id4$CP". $_;

   &do_cmd_subsubmenuitemstar($_);
}

sub do_cmd_popupmenunameNI{
   local($_) = @_;

   local($id) = ++$global{'max_id'};

   "\\gls$OP$id${CP}popupmenu$OP$id$CP".$_
}

sub do_cmd_popupmenunameI{ "popup menu".$_[0] }

sub do_cmd_popupmenu{
   local($_) = @_;
   local($menuname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   local($space) = '';

   if (/^\s/)
   {
      $space = '&#032;';
   }

   join(" ",$menuname, "\\popupmenunameNI", $space.$_)
}

sub do_cmd_popupmenustar{
   local($_) = @_;
   local($menuname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#popupmenu_${menuname}><2><popup menu><$menuname>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"popupmenu_${menuname}\">", 
       $menuname, '</A> \popupmenunameNI ', $_);
}

sub do_cmd_popupmenuitem{
   local($_) = @_;
   local($menuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   join('', '<B>', $itemname, '</B>', $_);
}

sub do_cmd_popupmenuitemstar{
   local($_) = @_;
   local($menuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#popupmenu_${menuname}_$itemname><3><popup menu><$menuname><$itemname>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"popupmenu_${menuname}_$itemname\">", '<B>',
       $itemname, '</B></A>', $_);
}

sub do_cmd_apppopupmenuitem{
   local($_) = @_;
   local($menuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   local ($key);

   if ($menuname=~/text\s+area/s)
   {
      $key = "text.$itemname";
   }
   elsif ($menuname=~/edit\s+path/s)
   {
      $key = "editpath.$itemname";
   }
   else
   {
      &write_warnings("Unknown popup menu '$menuname'");
   }

   unless (defined($applang{$key}))
   {
      &write_warnings("Unknown key '$key'");
   }

   $itemname = $applang{$key};

   join('', '<B>', $itemname, '</B>', $_);
}

sub do_cmd_apppopupmenuitemstar{
   local($_) = @_;
   local($menuname,$itemname);

   $menuname = &missing_braces unless
             s/$next_pair_pr_rx/$menuname=$2;''/eo;

   $itemname = &missing_braces unless
             s/$next_pair_pr_rx/$itemname=$2;''/eo;

   local ($key);

   if ($menuname eq "text area")
   {
      $key = "text.$itemname";
   }
   else
   {
      $key = "editpath.$itemname";
   }

   unless (defined($applang{$key}))
   {
      &write_warnings("Unknown key '$key'");
   }

   $itemname = $applang{$key};

   local($this_file) = $CURRENT_FILE;

   local($tmpidxfile) = "${FILE}IDX.tmp";

   if (open FD, ">>$tmpidxfile")
   {
      print FD "<${CURRENT_FILE}#popupmenu_${menuname}_$itemname><3><popup menu><$menuname><$itemname>\n";
      close FD;
   }
   else
   {
      &write_warnings("can't append to '$hsidxfile'\n");
   }

   join('', "<A NAME=\"popupmenu_${menuname}_$itemname\">", '<B>',
       $itemname, '</B></A>', $_);
}

sub do_cmd_meta{
   local($_) = @_;

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   join('', '<EM>&lt;', $text, '&gt;</EM>', $_);
}

$iconsizes{'up'}='WIDTH="32" HEIGHT="32"';
$iconsizes{'previous'}='WIDTH="10" HEIGHT="7" ALIGN="middle"';
$iconsizes{'next'}='WIDTH="10" HEIGHT="7" ALIGN="middle"';

# redefine navigation panel

sub navigation_panel {
    "<!--Navigation Panel-->"

    . "<TABLE WIDTH=\"100%\">"

    #. "<TR><TD ALIGN=LEFT>$PREVIOUS</TD>"
    #. "<TD>$UP</TD>"
    #. "<TD ALIGN=RIGHT>$NEXT</TD>"
    #. "</TR>\n"

    . "<TR><TD ALIGN=LEFT>"

    . ($PREVIOUS_TITLE ? "$PREVIOUS &nbsp; $PREVIOUS_TITLE" : "&nbsp;")

    #. "</TD><TD>"

    #. ($UP_TITLE ? "$UP_TITLE" : "&nbsp;")

    . "</TD><TD ALIGN=RIGHT>"

    . ($NEXT_TITLE ? "$NEXT_TITLE &nbsp; $NEXT" : "&nbsp;")

    . "</TD></TR>\n"

    . "</TABLE>\n"
    . "<HR>";
}

sub bot_navigation_panel{
    "<HR>"
    . "<!--Navigation Panel-->"

    . "<TABLE WIDTH=\"100%\">"

    #. "<TR><TD ALIGN=LEFT>$PREVIOUS</TD>"
    #. "<TD>$UP</TD>"
    #. "<TD ALIGN=RIGHT>$NEXT</TD>"
    #. "</TR>\n"

    . "<TR><TD ALIGN=LEFT>"

    . ($PREVIOUS_TITLE ? "$PREVIOUS &nbsp; $PREVIOUS_TITLE" : "&nbsp;")

    #. "</TD><TD>"

    #. ($UP_TITLE ? "$UP_TITLE" : "&nbsp;")

    . "</TD><TD ALIGN=RIGHT>"

    . ($NEXT_TITLE ? "$NEXT_TITLE &nbsp; $NEXT" : "&nbsp;")

    . "</TD></TR>\n"

    . "</TABLE>\n";
}

sub do_cmd_TeX{
   join('', 'TeX', $_[0]);
}

sub do_cmd_LaTeX{
   join('', 'LaTeX', $_[0]);
}

sub do_cmd_LaTeXe{
   join('', 'LaTeX2e', $_[0]);
}

sub do_cmd_iconstartpar{
   local($_)=@_;
   local($name);

   # ignore optional argument
   local($afterref,$pat) = &get_next_optional_argument;

   $name = &missing_braces unless
             s/$next_pair_pr_rx/$name=$2;''/eo;

   "<IMG SRC=\"$name\" ALIGN=LEFT>" . $_;
}

sub do_cmd_texrelated{
   local($_)=@_;

   "<IMG WIDTH=30 SRC=\"texrelated.png\" ALIGN=LEFT TAG=\"TeX related information\">" . $_;
}

sub do_cmd_pgfrelated{
   local($_)=@_;

   "<IMG WIDTH=30 SRC=\"pgfrelated.png\" ALIGN=LEFT TAG=\"PGF related information\">" . $_;
}

sub do_cmd_flfrelated{
   local($_)=@_;

   "<IMG WIDTH=30 SRC=\"flfrelated.png\" ALIGN=LEFT TAG=\"flowframe related information\">" . $_;
}

sub do_cmd_texttrademark{
   local($_)=@_;
   join('','&#153;',$_);
}

sub do_cmd_seealsoname{ "See also".$_[0] }

sub do_env_seealsolinks{
   join('', "<P>\\seealsoname:\n", &do_env_itemize(@_));
}

$t_version='??';

sub do_cmd_version{
   local($_)=@_;

   $t_version = &missing_braces unless
             s/$next_pair_pr_rx/$t_version=$2;''/eo;

   $_;
}

sub do_cmd_theversion{
   local($_) = @_;

   $t_version.$_;
}

sub do_cmd_versionname{
   local($_) = @_;

   "Version$_";
}

# modify maketitle to include version
sub do_cmd_maketitle {
    local($_) = @_;
    local($the_title) = '';
    local($alignc, $alignl);
    if ($HTML_VERSION > 2.1) {
	$alignc = " ALIGN=\"CENTER\""; 
	$alignl = " ALIGN=\"LEFT\""; 
    }
    if ($t_title) {
	$the_title .= "<H1$alignc>$t_title</H1>";
        # add version info
        if ($t_version)
        {
           local($ver) = &translate_commands("\\versionname\\ $t_version");
           $the_title .= "\n<P$alignc><STRONG>$ver</STRONG></P>";
        }
    } else { &write_warnings("\nThis document has no title."); }
    if ($t_author) {
	if ($t_authorURL) {
	    local($href) = &translate_commands($t_authorURL);
	    $href = &make_named_href('author'
		         , $href, "<STRONG>${t_author}</STRONG>");
	    $the_title .= "\n<P$alignc>$href</P>";
	} else {
	    $the_title .= "\n<P$alignc><STRONG>$t_author</STRONG></P>";
	}
    } else { &write_warnings("\nThere is no author for this document."); }
    if ($t_institute&&!($t_institute=~/^\s*(($O|$OP)\d+($C|$CP))\s*\1\s*$/)) {
        $the_title .= "\n<P$alignc><SMALL>$t_institute</SMALL></P>";}
    if ($t_affil&&!($t_affil=~/^\s*(($O|$OP)\d+($C|$CP))\s*\1\s*$/)) {
	$the_title .= "\n<P$alignc><I>$t_affil</I></P>";}
    if ($t_date&&!($t_date=~/^\s*(($O|$OP)\d+($C|$CP))\s*\1\s*$/)) {
	$the_title .= "\n<P$alignc><STRONG>$t_date</STRONG></P>";}
    if ($t_address&&!($t_address=~/^\s*(($O|$OP)\d+($C|$CP))\s*\1\s*$/)) {
	$the_title .= "<BR>\n<P$alignl><SMALL>$t_address</SMALL></P>";
    }  # else { $the_title .= "\n<P$alignl>"}
    if ($t_email&&!($t_email=~/^\s*(($O|$OP)\d+($C|$CP))\s*\1\s*$/)) {
	$the_title .= "\n<P$alignl><SMALL>$t_email</SMALL></P>";
    }  # else { $the_title .= "</P>" }
    $the_title . $_ ;
}

sub do_cmd_refstar{
   local($_)=@_;

   local($label,$id);

   $label = &missing_braces unless
             s/$next_pair_pr_rx/($id,$label)=($1,$2);''/eo;

   local($ref_mark)= &get_ref_mark($label,$id);
   $ref_mark.$_;
}

sub do_env_altdescription{ &do_env_description(@_) }

sub read_app_dictionary{
   local($file) = &fulltexpath("dictionary.txt");

   if (open FD, $file)
   {
      while (<FD>)
      {
         s/#.*//;

         next if m/^[\s]*$/;

         if (m/([^=]+)=(.*)/)
         {
            $applang{$1}=$2;
         }
      }
      close FD;
   }
   else
   {
      &write_warnings("unable to open 'dictionary.txt' $!");
   }
}

sub do_cmd_appstring{
   local($_) = @_;
   local($key);

   $key = &missing_braces unless
             s/$next_pair_pr_rx/$key=$2;''/eo;

   unless (defined($applang{$key}))
   {
      &write_warnings("Unknown key '$key'");
   }

   $applang{$key}.$_;
}

sub do_cmd_mnemonic{
   local($_) = @_;
   local($key);

   $key = &missing_braces unless
             s/$next_pair_pr_rx/$key=$2;''/eo;

   $key = "$key.mnemonic";

   local $mnemonic = "";

   if (defined($applang{$key}))
   {
      $mnemonic = $applang{$key};
   }
   else
   {
      &write_warnings("Unknown key '$key'");
   }

   "<TT>Alt-$mnemonic</TT>$_";
}

sub do_cmd_menuItemMnemonic{
   local($_) = @_;
   local($key1, $key2);

   $key1 = &missing_braces unless
             s/$next_pair_pr_rx/$key1=$2;''/eo;

   $key2 = &missing_braces unless
             s/$next_pair_pr_rx/$key2=$2;''/eo;

   $key2 = "$key1.$key2.mnemonic";

   $key1 = "$key1.mnemonic";

   local $mnemonic1 = "";

   if (defined($applang{$key1}))
   {
      $mnemonic1 = $applang{$key1};
   }
   else
   {
      &write_warnings("Unknown key '$key1'");
   }

   local $mnemonic2 = "";

   if (defined($applang{$key2}))
   {
      $mnemonic2 = $applang{$key2};
   }
   else
   {
      &write_warnings("Unknown key '$key2'");
   }

   "<TT>Alt-$mnemonic1 $mnemonic2</TT>$_";
}

sub do_cmd_subMenuItemMnemonic{
   local($_) = @_;
   local($key1, $key2, $key3);

   $key1 = &missing_braces unless
             s/$next_pair_pr_rx/$key1=$2;''/eo;

   $mKey1 = "$key1.mnemonic";

   $key2 = &missing_braces unless
             s/$next_pair_pr_rx/$key2=$2;''/eo;

   $mKey2 = "$key1.$key2.mnemonic";

   $key3 = &missing_braces unless
             s/$next_pair_pr_rx/$key3=$2;''/eo;

   $mKey3 = "$key1.$key2.$key3.mnemonic";

   local $mnemonic1 = "";

   if (defined($applang{$mKey1}))
   {
      $mnemonic1 = $applang{$mKey1};
   }
   else
   {
      &write_warnings("Unknown key '$mKey1'");
   }

   local $mnemonic2 = "";

   if (defined($applang{$mKey2}))
   {
      $mnemonic2 = $applang{$mKey2};
   }
   else
   {
      &write_warnings("Unknown key '$mKey2'");
   }

   local $mnemonic3 = "";

   if (defined($applang{$mKey3}))
   {
      $mnemonic3 = $applang{$mKey3};
   }
   else
   {
      &write_warnings("Unknown key '$mKey3'");
   }

   "<TT>Alt-$mnemonic1 $mnemonic2 $mnemonic3</TT>$_";
}

sub do_cmd_subSubMenuItemMnemonic{
   local($_) = @_;
   local($key1, $key2, $key3, $key4);

   $key1 = &missing_braces unless
             s/$next_pair_pr_rx/$key1=$2;''/eo;

   $mKey1 = "$key1.mnemonic";

   $key2 = &missing_braces unless
             s/$next_pair_pr_rx/$key2=$2;''/eo;

   $mKey2 = "$key1.$key2.mnemonic";

   $key3 = &missing_braces unless
             s/$next_pair_pr_rx/$key3=$2;''/eo;

   $mKey3 = "$key1.$key2.$key3.mnemonic";

   $key4 = &missing_braces unless
             s/$next_pair_pr_rx/$key4=$2;''/eo;

   $mKey4 = "$key1.$key2.$key3.$key4.mnemonic";

   local $mnemonic1 = "";

   if (defined($applang{$mKey1}))
   {
      $mnemonic1 = $applang{$mKey1};
   }
   else
   {
      &write_warnings("Unknown key '$mKey1'");
   }

   local $mnemonic2 = "";

   if (defined($applang{$mKey2}))
   {
      $mnemonic2 = $applang{$mKey2};
   }
   else
   {
      &write_warnings("Unknown key '$mKey2'");
   }

   local $mnemonic3 = "";

   if (defined($applang{$mKey3}))
   {
      $mnemonic3 = $applang{$mKey3};
   }
   else
   {
      &write_warnings("Unknown key '$mKey3'");
   }

   local $mnemonic4 = "";

   if (defined($applang{$mKey4}))
   {
      $mnemonic4 = $applang{$mKey4};
   }
   else
   {
      &write_warnings("Unknown key '$mKey4'");
   }

   "<TT>Alt-$mnemonic1 $mnemonic2 $mnemonic3 $mnemonic4</TT>$_";
}

sub do_cmd_subSubSubMenuItemMnemonic{
   local($_) = @_;
   local($key1, $key2, $key3, $key4, $key5);

   $key1 = &missing_braces unless
             s/$next_pair_pr_rx/$key1=$2;''/eo;

   $mKey1 = "$key1.mnemonic";

   $key2 = &missing_braces unless
             s/$next_pair_pr_rx/$key2=$2;''/eo;

   $mKey2 = "$key1.$key2.mnemonic";

   $key3 = &missing_braces unless
             s/$next_pair_pr_rx/$key3=$2;''/eo;

   $mKey3 = "$key1.$key2.$key3.mnemonic";

   $key4 = &missing_braces unless
             s/$next_pair_pr_rx/$key4=$2;''/eo;

   $mKey4 = "$key1.$key2.$key3.$key4.mnemonic";

   $key5 = &missing_braces unless
             s/$next_pair_pr_rx/$key5=$2;''/eo;

   $mKey5 = "$key1.$key2.$key3.$key4.$key5.mnemonic";

   local $mnemonic1 = "";

   if (defined($applang{$mKey1}))
   {
      $mnemonic1 = $applang{$mKey1};
   }
   else
   {
      &write_warnings("Unknown key '$mKey1'");
   }

   local $mnemonic2 = "";

   if (defined($applang{$mKey2}))
   {
      $mnemonic2 = $applang{$mKey2};
   }
   else
   {
      &write_warnings("Unknown key '$mKey2'");
   }

   local $mnemonic3 = "";

   if (defined($applang{$mKey3}))
   {
      $mnemonic3 = $applang{$mKey3};
   }
   else
   {
      &write_warnings("Unknown key '$mKey3'");
   }

   local $mnemonic4 = "";

   if (defined($applang{$mKey4}))
   {
      $mnemonic4 = $applang{$mKey4};
   }
   else
   {
      &write_warnings("Unknown key '$mKey4'");
   }

   local $mnemonic5 = "";

   if (defined($applang{$mKey5}))
   {
      $mnemonic5 = $applang{$mKey5};
   }
   else
   {
      &write_warnings("Unknown key '$mKey5'");
   }

   "<TT>Alt-$mnemonic1 $mnemonic2 $mnemonic3 $mnemonic4 $mnemonic5</TT>$_";
}

sub do_cmd_dialoglabel{
   local($_) = @_;
   local($label);

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   "<B>$label</B>" . $_;
}

sub do_cmd_appdialoglabel{
   local($_) = @_;
   local($label);

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$label=$2;''/eo;

   if (defined($applang{$label}))
   {
      $label = $applang{$label};
   }
   else
   {
      &write_warnings("Unknown key '$label'");
   }

   "<B>$label</B>" . $_;
}

sub do_cmd_keyfmt{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   "<TT>$text</TT>" . $_;
}

sub do_cmd_dq{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   "&quot;$text&quot;" . $_;
}

sub do_cmd_figureref{
   local($_) = @_;
   local($label,$id);

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$id=$1;$label=$2;''/eo;

   local($id1) = ++$global{'max_id'};
   local($id2) = ++$global{'max_id'};

   "\\htmlref$OP$id1$CP\\figurename~$OP$id1$CP$OP$id2$CP$label$OP$id2$CP\\ref$OP$id$CP$label$OP$id$CP" . $_;
}

sub do_cmd_tableref{
   local($_) = @_;
   local($label,$id);

   $label = &missing_braces unless
             s/$next_pair_pr_rx/$id=$1;$label=$2;''/eo;

   local($id1) = ++$global{'max_id'};
   local($id2) = ++$global{'max_id'};

   "\\htmlref$OP$id1$CP"
   . "\\tablename~$OP$id1$CP"
   . "$OP$id2$CP$label$OP$id2$CP"
   ."\\ref$OP$id$CP$label$OP$id$CP" . $_;
}

sub do_cmd_fndagmark{
   local($_) = @_;

   "<SUP>&#134;</SUP>" . $_;
}

sub do_cmd_degrees{
   local($_) = @_;
   local($text);

   $text = &missing_braces unless
             s/$next_pair_pr_rx/$text=$2;''/eo;

   "$text&deg;" . $_;
}

$nodename='';

sub custom_title_hook{
   local($title) = @_;

   local($name) = $nodename;

   unless ($name)
   {
      $name = lc($title);
      $name=~s/[^\w]//g;
   }

   $nodename = '';

   $name
}

%sectiontitles = ();

sub do_cmd_sectionlabel{
    local($_) = @_;
    local($label, $title);

    $label = &missing_braces unless (
        (s/$next_pair_pr_rx/$label = $2;''/eo)
        ||(s/$next_pair_rx/$label = $2;''/eo));

    $title = &missing_braces unless (
        (s/$next_pair_pr_rx/$title = $2;''/eo)
        ||(s/$next_pair_rx/$title = $2;''/eo));

    $sectiontitles{$label}=$title;

    $_
}

sub do_cmd_sectionref{
   local($_) = @_;

   local($label, $id);

   $label = &missing_braces unless 
       s/$next_pair_pr_rx/$id=$1;$label = $2;''/eo;

   local($id2) = ++$global{'max_id'};

   local($title) = $sectiontitles{$label};

   "\\htmlref$OP$id$CP$title$OP$id$CP$OP$id2$CP$label$OP$id2$CP".$_;
}

sub do_cmd_appitemref{
   local($_) = @_;

   local($label, $id);

   $label = &missing_braces unless 
       s/$next_pair_pr_rx/$id=$1;$label = $2;''/eo;

   local($id2) = ++$global{'max_id'};

   local($title) = $sectiontitles{$label};

   "\\htmlref$OP$id$CP$title$OP$id$CP$OP$id2$CP$label$OP$id2$CP".$_;
}

sub do_cmd_dash{ "---".$_[0] }

&ignore_commands( <<_IGNORED_CMDS_ );
makeglossaries
image # {}
hypertarget # {}
texorpdfstring # {}
ifpdf
fi
pdfinfo # {}
_IGNORED_CMDS_

1;
