
/**	@file	dkwt.h	Header file for the dkwt program.
*/

#ifndef DKWT_H_INCLUDED
#define DKT3_H_INCLUDED 1

#include <lm.h>
#include <Sddl.h>

/**	Description for one Windows account.
*/
typedef struct {
  dkChar const	*logname;	/**< Login name. */
  dkChar const	*fullname;	/**< Full name. */
  dkChar const	*comment;	/**< Account comment. */
  dkChar const	*usrcomment;	/**< User comment. */
  dkChar const	*homedir;	/**< Home directory. */
  dkChar const	*profile;	/**< User profile. */
  dkChar const	*textsid;	/**< SID as text. */
  DWORD		 priv;		/**< usri4_priv. */
  DWORD		 flags;		/**< usri4_flags. */
  DWORD		 a_flags;	/**< usri4_authflags. */
  DWORD		 coco;		/**< Country code. */
  DWORD		 codepage;	/**< Code page. */
  int		 f_long;	/**< Flag: Long information was found. */
} dkwt_account_t;



/**	Windows group.
*/
typedef struct {
  dkChar const	*groupname;	/**< Group name. */
  dkChar const	*comment;	/**< Comment. */
  dkChar const	*textsid;	/**< SID as text. */
  DWORD		 attributes;	/**< Attributes. */
  int		 gtype;		/**< Group type (net or local). */
} dkwt_group_t;



/**	List of Windows accounts.
*/
typedef struct {
  dk3_sto_t	*s_acc;	/**< Accounts storage. */
  dk3_sto_it_t	*i_acc;	/**< Accounts iterator. */
} dkwt_account_list_t;



/**	List of Windows groups.
*/
typedef struct {
  dk3_sto_t	*s_grp;	/**< Groups storage. */
  dk3_sto_it_t	*i_grp;	/**< Groups iterator. */
} dkwt_group_list_t;



/**	Printer information.
*/
typedef struct {
  dkChar const	*printerName;		/**< Printer name. */
  dkChar const	*comment;		/**< Comment for the printer. */
  dkChar const	*description;		/**< Printer description text. */
  dkChar const	*location;		/**< Printer location. */
  dkChar const	*serverName;		/**< Print server managing queue. */
  dkChar const	*shareName;		/**< Share name of the printer. */
  dkChar const	*portName;		/**< Port name. */
  dkChar const	*driverName;		/**< Driver name. */
  dkChar const	*deviceName;		/**< Device name. */
  dkChar const	*separatorFile;		/**< File printed to separate jobs. */
  dkChar const	*printProcessor;	/**< Print processor name. */
  dkChar const	*dataType;		/**< Print data type. */
  dkChar const	*parameters;		/**< Printer parameters. */
  DWORD		 attributes;		/**< Printer attributes. */
  DWORD		 status;		/**< Printer status. */
  DWORD		 nJobs;			/**< Number of jobs in queue. */
} dkwt_printer_t;



/**	List of printers.
*/
typedef struct {
  dk3_sto_t	*s_prn;		/**< Storage containing the printers. */
  dk3_sto_it_t	*i_prn;		/**< Iterator through @a s_prn storage. */
} dkwt_printer_list_t;



#ifdef __cplusplus
extern "C" {
#endif

/**	Function for dkwt boot|shutdown|login|logout.
	@param	app	Application structure.
	@param	msg	Localized message texts.
	@param	kwnl	Keywords, not localized.
	@param	evt	Event type: 0=boot, 1=shutdown, 2=login, 3=logout
	@return	0 on success, DKT_RESULT_ERR_xxx on error.
*/
int
dkwt_event(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl,
  int			 evt
);

/**	Show or modify environment.
	@param	app	Application structure.
	@param	msg	Localized message texts.
	@param	kwnl	Keywords, not localized.
	@return	0 on success, DKT_RESULT_ERR_xxx on error.
*/
int
dkwt_env(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl
);

/**	Function to keep data while boot/login/logout/shutdown.
	@param	app	Application structure.
	@param	msg	Localized message texts.
	@param	kwnl	Keywords, not localized.
	@return	0 on success, DKT_RESULT_ERR_xxx on error.
*/
int
dkwt_event_keep(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl
);

/**	Function to reset keep flag for boot/login/logout/shutdown.
	@param	app	Application structure.
	@param	msg	Localized message texts.
	@param	kwnl	Keywords, not localized.
	@return	0 on success, DKT_RESULT_ERR_xxx on error.
*/
int
dkwt_event_unkeep(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl
);

/**	Function for dkwt accounts.
	@param	app	Application structure.
	@param	msg	Localized message texts.
	@param	kwnl	Keywords, not localized.
	@return	0 on success, DKT_RESULT_ERR_xxx on error.
*/
int
dkwt_accounts(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl
);

/**	Create account, allocate memory.
	@param	logname	Login name of user.
	@param	app	Application structure for diagnostics.
	@return	Pointer to new account on success, NULL on error.
*/
dkwt_account_t *
dkwt_tool_open_account(dkChar const *logname, dk3_app_t *app);

/**	Destroy account, release memory.
	@param	ap	Account to destroy.
*/
void
dkwt_tool_close_account(dkwt_account_t *ap);

/**	Create account list, allocate memory.
	@param	app	Application structure for diagnostics.
	@param	f_long	Flag: Long listing required.
	@return	Pointer to new list on success, NULL on error.
*/
dkwt_account_list_t *
dkwt_tool_open_account_list(dk3_app_t *app, int f_long);

/**	Destroy account list, release memory.
	@param	lp	List to destroy.
*/
void
dkwt_tool_close_account_list(dkwt_account_list_t *lp);

/**	Reset account list before traversing it.
	@param	lp	List to reset.
*/
void
dkwt_tool_reset_account_list(dkwt_account_list_t *lp);

/**	Get next account from list.
	@param	lp	List to traverse.
	@return	Pointer to next account or NULL (list end).
*/
dkwt_account_t *
dkwt_tool_get_account_from_list(dkwt_account_list_t *lp);

/**	The dkwt groups command.
	@param	app	Application structure.
	@param	msg	Localized messages texts,
	@param	kwnl	Keywords, not localized.
*/
int
dkwt_groups(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl
);

/**	The dkwt printers command.
	@param	app	Application structure.
	@param	msg	Localized messages texts,
	@param	kwnl	Keywords, not localized.
*/
int
dkwt_printers(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl
);

/**	The dkwt print command.
	@param	app	Application structure.
	@param	msg	Localized message texts.
	@param	kwnl	Keywords, not localized.
*/
int
dkwt_print(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl
);

/**	Create a new group list.
	@param	app	Application structure.
	@param	f_long	Flag: Long (full) listing.
	@return	Pointer to new group list on success, NULL on error.
*/
dkwt_group_list_t *
dkwt_tool_open_group_list(dk3_app_t *app, int f_long);

/**	Reset a group list.
	@param	glp	Group list to reset.
*/
void
dkwt_tool_reset_group_list(dkwt_group_list_t *glp);

/**	Retrieve next group from list.
	@param	glp	Group list to traverse.
	@return	Pointer to next group on success, NULL on error.
*/
dkwt_group_t *
dkwt_tool_get_group_from_list(dkwt_group_list_t *glp);

/**	Close group list, release memory.
	@param	glp	Group list to close.
*/
void
dkwt_tool_close_group_list(dkwt_group_list_t *glp);

/**	Open printer list.
	@param	app	Application structure.
	@param	f_long	Flag: Long information about printers.
	@return	Pointer to new printer list on success, NULL on errors.
*/
dkwt_printer_list_t *
dkwt_tool_open_printer_list(dk3_app_t *app, int f_long);

/**	Close printer list, release memory.
	@param	pl	List to close.
*/
void
dkwt_tool_close_printer_list(dkwt_printer_list_t *pl);

/**	Reset printer list.
	@param	pl	Printer list to reset.
*/
void
dkwt_tool_printer_list_reset(dkwt_printer_list_t *pl);

/**	Get next printer description from list.
	@param	pl	Printer list.
	@return	Pointer to next printer on success, NULL on error.
*/
dkwt_printer_t *
dkwt_tool_printer_list_next(dkwt_printer_list_t *pl);

/**	Open a registry key.
	@param	key
	@param	skn
	@param	sam
	@param	phk
	@return	ERROR_SUCCESS on success, any other value indicates
	an error.
*/
long
dkwt_tool_reg_open_key(
  HKEY		 key,
  dkChar const	*skn,
  REGSAM	 sam,
  PHKEY		 phk
);

/**	Query a registry value.
	@param	hk	Registry key.
	@param	vn	Value name.
	@param	pDwTp	Pointer to type variable (in/out).
	@param	buf	Pointer to result buffer.
	@param	pDwSz	Result buffer length available/used.
	@return	ERROR_SUCCESS on success, any other value indicates
	an error.
*/
long
dkwt_tool_reg_query_value(
  HKEY			 hk,
  dkChar const		*vn,
  DWORD			*pDwTp,
  void			*buf,
  DWORD			*pDwSz
);

/**	Delete a registry entry (value).
	@param	hk	Registry key.
	@param	en	Name of entry to delete.
	@return	ERROR_SUCCESS on success, any other value indicates an error.
*/
long
dkwt_tool_reg_delete_value(
  HKEY		 hk,
  dkChar const	*en
);

/**	Delete subkey hierarchy.
	@param	hk	Parent registry key.
	@param	tn	Name of subkey to delete.
	@return	ERROR_SUCCESS on success, any other value indicates an error.
*/
long
dkwt_tool_reg_delete_tree(
  HKEY		 hk,
  dkChar const	*tn
);

/**	Enumerate subkeys of a key.
	@param	hk		Registry key.
	@param	dwIndex		Index of subkey to enumerate.
	@param	pResult		Pointer to result buffer.
	@param	pDwSzResult	Pointer to buffer length / result length.
	@param	pClass		Subkey class name, may be NULL.
	@param	pDwSzClass	Subkey class name length, may be NULL.
	@param	pLastWrite	Pointer to timestamp, may be NULL.
	@return ERROR_SUCCESS on success, ERROR_NO_MORE_DATA on end,
	any other value indicates an error.
*/
long
dkwt_tool_reg_enum_key(
  HKEY			 hk,
  DWORD			 dwIndex,
  dkChar		*pResult,
  DWORD			*pDwSzResult,
  dkChar		*pClass,
  DWORD			*pDwSzClass,
  PFILETIME		 pLastWrite
);

/**	Create a registry key.
	@param	hk		Parent key.
	@param	kn		Subkey name.
	@param	dwOptions	Options (0 is a good choice).
	@param	sam		Access mask.
	@param	pHkeyResult	Pointer to result HKEY.
	@param	lpDwDisposition	Pointer to disposition DWORD, may be NULL.
	@return	ERROR_SUCCESS on success, any other value indicates an error.
*/
long
dkwt_tool_reg_create_key(
  HKEY		 hk,
  dkChar const	*kn,
  DWORD		 dwOptions,
  REGSAM	 sam,
  PHKEY		 pHkeyResult,
  LPDWORD	 lpDwDisposition
);

/**	Set a registry value.
	@param	hk	Handle for registry key.
	@param	vn	Value name.
	@param	dwType	Value type.
	@param	pData	Value data.
	@param	szData	Length of value data.
	@return	ERROR_SUCCESS on success, any other value indicates an error.
*/
long
dkwt_ev_reg_set_value(
  HKEY		 hk,
  dkChar const	*vn,
  DWORD		 dwType,
  void		*pData,
  DWORD		 szData
);

/**	Clear local print queues.
	@param	app		Application structure for diagnostics.
	@param	msg		Localized message texts.
	@param	kwnl		Keywords, not localized.
	@param	userName	User for which to delete print jobs.
*/
void
dkwt_clear_printers_local(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl,
  dkChar const		*userName
);

/**	Clear LPRng/LPD remote print queues.
	@param	app		Application structure for diagnostics.
	@param	msg		Localized message texts.
	@param	kwnl		Keywords, not localized.
	@param	userName	User name for print jobs.
	@param	iterator	Iterator for print queue names storage.
*/
void
dkwt_clear_printers_remote(
  dk3_app_t		*app,
  dkChar const * const	*msg,
  dkChar const * const	*kwnl,
  dkChar const		*userName,
  dk3_sto_it_t		*iterator
);



#ifdef __cplusplus
}
#endif

/**	Show groups found on network server.
*/
#define	DKWT_GROUP_NET		0

/**	Show groups found on local computer.
*/
#define	DKWT_GROUP_LOCAL	1

#endif

