
/**	@file	dkct-tr.h	Header file for the dkct-tr module.
*/

#ifndef DKCT_TR_H_INCLUDED
#define	DKCT_TR_H_INCLUDED 1


#if DK3_USE_WX
#include "DkWxCommunicator.h"
#endif


/**	State machine entry (input, output state).
*/
typedef struct {
  char const	*name;		/**< Name. */
  char const	*comment;	/**< Comment text. */
  unsigned long	 lineno;	/**< Line number of definition. */
  int		 value;		/**< Numeric value. */
  char		 flag;		/**< Flag: Numeric value set. */
} DKCT_STM_ENTRY;



/**	Transition rule.
*/
typedef struct {
  DKCT_STM_ENTRY	*state;		/**< Current (old) state. */
  DKCT_STM_ENTRY	*input;		/**< Input. */
  DKCT_STM_ENTRY	*newstate;	/**< New state. */
  DKCT_STM_ENTRY	*output;	/**< Output. */
  unsigned long		 lineno;	/**< Line number of definition. */
} DKCT_STM_RULE;



/**	State machine description.
*/
typedef struct {
  dk3_sto_t	*s_states;	/**< Storage for states. */
  dk3_sto_it_t	*i_states;	/**< Iterator for states. */
  dk3_sto_t	*s_inputs;	/**< Storage for inputs. */
  dk3_sto_it_t	*i_inputs;	/**< Iterator for inputs. */
  dk3_sto_t	*s_outputs;	/**< Storage for outputs. */
  dk3_sto_it_t	*i_outputs;	/**< Iterator for outputs. */
  dk3_sto_t	*s_exact;	/**< Storage for exact rules. */
  dk3_sto_it_t	*i_exact;	/**< Iterator for exact rules. */
  dk3_sto_t	*s_wildcard;	/**< Storage for one-wildcard rules. */
  dk3_sto_it_t	*i_wildcard;	/**< Iterator for one-wildcard rules. */
  char const	*name;		/**< State machine name. */
  DKCT_STM_ENTRY	*rs;	/**< Reset state. */
  DKCT_STM_ENTRY	*ds;	/**< Default state. */
  DKCT_STM_ENTRY	*dout;	/**< Default output. */
  unsigned long	 stmno;		/**< State machine number. */
  unsigned long	 lineno;	/**< Line number of start. */
  unsigned long	 lngr;		/**< Line number of general rule. */
  int		 wrh;		/**< Flag: Write definitions to header. */
  int		 section;	/**< Current section. */
} DKCT_STM;



/**	One input source line.
*/
typedef struct {
  char const	*t;	/**< Text. */
  unsigned long	 l;	/**< Line number. */
} DKCT_LINE;



/**	One source file read in.
*/
typedef struct {
  DKCT_OPTION_SET	dkcto;	/**< Conversion options. */
  dk3_app_t		*app;	/**< Application structure. */
  dkChar const * const	*msg;	/**< Localized message texts. */
  dkChar const		*ffn;	/**< Full source file name. */
  dk3_sto_t		*hss;	/**< Header start storage. */
  dk3_sto_it_t		*hsi;	/**< Header start iterator. */
  dk3_sto_t		*hes;	/**< Header end storage. */
  dk3_sto_it_t		*hei;	/**< Header end iterator. */
  dk3_sto_t		*clss;	/**< Class start storage. */
  dk3_sto_it_t		*clsi;	/**< Class start iterator. */
  dk3_sto_t		*cles;	/**< Class end storage. */
  dk3_sto_it_t		*clei;	/**< Class end iterator. */
  dk3_sto_t		*mss;	/**< Module start storage. */
  dk3_sto_it_t		*msi;	/**< Module start iterator. */
  dk3_sto_t		*mes;	/**< Module end iterator. */
  dk3_sto_it_t		*mei;	/**< Module end iterator. */
  dk3_sto_t		*coss;	/**< Constructor start storage. */
  dk3_sto_it_t		*cosi;	/**< Constructor start iterator. */
  dk3_sto_t		*coes;	/**< Constructor end iterator. */
  dk3_sto_it_t		*coei;	/**< Constructor end iterator. */
  dk3_sto_t		*s_stm;	/**< Storage for state machines. */
  dk3_sto_it_t		*i_stm;	/**< Iterator for state machines. */
  dk3_sto_it_t		*cursr;	/**< Current source. */
  DKCT_STM		*cstm;	/**< Current state machine. */
  void			*gui;	/**< GUI description. */
  FILE			*fipo;	/**< Output file. */
  FILE			*cbfip;	/**< Comment box file. */
  char const		*sfn;	/**< Short file name. */
  char const		*crn;	/**< Copyright name. */
  char const		*cry;	/**< Copyright year. */
  char			*cmdb;	/**< Command buffer. */
  char			*cbfn;	/**< Comment box file name. */
  unsigned long		lineno;	/**< Current line number. */
  size_t		 szcmd;	/**< Size of command buffer. */
  size_t		 ucmd;	/**< Used characters in command buffer. */
  size_t		 cbcp;	/**< Comment box current position. */
  size_t		 cbmx;	/**< Comment box max line length. */
  int			 lict;	/**< License type. */
  int			 type;	/**< Source type. */
  int			 ec;	/**< Error code. */
  int			 st;	/**< Current state. */
  int			 sufi;	/**< Source suffix index. */
  int			 shln;	/**< Flag: Show line number. */
  int			 curdi;	/**< Flag: Processing the current directory. */
  char			 cchr;	/**< Current character to process. */
#if DK3_USE_WX
  DkWxCommunicator	*pComm;	/**< Communicator for threads. */
#endif
} DKCT_SRC;


/**	@defgroup	dkctsections	Source file sections.
*/
/**@{*/

/**	Start of module.
*/
#define	DKCT_SECTION_MODULE_START	0

/**	End of module
*/
#define DKCT_SECTION_MODULE_END		1

/**	Start of constructor.
*/
#define	DKCT_SECTION_CONSTRUCTOR_START	2

/**	End of constructor.
*/
#define DKCT_SECTION_CONSTRUCTOR_END	3

/**	Start of header.
*/
#define	DKCT_SECTION_HEADER_START	4

/**	End of header.
*/
#define	DKCT_SECTION_HEADER_END		5

/**	Start of class.
*/
#define	DKCT_SECTION_CLASS_START	6

/**	End of class.
*/
#define	DKCT_SECTION_CLASS_END		7

/**	Options.
*/
#define	DKCT_SECTION_OPTIONS		8

/**	wxWidgets GUI.
*/
#define	DKCT_SECTION_GUI		9

/**	State machine description.
*/
#define	DKCT_SECTION_STATE_MACHINE	10

/**@}*/



#ifdef __cplusplus
extern "C" {
#endif

/**	Destroy line structure.
	@param	p	Structure to destroy.
*/
void
dkct_tr_line_delete(DKCT_LINE *p);

/**	Create new source line.
	@param	t	Text.
	@param	l	Line number.
	@param	app	Application structure for diagnostics, may be NULL.
	@return	Pointer to new line structure on success, NULL on error.
*/
DKCT_LINE *
dkct_tr_line_new(char const *t, unsigned long l, dk3_app_t *app);

/**	Destroy dkct-tr source structure.
	@param	p	Structure to destroy.
*/
void
dkct_tr_delete(DKCT_SRC *p);

/**	Create new dkct-tr source structure.
	@param	o	Option set.
	@param	fn	Short file name.
	@param	ai	File suffix array index.
	@param	app	Application structure for diagnostics, may be NULL.
	@param	ffn	Full source file name.
	@return	Pointer to new structure on success, NULL on error.
*/
DKCT_SRC *
dkct_tr_new(
  DKCT_OPTION_SET	*o,
  char const		*fn,
  int			 ai,
  dk3_app_t		*app,
  dkChar const		*ffn
);

/**	Read source file.
	@param	psrc	Source file structure to store data.
	@param	fn	Source file name.
	@param	sufi	File name suffix index.
	@return	1 on success, 0 on error.
*/
int
dkct_tr_read(DKCT_SRC *psrc, dkChar const *fn, int sufi);

/**	Check setup (especially GUI).
	@param	psrc	Source file structure to store data.
	@param	fn	Source file name.
	@param	sufi	File name suffix index.
	@return	1 on success, 0 on error.
*/
int
dkct_tr_check(DKCT_SRC *psrc, dkChar const *fn, int sufi);

/**	Write output.
	@param	psrc	Source file structure to store data.
	@param	fn	Source file name.
	@param	sufi	File name suffix index.
	@return	1 on success, 0 on error.
*/
int
dkct_tr_write(DKCT_SRC *psrc, dkChar const *fn, int sufi);

/**	Create new state machine description.
	@param	lineno	Source line number.
	@param	app	Application structure for diagnostics, may be NULL.
	@return	Pointer to new structure on success, NULL on error.
*/
DKCT_STM *
dkct_au_stm_new(unsigned long lineno, dk3_app_t *app);

/**	Destroy state machine description.
	@param	p	Structure to destroy.
*/
void
dkct_au_stm_delete(DKCT_STM *p);

/**	Add one text line to state machine description.
	@param	stm	State machine description.
	@param	psrc	Source structure.
	@param	line	Text to add.
	@param	lineno	Line number.
	@return	1 on success, 0 on error.
*/
int
dkct_au_add_line(
  DKCT_STM *stm, DKCT_SRC *psrc, char *line, unsigned long lineno
);

/**	Check state machine consistency.
	@param	psrc	Source structure.
	@param	stm	State machine to check.
	@return	1 on success, 0 on error.
*/
int
dkct_au_stm_check(DKCT_SRC *psrc, DKCT_STM *stm);

/**	Write "#line ..." information to output.
	@param	psrc	Source structure.
	@param	newlines	Number of newlines to use in counting.
*/
void
dkct_tr_show_source_line(DKCT_SRC *psrc, size_t newlines);

/**	Write state, input and output constants.
	@param	psrc	Source structure.
	@param	stm	State machine.
*/
void
dkct_au_write_entries(DKCT_SRC *psrc, DKCT_STM *stm);

/**	Write function prototypes for state machine.
	@param	psrc	Source structure.
	@param	stm	State machine.
*/
void
dkct_au_write_prototypes(DKCT_SRC *psrc, DKCT_STM *stm);

/**	Write state machine functions.
	@param	psrc	Source structure.
	@param	stm	State machine.
	@param	f_static	Flag: Use static keyword.
*/
void
dkct_au_write_functions(DKCT_SRC *psrc, DKCT_STM *stm, int f_static);

/**	Get one of the dkct-tr keywords.
	@param	sz	Keyword index.
	@return	Pointer to keyword or NULL.
*/
char const *
dkct_tr_get_kw8(size_t sz);

/**	Write string containing file name and line number to output.
	@param	psrc	Source structure.
*/
void
dkct_tr_show_filename_and_lineno(DKCT_SRC *psrc);

/**	Log one message.
	@param	psrc	Source structure.
	@param	scope	Scope (0=GUI only, 1=app+GUI).
	@param	ll	Log level.
	@param	i1	Index of message text in \a psrc->msg.
*/
void
dkct_to_log_1(DKCT_SRC *psrc, int scope, int ll, size_t i1);

/**	Log one message.
	@param	psrc	Source structure.
	@param	scope	Scope (0=GUI only, 1=app+GUI).
	@param	ll	Log level.
	@param	i1	Index of message text in \a psrc->msg.
	@param	i2	Index of second fixed text in \a psrc->msg.
	@param	text	Variable text part.
*/
void
dkct_to_log_3(
  DKCT_SRC *psrc, int scope, int ll, size_t i1, size_t i2, char const *text
);

/**	Log one message.
	@param	psrc	Source structure.
	@param	scope	Scope (0=GUI only, 1=app+GUI).
	@param	ll	Log level.
	@param	i1	Index of message text in \a psrc->msg.
	@param	i2	Index of second fixed text in \a psrc->msg.
	@param	text	Variable text part.
*/
void
dkct_to_log_dk_3(
  DKCT_SRC *psrc, int scope, int ll, size_t i1, size_t i2, dkChar const *text
);


#ifdef __cplusplus
}
#endif

#endif

