
/**	@file	dk3types.h	Type definitions for the dk3 library.
 */

#ifndef DK3_TYPES_H_INCLUDED
/** Avoid multiple file inclusions. */
#define DK3_TYPES_H_INCLUDED 1

#include <dk3conf.h>

#if DK3_HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#if DK3_HAVE_STDINT_H
#include <stdint.h>
#endif
#if DK3_HAVE_INTTYPES_H
#include <inttypes.h>
#endif
#if DK3_HAVE_SIGNAL_H
#include <signal.h>
#endif
#if DK3_HAVE_WCHAR_H
#include <wchar.h>
#endif
#if DK3_HAVE_TERMIOS_H
#include <termios.h>
#endif
#if DK3_HAVE_SYS_TERMIOS_H
#include <sys/termios.h>
#endif
#if DK3_HAVE_SYS_TTOLD_H
#include <sys/ttold.h>
#endif
#if DK3_HAVE_ZLIB_H
#include <zlib.h>
#endif

#ifndef DK3_CHAR_SIZE
#if DK3_ON_WINDOWS && (defined(_UNICODE))
/**	Specify default character size 16 bit.
*/
#define	DK3_CHAR_SIZE	2
#else
/**	Specify default character size 8 bit.
*/
#define	DK3_CHAR_SIZE	1
#endif
#endif


/*
	Simple data types.
*/

/**	A 8-bit unsigned int  type.
*/
typedef unsigned char	dk3_u8_t;

/**	An 8-bit signed int type.
*/
typedef signed char dk3_i8_t;

/**	A 8-bit character type.
*/
typedef char /*^unsignedintegraltype^*/	dk3_c8_t;

/**	A 16-bit unsigned int type.
*/
typedef	unsigned short	dk3_u16_t;

/**	A 16-bit signed int type.
*/
typedef short dk3_i16_t;

/**	A 16 bit character type (used on Windows).
*/
#if DK3_SIZEOF_WCHAR_T == 2
typedef wchar_t		/*^unsignedintegraltype^*/	dk3_c16_t;
#else
typedef unsigned short	/*^unsignedintegraltype^*/	dk3_c16_t;
#endif

/**	A 32-bit character/unsigned int type (wchar_t on UNIX, not used).
*/
typedef unsigned long dk3_u32_t;

/**	A 32-bit signed int type.
*/
typedef long dk3_i32_t;

/**	A 32-bit character type (used i.e. by X11).
*/
#if DK3_SIZEOF_WCHAR_T == 4
typedef	wchar_t	      /*^unsignedintegraltype^*/	dk3_c32_t;
#else
typedef unsigned long /*^unsignedintegraltype^*/	dk3_c32_t;
#endif



#if DK3_CHAR_SIZE > 1
#if defined(DK3_SIZEOF_WCHAR_T)
#if DK3_CHAR_SIZE == DK3_SIZEOF_WCHAR_T
/**	Text character type.
*/
typedef	wchar_t		/*^unsignedintegraltype^*/	dkChar;
/**	Macro to switch text literals between 8-bit and 16-bit text.
*/
#define	dkT(s)		L ## s
/**	Macro to switch the main function between 8 and 16 bit.
*/
#define	DK3_MAIN	int wmain(int argc, wchar_t *argv[])
#else
/* if DK3_CHAR_SIZE == DK3_SIZEOF_WCHAR_T */
#if DK3_CHAR_SIZE > 2
/**	Text character type.
*/
typedef	dk3_c32_t	/*^unsignedintegraltype^*/	dkChar;
#error	"Can not use prefix L to specify dkChar string literals!"
#error	"Can not define DK3_MAIN to dkChar!"
#else
/* if DK3_CHAR_SIZE > 2 */
/**	Text character type.
*/
typedef	dk3_c16_t	/*^unsignedintegraltype^*/	dkChar;
#error	"Can not use prefix L to specify dkChar string literals!"
#error	"Can not define DK3_MAIN to dkChar!"
#endif
/* if DK3_CHAR_SIZE > 2 */
#endif
/* if DK3_CHAR_SIZE == DK3_SIZEOF_WCHAR_T */
#else
/* if defined(DK3_SIZEOF_WCHAR_T) */
#error	"Missing definition for DK3_SIZEOF_WCHAR_T!"
#endif
/* if defined(DK3_SIZEOF_WCHAR_T) */
#if !DK3_ON_WINDOWS
#error "Usable support for UNICODE/MBCS is available on Windows platforms only!"
#endif
/* if !DK3_ON_WINDOWS */
#else
/* if DK3_CHAR_SIZE > 1 */
/* +++ DK3_CHAR_SIZE == 1 */
/**	Text character type.
*/
typedef	char		/*^unsignedintegraltype^*/	dkChar;
/**	Macro to switch the main function between 8 and 16 bit.
*/
#define	dkT(s)		s
/**	Macro to switch the main function between 8 and 16 bit.
*/
#define	DK3_MAIN	int main(int argc, char *argv[])
/* --- DK3_CHAR_SIZE == 1 */
#endif
/* if DK3_CHAR_SIZE > 1 */



#if DK3_HAVE_INTMAX_T

/**	Maximum size integer type.
*/
typedef intmax_t			dk3_im_t;

/**	Maximum size unsigned integer type.
*/
typedef uintmax_t			dk3_um_t;

#else

#if DK3_HAVE_LONG_LONG

/**	Maximum size integer type.
*/
typedef long long int			dk3_im_t;

/**	Maximum size unsigned integer type.
*/
typedef long long unsigned		dk3_um_t;

#else

/**	Maximum size integer type.
*/
typedef long int			dk3_im_t;

/**	Maximum size unsigned integer type.
*/
typedef long unsigned			dk3_um_t;

#endif
#endif


/**	Timer.
*/

#if DK3_ON_WINDOWS
/**	Portable type to save a timestamp.
*/
typedef	__time64_t	dk3_time_t;
#else
#if (defined(DK3_HAVE_SYS_TIME_H) || defined(DK3_HAVE_TIME_H)) && defined(DK3_HAVE_TIME)
/**	Portable type to save a timestamp.
*/
typedef	time_t		dk3_time_t;
#else
#error "No mechanism known to obtain current time!"
#endif
#endif



#if DK3_HAVE_LONG_DOUBLE && (DK3_SIZEOF_LONG_DOUBLE > DK3_SIZEOF_DOUBLE)
/**	Most precise double type.
*/
typedef long double	dk3_double_t;
#else
/**	Most precise double type.
*/
typedef double		dk3_double_t;
#endif


#if DK3_ON_WINDOWS
/**	Number of bytes successfully read/written by read()/write().
*/
#define	dk3_rw_res_t	int
#else
#if DK3_HAVE_SSIZE_T
/**	Number of bytes successfully read/written by read()/write().
*/
#define dk3_rw_res_t	ssize_t
#else
/**	Number of bytes successfully read/written by read()/write().
*/
#define	dk3_rw_res_t	int
#endif
#endif

#if DK3_ON_WINDOWS
/**	Buffer size in read()/write().
*/
#define	dk3_rw_arg_t	unsigned int
#else
/**	Buffer size in read()/write().
*/
#define	dk3_rw_arg_t	size_t
#endif


/*
	Structured data types.
*/



/**	Timestamp.
*/
typedef struct _dk3_time_t_ {
  /**	Year.
  */
  int	Y;

  /**	Month.
  */
  int	M;

  /**	Day in month.
  */
  int	D;

  /**	Hours.
  */
  int	h;

  /**	Minutes.
  */
  int	m;

  /**	Seconds.
  */
  int	s;
} dk3_tm_t;



/**	File information.
	When retrieving information about a symbolic link the dk3sf_stat()
	function retrieves information about both the link destination and
	the link itself.
*/
typedef struct {

  /**	Creation timestamp.
  */
  dk3_time_t	cre;

  /**	Modification timestamp.
  */
  dk3_time_t	mod;

  /**	Last access timestamp.
  */
  dk3_time_t	acc;

  /**	Link creation timestamp.
  */
  dk3_time_t	lcre;

  /**	Link modification timestamp.
  */
  dk3_time_t	lmod;

  /**	Link last access timestamp.
  */
  dk3_time_t	lacc;

  /**	User ID.
  */
  dk3_um_t	u;

  /**	Group ID.
  */
  dk3_um_t	g;

  /**	Device number.
  */
  dk3_um_t	device;

  /**	Relative device number.
  */
  dk3_um_t	rdev;

  /**	Inode number.
  */
  dk3_um_t	inode;

  /**	Number of hard links to inode.
  */
  dk3_um_t	nlink;

  /**	File size in bytes.
  */
  dk3_um_t	sz;

  /**	Link user ID.
  */
  dk3_um_t	lu;

  /**	Link group ID.
  */
  dk3_um_t	lg;

  /**	Link device number.
  */
  dk3_um_t	ldevice;

  /**	Link relative device number.
  */
  dk3_um_t	lrdev;

  /**	Link inode number.
  */
  dk3_um_t	linode;

  /**	Number of hard links to the link.
  */
  dk3_um_t	lnlink;

#if DK3_ON_WINDOWS
  long		dwReparse;
#endif

  /**	File type, see @ref filetypes.
  */
  int		ft;

  /**	Permissions, see @ref filepermissions.
  */
  int		perm;

  /**	Link file type.
  */
  int		lft;

  /**	Link permissions.
  */
  int		lperm;

  /**	Link file size.
  */
  dk3_um_t	lsz;

  /**	Additional information about file, see @ref statadditional.
  */
  unsigned char	ai;

#if DK3_ON_WINDOWS
  /**	Reparse point information.
  */
  char		cReparse;
#endif
} dk3_stat_t;




/**	Evaluation result for an object.
*/
typedef union {

    /**	Evaluation to double.
    */
    double d;

    /**	Evaluation to float.
    */
    float f;

    /**	Evaluation to long.
    */
    long l;

    /**	Evaluation to unsigned long.
    */
    unsigned long ul;  

    /**	Evaluation to int.
    */
    int i;

    /**	Evaluation to unsigned int.
    */
    unsigned int ui;  

    /**	Evaluation to short.
    */
    short s;

    /**	Evaluation to unsigned short.
    */
    unsigned short us;

    /**	Evaluation to char.
    */
    char c;

    /**	Evaluation to unsigned char.
    */
    unsigned char uc;
} dk3_object_eval_res_t;


/**	Internal node for dk3_sto_t.
	This structure is used internally by the dksto module.
*/
struct _dk3_sto_node_ {

  /**	Object evaluation of @a o.
  */
  dk3_object_eval_res_t	   v;

  /**	Parent node in tree.
  */
  struct _dk3_sto_node_ *p;

  /**	Left child node (tree) or preceeding node (list).
  */
  struct _dk3_sto_node_ *l;

  /**	Right child node (tree) or following node (list).
  */
  struct _dk3_sto_node_ *r;

  /**	Object to store.
  */
  void                    *o;

  /**	Balance flag (used in trees only).
  */
  short                    b;

  /**	Direction to walk in tree.
  */
  short                    w;	/**< Direction to walk. */
};

/** Internal node for dk3_sto_t.
*/
typedef struct _dk3_sto_node_ dk3_sto_node_t;

/** Pointer to internal node for dk3_sto_t.
*/
typedef dk3_sto_node_t *dk3_sto_node_p;



/**	Evaluation functions.
	These function type are used to evaluate and compare objects
	stored in a sorted dk3_sto_t.
*/

/** Object evaluation function for sorted storage.
*/
typedef char dk3_fct_eval_c_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef unsigned char dk3_fct_eval_uc_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef short dk3_fct_eval_s_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef unsigned short dk3_fct_eval_us_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef int dk3_fct_eval_i_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef unsigned int dk3_fct_eval_ui_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef long dk3_fct_eval_l_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef unsigned long dk3_fct_eval_ul_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef float dk3_fct_eval_f_t(void const *obj, int crit) /*^*/ ;

/** Object evaluation function for sorted storage.
*/
typedef double dk3_fct_eval_d_t(void const *obj, int crit) /*^*/ ;

/** Object comparison function for sorted storage.
    When comparing to objects while inserting a new object into
    a storage both @a o1 and @a o2 are object pointers.
    When this function is called from dk3sto_it_find_like() the
    @a o1 argument is an object pointer from the storage,
    @a o2 is the pointer passed to dk3sto_it_find_like().
*/
typedef int dk3_fct_comp_t(
  /*^null^*/	void const *o1,
  /*^null^*/	void const *o2,
		int crit
) /*^*/ ;



/**	Choice for object evaluation function.
*/
typedef union {

  /** Evaluation function returning a char value.
  */
  dk3_fct_eval_c_t *c;

  /** Evaluation function returning an unsigned char value.
  */
  dk3_fct_eval_uc_t *uc;

  /** Evaluation function returning a short value.
  */
  dk3_fct_eval_s_t *s;

  /** Evaluation function returning an unsigned short value.
  */
  dk3_fct_eval_us_t *us;

  /** Evaluation function returning an int value.
  */
  dk3_fct_eval_i_t *i;

  /** Evaluation function returning an unsigned int value.
  */
  dk3_fct_eval_ui_t *ui;

  /** Evaluation function returning a long value.
  */
  dk3_fct_eval_l_t *l;

  /** Evaluation function returning an unsigned long value.
  */
  dk3_fct_eval_ul_t*ul;

  /** Evaluation function returning a float value.
  */
  dk3_fct_eval_f_t *f;

  /** Evaluation function returning a double value.
  */
  dk3_fct_eval_d_t *d;

  /** Function to compare two objects.
  */
  dk3_fct_comp_t *comp;
} dk3_object_eval_fct_t;



/**	Object storage.
	A storage can be used to store pointers to objects.
*/
typedef struct _dk3_sto_t_ {

  /**	Comparison or evaluation function.
  */
  dk3_object_eval_fct_t	e;

  /**	Application structure for diagnostics.
  */
  void		  	*app;

  /**	Critical path for delete operations.
  */
  dk3_sto_node_p 	*d;

  /**	Root element (tree) or left element (list).
  */
  dk3_sto_node_t 	*r;

  /**	Double-linked list of iterators.
  */
  void              	*i;

  /**	Algorithm selection for comparison or evaluation.
  */
  int 		     	h;

  /**	Criteria for comparison or evaluation.
  */
  int                	c;

  /**	Flag: Use tree, 1=tree, 0=list.
  */
  int		     	t;

  /**	Path length of critical path (number of elements).
  */
  short              	l;
} dk3_sto_t;



/**	Storage iterator.
	This structure can be used to iterate through a dk3_sto_t.
*/
struct _dk3_sto_iterator_ {

  /**	Preceeding iterator in iterator linked list. 
  */
  struct _dk3_sto_iterator_ *l;

  /**	Following iterator in iterator linked list.
  */
  struct _dk3_sto_iterator_ *r;

  /**	Storage owning this iterator.
  */
  dk3_sto_t                *s;

  /**	Current node.
  */
  dk3_sto_node_t           *c;
};
/**	Storage iterator. Can be used to iterate through a dk3_sto_t. */
typedef struct _dk3_sto_iterator_ dk3_sto_it_t;



/**	Key value pair.
*/
typedef struct {
  dkChar	*key;	/**< Key. */
  dkChar	*val;	/**< Value. */
} dk3_key_value_t;



/**	Directory contents.
*/
typedef struct _dk3_dir_t_ {

  /**	Stat buffer for current entry to process.
  */
  dk3_stat_t		stb;

  /**	Storage containing the directory names.
  */
  dk3_sto_t		*sdi;

  /**	Iterator for directory names storage.
  */
  dk3_sto_it_t		*idi;

  /**	Storage containing file names.
  */
  dk3_sto_t		*sfi;

  /**	Iterator for file names storage.
  */
  dk3_sto_it_t		*ifi;

  /**	Directory name.
  */
  dkChar const		*dirname;

  /**	Full path of current entry to process.
  */
  dkChar		*fullname;

  /**	Short name of current entry to process.
  */
  dkChar		*shortname;

  /**	Application used for diagnostics.
  */
  void			*app;

  /**	Number of directories found.
  */
  dk3_um_t		ndir;

  /**	Number of files found.
  */
  dk3_um_t		nfi;
} dk3_dir_t;



/**	Arguments in a stream command API call.
*/
typedef struct {

  /**	Low-level data.
  */
  void	*d;

  /**	Buffer for bytes to read/write.
  */
  char	*b;

  /**	Buffer size (number of bytes to read/write).
  */
  size_t	sz;
} dk3_stream_api_arg_t;



/**	Results of a stream command API call.
*/
typedef struct {

  /**	Number of bytes read.
  */
  size_t	sz;

  /**	Error code DK3_ERROR_xxx, see @ref errorcodes.
  */
  int		ec;
} dk3_stream_api_result_t;



/**	Stream command API.
*/
typedef struct _dk3_stream_api_t_ {

  /**	API call arguments.
  */
  dk3_stream_api_arg_t	arg;

  /**	API call results.
  */
  dk3_stream_api_result_t	res;

  /**	Command to execute, DK3_STREAM_API_xxx, see @ref streamapicommands.
  */
  int	cmd;

  /**	Success flag, 1=success, 0=error.
  */
  int	suc;
} dk3_stream_api_t;

/**	Stream API function type.
*/
typedef void dk3_stream_api_fct_t(dk3_stream_api_t *);

/**	Generic I/O stream.
*/
typedef struct _dk3_stream_t_ {

  /**	Optional application structure for diagnostics.
  */
  void			*app;

  /**	Low-level data.
  */
  void			*d;

  /**	Stream API function.
  */
  dk3_stream_api_fct_t	*f;

  /**	Input buffer.
  */
  char			*ib;

  /**	Output buffer.
  */
  char			*ob;

  /**	Number of bytes written.
  */
  dk3_um_t		wr;

  /**	Size of input buffer.
  */
  size_t		sz_ib;

  /**	Bytes available in input buffer.
  */
  size_t		av_ib;

  /**	Number of bytes already used from input buffer.
  */
  size_t		us_ib;

  /**	Size of output buffer.
  */
  size_t		sz_ob;

  /**	Number of bytes already placed in output buffer.
  */
  size_t		us_ob;

  /**	Output encoding, see @ref fileenc.
  */
  int			oe;

  /**	Usage flags, see @ref streamuseflags.
  */
  int			fl;

  /**	Flag: End of input found.
  */
  int			fe;

  /**	Flag: Reading 0 bytex indicates end of input.
  */
  int			ze;

  /**	Flag: Flush output buffer at end of line.
  */
  int			lf;

  /**	Flag: Error occured.
  */
  int			er;

  /**	Last error occured, see @ref errorcodes.
  */
  int			ec;
} dk3_stream_t;



/**	Application structure.
*/
typedef struct _dk3_app_t_ {

  /**	Messages array, either localized messages or default texts.
  */
  dkChar const * const	*msg;

  /**	Input file name currently processed (used in error messages).
  */
  dkChar const		*n_sourcefile;

  /**	Command line arguments.
  */
  dkChar const * 	*argv;

  /**	Application name.
  */
  dkChar const		*n_app;

  /**	Name of executable file for process.
  */
  dkChar const		*n_execfile;

  /**	Directory bin (directory for binaries - bindir).
  */
  dkChar const		*n_bindir;

  /**	Directory etc (system configuration directory - sysconfdir).
  */
  dkChar const		*n_etcdir;

  /**	Directory share (data root directory - datarootdir).
  */
  dkChar const		*n_sharedir;

  /**	Directory var (local state directory - vardir).
  */
  dkChar const		*n_vardir;

  /**	Login name of process owner.
  */
  dkChar const		*n_logname;

  /**	Home directory of process owner.
  */
  dkChar const		*n_homedir;

  /**	Aplication group name.
  */
  dkChar const		*n_appgroup;

  /**	File name for log file.
  */
  dkChar const		*n_logfilename;

  /**	Directory for temporary files.
  */
  dkChar const		*n_tmpdir;

  /**	Users language.
  */
  dkChar const		*n_language;

  /**	Users region.
  */
  dkChar const		*n_region;

  /**	Language/region setting string from environment or registry.
  */
  dkChar const		*n_langstr;

  /**	Host name.
  */
  dkChar const		*n_hostname;

  /**	Container for message catalogs.
  */
  dk3_sto_t		*s_mc;

  /**	Iterator through message catalogs container.
  */
  dk3_sto_it_t		*i_mc;

  /**	Container for preferences read from command line.
  */
  dk3_sto_t		*s_cmdprefs;

  /**	Iterator through command line preferences container.
  */
  dk3_sto_it_t		*i_cmdprefs;

  /**	Container for self-set preferences.
  */
  dk3_sto_t		*s_selfprefs;

  /**	Iterator through self-set preferences container.
  */
  dk3_sto_it_t		*i_selfprefs;

  /**	Container for system-wide preferences.
  */
  dk3_sto_t		*s_sysprefs;

  /**	Iterator through system-wide preferences container.
  */
  dk3_sto_it_t		*i_sysprefs;

  /**	Container for users constant preferences.
  */
  dk3_sto_t		*s_constprefs;

  /**	Iterator through users constant preferences container.
  */
  dk3_sto_it_t		*i_constprefs;

  /**	Container for users variable preferences.
  */
  dk3_sto_t		*s_varprefs;

  /**	Iterator through users variable preferences container.
  */
  dk3_sto_it_t		*i_varprefs;

  /**	Timestamp: Last logging to stderr.
  */
  dk3_time_t		 t_last_log_stderr;

  /**	Timestamp: Last logging to file.
  */
  dk3_time_t		 t_last_log_file;

  /**	Line number in current input file (used in error messages).
  */
  unsigned long		sourceline;

  /**	Name part of next temporary file name.
  */
  unsigned long		ul_tmpname;

  /**	Suffix part of next temporary file name.
  */
  unsigned		u_tmpsuffix;

  /**	Application type, see @ref apptypes.
  */
  int	app_type;

  /**	Error coce, report delayed after finishing application initialization.
  */
  int	de;

  /**	Log level required for reporting to stderr, see @ref loglevels.
  */
  int	ll_stderr;

  /**	Log level required for reporting to file, see @ref loglevels.
  */
  int	ll_file;

  /**	Log level required to keep log file, see @ref loglevels.
  */
  int	ll_file_keep;

  /**	Log level required to keep temporary files, see @ref loglevels.
  */
  int	ll_tmp_keep;

  /**	Most relevant log level found, see @ref loglevels.
  */
  int	ll_highest;

  /**	Carry flag, too many temporary file names requested.
  */
  int	i_tmpcarry;

  /**	Number of command line arguments.
  */
  int	argc;

  /**	Internal encoding, used for strings in memory, see @ref stringenc.
  */
  int	i_encoding;

  /**	Encoding for output files, see @ref fileenc.
  */
  int	i_output_encoding;

  /**	Encoding expected on stdin, see @ref fileenc.
  */
  int	i_stdin_input_encoding;

  /**	Encoding expected when reading from files, see @ref fileenc.
  */
  int	i_file_input_encoding;

  /**	Flag: Unconfigure application (delete users variable preferences)
	when finishing the application.
  */
  int	f_unconfigure;

  /**	Flag: Changes applied to preferences, need to save when finishing
	the application.
  */
  int	f_prefschanged;

  /**	Flag: Setup completed, we can write log messages.
  */
  int	f_readytolog;

  /**	Flag: First attempt to log to file.
  */
  int	f_first_file_log;

  /**	Flag: Already attempted to initialize PRNG.
  */
  int	f_rand_initialized;

  /**	Flag: PRNG was initialized successfully.
  */
  int	f_rand_success;

  /**	PRNG type to use, see @ref rndtypes.
  */
  int	rand_type;
} dk3_app_t;



/**	File search result.
*/
typedef struct _dk3_search_t_ {

  /**	Container for all file names found.
  */
  dk3_sto_t	*s_fn;

  /**	Iterator for all file names container.
  */
  dk3_sto_it_t	*i_fn;

  /**	Application structure, used for diagnostics.
  */
  dk3_app_t	*app;

  /**	Number of next file to add.
  */
  unsigned long	nf;

  /**	Flag: Inverted search order (most relevant files first).
  */
  int		inverted;
} dk3_search_t;



/**	Bit field.
*/
typedef struct _dk3_bf_t_ {

  /**	Data bytes to store the bits.
  */
  unsigned char	*data;

  /**	Optional application structure for diagnostics.
  */
  dk3_app_t	*app;

  /**	Number of bits to save.
  */
  size_t	nbits;
} dk3_bf_t;



/**	Bit matrix.
*/
typedef struct _dk3_bm_t_ {

  /**	Data bytes to store the bits.
  */
  unsigned char	**data;

  /**	Optional application structure for diagnostics.
  */
  dk3_app_t	*app;

  /**	Size of available cache for dk3bm_expand.
  */
  size_t	cachesize;

  /**	Number of bits per row.
  */
  size_t	columns;

  /**	Number of rows.
  */
  size_t	rows;
} dk3_bm_t;



/**	LaTeX font encoding information.
*/
typedef unsigned char dk3_font_encoding_t;


/**	Information about one package needed for an encoding.
*/
typedef struct {
  char const	*name;	/**< Package name. */
  char		 used;	/**< Flag: Package used. */
} dk3_uc2lat_pkg_t;

/**	Pointer to a package information.
*/
typedef dk3_uc2lat_pkg_t *dk3_uc2lat_pkg_ptr;



/**	Directory entry for UC to LaTeX.
*/
typedef struct {

  /**	Container to store ranges.
  */
  dk3_sto_t		*s_ran;

  /**	Iterator through ranges container.
  */
  dk3_sto_it_t		*i_ran;

  /**	Short file name containing data for this directory entry.
  */
  dkChar const		*sn;

  /**	Flag: Data was loaded.
  */
  unsigned char		loaded;
} dk3_uc2lat_dir_t;



/**	Range of UC characters.
*/
typedef struct {

  /**	LaTeX encodings for both text and math mode.
  */
  char const		**a;

  /**	LaTeX encodings for text mode only.
  */
  char const		**t;

  /**	LaTeX encodings for math mode only.
  */
  char const		**m;

  /**	Descriptions texts for character.
  */
  dkChar const		**dsc;

  /**	Package collections required for character.
  */
  dk3_uc2lat_pkg_t	**p;

  /**	Information about font encodings required by characters.
  */
  dk3_font_encoding_t	*f;

  /**	Directory entry responsible for this range.
  */
  dk3_uc2lat_dir_t	*dir;

  /**	First character in range.
  */
  dk3_c32_t		start;

  /**	Last character in range.
  */
  dk3_c32_t		end;

  /**	Flag: Initialization of range attempted, no second attempt.
  */
  unsigned char		ia;
} dk3_uc2lat_range_t;



/**	Data for U32 to LaTeX conversion.
*/
typedef struct {

  /**	Application structure for diagnostics and file search.
  */
  dk3_app_t	*app;

  /**	Name of data directory containing the LaTeX encodings.
  */
  dkChar const	*dir;

  /**	Allocated buffer for characters directly passed through.
  */
  char		*buf;

  /**	Container for ranges.
  */
  dk3_sto_t	*s_ran;

  /**	Iterator through ranges container.
  */
  dk3_sto_it_t	*i_ran;

  /**	Container for directories.
  */
  dk3_sto_t	*s_dir;

  /**	Iterator for directories container.
  */
  dk3_sto_it_t	*i_dir;

  /**	Containter for packages.
  */
  dk3_sto_t	*s_pkg;

  /**	Iterator through packages container.
  */
  dk3_sto_it_t	*i_pkg;

  /**	Cache: Last range used.
  */
  dk3_uc2lat_range_t	*rca;

  /**	Size of buffer @a buf.
  */
  size_t	 szbuf;

  /**	Flag: Load descriptions for characters too (normally the
	descriptions are ignored).
  */
  int		 f_dsc;

  /**	Flag: Generate UTF-8 output.
  */
  int		 f_utf8;

  /**	Font encoding information.
  */
  dk3_font_encoding_t	fe;
} dk3_uc2lat_t;



/**	Handler function for one line.
	@param	obj	Object to modify using the line.
	@param	il	Input line to process.
	@return	1=OK, 0=error, can continue, -1=error, abort processing.
*/
typedef
int
dk3_line_handler_t(
  void		*obj,
  dkChar	*il
);



/**	Struct for processing a file line by line.
*/
typedef struct {
  /**	Application structure for diagnostics.
  */
  dk3_app_t		*app;
  /**	Data object to modify while processing input lines.
  */
  void			*obj;
  /**	Handler function.
  */
  dk3_line_handler_t	*hf;
  /**	Buffer for input lines.
  */
  dkChar		*buf;
  /**	Current insertion position for next character to add.
  */
  dkChar		*cp;
  /**	Size of input buffer @a buf.
  */
  size_t		 szbuf;
  /**	Number of characters used in @a buf.
  */
  size_t		 bufus;
  /**	System encoding, see @ref stringenc.
  */
  int			 se;
} dk3_line_handler_data_t;



/**	One option.
*/
typedef struct {
  /**	Short option character.
  */
  dkChar 		 so;
  /**	Long option text.
  */
  dkChar const		*lo;
  /**	Flag: Option needs an argument.
  */
  int			 na;
} dk3_option_t;



/**	Options set.
*/
typedef struct {
  /**	Application structure for diagnostics.
  */
  dk3_app_t		 *app;
  /**	Options to find in command line arguments.
  */
  dk3_option_t const	 *options;
  /**	Array of flags, one for each options, whether or not the option was
	found.
  */
  int		 	 *found;
  /**	Array for option arguments.
  */
  dkChar const		**optargs;
  /**	Array of normal arguments (no options and no option arguments).
  */
  dkChar const		**args;
  /**	Further options.
  */
  dkChar const		**fo;
  /**	Short option to introduce further options.
  */
  dkChar		  foc;
  /**	Long option to introduce further options.
  */
  dkChar const		 *focl;
  /**	Number of elements in @a options.
  */
  size_t		  szoptions;
  /**	Number of normal arguments available.
  */
  int			  argsav;
  /**	Number of normal arguments used (found).
  */
  int			  argsused;
  /**	Number of further options available.
  */
  int			  foav;
  /**	Number of further options found.
  */
  int			  fou;
  /**	Error code of last error occured, see @ref errorcodes.
  */
  int			  ec;
} dk3_option_set_t;



/**	Output filter cell details for ASCII/Hex encoding.
*/
typedef struct {
  /**	Number of characters already printed in current line.
  */
  size_t			nl;
} dk3_of_cell_ah_details_t;

/**	Output filter cell details for ASCII85 encoding.
*/
typedef struct {
  /**	32-bit value built up by for bytes.
  */
  unsigned long			val;
  /**	Number of characters merged into @a val.
  */
  size_t			nc;
  /**	Number of characters already printed in current line.
  */
  size_t			nl;
} dk3_of_cell_a85_details_t;



/**	Output filter cell details for run-length compression.
*/
typedef struct {
  /**	Data buffer.
  */
  unsigned char		*bu;
  /**	Number of bytes used in buffer @a bu.
  */
  size_t		 used;
  /**	Number of same bytes repeated.
  */
  size_t		 same;
  /**	Previous character processed.
  */
  unsigned char		 pc;
  /**	Flag: Have a previous character.
  */
  unsigned char		 hc;
  /**	Flag: In a run.
  */
  unsigned char		 run;
} dk3_of_cell_rl_details_t;



#if DK3_HAVE_ZLIB_H
/**	Output filter cell details for flate compression.
*/
typedef struct {
  /**	zlib stream.
  */
  z_stream		*zs;
  /**	Input buffer.
  */
  Bytef			*ibu;
  /**	Output buffer.
  */
  Bytef			*obu;
  /**	Number of input bytes used.
  */
  uLong			 iused;
  /**	Flag: Everything ok, no errors yet.
  */
  int			 ok;
} dk3_of_cell_flate_details_t;
#endif


/**	Output filter cell details.
*/
typedef union {
#if DK3_HAVE_ZLIB_H
    /**	Flate compression data.
    */
    dk3_of_cell_flate_details_t	flate;
#endif
    /**	Run-length compression data.
    */
    dk3_of_cell_rl_details_t	psrl;
    /**	ASCII85 encoding data.
    */
    dk3_of_cell_a85_details_t	a85;
    /**	ASCII-Hex encoding data.
    */
    dk3_of_cell_ah_details_t	ah;
} dk3_of_cell_details_t;



/**	Output filter cell.
*/
typedef struct _dk3_of_cell_t_ {
  /**	Compression/encoding algorithm-specific data.
  */
  dk3_of_cell_details_t		data;
  /**	Next cell downwards.
  */
  struct _dk3_of_cell_t_	*down;
  /**	Next cell upwards.
  */
  struct _dk3_of_cell_t_	*up;
  /**	Output filter owning this cell.
  */
  void				*of;	/**< Output filter. */
  /**	Output filter cell type DK3_OF_CELL_TYPE_xxx,
	see @ref outputfiltercelltypes.
  */
  int				 type;
} dk3_of_cell_t;



/**	Output filter on top of a stream.
*/
typedef struct {
  /**	Application structure for diagnostics.
  */
  dk3_app_t		*app;
  /**	Output stream to receive the filtered data.
  */
  dk3_stream_t		*os;
  /**	Bottom filter cell (next to the stream).
  */
  dk3_of_cell_t		*bottom;
  /**	Top filter cell (input goes to here).
  */
  dk3_of_cell_t		*top;
  /**	Number of cells.
  */
  size_t		 nCells;
  /**	Number of bits used in the @a bits variable.
  */
  size_t		 nBits;
  /**	The bits stored so far.
  */
  unsigned char		 bits;
  /**	Flag set for output filter.
  */
  unsigned char		 flags;
} dk3_of_t;



#if 0

/**	One datum stored in the in-memory database.
*/
typedef struct {
  /**	Start address of data.
  */
  void		*data;
  /**	Data size (number of bytes).
  */
  size_t	 size;
} dk3_memdb_datum_t;



/**	In-memory database.
*/
typedef struct {
  /**	Container for the key/value pairs.
  */
  dk3_sto_t		*st;
  /**	Iterator through key/value pair container.
  */
  dk3_sto_it_t		*it;
  /**	Application structure for diagnostics.
  */
  dk3_app_t		*app;
  /**	File name of database on disk.
  */
  dkChar const		*fn;
  /**	Start of key.
  */
  char			*kb;
  /**	Start of value.
  */
  char			*vb;
  /**	Input line buffer.
  */
  char			*il;
  /**	Binary key size.
  */
  size_t		 szkb;
  /**	Text key size.
  */
  size_t		 szkt;
  /**	Binary value size.
  */
  size_t		 szvb;
  /**	Text value size.
  */
  size_t		 szvt;
  /**	Text line size.
  */
  size_t		 szl;
  /**	Error code of last error occured, see @ref errorcodes.
  */
  int			 ec;
  /**	Flag: Dirty (synchronization needed).
  */
  int			 f_dt;
} dk3_memdb_t;

#endif



/**	Data to store in a database as key or value.
*/
typedef struct {
  /**	Data bytes.
  */
  void		*dt;
  /**	Number of bytes.
  */
  size_t	 sz;
} dk3_datum_t;



/**	Details for a memory database.
*/
typedef struct {
  /**	Container for key/value pairs.
  */
  dk3_sto_t		*s_mem;
  /**	Container iterator.
  */
  dk3_sto_it_t		*i_mem;
} dk3_dbi_mem_details_t;



/**	Details for Berkeley DB database.
*/
typedef struct {
  /**	The BDB database.
  */
  void			*dbptr;
} dk3_dbi_bdb_details_t;



/**	Details for NDBM database.
*/
typedef struct {
  /**	8-bit character file name, only used if dkChar is 16 bit or above.
  */
  void			*dbptr;
} dk3_dbi_ndbm_details_t;



/**	Database type specific details.
*/
typedef union {
  /**	Memory database.
  */
  dk3_dbi_mem_details_t		mem;
  /**	Berkeley database details.
  */
  dk3_dbi_bdb_details_t		bdb;
  /**	NDBM database details.
  */
  dk3_dbi_ndbm_details_t	ndbm;
} dk3_dbi_details_t;



/**	Database interface.
*/
typedef struct {
  /**	Database type specific details.
   */
  dk3_dbi_details_t	details;
  /**	Application structure for diagnostics.
  */
  dk3_app_t		*app;
  /**	File name for database on disk.
  */
  dkChar const		*fn;
  /**	Database type, DK3_DB_TYPE_xxx, see @ref databasetypes.
  */
  int			tp;
  /**	Flag: Database opened for writing.
  */
  int			wr;
  /**	Flag: Database modified, sync to file necessary.
  */
  int			mod;
  /**	Flag: Write error messages for keys not found.
  */
  int			rfk;
  /**	Error code for last error occured.
  */
  int			ec;
} dk3_dbi_t;



/**	BIF coordinates type.
	Must include negative values as predictor operators may
	try to obtain values in row and column -1.
*/
typedef long	dk3_bif_coord_t;



/**	BIF pixel value type.
*/
typedef unsigned short dk3_bif_pixel_t;

/**	Size of BIF pixel value.
*/
#define	DK3_SIZEOF_BIF_PIXEL_T	DK3_SIZEOF_SHORT

/**	Pixel resample information.
*/
typedef struct {
  size_t		iw;	/**< Input width. */
  size_t		ow;	/**< Output width. */
  dk3_bif_pixel_t	im;	/**< Input maximum. */
  dk3_bif_pixel_t	om;	/**< Output maximum. */
  dk3_bif_pixel_t	im2;	/**< Half of input minimum. */
} dk3_pixel_resample_t;



/**	One frame in bitmap image file.
*/
typedef struct {
  /**	X resolution.
  */
  double		 xres;
  /**	Y resolution.
  */
  double		 yres;
  /**	Bitmap type specific data per frame.
  */
  void			*implspec;
  /**	Frame width.
  */
  dk3_bif_coord_t	 w;
  /**	Frame height.
  */
  dk3_bif_coord_t	 h;
  /**	Bits per component for frame found in file.
  */
  size_t		 bits;
  /**	Number of non-redundant bits in @ bits.
  */
  size_t		 realbits;
  /**	Frame number.
  */
  size_t		 n;		/**< Frame number. */
  /**	Color space, DK3_COLOR_SPACE_xxx, see @ref colorspaces.
  */
  int			 cs;
  /**	Flag: Really colored, 1=colored, 0=gray.
  */
  int			 realcolor;
  /**	Flag: Really found alpha data.
  */
  int			 realalpha;
  /**	Flag: Background color defined for frame.
  */
  int			 bg;
  /**	Background red found in file.
  */
  dk3_bif_pixel_t	 bgr;
  /**	Background green found in file.
  */
  dk3_bif_pixel_t	 bgg;
  /**	Background blue found in file.
  */
  dk3_bif_pixel_t	 bgb;
  /**	Background gray found in file.
  */
  dk3_bif_pixel_t	 bggr;
} dk3_bif_frame_t;



/**	Bitmap image file.
*/
typedef struct {
  /**	Structure to upsample or downsample pixel values.
  */
  dk3_pixel_resample_t	 pr;
  /**	Application structure for diagnostics.
  */
  dk3_app_t		*app;
  /**	Container for frames.
  */
  dk3_sto_t		*s_frames;
  /**	Iterator through frames container.
  */
  dk3_sto_it_t		*i_frames;
  /**	Current frame to process.
  */
  dk3_bif_frame_t	*cf;
  /**	Bitmap image type specific data per file.
  */
  void			*implspec;
  /**	Name of temporary file (if any), may be NULL.
  */
  char const		*tmpfn;
  /**	Temporary file (if any), may be NULL.
  */
  FILE			*tmpfipo;
  /**	Number of bitmap image file within process.
  */
  unsigned long		 n;
  /**	Number of frames available.
  */
  size_t		 n_frames;
  /**	Bits per component requested for output.
  */
  size_t		 bits;
  /**	BIF image type DK3_BIF_IMAGE_TYPE_xxx, see @ref bifimagetypes.
  */
  int			 bt;
  /**	Flag: Force use of requested background over bg found in file.
  */
  int			 bg;
  /**	Flag: Remove temporary file when closing the structure.
  */
  int			 remtmpf;
  /**	Background red requested.
  */
  dk3_bif_pixel_t	 bgr;
  /**	Background green requested.
  */
  dk3_bif_pixel_t	 bgg;
  /**	Background blue requested.
  */
  dk3_bif_pixel_t	 bgb;
} dk3_bif_t;



/**	Paper size description. All sizes are specified in PS points.
*/
typedef struct {
  /**	Width.
  */
  double	w;
  /**	Height.
  */
  double	h;
  /**	Left border (inner border for duplex without tumble).
  */
  double	i;
  /**	Right border (outer border for duplex without tumble).
  */
  double	o;
  /**	Top border (inner border for duplex with tumble).
  */
  double	t;
  /**	Bottom bofder (outer border for duplex with tumble).
  */
  double	b;
} dk3_paper_size_t;



/**	Named paper size.
*/
typedef struct {
  /**	Size.
  */
  dk3_paper_size_t		 size;
  /**	Size name.
  */
  dkChar const		*name;
} dk3_named_paper_size_t;



/**	Collection of multiple named paper sizes.
*/
typedef struct {
  /**	Application structure for diagnostics and file search.
  */
  dk3_app_t		*app;
  /**	Container of named paper sizes.
  */
  dk3_sto_t		*s_sizes;
  /**	Iterator through named paper sizes container.
  */
  dk3_sto_it_t		*i_sizes;
} dk3_paper_size_collection_t;






/**	Area to place an image.
*/
typedef struct {
  /**	Left x.
  */
  double	xl;
  /**	Right x.
  */
  double	xr;
  /**	Top y.
  */
  double	yt;
  /**	Bottom y.
  */
  double	yb;
} dk3_image_area_t;



/**	Area to place a PS/PDF image.
*/
typedef union {
  /**	Paper size for creating a document.
  */
  dk3_paper_size_t	ps;
  /**	Image area to use when converting Fig to PS.
  */
  dk3_image_area_t	ia;
} dk3_bm_eps_area_t;



/**	Object start position.
*/
typedef struct {
  /**	Object number.
  */
  unsigned long		objno;

  /**	Start position in file.
  */
  dk3_um_t		startpos;
} dk3_pdf_position_t;



/**	XObject representing one image or alpha mask.
*/
typedef struct {
  /**	Name of temporary file with contents.
  */
  dkChar const		*tempfilename;

  /**	Xobject for alpha mask.
  */
  void			*mask;

  /**	Stream length.
  */
  dk3_um_t		slgt;

  /**	Image width (number of pixels).
  */
  dk3_bif_coord_t	width;

  /**	Image height (number of pixels).
  */
  dk3_bif_coord_t	height;

  /**	Object number for XObject.
  */
  unsigned long		objno;

  /**	XObject type DK3_PDF_XO_TYPE_xxx, see @ref pdfxobjecttype.
  */
  int			xot;

  /**	Procedure for object, DK3_BMEPS_xxx, see @ref bmepscompression.
  */
  int			procedure;

  /**	Bits per component.
  */
  int			bpc;

  /**	Flag: Stream is DCT (1) or flate (0).
  */
  int			dct;

  /**	Predictor DK3_COMPRESSIONPREDICTOR_xxx, see @ref flatepredictors.
  */
  int			pred;

  /**	Flag: Adobe marker was found in CMYK DCT file.
  */
  int			adobe_marker_found;

  /**	Flag: Used by current page.
  */
  int			used;

  /**	Flag: Interpolate this object.
  */
  int			interpolate;
} dk3_pdf_xobject_t;



/**	One PDF page.
*/
typedef struct {
  /**	Name of temporary file for page description stream.
  */
  dkChar const		*tempfilename;

  /**	Temporary file.
  */
  FILE			*tempfile;

  /**	Stream to write to temporary file.
  */
  dk3_stream_t		*tempstream;

  /**	Output filter, used to write to temporary file.
  */
  dk3_of_t		*tempof;

  /**	Collection of all XObjects used by this page.
  */
  dk3_sto_t		*s_xobjects;

  /**	Iterator through XObjects collection.
  */
  dk3_sto_it_t		*i_xobjects;

  /**	Stream length.
  */
  dk3_um_t		slgt;

  /**	PDF object number.
  */
  unsigned long		objno;

  /**	Page number.
  */
  unsigned long		pageno;

  /**	Media box x0.
  */
  long			mbx0;

  /**	Media box x1.
  */
  long			mbx1;

  /**	Media box y0.
  */
  long			mby0;

  /**	Media box y1.
  */
  long			mby1;
} dk3_pdf_page_t;



/**	Structure used to write a PDF file.
*/
typedef struct {
  /**	Application structure, used to find temporary file names.
  */
  dk3_app_t		*app;

  /**	XObject storage (bitmaps).
  */
  dk3_sto_t		*s_xobjects;

  /**	Iterator through XObject storage.
  */
  dk3_sto_it_t		*i_xobjects;

  /**	Pages storage.
  */
  dk3_sto_t		*s_pages;

  /**	Iterator through pages storage.
  */
  dk3_sto_it_t		*i_pages;

  /**	Current page.
  */
  dk3_pdf_page_t	*cp;

  /**	Next XObject object number.
  	First Xobject is object number 4 as we have info object,
	root object and pages collection object as 1, 2, and 3.
  */
  unsigned long		nextobject;

  /**	Next page number.
  */
  unsigned long		nextpage;

  /**	Media box x0 for next page.
  */
  long			mbx0;

  /**	Media box x1 for next page.
  */
  long			mbx1;

  /**	Media box y0 for next page.
  */
  long			mby0;

  /**	Media box y1 for next page.
  */
  long			mby1;

  /**	Flag: Document mode (1) or image/object (0).
  */
  int			documentMode;

} dk3_pdf_t;


/**	Options to convert a bitmap image to PS/PDF.
*/
typedef struct {
  /**	Image area.
  */
  dk3_bm_eps_area_t	ima;

  /**	Resolution in dpi.
  */
  double		resolution;

  /**	Application structure for diagnostics, may be NULL.
  */
  dk3_app_t		*app;

  /**	Output driver DK3_BMEPS_DRIVER_xxx, see @ref bmepsdrivers.
  */
  int			dr;

  /**	Run mode DK3_BMEPS_MODE_xxx, see @ref dkbmmodes.
  */
  int			mode;

  /**	Size conversion mode.
  */
  int			szmode;

  /**	Compression predictor for flate compression,
	DK3_COMPRESSION_PREDICTOR_xxx, see @ref flatepredictors.
  */
  int			pred;

  /**	Image origin position DK3_IMAGE_ORIGIN, see @ref imageorigin.
  */
  int			io;

  /**	Flag: Draft mode.
  */
  int			draft;

  /**	Flag: Printout in duplex.
  */
  int			duplex;

  /**	Flag: Printout in tumble mode (like a calendar).
  */
  int			tumble;

  /**	PS level (2 or 3).
  */
  int			psl;

  /**	Flag: Write DSC comments.
  */
  int			dsc;

  /**	Flag: Allow direct DCT data transfer.
  */
  int			dct;

  /**	Flag: Keep aspect ratio.
  */
  int			kar;

  /**	Requested/suggested background red, 8 bits per pixel.
  */
  int			bgr;

  /**	Requested/suggested background green, 8 bits per pixel.
  */
  int			bgg;

  /**	Requested/suggested background blue, 8 bits per pixel.
  */
  int			bgb;

  /**	Flag: Enforce this background over bg color found in file.
  */
  int			fbg;

  /**	Flag: Image interpolation enabled.
  */
  int			ip;

  /**	Flag: Image interpolation enabled for JPEG files too.
  */
  int			jip;

  /**	Flag: Make mode.
  */
  int			make;

  /**	Write colored output.
  */
  int			color;

  /**	Flag: Reduce bits per component.
  */
  int			rbpc;

  /**	Flag: Write to stdout.
  */
  int			to_stdout;
} dk3_bm_eps_options_t;



/**	Internal conversion job structure.
*/
typedef struct {

  /**	Conversion option set.
  */
  dk3_bm_eps_options_t		*opt;

  /**	Localized message texts.
  */
  dkChar const * const		*msg;

  /**	Source file name.
  */
  dkChar const			*sfn;

  /**	Destination file name.
  */
  dkChar const			*dfn;

  /**	Destination file.
  */
  FILE				*dfile;

  /**	PDF writer structure.
  */
  dk3_pdf_t			*pdf;

  /**	Bitmap image file.
  */
  dk3_bif_t			*bif;

  /**	Communication object.
  */
  void				*pComm;

  /**	Translation x.
  */
  double			 x_translate;

  /**	Translation y.
  */
  double			 y_translate;

  /**	Scale x.
  */
  double			 x_scale;

  /**	Scale y.
  */
  double			 y_scale;

  /**	Left x coordinate (used for the draft).
  */
  double			 x_0;

  /**	Right x coordinate (used for the draft).
  */
  double			 x_1;

  /**	Lower y coordinate (used for the draft).
  */
  double			 y_0;

  /**	Upper y coordinate (used for the draft).
  */
  double			 y_1;

  /**	Image width (bounding box).
  */
  long				 x_size;

  /**	Image width (bounding box).
  */
  long				 y_size;

  /**	Number of bits per component.
  */
  size_t			 bpc;

  /**	Number of pages to produce.
  */
  size_t			 nPages;

  /**	Number of current page.
  */
  size_t			 currentPage;

  /**	Minimum value for progress bar.
  */
  int				 minpb;

  /**	Maximum value for progress bar.
  */
  int				 maxpb;

  /**	Image file type.
  */
  int				 biftype;

  /**	Flag: Must translate coordinates.
  */
  int				 do_translate;

  /**	Rotation (multiples of 90 degree).
  */
  int				 do_rotate;

  /**	Use floating point numbers for scaling.
  */
  int				 do_float_scale;

  /**	Flag: Must do header and footer.
  */
  int				 do_header_footer;

  /**	Output procedure DK3_BMEPS_xxx, see @ref bmepscompression.
  */
  int				 procedure;

} dk3_bm_conversion_job_t;



/**	Information about one font.
*/
typedef struct {
  char const		*psFontName;	/**< PS font name. */
  char const		*texFontName;	/**< LaTeX font name. */
  char const		*texFamilyName;	/**< LaTeX font family name. */
  char const		*gsFontName;	/**< GhostScript font name. */
  char const		*gsFileOld;	/**< Old GhostScript font file name. */
  char const		*gsFileNew;	/**< New GhostScript font file name. */
  char const		*psFamilyName;	/**< PS font family name. */
  char const		*svgFamilyName;	/**< SVG font family name. */
  char const		*svgFontId;	/**< SVG font identifier. */
  char const * const	*replacements;	/**< Replacement family names. */
  int			 features;	/**< Font features (bold/italic). */
  int			 svgFeatures;	/**< Font features when used in SVG. */
  int			 svgFontWeight;	/**< Font weight. */
} dk3_font_t;



/**	Bounding box data.
*/
typedef struct {
  double	xmin;	/**< Minimum x. */
  double	xmax;	/**< Maximum x. */
  double	ymin;	/**< Minimum y. */
  double	ymax;	/**< Maximum y. */
  int		xused;	/**< Flag: x-value specified. */
  int		yused;	/**< Flag: y-value specified. */
} dk3_bb_t;



/**	Coordinates transformation-
*/
typedef struct {
  double	mx;	/**< X factor. */
  double	nx;	/**< X summand. */
  double	my;	/**< Y factor. */
  double	ny;	/**< Y summand. */
} dk3_ct_2d_t;



/**	Color specified as RGB.
*/
typedef struct {
  double	r;	/**< Red. */
  double	g;	/**< Green. */
  double	b;	/**< Blue. */
} dk3_rgb_color_t;



/**	Coordinates system transformation to place an image.
	Order of operations is translate, rotate, scale.
*/
typedef struct {
  double	x_translate;	/**< X translation. */
  double	y_translate;	/**< Y translation. */
  double	x_scale;	/**< X scale factor. */
  double	y_scale;	/**< Y scale factor. */
  int		rot;		/**< Rotation angle, multiple of 90 degrees. */
} dk3_coordinates_transformation_t;



/**	Details for a Windows printer.
*/
typedef struct {
  int			dummy;		/**< No details for Windows. */
} dk3_printer_windows_details_t;



/**	Details for LPRng/LPD queue.
*/
typedef struct {
  dkChar const		*hostname;	/**< Destination host name. */
  dkChar const		*queuename;	/**< Destination queue name. */
  double		 to_c;		/**< Timeout for connect. */
  double		 to_r;		/**< Timeout for receive. */
  double		 to_s;		/**< Timeout for send. */
} dk3_printer_lprng_details_t;



/**	Details for a socket connected printer.
*/
typedef struct {
  dkChar const		*hostname;	/**< Destination host or address. */
  unsigned short	 portno;	/**< Port no, host representation. */
  unsigned char		 ordrel;	/**< Flag: Orderly release. */
  double		 to_c;		/**< Timeout for connect. */
  double		 to_r;		/**< Timeout for receive. */
  double		 to_s;		/**< Timeout for send. */
} dk3_printer_socket_t;



/**	Details for a printer.
*/
typedef union {
  dk3_printer_windows_details_t		win;	/**< Windows printer details. */
  dk3_printer_lprng_details_t		lprng;	/**< LPRng printer details. */
  dk3_printer_socket_t			sock;	/**< Socket printer details. */
} dk3_printer_details_t;



/**	Printer.
	Windows printer ports can be a connection to a remote
	raw socket or LPD/LPRng queue. In these cases t_p and det_p
	indicate "Windows printer" and the t_s and det_s contain
	information about the backend.
	For the "clean" and "status" commands we need to access
	the backend, other operations only access the Windows printer.

*/
typedef struct {
  dk3_printer_details_t	 det_p;	/**< Primary type printer details. */
  dk3_printer_details_t	 det_s;	/**< Secondary type printer details. */
  dkChar const		*name;	/**< Printer name. */
  dkChar const		*h_snmp;	/**< Host to access for SNMP. */
  dkChar const		*c_snmp;	/**< SNMP community. */
  int			 t_p;	/**< Primary type,see @ref printertypes.  */
  int			 t_s;	/**< Secondard type, see @ref printertypes.  */
  int			 v_snmp;	/**< SNMP version. */
  int			 ps;	/**< Flag: PS printer. */
} dk3_printer_t;



/**	Queue name alias.
*/
typedef struct {
  dkChar const		*name;		/**< Alias name. */
  dk3_printer_t		*printer;	/**< Alias destination. */
} dk3_printer_alias_t;



/**	Print host.
*/
typedef struct {
  dkChar const		*name;		/**< Host name. */
  double		 to_c;		/**< Timeout for connect. */
  double		 to_r;		/**< Timeout for receive. */
  double		 to_s;		/**< Timeout for send. */
  int			 enc;		/**< Host encoding (plain or UTF-8). */
} dk3_print_host_t;



/**	Print host alias.
*/
typedef struct {
  dkChar const		*name;		/**< Alias name. */
  dk3_print_host_t	*host;		/**< Destination host. */
} dk3_print_host_alias_t;



/**	Print configuration for a computer system.
*/
typedef struct {
  dk3_app_t		*app;		/**< Application structure. */
  dkChar const * const	*msg;		/**< Localized message texts. */
  dk3_sto_t		*sPrinters;	/**< Printers storage. */
  dk3_sto_it_t		*iPrinters;	/**< Iterator through printers. */
  dk3_sto_t		*sPrintAliases;	/**< Printer aliases storage. */
  dk3_sto_it_t		*iPrintAliases;	/**< Iterator through print aliases. */
  dk3_sto_t		*sPrintHosts;	/**< Print hosts storage. */
  dk3_sto_it_t		*iPrintHosts;	/**< Iterator through hosts. */
  dk3_sto_t		*sHostAliases;	/**< Print host aliases storage. */
  dk3_sto_it_t		*iHostAliases;	/**< Iterator through host aliases. */
  dk3_printer_t		*defPrinter;	/**< Default printer. */
  dk3_print_host_t	*defHost;	/**< Default host. */
  int			 defPsl;	/**< Default PS level. */
} dk3_print_conf_t;



#if DK3_HAVE_VOLATILE
/**	The volatile keyword can be used.
*/
#define	DK3_VOLATILE	volatile
#else
/**	The volatile keyword can not be used.
*/
#define	DK3_VOLATILE	/* NIX */
#endif

typedef
#if DK3_HAVE_SIG_ATOMIC_T
/**	Atomic type for signal handlers.
*/
sig_atomic_t
#else
/**	Atomic type or int for signal handlers.
*/
int
#endif
dk3_sig_atomic_t;

/**	Pointer to dkChar.
*/
typedef dkChar		*DK3_PDKCHAR;

/**	Pointer to const dkChar.
*/
typedef	dkChar const	*DK3_PCDKCHAR;

/**	Pointer to unsigned char.
*/
typedef	unsigned char	*DK3_PUCHAR;

/**	Pointer to char.
*/
typedef char		*DK3_PCHAR;

/**	Pointer to char const.
*/
typedef char const	*DK3_PCCHAR;

/**	Return type for signal handlers.
*/
typedef	void		dk3_signal_ret_t;

/**	Signal handler function prototype.
*/
typedef
dk3_signal_ret_t
dk3_signal_fct_t(int)
/*^modifies internalState^*/
;

/**	Signal disposition (handler function address).
*/
typedef	dk3_signal_fct_t	*dk3_signal_disp_t;

/**	Handler function for one character.
	@param	obj	Object to modify using the line.
	@param	ch	Character to process.
	@return	1=OK, 0=error, can continue, -1=error, abort processing.
*/
typedef
int
dk3_char_handler_t(
  void		*obj,
  dk3_c32_t	 ch
);

/**	Function to traverse a simple database.
	@param	obj	Object to modify while traversing the database.
	@param	key	Key data.
	@param	val	Value data.
	@return	1 on success, 0 on error (can continue), -1 on error (abort).
*/
typedef
int
dk3_db_traverse_fct_t(
  void		*obj,
  dk3_datum_t	*key,
  dk3_datum_t	*val
)
/*^modifies *obj,*key,*val^*/
;

#endif

