
/**	@file	dk3const.h	Constants for the dk3 libraries.
*/

#ifndef DK3_CONST_H_INCLUDED
/** Avoid multiple inclusion. */
#define DK3_CONST_H_INCLUDED 1

#include <dk3conf.h>

/**	@defgroup	apptypes	Application types.
*/
/**@{*/
enum {
/**	Application type: Not (yet) specified.

No application type was specified.
*/
DK3_APP_TYPE_UNKNOWN	=	0,

/**	Application type: Command line application.

Diagnostics go to a log file and to standard error output.
The log file is deleted by dk3app_close() unless keeping the
log file is required by the preference /log/file/keep.
*/
DK3_APP_TYPE_COMMAND,

/**	Application type: GUI application.

Diagnostics go to a log file.
The log file is deleted by dk3app_close() unless keeping the
log file is required by the preference /log/file/keep.
*/
DK3_APP_TYPE_GUI,

/**	Application type: Daemon.

Diagnostics go to a log file.
The same log file is used again and again, messages
are appended. The log file is not deleted automatically.
*/
DK3_APP_TYPE_DAEMON,

/**	Application type: Silent application.

No diagnostic messages are produced.
*/
DK3_APP_TYPE_SILENT
};
/**@}*/



/**	@defgroup	fileenc	Encodings for text files.
	These encodings can be used when writing texts to file.
	For multi-byte characters we have to decide whether to write
	MSB first or LSB first.
*/
/**@{*/
enum {
/**	Plain ascii text.
*/
DK3_FILE_ENCODING_ASCII			=	0,

/**	UTF-8 encoding.
*/
DK3_FILE_ENCODING_UTF8,

/**	UTF-16 encoding, MSB comes first.
*/
DK3_FILE_ENCODING_UTF16_MSB_FIRST,

/**	UTF-16 encoding, LSB comes first.
 */
DK3_FILE_ENCODING_UTF16_LSB_FIRST,

/**	32-bit UNICODE used directly, MSB comes first.
 */
DK3_FILE_ENCODING_UNICODE_MSB_FIRST,

/**	32-bit UNICODE used directly, LSB comes first.
 */
DK3_FILE_ENCODING_UNICODE_LSB_FIRST
};
/**@}*/



/**	@defgroup	stringenc	Encodings for text.
	These encoding are used when keeping text in memory.
*/
/**@{*/
enum {
/**	Plain ascii text.
*/
DK3_ENCODING_PLAIN	=	DK3_FILE_ENCODING_ASCII,

/**	UTF-8 encoding.
*/
DK3_ENCODING_UTF8	=	DK3_FILE_ENCODING_UTF8,

/**	UTF-16 encoding, MSB comes first.
*/
DK3_ENCODING_UTF16	=	DK3_FILE_ENCODING_UTF16_MSB_FIRST,

/**	32-bit UNICODE used directly, MSB comes first.
 */
DK3_ENCODING_UNICODE	=	DK3_FILE_ENCODING_UNICODE_MSB_FIRST
};
/**@}*/




/**	@defgroup	filetypes	File types
	These file types are returned in the ft component of the
	dk3_stat_t type.
*/
/**@{*/
enum {

/**	Regular file.
*/
DK3_FT_REGULAR			=	1,

/**	Directory.
*/
DK3_FT_DIRECTORY,

/**	FIFO.
*/
DK3_FT_FIFO,

/**	Character special device.
*/
DK3_FT_SPECIAL_CHAR,

/**	Block special device.
*/
DK3_FT_SPECIAL_BLOCK,

/**	Socket.
*/
DK3_FT_SOCKET,

/* Next there are some exotic file types. */

/**	Door.
*/
DK3_FT_DOOR,

/**	Event port.
*/
DK3_FT_EVENT_PORT,

/**	Xenix special file.
*/
DK3_FT_XENIX_SPECIAL,

/**	Xenix semaphore.
*/
DK3_FT_XENIX_SEMAPHORE,

/**	Xenix shared data.
*/
DK3_FT_XENIX_SHARED_DATA,

/**	Multiplexed character device.
 */
DK3_FT_MUX_CHAR,

/**	Multiplexed block device.
 */
DK3_FT_MUX_BLOCK,

/**	VxFS compressed file.
 */
DK3_FT_VXFS_COMPRESSED,

/**	Network special file.
 */
DK3_FT_NETWORK_SPECIAL,

/**	Whiteout (not used for inode).
 */
DK3_FT_WHITEOUT,

/**	ACL shadow inode.
 */
DK3_FT_ACL_SHADOW,

/**	Symlink with non-existing target.
*/
DK3_FT_BAD_SYMLINK,

/**	Maximum found so far. Applications must use int i[DK3_FT_MAX+1]
	to store information about all the file types.
 */
DK3_FT_MAX			=	DK3_FT_BAD_SYMLINK,

/**	Symbolic link (this flag is or-combined with one of the values below).
*/
DK3_FT_SYMLINK			=	256,

};
/**@}*/



/**	@defgroup	filepermissions	File permissions.
*/
/**@{*/
enum {
/**	Read permission for owner.
*/
DK3_FPERM_U_READ		=	0400,

/**	Write permission for owner.
*/
DK3_FPERM_U_WRITE		=	0200,

/**	Execute permission for owner.
*/
DK3_FPERM_U_EXEC		=	0100,

/**	Read permission for group.
*/
DK3_FPERM_G_READ		=	0040,

/**	Write permission for group.
*/
DK3_FPERM_G_WRITE		=	0020,

/**	Execute permission for group.
*/
DK3_FPERM_G_EXEC		=	0010,

/**	Read permission for others.
*/
DK3_FPERM_O_READ		=	0004,

/**	Write permission for others.
*/
DK3_FPERM_O_WRITE		=	0002,

/**	Execute permission for others.
*/
DK3_FPERM_O_EXEC		=	0001,

/**	Set user ID on execution.
*/
DK3_FPERM_SUID			=	04000,

/**	Set group ID on execution.
*/
DK3_FPERM_SGID			=	02000,

/**	VTX: Save swapped text segment even after use.
*/
DK3_FPERM_VTX			=	01000,

/**	Directory create mode: Only for user.
*/
DK3_FPERM_CREATE_PRIVATE	=	0700,

/**	File create mode: Only for user.
*/
DK3_FPERM_CREATE_FILE_PRIVATE	=	0600,

/**	Directory create mode: Full permissions for owner and group.
*/
DK3_FPERM_CREATE_GROUP		=	0770,

/**	File create mode: Full permissions for owner and group.
*/
DK3_FPERM_CREATE_FILE_GROUP	=	0660
};
/**@}*/



/**	@defgroup	streamuseflags	Flags how to use a stream.
	A stream can be opened for reading, writing or both.
*/
/**@{*/
enum {
/**	Allowed to read from stream.
*/
DK3_STREAM_FLAG_READ	=	1,

/**	Allowed to write to stream.
*/
DK3_STREAM_FLAG_WRITE
};
/**@}*/



/**	@defgroup	streamapicommands	Stream API commands.
	These commands are used in the dk3stream.c module for
	communication between the exported functions and the low-level
	function dealing with the stream destination.
*/
/**@{*/
enum {
/**	Read into input buffer.
*/
DK3_STREAM_API_READ		=	1,

/**	Write output buffer.
*/
DK3_STREAM_API_WRITE,

/**	Flush output buffer.
*/
DK3_STREAM_API_FLUSH,

/**	Check whether we have reached the end of data.
*/
DK3_STREAM_API_AT_END,

/**	Close the low-level data.
*/
DK3_STREAM_API_CLOSE,

/**	Check whether a zero read result indicates end of data.
*/
DK3_STREAM_API_ZERO_IS_END,

/**	Check for error.
*/
DK3_STREAM_API_ERROR
};
/**@}*/



/**	@defgroup	loglevels	Log levels.
	These levels are used to describe the importance of log messages.
	Log messages are written to a log file, command line applications
	also write to standard error output (stderr).
	For both log destinations there is a required level, only messages
	on that level or higher priorized (lower numeric value) are written
	to that destination.
*/
/**@{*/
enum {
/** This message priority never occurs. */
DK3_LL_NONE	=	0,

/** Panic message. */
DK3_LL_PANIC,

/** Fatal error message. */
DK3_LL_FATAL,

/** Error message. */
DK3_LL_ERROR,

/** Warning message. */
DK3_LL_WARNING,

/** Informational message. */
DK3_LL_INFO,

/** Progress message. */
DK3_LL_PROGRESS,

/** Debug message. */
DK3_LL_DEBUG,

/** Message can be ignored. */
DK3_LL_IGNORE
};
/**@}*/



/**	Preferences priorities.
*/
enum {
/**	Mask to retrieve scope value.
*/
DK3_PREF_SCOPE			=	0x0007,

/**	Scope: Host.
*/
DK3_PREF_SCOPE_HOST		=	0x0001,

/**	Scope: Application.
*/
DK3_PREF_SCOPE_APPLICATION	=	0x0002,

/**	Scope: User name.
*/
DK3_PREF_SCOPE_USER		=	0x0004,

/**	Mask to retrieve source.
*/
DK3_PREF_SOURCE			=	0x0018,

/**	Source: System configuration settings.
*/
DK3_PREF_SOURCE_SYSTEM		=	0x0000,

/**	Source: Users general configuration settings.
*/
DK3_PREF_SOURCE_USER		=	0x0008,

/**	Source:	Command line arguments.
*/
DK3_PREF_SOURCE_COMMANDLINE	=	0x0010,
};



/**	@defgroup	errorcodes	Error codes. */
/**@{*/
enum {
/**	No error occured.
*/
DK3_ERROR_NONE				=	0,

/**	System error, see errno.
*/
DK3_ERROR_SYSTEM,

/**	Aborted by user (cancel button).
*/
DK3_ERROR_ABORTED,

/**	Mathematical error (division by zero).
*/
DK3_ERROR_MATH_DIVZERO,

/**	Mathematical error (result out of range).
*/
DK3_ERROR_MATH_OVERFLOW,

/**	Mathematical error (at least one argument out of range).
*/
DK3_ERROR_MATH_OUT_OF_RANGE,

/**	Not enough memory.
*/
DK3_ERROR_MEMORY,

/**	Invalid arguments passed to function.
*/
DK3_ERROR_INVALID_ARGS,

/**	Incomplete configuration.
*/
DK3_ERROR_INCOMPLETE_CONFIGURATION,

/**	Stream not opened for writing.
*/
DK3_ERROR_NOT_OPENED_FOR_WRITING,

/**	Stream not opened for reading.
*/
DK3_ERROR_NOT_OPENED_FOR_READING,

/**	An error occured during a write operation.
*/
DK3_ERROR_DURING_WRITE,

/**	An error occured during a read operation.
*/
DK3_ERROR_DURING_READ,

/**	No supported (library was not available during build time).
*/
DK3_ERROR_NOT_SUPPORTED,

/**	Entry not found.
*/
DK3_ERROR_NOT_FOUND,

/**	Conversion of binary data to text failed.
*/
DK3_ERROR_CONVERSION_FAILED,

/**	Data too large.
*/
DK3_ERROR_TOO_LARGE,

/**	Data unusable, probably damaged.
*/
DK3_ERROR_DATA_DAMAGED,

/**	Syntax error in input file.
*/
DK3_ERROR_SYNTAX,

/**	Network protocol is not supported.
*/
DK3_ERROR_PROTOCOL_NOT_SUPPORTED,

/**	Network address family is not supported.
*/
DK3_ERROR_ADDRESS_FAMILY_NOT_SUPPORTED,

/**	Insufficient permissions.
*/
DK3_ERROR_INSUFFICIENT_PERMISSIONS,

/**	Illegal network protocol family.
*/
DK3_ERROR_ILLEGAL_PROTOCOL_FAMILY,

/**	Process table overflow.
*/
DK3_ERROR_PROCESS_TABLE_OVERFLOW,

/**	Limit reached for number of open filedescriptors.
*/
DK3_ERROR_TOO_MANY_OPEN_FILES,

/**	Error in I/O operation.
*/
DK3_ERROR_IO,

/**	Signal was received.
*/
DK3_ERROR_INTERRUPTED,

/**	IP address not found for host.
*/
DK3_ERROR_HOST_NOT_FOUND,

/**	Local address already in use.
*/
DK3_ERROR_ADDR_IN_USE,

/**	Out of resources.
*/
DK3_ERROR_RESOURCES,

/**	Previous connection request in processing.
*/
DK3_ERROR_IN_PROGRESS,

/**	Connection refused, no-one listening on the port.
*/
DK3_ERROR_CONNECTION_REFUSED,

/**	Socket is already connected.
*/
DK3_ERROR_ALREADY_CONNECTED,

/**	Network is unreachable.
*/
DK3_ERROR_NETWORK_UNREACHABLE,

/**	Operation timed out.
*/
DK3_ERROR_TIMEOUT,

/**	Failed to retrieve or set socket options.
*/
DK3_ERROR_SOCKET_OPTIONS,

/**	Failed to bind local address.
*/
DK3_ERROR_BIND_FAILED,

/**	The socket is already bound to an address.
*/
DK3_ERROR_ALREADY_BOUND,

/**	Operation would block.
*/
DK3_ERROR_WOULD_BLOCK,

/**	Connection reset by peer.
*/
DK3_ERROR_CONNECTION_RESET,

/**	Peer address is required to send over connectionless socket.
*/
DK3_ERROR_ADDRESS_REQUIRED,

/**	Writing to a pipe no-one is reading.
*/
DK3_ERROR_PIPE,

/**	End of input found.
*/
DK3_ERROR_END_OF_INPUT,

/**	Socket not connected.
*/
DK3_ERROR_SOCKET_NOT_CONNECTED,

/**	Listen operation failed on socket.
*/
DK3_ERROR_LISTEN_FAILED,

/**	Failed to accept incoming connection request.
*/
DK3_ERROR_ACCEPT_FAILED,

/**	Connection attempt aborted.
*/
DK3_ERROR_CONNECTION_ABORTED,

/**	Error in network protocol.
*/
DK3_ERROR_NETWORK_PROTOCOL,

/**	Message was not sent completely.
*/
DK3_ERROR_MESSAGE_NOT_SENT_COMPLETELY,

/**	Network subsystem or Winsock not ready.
*/
DK3_ERROR_NETWORK_SUBSYSTEM_NOT_READY,

/**	The address is not available!
*/
DK3_ERROR_ADDRESS_NOT_AVAILABLE,

/**	Host (temporarily) unreachable!
*/
DK3_ERROR_HOST_UNREACHABLE,

/**	Stale NFS file handle.
*/
DK3_ERROR_STALE_NFS_FILE_HANDLE,

/**	No select function available.
*/
DK3_ERROR_NO_SELECT,

/**	Iteration failed (no result after maximum number of passes).
*/
DK3_ERROR_ITERATION
};
/**@}*/



/**	Installation directories.
	Order must correspond to order of directories in the dk3inst.c module.
*/
enum {

/**	Installation prefix (/usr/local).
*/
DK3_INST_PREFIX		=	0,

/**	System configuration directory (/usr/local/etc).
*/
DK3_INST_SYSCONF	=	1,

/**	Data root directory (/usr/local/share).
*/
DK3_INST_DATAROOT	=	2,

/**	Local state directory (/usr/local/var).
*/
DK3_INST_LOCALSTATE	=	3,

/**	bin directory (/usr/local/bin).
*/
DK3_INST_BIN		=	4,

/**	sbin directory (/usr/local/sbin).
*/
DK3_INST_SBIN		=	5,

/**	lib directory (/usr/local/lib).
*/
DK3_INST_LIB		=	6,

/**	include directory (/usr/local/include).
*/
DK3_INST_INCLUDE	=	7,

/**	man directory (/usr/local/share/man).
*/
DK3_INST_MAN		=	8,

/**	Installation prefix again (/usr/local).
*/
DK3_INST_PREFIX2	=	9,

/**	libexec directory (/usr/local/libexec).
*/
DK3_INST_LIBEXEC	=	10
};



/**	@defgroup	mdtypes	Message digest types.
	Order here must match order in dk3chks -> dk3_checksum_type_names.
*/
/**@{*/
enum {
/**	RSA Security, Inc. MD5 message digest.
*/
DK3_MD_TYPE_MD5		=	1,

/**	SHA-1 message digest.
*/
DK3_MD_TYPE_SHA_1,

/**	SHA-224 message digest.
*/
DK3_MD_TYPE_SHA_224,

/**	SHA-256 message digest.
*/
DK3_MD_TYPE_SHA_256,

/**	SHA-384 message digest.
*/
DK3_MD_TYPE_SHA_384,

/**	SHA-512 message digest.
*/
DK3_MD_TYPE_SHA_512,

/**	RIPEMD-160 message digest.
*/
DK3_MD_TYPE_RIPEMD_160,

/**	Maximum number for digest type.
*/
DK3_MD_TYPE_MAX	=	DK3_MD_TYPE_RIPEMD_160
};
/**@}*/



/**	@defgroup	dataencoding	Data encoding.
	Order here must match order in dk3enc -> dk3_data_encoding_names.
*/
/**@{*/
enum {
/**	Hexadecimal encoding.
*/
DK3_DATA_ENCODING_HEX			=	0,

/**	ASCII-85 encoding.
*/
DK3_DATA_ENCODING_ASCII85,

/**	Reverse ASCII-85 encoding.
*/
DK3_DATA_ENCODING_REVERSE_ASCII85,

/**	Maximum number for encoding.
*/
DK3_DATA_ENCODING_MAX		=	DK3_DATA_ENCODING_REVERSE_ASCII85
};
/**@}*/



/**	Signal handling types.
	Different signal handling mechanisms are available for different
	operating systems.
*/
enum {
/**	No signal handling available.
*/
DK3_SIGNAL_HANDLING_TYPE_NONE		=	0,

/**	Use signal() function.
*/
DK3_SIGNAL_HANDLING_TYPE_SIGNAL,

/**	Use sigset() function.
*/
DK3_SIGNAL_HANDLING_TYPE_SIGSET,

/**	Use sigaction() function.
*/
DK3_SIGNAL_HANDLING_TYPE_SIGACTION
};


/**	@defgroup	rndtypes	Random number generator types.
	Depending on the OS and the installed libraries there are different
	PRNGs available.
*/
/**@{*/
enum {
/**	PRNG type: OpenSSL.
*/
DK3_PRNG_OPENSSL	=	1,

/**	PRNG type: initstate()/setstate().
*/
DK3_PRNG_STATE		=	2,

/**	PRNG type: nrand48().
*/
DK3_PRNG_RAND48		=	4,

/**	PRNG type: rand().
*/
DK3_PRNG_SIMPLE		=	8,

/**	Mask to enable all PRNG types.
*/
DK3_PRNG_ALL		=
(DK3_PRNG_OPENSSL + DK3_PRNG_STATE + DK3_PRNG_RAND48 + DK3_PRNG_SIMPLE)
};
/**@}*/



/**	Application commands.
*/
enum {
/**	Show help.
*/
DK3_APP_CMD_HELP		=	1,

/**	Show version.
*/
DK3_APP_CMD_VERSION		=	2,

/**	Show license conditions.
*/
DK3_APP_CMD_LICENSE		=	4
};


/**	@defgroup	outputfiltercelltypes	Output filter cell types.
*/
/**@{*/
enum {
/**	Output filter cell type: Binary to hex encoding.
*/
DK3_OF_CELL_TYPE_HEX		=	0,

/**	Output filter cell type: Binary to ASCII85 encoding.
*/
DK3_OF_CELL_TYPE_A85,

/**	Output filter cell type: Run-length compression.
*/
DK3_OF_CELL_TYPE_RUN_LENGTH,

/**	Output filter cell type: Flate compression.
*/
DK3_OF_CELL_TYPE_FLATE
};
/**@}*/



/**	Sizes for flate buffers. */
enum {
/**	Input buffer size for flate compression.
*/
DK3_OF_FLATE_IN_BUFFER_SIZE	=	1024,

/**	Output buffer size for flate compression
	(in theory 1057 would suffice).
*/
DK3_OF_FLATE_OUT_BUFFER_SIZE	=	1536
};



/**	@defgroup	databasetypes	Database types. */
/**@{*/
enum {
/**	Database type: Not yet set.
*/
DK3_DB_TYPE_UNKNOWN	=	-1,
/**	Database type: In-memory database, synchronize on
	explicit request and on close only.
*/
DK3_DB_TYPE_MEMORY	=	0,

/**	Database type: Berkeley DB.
*/
DK3_DB_TYPE_BDB		=	1,

/**	Database type: NDBM.
*/
DK3_DB_TYPE_NDBM	=	2
};
/**@}*/



/**	Database type for SQL databases. */
enum {
/**	SQL database type: Not type configured yet.
*/
DK3_SQLDB_TYPE_UNKNOWN	=	-1,

/**	SQL database type: Not type configured yet.
*/
DK3_SQLDB_TYPE_MYSQL	=	0
};



/**	@defgroup	databaseaccess	Access types for databases. */
/**@{*/
enum {
/**	Database access: Read only.
*/
DK3_DB_ACCESS_READ	=	1,

/**	Database access: Write entries.
*/
DK3_DB_ACCESS_WRITE	=	2
};
/**@}*/



/**	Text and object alignments.
*/
enum {
/**	Align: Horizontally and vertically centered.
*/
DK3_ALIGN_CENTER		=	0,

/**	Align: Horizontally centered.
*/
DK3_ALIGN_CENTER_HORIZONTAL	=	0,

/**	Align: Left.
*/
DK3_ALIGN_LEFT			=	1,

/**	Align: Right.
*/
DK3_ALIGN_RIGHT			=	2,

/**	Align: Mask to retrieve horizontal alignment.
*/
DK3_ALIGN_HORIZONTAL		=	3,

/**	Align: Vertically centered.
*/
DK3_ALIGN_CENTER_VERTICAL	=	0,

/**	Align: Top.
*/
DK3_ALIGN_TOP			=	4,

/**	Align: Bottom.
*/
DK3_ALIGN_BOTTOM		=	8,

/**	Align: Mask to retrieve vertical alignment.
*/
DK3_ALIGN_VERTICAL		=	12
};



/**	@defgroup	retrieveguialignment Retrieve GUI alignment.
*/
/**@{*/
/**	Retrieve horizontal alignment from horizontal/vertical combination.
*/
#define	DK3_ALIGN_GET_HORIZONTAL(x)	((x) & DK3_ALIGN_HORIZONTAL)

/**	Retrieve vertical alignment from horizontal/vertical combination.
*/
#define	DK3_ALIGN_GET_VERTICAL(x)	((x) & DK3_ALIGN_VERTICAL)
/**@}*/



/**	@defgroup	colorspaces	Color spaces for bitmaps.
*/
/**@{*/
enum {
/**	Color space: Not defined yet.
*/
DK3_COLOR_SPACE_UNKNOWN		=	0,

/**	Color space: Grey.
*/
DK3_COLOR_SPACE_GRAY,

/**	Color space: Grey with alpha channel.
*/
DK3_COLOR_SPACE_GRAY_ALPHA,

/**	Color space: RGB.
*/
DK3_COLOR_SPACE_RGB,

/**	Color space: RGB with alpha channel.
*/
DK3_COLOR_SPACE_RGBA,

/**	Color space: CMYK.
*/
DK3_COLOR_SPACE_CMYK,

/**	Color space: Y Cb Cr
*/
DK3_COLOR_SPACE_YCbCr,

/**	Color space: Y Cb Cr K (inverted CMYK?)
*/
DK3_COLOR_SPACE_YCCK,

/**	Color space: Hue, saturation, brightness.
*/
DK3_COLOR_SPACE_HSB,
};
/**@}*/



/**	@defgroup	linecap	Line cap styles. */
/**@{*/
enum {
/**	Line cap: Butted.
*/
DK3_LINECAP_BUTT	=	0,

/**	Line cap: Rounded.
*/
DK3_LINECAP_ROUND,

/**	Line cap: Half linewidt added.
*/
DK3_LINECAP_SQUARE
};
/**@}*/



/**	Line join styles.  */
enum {
/**	Line join: Mitered.
*/
DK3_LINEJOIN_MITER		=	0,

/**	Line join: Rounded corner.
*/
DK3_LINEJOIN_ROUND,

/**	Line join: Cut corner.
*/
DK3_LINEJOIN_BEVEL
};



/**	@defgroup	bifimagetypes	BIF image types.
*/
/**@{*/
enum {
/**	Image type: Unknown.
*/
DK3_BIF_IMAGE_TYPE_UNKNOWN	=	-1,

/**	Image type: PNG.
*/
DK3_BIF_IMAGE_TYPE_PNG		=	0,

/**	Image type: JPEG.
*/
DK3_BIF_IMAGE_TYPE_JPEG		=	1,

/**	Image type: TIFF.
*/
DK3_BIF_IMAGE_TYPE_TIFF		=	4
};
/**@}*/



/**	@defgroup	bifanalysis	Analysis types for bitmap images.
*/
/**@{*/
enum {
/**	Bitmap image analysis: Number of bits necessary.
*/
DK3_BIF_ANALYSIS_BITS		=	1,

/**	Bitmap image analysis: Color really used.
*/
DK3_BIF_ANALYSIS_COLOR		=	2,

/**	Bitmap image analysis: Alpha channel really used.
*/
DK3_BIF_ANALYSIS_ALPHA		=	4,

/**	Bitmap image analysis: Complete analysis
*/
DK3_BIF_ANALYSIS_ALL		=	7,

/**	Bitmap image analysis: Analysis to prepare PS output.
*/
DK3_BIF_ANALYSIS_FOR_PS		=	3,
};
/**@}*/



/**	@defgroup	dkbmmodes	Run modes for dkbm and wxdkbm.
*/
/**@{*/
enum {
/**	Bitmap to PS/PDF conversion: Create standalone image.

	Alpha channel: The image is mixed against a background color
	defined in the source file or on the command line or mixed against
	white if there is no background chunk.

	Frames: If the input contains multiple frames only the first
	frame is used.

	An optional resolution chunk from the source file is used
	to scale the image, otherwise each pixel is sized to 1 PS/PDF point.
	No automatic rotation.

	A showpage instruction is written to PS output, not to EPS
	output.
*/
DK3_BMEPS_MODE_IMAGE		=	0,

/**	Bitmap to PS/PDF conversion: Create object to embed in a document.

	Alpha channel for EPS output: The image is mixed against
	a background specified in the file or on the command line or against
	a white background.

	Alpha channel for PDF output: The alpha channel is transferred
	to the output so the object can be placed over other objects.

	Frames: If the input file contains multiple frames only the first
	frame is used.

	Each pixel is sized to 1 PS/PDF point. No automatic rotation.
	Scaling and rotating is up to the application processing
	the document in which the object is embedded.

	No showpage instruction is written to PS output.
*/
DK3_BMEPS_MODE_OBJECT		=	1,

/**	Bitmap to PS/PDF conversion: Create print job or standalone PDF file.

	Alpha channel: The image is mixed against a background color
	defined in the source file or on command line or mixed against
	white if there is no background chunk.

	Frames: If the input contains multiple frames all frames are
	used, the output contains multiple pages.

	The image is placed in a given area to fill it optimally.
	Automatic rotation may occur to best fit the area.

	For PS output a showpage instruction is written at the end
	of each page.
*/
DK3_BMEPS_MODE_DOCUMENT		=	2,

/**	Bitmap to PS/PDF conversion: Create object to embed in a document
	in a given area.

	Alpha channel for EPS output: The image is mixed against
	a background specified in the file or on the command line or against
	a white background.

	Alpha channel for PDF output: The alpha channel is transferred
	to the output so the object can be placed over other objects.

	Frames: If the input contains multiple frames only the first
	frame is used.

	The image is placed in a given area with a specified rotation
	mode. Rotation depends on a DK3_IMAGE_ORIGIN_xxx value,
	see @ref imageorigin.

	No showpage instruction is written to PS output.
*/
DK3_BMEPS_MODE_PLACED_OBJECT	=	3
};
/**@}*/



/**	@defgroup	bmepsdrivers	Output drivers for dkbm/wxdkbm.
*/
/**@{*/
enum {
/**	Bitmap to PS/PDF conversion: Output driver to create PS.
*/
DK3_BMEPS_DRIVER_PS		=	0,

/**	Bitmap to PS/PDF conversion: Output driver to create EPS.
*/
DK3_BMEPS_DRIVER_EPS,

/**	Bitmap to PS/PDF conversion: Output driver to create PDF.
*/
DK3_BMEPS_DRIVER_PDF,

/**	Bitmap to PS/PDF conversion: Output driver to create a bounding box.
*/
DK3_BMEPS_DRIVER_BB,

/**	Bitmap to PS/PDF conversion: Output driver for print using wxWidgets.
*/
DK3_BMEPS_DRIVER_WXPRINT
};
/**@}*/



/**	@defgroup	bmepssizehandling	Size handling in bmeps.
*/
/**@{*/
enum {
/**	Bmeps size handling:	1 pixel corresponds to 1 PS point.
*/
DK3_BMEPS_SIZE_1_1		=	0,

/**	Bmeps size handling:	Use resolution chunk from input file.
*/
DK3_BMEPS_SIZE_CHUNK,

/**	Bmeps size handling:	DPI size specified.
*/
DK3_BMEPS_SIZE_SPECIFIED,

/**	Bmeps size handling:	Paper size specified.
*/
DK3_BMEPS_SIZE_PAPER
};
/**@}*/



/**	@defgroup	imageorigin	Rotation instructions for images.
	In XFig and transfig images are rotated depending on the
	position of the first image rectangle point click and
	the flipped flag.
*/
/**@{*/
enum {
/**	Image placement: First click in left top point, not flipped.
*/
DK3_IMAGE_ORIGIN_LEFT_TOP		=	0,

/**	Image placement: First click in left bottom point, not flipped.
*/
DK3_IMAGE_ORIGIN_LEFT_BOTTOM,

/**	Image placement: First click in right top point, not flipped.
*/
DK3_IMAGE_ORIGIN_RIGHT_TOP,

/**	Image placement: First click in right bottom point, not flipped.
*/
DK3_IMAGE_ORIGIN_RIGHT_BOTTOM,

/**	Image placement: First click in left top point, flipped.
*/
DK3_IMAGE_ORIGIN_LEFT_TOP_FLIPPED,

/**	Image placement: First click in left bottom point, flipped.
*/
DK3_IMAGE_ORIGIN_LEFT_BOTTOM_FLIPPED,

/**	Image placement: First click in right top point, flipped.
*/
DK3_IMAGE_ORIGIN_RIGHT_TOP_FLIPPED,

/**	Image placement: First click in right bottom point, flipped.
*/
DK3_IMAGE_ORIGIN_RIGHT_BOTTOM_FLIPPED
};
/**@}*/



/**	@defgroup	flatepredictors	Predictors for flate compression.
*/
/**@{*/
enum {
/**	Compression predictor: None.
*/
DK3_COMPRESSION_PREDICTOR_NONE		=	0,

/**	Compression predictor: TIFF.
*/
DK3_COMPRESSION_PREDICTOR_TIFF,

/**	Compression predictor: PNG sub (value left).
*/
DK3_COMPRESSION_PREDICTOR_PNG_SUB,

/**	Compression predictor: PNG up (value above).
*/
DK3_COMPRESSION_PREDICTOR_PNG_UP,

/**	Compression predictor: PNG average.
*/
DK3_COMPRESSION_PREDICTOR_PNG_AVERAGE,

/**	Compression predictor: PNG paeth.
*/
DK3_COMPRESSION_PREDICTOR_PNG_PAETH
};
/**@}*/



/**	Run state.
*/
enum {
/**	Run state: No autorun required.
*/
DK3_WX_AUTOSTART_NORMAL		=	0,

/**	Run state: Waiting for autorun start.
*/
DK3_WX_AUTOSTART_START,

/**	Run state: Running.
*/
DK3_WX_AUTOSTART_RUNNING,

/**	Run state: Finished, application can exit.
*/
DK3_WX_AUTOSTART_FINISHED,

/**	Run state: Application allowed to exit and notified about that.
*/
DK3_WX_AUTOSTART_WAIT_FOR_EXIT
};



/**	OpenSSL PRNG seeding.
*/
enum {
/**	Number of bytes required to seed OpenSSL PRNG.
*/
DK3_SEED_OPENSSL_BYTES	=	1024
};



/**	Paper size configuration file.
*/
enum {
/**	Configuration line size for paper formats.
*/
DK3_PAPER_CONFIG_LINE_SIZE	=	256
};



/**	@defgroup	pdfxobjecttype	XObject types for PDF creation.
*/
/**@{*/
enum {
/**	XObject represents an image.
*/
DK3_PDF_XO_TYPE_IMAGE	=	1,

/**	XObject represents an alpha mask.
*/
DK3_PDF_XO_TYPE_ALPHA
};
/**@}*/



/**	@defgroup	shutdownvalues	Values for socket shutdown.
*/
/**@{*/
enum {
/**	Shutdown socket for read access.
*/
DK3_TCPIP_SHUTDOWN_READ		=	0,

/**	Shutdown socket for write access.
*/
DK3_TCPIP_SHUTDOWN_WRITE,

/**	Shutdown socket for both read and write access.
*/
DK3_TCPIP_SHUTDOWN_RW
};
/**@}*/



#if 0
/**	Socket states.
*/
enum {
/**	Socket state: Initialized.
*/
DK3_SOCKET_STATE_INITIALIZED	=	1
};
#endif



/**	SNMP versions.
*/
enum {
/**	SNMP version not (yet) known.
*/
DK3_SNMP_VERSION_UNKNOWN	=	0,

/**	SNMP version: 1.
*/
DK3_SNMP_VERSION_1,

/**	SNMP version: 2c.
*/
DK3_SNMP_VERSION_2C,

/**	SNMP version: 2p.
*/
DK3_SNMP_VERSION_2P,

/**	SNMP version: 3.
*/
DK3_SNMP_VERSION_3
};



/**	Font features.
*/
enum {
/**	Roman font.
*/
DK3_FONT_ROMAN			=	0,

/**	Sans-serif font.
*/
DK3_FONT_SANS_SERIF		=	1,

/**	Typewriter font.
*/
DK3_FONT_TYPEWRITER		=	2,

/**	Mask to retrieve font category.
*/
DK3_FONT_FAMILY			=	3,

/**	Bold font.
*/
DK3_FONT_BOLD			=	4,

/**	Italic font (in german: kursiv).
*/
DK3_FONT_ITALIC			=	8,

/**	Oblique font (in german: schraeggestellt).
*/
DK3_FONT_OBLIQUE		=	16,

/**	Mask value to check if text is either bold or oblique.
*/
DK3_FONT_ITOB			=	24
};



/**	PS fonts.
	We use the same font order as the Fig file standard.
*/
enum {
/**	PS font: Default font (Times Roman).
*/
DK3_FONT_PS_DEFAULT					=	-1,

/**	PS font: Times Roman.
*/
DK3_FONT_PS_TIMES_ROMAN					=	0,

/**	PS font: Times Roman Italic.
*/
DK3_FONT_PS_TIMES_ITALIC,

/**	PS font: Times Roman Bold.
*/
DK3_FONT_PS_TIMES_BOLD,

/**	PS font: Times Roman Bold Italic.
*/
DK3_FONT_PS_TIMES_BOLD_ITALIC,

/**	PS font: Avantgarde Book.
*/
DK3_FONT_PS_AVANTGARDE_BOOK,

/**	PS font: Avantgarde Book Oblique.
*/
DK3_FONT_PS_AVANTGARDE_BOOK_OBLIQUE,

/**	PS font: Avantgarde Demi.
*/
DK3_FONT_PS_AVANTGARDE_DEMI,

/**	PS font: Avantgarde Demi Oblique.
*/
DK3_FONT_PS_AVANTGARDE_DEMI_OBLIQUE,

/**	PS font: Bookman Light.
*/
DK3_FONT_PS_BOOKMAN_LIGHT,

/**	PS font: Bookman Light Italic.
*/
DK3_FONT_PS_BOOKMAN_LIGHT_ITALIC,

/**	PS font: Bookman Demi.
*/
DK3_FONT_PS_BOOKMAN_DEMI,

/**	PS font: Bookman Demi Italic.
*/
DK3_FONT_PS_BOOKMAN_DEMI_ITALIC,

/**	PS font: Courier.
*/
DK3_FONT_PS_COURIER,

/**	PS font: Courier Oblique.
*/
DK3_FONT_PS_COURIER_OBLIQUE,

/**	PS font: Courier Bold.
*/
DK3_FONT_PS_COURIER_BOLD,

/**	PS font: Courier Bold Oblique.
*/
DK3_FONT_PS_COURIER_BOLD_OBLIQUE,

/**	PS font: Helvetica.
*/
DK3_FONT_PS_HELVETICA,
	
/**	PS font: Helvetica Oblique.
*/
DK3_FONT_PS_HELVETICA_OBLIQUE,

/**	PS font: Helvetica Bold.
*/
DK3_FONT_PS_HELVETICA_BOLD,

/**	PS font: Helvetica Bold Oblique.
*/
DK3_FONT_PS_HELVETICA_BOLD_OBLIQUE,

/**	PS font: Helvetica Narrow.
*/
DK3_FONT_PS_HELVETICA_NARROW,

/**	PS font: Helvetica Narrow Oblique.
*/
DK3_FONT_PS_HELVETICA_NARROW_OBLIQUE,

/**	PS font: Helvetica Narrow Bold.
*/
DK3_FONT_PS_HELVETICA_NARROW_BOLD,

/**	PS font: Helvetica Narrow Bold Oblique.
*/
DK3_FONT_PS_HELVETICA_NARROW_BOLD_OBLIQUE,

/**	PS font: New Century Schoolbook Roman.
*/
DK3_FONT_PS_NEW_CENTURY_SCHOOLBOOK_ROMAN,

/**	PS font: New Century Schoolbook Italic.
*/
DK3_FONT_PS_NEW_CENTURY_SCHOOLBOOK_ITALIC,

/**	PS font: New Century Schoolbook Bold.
*/
DK3_FONT_PS_NEW_CENTURY_SCHOOLBOOK_BOLD,

/**	PS font: New Century Schoolbook Bold Italic.
*/
DK3_FONT_PS_NEW_CENTURY_SCHOOLBOOK_BOLD_ITALIC,

/**	PS font: Palatino Roman.
*/
DK3_FONT_PS_PALATINO_ROMAN,

/**	PS font: Palatino Roman Italic.
*/
DK3_FONT_PS_PALATINO_ITALIC,

/**	PS font: Palatino Roman Bold.
*/
DK3_FONT_PS_PALATINO_BOLD,

/**	PS font: Palatino Roman Bold Italic.
*/
DK3_FONT_PS_PALATINO_BOLD_ITALIC,

/**	PS font: Symbol.
*/
DK3_FONT_PS_SYMBOL,

/**	PS font: Zapf Chancery.
*/
DK3_FONT_PS_ZAPF_CHANCERY_MEDIUM_ITALIC,

/**	PS font: Zapf Dingbats.
*/
DK3_FONT_PS_ZAPF_DINGBATS
};



/**	@defgroup	printertypes	Printer types.
*/
/**@{*/
enum {
/**	Printer type: Unspecified/unknown.
*/
DK3_PRINTER_TYPE_NONE		=	0,

/**	Printer type: Windows system printer.
*/
DK3_PRINTER_TYPE_WINDOWS,

/**	Printer type: Raw socket.
*/
DK3_PRINTER_TYPE_SOCKET,

/**	Printer type: LPD.
*/
DK3_PRINTER_TYPE_LPD,

/**	Printer type: LPRng.
*/
DK3_PRINTER_TYPE_LPRNG
};
/**@}*/



/*	Not an enum:
	The dkChar data type has different size, can be 8, 16 or
	32 bit long. So our constants can not be type int, and thus
	not enum.
*/
/**	@defgroup	characters	Characters. */
/**@{*/
#if DK3_CHAR_SIZE > 1
#if DK3_CHAR_SIZE > 2
/* +++ 32 bit +++ */
/**	String finalizer.
*/
#define	DK3_CHAR_0	0UL

/**	Byte order marker (BOM)
*/
#define	DK3_CHAR_BOM	0x0000FEFFUL

/**	Backslash.
*/
#define	DK3_CHAR_BS	0x0000005CUL

/**	Slash.
*/
#define	DK3_CHAR_SLASH	0x0000002FUL

/**	Minus.
*/
#define	DK3_CHAR_MINUS	0x0000002DUL

#if DK3_HAVE_BACKSLASH
/**	File name separator.
*/
#define	DK3_CHAR_SEP	0x0000005CUL
/**	Not a file name separator.
*/
#define	DK3_CHAR_NOSEP	0x0000002FUL
#else
/**	File name separator.
*/
#define	DK3_CHAR_SEP	0x0000002FUL
/**	Not a file name separator.
*/
#define	DK3_CHAR_NOSEP	0x0000005CUL
#endif
/**	Newline.
*/
#define	DK3_CHAR_NL	0x0000000AUL
/**	Dot.
*/
#define	DK3_CHAR_DOT	0x0000002EUL
/* --- 32 bit --- */
#else
/* +++ 16 bit +++ */
/**	String finalizer.
*/
#define	DK3_CHAR_0	0U

/**	Byte order marker (BOM).
*/
#define DK3_CHAR_BOM	0xFEFFU

/**	Backslash.
*/
#define	DK3_CHAR_BS	0x005CU

/**	Slash.
*/
#define	DK3_CHAR_SLASH	0x002FU

/**	Minus.
*/
#define	DK3_CHAR_MINUS	0x002DU

#if DK3_HAVE_BACKSLASH
/**	File name separator.
*/
#define	DK3_CHAR_SEP	0x005CU
/**	Not a file name separator.
*/
#define	DK3_CHAR_NOSEP	0x002FU
#else
/**	File name separator.
*/
#define	DK3_CHAR_SEP	0x002FU
/**	Not a file name separator.
*/
#define	DK3_CHAR_NOSEP	0x005CU
#endif
/**	Newline.
*/
#define	DK3_CHAR_NL	0x000AU
/**	Dot.
*/
#define	DK3_CHAR_DOT	0x002EU
/* --- 16 bit --- */
#endif
#else
/* +++ 8 bit +++ */
/**	String finalizer.
*/
#define	DK3_CHAR_0	'\0'

/**	Backslash.
*/
#define	DK3_CHAR_BS	'\\'

/**	Slash.
*/
#define	DK3_CHAR_SLASH	'/'

/**	Minus
*/
#define	DK3_CHAR_MINUS	'-'

#if DK3_HAVE_BACKSLASH
/**	File name separator.
*/
#define	DK3_CHAR_SEP	'\\'
/**	Not a file name separator.
*/
#define	DK3_CHAR_NOSEP	'/'
#else
/**	File name separator.
*/
#define	DK3_CHAR_SEP	'/'
/**	Not a file name separator.
*/
#define	DK3_CHAR_NOSEP	'\\'
#endif
/**	Newline.
*/
#define	DK3_CHAR_NL	'\n'
/**	Dot.
*/
#define	DK3_CHAR_DOT	'.'
/* --- 8 bit --- */
#endif

/**@}*/



#ifndef DK3_MAX_PATH
#if DK3_ON_WINDOWS
#ifdef _MAX_PATH
#define	DK3_MAX_PATH	_MAX_PATH
#else
/**	Maximum file name (path) length.
*/
#define	DK3_MAX_PATH	1024
#endif
#else
#ifdef MAX_PATH_LEN
/**	Maximum file name (path) length.
*/
#define	DK3_MAX_PATH	MAX_PATH_LEN
#else
/**	Maximum file name (path) length.
*/
#define	DK3_MAX_PATH	1024
#endif
#endif
#endif



/*	Not an enum:
	As we might have multiple dk3_stat_t in use at same time
	we used the smallest possible data type -- unsigned char -- for
	this information.
*/
/** @defgroup	statadditional Additional information from stat
The dk3sf_stat() function internally uses both stat() and lstat() to
retrieve information.
*/
/**@{*/
/**	The link owner is not the same user as the link targets owner.
*/
#define	DK3_STAT_AI_USER_DIFFERS	0x01

/**	The link owner group is not the same as the link targets owner group.
*/
#define	DK3_STAT_AI_GROUP_DIFFERS	0x02

/**	Link and link target are on different file systems.
*/
#define	DK3_STAT_AI_FAR_LINK		0x04
/**@}*/



/*	Not an enum:
	As we might have multiple outut streams in use at same
	time we used the smallest possible data type.
*/
/**	@defgroup	outputfiltertypes	Output filter flags.
*/
/**@{*/
/**	Flag: Use cr/nl sequence.
*/
#define	DK3_OF_FLAG_CRNL	0x01

/**	Flag: Add finalizer to end of output.
*/
#define	DK3_OF_FLAG_FINALIZE	0x02
/**@}*/



/*	Not an enum:
	As there are a lot of probable error conditions we
	use an unsigned long here for the bits.
*/
/**	@defgroup	snmpprintererrors	SNMP reported printer errors.
	Note: The unsigned long value is filled beginning from the LSB
	when reading the SNMP response beginning from the MSB of the
	first byte.
*/
/**@{*/

/**	Printer error condition: Paper low.
*/
#define	DK3_SNMP_PRINTER_ERROR_LOW_PAPER	0x00000001UL

/**	Printer error condition: No paper available.
*/
#define	DK3_SNMP_PRINTER_ERROR_NO_PAPER		0x00000002UL

/**	Printer error condition: Toner low.
*/
#define DK3_SNMP_PRINTER_ERROR_LOW_TONER	0x00000004UL

/**	Printer error condition: Toner empty.
*/
#define DK3_SNMP_PRINTER_ERROR_NO_TONER		0x00000008UL

/**	Printer error condition: Door is open
*/
#define DK3_SNMP_PRINTER_ERROR_DOOR_OPEN	0x00000010UL

/**	Printer error condition: Paper jam.
*/
#define DK3_SNMP_PRINTER_ERROR_JAMMED		0x00000020UL

/**	Printer error condition: Printer offline for whatever reason.
*/
#define DK3_SNMP_PRINTER_ERROR_OFFLINE		0x00000040UL

/**	Printer error condition: Service needed.
*/
#define DK3_SNMP_PRINTER_ERROR_SERVICE_NEEDED	0x00000080UL

/**	Printer error condition: Input tray is missing.
*/
#define DK3_SNMP_PRINTER_ERROR_INPUT_TRAY	0x00000100UL

/**	Printer error condition: Output tray is missing.
*/
#define DK3_SNMP_PRINTER_ERROR_OUTPUT_TRAY	0x00000200UL

/**	Printer error condition: Marker supply is missing.
*/
#define DK3_SNMP_PRINTER_ERROR_MARKER_SUPPLY	0x00000400UL

/**	Printer error condition: Output tray nearly full.
*/
#define DK3_SNMP_PRINTER_ERROR_OUTPUT_NEAR	0x00000800UL

/**	Printer error condition: Output tray full.
*/
#define DK3_SNMP_PRINTER_ERROR_OUTPUT_FULL	0x00001000UL

/**	Printer error condition: Input tray is empty.
*/
#define DK3_SNMP_PRINTER_ERROR_INPUT_EMPTY	0x00002000UL

/**	Printer error condition: Overdue preventive maintenance.
*/
#define DK3_SNMP_PRINTER_ERROR_MAINTENANCE	0x00004000UL
/**@}*/



/*	Not an enum:
	As the font encoding information is stored for each
	Unicode to LaTeX character conversion we used the smallest
	possible data type.
*/
/**	@defgroup	fontenc	LaTeX font encodings.
*/
/**@{*/

/**	No specific requirements.
*/
#define	DK3_FONT_ENCODING_NONE		((dk3_font_encoding_t)0x00)

/**	OT1 required.
*/
#define DK3_FONT_ENCODING_OT1		((dk3_font_encoding_t)0x01)

/**	Do not use OT1.
*/
#define DK3_FONT_ENCODING_NOT_OT1	((dk3_font_encoding_t)0x02)

/**	T1 required.
*/
#define DK3_FONT_ENCODING_T1		((dk3_font_encoding_t)0x04)

/**	Do not use T1.
*/
#define DK3_FONT_ENCODING_NOT_T1	((dk3_font_encoding_t)0x08)

/**	T4 required.
*/
#define DK3_FONT_ENCODING_T4		((dk3_font_encoding_t)0x10)

/**	Do not use T4.
*/
#define DK3_FONT_ENCODING_NOT_T4	((dk3_font_encoding_t)0x20)

/**	T5 required.
*/
#define DK3_FONT_ENCODING_T5		((dk3_font_encoding_t)0x40)

/**	Do not use T5.
*/
#define DK3_FONT_ENCODING_NOT_T5	((dk3_font_encoding_t)0x80)
/**@}*/





#endif

