unit ch_parse;

interface

uses
  crt, dos, ch_chess, ch_files;

const
  use_short_notation : boolean = true;



procedure parse_nic_gamehead(VAR game : gametype);

procedure parse_cb_gamehead(VAR game : gametype);

procedure write_cb_gamehead(VAR game: gametype);

procedure write_nic_gamehead(VAR game: gametype);

procedure parse_nic_moves(VAR game:gametype);

procedure print_moves(VAR f1: text; VAR game: gametype);

procedure print_pgn(VAR f1: text; VAR game: gametype);

function parse_cb_move(VAR game: gametype; VAR board:boardtype;
                       move_numb: byte): string;

procedure parse_cb_moves(VAR game: gametype; first_byte: word);

procedure write_cb_moves(VAR game: gametype; byte_numb: word);

procedure move2pgn(VAR board: boardtype; VAR move: string);

function move2cb(VAR board: boardtype; move: movetype):byte;

procedure write_nic_moves(VAR game: gametype);

implementation

const

  piece_token : array[1..6] of char
  = ('Q','N','R','B','K','P');

type
  nic_piecetype = array[0..16] of string[3];
  nic_piece_listtype = array[false..true,1..6] of string;
  numb_of_piecestype = array[false..true] of shortint;


function get1bit(VAR game: gametype) : byte;
var
b : boolean;
i, j : longint;
b2 : byte;

begin
  i := game.bitnumb div 8;
  j := game.bitnumb mod 8;
  b:=((game.bytes[i] and bitval[j])=bitval[j]);
  if b then
    b2:=1
  else
    b2:=0;
  inc(game.bitnumb);
  get1bit:=b2;
end; {get1bit}

function getbits(VAR game: gametype; numb_of_bits:byte) : word;
var
  b : word;
  i : byte;

begin
  b:=0;
  for i:=1 to numb_of_bits do
  begin
    b:=b+bitval[i-1]*get1bit(game);
  end;
  getbits:=b;
end;


function get_text(VAR game: gametype): string;
var
  s : string;
  b : byte;

begin
  s:='';
  repeat
    b:=getbits(game,7);
    if b<>0 then
    begin
      if b>=32 then
        s:=s+chr(b)
      else
        s:=s+chr(b+128);
    end;
  until b=0;
  get_text:=s;
end;

procedure put1bit(VAR game: gametype; b : word);
var
  i, j: longint;
  b2 : byte;

begin
  i := game.bitnumb div 8;
  j := game.bitnumb mod 8;
  if j=0 then
    game.bytes[i]:=0;
  b2:=bitval[j];
  if b<>0 then
    game.bytes[i]:=(game.bytes[i] and (255-b2))+ b2
  else
    game.bytes[i]:=(game.bytes[i] and (255-b2));
  inc(game.bitnumb);
end;

procedure putbits(VAR game: gametype; numb_of_bits: byte; w: word);
var
  i : integer;

begin
  for i:=0 to numb_of_bits-1 do
  begin
    put1bit(game,w and bitval[i]);
  end;
end;

procedure puttext(VAR game: gametype; s: string);
var
  i : integer;
  b : byte;

begin
  for i:=1 to length(s) do
  begin
    b:=ord(s[i]) and 127;
    putbits(game,7,b);
  end;
  putbits(game,7,0);
end;

procedure move2pgn(VAR board: boardtype; VAR move: string);
var
  x1, y1, x2, y2 : byte;
  s : string;

begin
  x1:=ord(move[2])-96;
  y1:=ord(move[3])-48;
  x2:=ord(move[4])-96;
  y2:=ord(move[5])-48;
  if (copy(move,1,2)='Ke') and (pos(move[4],'gc')>0) then
  begin
    if (move[4]='g') then
       move:='O-O';
    if (move[4]='c') then
       move:='O-O-O';
  end
  else
  begin
    if move[1]='P' then
      s:=copy(move,2,2)
    else
      s:=copy(move,1,3);

    if board.square[x2,y2].color='.' then
      s:=s+'-'
    else
      s:=s+'x';
    s:=s+copy(move,4,length(move)-3);
    move:=s;
  end;
end;

procedure parse_nic_gamehead(VAR game : gametype);
var
  byte0, byte1, byte2,temp1, temp2, temp3, numb_of_codes,i  : byte;
  resultnumb, w_titlenumb, b_titlenumb : byte;
  keystr: array[1..4] of string[3];
  keycode: array[1..4] of byte;
  numb2str : string;

begin
  game.bitnumb:=0;
  byte0:=getbits(game,8);
  if byte0=0 then
  begin
    game.overwritten:=true;
    byte0:=getbits(game,8);
  end
  else
  begin
    game.overwritten:=false;
  end;
  byte1:=getbits(game,8);
  byte2:=getbits(game,8);
  game.numb_of_moves:=getbits(game,8);

  (***)
  if game.numb_of_moves>127 then
    temp1:=getbits(game,7);

  game.deleted:=       ((byte0 and 1)=1);
  game.position:=      ((byte0 and 2)=2);
  game.comment:=       ((byte0 and 4)=4);
  game.quiz:=          ((byte0 and 8)=8);
  game.alternatives:=  ((byte0 and 32)=32);
  game.normal_game:=   ((byte0 and 128)=128);

  game.nic_opening:='';
  if (byte0 and 16)=16 then
  begin
    game.nic_opening:=game.nic_opening+chr(getbits(game,5)+64)+chr(getbits(game,5)+64)+' ';

    temp1:=getbits(game,2);

    temp1:=getbits(game,4);
    if temp1=0 then
      numb_of_codes:=1
    else
    if temp1=4 then
      numb_of_codes:=2
    else
    if temp1=5 then
      numb_of_codes:=3
    else
    if temp1=6 then
      numb_of_codes:=3
    else
    if temp1=8 then
      numb_of_codes:=2
    else
    if temp1=12 then
      numb_of_codes:=2
    else
      numb_of_codes:=4;

    keycode[1]:=0;
    temp2:=getbits(game,2);
    if temp2=1 then
    begin
      keycode[1]:=getbits(game,4);
    end
    else
    if temp2=2 then
    begin
      keycode[1]:=getbits(game,4)*16;
    end
    else
    if temp2=3 then
    begin
      keycode[1]:=getbits(game,4)+getbits(game,4)*16;
    end;
    i:=1;
    if keycode[1]<>0 then
    begin
      str(keycode[1],keystr[1]);
      game.nic_opening:=game.nic_opening+keystr[1];
    end;
    while i<numb_of_codes do
    begin
      inc(i);
      keycode[i]:=getbits(game,4);

      if (temp1=8) and (i=2) then
        keycode[2]:=keycode[2]*16;

      if (temp1=12) and (i=2) then
      begin
        keycode[2]:=keycode[2]+16;
        temp3:=getbits(game,4);
      end;

      if (temp1=6) and (i=3) then
        keycode[3]:=keycode[3]*16;

      str(keycode[i],keystr[i]);
      game.nic_opening:=game.nic_opening+'.'+keystr[i];
    end;

  end;

  if (byte1 and 1)=1 then
    game.w_name:=get_text(game)
  else
    game.w_name:='';
  if (byte1 and 2)=2 then
    game.b_name:=get_text(game)
  else
    game.b_name:='';
  if (byte1 and 4)=4 then
    game.w_elo:=getbits(game,13)
  else
    game.w_elo:=0;
  if (byte1 and 8)=8 then
    game.b_elo:=getbits(game,13)
  else
    game.b_elo:=0;
  if (byte1 and 16)=16 then
    w_titlenumb:=getbits(game,4)
  else
    w_titlenumb:=0;
  case w_titlenumb of
  1: game.w_title:='IGM';
  2: game.w_title:='IM';
  3: game.w_title:='FM';
  4: game.w_title:='NM';
  5: game.w_title:='WGM';
  6: game.w_title:='WIM';
  7: game.w_title:='WFM';
  else
    game.w_title:='';
  end;
  if (byte1 and 32)=32 then
    b_titlenumb:=getbits(game,4)
  else
    b_titlenumb:=0;
  case b_titlenumb of
  1: game.b_title:='IGM';
  2: game.b_title:='IM';
  3: game.b_title:='FM';
  4: game.b_title:='NM';
  5: game.b_title:='WGM';
  6: game.b_title:='WIM';
  7: game.b_title:='WFM';
  else
    game.b_title:='';
  end;
  if (byte1 and 64)=64 then
    game.place:=get_text(game)
  else
    game.place:='';
  if (byte1 and 128)=128 then
    game.annotator:=get_text(game)
  else
    game.annotator:='';
  if (byte2 and 1)=1 then
    game.source:=get_text(game)
  else
    game.source:='';
  if (byte2 and 2)=2 then
    game.info:=get_text(game)
  else
    game.info:='';
  if (byte2 and 16)=16 then
    game.year:=getbits(game,8)+1800
  else
    game.year:=0;
  if (byte2 and 8)=8 then
    resultnumb:=getbits(game,4)
  else
    resultnumb:=0;
  case resultnumb of
  1: game.result:='1-0';
  2: game.result:='1/2';
  3: game.result:='0-1';
  else
    game.result:=' L ';
  end;

  if (byte2 and 32)=32 then
    game.round:=getbits(game,7)
  else
    game.round:=0;

  if (byte2 and 64)=64 then
  begin
    game.nictools_used_to_convert:=true;
    temp1:=getbits(game,8);
    temp1:=getbits(game,8);
  end
  else
  begin
    game.nictools_used_to_convert:=false;
  end;

  game.numb_of_w_moves:=(game.numb_of_moves+1) div 2;
  game.cb_players:=game.w_name+' - '+game.b_name;
  game.cb_source:=game.place;
  str(game.round,numb2str);
  if game.round<>0 then
    game.cb_source:=game.cb_source+'('+numb2str+')';

  while (game.bitnumb mod 8)<>0 do inc(game.bitnumb);
  if game.bytes[game.bitnumb div 8]=0 then
    game.bitnumb:=game.bitnumb+8;
end;

procedure write_nic_gamehead(VAR game: gametype);
var
  byte0, byte1, byte2, byte3 : byte;
  w_title_numb, b_title_numb : byte;
  p, result_numb: byte;
  i: integer;

begin
  if not(game.deleted or game.position or game.quiz) and
  game.normal_game then
  begin
    game.bitnumb:=32;
    byte0:=$80;
    byte1:=0;
    byte2:=0;
    byte3:=game.numb_of_moves and 255;
    if game.numb_of_moves>127 then
      putbits(game,7,1);

    if length(game.w_name)>0 then
    begin
      byte1:=byte1 or 1;
      puttext(game,game.w_name);
    end;

    if length(game.b_name)>0 then
    begin
      byte1:=byte1 or 2;
      puttext(game,game.b_name);
    end;

    if game.w_elo>0 then
    begin
      byte1:=byte1 or 4;
      putbits(game,13,game.w_elo);
    end;

    if game.b_elo>0 then
    begin
      byte1:=byte1 or 8;
      putbits(game,13,game.b_elo);
    end;

    p:=pos(game.w_title,'IGM IM  FM  NM  WGM WIM WFM ');
    if (p>0) and (length(game.w_title)<>0) then
    begin
      w_title_numb:=((p-1) div 4)+1;
      byte1:=byte1 or 16;
      putbits(game,4,w_title_numb);
    end;

    p:=pos(game.b_title,'IGM IM  FM  NM  WGM WIM WFM ');
    if (p>0) and (length(game.b_title)<>0) then
    begin
      b_title_numb:=((p-1) div 4)+1;
      byte1:=byte1 or 32;
      putbits(game,4,b_title_numb);
    end;

    if (length(game.place)>0) then
    begin
      byte1:=byte1 or 64;
      puttext(game,game.place);
    end;

    if (length(game.annotator)>0) then
    begin
      byte1:=byte1 or 128;
      puttext(game,game.annotator);
    end;


    byte2:=0;
    if (length(game.source)>0) then
    begin
      byte2:=byte2 or 1;
      puttext(game,game.source);
    end;

    if (length(game.info)>0) then
    begin
      byte2:=byte2 or 2;
      puttext(game,game.info);
    end;

    if (game.year>1800) then
    begin
      byte2:=byte2 or 16;
      putbits(game,8,game.year-1800);
    end;

    p:=pos(game.result,'1-0 1/2 0-1  L ');
    if (length(game.result)<>0) and (p>0) then
    begin
      result_numb:=((p-1) div 4)+1;
      byte2:=byte2 or 8;
      putbits(game,4,result_numb);
    end;


    if (game.round>0) then
    begin
      byte2:=byte2 or 32;
      putbits(game,7,game.round);
    end;

    game.bytes[0]:=byte0;
    game.bytes[1]:=byte1;
    game.bytes[2]:=byte2;
    game.bytes[3]:=byte3;

    while (game.bitnumb mod 8)<>0 do
      put1bit(game,0);

    write_nic_moves(game);

    game.len:=(game.bitnumb div 8);
    if (game.bitnumb mod 8)>0 then
      inc(game.len);

  end
  else
  begin
    game.len:=0;
  end;
end;

function get_cb_text(VAR game: gametype; first_byte,
                     numb_of_bytes: byte):string;

var
  i : integer;
  last_byte : integer;
  s : string;

begin
  last_byte:=first_byte-1+numb_of_bytes;
  s:='';
  if numb_of_bytes>0 then
  begin
    i:=first_byte;
    while (i<=last_byte) do
    begin
      s:=s+chr(game.bytes[i]);
      inc(i);
    end;
  end;
  get_cb_text:=s;
end;



procedure xor_cb_gamehead(VAR game: gametype);
var
  k : longint;
  i : integer;

begin
  k:=101;
  for i:=13 downto 0 do
  begin
    game.bytes[i]:=game.bytes[i] xor (k and 255);
    k:=k*3;
  end;
end;

procedure xor_text(VAR game: gametype; first_byte: integer;
                   total_text_len: byte);
var
  i, k : longint;
begin
  k:=3*total_text_len;
  for i:=first_byte+total_text_len-1 downto first_byte do
  begin
    game.bytes[i]:=game.bytes[i] xor (k and 255);
    k:=k*3;
  end;
end;

procedure xor_moves(VAR game: gametype; first_byte: integer;
                    total_move_len: integer);

var
  i, k : longint;

begin
  k:=49*total_move_len;
  for i:=first_byte+total_move_len-2 downto first_byte+1 do
  begin
    game.bytes[i]:=game.bytes[i] xor (k and 255);
    k:=k*7;
  end;
end;


procedure parse_cb_gamehead(VAR game: gametype);
var
  i, p, p1, p2 : integer;
  numb_of_games, game_numb, temp : longint;
  tempstr : string;
  round: integer;
  first_move_byte, first_text_byte: byte;

begin
  init_game(game);
  xor_cb_gamehead(game);
  p:=game.bytes[0];
  if p=127 then
    game.year:=0
  else
  begin
    if p>=128 then
      p:=p-256;
    game.year:=1900+p;
  end;
  case game.bytes[1] of
  2: game.result:='1-0';
  0: game.result:='0-1';
  1: game.result:='1/2';
  else
    game.result:=' L ';
  end;

  game.cb_bytes_on_moves:=game.bytes[2]*256+game.bytes[3];
  game.numb_of_moves:=0;
  game.cb_players_len:=game.bytes[4] and 63;
  game.cb_source_len:=game.bytes[5] and 63;
  game.w_elo:=1600+game.bytes[8]*5;
  if game.w_elo=1600 then
    game.w_elo:=0;
  game.b_elo:=1600+game.bytes[9]*5;
  if game.b_elo=1600 then
    game.b_elo:=0;

  game.position:=(game.bytes[10] and 1)=1;
  game.alternatives:=(game.numb_of_w_moves)=((game.numb_of_moves+1) div 2);

  first_move_byte:=game.bytes[11];

  game.numb_of_w_moves:=game.bytes[12];
  if not game.position then
  begin
    first_text_byte:=14;
    xor_text(game,first_text_byte,game.cb_source_len+game.cb_players_len);
    tempstr:=get_cb_text(game,14,game.cb_players_len+game.cb_source_len);

    game.cb_players:=copy(tempstr,1,game.cb_players_len);
    game.cb_source:=copy(tempstr,game.cb_players_len+1,game.cb_source_len);

    p1:=pos('(',game.cb_players);
    p2:=pos(')',game.cb_players);
    if (p2>p1) and (p1>0) then
    begin
      tempstr:=copy(game.cb_players,p1+1,p2-p1-1);
      val(tempstr,round,p);
      if p=0 then
        delete(game.cb_players,p1,p2-p1+1);
    end;
    p1:=pos('(',game.cb_source);
    p2:=pos(')',game.cb_source);
    if (p2>p1) and (p1>0) and (p<>0) then
    begin
      tempstr:=copy(game.cb_source,p1+1,p2-p1-1);
      val(tempstr,round,p);
      if p=0 then
        delete(game.cb_source,p1,p2-p1+1);
    end;
    if (p=0) and (round<=255) then
     game.round:=round
    else
     game.round:=0;

    tempstr:=game.cb_players;
    p:=pos('-',tempstr);
    if p=0 then
      p:=pos(',',tempstr);
    if p=0 then
    begin
      if length(tempstr)<=25 then
      begin
        game.w_name:=tempstr;
        game.b_name:='';
      end
      else
      begin
        p:=length(tempstr) div 2;
        game.w_name:=copy(tempstr,1,p);
        game.b_name:=copy(tempstr,p+1, length(tempstr)-p);
      end;
    end
    else
    begin
      game.w_name:=copy(tempstr,1,p-1);
      game.b_name:=copy(tempstr,p+1, length(tempstr)-p);

    end;

    remove_spaces_in_name(game.w_name);
    remove_spaces_in_name(game.b_name);
    game.place:=game.cb_source;

    first_move_byte:=first_text_byte+game.cb_source_len+game.cb_players_len;
    xor_moves(game,first_move_byte,game.cb_bytes_on_moves);
    parse_cb_moves(game,first_move_byte);
  end
  else
  begin
    game.skip:=true;
  end;

end;

procedure write_cb_gamehead(VAR game: gametype);
var
  p,i : integer;
  byte_numb : integer;
  tempstr : string;

begin
  byte_numb:=0;
  p:=game.year-1900;
  if p<=-126 then
    p:=127
  else
  if p<0 then
    p:=p+256;
  game.bytes[0]:=p;
  inc(byte_numb);

  if game.result='1-0' then
    game.bytes[1]:=2
  else
  if game.result='0-1' then
    game.bytes[1]:=0
  else
  if pos('1/2',game.result)>0 then
    game.bytes[1]:=1
  else
    game.bytes[1]:=3;
  inc(byte_numb);

  game.bytes[2]:=(game.numb_of_moves +1) div 256;
  inc(byte_numb);

  game.bytes[3]:=((game.numb_of_moves + 1) mod 256);
  inc(byte_numb);

  game.cb_bytes_on_moves:=game.numb_of_moves +1;

  game.bytes[4]:=(length(game.cb_players));
  inc(byte_numb);

  game.bytes[5]:=(length(game.cb_source));
  inc(byte_numb);

  game.cb_players_len:=length(game.cb_players);
  game.cb_source_len:=length(game.cb_source);

  game.bytes[6]:=0;
  inc(byte_numb);

  game.bytes[7]:=0;
  inc(byte_numb);

  if (game.w_elo<1605) then
    game.bytes[8]:=0
  else
    game.bytes[8]:=((game.w_elo - 1600) div 5);
  inc(byte_numb);

  if (game.b_elo<1605) then
    game.bytes[9]:=0
  else
    game.bytes[9]:=((game.b_elo - 1600) div 5);
  inc(byte_numb);

  game.bytes[10]:=0;
  inc(byte_numb);

  game.bytes[11]:=14+length(game.cb_source)+length(game.cb_players);
  inc(byte_numb);


  game.bytes[12]:=(game.numb_of_w_moves);
  inc(byte_numb);

  game.bytes[13]:=(game.bytes[0]*$25+game.bytes[5]+game.bytes[9]) and 255;
  inc(byte_numb);


  tempstr:=game.cb_players+game.cb_source;
  p:=length(tempstr);
  for i:=1 to p do
  begin
    game.bytes[byte_numb]:=ord(tempstr[i]);
    inc(byte_numb);
  end;

  write_cb_moves(game,byte_numb);
  xor_cb_gamehead(game);
  xor_text(game,14,game.cb_players_len+game.cb_source_len);
  xor_moves(game,14+game.cb_players_len+game.cb_source_len,
            game.cb_bytes_on_moves);


end;

procedure init_nic_piece_list(VAR board: boardtype;
                              VAR piece_list:nic_piece_listtype;
                              VAR numb_of_pieces: numb_of_piecestype);


var
  c1, c2, p : shortint;
  square_rec : squaretype;

begin
  for p:=1 to 6 do
  begin
    piece_list[FALSE,p]:='';
    piece_list[TRUE,p]:='';
  end;
  numb_of_pieces[FALSE]:=0;
  numb_of_pieces[TRUE]:=0;
  for c1:=1 to 8 do
  begin
    for c2:=1 to 8 do
    begin
      square_rec:=board.square[c1,c2];
      p:=pos(square_rec.rank,'QNRBKP.');
      if (p>0) and (p<7) then
      begin
        if (square_rec.color='W') then
        begin
          piece_list[TRUE,p]:=piece_list[TRUE,p]+chr(96+c1)+chr(48+c2);
          inc(numb_of_pieces[TRUE]);
        end
        else
        begin
          piece_list[FALSE,p]:=piece_list[FALSE,p]+chr(96+c1)+chr(48+c2);
          inc(numb_of_pieces[FALSE]);
        end;
      end;
    end;
  end;
end;


procedure parse_nic_moves(VAR game:gametype);


var
  board: boardtype;
  numb_of_pieces: numb_of_piecestype;
  w_moves : boolean;
  piece_list: nic_piece_listtype;


  function find_choosen_piece(piece_numb: byte) : string;
  var
    p, c1 : byte;
    choosen_piece, tempstr : string;

  begin
    p:=1;
    c1:=0;
    while (c1<(piece_numb+1)) do
    begin
      tempstr:=piece_list[w_moves,p];
      while (length(tempstr)>0) and (c1<(piece_numb+1)) do
      begin
        choosen_piece:=piece_token[p]+copy(tempstr,1,2);
        tempstr:=copy(tempstr,3,length(tempstr)-2);
        inc(c1);
      end;
      inc(p);
    end;
    find_choosen_piece:=choosen_piece;
  end;



var


  piece_numb, i,j,k,p,c1,c2, temp1 : integer;
  move_numb : integer;
  choosen_move, numb_of_legal_moves : byte;
  bits_needed : integer;
  tempstr1, moves_string, piece : string;
  square : string[2];
  legal_moves: array[1..28] of string[3];
  stop, flag1, flag2 : boolean;

begin
  init_board(board);
  game.move[0]:='';
  i:=1;
  stop:=false;
  repeat
    w_moves:=board.white_to_move;
    init_nic_piece_list(board,piece_list,numb_of_pieces);
    if game.alternatives or game.comment then
      temp1:=get1bit(game);
    bits_needed:=0;
    while bitval[bits_needed]<numb_of_pieces[w_moves] do
      inc(bits_needed);
    piece_numb:=getbits(game,bits_needed);
    if (numb_of_pieces[w_moves]+piece_numb<bitval[bits_needed]) or
       (piece_numb>=numb_of_pieces[w_moves]) then
    begin
      flag1:=true;
      if piece_numb>=numb_of_pieces[w_moves] then
      begin
        piece_numb:=piece_numb-numb_of_pieces[w_moves];
        flag2:=true;
      end
      else
        flag2:=false;
    end
    else
      flag1:=false;
    piece:=find_choosen_piece(piece_numb);
    square:=copy(piece,2,2);
    moves_string:=nic_gen_moves(board,square);
    if length(moves_string)>0 then
    begin
      j:=5; k:=1;
      p:= pos(' ',copy(moves_string,j,length(moves_string)-j+1));
      while (p>0) and (j<length(moves_string)) do
      begin
        legal_moves[k]:=copy(moves_string,j,p-1);
        j:=j+p;
        inc(k);
        p:= pos(' ',copy(moves_string,j,length(moves_string)-j+1));
      end;
      legal_moves[k]:='';
      numb_of_legal_moves:=k-1;
      bits_needed:=0;
      while bitval[bits_needed]<numb_of_legal_moves do
        inc(bits_needed);
      if bits_needed=0 then
        choosen_move:=0
      else
      begin
        if not flag1 then
          choosen_move:=getbits(game,bits_needed)
        else
        begin
          if flag2 then
            choosen_move:=getbits(game,bits_needed-1)+bitval[bits_needed-1]
          else
            choosen_move:=getbits(game,bits_needed-1);
        end;
      end;
      choosen_move:=choosen_move+1;
      if choosen_move<=numb_of_legal_moves then
      begin
        j:=ord(upcase(legal_moves[choosen_move][1]))-64;
        k:=ord(legal_moves[choosen_move][2])-48;
        if board.square[j,k].rank='.' then
          game.move[i]:=piece+'-'+legal_moves[choosen_move]
        else
          game.move[i]:=piece+'x'+legal_moves[choosen_move];
        if game.move[i][1]='P' then
          game.move[i]:=copy(game.move[i],2,length(game.move[i])-1);
        game.move[i+1]:='';
        if (game.move[i][1]='K') and (game.move[i][2]='e') and
           (game.move[i][5]='g') then
        begin
          game.move[i]:='O-O';
        end;
        if (game.move[i][1]='K') and (game.move[i][2]='e') and
           (game.move[i][5]='c') then
        begin
          game.move[i]:='O-O-0';
        end;
        do_move(board,game.move[i]);
        inc(i);
      end
      else
        stop:=true;
    end
    else
    begin
      stop:=true;
    end;
  until (i>game.numb_of_moves) or stop;
  if stop then
  begin
    numb_of_legal_moves:=0;
    game.numb_of_moves:=i-1;
  end;
  game.illegal_move:=stop or board.illegal_move;
end;

procedure write_cb_moves(VAR game: gametype; byte_numb: word);
var
  board : boardtype;
  p, i: integer;

begin
  init_board(board);
  p:=game.numb_of_moves;
  i:=1;
  while (i<=p) do
  begin
    game.bytes[byte_numb]:=move2cb(board, game.move[i]);
    inc(byte_numb);
    inc(i);
  end;
  game.len:=byte_numb;

end;

procedure write_nic_moves(VAR game: gametype);
var
  numb_of_pieces : numb_of_piecestype;

  procedure write_nic_move(VAR game: gametype; VAR board: boardtype;
                           move_numb:integer);
  var
    i,j,p,k : integer;
    promotion: string[1];
    numb_of_legal_moves, choosen_move_numb : byte;
    move: movetype;
    w_mov : boolean;
    piece_moving, moving_from, moving_to, found_piece: string;
    moves_string, tempstr: string;
    piece_list: nic_piece_listtype;
    bits_needed1, bits_needed2, piece_numb : byte;
    flag1, flag2 : boolean;

  begin
    init_nic_piece_list(board,piece_list,numb_of_pieces);
    w_mov:=(move_numb mod 2)=1;
    move:=game.move[move_numb];
    parse_move_string(board, move);
    if pos(move[1],'QNRBKP')>0 then
    begin
      piece_moving:=copy(move,1,3);
      promotion:=copy(move+' ',6,1);
      moving_to:=copy(move,4,2);
    end
    else
    begin
      writeln('Error in write_nic_move');
      halt;
    end;

    moving_from:=copy(piece_moving,2,2);

    i:=0;
    j:=1;
    found_piece:='';
    while (j<=6) and (i<=numb_of_pieces[w_mov]) and
          (found_piece<>piece_moving) do
    begin
      tempstr:=piece_list[w_mov,j];
      found_piece:=piece_token[j]+copy(tempstr,1,2);
      while (i<=numb_of_pieces[w_mov]) and (tempstr<>'')
      and (found_piece<>piece_moving) do
      begin
        tempstr:=copy(tempstr,3,length(tempstr)-2);
        found_piece:=piece_token[j]+copy(tempstr,1,2);
        inc(i);
      end;
      inc(j);
    end;
    if (i<=numb_of_pieces[w_mov]) then
    begin
      piece_numb:=i;
    end
    else
    begin
      writeln('Error in write_nic_move!');
      halt;
    end;


    moves_string:=nic_gen_moves(board,moving_from);
    tempstr:=copy(moves_string,5,length(moves_string)-4);
    k:=0;
    p:=pos(' ',tempstr);
    while (p>0) and (length(tempstr)>0) do
    begin
      tempstr:=copy(tempstr,p+1,length(tempstr)-p);
      p:=pos(' ',tempstr);
      inc(k);
    end;
    numb_of_legal_moves:=k;

    bits_needed2:=0;
    while bitval[bits_needed2]<numb_of_legal_moves do
      inc(bits_needed2);


    tempstr:=copy(moves_string,5,length(moves_string)-4);
    k:=0;
    p:=pos(' ',tempstr);
    while (p>0) and (length(tempstr)>0) and
          (copy(tempstr,1,3)<>moving_to+promotion) do
    begin
      tempstr:=copy(tempstr,p+1,length(tempstr)-p);
      p:=pos(' ',tempstr);
      inc(k);
    end;
    if k<numb_of_legal_moves then
    begin
      choosen_move_numb:=k;
    end
    else
    begin
      writeln('Error in write_nic_move!!!');
      halt;
    end;

    bits_needed1:=0;
    while bitval[bits_needed1]<numb_of_pieces[w_mov] do
      inc(bits_needed1);

    if (numb_of_pieces[w_mov]+piece_numb<bitval[bits_needed1]) then
    begin
      flag1:=true;
    end
    else
    begin
      flag1:=false;
    end;


    if flag1 then
    begin
      if bits_needed2>0 then
        dec(bits_needed2);
      if choosen_move_numb>=bitval[bits_needed2] then
      begin
        piece_numb:=piece_numb+numb_of_pieces[w_mov];
      end;
    end;


    putbits(game,bits_needed1,piece_numb);

    putbits(game,bits_needed2,choosen_move_numb);
  end;

var
  i : integer;
  board: boardtype;

begin
  init_board(board);

  for i:=1 to game.numb_of_moves do
  begin
    write_nic_move(game,board,i);
    do_move(board,game.move[i]);

  end;
end;

function parse_cb_move(VAR game: gametype; VAR board:boardtype;
                       move_numb: byte): string;
var
  i, j, k, p: byte;
  s, tempstr, move, target_square : string;


begin
  k:=0;
  i:=1;
  j:=1;
  game.illegal_move:=false;
  while (k<move_numb) and (not game.illegal_move) do
  begin
    s:=cb_gen_moves(board,chr(i+96)+chr(48+j));
    tempstr:=s;
    s:=copy(s,5,length(s)-4);
    p:=pos(' ',s);
    while (p>0) and (k<move_numb) do
    begin
      move:=board.square[i,j].rank+chr(i+96)+chr(48+j);
      target_square:=copy(s,1,p-1);
      move:=move+target_square;
      inc(k);
      s:=copy(s,p+1,length(s)-p);
      p:=pos(' ',s);
    end;
    inc(j);
    if j>8 then
    begin
      j:=1;
      inc(i);
    end;
    if (k<move_numb) and ((i>8) or (j>8)) then
    begin
      game.illegal_move:=true;
    end;
  end;
  if not game.illegal_move then
  begin
    move2pgn(board,move);
    do_move(board,move);
  end;
  parse_cb_move:=move;
end;

procedure parse_cb_moves(VAR game: gametype; first_byte: word);


var
  i, move_numb, choosen_move : word;
  tempstr : string;
  board : boardtype;
  line_numb : byte;
  end_of_game : boolean;


begin
  init_board(board);
  game.move[0]:='';
  i:=first_byte;
  move_numb:=0;
  line_numb:=0;
  end_of_game:=false;
  while (i<=first_byte+game.cb_bytes_on_moves-2) and
        (not game.illegal_move) and (not end_of_game) do
  begin
    end_of_game:=end_of_game or (game.bytes[i]<0);
    if not end_of_game then
    begin
      choosen_move:=game.bytes[i] and 127;
      if choosen_move=127 then
      begin
        inc(line_numb);
        repeat
          inc(i);
          choosen_move:=game.bytes[i];
          if choosen_move=255 then
            inc(line_numb);
          if choosen_move=128 then
            dec(line_numb);
          end_of_game:=end_of_game or (game.bytes[i+1]<0);
        until (line_numb=0) or (i>first_byte+game.cb_bytes_on_moves-1) or
              (end_of_game);
        inc(i);
      end
      else
      if choosen_move=0 then
      begin
        end_of_game:=true;
      end
      else
      begin
        tempstr:=parse_cb_move(game,board,choosen_move);
        if not game.illegal_move then
        begin
          inc(move_numb);
          game.move[move_numb]:=tempstr;
        end
        else
        begin
          game.numb_of_moves:=move_numb;
        end;
        inc(i);
      end;
    end;
  end;
  game.move[move_numb+1]:='';
  game.numb_of_moves:=move_numb;
end;



function move2cb(VAR board: boardtype; move: movetype):byte;
var
  i,j, k, p : byte;
  s, tempstr, squares, target_square, old_move : string;
  x1, y1, x2, y2 : integer;
  stop : boolean;
  move_numb : byte;

begin
  parse_move_string(board,move);
  i:=1;
  j:=1;
  k:=0;
  old_move:=move;
  squares:=copy(move,2,4);
  x1:=ord(upcase(move[2]))-64;
  y1:=ord(upcase(move[3]))-48;
  x2:=ord(upcase(move[4]))-64;
  y2:=ord(upcase(move[5]))-48;
  if legal_move(board,x1,y1,x2,y2) then
  begin
    stop:=false;
    while (not stop) do
    begin
      s:=cb_gen_moves(board,chr(i+96)+chr(48+j));
      tempstr:=s;
      s:=copy(s,5,length(s)-4);
      p:=pos(' ',s);
      stop:=false;
      move:='';
      while (p>0) and (not stop) do
      begin
        move:=chr(i+96)+chr(48+j);
        target_square:=copy(s,1,2);
        move:=move+target_square;
        inc(k);
        s:=copy(s,p+1,length(s)-p);
        p:=pos(' ',s);
        stop:=(move=squares) or ((i>8) or (j>8));
      end;
      inc(j);
      if j>8 then
      begin
        j:=1;
        inc(i);
      end;
      if (move<>squares) and ((i>8) or (j>8)) then
      begin
        writeln('Error in do_cb_move!!!');
       { halt; }
      end;
    end;
    move:=old_move;
    p:=pos(move[length(move)],'QRBN');
    if (board.square[x1,y1].rank='P') and (p>0) then
    begin
      k:=k+p-1;
    end;
    do_move(board,old_move);
    move_numb:=k;
  end
  else
  begin
    move_numb:=0;
    writeln(' Error in move2cb!!');
    board.illegal_move:=true;
  end;
  move2cb:=move_numb;
end;

function long2short(VAR board: boardtype; in_move: string): string;
var
  tempstr, new_move : string[10];
  piece: char;
  i, j, numb_of_same_piece : integer;
  move_color : char;
  piece_list : array[1..10] of string[2];
  source_square, target_square: string[2];
  current_square : string;
  numb_on_right_file : integer;

begin
  if pos('O-O',in_move)=0 then
  begin
    if pos(in_move[1],'NBRQK')>0 then
    begin
      piece:=in_move[1];
      source_square:=copy(in_move,2,2);
      target_square:=copy(in_move,5,2);
      case piece of
      'K':
      begin
        if in_move[4]='-' then
          tempstr:=piece+target_square
        else
          tempstr:=piece+'x'+target_square;
      end;

      'B','R','Q','N':
      begin
        numb_of_same_piece:=0;
        if board.white_to_move then
          move_color:='W'
        else
          move_color:='B';

        for i:=1 to 8 do
        begin
          for j:=1 to 8 do
          begin
            if (board.square[i,j].color=move_color) and
               (board.square[i,j].rank=piece) then
            begin
              current_square:=chr(96+i)+chr(48+j);
              if (pos(target_square,nic_gen_moves(board,current_square))>0) then
              begin
                new_move:=piece+current_square+target_square;
                { if not in_check_after(board,new_move) then }
                begin
                  inc(numb_of_same_piece);
                  piece_list[numb_of_same_piece]:=current_square;
                end;
              end;
            end;
          end;
        end;

        if numb_of_same_piece=1 then
        begin
          if in_move[4]='-' then
            tempstr:=piece+copy(in_move,5,2)
          else
            tempstr:=piece+'x'+copy(in_move,5,2);
        end
        else
        begin
          numb_on_right_file:=0;
          for i:=1 to numb_of_same_piece do
          begin
            if piece_list[i][1]=source_square[1] then
              inc(numb_on_right_file);
          end;
          if numb_on_right_file<2 then
            tempstr:=piece+source_square[1]
          else
            tempstr:=piece+source_square[2];

          if in_move[4]='-' then
            tempstr:=tempstr+copy(in_move,5,2)
          else
            tempstr:=tempstr+'x'+copy(in_move,5,2);
        end;
      end;
      else
      end;
    end
    else
    begin
      if in_move[3]='x' then
        tempstr:=in_move[1]+'x'+copy(in_move,4,length(in_move)-3)
      else
        tempstr:=copy(in_move,4,length(in_move)-3);
    end;
  end
  else
    tempstr:=in_move;
  if giving_check_after(board,in_move) then
      tempstr:=tempstr+'+';
  do_move(board,in_move);
  long2short:=tempstr;
end;

procedure print_moves(VAR f1: text; VAR game: gametype);
const
  max_line_len = 78;
var
  i : integer;
  linestr, tempstr, tempstr2, one_move : string;
  board : boardtype;

begin
  linestr:=''; tempstr:='';
  writeln(f1);
  if game.numb_of_moves>0 then
  begin
    init_board(board);
    for i:=1 to game.numb_of_moves do
    begin
      tempstr:='';

      if (pos(game.move[i][length(game.move[i])],'QRBN')>0) then
      begin
        one_move:=game.move[i];
        insert('=',one_move,length(one_move));
        tempstr:=tempstr+one_move;
      end
      else
        tempstr:=tempstr+game.move[i];

      if use_short_notation then
        tempstr:=long2short(board,tempstr);

      tempstr:=tempstr+' ';
      if (i mod 2)=1 then
      begin
        str(i div 2 +1, tempstr2);
        tempstr:=tempstr2+'. '+ tempstr;
      end;
      if ((length(tempstr)+length(linestr))>max_line_len) then
      begin
        writeln(f1,linestr);
        linestr:=tempstr;
      end
      else
        linestr:=linestr+tempstr;
    end;
    writeln(f1,linestr);
  end;
  writeln(f1);
  if (game.result<>'') and (pos('L',game.result)=0) then
  begin
    if pos('1/2',game.result)=0 then
      writeln(f1,'':29,game.result)
    else
      writeln(f1,'':29,'1/2-1/2');
  end
  else
  begin
    writeln(f1,'':29,'*');
  end;
end;

procedure print_pgn(VAR f1: text; VAR game: gametype);

begin
  writeln(f1);

  writeln(f1,  '[Event       "?"]');
  if game.place<>'' then
    writeln(f1,'[Site        "',game.place,'"]');
  if game.year>0 then
    writeln(f1,'[Date        "',game.year,'.??.??"]')
  else
    writeln(f1,'[Date        "????.??.??"]');
  if game.round>0 then
    writeln(f1,'[Round       "',game.round,'"]')
  else
    writeln(f1,'[Round       "??"]');
  if game.w_name<>'' then
    writeln(f1,'[White       "',game.w_name,'"]')
  else
    writeln(f1,'[White       "????????????????"]');
  if game.b_name<>'' then
    writeln(f1,'[Black       "',game.b_name,'"]')
  else
    writeln(f1,'[Black       "????????????????"]');
  if game.result<>'' then
  begin
    write(f1,  '[Result      "');
    if pos('1/2',game.result)>0 then
      writeln(f1,'1/2-1/2"]')
    else
    if pos('1-0',game.result)>0 then
      writeln(f1,'1-0"]')
    else
    if pos('0-1',game.result)>0 then
      writeln(f1,'0-1"]')
    else
      writeln(f1,'*"]');
  end
  else
    writeln(f1,'[Result      "???"]');

  if game.w_title<>'' then
    writeln(f1,'[WhiteTitle  "',game.w_title,'"]');

  if game.b_title<>'' then
    writeln(f1,'[BlackTitle  "',game.b_title,'"]');

  if game.w_elo<>0 then
    writeln(f1,'[WhiteElo    "',game.w_elo,'"]');

  if game.b_elo<>0 then
    writeln(f1,'[BlackElo    "',game.b_elo,'"]');

  if game.nic_opening<>'' then
    writeln(f1,'[NIC         "',game.nic_opening,'"]');

  print_moves(f1, game);
end;

begin
end.

