/*
 *	This file contains the PostScript image-paste functionality.
 */

#include "types.h"
#include "devtab.h"
#include "tracedef.h"

#include "raster.p"
#include "strng.p"
#include "fileio.p"
#include "fio.p"

pointer *PSC_File;

/*
 *	The parameters to PS_Paste specify the output device descriptor,
 *	the location of the upper left corner of the output box, the
 *	width, depth and orientation of the output box, the relative
 *	picture number and the file name containing the graphical input
 *	data. All dimensions are in 1/1024 pixels.
 *
 *	The 'orientation' option allows the picture to be pasted into
 *	the box in any of four orientations, corresponding to rotations
 *	of multiples of 90 degrees. Additionally, if X_Scale is negative,
 *	it indicates a mirror-image of the picture is to be produced.
 *
 *	A problem may exist here if the output device has extremely high
 *	resolution. The width and/or the height of the bitmap image may
 *	exceed 65535, the limit that can be stored in an unsigned short
 *	value.
 */
int PS_Paste (Dev_Ptr, File, X0, Y0, X_Scale, Y_Scale, Orientation, Picture,
	      Optv, Optc)
struct Device_Table *Dev_Ptr;
char *File;
long X0, Y0;
long X_Scale, Y_Scale;
unsigned int Orientation, Picture;
char *Optv[];
int Optc;
{
	msgcode DVIOUT_INOPENFAIL;
	char c;
	char Scratch_String[256];
/*
 *	Open the graphics file for input:
 */
	if (trace_paste)
		printf ("PostScript file name: %s\n", File);
	if ((PSC_File = Open_Gen_File_M (File, ".PS", "r", "\n")) == 0) {
		Message (DVIOUT_INOPENFAIL, File, 1);
		return (0);
	}
/*
 *	Save graphics state, set up picture size in some variables and
 *      translate coordinate system
 */
	(*Dev_Ptr->Output_String) ("\nsave ");
	sprintf (Scratch_String,
                 "/XWidth %ld 1024 div sc@le mul def /YWidth %ld 1024 div sc@le mul def ", abs (X_Scale), Y_Scale);
	(*Dev_Ptr->Output_String) (Scratch_String);
	switch (Orientation + ((X_Scale < 0) ? 1 : 0)) {
		case 0:				/* UP */
		case 4: Y0 += Y_Scale;
			break;
		case 1: X0 += abs (X_Scale);    /* LEFT */
			Y0 += Y_Scale;
			break;
		case 2: X0 += abs (X_Scale);	/* DOWN */
			break;
		case 3:				/* RIGHT */
			break;
	}
	sprintf (Scratch_String, "%ld 1024 div %ld 1024 div translate", X0, Y0);
	(*Dev_Ptr->Output_String) (Scratch_String);
	sprintf (Scratch_String, " %ld rotate", -90 * Orientation);
	(*Dev_Ptr->Output_String) (Scratch_String);
	if (X_Scale < 0) {
		sprintf (Scratch_String, " -1 1 scale");
		(*Dev_Ptr->Output_String) (Scratch_String);
	}
	(*Dev_Ptr->Output_String) (" r@ster 72 div ");
	if (X_Scale < 0) (*Dev_Ptr->Output_String) ("neg dup");
	else (*Dev_Ptr->Output_String) ("dup neg");
	(*Dev_Ptr->Output_String) (" scale\n");
/*
 *	Read in each row of the image and place it into the unpacked
 *	image buffer:
 */
	while ((c = Read_Gen_Character_M (PSC_File)) != 0) {
		(*Dev_Ptr->Output_Character) (c);
	}
/*
 *	Restore graphics state
 */
	(*Dev_Ptr->Output_String) ("\nrestore\n");
/*
 *	Finish up:
 */
	Close_Gen_File_M (PSC_File);
}
