/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This is the PostScript low-level driver.
 */

#include "types.h"
#include "drivutil.h"
#include "font.h"

#include "fio.p"
#include "raster.p"
#include "strng.p"

#define OBJECT_OVERHEAD     12
#define DICT_ENTRY_OVERHEAD 16
#define DICT_OVERHEAD       36
#define MAX_TEMP_FONTS      25

char Scratch_String[256];
struct Font_Definition *PS_Current_Font; /* Current active font */
pointer PS_File;		/* Output file descriptor pointer */
long PS_VMSize;			/* Amount of remaining memory */
long PS_Saved_VMSize;		/* Size for restoring between save/restore */
unsigned long PS_Fonts_Available; /* Number of fonts that can yet be downloaded */
unsigned long PS_Temp_Fonts_Available; /* Number of temporary fonts that can yet be downloaded */
long PS_GX, PS_GY;		/* Graphics current position ('X@','Y@') */
long PS_Y;			/* Current Y coordinate */
unsigned long PS_Pageno;	/* Page number */

#define Out_Character(c) Write_Character_M (c, PS_File)
#define Out_String(str) Write_Block_M (str, sizeof(str)-1, PS_File)
#define Out_Scratch Write_Block_M (Scratch_String, strlen (Scratch_String), PS_File)
#define Round_Pixel(size) (((size) + ((size) >= 0 ? 512 : -512)) >> 10)
#define Trunc_Pixel(size) ((size) >> 10)
#define Ceil_Pixel(size) (((size) + ((size) >= 0 ? 1023 : -1023)) >> 10)
extern int strlen();

/*
 *	The following encoding vector maps "tex text" encoding to
 *	"postscript text" encoding.
 *
 *	A code value greater than '377 calls for a character in the Symbol
 *	font. This is how Greek letters are done. The Greek letters can
 *	only satisfactorily be accomodated for the "Times-Roman" font, since
 *	there are no "Symbol-Bold" or "Symbol-Italic" fonts. Therefore the
 *	Greek characters should not be defined in the Times-Italic,
 *	Times-Bold, etc. font TFM files. If for some reason they are defined,
 *	the Romanish-style Greek characters from the Symbol font will appear
 *	instead.
 *
 *	The following differences between pure "tex text" encoding
 *	and the encoding expected here (defined by the 'querylw' program)
 *	are:
 *
 *	The dotless-j, ff ligature, ffi ligature and ffl ligature are not
 *	supported, and should not to be defined within the TFM file for the
 *	font. Also, the slash for the suppressed-L characters cannot be
 *	generated by itself (who would want to, anyway?). The suppressed-L
 *	characters are done by putting these two characters in font positions
 *	'16 and '17 (replacing ffi and ffl) and defining these characters as
 *	ligatures instead of kerns in the TFM file.
 */

#define UNDF 0040
unsigned long TeX_to_PS_Text_Vector[128] = {
	0507, 0504, 0521, 0514, 0530, 0520, 0523, 0641,
	0506, 0531, 0527, UNDF, 0256, 0257, 0370, 0350,
	0365, UNDF, 0301, 0302, 0317, 0306, 0305, 0312,
	0313, 0373, 0361, 0372, 0371, 0341, 0352, 0351,
	UNDF, 0041, 0272, 0043, 0044, 0045, 0046, 0047,
	0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
	0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
	0070, 0071, 0072, 0073, 0241, 0075, 0277, 0077,
	0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
	0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
	0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,
	0130, 0131, 0132, 0133, 0252, 0135, 0303, 0307,
	0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
	0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
	0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
	0170, 0171, 0172, 0261, 0320, 0315, 0304, 0310
};
struct Encoding TeX_to_PS_Text_Encoding = {
	&TeX_to_PS_Text_Vector[0],
	arraysize (TeX_to_PS_Text_Vector),
	UNDF
};

/*
 *	The following encoding definition maps "tex typewriter text"
 *	to "postscript text". There are no appropriate Greek letters
 *	available, but few occasions call for the upper case Greek
 *	letters in typewriter style. The dotless-j is still missing.
 *	There is no acceptable substitute for the square cup "space"
 *	character in the tex typewriter fonts, so just a blank is used.
 *	The up arrow and down arrow are obtained from the Symbol font
 *	(the widths of these two characters are very close to the
 *	fixed width of the Courier fonts).
 *
 *	It is not certain which choice for circumflex and tilde
 *	should be used. The choice is between "asciicircum" ('136)
 *	vs. "circumflex" ('303) and "asciitilde" ('176) vs.
 *	"tilde" ('304). I guess it depends on whether the characters
 *	will be used for accents or not. The Tex versions of these
 *	characters look like they are intended as accents, not ASCII
 *	characters, so that is what is used.
 *
 *	This encoding is intended to be used with the Courier fonts.
 */

unsigned long TeX_TT_to_PS_Text_Vector[128] = {
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, 0655, 0657, 0251, 0241, 0277,
	0365, UNDF, 0301, 0302, 0317, 0306, 0305, 0312,
	0313, 0373, 0361, 0372, 0371, 0341, 0352, 0351,
	0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,
	0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
	0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
	0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
	0100, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
	0110, 0111, 0112, 0113, 0114, 0115, 0116, 0117,
	0120, 0121, 0122, 0123, 0124, 0125, 0126, 0127,
	0130, 0131, 0132, 0133, 0134, 0135, 0303, 0137,
	0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
	0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
	0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
	0170, 0171, 0172, 0173, 0174, 0175, 0304, 0310
};
struct Encoding TeX_TT_to_PS_Text_Encoding = {
	&TeX_TT_to_PS_Text_Vector[0],
	arraysize (TeX_TT_to_PS_Text_Vector),
	UNDF
};

/*
 *	The following encoding definition maps "ebcdic" encoding
 *	"postscript text". Note that the logical not character,
 *	the logical or and the long vertical bar are obtained from
 *	the Symbol font.
 */

unsigned long EBCDIC_to_PS_Text_Vector[256] = {
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	0040, 0040, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, 0242, 0056, 0074, 0050, 0053, 0732,
	0046, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, 0041, 0044, 0052, 0051, 0073, 0730,
	0055, 0057, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, 0174, 0054, 0045, 0137, 0076, 0077,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, 0301, 0072, 0043, 0100, 0251, 0075, 0042,
	UNDF, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
	0150, 0151, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, 0152, 0153, 0154, 0155, 0156, 0157, 0160,
	0161, 0162, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, 0176, 0163, 0164, 0165, 0166, 0167, 0170,
	0171, 0172, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	0173, 0101, 0102, 0103, 0104, 0105, 0106, 0107,
	0110, 0111, 0261, UNDF, UNDF, UNDF, UNDF, UNDF,
	0175, 0112, 0113, 0114, 0115, 0116, 0117, 0120,
	0121, 0122, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	0134, 0040, 0123, 0124, 0125, 0126, 0127, 0130,
	0131, 0132, UNDF, UNDF, UNDF, UNDF, UNDF, UNDF,
	0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
	0070, 0071, 0675, UNDF, UNDF, UNDF, UNDF, UNDF
};
struct Encoding EBCDIC_to_PS_Text_Encoding = {
	&EBCDIC_to_PS_Text_Vector[0],
	arraysize (EBCDIC_to_PS_Text_Vector),
	UNDF
};

/*
 *	PostScript procedures description section. Procedures used
 *	by DVIOUT all have '@' symbols in the names, so as (hopefully)
 *	not to conflict with any PostScript built-in name. I have
 *	described the PostScript procedures in a pseudo language, since
 *	PostScript itself is intrinsically unreadable.
 *
 *	'pagem@trix' is the transformation matrix that pre-multiplies
 *	the default transformation matrix to provide for all coordinates
 *	to be specified in device coordinates with the required layout.
 *
 *	'im@gedict' is used as a global variable storage area between
 *	procedures 'buildch@r' and 'exp@nd'.
 *
 *	'r@ster' is the device resolution in pixels per inch. Note that
 *	this is what DVIOUT thinks the resolution is, not what the actual
 *	resolution may be.
 *
 *	'sc@le' is the scale factor to use to convert from pixels to
 *	PostScript coordinates.
 *
 *	'n@mstr' is a string used for building numbers.
 *
 *	'fontm@trix' is the matrix used for all fonts.
 *
 *	'ch@rvector' is the encoding vector used for all fonts.
 *
 *	'F@ntDir' is our own font directory where font names and
 *	dictionaries are stored.
 *
 *	Function 'xr@w' expands one row of a compressed bitmap.
 *
 *	function xr@w (buffer:string, raster:string, idx:int, width:int): int
 *	{ auto i:int, j:int, k:int, count:int
 *	  begin
 *	    j = 0
 *	    for (i = 0; i < width; ) {
 *	      count = raster[idx]
 *	      idx = idx + 1
 *	      if ((count & 0x80) == 0) {
 *	        count = count + 1
 *	        buffer[j..j+count-1] = raster[idx..idx+count-1]
 *	        idx = idx + count
 *	      } else {
 *	        count = (count & 127) + 1
 *	        for (k = j; k <= j+count-1; k = k + 1)
 *	          buffer[k] = raster[idx]
 *	        idx = idx + 1
 *	      }
 *	      j = j + count
 *	    }
 *	    return (idx)
 *	  end
 *	}
 *
 *	Function 'exp@nd' returns the next row of a bitmap descriptor
 *	for use by the imagemask operator in constructing a bitmap
 *	character. Variables 'w@dth', 'ind@x', 'c@unt', and 'bitm@p'
 *	are global variables used by exp@nd to keep track of the current
 *	context of the expansion.
 *
 *	function exp@nd ( ): string
 *	{ begin
 *	    with im@gedict begin
 *	      if (c@unt == 0) {
 *	        c@unt = bitm@p[ind@x]
 *	        ind@x = xr@w (b@f, bitm@p, ind@x+1, w@dth)
 *	      }
 *	      c@unt = c@unt - 1
 *	      return (b@f[0..w@dth-1])
 *	    end
 *        end
 *	}
 *
 *	Procedure 'buildch@r' constructs one character. It is
 *	referenced by the local BuildChar procedure within each
 *	font dictionary. It constructs the bounding box, then sets
 *	up to pass the actual image generation to procedure 'exp@nd'.
 *
 *	procedure buildch@r (fontdict:dict, charcode:int)
 *	{ static chardata:array, w:int, h:int, x0:int, y0:int
 *	  begin
 *	    with fontdict begin
 *	      chardata = CharInfo[Encoding[charcode]]
 *	      w = chardata[1]
 *	      h = chardata[2]
 *	      x0 = chardata[3] + 0.5
 *	      y0 = chardata[4] + 0.5
 *	      setcachedevice (chardata[0] / 1024, 0, -x0, h+1-y0, w+1-x0, -y0)
 *	      maskm@trix[4] = x0
 *	      maskm@trix[5] = y0
 *	      if (chardata[6]) {
 *	        with im@gedict begin
 *	          w@dth = (w + 7) / 8
 *	          ind@x = 0
 *	          c@unt = 0
 *	          bitm@p = chardata[5]
 *	          b@f = Buffer
 *	        end
 *	        imagemask (w, h, true, maskm@trix, { exp@nd })
 *	      } else {
 *	        imagemask (w, h, true, maskm@trix, { chardata[5] })
 *	      }
 *	    end
 *	  end
 *	}
 *
 *	Function strc@mp complements the values in a string. It returns
 *	the complemented string as the result.
 *
 *	Procedure 'm@kefont' constructs a font dictionary without,
 *	for the time being, any character definitions. It returns
 *	the resulting font dictionary. As much stuff as possible is
 *	shared among fonts.
 *
 *	function m@kefont (id:int, nchars:int): dict
 *	{ auto fontdict:dict
 *	  begin
 *	    fontdict = dict (11)
 *	    with fontdict begin
 *	      CharInfo = dict (nchars)
 *	      UniqueID = id
 *	      n@mstr[0..1] = ['F','@']
 *	      n@mstr[2..k] = cvrs (UniqueID, 16, n@mstr[2..6])
 *	      FontName = cvn (n@mstr[0..k])
 *	      FontMatrix = fontm@trix
 *	      FontType = 3
 *	      PaintType = 3
 *	      FontBBox = [ 0 0 0 0 ]
 *	      Encoding = ch@rvector
 *	      BuildChar = { buildch@r }
 *          end
 *	    return (fontdict)
 *	  end
 *	}
 *
 *	procedure 'c@' places information on one character
 *	of a font into the font dictionary.
 *
 *	procedure c@ (fontdict:dict, charcode:int, width:int, columns:int,
 *	              rows:int, x0:int, y0:int, bitmap:string, compressed:boolean)
 *	{ auto chardata:array
 *	  begin
 *	    chardata[0..6] = [ width, columns, rows, x0, y0, bitmap, compressed ]
 *	    with fontdict begin
 *	      CharInfo[Encoding[charcode]] = chardata
 *	    end
 *	  end
 *	}
 *
 *	Procedure 's@tbbox' computes the font bounding box from the
 *	inidividual character descriptions.
 *
 *	procedure s@tbbox (fontdict:dict)
 *      { auto w:int, h:int, x0:int, y0:int, CharArray:array
 *	  begin
 *          with fontdict begin
 *            forall (CharArray = CharInfo[*]) {
 *              w = CharArray[1]
 *              h = CharArray[2]
 *              x0 = CharArray[3] + 0.5
 *              y0 = CharArray[4] + 0.5
 *              FontBBox[0] = min (FontBBox[0], -x0)
 *              FontBBox[1] = max (FontBBox[1], h+1-y0)
 *              FontBBox[2] = max (FontBBox[2], w+1-x0)
 *              FontBBox[3] = min (FontBBox[3], -y0)
 *            }
 *          end
 *        end
 *	}
 *
 *	Procedure 's@tbuf' determines the size of and allocates a buffer
 *	for a font:
 *
 *	procedure s@tbuf (fontdict:dict)
 *	{ auto maxw, compressed
 *        begin
 *          with fontdict begin
 *	      maxw = 0
 *	      compressed = false
 *            forall (CharArray = CharInfo[*]) {
 *	        if (CharArray[6]) {
 *                compressed = true
 *	          maxw = max ((CharArray[1] + 7) / 8, maxw)
 *	        }
 *            }
 *	      if (compressed)
 *	        Buffer = string (maxw)
 *          end
 *        end
 *	}
 *
 *	Procedure sc@lefont is a slight variant of scalefont;
 *	sc@lesym scales and sets the Symbol font.
 *
 *	m@ is the same as 'moveto'.
 *
 *	l@ is the same as 'lineto'.
 *
 *	rm@ is the same as 'rmoveto'.
 *
 *	rl@ is the same as 'rlineto'.
 *
 *	s@ is the same as 'show'.
 *
 *	n@ is a moveto-show sequence.
 *
 *	@ is like n@, but the y value is taken from the current position.
 *	This is the only single-letter name used here, because it occurs
 *	the most frequently.
 *
 *	f@ finds ands sets the font.
 *
 *	procedure r@ generates a filled rectangle at a point (x,y) with
 *	size (width,depth).
 *
 *	procedure p@ sets the current graphics point to (x,y).
 *
 *	procedure v@ generates a line from the current graphics point
 *	to (x,y) specified relative to the current point, using the
 *	current line width and linestyle.
 *
 *	procedure d@ draws a solid dot (point) at a specified (x,y)
 *	position.
 *
 *	procedure a@ generates an arc, using the current line width and
 *	linestyle.
 *
 *	Procedure i@ outputs an image mask at a specified position:
 *
 *	procedure i@ (w:int, h:int, x:int, y:int, mask:string,
 *	              compressed:boolean)
 *	{ begin
 *	    maskm@trix[4] = -x + 0.5
 *	    maskm@trix[5] = -y + 0.5
 *	    if (compressed) {
 *	      with im@gedict begin
 *	        bitm@p = mask
 *	        ind@x = 0
 *	        c@unt = 0
 *	        w@dth = (w + 7) / 8
 *	        b@f = string (w@dth)
 *	      end
 *	      imagemask (w, h, true, maskm@trix, { exp@nd })
 *	    } else
 *	      imagemask (w, h, true, maskm@trix, { mask })
 *	  end
 *	}
 *
 *	Procedure 'j@' fills a large rectangular area with a pattern
 *	specified as an image bitmap. The pattern is repeated as often
 *	as necessary to completely fill in the rectangle. It uses the
 *	global 'im@gedict' dictionary.
 *
 *	procedure j@ (wscale:int, hscale:int, x:int, y:int, w:int, h:int,
 *	              mask:string)
 *	{ begin
 *	    with im@gedict begin
 *	      bitm@p = strc@mp (mask)
 *	      h@ight = h
 *	      w@dth = (w + 7) / 8
 *	      wsc@le = wscale
 *	      ind@x = 0
 *	      c@unt = 0
 *	      hc@unt = 0
 *	    end
 *	    maskm@trix[5] = -y + 0.5
 *	    maskm@trix[4] = -x + 0.5
 *	    image (w@dth*8*wscale, h*hscale, 1, maskm@trix, { f@ll })
 *	  end
 *	}
 *
 *	Function 'f@ll' returns the next row of the pattern bitmap when
 *	filling a rectangular area with a fill pattern:
 *
 *	function f@ll ( ): string
 *	{ begin
 *	    with im@gedict begin
 *	      if (c@unt == wsc@le) {
 *	        c@unt = 0
 *	        hc@unt = hc@unt + 1
 *	        ind@x = ind@x + w@dth
 *	        if (hc@unt == h@ight) {
 *	          hc@unt = 0
 *	          ind@x = 0
 *	        }
 *            }
 *	      c@unt = c@unt + 1
 *	      return (bitm@p[ind@x..ind@x+w@dth-1])
 *	    end
 *	  end
 *	}
 *
 *	Just a note here on the use of the 'bind' operator. When 'bind'
 *	is used, care must be exercised that none of the local names
 *	used within a procedure coincide with PostScript operator names;
 *	otherwise they get bound to the wrong thing. It is possible that
 *	different PostScript implementations use operators that are
 *	non-standard and/or not documented that may possibly collide
 *	with names used locally within procedures. This is one reason
 *	why '@'s are used in names.
 */

char *PS_Defs[] = {
"/im@gedict 8 dict def\
/n@mstr 12 string def\
/fontm@trix matrix def\
/maskm@trix matrix def\
/sc@le 72 r@ster div def",

"/min{2 copy lt{pop}{exch pop}ifelse}bind def\
/max{2 copy gt{pop}{exch pop}ifelse}bind def\
/T{true}bind def /F{false}bind def\
/strc@mp{0 1 2 index length 1 sub{1 index exch 2 copy get not 255 and put}for}bind def",

"/m@{moveto}bind def\
/l@{lineto}bind def\
/rm@{rmoveto}bind def\
/rl@{rlineto}bind def\
/s@{show}bind def\
/n@{3 1 roll moveto show}bind def\
/@{exch currentpoint exch pop moveto show}bind def\
/f@{F@ntDir exch get setfont}bind def\
/p@{Y@ add/Y@ exch def X@ add/X@ exch def}bind def",

"/v@{gsave newpath X@ Y@ moveto 2 copy p@ rlineto 1 setlinecap stroke grestore}bind def\
/d@{gsave newpath moveto 0 0 rlineto 1024 div setlinewidth 1 setlinecap\
 [] 0 setdash stroke grestore}bind def\
/a@{gsave newpath 3 -1 roll 1024 div 3 1 roll 1000 div exch 1000 div exch\
 2 copy le{arc}{arcn}ifelse 1 setlinecap stroke grestore}bind def\
/r@{gsave newpath moveto 1 index 0 rlineto 0 exch rlineto neg 0 rlineto closepath\
 fill grestore}bind def",

"/ch@rvector 256 array def \
0 1 255{ch@rvector exch dup StandardEncoding exch get dup/.notdef eq\
{pop n@mstr dup 0 99 put 0 2 index 36 n@mstr 1 5 getinterval cvrs \
length 1 add getinterval cvn}if put}for",

"/xr@w{0{2 copy le{exit}if 4 2 roll 2 copy get exch 1 add exch \
dup 128 and 0 eq{1 add 3 copy getinterval 6 index 5 index 3 -1 roll putinterval \
dup 3 -1 roll add exch}{127 and 1 add 3 1 roll 2 copy get 4 index dup 5 index add 1 sub 1 exch\
{7 index exch 2 index put}for pop 1 add 3 -1 roll}ifelse 5 -2 roll 3 -1 roll add}loop \
pop pop 3 1 roll pop pop}bind def\
/exp@nd{im@gedict begin \
c@unt 0 eq{/c@unt bitm@p ind@x get def b@f bitm@p ind@x 1 add w@dth xr@w/ind@x exch def}if\
/c@unt c@unt 1 sub def b@f 0 w@dth getinterval end}bind def",

"/buildch@r{exch begin 0 begin/chardata CharInfo Encoding 4 -1 roll get get def\
/w chardata 1 get def/h chardata 2 get def/x0 chardata 3 get 0.5 add def/y0 chardata 4 get 0.5 add def \
chardata 0 get 1024 div w h mul 0 eq{0 setcharwidth}{0 x0 neg h 1 add y0 sub w 1 add x0 sub y0 neg \
setcachedevice maskm@trix dup 4 x0 put 5 y0 put \
chardata 6 get{im@gedict begin/w@dth w 7 add 8 idiv def/ind@x 0 def/c@unt 0 def\
/bitm@p chardata 5 get def/b@f Buffer def end \
w h T maskm@trix{exp@nd}imagemask}{w h T maskm@trix{chardata 5 get}imagemask}ifelse}ifelse \
end end}dup 2 5 dict put def",

"/i@{4 2 roll maskm@trix exch neg 0.5 add 5 exch put maskm@trix exch neg 0.5 add 4 exch put\
{im@gedict begin/bitm@p exch def/w@dth 2 index 7 add 8 idiv def/b@f w@dth string def\
/ind@x 0 def/c@unt 0 def end T maskm@trix{exp@nd}imagemask}{3 1 roll T maskm@trix{}imagemask}\
ifelse}bind def",

"/f@ll{im@gedict begin c@unt wsc@le eq{/c@unt 0 def/hc@unt hc@unt 1 add def\
/ind@x ind@x w@dth add def hc@unt h@ight eq{/hc@unt 0 def/ind@x 0 def}if}if\
/c@unt c@unt 1 add def bitm@p ind@x w@dth getinterval end}bind def\
/j@{im@gedict begin strc@mp/bitm@p exch def/h@ight exch def/w@dth exch 7 add 8 idiv def \
3 index/wsc@le exch def/ind@x 0 def/c@unt 0 def/hc@unt 0 def \
neg 0.5 add maskm@trix exch 5 exch put neg 0.5 add maskm@trix exch 4 exch put \
h@ight mul exch w@dth 8 mul mul exch end 1 maskm@trix{f@ll}image}bind def",

"/m@kefont{11 dict dup begin 3 1 roll \
dict/CharInfo exch def/UniqueID exch def/FontType 3 def/PaintType 3 def\
/FontMatrix fontm@trix def/FontBBox 4 array def/Encoding ch@rvector def\
/FontName n@mstr dup 0 70 put dup 1 64 put 0 UniqueID 16 n@mstr 2 5 getinterval cvrs \
length 2 add getinterval cvn def/BuildChar/buildch@r load def end}def",

"/c@{7 array astore 3 -1 roll begin CharInfo exch Encoding 4 -1 roll \
get exch put end}def",

"/s@tbbox{begin CharInfo [65535 -65536 -65536 65535] FontBBox copy pop\
{exch pop aload 8 -1 roll pop pop pop pop \
0.5 add neg dup FontBBox dup 3 get 3 -1 roll min 3 exch put \
3 -1 roll 1 add add FontBBox dup 1 get 3 -1 roll max 1 exch put \
0.5 add neg dup FontBBox dup 0 get 3 -1 roll min 0 exch put \
1 add add FontBBox dup 2 get 3 -1 roll max 2 exch put}forall end}def\
/s@tbuf{begin 0 F CharInfo\
{exch pop dup 6 get{exch pop T exch 1 get 7 add 8 idiv 3 -1 roll max exch}{pop}ifelse}forall\
{/Buffer exch string def}{pop}ifelse end}def",

"/s@tfont{F@ntDir exch dup s@tbbox dup s@tbuf dup/FontName get exch 2 copy definefont pop put}def\
/sc@lefont{exch 1024 div exch -1024 div 0 0 3 -1 roll 0 0 matrix astore makefont}def\
/sc@lesym{/Symbol findfont 3 1 roll sc@lefont setfont}def ",
0 };

int PS_Set_Layout (Widest_Width, Tallest_Height, Layout_Mode, Two_Sided,
		   N_Fonts, Max_Size, Max_Primitives, Memory_Size, Max_Fonts,
		   Resolution, Page_Size, Copies, Input_Name, Output_Name)
unsigned long Widest_Width, Tallest_Height, N_Fonts, Max_Size,
	      Max_Primitives, Memory_Size, Max_Fonts, Layout_Mode,
	      Two_Sided;
struct Ratio *Resolution, Page_Size[2];
unsigned short Copies;
char *Input_Name, *Output_Name;
{
	auto   char **Str_Ptr;
/*
 *	Construct file name; open the output file:
 */
	stringcpy_m (Scratch_String, Input_Name, sizeof (Scratch_String));
	stringcat_m (Scratch_String, ".PS", sizeof (Scratch_String));
	if ((PS_File = Open_File_M ((Output_Name == 0) ? "" : Output_Name,
				  Scratch_String, "w", 0)) == 0)
		return (0);
/*
 *	Output the file header comments:
 */
	sprintf (Scratch_String, "%%!PS-Adobe-1.0\n%%%%Title: %s\n%%%%Pages: (atend)\n%%%%EndComments\n",
		 Input_Name);
	Out_Scratch;
/*
 *	Output all definitions:
 */
	sprintf (Scratch_String, "/r@ster %lu %lu div def ", Resolution->Numerator,
		 Resolution->Denominator);
	Out_Scratch;
	for (Str_Ptr = &PS_Defs[0]; *Str_Ptr != 0; Str_Ptr++)
		Write_Block_M (*Str_Ptr, strlen (*Str_Ptr), PS_File);
/*
 *	Set up the transformation matrix to go from DVIOUT's coordinate
 *	system, with (0,0) at the top left corner, to the PostScript
 *	coordinate system, with (0,0) at the lower left. For Landscape
 *	mode, the two coordinate systems are the same, except that the x
 *	and y coordinates reversed. In either case, the coordinate system
 *	is scaled so that one unit is one pixel. All arithmetic is done
 *	using the device's processor.
 */
	if (Layout_Mode == 0)	/* Regular */
		sprintf (Scratch_String, "sc@le 0.0 0.0 sc@le neg 0.0 %lu %lu div 72 mul",
			 Page_Size[1].Numerator, Page_Size[1].Denominator);
	else			/* Landscape */
		sprintf (Scratch_String, "0.0 sc@le sc@le 0.0 0.0 0.0");
	Out_String ("/pagem@trix [ ");
	Out_Scratch;
	Out_String (" ] def ");
/*
 *	Create Font Directory:
 */
	sprintf (Scratch_String, "/F@ntDir %lu dict def ", N_Fonts + MAX_TEMP_FONTS);
	Out_Scratch;
/*
 *	Define 'ej@ct' control sequence to output the proper number of
 *	copies:
 */
	if (Copies == 1)
		Out_String ("/ej@ct{copypage}def ");
	else {
		sprintf (Scratch_String, "/ej@ct{1 1 %u{pop copypage}for}def ", Copies);
		Out_Scratch;
	}
/*
 *	Remainder of initialization:
 */
	Out_String ("initgraphics pagem@trix concat ");
	Init_GState (&Current_State, 1);	/* initgraphics => linewidth = 1 */
	PS_Current_Font = 0;
	PS_Fonts_Available = (Max_Fonts > 0) ? Max_Fonts : 65535;
	PS_Temp_Fonts_Available = MAX_TEMP_FONTS;
	PS_VMSize = (Memory_Size << 10) - 20000 - (N_Fonts + MAX_TEMP_FONTS) * DICT_ENTRY_OVERHEAD;
	PS_Pageno = 0;
}

PS_Terminate ()
{
	sprintf (Scratch_String, "\n%%%%Trailer\n%%%%Pages: %u\n", PS_Pageno);
	Out_Scratch;
	Close_File_M (PS_File);
	PS_File = 0;
}

/*
 *	Routine PS_Download_Font downloads a font to the device.
 *	Native fonts simply scale an existing font and place it in
 *	the font directory; otherwise the characters definitions for
 *	each character in the font have to be constructed and loaded.
 *	Characters that would require a raster description larger than
 *	65536 bytes (before compression) are not downloaded.
 */

PS_Download_Font (Font_Ptr, Char_Vector, N_Chars)
struct Font_Definition *Font_Ptr;
struct Char_Definition **Char_Vector;
unsigned int N_Chars;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned char *Raster_Ptr;
	auto   long Font_Scale;
	auto   unsigned int Raster_Size, Compressed_Size;
	auto   int Index, Compressed;
	extern struct Encoding Identity_Encoding;
	static struct {
		char *Encoding_Name;
		struct Encoding *Encoding_Value;
	} Encoding_List[] = {
		{ "tex text", &TeX_to_PS_Text_Encoding },
		{ "tex typewriter text", &TeX_TT_to_PS_Text_Encoding },
		{ "ascii", &Identity_Encoding },
		{ "ebcdic", &EBCDIC_to_PS_Text_Encoding },
		{ "postscript text", &Identity_Encoding },
		{ "postscript symbol", &Identity_Encoding },
	};
	static char Full_Font_Name[32];
	extern char *Mem_Alloc();
	extern long Convert();
	extern unsigned int PS_Write_Hex();
	extern int strcmp();
/*
 *	Check first for native fonts, which are (relatively) easy to do.
 *	We simply scale the native font by the magnification value for that
 *	font, then put it into F@ntDir. The X scaling is negative for
 *	mirror-image fonts. We also determine the encoding to use at this
 *	point.
 */
	if ((Font_Ptr->Flags & NATIVE_FONT) != 0) {
		PS_Build_Font_Name (Font_Ptr->Font_Family, Font_Ptr->Font_Face, Full_Font_Name);
		Font_Scale = Convert (Font_Ptr->At_Size);
		sprintf (Scratch_String, "F@ntDir /F@%X /%s findfont %s%ld %ld sc@lefont put ",
			 Font_Ptr->Font_Index + 1, Full_Font_Name,
			 ((Font_Ptr->Flags & MIRROR_FONT) == 0) ? "" : "-",
			 Font_Scale, Font_Scale);
		Out_Scratch;
		for (Index = 0; Index < arraysize (Encoding_List); Index++)
		if (strcmp (Encoding_List[Index].Encoding_Name, Font_Ptr->Font_Coding) == 0) {
			Font_Ptr->Font_Encoding = Encoding_List[Index].Encoding_Value;
			break;
		}
		for (Index = 0; Index < N_Chars; Index++)
			Char_Vector[Index]->Driver_Id = 1;
		return;
	}
/*
 *	Check if enough memory is available:
 */
	if (PS_VMSize <= 0 || PS_Fonts_Available == 0 ||
	    (PS_Temp_Fonts_Available == 0 && (Font_Ptr->Flags & TEMP_FONT) != 0)) {
		for (Index = 0; Index < N_Chars; Index++)
			Char_Vector[Index]->Driver_Id = 0;
		return;
	}
/*
 *	Pre-allocate some work storage to store the compressed
 *	raster data:
 */
	Raster_Ptr = (unsigned char *) Mem_Alloc (Font_Ptr->Max_Compressed);
/*
 *	First, build a font dictionary for the font. This leaves the
 *	resulting dictionary on the operand stack:
 */
	sprintf (Scratch_String, "%u %u m@kefont ", Font_Ptr->Font_Index + 1, N_Chars);
	Out_Scratch;
	PS_VMSize -= 424 + N_Chars * DICT_ENTRY_OVERHEAD;
	PS_Fonts_Available--;
	if ((Font_Ptr->Flags & TEMP_FONT) != 0)
		PS_Temp_Fonts_Available--;
/*
 *	For each character needed to be downloaded, compress the raster
 *	information, then send down the character's data:
 */
	Compressed = 0;
	for (Index = 0; Index < N_Chars; Index++) {
		Char_Ptr = Char_Vector[Index];
		if (PS_VMSize <= 0 || Char_Ptr->Pixel_Width == 0 || Char_Ptr->Pixel_Height == 0 ||
		    (Raster_Size = ((Char_Ptr->Pixel_Width + 7) >> 3) * Char_Ptr->Pixel_Height) > 65536) {
			Char_Ptr->Driver_Id = 0;
			continue;
		}
		sprintf (Scratch_String, "dup %lu %ld %u %u %d %d", Char_Ptr->Character_Code,
			 Char_Ptr->H_Escapement, Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
			 Char_Ptr->X_Origin, Char_Ptr->Y_Origin);
		Out_Scratch;
		Compressed_Size = Compress_Raster_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
						     Char_Ptr->Pixel_Array, Raster_Ptr);
		if (Compressed_Size < Raster_Size) {
			PS_Write_Hex (Raster_Ptr, Compressed_Size);
			Out_String ("T ");
			Raster_Size = Compressed_Size;
			Compressed++;
		} else {
			PS_Write_Hex (Char_Ptr->Pixel_Array, Raster_Size);
			Out_String ("F ");
		}
		Out_String ("c@ ");
		Char_Ptr->Driver_Id = 1;
		PS_VMSize -= 7 * OBJECT_OVERHEAD + ((Raster_Size + 3) & ~0x03);
	}
/*
 *	Finish up building the font; decrement memory size by the size
 *	of the scratch buffer that is created for fonts with compressed
 *	characters:
 */
	Out_String ("s@tfont ");
	if (Compressed > 0)
		PS_VMSize -= (((Font_Ptr->Max_Width + 7) >> 3) + 3) & ~0x03;
	Mem_Free (Raster_Ptr);
}

/*
 *	Routine Build_Font_Name constructs the PostScript font name
 *	(e.g. Times-BoldItalic) from the font family and face information
 *	in the font TFM file. This is done by correlating the name and
 *	face to information in a table. All the LaserWriter-Plus fonts
 *	are supported. Other PostScript machines that have fonts other
 *	than these will not be able to use those fonts.
 *
 *	It is (unfortunately) necessary to know about what fonts are
 *	defined on the output device, since there is no standard naming
 *	convention for font names (for example, whether a font is 'Italic'
 *	or 'Oblique' is a function of its name). We are trying to use the
 *	pre-defined face specification conventions (i.e. MRR, MIR, etc.)
 *	established for face codes 0-17. We could, of course, develop our
 *	own meaning for the face code. If we did, we would need unique
 *	identifiers for non-slanted and slanted; an indicator whether the
 *	slanted version of the font is italic or oblique; weights light,
 *	medium, bold, demi, and book; and some indicator when to put in
 *	'Roman'. (The latter problem is the real sticking point -- some-
 *	times 'Roman' is the non-slanted suffix, with the 'Roman' being
 *	dropped when the font is slanted, and sometimes there is no suffix
 *	for the non-slanted fonts. Times-Roman, Palatino-Roman and
 *	NewCenturySchlbk-Roman should not have -Roman as a suffix.
 *	Now, you could say that 'when the slanted version is Italic, then
 *	the non-slanted medium weight font will have suffix -Roman'. This
 *	is fine, except for the ZapfChancery-MediumItalic, which breaks
 *	that rule.) In any case, because of the spelling of the font names
 *	(i.e. use of upper case letters), it is necessary to know which
 *	fonts are out there so we can spell their names right (I don't
 *	think PLtoTF preserves case when writing the family name into
 *	the TFM file).
 */

PS_Build_Font_Name (Family, Face_Code, Str)
char *Family, *Str;
unsigned char Face_Code;
{
	auto   char *Slant, *Weight, *Name, *Ptr;
	auto   unsigned char Face, Expansion;
	auto   int Index, Upcase;
	static char Roman[] = "Roman", Italic[] = "Italic", Oblique[] = "Oblique";
	static char Demi[] = "Demi", Book[] = "Book", Light[] = "Light";
	static char Medium[] = "Medium", Bold[] = "Bold", Null[] = "";
	static char Fallback_Name[40];
	static char *Fallback_Weight_Names[3] = { Null, Bold, Light };
	static struct {
		char *Family_Name;
		char *Full_Family_Name;
		char *Slant_Name[2];
		char *Weight_Name[3];
	} Font_Info[] = {
		{ "times",            "Times",            { Roman, Italic }, { Null, Bold, Null } },
		{ "helvetica",        "Helvetica",        { Null, Oblique }, { Null, Bold, Null } },
		{ "courier",          "Courier",          { Null, Oblique }, { Null, Bold, Null} },
		{ "symbol",           "Symbol",           { Null, Null },    { Null, Null, Null } },
		{ "newcenturyschlbk", "NewCenturySchlbk", { Roman, Italic }, { Null, Bold, Null} },
		{ "palatino",         "Palatino",         { Roman, Italic }, { Null, Bold, Null} },
		{ "bookman",          "Bookman",          { Null, Italic },  { Demi, Null, Light } },
		{ "helvetica-narrow", "Helvetica-Narrow", { Null, Oblique }, { Null, Bold, Null } },
		{ "avantgarde",       "AvantGarde",       { Null, Oblique }, { Demi, Book, Null } },
		{ "zapfchancery",     "ZapfChancery",     { Null, Italic },  { Medium, Null, Null } },
		{ "zapfdingbats",     "ZapfDingbats",     { Null, Null },    { Null, Null, Null } }
	};
	extern int strcmp();
	extern char toupper();

	for (Face = Face_Code, Expansion = 0; Face > 5; Face -= 6, Expansion++)
		;
	for (Index = 0; Index < arraysize (Font_Info); Index++)
	if (strcmp (Family, Font_Info[Index].Family_Name) == 0) {
		Name = Font_Info[Index].Full_Family_Name;
		Slant = Font_Info[Index].Slant_Name[Face&0x01];
		Weight = Font_Info[Index].Weight_Name[Face>>1];
		break;
	}
	if (Index >= arraysize (Font_Info)) {	/* Unknown font, fudge it */
		Name = &Fallback_Name[0];
		stringcpy_m (Name, Family, sizeof (Fallback_Name));
		Upcase = 1;
		for (Ptr = Name; *Ptr != '\0'; Ptr++)
		if (Upcase != 0) {
			*Ptr = toupper (*Ptr);
			Upcase = 0;
		} else if (*Ptr == '-')
			Upcase++;
		Slant = ((Face & 0x01) == 0) ? Null : Italic;
		Weight = Fallback_Weight_Names[Face>>1];
	}
	if (Slant == Roman && Weight != Null)
		Slant = Null;
	sprintf (Str, "%s%s%s%s", Name, (*Slant == '\0' && *Weight == '\0') ? "" : "-",
		 Weight, Slant);
}

/*
 *	Routine PS_Setup_New_Page is responsible for clearing the
 *	output device and setting known values for various things.
 *	we must keep track of the current implicit graphics attributes
 *	because the PostScript save/restore operations affect them.
 */

PS_Setup_New_Page (Page_Number)
long Page_Number[10];
{
	static char Page_Str[32];

	Out_String ("\n");
	if (PS_Pageno == 0)
		Out_String ("%%EndProlog\n");
	Format_Page_Number (Page_Number, Page_Str, sizeof (Page_Str));
	sprintf (Scratch_String, "%%%%Page: %s %u\n", Page_Str, ++PS_Pageno);
	Out_Scratch;
	PS_GX = PS_GY = PS_Y = 0;
	PS_Saved_VMSize = PS_VMSize;
	sprintf (Scratch_String, "erasepage save 0 %ld m@ /X@ %ld def /Y@ %ld def ",
		 PS_Y, PS_GX, PS_GY);
	Out_Scratch;
	Save_GState (&Current_State, &PS_File);
}

/*
 *	Routine PS_Eject_Page outputs however many copies of the page
 *	are required, restores the VM to the previously saved state,
 *	and resets the implicit graphics attributes if they changed
 *	on the page just completed (these attributes are supposed to
 *	persist across pages). The restore may also change the current
 *	font, so we set it to 'undefined' to be certain is gets set
 *	properly on the next Typeset_String operation. Finally, state
 *	variables that could have changed after the "save" command
 *	are restored.
 */

PS_Eject_Page ()
{
	extern int Restore_GState();
	static struct Graphics_State GState;

	Out_String ("ej@ct restore ");
	GState = Current_State;
	if (Restore_GState (&Current_State, &PS_File) != 0)
		PS_Set_State (&GState);
	PS_Current_Font = 0;
	PS_VMSize = PS_Saved_VMSize;
	PS_Fonts_Available += MAX_TEMP_FONTS - PS_Temp_Fonts_Available;
	PS_Temp_Fonts_Available = MAX_TEMP_FONTS;
}

PS_Typeset_String (X, Y, Font_Ptr, Char_Vector, N_Chars)
long X, Y;
struct Font_Definition *Font_Ptr;
unsigned int N_Chars;
struct Char_Definition **Char_Vector;
{
	auto   struct Font_Definition *Font;
	auto   char *Term_Str;
	auto   long Font_Scale, Dev_X, Dev_Y;
	auto   int Index, Count, Term_Len;
	auto   unsigned int Char_Code;
	static struct Font_Definition Symbol_Font;
	extern long Convert();
/*
 *	Move to starting point of string. Construct the output string from
 *	each character description. If a character code greater that 255
 *	is encountered, this means to switch to the Symbol font:
 */
	Dev_X = Round_Pixel (X);
	Dev_Y = Round_Pixel (Y);
	if (Dev_Y != PS_Y) {
		sprintf (Scratch_String, "%ld %ld", Dev_X, Dev_Y);
		PS_Y = Dev_Y;
		Term_Str = ")n@ ";
		Term_Len = 4;
	} else {
		sprintf (Scratch_String, "%ld", Dev_X);
		Term_Str = ")@ ";
		Term_Len = 3;
	}
	Out_Scratch;
	Count = 0;
	for (Index = 0; Index < N_Chars; Index++) {
		Char_Code = (unsigned int) Char_Vector[Index]->Character_Code;
		Font = (Char_Code < 256) ? Font_Ptr : &Symbol_Font;
		if (Font != PS_Current_Font ||
				(Font == &Symbol_Font && Font_Ptr->At_Size != Symbol_Font.At_Size)) {
			if (Count > 0) {
				Write_Block_M (Term_Str, Term_Len, PS_File);
				Count = 0;
				Term_Str = ")s@ ";
				Term_Len = 4;
			}
			if (Font == Font_Ptr)
				sprintf (Scratch_String, "/F@%X f@", Font->Font_Index + 1);
			else {
				Font_Scale = Convert (Font_Ptr->At_Size);
				sprintf (Scratch_String, " %s%ld %ld sc@lesym",
					 ((Font_Ptr->Flags & MIRROR_FONT) == 0) ? "" : "-",
					 Font_Scale, Font_Scale);
				Symbol_Font.At_Size = Font_Ptr->At_Size;
				Char_Code -= 256;
			}
			Out_Scratch;
			PS_Current_Font = Font;
		}
		if (Count == 0)
			Out_Character ('(');
		if (Char_Code == '(' || Char_Code == ')' || Char_Code == '\\' || Char_Code == '%') {
			Out_Character ('\\');
			Out_Character (Char_Code);
		} else if (Char_Code < 32 || Char_Code > 126) {
			sprintf (Scratch_String, "\\%03o", Char_Code);
			Out_Scratch;
		} else
			Out_Character (Char_Code);
		Count++;
	}
	Write_Block_M (Term_Str, Term_Len, PS_File);
}

/*
 *	Routine PS_Typeset_Pixel outputs a raster description at the
 *	specified (x,y) point. Some special logic is needed to account
 *	for raster descriptions that are more than 65536 bytes long
 *	(PostScript limit).
 */

PS_Typeset_Pixel (X, Y, Width, Height, Pixel_Array)
long X, Y;
unsigned short Width, Height;
unsigned char *Pixel_Array;
{
	auto   unsigned char *Pixel_Ptr;
	auto   unsigned short Raster_Width, Raster_Height, H_Max, H_Left;
	auto   long Y_Val;

	if (Width == 0 || Height == 0)
		return;
	Raster_Width = (Width + 7) >> 3;
	H_Max = 65535 / (Raster_Width + (Raster_Width >> 7) + 1);
	if ((H_Left = Height) <= H_Max)
		PS_Output_Image (X, Y, Width, H_Left, Pixel_Array);
	else {
		Y_Val = Y;
		Pixel_Ptr = Pixel_Array;
		for (; H_Left > 0; H_Left -= Raster_Height) {
			if ((Raster_Height = H_Left) > H_Max)
				Raster_Height = H_Max;
			PS_Output_Image (X, Y_Val, Width, Raster_Height, Pixel_Ptr);
			Pixel_Ptr = &Pixel_Ptr[Raster_Width*Raster_Height];
			Y_Val += Raster_Height << 10;
		}
	}
}

PS_Output_Image (X, Y, Width, Height, Pixel_Array)
long X, Y;
unsigned short Width, Height;
unsigned char *Pixel_Array;
{
	auto   char *Raster_Ptr;
	auto   unsigned int Raster_Size, Compressed_Size;
	extern char *Mem_Alloc();
	extern unsigned int PS_Write_Hex();

	Raster_Size = ((Width + 7) >> 3) * Height;
	Raster_Ptr = (unsigned char *) Mem_Alloc (Compressed_Size_Func_M (Width, Height));
	sprintf (Scratch_String, "%u %u %ld %ld", Width, Height, Round_Pixel (X), Round_Pixel (Y));
	Out_Scratch;
	Compressed_Size = Compress_Raster_M (Width, Height, Pixel_Array, Raster_Ptr);
	if (Compressed_Size < Raster_Size) {
		PS_Write_Hex (Raster_Ptr, Compressed_Size);
		Out_String ("T ");
	} else {
		PS_Write_Hex (Pixel_Array, Raster_Size);
		Out_String ("F ");
	}
	Out_String ("i@ ");
	Mem_Free (Raster_Ptr);
}

/*
 *	Routine PS_Typeset_Rule outputs a rule at the specified position
 *	with the specified width and depth. We must be VERY careful when
 *	doing rules to make sure that, if given the same y coordinate as
 *	a character, the bottom of the rule lines up properly with the
 *	baseline of the character. Also, the width and depth of the rule
 *	should be independent of its position on the page.
 */

PS_Typeset_Rule (X, Y, Width, Height)
long X, Y;
unsigned long Width, Height;
{
	auto   unsigned long w, h;

	if ((w = Ceil_Pixel (Width)) > 0)
		w--;
	if ((h = Ceil_Pixel (Height)) > 0)
		h--;
	sprintf (Scratch_String, "%u -%u %ld %ld r@ ", w, h, Round_Pixel (X), Round_Pixel (Y));
	Out_Scratch;
}

/*
 *	Routine PS_Typeset_Filled generates a filled polygon. If the
 *	fill pattern is not solid (Width and Height > 0), this is done
 *	by setting the clipping path to the specified polygon and filling
 *	in a rectangle that completely encloses the polygon. The coordinates
 *	and size of the rectangle are set in such a way that the fill
 *	patterns of overlapping or adjacent polygons mesh properly.
 */

PS_Typeset_Filled (Vertex_Count, Poly_X, Poly_Y, Width, Height, Pixel_Array,
		   Do_Perimeter)
unsigned int Vertex_Count;
long Poly_X[], Poly_Y[];
unsigned short Width, Height;
unsigned char *Pixel_Array;
int Do_Perimeter;
{
	auto   long Min_X, Max_X, Min_Y, Max_Y, X, Y;
	auto   unsigned int Index;
	extern unsigned int PS_Write_Hex();
/*
 *	Set line attributes, if necessary (must be outside gsave/grestore):
 */
	if (Do_Perimeter != 0)
		PS_Set_Line_Attributes ();
/*
 *	First, generate a path defined by the polygon:
 */
	Min_X = Max_X = Round_Pixel (Poly_X[0]);
	Min_Y = Max_Y = Round_Pixel (Poly_Y[0]);
	sprintf (Scratch_String, "gsave newpath %ld %ld m@ ", Min_X, Min_Y);
	Out_Scratch;
	for (Index = 1; Index < Vertex_Count; Index++) {
		X = Round_Pixel (Poly_X[Index]);
		Y = Round_Pixel (Poly_Y[Index]);
		if (X < Min_X) Min_X = X;
		if (X > Max_X) Max_X = X;
		if (Y < Min_Y) Min_Y = Y;
		if (Y > Max_Y) Max_Y = Y;
		sprintf (Scratch_String, "%ld %ld l@ ", X, Y);
		Out_Scratch;
	}
	Out_String ("closepath ");
	if (Do_Perimeter != 0)
		Out_String ("gsave ");
/*
 *	If the fill pattern is solid, do a simple 'fill'; otherwise
 *	output a filled rectangle clipped to the polygon interior:
 */
	if (Width == 0 || Height == 0)
		Out_String ("fill ");
	else {
		Min_X -= Min_X % (long) Width;
		Min_Y -= Min_Y % (long) Height;
		sprintf (Scratch_String, "clip %lu %lu %ld %ld %u %u",
			 (Max_X - Min_X + Width) / (long) Width, (Max_Y - Min_Y + Height) / (long) Height,
			 Min_X, Min_Y, Width, Height);
		Out_Scratch;
		PS_Write_Hex (Pixel_Array, ((Width + 7) >> 3) * Height);
		Out_String ("j@ ");
	}
/*
 *	Draw the lines around the perimeter, if specified:
 */
	if (Do_Perimeter != 0)
		Out_String ("grestore 1 setlinecap stroke ");
	Out_String ("grestore ");
}

PS_Typeset_Line (X1, Y1, X2, Y2)
long X1, Y1, X2, Y2;
{
	auto   long x0, y0, x, y;

	PS_Set_Line_Attributes ();
	x0 = Round_Pixel (X1);
	y0 = Round_Pixel (Y1);
	if (x0 != PS_GX || y0 != PS_GY) {
		sprintf (Scratch_String, "%ld %ld p@ ", x0 - PS_GX, y0 - PS_GY);
		Out_Scratch;
	}
	PS_GX = x = Round_Pixel (X2);
	PS_GY = y = Round_Pixel (Y2);
	sprintf (Scratch_String, "%ld %ld v@ ", x - x0, y - y0);
	Out_Scratch;
}

PS_Typeset_Point (X, Y, Diameter)
long X, Y;
unsigned long Diameter;
{
	sprintf (Scratch_String, "%lu %ld %ld d@ ", Diameter, Round_Pixel (X), Round_Pixel (Y));
	Out_Scratch;
}

PS_Typeset_Arc (X, Y, Radius, Start_Ang, Stop_Ang)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius;
{
	PS_Set_Line_Attributes ();
	sprintf (Scratch_String, "%ld %ld %lu %ld %ld a@ ", Round_Pixel (X), Round_Pixel (Y),
		 Radius, Start_Ang, Stop_Ang);
	Out_Scratch;
}

PS_Set_Linewidth (Width)
unsigned long Width;
{
	if (Current_State.Line_Width != Width) {
		Current_State.Line_Width = Width;
		Current_State.Flags |= CHANGE_LWIDTH;
	}
}

PS_Set_Linestyle (Segment_Len, Segment_Desc, Start_Phase)
unsigned int Segment_Len, Segment_Desc[], Start_Phase;
{
	auto   unsigned int Len, Index;
	extern int Compare_Linestyle();

	Len = (Segment_Len <= 20) ? Segment_Len : 20;
	if (Compare_Linestyle (Current_State.Dashline_Len, Current_State.Dashline_Spec,
			       Len, Segment_Desc) != 0) {
		Current_State.Dashline_Len = Len;
		for (Index = 0; Index < Len; Index++)
			Current_State.Dashline_Spec[Index] = Segment_Desc[Index];
		Current_State.Dashline_Phase = Start_Phase;
		Current_State.Flags |= CHANGE_LSTYLE;
	}
}

PS_Set_Line_Attributes ()
{
	if ((Current_State.Flags & CHANGE_LWIDTH) != 0) {
		sprintf (Scratch_String, "%lu 1024 div setlinewidth ", Current_State.Line_Width);
		Out_Scratch;
	}
	if ((Current_State.Flags & (CHANGE_LWIDTH | CHANGE_LSTYLE)) != 0) {
		PS_Scale_Linestyle ();
		Current_State.Flags &= ~(CHANGE_LWIDTH | CHANGE_LSTYLE);
	}
}

PS_Scale_Linestyle ()
{
	auto   char *Ptr;
	auto   unsigned int Index;
	extern unsigned long Scale_Linestyle();

	Current_State.Scaled_Phase = Scale_Linestyle (Current_State.Dashline_Len,
							   Current_State.Dashline_Spec,
							   Current_State.Line_Width,
							   Current_State.Dashline_Phase,
							   Current_State.Scaled_Spec);
	Ptr = &Scratch_String[0];
	*Ptr++ = '[';
	for (Index = 0; Index < Current_State.Dashline_Len && Ptr < &Scratch_String[sizeof(Scratch_String)-33]; Index++) {
		sprintf (Ptr, "%lu ", Round_Pixel (Current_State.Scaled_Spec[Index]));
		Ptr = &Ptr[strlen(Ptr)];
	}
	sprintf (Ptr, "] %lu setdash ", Round_Pixel (Current_State.Scaled_Phase));
	Out_Scratch;
}

PS_Set_Color (Red, Green, Blue)
unsigned int Red, Green, Blue;
{
	if (Current_State.Color.Red_Component != Red ||
	    Current_State.Color.Green_Component != Green ||
	    Current_State.Color.Blue_Component != Blue) {
		sprintf (Scratch_String, "%u 1000 div %u 1000 div %u 1000 div setrgbcolor ",
			 Red, Green, Blue);
		Out_Scratch;
		Current_State.Color.Red_Component = Red;
		Current_State.Color.Green_Component = Green;
		Current_State.Color.Blue_Component = Blue;
	}
}

PS_Save_Graphics (Identifier)
pointer Identifier;
{
	Save_GState (&Current_State, Identifier);
}

int PS_Restore_Graphics (Identifier)
pointer Identifier;
{
	auto   int Status;
	static struct Graphics_State GState;
	extern int Restore_GState();

	if ((Status = Restore_GState (&GState, Identifier)) != 0)
		PS_Set_State (&GState);
	return (Status);
}

PS_Set_State (GState)
struct Graphics_State *GState;
{
	PS_Set_Color (GState->Color.Red_Component, GState->Color.Green_Component,
		      GState->Color.Blue_Component);
	PS_Set_Linewidth (GState->Line_Width);
	PS_Set_Linestyle (GState->Dashline_Len, GState->Dashline_Spec, GState->Dashline_Phase);
	Current_State.Graphics_X = GState->Graphics_X;
	Current_State.Graphics_Y = GState->Graphics_Y;
	Current_State.Flags |= (GState->Flags & XY_DEFINED);
}

/*
 *	Routine PS_Write_Hex converts a byte-valued array to an ascii
 *	Hexadecimal string and outputs it to the output file.
 */

unsigned int PS_Write_Hex (In, Size)
unsigned char *In;
unsigned int Size;
{
	auto   unsigned char *In_Ptr, Value;
	auto   unsigned int Index;
	static char Hex_Digits[16] = {
		'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
	};

	Out_Character ('<');
	for (In_Ptr = In, Index = Size; Index > 0; Index--) {
		Value = *In_Ptr++;
		Out_Character (Hex_Digits[Value >> 4]);
		Out_Character (Hex_Digits[Value & 0x0F]);
	}
	Out_Character ('>');
	return (Size);
}
