/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This is DVIOUT, a full-featured DVI-to-output-device program
 *	that can support multiple output devices.
 */

#include "debug.h"
#include "types.h"
#include "devtab.h"
#include "dvi.h"
#include "options.h"
#include "font.h"
#include "tracedef.h"

#include "arith.p"
#include "fileio.p"
#include "fio.p"
#include "font.p"
#include "strng.p"

/*
 *	Define some macros needed for the big-switch statement:
 */

#define case_4(k)   case k: case k+1: case k+2: case k+3:
#define case_16(k)  case_4(k) case_4(k+4) case_4(k+8) case_4(k+12)
#define case_64(k)  case_16(k) case_16(k+16) case_16(k+32) case_16(k+48)
#define case_128(k) case_64(k) case_64(k+64)

#define SET_H 0x01
#define SET_V 0x02
#define SET_W 0x04
#define SET_X 0x08
#define SET_Y 0x10
#define SET_Z 0x20
#define SET_STACK 0x40

#define NO   0
#define YES  1
#define QUIT 2
#define GO   3

char Scratch_String[256];		/* Globally shared scratch string */
struct Device_Table *Dev_Ptr;		/* Pointer to device table entry */
unsigned long Job_Magnification;	/* Overall magnification */
long H0, V0;				/* Position where H = V = 0 (top left) */
struct Ratio Dev_Res;			/* Scaled device resolution */
unsigned long Max_N_Primitives;		/* Total number of output primitives */
unsigned long Interrupt_Flag;		/* Non-zero if user interrupts */
long Mark_Width;			/* Width, in DVI units, of mark string */

/*
 *	These next several things are used to batch together
 *	sets of contiguous characters into 'strings', which are
 *	output together to the device:
 */

#define MAX_STRING_SIZE 150
struct Char_Definition *Char_Vector[MAX_STRING_SIZE]; /* Pointers to char descriptors */
unsigned long String_X, String_Y;	/* Position of string */
unsigned long Next_String_X;		/* X coordinate following last character in string */
unsigned int String_Size;		/* Size of accumulated list of characters */

/*
 *	Things from the DVI file Preamble:
 */

struct Ratio DVI_Fraction;		/* DVI to 10**-7 fraction */
unsigned char DVI_Format;		/* File's DVI format id */
unsigned long DVI_Magnification;	/* \mag */

/*
 *	Things from the Postamble:
 */

unsigned long Last_Page_Addr;		/* Position of last bop */
unsigned long Tallest_Page_Height;	/* Height of the tallest page */
unsigned long Widest_Page_Width;	/* Width of the widest page */
unsigned short Maximum_Stack_Depth;	/* Required size of stack */
unsigned short Number_of_Pages;		/* Total number of pages */

/*
 *	The Page Index contains the file addresses for each page
 *	in the DVI file. There are 'Number_of_Pages' of them allocated
 *	dynamically.
 */

struct Page_Index_Entry {
	unsigned long Page_Addr;	/* DVI file page address */
	long Page_No[10];		/* \count0 .. \count9 */
};
struct Page_Index_Entry *Page_Index;	/* Pointer to array of indices */

/*
 *	The position data structure and Stack:
 */

struct Position {
	long H, V;			/* Current position */
	long W, X;			/* Horizontal spacing */
	long Y, Z;			/* Vertical spacing */
};
struct Position *Stack;			/* Allocated dynamically */

/*
 *	The Main Program retrieves one file specification at a time,
 *	then invokes Process_File to generate output from the DVI
 *	file.
 */

main ()
{
	auto   pointer DVI_File;
	auto   unsigned long Status;
	static char Input_File[256], File_Name[40];
	extern unsigned long Process_File();
	extern int Get_Input_File_Name();
	msgcode DVIOUT_INOPENFAIL, DVIOUT_INTERRUPT, SS$_NORMAL;

	Initialize();
	Process_Run_Options (&Global_Options, -1);
#ifdef DEBUG
	Show_Options (&Global_Options, "Global");
#endif
	Enable_Interrupt (&Interrupt_Flag);
	Status = SS$_NORMAL;
	while (Status == SS$_NORMAL &&
	    Get_Input_File_Name (Input_File, sizeof (Input_File)) != 0) {
		Process_Run_Options (&Local_Options, 1);
#ifdef DEBUG
		Show_Options (&Local_Options, "Local");
#endif
		Apply_Options (&Global_Options, &Local_Options);
		Trace_Ptr = &Local_Options.Trace;
		Dev_Ptr = Local_Options.Device_Ptr;
		Dev_Res.Numerator = Dev_Ptr->Device_Resolution.Numerator * 64;
		Dev_Res.Denominator = Dev_Ptr->Device_Resolution.Denominator * 15875;
		Provide_Name_Only (Input_File, File_Name);
		if (Interrupt_Flag != 0)
			Status = DVIOUT_INTERRUPT;
		else if ((DVI_File = Open_File_M (Input_File, "TeXPUT.DVI",
                                                  "r", 0)) == 0)
			Message (DVIOUT_INOPENFAIL, Input_File, 1);
		else {
			if (trace_dvi)
				printf ("Input file is \"%s\"\n",
                                        Full_File_Name_M (DVI_File));
			Status = Process_File (DVI_File, File_Name,
					       ((Local_Options.Flags &
                                                 OUTPUT) == 0) ?
					       0 : Local_Options.Output);
		}
		Liberate_Memory ();
	}
	exit (Status);
}

/*
 *	Routine Process_File is the main driver for DVIOUT.
 *	It processes one input DVI file.
 */

unsigned long Process_File (DVI_File, Input_Name, Output_Name)
pointer DVI_File;
char *Input_Name, *Output_Name;
{
	auto   unsigned long Status;
	static struct Ratio Space_Scale = { 4, 10 };
	extern struct Font_Definition *Set_Font();
	extern int Process_Preamble(), Process_Postamble();
	extern int Generate_Page_Index(), Load_All_Fonts();
	extern int Pass_File(), Collapse_XY_Stack();
	extern char *Mem_Alloc();
	msgcode DVIOUT_BADDVIFILE, DVIOUT_FONTLOADERR, DVIOUT_UNKDVIFORMAT,
		DVIOUT_DEVINITFAIL, DVIOUT_INTERRUPT, DVIOUT_CANTDELETE,
		DVIOUT_BADXYSTACK, SS$_NORMAL;

	Initialize_For_File ();
	if (Process_Preamble (DVI_File) == 0 || Process_Postamble (DVI_File) == 0)
		Status = DVIOUT_BADDVIFILE;
	else if (DVI_Format != DVI_FORMAT_B)
		Status = DVIOUT_UNKDVIFORMAT;
	else if (Interrupt_Flag != 0)
		Status = DVIOUT_INTERRUPT;
	else if (Generate_Page_Index (DVI_File, Local_Options.Flags & SORTED) == 0)
		Status = DVIOUT_BADDVIFILE;
	else if (Interrupt_Flag != 0)
		Status = DVIOUT_INTERRUPT;
	else if (Pass_File (DVI_File, 0) == 0)
		Status = DVIOUT_BADDVIFILE;
	else if (Interrupt_Flag != 0)
		Status = DVIOUT_INTERRUPT;
	else if (Load_All_Fonts (Dev_Ptr, Local_Options.Flags & USETFM, &DVI_Fraction, Job_Magnification,
				 Local_Options.Pixel_Load_Func) == 0)
		Status = DVIOUT_FONTLOADERR;
	else if (Interrupt_Flag != 0)
		Status = DVIOUT_INTERRUPT;
	else {
/*
 *	Allocate stack; build page mark character box, if necessary:
 */
		Stack = (struct Position *) Mem_Alloc (sizeof (struct Position) * Maximum_Stack_Depth);
		if ((Local_Options.Flags & MARK) != 0)
			Mark_Width = String_Width_M (Set_Font (Mark_Font_Number, 0), Local_Options.Page_Mark,
						     &Space_Scale);
/*
 *	Initialize the device and Download the characters determined
 *	during pass one:
 */
		if ((*Dev_Ptr->Set_Layout) (Widest_Page_Width, Tallest_Page_Height,
					    Local_Options.Flags & ROTATED, Local_Options.Flags & TWO_SIDED,
					    Font_Count, Max_Char_Size, Max_N_Primitives,
					    ((Local_Options.Flags & MEMORY) == 0) ?
					    Dev_Ptr->Device_Memory : Local_Options.Memory_Size,
					    Dev_Ptr->Max_Fonts, &Dev_Ptr->Device_Resolution,
					    &Local_Options.Form_Size[0], Local_Options.Number_of_Copies,
					    Input_Name, Output_Name) == 0)
			Status = DVIOUT_DEVINITFAIL;
		else {
			Download_Fonts (Dev_Ptr->Download_Font, &Dev_Ptr->Device_Resolution);
			if (trace_dvi) {
				printf ("Total number of characters used is %u\n", Char_Count);
				printf ("Maximum character size of characters used is %u\n", Max_Char_Size);
			}
/*
 *	Now generate output on the device:
 */
			Pass_File (DVI_File, 1);
			if (Collapse_XY_Stack () != 0)
				Message (DVIOUT_BADXYSTACK);
/*
 *	Terminate output on the device; set return status:
 */
			(*Dev_Ptr->Terminate) ();
			if (Interrupt_Flag != 0)
				Status = DVIOUT_INTERRUPT;
			else
				Status = SS$_NORMAL;
		}
	}
/*
 *	Close the input file; delete it if /DELETE was specified
 *	and there were no errors or interrupt:
 */
	if (Status == SS$_NORMAL) {
		if ((Local_Options.Flags & DELETE) == 0)
			Close_File_M (DVI_File);
		else {
			stringcpy_m (Scratch_String, Full_File_Name_M (DVI_File), sizeof (Scratch_String));
			if (Delete_File_M (DVI_File) == 0)
				Message (DVIOUT_CANTDELETE, Scratch_String, 1);
		}
	} else {
		if (Status != DVIOUT_INTERRUPT) {
			Message (Status, Full_File_Name_M (DVI_File), 1);
			Status = SS$_NORMAL;
		}
		Close_File_M (DVI_File);
	}
	return (Status);
}

/*
 *	Routine Pass_File makes a pass through the file, using
 *	only those pages selected.
 */

int Pass_File (DVI_File, Pass)
pointer DVI_File;
int Pass;
{
	auto   int (*Page_Func)();
	auto   struct Pages_Spec *Page_Ptr;
	auto   int Do_Page;
	auto   unsigned int Index, Jndex;
	static struct Pages_Spec All_Pages = {	/* Equivalent to /PAGES=*.*.*.*.*.*.*.*.*.* */
		0, { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 }, 0
	};
	extern int Match_Page(), Query_Do_Page();
	extern int Process_Page_1(), Process_Page_2();
/*
 *	Convert horizontal and vertical offset values to 1/1024 pixels
 *	(these numbers are always in scaled points). These values are
 *	not scaled by the magnification.
 */
	H0 = XN_Div_D_T_M (Local_Options.Hoffset, Dev_Ptr->Device_Resolution.Numerator * 100,
			   Dev_Ptr->Device_Resolution.Denominator * 462528);
	V0 = XN_Div_D_T_M (Local_Options.Voffset, Dev_Ptr->Device_Resolution.Numerator * 100,
			   Dev_Ptr->Device_Resolution.Denominator * 462528);
/*
 *	Determine which function to call to process a page, based on
 *	which file pass we are on:
 */
	if ((Page_Ptr = Local_Options.Pages_Head) == 0)	/* No /PAGES specified */
		Page_Ptr = &All_Pages;
	Page_Func = (Pass == 0) ? &Process_Page_1 : Process_Page_2;
/*
 *	Process each page specified:
 */
	for (Index = 0; Index < Number_of_Pages && Interrupt_Flag == 0; Index++) {
		Jndex = ((Local_Options.Flags & REVERSE) == 0) ? Index : Number_of_Pages - Index - 1;
		if (Match_Page (Page_Ptr, Page_Index[Jndex].Page_No) != 0) {
			if (Pass == 0 || (Local_Options.Flags & INTERACTIVE) == 0)
				Do_Page = YES;
			else if ((Do_Page = Query_Do_Page (Page_Index[Jndex].Page_No)) == QUIT)
				return (1);
			else if (Do_Page == GO) {
				Local_Options.Flags &= ~INTERACTIVE;
				Do_Page = YES;
			}
			if (Do_Page == YES && (*Page_Func) (Page_Index[Jndex].Page_Addr,
							    DVI_File) == 0)
				return (0);
		}
	}
	return (1);
}

/*
 *	Routine Match_Page determines if a page number from the DVI
 *	file matches the /PAGES specification.
 */

int Match_Page (Page_Head, Page_Number)
struct Pages_Spec *Page_Head;
long Page_Number[10];
{
	auto   struct Pages_Spec *Page_Ptr;
	auto   int Odd, Match;
	auto   unsigned int Index;
	extern int Compare_Page_Number();
/*
 *	First, determine if the page number matches any of the search or
 *	range parameter(s):
 */
	Page_Ptr = Page_Head;
	do {
	 	if (Page_Ptr->Range == 0) {	/* Search-style page specification */
			Match = 1;
			for (Index = 0; Index < 10 && Match != 0; Index++)
			if (Page_Ptr->High_Page[Index] == 0 && Page_Ptr->Low_Page[Index] != Page_Number[Index])
				Match = 0;
		} else if (Compare_Page_Number (Page_Number, Page_Ptr->Low_Page) < 0 ||
			   Compare_Page_Number (Page_Number, Page_Ptr->High_Page) > 0)
			Match = 0;
		else
			Match = 1;
	} while (Match == 0 && (Page_Ptr = Page_Ptr->Link) != 0);
/*
 *	Now determine if it matches the /ODD or /EVEN requirement.
 *	A page number is 'odd' if the first page number element is
 *	odd (e.g. page 1.6 is odd; 2.3 is even). This corresponds
 *	to \count0.
 */
	if (Match != 0 && (Local_Options.Flags & (EVEN_PAGES | ODD_PAGES)) != 0) {
		Odd = Page_Number[0] & 0x01;
		if (!((Odd == 0 && (Local_Options.Flags & EVEN_PAGES) != 0) ||
		      (Odd != 0 && (Local_Options.Flags & ODD_PAGES) != 0)))
			Match = 0;
	}
/*
 *	Finally, check for /EXCLUDE, which reverses the logic of the
 *	above tests:
 */
	return (((Local_Options.Flags & EXCLUDE) == 0) ? Match : 1-Match);
}

/*
 *	Routine Compare_Page_Number compares two page numbers and returns
 *	-1 if first < second; 0 if first = second; and +1 if first > second.
 *	NOTE: Page numbers are compared as if \count0 is the least significant
 *	digit, \count1 the next most significant digit, and so on.
 */

int Compare_Page_Number (Page1, Page2)
long Page1[10], Page2[10];
{
	auto   unsigned int Index;

	for (Index = 9; Page1[Index] == Page2[Index]; Index--)
	if (Index == 0)
		return (0);
	return ((Page1[Index] < Page2[Index]) ? -1 : 1);
}

int Query_Do_Page (Page_Number)
long Page_Number[10];
{
	auto   int Answer;
	auto   char c;
	static char Response[8], Page_Str[32];
	extern int Tty_In_With_Prompt();

	Format_Page_Number (Page_Number, Page_Str, sizeof (Page_Str));
	sprintf (Scratch_String, "Next page is page number %s. Enter option [Y/N/G/Q]: ", Page_Str);
	if (Tty_In_With_Prompt (Scratch_String, Response, sizeof (Response)) == 0)
		Answer = QUIT;
	else if ((c = Response[0]) == 'y' || c == 'Y')
		Answer = YES;
	else if (c == 'g' || c == 'G')
		Answer = GO;
	else if (c == 'q' || c == 'Q')
		Answer = QUIT;
	else
		Answer = NO;
	return (Answer);
}

/*
 *	Pass 1 over the file collects information on the frequency
 *	of character usage, to minimize the amount of processing to
 *	only that required to generate output for this job.
 */

int Process_Page_1 (Page_Addr, DVI_File)
unsigned long Page_Addr;
pointer DVI_File;
{
	auto   struct Font_Definition *Current_Font;
	auto   unsigned long Max_Primitives_on_Page;
	auto   unsigned int Reflection_Level;
	auto   unsigned char Command;
	extern struct Font_Definition *Set_Font(), *Reset_Font();
	extern unsigned long Record_String();
/*
 *	Position to the beginning of page, read in BOP command:
 */
	if (Set_File_Position_M (Page_Addr, DVI_File) != Page_Addr || Read_Character_M (DVI_File) != BOP)
		return (0);
	Current_Font = 0;
	Max_Primitives_on_Page = 0;
	Reflection_Level = 0;
	Skip (44, DVI_File);
/*
 *	Now scan through the page commands, processing each as we go:
 */
	while (File_At_EOF_M (DVI_File) == 0 && Interrupt_Flag == 0) {
		switch (Command = Read_Character_M (DVI_File)) {

		case_128(SET_CHAR_0)
			Record_Character (Current_Font, Command - SET_CHAR_0);
			Max_Primitives_on_Page++;
			break;

		case_4(SET1)
			Record_Character (Current_Font, Read_Unsigned_Int_M (Command-SET1+1, DVI_File));
			Max_Primitives_on_Page++;
			break;

		case SET_RULE: case PUT_RULE:
			Read_Signed_Int_M (4, DVI_File);
			Read_Signed_Int_M (4, DVI_File);
			Max_Primitives_on_Page++;
			break;

		case_4(PUT1)
			Record_Character (Current_Font, Read_Unsigned_Int_M (Command-PUT1+1, DVI_File));
			Max_Primitives_on_Page++;
			break;

		case NOP: case PUSH: case POP: case W0: case X0: case Y0: case Z0: break;

		case EOP:
			if ((Local_Options.Flags & MARK) != 0) {
				Current_Font = Set_Font (Mark_Font_Number, 0);
				Max_Primitives_on_Page += Record_String (Current_Font, Local_Options.Page_Mark) +
							  Record_String (Current_Font, Local_Options.Page_Mark);
			}
			if (Max_Primitives_on_Page > Max_N_Primitives)
				Max_N_Primitives = Max_Primitives_on_Page;
			return (1);

		case_4(RIGHT1)
			Read_Signed_Int_M (Command-RIGHT1+1, DVI_File);
			break;

		case_4(W1)
			Read_Signed_Int_M (Command-W1+1, DVI_File);
			break;

		case_4(X1)
			Read_Signed_Int_M (Command-X1+1, DVI_File);
			break;

		case_4(DOWN1)
			Read_Signed_Int_M (Command-DOWN1+1, DVI_File);
			break;

		case_4(Y1)
			Read_Signed_Int_M (Command-Y1+1, DVI_File);
			break;

		case_4(Z1)
			Read_Signed_Int_M (Command-Z1+1, DVI_File);
			break;

		case_64(FNT_NUM_0)
			Current_Font = Set_Font (Command - FNT_NUM_0, Reflection_Level);
			break;

		case_4(FNT1)
			Current_Font = Set_Font (Read_Unsigned_Int_M (Command-FNT1+1, DVI_File),
						 Reflection_Level);
			break;

		case_4(XXX1)
			Skip (Read_Unsigned_Int_M (Command-XXX1+1, DVI_File), DVI_File);
			break;

		case_4(FNT_DEF1)
			Skip (Command - FNT_DEF1 + 1 + 12, DVI_File);
			Skip (Read_Unsigned_Int_M (1, DVI_File) + Read_Unsigned_Int_M (1, DVI_File), DVI_File);
			break;

		case BEGIN_REFLECT:
			Reflection_Level++;
			Current_Font = Reset_Font (Current_Font, Reflection_Level);
			break;

		case END_REFLECT:
			if (Reflection_Level > 0)
				Reflection_Level--;
			Current_Font = Reset_Font (Current_Font, Reflection_Level);
			break;

		default:
			return (0);
		}
	}
	return ((Interrupt_Flag == 0) ? 0 : 1);
}

/*
 *	Routine Process_Page_2 reads in the DVI data for one page and
 *	processes it. Pass 2 actually generates output to the device.
 */

int Process_Page_2 (Page_Addr, DVI_File)
unsigned long Page_Addr;
pointer DVI_File;
{
	auto   unsigned long Font_Count_Save, Font_Number_Save;
	auto   int Index, Status;
	static struct Position Start_Position = { 0, 0, 0, 0, 0, 0 };
	static long Page_Number[10];
	static char Page_Str[32];
	extern int Execute_DVI();
/*
 *	Position to the beginning of page, read in BOP command;
 *	execute page description commands; initialize and terminate
 *	each page at the device:
 */
	if (Set_File_Position_M (Page_Addr, DVI_File) != Page_Addr || Read_Character_M (DVI_File) != BOP)
		Status = 0;
	else {
		for (Index = 0; Index < 10; Index++)
			Page_Number[Index] = Read_Signed_Int_M (4, DVI_File);
		if (trace_dvi) {
			Format_Page_Number (Page_Number, Page_Str, sizeof (Page_Str));
			printf ("Page number is %s\n", Page_Str);
		}
		Font_Count_Save = Font_Count;
		Font_Number_Save = Max_Font_Number;
		(*Dev_Ptr->Setup_New_Page) (Page_Number);
		Status = Execute_DVI (&Start_Position, Maximum_Stack_Depth, 0, 0, Page_Addr + 45, DVI_File);
		(*Dev_Ptr->Eject_Page) ();
		Font_Count = Font_Count_Save;
		Max_Font_Number = Font_Number_Save;
	}
	return (Status);
}

/*
 *	Routine Execute_DVI executes the DVI file from the given
 *	file address up until the specified command is seen (either
 *	EOP or END_REFLECT). The inputs to this routine determine
 *	the transformation of (h,v) positions in the DVI file to
 *	the (h',v) reflected positions. This routine is called
 *	recursively for each reflected segment of the DVI file.
 *
 *	This routine implements the 'reflection' mode of TeX-XeT
 *	described in "Mixing right-to-left texts with left-to-right
 *	texts", by Donald Knuth and Pierre MacKay in the April
 *	1987 issue of TUGboat magazine. While not at all a trivial
 *	task to implement, it is conceptually easy to understand
 *	what must be done to do it. Essentially, each segment of
 *	DVI text between matching BEGIN_REFLECT and END_REFLECT
 *	commands must be reflected about the centerpoint of the h
 *	positions of the text at those two points in the file, as
 *	described in the TUGboat article. This process can be
 *	recursive, so that reflected text can appear within reflected
 *	text. Since we do not know how many levels this will occur to
 *	in the DVI file, the process is implemented in a recursive
 *	procedure, allowing the necessary storage to be allocated on
 *	the run-time stack. This is implemented as follows:
 *
 *	  1. When the first BEGIN_REFLECT is encountered, the current
 *	     state is saved, consisting of the current file position,
 *	     the current font, the (h,v,w,x,y,z) values, and the current
 *	     stack pointer (needed for error detection).
 *	  2. The reflection level is incremented by one. While this
 *	     is above the initial value, no output is produced at the
 *	     device.
 *	  3. When the matching END_REFLECT is seen, the procedure
 *	     is called recursively with the parameters that were saved
 *	     when the BEGIN_REFLECT was encountered. Output of the
 *	     reflected segment then occurs.
 *	  4. BEGIN_REFLECT / END_REFLECT pairs encountered between
 *	     the pair of interest only serve to increment / decrement
 *	     the reflection level; nothing else is done with these (yet).
 *
 *	Restrictions on the reflection mode consist of the following:
 *
 *	  1. BEGIN_REFLECT / END_REFLECT pairs must be wholly
 *	     contained on each page; the reflection cannot cross
 *	     page boundaries.
 *	  2. Furthermore, the reflection pairs cannot cross line
 *	     boundaries, either. That is, the matching END_REFLECT
 *	     to a BEGIN_REFLECT should be in the same text line.
 *	     While this condition is not checked for, peculiar results
 *	     would occur should it ever happen. (We cannot be REALLY
 *	     sure that the two points in the file are on the same line,
 *	     since some small \raising or \lowering may be in effect at
 *	     either endpoint.)
 *	  3. The stack position at the matching END_REFLECT must be
 *	     the same as the stack position at the BEGIN_REFLECT
 *	     command. For example, it is not permissable to have
 *	     a BEGIN_REFLECT, PUSH, END_REFLECT sequence. The matching
 *	     POP must occur prior to the matching END_REFLECT. Similarly,
 *	     a POP command cannot be present between a BEGIN_REFLECT /
 *	     END_REFLECT pair.
 */

#define Add_to_H(x) if ((Start_Level & 0x01) == 0) HH += x; else HH -= x

int Execute_DVI (Start_Position, Stack_Level, Font, Start_Level, Page_Addr, DVI_File)
struct Position *Start_Position;
struct Font_Definition *Font;
unsigned int Stack_Level, Start_Level;
unsigned long Page_Addr;
pointer DVI_File;
{
	auto   struct Font_Definition *Current_Font, *Save_Font;
	auto   struct Char_Definition *Char_Ptr;
	auto   struct Position *Stack_Ptr, *Save_Stack_Ptr, Save_Position;
	auto   char *Special_String;
	auto   unsigned long Save_Addr;
	auto   long HH, VV, WW, XX, YY, ZZ, Param_1, Param_2;
	auto   unsigned int Reflection_Level;
	auto   unsigned char Command, Trace_Mask;
	static char *Literal_Commands_A[] = {
		"Set1", "Set2", "Set3", "Set4", "Set_Rule",
		"Put1", "Put2", "Put3", "Put4", "Put_Rule",
		"Nop", "Bop", "Eop", "Push", "Pop",
		"Right1", "Right2", "Right3", "Right4",
		"W0", "W1", "W2", "W3", "W4",
		"X0", "X1", "X2", "X3", "X4",
		"Down1", "Down2", "Down3", "Down4",
		"Y0", "Y1", "Y2", "Y3", "Y4",
		"Z0", "Z1", "Z2", "Z3", "Z4"
	}, *Literal_Commands_B[] = {
		"Fnt1", "Fnt2", "Fnt3", "Fnt4",
		"XXX1", "XXX2", "XXX3", "XXX4",
		"Fnt_Def1", "Fnt_Def2", "Fnt_Def3", "Fnt_Def4",
		"Pre", "Post", "Post_Post",
		"Begin_Reflect", "End_Reflect",
		"Undef", "Undef", "Undef", "Undef"
	};
	extern struct Font_Definition *Set_Font(), *Reset_Font();
	extern struct Char_Definition *Get_Character();
	extern char *Mem_Alloc();
	extern long Convert();
	extern long Set_Rule();
	msgcode DVIOUT_STACKOVERFLOW, DVIOUT_STACKUNDERFLOW, DVIOUT_REFLECTION;
/*
 *	Position to the starting location in the file:
 */
	if (Set_File_Position_M (Page_Addr, DVI_File) != Page_Addr)
		return (0);
/*
 *	Initialize:
 */
	Stack_Ptr = &Stack[Stack_Level];
	HH = Start_Position->H; VV = Start_Position->V;
	WW = Start_Position->W; XX = Start_Position->X;
	YY = Start_Position->Y; ZZ = Start_Position->Z;
	Reflection_Level = Start_Level;
	Current_Font = Reset_Font (Font, Reflection_Level);
	String_Size = 0;
/*
 *	Now scan through the page commands, processing each as we go:
 */
	while (File_At_EOF_M (DVI_File) == 0 && Interrupt_Flag == 0) {
		Trace_Mask = 0;
		switch (Command = Read_Character_M (DVI_File)) {

		case_128(SET_CHAR_0)
			if ((Char_Ptr = Get_Character (Current_Font, Command-SET_CHAR_0)) != 0) {
				if (Reflection_Level == Start_Level)
					Set_Character (Current_Font, HH, VV, Char_Ptr, Reflection_Level);
				Add_to_H (Char_Ptr->DVI_Width);
			}
			Trace_Mask = SET_H;
			break;

		case_4(SET1)
			if ((Char_Ptr = Get_Character (Current_Font,
						       Read_Unsigned_Int_M (Command-SET1+1, DVI_File))) != 0) {
				if (Reflection_Level == Start_Level)
					Set_Character (Current_Font, HH, VV, Char_Ptr, Reflection_Level);
				Add_to_H (Char_Ptr->DVI_Width);
			}
			Trace_Mask = SET_H;
			break;

		case SET_RULE:
			Param_1 = Read_Signed_Int_M (4, DVI_File);
			Param_2 = Read_Signed_Int_M (4, DVI_File);
			if (Reflection_Level == Start_Level)
				Set_Rule (HH, VV, Param_1, Param_2, Reflection_Level);
			Add_to_H (Param_2);
			Trace_Mask = SET_H;
			break;

		case_4(PUT1)
			if ((Char_Ptr = Get_Character (Current_Font,
						       Read_Unsigned_Int_M (Command-PUT1+1, DVI_File))) != 0)
				if (Reflection_Level == Start_Level)
					Set_Character (Current_Font, HH, VV, Char_Ptr, Reflection_Level);
			break;

		case PUT_RULE:
			Param_1 = Read_Signed_Int_M (4, DVI_File);
			Param_2 = Read_Signed_Int_M (4, DVI_File);
			if (Reflection_Level == Start_Level)
				Set_Rule (HH, VV, Param_1, Param_2, Reflection_Level);
			break;

		case NOP: break;

		case EOP:
			Set_String (Current_Font);
			if ((Local_Options.Flags & MARK) != 0) {
				Current_Font = Set_Font (Mark_Font_Number, 0);
				if ((Local_Options.Flags & ROTATED) == 0)
					Output_Mark (Current_Font, Local_Options.Page_Mark,
						     Mark_Width, &Local_Options.Form_Size[0],
						     &Local_Options.Form_Size[1]);
				else
					Output_Mark (Current_Font, Local_Options.Page_Mark,
						     Mark_Width, &Local_Options.Form_Size[1],
						     &Local_Options.Form_Size[0]);
			}
			if (Reflection_Level != Start_Level)
				Message (DVIOUT_REFLECTION);
			return (1);

		case PUSH:
			if (Stack_Ptr == &Stack[0])
				Message (DVIOUT_STACKOVERFLOW);
			else {
				Stack_Ptr--;
				Stack_Ptr->H = HH; Stack_Ptr->V = VV;
				Stack_Ptr->W = WW; Stack_Ptr->X = XX;
				Stack_Ptr->Y = YY; Stack_Ptr->Z = ZZ;
			}
			Trace_Mask = SET_H | SET_V | SET_W | SET_X | SET_Y | SET_Z | SET_STACK;
			break;

		case POP:
			if (Stack_Ptr == &Stack[Stack_Level])
				Message (DVIOUT_STACKUNDERFLOW);
			else {
				HH = Stack_Ptr->H; VV = Stack_Ptr->V;
				WW = Stack_Ptr->W; XX = Stack_Ptr->X;
				YY = Stack_Ptr->Y; ZZ = Stack_Ptr->Z;
				Stack_Ptr++;
			}
			Trace_Mask = SET_H | SET_V | SET_W | SET_X | SET_Y | SET_Z | SET_STACK;
			break;

		case_4(RIGHT1)
			Param_1 = Read_Signed_Int_M (Command-RIGHT1+1, DVI_File);
			Add_to_H (Param_1);
			Trace_Mask = SET_H;
			break;

		case W0:
			Add_to_H (WW);
			Trace_Mask = SET_H;
			break;

		case_4(W1)
			WW = Read_Signed_Int_M (Command-W1+1, DVI_File);
			Add_to_H (WW);
			Trace_Mask = SET_H | SET_W;
			break;

		case X0:
			Add_to_H (XX);
			Trace_Mask = SET_H;
			break;

		case_4(X1)
			XX = Read_Signed_Int_M (Command-X1+1, DVI_File);
			Add_to_H (XX);
			Trace_Mask = SET_H | SET_X;
			break;

		case_4(DOWN1)
			VV += Read_Signed_Int_M (Command-DOWN1+1, DVI_File);
			Trace_Mask = SET_V;
			break;

		case Y0:
			VV += YY;
			Trace_Mask = SET_V;
			break;

		case_4(Y1)
			YY = Read_Signed_Int_M (Command-Y1+1, DVI_File);
			VV += YY;
			Trace_Mask = SET_V | SET_Y;
			break;

		case Z0:
			VV += ZZ;
			Trace_Mask = SET_V;
			break;

		case_4(Z1)
			ZZ = Read_Signed_Int_M (Command-Z1+1, DVI_File);
			VV += ZZ;
			Trace_Mask = SET_V | SET_Z;
			break;

		case_64(FNT_NUM_0)
			Set_String (Current_Font);
			Current_Font = Set_Font (Command - FNT_NUM_0, Reflection_Level);
			break;

		case_4(FNT1)
			Set_String (Current_Font);
			Current_Font = Set_Font (Read_Unsigned_Int_M (Command-FNT1+1, DVI_File),
						 Reflection_Level);
			break;

		case_4(XXX1)
			Param_1 = Read_Unsigned_Int_M (Command-XXX1+1, DVI_File);
			Special_String = Mem_Alloc (Param_1 + 1);
			Read_Block_M (Special_String, Param_1, DVI_File);
			Special_String[Param_1] = '\0';
			if (Reflection_Level == Start_Level) {
				if (trace_dvi)
					printf ("%s \"%s\"", Literal_Commands_B[Command-FNT1], Special_String);
				Execute_Special (Special_String, Convert (HH) + H0, Convert (VV) + V0,
						 Job_Magnification, Reflection_Level & 0x01, Dev_Ptr);
			}
			Mem_Free (Special_String);
			break;

		case_4(FNT_DEF1)
			Skip (Command - FNT_DEF1 + 1 + 12, DVI_File);
			Skip (Read_Unsigned_Int_M (1, DVI_File) + Read_Unsigned_Int_M (1, DVI_File), DVI_File);
			break;

		case BEGIN_REFLECT:
			if (Reflection_Level == Start_Level) {
				Save_Font = Current_Font;
				Save_Position.H = HH; Save_Position.V = VV;
				Save_Position.W = WW; Save_Position.X = XX;
				Save_Position.Y = YY; Save_Position.Z = ZZ;
				Save_Stack_Ptr = Stack_Ptr;
				Save_Addr = File_Position_M (DVI_File);
				if (trace_dvi)
					printf ("%s: h = %d v = %d.\n", Literal_Commands_B[BEGIN_REFLECT-FNT1],
						HH, VV);
			}
			Reflection_Level++;
			goto Set_Reflect_Font;

		case END_REFLECT:
			if (Reflection_Level == Start_Level) {
				if (Start_Level > 0) {	/* Acts like EOP in this case */
					Set_String (Current_Font);
					return (1);
				} else
					Message (DVIOUT_REFLECTION);
			} else if (--Reflection_Level == Start_Level) {
				if (Save_Stack_Ptr != Stack_Ptr)
					Message (DVIOUT_REFLECTION);
				Save_Position.H = HH;
				if (Execute_DVI (&Save_Position, Save_Stack_Ptr-Stack, Save_Font,
						 Start_Level+1, Save_Addr, DVI_File) == 0)
					return (0);
				Trace_Mask = SET_H | SET_V;
			}
Set_Reflect_Font:	Set_String (Current_Font);
			Current_Font = Reset_Font (Current_Font, Reflection_Level);
			break;

		default:
			return (0);
		}
		if (trace_dvi && Reflection_Level == Start_Level) {
			if (Command < SET1)
				printf ("Set_Char_%u", Command-SET_CHAR_0);
			else if (Command < FNT_NUM_0)
				printf ("%s", Literal_Commands_A[Command-SET1]);
			else if (Command < FNT1)
				printf ("Fnt_Num_%u", Command-FNT_NUM_0);
			else if (Command < XXX1 || Command > XXX1+3)
				printf ("%s", Literal_Commands_B[Command-FNT1]);
			if (Trace_Mask != 0) {
				printf (":");
				if ((Trace_Mask & SET_H) != 0)
					printf (" h = %d", HH);
				if ((Trace_Mask & SET_V) != 0)
					printf (" v = %d", VV);
				if ((Trace_Mask & SET_W) != 0)
					printf (" w = %d", WW);
				if ((Trace_Mask & SET_X) != 0)
					printf (" x = %d", XX);
				if ((Trace_Mask & SET_Y) != 0)
					printf (" y = %d", YY);
				if ((Trace_Mask & SET_Z) != 0)
					printf (" z = %d", ZZ);
				if ((Trace_Mask & SET_STACK) != 0)
					printf (" sp = %d", &Stack[Maximum_Stack_Depth]-Stack_Ptr);
			}
			printf (".\n");
		}
	}
	return ((Interrupt_Flag == 0) ? 0 : 1);
}
#undef Add_to_H

/*
 *	Initialize run parameters:
 */

Initialize ()
{
	Mem_Init ();
	Build_Ord ();
}

Initialize_For_File ()
{
	Font_Def_Head = 0;
	Max_N_Primitives = 0;
	Page_Index = 0;
}

/*
 *	Routine Process_Preamble reads in the preamble and stores the
 *	information for later use.
 */

int Process_Preamble (DVI_File)
pointer DVI_File;
{
	auto   unsigned char K;

	Set_File_Position_M (0, DVI_File);
	if (Read_Character_M (DVI_File) != PRE || File_At_EOF_M (DVI_File) != 0 ||
	    (DVI_Format = Read_Unsigned_Int_M (1, DVI_File)) == 0 || File_At_EOF_M (DVI_File) != 0 ||
	    (DVI_Fraction.Numerator = Read_Unsigned_Int_M (4, DVI_File)) == 0 || File_At_EOF_M (DVI_File) != 0 ||
	    (DVI_Fraction.Denominator = Read_Unsigned_Int_M (4, DVI_File)) == 0 || File_At_EOF_M (DVI_File) != 0 ||
	    (DVI_Magnification = Read_Unsigned_Int_M (4, DVI_File)) == 0 || File_At_EOF_M (DVI_File) != 0)
		return (0);
	if ((K = Read_Unsigned_Int_M (1, DVI_File)) != 0)
		Read_Block_M (Scratch_String, K, DVI_File);
	Scratch_String[K] = '\0';
	if ((Local_Options.Flags & MAGNIFICATION) == 0)
		Job_Magnification = DVI_Magnification;
	else if ((Local_Options.Flags & CONCATENATE) == 0)
		Job_Magnification = XN_Div_D_R_M (Local_Options.Magnification, DVI_Magnification, 1000);
	else
		Job_Magnification = Local_Options.Magnification;
	if (trace_dvi) {
		printf ("DVI File Format is %u\n", DVI_Format);
		printf ("Scale ratio is %u / %u\n", DVI_Fraction.Numerator, DVI_Fraction.Denominator);
		printf ("Magnification is %u\n", DVI_Magnification);
		printf ("Comment is \"%s\"\n", Scratch_String);
	}
	if (File_At_EOF_M (DVI_File) != 0)
		return (0);
	return (1);
}

/*
 *	Routine Process_Postamble reads in the postamble file and
 *	stores the information found for later use, checks various
 *	values against those found in the Preamble, and reads in
 *	the Font definitions into the temporary font definition
 *	list.
 */

int Process_Postamble (DVI_File)
pointer DVI_File;
{
	auto   unsigned long Postamble_Addr;
	extern unsigned long Find_Postamble();
	extern int Process_Font_Definitions();

	if ((Postamble_Addr = Find_Postamble (DVI_File)) == 0 ||
	    Set_File_Position_M (Postamble_Addr, DVI_File) != Postamble_Addr ||
	    Read_Character_M (DVI_File) != POST || File_At_EOF_M (DVI_File) != 0 ||
	    (Last_Page_Addr = Read_Unsigned_Int_M (4, DVI_File)) == 0 || File_At_EOF_M (DVI_File) != 0 ||
	    Read_Unsigned_Int_M (4, DVI_File) != DVI_Fraction.Numerator || File_At_EOF_M (DVI_File) != 0 ||
	    Read_Unsigned_Int_M (4, DVI_File) != DVI_Fraction.Denominator || File_At_EOF_M (DVI_File) != 0 ||
	    Read_Unsigned_Int_M (4, DVI_File) != DVI_Magnification || File_At_EOF_M (DVI_File) != 0)
		return (0);
	Tallest_Page_Height = Read_Unsigned_Int_M (4, DVI_File);
	Widest_Page_Width = Read_Unsigned_Int_M (4, DVI_File);
	Maximum_Stack_Depth = Read_Unsigned_Int_M (2, DVI_File);
	Number_of_Pages = Read_Unsigned_Int_M (2, DVI_File);
	if (trace_dvi) {
		printf ("Tallest page height is %u\n", Tallest_Page_Height);
		printf ("Widest page width is %u\n", Widest_Page_Width);
		printf ("Maximum stack depth is %u\n", Maximum_Stack_Depth);
		printf ("Number of pages is %u\n", Number_of_Pages);
	}
	if (File_At_EOF_M (DVI_File) != 0)
		return (0);
	return (Process_Font_Definitions (DVI_File));
}

/*
 *	Routine Find_Postamble locates the Postamble of the DVI
 *	file. This is done by setting to the end of file and reading
 *	backwards until the postamble address is found.
 */

unsigned long Find_Postamble (DVI_File)
pointer DVI_File;
{
	auto   unsigned long Postamble_Address;
	auto   unsigned int Sig_Count;
	auto   unsigned char Value;

	Set_EOF_M (DVI_File);
	Sig_Count = 0;
	while ((Value = Read_Character_Reverse_M (DVI_File)) == SIGNATURE)
		Sig_Count++;
	if (Value != DVI_Format)
		return (0);
	Postamble_Address = Read_Unsigned_Int_Reverse_M (4, DVI_File);
	if (trace_dvi) {
		printf ("Number of %d's found: %u\n", SIGNATURE, Sig_Count);
		printf ("Postamble address is %u\n", Postamble_Address);
	}
	return (Postamble_Address);
}

/*
 *	Routine Process_Font_Definitions reads in the font definitions
 *	in the Postamble into a temporary linked list of entries to be
 *	loaded later.
 */

int Process_Font_Definitions (DVI_File)
pointer DVI_File;
{
	auto   struct Font_Definition *Font_Ptr;
	auto   unsigned long *Ptr;
	auto   int Index;
	auto   unsigned char Command, A, L;
	static unsigned long Param_Array[4];
	extern struct Encoding Identity_Encoding;
	extern char *Mem_Alloc();
	extern int strcmp();

	Font_Def_Head = 0;
	Font_Count = 0;
	Max_Font_Number = 0;
	while (File_At_EOF_M (DVI_File) == 0 &&
	    (Command = Read_Character_M (DVI_File)) != POST_POST) {
		if (Command == NOP)
			continue;
		else if (Command < FNT_DEF1 || Command > FNT_DEF1 + 3)
			break;
		Ptr = &Param_Array[0];
		*Ptr++ = Read_Unsigned_Int_M (Command-FNT_DEF1+1, DVI_File);
		*Ptr++ = Read_Unsigned_Int_M (4, DVI_File);
		*Ptr++ = Read_Unsigned_Int_M (4, DVI_File);
		*Ptr++ = Read_Unsigned_Int_M (4, DVI_File);
		A = Read_Unsigned_Int_M (1, DVI_File);
		L = Read_Unsigned_Int_M (1, DVI_File);
		Font_Ptr = (struct Font_Definition *) Mem_Alloc (sizeof (struct Font_Definition) + A + L + 2);
		Font_Ptr->Font_Directory = (struct Char_Definition **)
					    Mem_Alloc (CHARS_PER_FONT * sizeof (struct Char_Definition *));
		Font_Ptr->Char_Dir_Count = CHARS_PER_FONT;
		Font_Ptr->Link = Font_Def_Head;
		Font_Def_Head = Font_Ptr;
		Font_Ptr->Font_Index = Font_Count++;
		Font_Ptr->Design_Size = *--Ptr;
		Font_Ptr->At_Size = *--Ptr;
		Font_Ptr->Checksum = *--Ptr;
		if ((Font_Ptr->Font_Number = (long) *--Ptr) > Max_Font_Number)
			Max_Font_Number = Font_Ptr->Font_Number;
		Font_Ptr->Magnification = XN_Div_D_R_M (Font_Ptr->At_Size, Job_Magnification,
							Font_Ptr->Design_Size);
		Font_Ptr->Area_Ptr = &Font_Ptr->Name[0];
		Font_Ptr->Name_Ptr = &Font_Ptr->Name[A+1];
		if (A != 0)
			Read_Block_M (Font_Ptr->Area_Ptr, A, DVI_File);
		Read_Block_M (Font_Ptr->Name_Ptr, L, DVI_File);
		Font_Ptr->Name[A] = '\0';
		Font_Ptr->Name[A+L+1] = '\0';
		for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
			Font_Ptr->Font_Directory[Index] = 0;
		Font_Ptr->Flags = 0;
		Font_Ptr->Font_Encoding = &Identity_Encoding;
		if (trace_dvi) {
			printf ("Font Number %u:\n", Font_Ptr->Font_Number);
			printf ("  Checksum is %u\n", Font_Ptr->Checksum);
			printf ("  Design size is %u\n", Font_Ptr->Design_Size);
			printf ("  At size is %u\n", Font_Ptr->At_Size);
			printf ("  Font area is \"%s\"\n", Font_Ptr->Area_Ptr);
			printf ("  Font name is \"%s\"\n", Font_Ptr->Name_Ptr);
		}
		fold_m (Font_Ptr->Area_Ptr);
		fold_m (Font_Ptr->Name_Ptr);
	}
	if (File_At_EOF_M (DVI_File) != 0 || Command != POST_POST)
		return (0);
/*
 *	If required, construct the MARK font (font used for outputting
 *	top and bottom page marks) as if it were in the postamble font
 *	list (note that metric information has to be filled in when the
 *	pixel file for the font is read in):
 */
	if ((Local_Options.Flags & MARK) != 0) {
		fold_m (Local_Options.Mark_Font_Name);
		for (Font_Ptr = Font_Def_Head; Font_Ptr != 0; Font_Ptr = Font_Ptr->Link)
		if (strcmp (Font_Ptr->Name_Ptr, Local_Options.Mark_Font_Name) == 0 &&
			    Font_Ptr->Magnification == Local_Options.Mark_Font_Scale) {
			Mark_Font_Number = Font_Ptr->Font_Number;
			Font_Ptr->Flags |= MARK_FONT;
			return (1);
		}
		Font_Ptr = (struct Font_Definition *) Mem_Alloc (sizeof (struct Font_Definition) + 1);
		Font_Ptr->Font_Directory = (struct Char_Definition **)
					    Mem_Alloc (CHARS_PER_FONT * sizeof (struct Char_Definition *));
		Font_Ptr->Char_Dir_Count = CHARS_PER_FONT;
		Font_Ptr->Link = Font_Def_Head;
		Font_Def_Head = Font_Ptr;
		Font_Ptr->Font_Index = Font_Count++;
		Font_Ptr->Design_Size = 0;
		Font_Ptr->At_Size = 0;
		Font_Ptr->Checksum = 0;
		Font_Ptr->Font_Number = Mark_Font_Number = ++Max_Font_Number;
		Font_Ptr->Magnification = Local_Options.Mark_Font_Scale;
		Font_Ptr->Name[0] = '\0';
		Font_Ptr->Area_Ptr = &Font_Ptr->Name[0];
		Font_Ptr->Name_Ptr = &Local_Options.Mark_Font_Name[0];
		for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
			Font_Ptr->Font_Directory[Index] = 0;
		Font_Ptr->Flags = MARK_FONT;
		Font_Ptr->Font_Encoding = &Identity_Encoding;
	}
	return (1);
}

/*
 *	Routine Generate_Page_Index builds the page index, each
 *	entry of which points to the file address for that page.
 *	It allows us to output (and optimize for) partial output
 *	of the DVI file. It will optionally sort the index in order
 *	of increasing page number.
 */

int Generate_Page_Index (DVI_File, Sort)
pointer DVI_File;
int Sort;
{
	auto   unsigned long Addr;
	auto   int Index, Count;
	static char Page_Str[32];
	extern char *Mem_Alloc();
/*
 *	Allocate storage to store the Index:
 */
	Page_Index = (struct Page_Index_Entry *)
		Mem_Alloc (Number_of_Pages * sizeof (struct Page_Index_Entry));
/*
 *	Scan through the file backwards beginning at the last page
 *	index found in the postamble, recording the file address
 *	for the BOP at the beginning of each page.
 */
	Addr = Last_Page_Addr;
	Index = Number_of_Pages;
	while (Index > 0 && Addr != 0xFFFFFFFF) {
		Set_File_Position_M (Addr, DVI_File);
		Page_Index[--Index].Page_Addr = Addr;
		if (File_At_EOF_M (DVI_File) != 0 || Read_Character_M (DVI_File) != BOP)
			break;
		for (Count = 0; Count < 10; Count++)
			Page_Index[Index].Page_No[Count] = Read_Signed_Int_M (4, DVI_File);
		Addr = Read_Unsigned_Int_M (4, DVI_File);
		if (File_At_EOF_M (DVI_File) != 0)
			break;
		if (trace_dvi) {
			Format_Page_Number (Page_Index[Index].Page_No, Page_Str, sizeof (Page_Str));
			printf ("Page Address of page %s is %u\n", Page_Str, Page_Index[Index].Page_Addr);
		}
	}
/*
 *	Set return value. At the successful completion of the above
 *	loop, the file address should be -1 and the page index should
 *	be zero.
 */
	if (Addr != 0xFFFFFFFF || Index != 0)
		return (0);
	if (Sort != 0)
		Sort_Page_Index (Page_Index, Number_of_Pages);
	return (1);
}

/*
 *	Routine Sort_Page_Index uses a Shell sort to sort the list
 *	of pages in non-decreasing order.
 */

Sort_Page_Index (Page_Array, Count)
struct Page_Index_Entry Page_Array[];
unsigned long Count;
{
	auto   unsigned long n, i, j, k, gap;
	static struct Page_Index_Entry Temp_Page_Entry;
	extern int Compare_Page_Number();

	n = Count;
	for (gap = n >> 1; gap > 0; gap >>= 1)
		for (i = gap; i < n; i++)
			for (j = i - gap; j >= 0; j -= gap) {
				k = j + gap;
				if (Compare_Page_Number (Page_Array[j].Page_No, Page_Array[k].Page_No) <= 0)
					break;
				Temp_Page_Entry = Page_Array[j];
				Page_Array[j] = Page_Array[k];
				Page_Array[k] = Temp_Page_Entry;
			}
}

Format_Page_Number (Page_Number, Str, Str_Size)
long Page_Number[10];
char *Str;
unsigned short Str_Size;
{
	auto   unsigned int Index, Jndex;
	static char Temp_Str[12];

	*Str = '\0';
	for (Jndex = 10; Jndex > 1 && Page_Number[Jndex-1] == 0; Jndex--)
		;
	for (Index = 0; Index < Jndex; Index++) {
		sprintf (Temp_Str, (Index == 0) ? "%d" : ".%d", Page_Number[Index]);
		stringcat_m (Str, Temp_Str, Str_Size);
	}
}

/*
 *	Set_Font locates the specified font number and returns its
 *	index. Zero is returned if it cannot be found. If the font
 *	is a mirror-image font and it does not yet exist, it is
 *	created (normally, this should happen only during pass 1).
 */

struct Font_Definition *Set_Font (Font_Num, Reflection_Level)
unsigned long Font_Num;
unsigned int Reflection_Level;
{
	auto   struct Font_Definition *Font_Ptr, *New_Font_Ptr;
	auto   long F;
	auto   int Index;
	extern char *Mem_Alloc();

	F = (long) Font_Num;
	if ((Reflection_Level & 0x01) != 0)
		F = ~F;
	for (Font_Ptr = Font_Def_Head; Font_Ptr != 0; Font_Ptr = Font_Ptr->Link)
	if (Font_Ptr->Font_Number == F)
		return (Font_Ptr);
	if ((Reflection_Level & 0x01) != 0) {	/* Mirror-image font */
		for (Font_Ptr = Font_Def_Head; Font_Ptr != 0; Font_Ptr = Font_Ptr->Link)
		if (Font_Ptr->Font_Number == Font_Num) {
			New_Font_Ptr = (struct Font_Definition *) Mem_Alloc (sizeof (struct Font_Definition));
			*New_Font_Ptr = *Font_Ptr;	/* copy entire structure */
			New_Font_Ptr->Font_Directory = (struct Char_Definition **)
						       Mem_Alloc (New_Font_Ptr->Char_Dir_Count *
									sizeof (struct Char_Definition *));
			Font_Ptr->Link = New_Font_Ptr;
			New_Font_Ptr->Font_Number = F;
			New_Font_Ptr->Font_Index = Font_Count++;
			New_Font_Ptr->Flags = MIRROR_FONT;
			for (Index = 0; Index < New_Font_Ptr->Char_Dir_Count; Index++)
				New_Font_Ptr->Font_Directory[Index] = 0;
			return (New_Font_Ptr);
		}
	}
	return (0);
}

/*
 *	Reset_Font is like Set_Font, except it is specifically invoked
 *	when the Reflection Level has changed.
 */

struct Font_Definition *Reset_Font (Font_Ptr, Reflection_Level)
struct Font_Definition *Font_Ptr;
unsigned int Reflection_Level;
{
	auto   struct Font_Definition *New_Font_Ptr;
	extern struct Font_Definition *Set_Font();

	if ((New_Font_Ptr = Font_Ptr) != 0)
		New_Font_Ptr = Set_Font (((New_Font_Ptr->Flags & MIRROR_FONT) == 0) ?
					 New_Font_Ptr->Font_Number : ~New_Font_Ptr->Font_Number,
					 Reflection_Level);
	return (New_Font_Ptr);
}

/*
 *	Routine Output_Mark outputs the page mark at the top and
 *	bottom of the page. The top mark is positioned so that the
 *	character baselines are 1/2 inch from the top of the page
 *	and the bottom mark baselines are 1/4 inch from the bottom
 *	of the page. Both marks are centered horizontally on the
 *	page.
 */

Output_Mark (Font_Ptr, Mark_String, Mark_Width, Form_Width, Form_Length)
struct Font_Definition *Font_Ptr;
char *Mark_String;
long Mark_Width;
struct Ratio *Form_Width, *Form_Length;
{
	auto   struct Ratio *Resolution;
	auto   long Width, Length, H_Pos, Top_Base;
	extern long Convert(), Unconvert();
/*
 *	Convert form width and length from inches to 1/1024 pixels;
 *	compute center of page and baseline positions for top and
 *	bottom marks in de-magnified, untranslated DVI units (required
 *	because they are re-magnified and translated later on):
 */
	Resolution = &Dev_Ptr->Device_Resolution;
	Width = XN_Div_D_T_M (XN_Div_D_T_M (1024, Form_Width->Numerator, Form_Width->Denominator),
			      Resolution->Numerator, Resolution->Denominator);
	Length = XN_Div_D_T_M (XN_Div_D_T_M (1024, Form_Length->Numerator, Form_Length->Denominator),
			       Resolution->Numerator, Resolution->Denominator);
	H_Pos = Unconvert (((Width - Convert (Mark_Width) + 1) >> 1) - H0);
	Top_Base = XN_Div_D_T_M (512, Resolution->Numerator, Resolution->Denominator); /* That's 1/2 inch */
	Set_Text (Font_Ptr, H_Pos, Unconvert (Top_Base - V0), Mark_String);
	Set_Text (Font_Ptr, H_Pos, Unconvert (Length - (Top_Base >> 1) - V0), Mark_String);
	Set_String (Font_Ptr);
}

Set_Text (Font_Ptr, Hor_Pos, Vert_Pos, String)
struct Font_Definition *Font_Ptr;
long Hor_Pos, Vert_Pos;
char *String;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   char *Ptr;
	auto   long H_Pos, Space_Width;
	auto   char c;
	extern unsigned char XOrd[];
	extern struct Char_Definition *Get_Character();

	Space_Width = XN_Div_D_T_M (Font_Ptr->Design_Size, 4, 10);
	H_Pos = Hor_Pos;
	for (Ptr = String; (c = *Ptr) != '\0'; Ptr++) {
		if (c == ' ')
			H_Pos += Space_Width;
		else if ((Char_Ptr = Get_Character (Font_Ptr, (unsigned long) XOrd[c])) != 0) {
			Set_Character (Font_Ptr, H_Pos, Vert_Pos, Char_Ptr, 0);
			H_Pos += Char_Ptr->DVI_Width;
		}
	}
}

struct Char_Definition *Get_Character (Font_Ptr, Char_Code)
struct Font_Definition *Font_Ptr;
unsigned long Char_Code;
{
	auto   struct Char_Definition *Char_Ptr;

	if (Font_Ptr == 0 || Char_Code >= CHARS_PER_FONT ||
	    (Char_Ptr = Font_Ptr->Font_Directory[Char_Code]) == 0)
		return (0);
	return (Char_Ptr);
}

/*
 *	Set_Character sets the specified character at the specified
 *	position. Characters are not output right away, but rather are
 *	accumulated for output by 'Set_String'; this significantly
 *	reduces the number of times the driver routines need to be
 *	called.
 */

Set_Character (Font_Ptr, Hor_Pos, Vert_Pos, Char_Ptr, Reflection_Level)
struct Font_Definition *Font_Ptr;
long Hor_Pos, Vert_Pos;
struct Char_Definition *Char_Ptr;
{
	auto   long X_Pos, Y_Pos;
	extern long Convert();

	if (Char_Ptr->Driver_Id != 0) {
		if (String_Size >= MAX_STRING_SIZE ||
		    (String_Size > 0 && (Vert_Pos != String_Y || Hor_Pos != Next_String_X)))
			Set_String (Font_Ptr);
		if (String_Size == 0) {
			String_X = Hor_Pos;
			String_Y = Vert_Pos;
		}
		Char_Vector[String_Size++] = Char_Ptr;
		if ((Reflection_Level & 0x01) == 0)
			Next_String_X = Hor_Pos + Char_Ptr->DVI_Width;
		else
			Next_String_X = Hor_Pos - Char_Ptr->DVI_Width;
	} else if (Char_Ptr->Pixel_Width != 0 && Char_Ptr->Pixel_Height != 0) {
		X_Pos = Convert (Hor_Pos) + H0;
		Y_Pos = Convert (Vert_Pos) + V0;
		(*Dev_Ptr->Typeset_Pixel) (X_Pos - (Char_Ptr->X_Origin << 10),
					   Y_Pos - (Char_Ptr->Y_Origin << 10),
					   Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
					   Char_Ptr->Pixel_Array);
	}
}

/*
 *	Routine Set_String typesets the current accumulated string
 *	of characters, all in the same font and adjacent to each
 *	other:
 */

Set_String (Font_Ptr)
struct Font_Definition *Font_Ptr;
{
	auto   long X_Pos, Y_Pos;
	extern long Convert();

	if (Font_Ptr != 0 && String_Size > 0) {
		X_Pos = Convert (String_X) + H0;
		Y_Pos = Convert (String_Y) + V0;
		(*Dev_Ptr->Typeset_String) (X_Pos, Y_Pos, Font_Ptr,
					    Char_Vector, String_Size);
	}
	String_Size = 0;
}

/*
 *	Routine Set_Rule typesets a rule at the specified position
 *	and size.
 */

Set_Rule (Hor_Pos, Vert_Pos, Height, Width, Reflection_Level)
long Hor_Pos, Vert_Pos, Height, Width;
int Reflection_Level;
{
	extern long Convert();

	if (Width > 0 && Height > 0)
		(*Dev_Ptr->Typeset_Rule) ((((Reflection_Level & 0x01) == 0) ?
						Convert (Hor_Pos) : Convert (Hor_Pos-Width)) + H0,
					  Convert (Vert_Pos) + V0,
					  Convert (Width), Convert (Height));
}

/*
 *	Routine Skip skips a specified number of characters in the input
 *	file.
 */

Skip (N_Chars, File)
unsigned int N_Chars;
unsigned long File;
{
	Set_File_Position_M (File_Position_M (File) + N_Chars, File);
}

/*
 *	Routine Convert converts a dimension in the DVI file to
 *	raster units * 1024. Convert_True does a similar thing, but
 *	excludes the /MAGNIFICATION. Routine Unconvert is the inverse
 *	of Convert.
 */

long Convert (Value)
long Value;
{
	return (XN_Div_D_T_M (XN_Div_D_T_M (XN_Div_D_T_M (Value, DVI_Fraction.Numerator,
							  DVI_Fraction.Denominator),
					    Job_Magnification, 1000),
			      Dev_Res.Numerator, Dev_Res.Denominator));
}

long Unconvert (Value)
long Value;
{
	return (XN_Div_D_T_M (XN_Div_D_T_M (XN_Div_D_T_M (Value, DVI_Fraction.Denominator,
							  DVI_Fraction.Numerator),
					    1000, Job_Magnification),
			      Dev_Res.Denominator, Dev_Res.Numerator));
}

long Convert_True (Value)
long Value;
{
	return (XN_Div_D_T_M (XN_Div_D_T_M (Value, DVI_Fraction.Numerator, DVI_Fraction.Denominator),
			      Dev_Res.Numerator, Dev_Res.Denominator));
}

/*
 *	Routine Record_String records the characters in a string
 *	during pass 1 processing.
 */

unsigned long Record_String (Font_Ptr, String)
struct Font_Definition *Font_Ptr;
char *String;
{
	auto   char *Ptr;
	auto   unsigned long Count;
	auto   char c;
	extern unsigned char XOrd[];

	Count = 0;
	for (Ptr = String; (c = *Ptr) != '\0'; Ptr++)
	if (c != ' ') {
		Record_Character (Font_Ptr, (unsigned long) XOrd[c]);
		Count++;
	}
	return (Count);
}

/*
 *	This routine, called during pass one upon a 'set_char' command,
 *	increments the frequency count for the character for optimal
 *	downloading prior to pass two.
 *
 *	Important Note: the Font_Directory array, which is an array
 *	of pointers, is used (somewhat recklessly) during pass one to
 *	store unsigned integers. My assumptions regarding the usage of
 *	pointers for alternate purposes are:
 *
 *	  a) A pointer will fit inside an unsigned long, and
 *	  b) An unsigned or signed int will fit inside a pointer.
 *
 *	All this to save one longword in the Char_Definition's.
 *	The value is transferred to the Driver_Id longword when
 *	the character descriptor is created during font downloading.
 */

Record_Character (Font_Ptr, Char_Code)
struct Font_Definition *Font_Ptr;
unsigned long Char_Code;
{
	auto   unsigned int Count;

	if (Font_Ptr != 0 && Char_Code < CHARS_PER_FONT) {
		Count = (unsigned int) Font_Ptr->Font_Directory[Char_Code];
		if (Count < 65535)
			Count++;
		Font_Ptr->Font_Directory[Char_Code] = (struct Char_Definition *) Count;
	}
}

/*
 *	The following routines are dummy routines for AST queueing
 *	that are not needed in this program:
 */

EnQueue_AST_Locked (Type, Arg1, Arg2, Priority)
unsigned long Type, Arg1, Arg2, Priority;
{
	EnQueue_AST (Type, Arg1, Arg2, Priority);
}

EnQueue_AST (Type, Arg1, Arg2, Priority)
unsigned long Type, Arg1, Arg2, Priority;
{

}
