%{
/* This lex file needs a lex that can work with 8 bit characters
   i.e. flex 2.3 */

#include <stdio.h>

#define B7_IN    "\033%/3"
#define B7_NAME  "BIG5"
#define B7_DUMMY '='
#define B7_SEVEN '7'
#define B7_EIGHT '8'

#define G7_IN   "\033$(A"

#define OUT  "\033(B"

#ifdef STX
#undef STX
#endif

#define STX  0x02

#define ENCODE 0
#define DECODE 1

#define GUOBIAO 0
#define BIG5    1

static int state = DECODE,
           encoding = BIG5,
           in7 = 0;

static char *program;

%}

%%

[\xa1-\xfe][\x40-\xfe]+ {
    int i, len;

    if (state == ENCODE) {
        len = strlen(yytext);
        switch(encoding) {
          case BIG5:
            printf("%s%c%c%s%c", B7_IN, B7_DUMMY, B7_DUMMY, B7_NAME, STX);
            break;
          case GUOBIAO:
            printf("%s", G7_IN);
            break;
        }
        for (i=0; i < len; i += 2) {
            putchar(yytext[i] & 0x7f); /* First byte always 8 bit */
            switch(encoding) {
              case BIG5:
                if (0x7f <= yytext[i+1] && yytext[i+1] < 0xa0) {
                    fprintf(stderr, "%s: illegal Big5 QuWei 0x%02x%02x\n",
                            program, yytext[i] & 0xff, yytext[i+1] & 0xff);
                    exit(1);
                }
                if (yytext[i+1] & 0x80) {
                    putchar(B7_EIGHT);
                    putchar(yytext[i+1] & 0x7f);
                } else {
                    putchar(B7_SEVEN);
                    putchar(yytext[i+1]);
                }
                break;
              case GUOBIAO:
                if (yytext[i+1] < 0xa1 || yytext[i+1] >= 0xff) {
                    fprintf(stderr, "%s: illegal GuoBiao QuWei 0x%02x%02x\n",
                            program, yytext[i] & 0xff, yytext[i+1] & 0xff);
                    exit(1);
                }
                putchar(yytext[i+1] & 0x7f);
                break;
            }
        }
        printf("%s", OUT);
    } else
      ECHO;
}

"\x1b%/3"[\x01-\x7f][\x01-\x7f][a-zA-Z0-9]+\x02 {
    if (encoding == BIG5 && state == DECODE)
        in7 = 1;
    else
      ECHO;
}

[\x21-\x7e][\x07\x08][\x01-\x7f] {
    if (encoding == BIG5 && state == DECODE) {
        if (!in7) {
            fprintf(stderr, "%s: missing B7 escape sequence at string start\n",
                    program);
            if (yyin != stdin)
              fclose(yyin);
            exit(1);
        }
        putchar(yytext[0] | 0x80);
        if (yytext[1] == 0x08)
          putchar(yytext[2] | 0x80);
        else
          putchar(yytext[2]);
    } else
      ECHO;
}

"\033$(A" {
    if (encoding == GUOBIAO && state == DECODE)
        in7 = 1;
    else
      ECHO;
}

[\x21-\x7e][\x21-\x7e] {
    if (encoding == GUOBIAO && state == DECODE) {
        if (!in7) {
            fprintf(stderr, "%s: missing G7 escape sequence at string start\n",
                    program);
            if (yyin != stdin)
              fclose(yyin);
            exit(1);
        }
        putchar(yytext[0] | 0x80);
        putchar(yytext[1] | 0x80);
    } else
      ECHO;
}

"\x1b(B" {
    if (state == DECODE)
      in7 = 0;
    else
      ECHO;
}

%%

main(argc, argv)
int argc;
char **argv;
{
    FILE *f;

    program = argv[0];
    while(*++argv) {
        if (argv[0][0] == '-') {
            switch(argv[0][1]) {
              case 'g': case 'G':
                encoding = GUOBIAO;
                state = (argv[0][2] && argv[0][2] == '7') ? ENCODE : DECODE;
                break;
              case 'b': case 'B':
                encoding = BIG5;
                state = (argv[0][2] && argv[0][2] == '7') ? ENCODE : DECODE;
                break;
            }
        } else {
            if (!(yyin = fopen(argv[0], "r"))) {
                fprintf(stderr, "%s: problem with file %s\n",
                        program, argv[0]);
                exit(1);
            }
            yylex();
            fclose(f);
        }
        argc--;
    }
    if (argc == 1) {
        yyin = stdin;
        yylex();
    }
}
