// File          : VectorizeBitmapDialog.java
// Description   : Dialog box for vectorizing a bitmap
// Date          : 25th May 2011
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog box for scanning a bitmap.
 * @author Nicola L C Talbot
 */
public class VectorizeBitmapDialog extends JDialog
   implements ActionListener
{
   public VectorizeBitmapDialog(JpgfDraw application)
   {
      super(application, JDRResources.getString("vectorize.title"),
            true);
      application_ = application;

      Box p = Box.createVerticalBox();

      JPanel incPanel = new JPanel();

      xIncPanel = new NonNegativeIntPanel("vectorize.xinc", 2);
      xIncPanel.add(Box.createHorizontalGlue());
      incPanel.add(xIncPanel);

      yIncPanel = new NonNegativeIntPanel("vectorize.yinc", 2);
      yIncPanel.add(Box.createHorizontalGlue());
      incPanel.add(yIncPanel);

      p.add(incPanel);

      JComponent straightenPanel = Box.createVerticalBox();

      straightenPanel.setBorder(BorderFactory.createEtchedBorder());

      straightenBox = new JCheckBox(
         JDRResources.getString("vectorize.straighten"), false);

      straightenBox.addActionListener(this);
      straightenBox.setActionCommand("straighten");

      straightenPanel.add(straightenBox);

      tolerance = new NonNegativeDoublePanel(
         "vectorize.tolerance", 1.0);

      straightenPanel.add(tolerance);

      p.add(straightenPanel);

      JComponent smoothPanel = Box.createVerticalBox();
      smoothPanel.setBorder(BorderFactory.createEtchedBorder());

      smoothBox = new JCheckBox(
         JDRResources.getString("vectorize.smooth"), true);
      smoothBox.addActionListener(this);
      smoothBox.setActionCommand("smooth");

      smoothPanel.add(smoothBox);

      JComponent smoothParametersPanel = new JPanel();

      smoothParametersPanel.setLayout(new GridLayout(5, 2));

      chi = new NonNegativeDoublePanel("vectorize.chi", 2.0);

      smoothParametersPanel.add(chi);

      delta = new NonNegativeDoublePanel("vectorize.delta", 0.01);

      smoothParametersPanel.add(delta);

      gamma = new NonNegativeDoublePanel("vectorize.gamma", 0.5);

      smoothParametersPanel.add(gamma);

      rho = new NonNegativeDoublePanel("vectorize.rho", 1.0);

      smoothParametersPanel.add(rho);

      sigma = new NonNegativeDoublePanel("vectorize.sigma", 0.5);

      smoothParametersPanel.add(sigma);

      smoothTolFun = new NonNegativeDoublePanel(
         "vectorize.smoothTolFun", 1e-6);

      smoothParametersPanel.add(smoothTolFun);

      smoothTol = new NonNegativeDoublePanel(
         "vectorize.smoothTol", 1e-6);

      smoothParametersPanel.add(smoothTol);

      maxIter = new NonNegativeIntPanel("vectorize.maxIter", 200);

      smoothParametersPanel.add(maxIter);

      maxFunEvals = new NonNegativeIntPanel(
         "vectorize.maxFunEvals", 1000);

      smoothParametersPanel.add(maxFunEvals);

      flatness = new NonNegativeDoublePanel("vectorize.flatness", 1.0);

      smoothParametersPanel.add(flatness);

      smoothPanel.add(smoothParametersPanel);

      p.add(smoothPanel);

      JPanel colPanel = new JPanel();

      colPanel.add(new JLabel(JDRResources.getString("vectorize.base")));

      base = Color.black;

      swatch = new JPanel()
      {
         public Dimension getPreferredSize()
         {
            return new Dimension(40,20);
         }

         public Color getBackground()
         {
            return base;
         }
      };

      colPanel.add(swatch);

      JButton colorChooserButton 
         = new JButton(JDRResources.getString("vectorize.selectbase"));
      colorChooserButton.setMnemonic(
         JDRResources.getChar("vectorize.selectbase.mnemonic"));
      colorChooserButton.addActionListener(this);
      colorChooserButton.setActionCommand("selectBase");
      colPanel.add(colorChooserButton);

      p.add(colPanel);

      colorChooser = new JColorChooser(swatch.getBackground());

      colorChooserDialog = JColorChooser.createDialog(application,
         JDRResources.getString("vectorize.selectbase.title"),
         true,
         colorChooser,
         new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               base = colorChooser.getColor();
               swatch.repaint();
            }
         },
         new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
            }
         }
      );

      getContentPane().add(p, "Center");

      JPanel p2 = new JPanel();

      p2.add(JDRResources.createOkayButton(this));
      p2.add(JDRResources.createCancelButton(this));

      getContentPane().add(p2, "South");
      pack();
      setLocationRelativeTo(application);
   }

   public void display()
   {
      mainPanel = application_.getCurrentFrame();
      bitmap = mainPanel.getSelectedBitmap();
      setVisible(true);
   }

   private void setStraighten(boolean flag)
   {
      if (straightenBox.isSelected() != flag)
      {
         straightenBox.setSelected(flag);
      }

      tolerance.setEnabled(flag);
   }

   private void setSmooth(boolean flag)
   {
      if (smoothBox.isSelected() != flag)
      {
         smoothBox.setSelected(flag);
      }

      chi.setEnabled(flag);
      delta.setEnabled(flag);
      gamma.setEnabled(flag);
      rho.setEnabled(flag);
      sigma.setEnabled(flag);
      smoothTolFun.setEnabled(flag);
      smoothTol.setEnabled(flag);
      maxIter.setEnabled(flag);
      maxFunEvals.setEnabled(flag);
      flatness.setEnabled(flag);
   }

   public int getXInc()
   {
      return xIncPanel.getValue();
   }

   public int getYInc()
   {
      return yIncPanel.getValue();
   }

   public Color getBase()
   {
      return base;
   }

   public double getStraightenTolerance()
   {
      return tolerance.getValue();
   }

   public double getChi()
   {
      return chi.getValue();
   }

   public double getDelta()
   {
      return delta.getValue();
   }

   public double getGamma()
   {
      return gamma.getValue();
   }

   public double getRho()
   {
      return rho.getValue();
   }

   public double getSigma()
   {
      return sigma.getValue();
   }

   public double getSmoothTolFun()
   {
      return smoothTolFun.getValue();
   }

   public double getSmoothTol()
   {
      return smoothTol.getValue();
   }

   public int getMaxIter()
   {
      return maxIter.getValue();
   }

   public int getMaxFunEvals()
   {
      return maxFunEvals.getValue();
   }

   public double getFlatness()
   {
      return flatness.getValue();
   }

   public boolean isStraightenSelected()
   {
      return straightenBox.isSelected();
   }

   public boolean isSmoothSelected()
   {
      return smoothBox.isSelected();
   }

   public void okay()
   {
      ScanImage scanImage = new ScanImage(mainPanel, bitmap,
         this);
      scanImage.performScan();
      setVisible(false);
   }

   public void actionPerformed(ActionEvent e)
   {
      String action = e.getActionCommand();

      if (action == null) return;

      if (action.equals("okay"))
      {
         okay();
      } 
      else if (action.equals("cancel"))
      {
         setVisible(false);
      }
      else if (action.equals("selectBase"))
      {
         colorChooserDialog.setVisible(true);
      }
      else if (action.equals("straighten"))
      {
         setStraighten(straightenBox.isSelected());
      }
      else if (action.equals("smooth"))
      {
         setSmooth(smoothBox.isSelected());
      }
   }

   private JpgfDraw application_;
   private JDRFrame mainPanel;
   private JDRBitmap bitmap;
   private Color base;
   private JDialog colorChooserDialog;
   private JColorChooser colorChooser;

   private NonNegativeIntPanel xIncPanel, yIncPanel;
   private NonNegativeDoublePanel tolerance;
   private JCheckBox straightenBox, smoothBox;
   private NonNegativeDoublePanel chi, delta, gamma, rho, sigma,
      smoothTolFun, smoothTol, flatness;
   private NonNegativeIntPanel maxIter, maxFunEvals;

   private Component swatch;
}
