// File          : TextSelector.java
// Description   : Dialog for editing text areas
// Date          : 6th February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog for editing text areas.
 * @author Nicola L C Talbot
 */

public class TextSelector extends JDialog
   implements ActionListener,SymbolSelectorListener
{
   public TextSelector(JpgfDraw application)
   {
      super(application, JDRResources.getString("edittext.title"),
         true);
      application_ = application;
      setLocationRelativeTo(application_);

      JPanel p1 = new JPanel();
      p1.setLayout(new GridLayout(3,1));

      textbox = new JTextField()
      {
         public void paintComponent(Graphics g)
         {
            Graphics2D g2 = (Graphics2D)g;

            RenderingHints hints = getRenderingHints();

            if (hints != null)
            {
               RenderingHints oldHints = g2.getRenderingHints();
               g2.setRenderingHints(hints);
               super.paintComponent(g);
               g2.setRenderingHints(oldHints);
            }
         }
      };


      symbolSelector = new CharacterSelector(this, this);

      textpopupMenu = new JPopupMenu();

      copyText = new JMenuItem(
          JDRResources.getString("edit.copy"),
          JDRResources.getChar("edit.copy.mnemonic"));
      textpopupMenu.add(copyText);
      copyText.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C,
         InputEvent.CTRL_MASK));
      copyText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textbox.copy();
            }
         });

      cutText = new JMenuItem(
          JDRResources.getString("edit.cut"),
          JDRResources.getChar("edit.cut.mnemonic"));
      textpopupMenu.add(cutText);
      cutText.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,
         InputEvent.CTRL_MASK));
      cutText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textbox.cut();
            }
         });

      JMenuItem pasteText = new JMenuItem(
          JDRResources.getString("edit.paste"),
          JDRResources.getChar("edit.paste.mnemonic"));
      textpopupMenu.add(pasteText);
      pasteText.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V,
         InputEvent.CTRL_MASK));
      pasteText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textbox.paste();
            }
         });

      JMenuItem select_allText = new JMenuItem(
          JDRResources.getString("edit.select_all"),
          JDRResources.getChar("edit.select_all.mnemonic"));
      textpopupMenu.add(select_allText);
      select_allText.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A,
         InputEvent.CTRL_MASK));
      select_allText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textbox.selectAll();
            }
         });


      JMenuItem insertSymbol = new JMenuItem(
         JDRResources.getString("text.insert_symbol"),
         JDRResources.getChar("text.insert_symbol.mnemonic"));
      textpopupMenu.add(insertSymbol);
      insertSymbol.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_INSERT,0));
      insertSymbol.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               symbolSelector.display();
            }
         });

      textbox.addMouseListener(new MouseAdapter()
        {
           public void mousePressed(MouseEvent evt)
           {
              checkForPopupTrigger(evt);
           }

           public void mouseReleased(MouseEvent evt)
           {
              checkForPopupTrigger(evt);
           }

           private void checkForPopupTrigger(MouseEvent evt)
           {
              if (evt.isPopupTrigger())
              {
                 showPopUp(evt.getX(), evt.getY());
              }
           }
        });

      textbox.registerKeyboardAction(this, "insert",
         KeyStroke.getKeyStroke(KeyEvent.VK_INSERT, 0),
         JComponent.WHEN_FOCUSED);

      textbox.registerKeyboardAction(this, "popup",
         KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0),
         JComponent.WHEN_FOCUSED);

      p1.add(textbox);

      JPanel p3 = new JPanel();

      p3.add(new JLabel(
         JDRResources.getString("edittext.latexlabel")));

      ButtonGroup group = new ButtonGroup();
      same = new JRadioButton(
         JDRResources.getString("edittext.same"),true);
      same.setMnemonic(
         JDRResources.getChar("edittext.same.mnemonic"));
      group.add(same);
      p3.add(same);
      same.setActionCommand("same");
      same.addActionListener(this);

      different = new JRadioButton(
         JDRResources.getString("edittext.different"),false);
      different.setMnemonic(
         JDRResources.getChar("edittext.different.mnemonic"));
      group.add(different);
      p3.add(different);
      different.addActionListener(this);
      different.setActionCommand("different");

      p1.add(p3);

      latexbox = new JTextField();

      p1.add(latexbox);

      getContentPane().add(p1, "Center");

      JPanel p2 = new JPanel();

      p2.add(JDRResources.createOkayButton(this));
      p2.add(JDRResources.createCancelButton(this));
      p2.add(JDRResources.createHelpButton("sec:edittext"));

      getContentPane().add(p2, "South");

      pack();
   }

   public void display()
   {
      JDRFrame mainPanel = application_.getCurrentFrame();
      JDRTextual text = mainPanel.getSelectedText();

      if (text == null)
      {
         JDRResources.internalError(
            JDRResources.getString("internal_error.cant_find_selected_text"));
         return;
      }

      textbox.setText(text.getText());
      textbox.requestFocusInWindow();
      latexbox.setText(text.getLaTeXText());

      if (text.getText().equals(text.getLaTeXText()))
      {
         same.setSelected(true);
         latexbox.setEnabled(false);
      }
      else
      {
         different.setSelected(true);
         latexbox.setEnabled(true);
      }

      buttonFont = mainPanel.getSymbolButtonFont();
      textbox.setFont(buttonFont);

      setVisible(true);
   }

   public void setText(String text)
   {
      textbox.setText(text);
      textbox.requestFocusInWindow();
   }

   public Font getSymbolFont()
   {
      return textbox.getFont();
   }

   public String getSymbolText()
   {
      return textbox.getText();
   }

   public Font getSymbolButtonFont()
   {
      return buttonFont;
   }

   public int getSymbolCaretPosition()
   {
      return textbox.getCaretPosition();
   }

   public void setSymbolCaretPosition(int position)
   {
      textbox.setCaretPosition(position);
   }

   public void requestSymbolFocus()
   {
      textbox.requestFocusInWindow();
   }

   public RenderingHints getRenderingHints()
   {
      return application_.getRenderingHints();
   }

   public void okay()
   {
      JDRFrame mainPanel = application_.getCurrentFrame();

      if (same.isSelected())
      {
         mainPanel.setSelectedText(textbox.getText());
      }
      else
      {
         mainPanel.setSelectedText(textbox.getText(),
                                   latexbox.getText());
      }
      setVisible(false);
   }

   public void actionPerformed(ActionEvent e)
   {
      String action = e.getActionCommand();

      if (action == null) return;

      if (action.equals("okay"))
      {
         okay();
      } 
      else if (action.equals("cancel"))
      {
         setVisible(false);
      }
      else if (action.equals("same"))
      {
         latexbox.setText(textbox.getText());
         latexbox.setEnabled(false);
         textbox.requestFocusInWindow();
      }
      else if (action.equals("different"))
      {
         latexbox.setEnabled(true);
         latexbox.requestFocusInWindow();
      }
      else if (action.equals("insert"))
      {
         symbolSelector.display();
      }
      else if (action.equals("popup"))
      {
         Point p = textbox.getCaret().getMagicCaretPosition();

         showPopUp(p.x, p.y);
      }
   }

   public void showPopUp(int x, int y)
   {
      String selectedText = textbox.getSelectedText();

      copyText.setEnabled(selectedText != null);
      cutText.setEnabled(selectedText != null);

      textpopupMenu.show(textbox, x, y);
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "";

      str += "TextSelector:"+eol;
      str += "has focus: "+hasFocus()+eol;
      str += "text box has focus: "+textbox.hasFocus()+eol;
      str += "LaTeX box has focus: "+latexbox.hasFocus()+eol;

      ActionMap actionMap = getRootPane().getActionMap();
      str += "action map: "+eol;

      Object[] allKeys = actionMap.allKeys();

      for (int i = 0; i < allKeys.length; i++)
      {
         str += "Key: "+allKeys[i]
             + " Action: "+actionMap.get(allKeys[i])+eol;
      }

      return str+eol;
   }

   private JpgfDraw application_;
   private JTextField textbox, latexbox;
   private CharacterSelector symbolSelector;
   private JPopupMenu textpopupMenu;
   private JRadioButton same, different;
   private Font buttonFont=null;
   private JMenuItem copyText, cutText;
}
