// File          : RadialGridPanel.java
// Description   : Panel in which to specify radial grid settings
// Date          : 14th Sept 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Panel in which to specify radial grid settings.
 * @author Nicola L C Talbot
 */
public class RadialGridPanel extends GridPanel
{
   public RadialGridPanel()
   {
      super();
      setLayout(new GridBagLayout());

      GridBagConstraints gbc = new GridBagConstraints();
      gbc.gridx = 0;
      gbc.gridy = 0;
      gbc.weightx = 100;
      gbc.weighty = 100;
      gbc.gridheight = 1;
      gbc.gridwidth = 2;

      JLabel majorLabel = new JLabel(
         JDRResources.getString("grid.major"));
      majorLabel.setDisplayedMnemonic(
         JDRResources.getChar("grid.major.mnemonic"));
      add(majorLabel, gbc);

      majorDivisions = new NonNegativeIntField(100);
      majorLabel.setLabelFor(majorDivisions);

      gbc.gridx = 2;
      gbc.gridwidth = 1;
      gbc.fill = GridBagConstraints.BOTH;
      add(majorDivisions, gbc);

      //unitBox = new JComboBox<String>(new String[] {"pt", "in", "cm", "bp"}); // Java7
      unitBox = new JComboBox(new String[] {"pt", "in", "cm", "bp"});

      gbc.gridx = 3;
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.WEST;
      add(unitBox, gbc);

      subDivisions = new NonNegativeIntField(10);

      gbc.gridx = 0;
      gbc.gridy++;
      gbc.gridwidth = 2;
      gbc.anchor = GridBagConstraints.CENTER;
      JLabel subdivisionsLabel = new JLabel(
         JDRResources.getString("grid.sub_divisions"));
      add(subdivisionsLabel, gbc);
      subdivisionsLabel.setDisplayedMnemonic(
         JDRResources.getChar("grid.sub_divisions.mnemonic"));
      subdivisionsLabel.setLabelFor(subDivisions);

      gbc.gridx = 2;
      gbc.gridwidth = 1;
      gbc.fill = GridBagConstraints.BOTH;
      add(subDivisions, gbc);

      gbc.gridx = 0;
      gbc.gridy++;
      gbc.gridwidth = 2;
      gbc.anchor = GridBagConstraints.CENTER;

      spokes = new NonNegativeIntField(8);

      spokesLabel = new JLabel(
         JDRResources.getString("grid.spokes"));
      spokesLabel.setDisplayedMnemonic(
         JDRResources.getChar("grid.spokes.mnemonic"));
      spokesLabel.setLabelFor(spokes);
      add(spokesLabel, gbc);

      gbc.gridx = 2;
      gbc.gridwidth = 1;
      gbc.fill = GridBagConstraints.BOTH;

      add(spokes, gbc);
   }

   public void requestDefaultFieldFocus()
   {
      majorDivisions.requestFocusInWindow();
   }

   public void setGrid(JDRGrid grid)
   {
      majorDivisions.setValue((int)((JDRRadialGrid)grid).getMajorInterval());
      subDivisions.setValue(((JDRRadialGrid)grid).getSubDivisions());
      setUnit(((JDRRadialGrid)grid).getUnit());
      spokes.setValue(((JDRRadialGrid)grid).getSpokes());
   }

   public JDRGrid getGrid(JDRGrid grid)
   {
      Point2D point = grid.getMinorTicDistance();

      double dist = Math.min(point.getX(), point.getY());

      if (dist < 4)
      {
            int selection = JOptionPane.showConfirmDialog(this,
               JDRResources.getString("warning.grid_too_fine"),
               JDRResources.getString("warning.title"),
               JOptionPane.YES_NO_OPTION,
               JOptionPane.WARNING_MESSAGE);

            if (selection != JOptionPane.YES_OPTION) return null;
      }

      try
      {
         if (grid instanceof JDRRadialGrid)
         {
            JDRRadialGrid g = (JDRRadialGrid)grid;
            g.set(getUnit(), getMajor(), getSubDivisions(), getSpokes());
         }
         else
         {
            grid = new JDRRadialGrid(getUnit(), getMajor(),
               getSubDivisions(), getSpokes());
         }
      }
      catch (InvalidGridArgumentException e)
      {
      }

      return grid;
   }

   public int getSpokes()
   {
      int s = spokes.getValue();

      if (s == 0) s = 1;

      return s;
   }

   public double getMajor()
   {
      double d = majorDivisions.getValue();
      if (d == 0) d = 1;

      return d;
   }

   public int getSubDivisions()
   {
      int d = subDivisions.getValue();
      if (d == 0) d = 1;

      return d;
   }

   public void setUnit(JDRUnit unit)
   {
      unitBox.setSelectedItem(unit.getLabel());
   }

   public JDRUnit getUnit()
   {
      switch (unitBox.getSelectedIndex())
      {
         case PT : return JDRUnit.pt;
         case IN : return JDRUnit.in;
         case CM : return JDRUnit.cm;
         case BP : return JDRUnit.bp;
      }

      return JDRUnit.bp;
   }

   private NonNegativeIntField majorDivisions, subDivisions, spokes;
   //private JComboBox<String> unitBox; Java7
   private JComboBox unitBox;
   private JLabel spokesLabel;
}
