// File          : PatternDialogBox.java
// Description   : Dialog box for converting shapes to rotational
//                 patterns
// Date          : 9th Sept 2010
// Last Modified : 10th April 2011
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog box for converting shapes to patterns.
 */

public class PatternDialogBox extends JDialog
   implements ActionListener
{
   public PatternDialogBox(JpgfDraw application)
   {
      super(application,
         JDRResources.getString("pattern.title"), true);
      application_ = application;

      Box box = Box.createVerticalBox();

      JPanel panel = new JPanel();
      JLabel label = new JLabel(
         JDRResources.getString("pattern.replicas"));
      label.setDisplayedMnemonic(
         JDRResources.getChar("pattern.replicas.mnemonic"));
      panel.add(label);

      replicaField = new NonNegativeIntField(1);
      label.setLabelFor(replicaField);
      panel.add(replicaField);

      JLabel modeLabel = new JLabel(
         JDRResources.getString("pattern.mode"));

      panel.add(modeLabel);

      ButtonGroup modeBG = new ButtonGroup();

      singlePath = new JRadioButton(
         JDRResources.getString("pattern.mode.single"), true);
      singlePath.setMnemonic(
         JDRResources.getChar("pattern.mode.single.mnemonic"));

      modeBG.add(singlePath);

      panel.add(singlePath);

      multiPath = new JRadioButton(
         JDRResources.getString("pattern.mode.multi"), false);
      multiPath.setMnemonic(
         JDRResources.getChar("pattern.mode.multi.mnemonic"));

      modeBG.add(multiPath);

      panel.add(multiPath);

      box.add(panel);

      showPath = new JCheckBox(
         JDRResources.getString("pattern.show_original"), true);

      showPath.setMnemonic(
         JDRResources.getChar("pattern.show_original.mnemonic"));

      box.add(showPath);

      tabbedPane = new JTabbedPane();

      rotPanel = new JPanel();

      rotAnglePanel = new AnglePanel(
         JDRResources.getString("rotate.label")+" ",
         JDRResources.getChar("rotate.mnemonic")
      );

      rotAnglePanel.setDegrees(90);

      rotPanel.add(rotAnglePanel);

      tabbedPane.addTab(JDRResources.getString("pattern.rotational"),
         null, rotPanel);

      tabbedPane.setMnemonicAt(0,
         JDRResources.getChar("pattern.rotational.mnemonic"));

      scaledPanel = new JPanel();

      label = new JLabel(
         JDRResources.getString("scale.x")+" ");
      scaledPanel.add(label);
      label.setDisplayedMnemonic(
         JDRResources.getChar("scale.x.mnemonic"));

      scaleXField = new DoubleField(2.0);

      label.setLabelFor(scaleXField);
      scaledPanel.add(scaleXField);

      label = new JLabel(
         JDRResources.getString("scale.y")+" ");
      scaledPanel.add(label);
      label.setDisplayedMnemonic(
         JDRResources.getChar("scale.y.mnemonic"));

      scaleYField = new DoubleField(2.0);

      label.setLabelFor(scaleYField);
      scaledPanel.add(scaleYField);

      tabbedPane.addTab(JDRResources.getString("pattern.scaled"),
         null, scaledPanel);

      tabbedPane.setMnemonicAt(1,
         JDRResources.getChar("pattern.scaled.mnemonic"));

      spiralPanel = new JPanel();

      spiralAnglePanel = new AnglePanel(
         JDRResources.getString("rotate.label")+" ",
         JDRResources.getChar("rotate.mnemonic")
      );

      spiralAnglePanel.setDegrees(20);

      spiralPanel.add(spiralAnglePanel);

      spiralDistancePanel = new NonNegativeLengthPanel(
         JDRResources.getString("pattern.spiral.distance")+" ",
         JDRResources.getChar("pattern.spiral.distance.mnemonic"));

      spiralDistancePanel.setValue(10);

      spiralPanel.add(spiralDistancePanel);

      tabbedPane.addTab(JDRResources.getString("pattern.spiral"),
         null, spiralPanel);

      tabbedPane.setMnemonicAt(2,
         JDRResources.getChar("pattern.spiral.mnemonic"));

      box.add(tabbedPane);

      getContentPane().add(box, "Center");

      JPanel p2 = new JPanel();

      p2.add(JDRResources.createOkayButton(this));
      p2.add(JDRResources.createCancelButton(this));
      p2.add(JDRResources.createHelpButton("sec:pattern"));

      getContentPane().add(p2, "South");

      pack();
      setLocationRelativeTo(application);
   }

   public void display()
   {
      display(-1);
   }

   public void display(int objectIndex)
   {
      JDRPattern pattern=null;
      index = objectIndex;

      if (index != -1)
      {
         JDRCompleteObject object 
            = application_.getCurrentFrame().getObject(index);

         pattern = object.getPattern();
      }

      if (pattern != null)
      {
         replicaField.setValue(pattern.getNumReplicas());

         if (pattern instanceof JDRRotationalPattern)
         {
            JDRRotationalPattern rotPattern 
               = (JDRRotationalPattern)pattern;

            rotAnglePanel.setValue(rotPattern.getRotationAngle());

            tabbedPane.setSelectedComponent(rotPanel);
         }
         else if (pattern instanceof JDRScaledPattern)
         {
            JDRScaledPattern scaledPattern
               = (JDRScaledPattern)pattern;

            scaleXField.setValue(scaledPattern.getScaleX());
            scaleYField.setValue(scaledPattern.getScaleY());

            tabbedPane.setSelectedComponent(scaledPanel);
         }
         else if (pattern instanceof JDRSpiralPattern)
         {
            JDRSpiralPattern spiralPattern 
               = (JDRSpiralPattern)pattern;

            spiralAnglePanel.setValue(spiralPattern.getRotationAngle());

            spiralDistancePanel.setValue(spiralPattern.getDistance());

            tabbedPane.setSelectedComponent(spiralPanel);
         }

         if (pattern.isSinglePath())
         {
            singlePath.setSelected(true);
         }
         else
         {
            multiPath.setSelected(true);
         }

         showPath.setSelected(pattern.showOriginal());
      }

      setVisible(true);
      replicaField.requestFocusInWindow();
   }

   public void actionPerformed(ActionEvent evt)
   {
      String action = evt.getActionCommand();

      if (action == null) return;

      if (action.equals("okay"))
      {
         okay();
      }
      else if (action.equals("cancel"))
      {
         setVisible(false);
      }
   }

   public JDRPattern getPattern()
     throws InvalidFormatException
   {
      Component comp = tabbedPane.getSelectedComponent();

      int replicas = replicaField.getValue();

      if (replicas < 1) replicas = 1;

      if (comp == rotPanel)
      {
         return new JDRRotationalPattern
            (null, null, rotAnglePanel.getRadians(),
             replicas, singlePath.isSelected(),
             showPath.isSelected());

      }
      else if (comp == scaledPanel)
      {
         double scaleX = scaleXField.getValue();
         double scaleY = scaleYField.getValue();

         if (scaleX == 0.0) scaleX = 1.0;
         if (scaleY == 0.0) scaleY = 1.0;

         return new JDRScaledPattern
            (null, null, scaleX, scaleY, replicas,
             singlePath.isSelected(),
             showPath.isSelected());
      }
      else if (comp == spiralPanel)
      {
         return new JDRSpiralPattern
            (null, null, null,
             spiralAnglePanel.getRadians(),
             spiralDistancePanel.getValue(), 
             replicas, singlePath.isSelected(),
             showPath.isSelected());

      }

      throw new InvalidFormatException("No tab selected");
   }

   public void okay()
   {
      JDRPattern pattern = null;

      try
      {
         pattern = getPattern();
      }
      catch (InvalidFormatException e)
      {
         JDRResources.internalError(null, e);
         return;
      }

      setVisible(false);

      if (index == -1)
      {
         application_.getCurrentFrame()
            .convertSelectedToPattern(pattern);
      }
      else
      {
         application_.getCurrentFrame()
            .updatePattern(index, pattern);
      }
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "";

      str += "RotationalPatternDialogBox:"+eol;
      str += "has focus: "+hasFocus()+eol;
      str += "replica field has focus: "+replicaField.hasFocus()+eol;

      ActionMap actionMap = getRootPane().getActionMap();
      str += "action map: "+eol;

      Object[] allKeys = actionMap.allKeys();

      for (int i = 0; i < allKeys.length; i++)
      {
         str += "Key: "+allKeys[i]+" Action: "+actionMap.get(allKeys[i])+eol;
      }

      return str+eol;
   }

   private AnglePanel rotAnglePanel, spiralAnglePanel;

   private DoubleField scaleXField, scaleYField;
   private NonNegativeLengthPanel spiralDistancePanel;

   private NonNegativeIntField replicaField;
   private JpgfDraw application_;

   private JRadioButton singlePath, multiPath;
   private JCheckBox showPath;

   private JTabbedPane tabbedPane;

   private int index=-1;

   private JPanel rotPanel, scaledPanel, spiralPanel;
}
